# ActsAsTree
[![Build Status](https://secure.travis-ci.org/amerine/acts_as_tree.svg?branch=master)](http://travis-ci.org/amerine/acts\_as\_tree)
[![Gem Version](https://badge.fury.io/rb/acts_as_tree.svg)](http://badge.fury.io/rb/acts\_as\_tree)

ActsAsTree extends ActiveRecord to add simple support for organizing items into parent–children relationships. By default, ActsAsTree expects a foreign key column called `parent_id`.

## Example

```ruby
class Category < ActiveRecord::Base
  acts_as_tree order: "name"
end

root      = Category.create("name" => "root")
child1    = root.children.create("name" => "child1")
subchild1 = child1.children.create("name" => "subchild1")

root.parent   # => nil
child1.parent # => root
root.children # => [child1]
root.children.first.children.first # => subchild1
```

We also have a convenient `TreeView` module you can mixin if you want a little visual representation of the tree strucuture. Example:

```ruby
class Category < ActiveRecord::Base
  extend ActsAsTree::TreeView

  acts_as_tree order: 'name'
end

> Category.tree_view(:name)
root
 |_ child1
 |    |_ subchild1
 |    |_ subchild2
 |_ child2
      |_ subchild3
      |_ subchild4
=> nil
```

And there's a `TreeWalker` module (traversing the tree using depth-first search (default) or breadth-first search) as well. Example given the Model `Page` as

```ruby
class Page < ActiveRecord::Base
  extend ActsAsTree::TreeWalker

  acts_as_tree order: 'rank'
end
```

In your view you could traverse the tree using

```erb
<% Page.walk_tree do |page, level| %>
  <%= link_to "#{'-'*level}#{page.name}", page_path(page) %><br />
<% end %>
```

You also could use walk\_tree as an instance method such as:

```erb
<% Page.first.walk_tree do |page, level| %>
  <%= link_to "#{'-'*level}#{page.name}", page_path(page) %><br />
<% end %>
```

## Compatibility

We no longer support Ruby 1.8 or versions of Rails/ActiveRecord older than 3.0. If you're using a version of ActiveRecord older than 3.0 please use 0.1.1.

Moving forward we will do our best to support the latest versions of ActiveRecord and Ruby.

## Change Log
* 2.7.1 - January 30, 2018
	* Fix column quoting if the `:order` option is a symbol, see #73, #74 -- felixbuenemann
* 2.7.0 - September 15, 2017
	* Added support for rails 5.1, see #67, #68 -- felixbuenemann, marcinwierzbicki
* 2.6.1 - January 18, 2017
	* Avoid conflicts of `#level` method with existing column, see #57, #58, #60 -- markhgbrewster
	* Fix tests on rails 4.2 with ruby < 2.1 -- felixbuenemann
* 2.6.0 - October 9, 2016
	* Add generations methods, see #56 -- markhgbrewster
* 2.5.1 - September 8, 2016
	* Fix early database connection in acts\_as\_tree, see #55 -- felixbuenemann
* 2.5.0 - August 14, 2016
	* Allow for use of a different primary key, see #50 -- Two9A
* 2.4.0 - January 12, 2016
	* Added support for rails 5.0, see #46 -- klacointe
* 2.3.0 - November 6, 2015
	* Added touch option to acts\_as\_tree relation. See #40 -- mbenitezm
	* Fix tests on rails 3.x with ruby 1.9.2. See #41 -- felixbuenemann
* 2.2.0 - June 15, 2015
	* Added TreeWalker.walk\_tree instance method. See #32, #37, #38 -- felixbuenemann, genewoo
	* Fix tests on rails 3.x. See #36 -- marshall-lee
* 2.1.0 - September 25, 2014
	* Added TreeWalker. See #30 -- 545ch4
* 2.0.0 - July 3, 2014
	* Renamed Presentation module to TreeView, see #27, #28 -- felixbuenemann
* 1.6.1 - May 29, 2014
	* Readme Improvements, see #26 -- schlick
	* Improvements and Fixes for counter cache (fix counter\_cache: true). See #24, #25 -- dv
	* Cleanup and fix tests, see #24.
* 1.6.0 - April 21, 2014
	* Added new `leaves` method. See #23 -- MichalPokorny
* 1.5.1 - March 28, 2014
	* Fixing descendants modification bug. See #20 -- amerine, tmuerell
* 1.5.0 - December 16, 2013
	* Added new `descendants` method -- adamkleingit
	* Fixed warning message -- akicho8
* 1.4.0 - June 25, 2013
	* `Presentation#tree_view` -- rainchen
	* `root?` && `leaf?` methods. -- xuanxu
* 1.3.0 - March 29, 2013
	* Rails 4.0 Support! -- mischa78
	* Readme Fixes -- mischa78 & seanhussey
* 1.2.0 - October 29, 2012
	* Adding new `self_with_ancestors` accessor -- felixbuenemann
	* `roots` is now a scope.
* 1.1.0 - April 24, 2012
	* Deprecate the ActiveRecord::Acts::Tree module in favor of ActsAsTree
* 1.0.1 - April 18, 2012
	* Include the Railtie for easier loading in Rails. Will reassess the forced module inclusion in the future.
* 1.0.0 - April 14, 2012
	* Official 1.0 release. Force users to include the ActiveRecord::Acts::Tree module.
* 0.2.0 - April 9, 2012
	* Rails 3 Support
* 0.1.1 - February 3, 2010
	* Bug Fixes
* 0.1.0 - October 9, 2009
	* First Gem Release

## Note on Patches/Pull Requests

1. Fork the project.
2. Make your feature addition or bug fix.
3. Add tests for it. This is important so we don't break it in a future version
   unintentionally.
4. Commit, do not mess with rakefile, version, or history. (if you want to have
   your own version, that is fine but bump version in a commit by itself so we can
   ignore when we pull)
5. Send us a pull request. Bonus points for topic branches.
6. All contributors to this project, after their first accepted patch, are given push
   access to the repository and are welcome as full contributors to ActsAsTree. All
   we ask is that all changes go through CI and a Pull Request before merging.

## Releasing new versions

1. We follow Semver. So if you're shipping interface breaking changes then bump
   the major version. We don't care if we ship version 1101.1.1, as long as
   people know that 1101.1.1 has breaking differences from 1100.0. If you're
   adding new features, but not changing existing functionality bump the minor
   version, if you're shipping a bugfix, just bump the patch.
2. Following the above rules, change the version found in lib/acts\_as\_tree/version.rb.
3. Make sure the Change log in the README includes a brief summary of the versions
   changes, with credit to the contributors.
4. Commit these changes in one "release-prep" commit (on the master branch).
5. Push that commit up to the repo.
6. Run `rake release`
   This will create and push a tag to Github, then generate a gem and push it to
   Rubygems.
7. Profit.

## License (MIT)

Copyright (c) 2007 David Heinemeier Hansson

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the 'Software'), to deal in the
Software without restriction, including without limitation the rights to use,
copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
Software, and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
