#!/bin/env ruby
#
#---------------------------------------------------------------------- 
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version. This program is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
# Public License for more details; it is available at
# <http://www.fsf.org/copyleft/gpl.html>, or by writing to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
# Written by Gordon Miller <gmiller@bittwiddlers.com>. Inspired by and partly
# derived from the Python version by Michael Haggerty. If you find a problem
# or have a suggestion, please let me know at <gmiller@bittwiddlers.com>.
# Other feedback would also be appreciated.
#
#----------------------------------------------------------------------
#
# This file contains the implementations of the Stream objects.  These objects
# represent Sinks of data for Gnuplot commands (probably should call them Sink
# objects instead). 
#
#---------------------------------------------------------------------- 

class GnuplotStream
  # Caches the data that is written into the stream into a string.  This
  # string can then be accessed via the data accessor.
  
  attr_reader :data

  def initialize ()
    @data = ""
  end

  def write (s)
    @data += s
  end
  
  def flush
  end

  def insert (s)
    write(s + "\n")
    flush
  end
end

class GnuplotFile

  # A file to which gnuplot commands can be written.
  # 
  # Sometimes it is convenient to write gnuplot commands to a command file for
  # later evaluation. In that case, one of these objects is used as a mock
  # gnuplot process. Note that temporary files may be deleted before you have
  # time to execute the file!
  # 
  # Members:
  # 
  # 'gnuplot' -- the file object gathering the commands.
  # 
  # Methods:
  # 
  # 'new'    -- open the file.
  # 'insert' -- write a gnuplot command to the file, followed by a
  #             newline.
  # 'write'  -- write an arbitrary string to the file.
  # 'flush'  -- cause pending output to be written immediately.

  def initialize (filename)
    @file = open(filename, 'w')
    insert("#!/bin/env gnuplot")
  end

  def write (s)
    @file.write (s)
  end
  
  def flush
    @file.flush
  end

  def insert (s)
    write(s + "\n")
    flush()
  end
end

class GnuplotProcess

  # Unsophisticated interface to a running gnuplot program.
  # 
  # This represents a running gnuplot program and the means to
  # communicate with it at a primitive level (i.e., pass it commands
  # or data).  When the object is destroyed, the gnuplot program exits
  # (unless the 'persist' option was set).  The communication is
  # one-way; gnuplot's text output just goes to stdout with no attempt
  # to check it for error messages.
  # 
  # Methods:
  # 
  # 'initialize' -- Start up the program.
  #
  # 'insert'     -- Pass an arbitrary string to the gnuplot program,
  #                 followed by a newline, and then flush it.
  #
  # 'flush'      -- Cause pending output to be written immediately.
  #
  # 'write'      -- Pass an arbitrary string to the gnuplot program.
  #

  attr_reader :cmd

  def initialize (persist = true)
    # Start a gnuplot process.
    # 
    # Create a 'GnuplotProcess' object.  This starts a gnuplot
    # program and prepares to write commands to it.
    #
    # persist - If set to true then the gnuplot command will be run with the
    #           'persist' command line option. 
    # 

    @cmd = "/usr/bin/gnuplot"
    @cmd += " -persist" if persist

    @gnuplot = IO::popen(@cmd, "w")
  end

  def flush
    # Cause pending output to be written immediately.
    @gnuplot.flush
  end

  def insert (s)
    # Send a command string to gnuplot, followed by newline, and flush it.
    write(s + "\n")
    flush()
  end

  def write (s)
    # Pass an arbitrary string to the gnuplot program
    @gnuplot.write ( s )
  end
  
end

# $Id: Stream.rb,v 1.3 2001/01/25 06:49:08 gmiller Exp $
