/*
* Copyright (C) 2015 Samsung System LSI
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
package android.bluetooth;

import android.os.Parcel;
import android.os.Parcelable;

/** @hide */
public class SdpMasRecord implements Parcelable {
    private final int mMasInstanceId;
    private final int mL2capPsm;
    private final int mRfcommChannelNumber;
    private final int mProfileVersion;
    private final int mSupportedFeatures;
    private final int mSupportedMessageTypes;
    private final String mServiceName;
    public static final class MessageType {
        public static final int EMAIL    = 0x01;
        public static final int SMS_GSM  = 0x02;
        public static final int SMS_CDMA = 0x04;
        public static final int MMS      = 0x08;
    }

    public SdpMasRecord(int mas_instance_id,
                                 int l2cap_psm,
                                 int rfcomm_channel_number,
                                 int profile_version,
                                 int supported_features,
                                 int supported_message_types,
                                 String service_name){
        this.mMasInstanceId = mas_instance_id;
        this.mL2capPsm = l2cap_psm;
        this.mRfcommChannelNumber = rfcomm_channel_number;
        this.mProfileVersion = profile_version;
        this.mSupportedFeatures = supported_features;
        this.mSupportedMessageTypes = supported_message_types;
        this.mServiceName = service_name;
    }

    public SdpMasRecord(Parcel in){
        this.mMasInstanceId = in.readInt();
        this.mL2capPsm = in.readInt();
        this.mRfcommChannelNumber = in.readInt();
        this.mProfileVersion = in.readInt();
        this.mSupportedFeatures = in.readInt();
        this.mSupportedMessageTypes = in.readInt();
        this.mServiceName = in.readString();
    }
    @Override
    public int describeContents() {
        // TODO Auto-generated method stub
        return 0;
    }

    public int getMasInstanceId() {
        return mMasInstanceId;
    }

    public int getL2capPsm() {
        return mL2capPsm;
    }

    public int getRfcommCannelNumber() {
        return mRfcommChannelNumber;
    }

    public int getProfileVersion() {
        return mProfileVersion;
    }

    public int getSupportedFeatures() {
        return mSupportedFeatures;
    }

    public int getSupportedMessageTypes() {
        return mSupportedMessageTypes;
    }
    
    public boolean msgSupported(int msg) {
        return (mSupportedMessageTypes & msg) != 0;
    }
    
    public String getServiceName() {
        return mServiceName;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {

        dest.writeInt(this.mMasInstanceId);
        dest.writeInt(this.mL2capPsm);
        dest.writeInt(this.mRfcommChannelNumber);
        dest.writeInt(this.mProfileVersion);
        dest.writeInt(this.mSupportedFeatures);
        dest.writeInt(this.mSupportedMessageTypes);
        dest.writeString(this.mServiceName);

    }
    @Override
    public String toString(){
        String ret = "Bluetooth MAS SDP Record:\n";

        if(mMasInstanceId != -1){
            ret += "Mas Instance Id: " + mMasInstanceId + "\n";
        }
        if(mRfcommChannelNumber != -1){
            ret += "RFCOMM Chan Number: " + mRfcommChannelNumber + "\n";
        }
        if(mL2capPsm != -1){
            ret += "L2CAP PSM: " + mL2capPsm + "\n";
        }
        if(mServiceName != null){
            ret += "Service Name: " + mServiceName + "\n";
        }
        if(mProfileVersion != -1){
            ret += "Profile version: " + mProfileVersion + "\n";
        }
        if(mSupportedMessageTypes != -1){
            ret += "Supported msg types: " + mSupportedMessageTypes + "\n";
        }
        if(mSupportedFeatures != -1){
            ret += "Supported features: " + mSupportedFeatures + "\n";
        }
        return ret;
    }

    public static final Parcelable.Creator CREATOR = new Parcelable.Creator() {
        public SdpMasRecord createFromParcel(Parcel in) {
            return new SdpMasRecord(in);
        }
        public SdpRecord[] newArray(int size) {
            return new SdpRecord[size];
        }
    };
}
