//===-- xray_interface.h ----------------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
// This file is a part of XRay, a dynamic runtime instrumentation system.
//
// APIs for controlling XRay functionality explicitly.
//===----------------------------------------------------------------------===//
#ifndef XRAY_XRAY_INTERFACE_H
#define XRAY_XRAY_INTERFACE_H

#include <cstdint>

extern "C" {

enum XRayEntryType { ENTRY = 0, EXIT = 1, TAIL = 2 };

// Provide a function to invoke for when instrumentation points are hit. This is
// a user-visible control surface that overrides the default implementation. The
// function provided should take the following arguments:
//
//   - function id: an identifier that indicates the id of a function; this id
//                  is generated by xray; the mapping between the function id
//                  and the actual function pointer is available through
//                  __xray_table.
//   - entry type: identifies what kind of instrumentation point was encountered
//                 (function entry, function exit, etc.). See the enum
//                 XRayEntryType for more details.
//
// Returns 1 on success, 0 on error.
extern int __xray_set_handler(void (*entry)(int32_t, XRayEntryType));

// This removes whatever the currently provided handler is. Returns 1 on
// success, 0 on error.
extern int __xray_remove_handler();

enum XRayPatchingStatus {
  NOT_INITIALIZED = 0,
  SUCCESS = 1,
  ONGOING = 2,
  FAILED = 3,
};

// This tells XRay to patch the instrumentation points. See XRayPatchingStatus
// for possible result values.
extern XRayPatchingStatus __xray_patch();

// Reverses the effect of __xray_patch(). See XRayPatchingStatus for possible
// result values.
extern XRayPatchingStatus __xray_unpatch();
}

#endif
