/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include <mediascanner/utilities.h>

// C Standard Library
#include <cxxabi.h>
#include <execinfo.h>
#include <stdio.h>

// C++ Standard Library
#include <string>

// Media Scanner Library
#include "mediascanner/locale.h"
#include "mediascanner/property.h"

namespace mediascanner {

void abort_with_backtrace() {
    static const int max_depth = 100;
    void *stack_addrs[max_depth];
    const int depth = backtrace(stack_addrs, max_depth);
    char **symbols = backtrace_symbols(stack_addrs, depth);
    std::string backtrace;

    if (symbols) {
        for (int i = 1; i < depth; ++i) {
            std::string line = symbols[i];
            std::string::size_type begin = line.find('(');
            std::string::size_type end = line.find('+', begin);

            if (end - begin > 1
                    && begin++ != std::string::npos
                    && end != std::string::npos) {
                int status;
                char *const name = abi::__cxa_demangle
                        (line.substr(begin, end - begin).c_str(),
                         nullptr, nullptr, &status);

                if (name) {
                    if (status == 0)
                        line = line.substr(0, begin) + name + line.substr(end);

                    free(name);
                }
            }

            backtrace += line + "\n";
        }

        const size_t len = write(1, backtrace.data(), backtrace.length());
        BOOST_ASSERT(len == backtrace.length());

        free(symbols);
    } else {
        const std::string message = "No backtrace available.\n";
        const size_t len = write(1, message.data(), message.length());
        BOOST_ASSERT(len == message.length());
    }

    abort();
}

std::wstring safe_wstring(const char *str) {
    return str ? ToUnicode(str) : std::wstring();
}

std::ostream &operator<<(std::ostream &os, const Property &p) {
    return os << "property: " << p.field_name();
}

} // namespace mediascanner

namespace std {

ostream& operator<<(ostream &os, const wstring &s) {
    return os << ::mediascanner::FromUnicode(s);
}

} // namespace std
