/*
 * Copyright (c) 2001, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 *
 */

package sun.jvm.hotspot.runtime;

import java.util.*;

import sun.jvm.hotspot.debugger.*;
import sun.jvm.hotspot.oops.*;
import sun.jvm.hotspot.types.*;
import sun.jvm.hotspot.utilities.Observable;
import sun.jvm.hotspot.utilities.Observer;

public class ObjectMonitor extends VMObject {
  static {
    VM.registerVMInitializedObserver(new Observer() {
        public void update(Observable o, Object data) {
          initialize(VM.getVM().getTypeDataBase());
        }
      });
  }

  private static synchronized void initialize(TypeDataBase db) throws WrongTypeException {
    heap = VM.getVM().getObjectHeap();
    Type type  = db.lookupType("ObjectMonitor");

    sun.jvm.hotspot.types.Field f = type.getField("_metadata");
    metadataFieldOffset = f.getOffset();
    f = type.getField("_object");
    objectFieldOffset = f.getOffset();
    f = type.getField("_owner");
    ownerFieldOffset = f.getOffset();
    f = type.getField("_stack_locker");
    stackLockerFieldOffset = f.getOffset();
    f = type.getField("_next_om");
    nextOMFieldOffset = f.getOffset();
    contentionsField  = new CIntField(type.getCIntegerField("_contentions"), 0);
    waitersField      = new CIntField(type.getCIntegerField("_waiters"), 0);
    recursionsField   = type.getCIntegerField("_recursions");

    ANONYMOUS_OWNER = db.lookupLongConstant("ObjectMonitor::ANONYMOUS_OWNER").longValue();
  }

  public ObjectMonitor(Address addr) {
    super(addr);
  }

  public Mark header() {
    return new Mark(addr.addOffsetTo(metadataFieldOffset));
  }

  // FIXME
  //  void      set_header(markWord hdr);

  // FIXME: must implement and delegate to platform-dependent implementation
  //  public boolean isBusy();
  public boolean isEntered(sun.jvm.hotspot.runtime.Thread current) {
    Address o = owner();
    if (current.threadObjectAddress().equals(o) ||
        current.isLockOwned(o)) {
      return true;
    }
    return false;
  }

  public boolean isOwnedAnonymous() {
    return addr.getAddressAt(ownerFieldOffset).asLongValue() == ANONYMOUS_OWNER;
  }

  public Address owner() { return addr.getAddressAt(ownerFieldOffset); }
  public Address stackLocker() { return addr.getAddressAt(stackLockerFieldOffset); }
  // FIXME
  //  void      set_owner(void* owner);

  public int    waiters() { return (int)waitersField.getValue(this); }

  public Address nextOM() { return addr.getAddressAt(nextOMFieldOffset); }
  // FIXME
  //  void      set_queue(void* owner);

  public long recursions() { return recursionsField.getValue(addr); }

  public OopHandle object() {
    Address objAddr = addr.getAddressAt(objectFieldOffset);
    if (objAddr == null) {
      return null;
    }
    return objAddr.getOopHandleAt(0);
  }

  public int contentions() {
      return (int)contentionsField.getValue(this);
  }

  // The following four either aren't expressed as typed fields in
  // vmStructs.cpp because they aren't strongly typed in the VM, or
  // would confuse the SA's type system.
  private static ObjectHeap    heap;
  private static long          metadataFieldOffset;
  private static long          objectFieldOffset;
  private static long          ownerFieldOffset;
  private static long          stackLockerFieldOffset;
  private static long          nextOMFieldOffset;
  private static CIntField     contentionsField;
  private static CIntField     waitersField;
  private static CIntegerField recursionsField;
  private static long          ANONYMOUS_OWNER;

  // FIXME: expose platform-dependent stuff
}
