# Copyright 1999 by Cayte Lindner.  All rights reserved.
# This code is part of the Biopython distribution and governed by its
# license.  Please see the LICENSE file that should have been included
# as part of this package.
#
# NOTE - This file has been split in two as a workaround for Jython JVM limits.

"""Tests for prosite2 module."""

import os
import unittest

from Bio.ExPASy import Prosite


class TestPrositeRead(unittest.TestCase):

    def test_read4(self):
        """Parsing Prosite record ps00432.txt."""
        filename = os.path.join("Prosite", "ps00432.txt")
        with open(filename) as handle:
            record = Prosite.read(handle)
        self.assertEqual(record.name, "ACTINS_2")
        self.assertEqual(record.type, "PATTERN")
        self.assertEqual(record.accession, "PS00432")
        self.assertEqual(record.created, "NOV-1990")
        self.assertEqual(record.data_update, "DEC-2004")
        self.assertEqual(record.info_update, "MAR-2006")
        self.assertEqual(record.pdoc, "PDOC00340")
        self.assertEqual(record.description, "Actins signature 2.")
        self.assertEqual(record.pattern, "W-[IVC]-[STAK]-[RK]-x-[DE]-Y-[DNE]-[DE].")
        self.assertEqual(record.matrix, [])
        self.assertEqual(record.rules, [])
        self.assertEqual(record.nr_sp_release, "49.3")
        self.assertEqual(record.nr_sp_seqs, 212425)
        self.assertEqual(record.cc_taxo_range, "??E??")
        self.assertEqual(record.cc_max_repeat, "1")
        self.assertEqual(len(record.cc_site), 0)
        self.read4_positive(record)
        self.read4_false_etc(record)

    def read4_positive(self, record):
        self.assertEqual(len(record.dr_positive), 295)
        self.assertEqual(record.dr_positive[0], ("O96019", "ACL6A_HUMAN"))
        self.assertEqual(record.dr_positive[1], ("Q9Z2N8", "ACL6A_MOUSE"))
        self.assertEqual(record.dr_positive[2], ("O94805", "ACL6B_HUMAN"))
        self.assertEqual(record.dr_positive[3], ("Q99MR0", "ACL6B_MOUSE"))
        self.assertEqual(record.dr_positive[4], ("P53496", "ACT11_ARATH"))
        self.assertEqual(record.dr_positive[5], ("P30171", "ACT11_SOLTU"))
        self.assertEqual(record.dr_positive[6], ("P53497", "ACT12_ARATH"))
        self.assertEqual(record.dr_positive[7], ("P30172", "ACT12_SOLTU"))
        self.assertEqual(record.dr_positive[8], ("P30173", "ACT13_SOLTU"))
        self.assertEqual(record.dr_positive[9], ("P84185", "ACT1D_ANOGA"))
        self.assertEqual(record.dr_positive[10], ("P02578", "ACT1_ACACA"))
        self.assertEqual(record.dr_positive[11], ("P49128", "ACT1_AEDAE"))
        self.assertEqual(record.dr_positive[12], ("P10671", "ACT1_ARATH"))
        self.assertEqual(record.dr_positive[13], ("P18600", "ACT1_ARTSX"))
        self.assertEqual(record.dr_positive[14], ("P83969", "ACT1_BACDO"))
        self.assertEqual(record.dr_positive[15], ("P07836", "ACT1_BOMMO"))
        self.assertEqual(record.dr_positive[16], ("P10983", "ACT1_CAEEL"))
        self.assertEqual(record.dr_positive[17], ("P23343", "ACT1_DAUCA"))
        self.assertEqual(record.dr_positive[18], ("P02577", "ACT1_DICDI"))
        self.assertEqual(record.dr_positive[19], ("P68556", "ACT1_DIPDE"))
        self.assertEqual(record.dr_positive[20], ("P10987", "ACT1_DROME"))
        self.assertEqual(record.dr_positive[21], ("P35432", "ACT1_ECHGR"))
        self.assertEqual(record.dr_positive[22], ("P68142", "ACT1_FUGRU"))
        self.assertEqual(record.dr_positive[23], ("P53460", "ACT1_HALRO"))
        self.assertEqual(record.dr_positive[24], ("P69002", "ACT1_HELER"))
        self.assertEqual(record.dr_positive[25], ("P69003", "ACT1_HELTB"))
        self.assertEqual(record.dr_positive[26], ("P92182", "ACT1_LUMTE"))
        self.assertEqual(record.dr_positive[27], ("P53465", "ACT1_LYTPI"))
        self.assertEqual(record.dr_positive[28], ("P02582", "ACT1_MAIZE"))
        self.assertEqual(record.dr_positive[29], ("P27131", "ACT1_NAEFO"))
        self.assertEqual(record.dr_positive[30], ("P30162", "ACT1_ONCVO"))
        self.assertEqual(record.dr_positive[31], ("Q98972", "ACT1_ORYLA"))
        self.assertEqual(record.dr_positive[32], ("P13362", "ACT1_ORYSA"))
        self.assertEqual(record.dr_positive[33], ("P53503", "ACT1_OXYFA"))
        self.assertEqual(record.dr_positive[34], ("P12715", "ACT1_OXYNO"))
        self.assertEqual(record.dr_positive[35], ("P53468", "ACT1_OXYTR"))
        self.assertEqual(record.dr_positive[36], ("P30164", "ACT1_PEA"))
        self.assertEqual(record.dr_positive[37], ("P22131", "ACT1_PHYIN"))
        self.assertEqual(record.dr_positive[38], ("P10988", "ACT1_PLAFA"))
        self.assertEqual(record.dr_positive[39], ("P43239", "ACT1_PNECA"))
        self.assertEqual(record.dr_positive[40], ("P41112", "ACT1_PODCA"))
        self.assertEqual(record.dr_positive[41], ("O18499", "ACT1_SACKO"))
        self.assertEqual(record.dr_positive[42], ("Q9Y702", "ACT1_SCHCO"))
        self.assertEqual(record.dr_positive[43], ("P53470", "ACT1_SCHMA"))
        self.assertEqual(record.dr_positive[44], ("P53504", "ACT1_SORBI"))
        self.assertEqual(record.dr_positive[45], ("P02581", "ACT1_SOYBN"))
        self.assertEqual(record.dr_positive[46], ("P10990", "ACT1_STRFN"))
        self.assertEqual(record.dr_positive[47], ("Q9Y701", "ACT1_SUIBO"))
        self.assertEqual(record.dr_positive[48], ("P10992", "ACT1_TETTH"))
        self.assertEqual(record.dr_positive[49], ("Q05214", "ACT1_TOBAC"))
        self.assertEqual(record.dr_positive[50], ("P12432", "ACT1_TRYBB"))
        self.assertEqual(record.dr_positive[51], ("P04751", "ACT1_XENLA"))
        self.assertEqual(record.dr_positive[52], ("P26197", "ACT2_ABSGL"))
        self.assertEqual(record.dr_positive[53], ("Q96292", "ACT2_ARATH"))
        self.assertEqual(record.dr_positive[54], ("P18601", "ACT2_ARTSX"))
        self.assertEqual(record.dr_positive[55], ("P45885", "ACT2_BACDO"))
        self.assertEqual(record.dr_positive[56], ("P07837", "ACT2_BOMMO"))
        self.assertEqual(record.dr_positive[57], ("P10984", "ACT2_CAEEL"))
        self.assertEqual(record.dr_positive[58], ("P23344", "ACT2_DAUCA"))
        self.assertEqual(record.dr_positive[59], ("P07827", "ACT2_DICDI"))
        self.assertEqual(record.dr_positive[60], ("P53456", "ACT2_DIPDE"))
        self.assertEqual(record.dr_positive[61], ("P02572", "ACT2_DROME"))
        self.assertEqual(record.dr_positive[62], ("Q03341", "ACT2_ECHGR"))
        self.assertEqual(record.dr_positive[63], ("P53485", "ACT2_FUGRU"))
        self.assertEqual(record.dr_positive[64], ("P27130", "ACT2_HALRO"))
        self.assertEqual(record.dr_positive[65], ("P92176", "ACT2_LUMTE"))
        self.assertEqual(record.dr_positive[66], ("P53466", "ACT2_LYTPI"))
        self.assertEqual(record.dr_positive[67], ("Q25472", "ACT2_MOLOC"))
        self.assertEqual(record.dr_positive[68], ("P27132", "ACT2_NAEFO"))
        self.assertEqual(record.dr_positive[69], ("P30163", "ACT2_ONCVO"))
        self.assertEqual(record.dr_positive[70], ("P17298", "ACT2_ORYSA"))
        self.assertEqual(record.dr_positive[71], ("P02583", "ACT2_OXYFA"))
        self.assertEqual(record.dr_positive[72], ("P55805", "ACT2_OXYNO"))
        self.assertEqual(record.dr_positive[73], ("P53469", "ACT2_OXYTR"))
        self.assertEqual(record.dr_positive[74], ("P30165", "ACT2_PEA"))
        self.assertEqual(record.dr_positive[75], ("P22132", "ACT2_PHYIN"))
        self.assertEqual(record.dr_positive[76], ("P14883", "ACT2_PLAFA"))
        self.assertEqual(record.dr_positive[77], ("O18500", "ACT2_SACKO"))
        self.assertEqual(record.dr_positive[78], ("Q9Y896", "ACT2_SCHCO"))
        self.assertEqual(record.dr_positive[79], ("P53471", "ACT2_SCHMA"))
        self.assertEqual(record.dr_positive[80], ("P69004", "ACT2_STRFN"))
        self.assertEqual(record.dr_positive[81], ("Q9Y707", "ACT2_SUIBO"))
        self.assertEqual(record.dr_positive[82], ("P10993", "ACT2_TETPY"))
        self.assertEqual(record.dr_positive[83], ("P12433", "ACT2_TRYBB"))
        self.assertEqual(record.dr_positive[84], ("P10995", "ACT2_XENLA"))
        self.assertEqual(record.dr_positive[85], ("P20399", "ACT2_XENTR"))
        self.assertEqual(record.dr_positive[86], ("Q25010", "ACT3A_HELAM"))
        self.assertEqual(record.dr_positive[87], ("P84184", "ACT3B_HELAM"))
        self.assertEqual(record.dr_positive[88], ("P18602", "ACT3_ARTSX"))
        self.assertEqual(record.dr_positive[89], ("P45886", "ACT3_BACDO"))
        self.assertEqual(record.dr_positive[90], ("P04829", "ACT3_BOMMO"))
        self.assertEqual(record.dr_positive[91], ("P07829", "ACT3_DICDI"))
        self.assertEqual(record.dr_positive[92], ("P53457", "ACT3_DIPDE"))
        self.assertEqual(record.dr_positive[93], ("P53501", "ACT3_DROME"))
        self.assertEqual(record.dr_positive[94], ("Q03342", "ACT3_ECHGR"))
        self.assertEqual(record.dr_positive[95], ("P53486", "ACT3_FUGRU"))
        self.assertEqual(record.dr_positive[96], ("P41340", "ACT3_LIMPO"))
        self.assertEqual(record.dr_positive[97], ("Q25379", "ACT3_LYTPI"))
        self.assertEqual(record.dr_positive[98], ("P17299", "ACT3_ORYSA"))
        self.assertEqual(record.dr_positive[99], ("P46258", "ACT3_PEA"))
        self.assertEqual(record.dr_positive[100], ("P41113", "ACT3_PODCA"))
        self.assertEqual(record.dr_positive[101], ("P30167", "ACT3_SOLTU"))
        self.assertEqual(record.dr_positive[102], ("P02580", "ACT3_SOYBN"))
        self.assertEqual(record.dr_positive[103], ("P04752", "ACT3_XENLA"))
        self.assertEqual(record.dr_positive[104], ("P53494", "ACT4_ARATH"))
        self.assertEqual(record.dr_positive[105], ("P18603", "ACT4_ARTSX"))
        self.assertEqual(record.dr_positive[106], ("P84183", "ACT4_BOMMO"))
        self.assertEqual(record.dr_positive[107], ("P10986", "ACT4_CAEEL"))
        self.assertEqual(record.dr_positive[108], ("P07828", "ACT4_DICDI"))
        self.assertEqual(record.dr_positive[109], ("P02574", "ACT4_DROME"))
        self.assertEqual(record.dr_positive[110], ("Q25380", "ACT4_LYTPI"))
        self.assertEqual(record.dr_positive[111], ("Q8RYC2", "ACT5_ARATH"))
        self.assertEqual(record.dr_positive[112], ("P45887", "ACT5_BACDO"))
        self.assertEqual(record.dr_positive[113], ("P53478", "ACT5_CHICK"))
        self.assertEqual(record.dr_positive[114], ("P53458", "ACT5_DIPDE"))
        self.assertEqual(record.dr_positive[115], ("P10981", "ACT5_DROME"))
        self.assertEqual(record.dr_positive[116], ("P53505", "ACT5_XENLA"))
        self.assertEqual(record.dr_positive[117], ("P53459", "ACT6_DIPDE"))
        self.assertEqual(record.dr_positive[118], ("P83967", "ACT6_DROME"))
        self.assertEqual(record.dr_positive[119], ("P83968", "ACT6_DROSI"))
        self.assertEqual(record.dr_positive[120], ("P30168", "ACT6_SOLTU"))
        self.assertEqual(record.dr_positive[121], ("P53492", "ACT7_ARATH"))
        self.assertEqual(record.dr_positive[122], ("P17300", "ACT7_ORYSA"))
        self.assertEqual(record.dr_positive[123], ("Q96293", "ACT8_ARATH"))
        self.assertEqual(record.dr_positive[124], ("P07830", "ACT8_DICDI"))
        self.assertEqual(record.dr_positive[125], ("P53506", "ACT8_XENLA"))
        self.assertEqual(record.dr_positive[126], ("P93738", "ACT9_ARATH"))
        self.assertEqual(record.dr_positive[127], ("P62739", "ACTA_BOVIN"))
        self.assertEqual(record.dr_positive[128], ("P08023", "ACTA_CHICK"))
        self.assertEqual(record.dr_positive[129], ("P62736", "ACTA_HUMAN"))
        self.assertEqual(record.dr_positive[130], ("P41339", "ACTA_LIMPO"))
        self.assertEqual(record.dr_positive[131], ("P62737", "ACTA_MOUSE"))
        self.assertEqual(record.dr_positive[132], ("P02576", "ACTA_PHYPO"))
        self.assertEqual(record.dr_positive[133], ("P62740", "ACTA_RABIT"))
        self.assertEqual(record.dr_positive[134], ("P62738", "ACTA_RAT"))
        self.assertEqual(record.dr_positive[135], ("P53472", "ACTA_STRPU"))
        self.assertEqual(record.dr_positive[136], ("Q7ZVI7", "ACTB1_BRARE"))
        self.assertEqual(record.dr_positive[137], ("Q7ZVF9", "ACTB2_BRARE"))
        self.assertEqual(record.dr_positive[138], ("P60712", "ACTB_BOVIN"))
        self.assertEqual(record.dr_positive[139], ("P84336", "ACTB_CAMDR"))
        self.assertEqual(record.dr_positive[140], ("O18840", "ACTB_CANFA"))
        self.assertEqual(record.dr_positive[141], ("Q71FK5", "ACTB_CAVPO"))
        self.assertEqual(record.dr_positive[142], ("Q76N69", "ACTB_CERAE"))
        self.assertEqual(record.dr_positive[143], ("P60706", "ACTB_CHICK"))
        self.assertEqual(record.dr_positive[144], ("P48975", "ACTB_CRIGR"))
        self.assertEqual(record.dr_positive[145], ("P83751", "ACTB_CTEID"))
        self.assertEqual(record.dr_positive[146], ("P83750", "ACTB_CYPCA"))
        self.assertEqual(record.dr_positive[147], ("P60708", "ACTB_HORSE"))
        self.assertEqual(record.dr_positive[148], ("P60709", "ACTB_HUMAN"))
        self.assertEqual(record.dr_positive[149], ("Q711N9", "ACTB_MESAU"))
        self.assertEqual(len(record.dr_positive), 295)
        self.assertEqual(record.dr_positive[150], ("P60710", "ACTB_MOUSE"))
        self.assertEqual(record.dr_positive[151], ("P68143", "ACTB_OREMO"))
        self.assertEqual(record.dr_positive[152], ("P79818", "ACTB_ORYLA"))
        self.assertEqual(record.dr_positive[153], ("Q5R1X3", "ACTB_PANTR"))
        self.assertEqual(record.dr_positive[154], ("P29751", "ACTB_RABIT"))
        self.assertEqual(record.dr_positive[155], ("P60711", "ACTB_RAT"))
        self.assertEqual(record.dr_positive[156], ("O42161", "ACTB_SALSA"))
        self.assertEqual(record.dr_positive[157], ("P60713", "ACTB_SHEEP"))
        self.assertEqual(record.dr_positive[158], ("Q91ZK5", "ACTB_SIGHI"))
        self.assertEqual(record.dr_positive[159], ("P53473", "ACTB_STRPU"))
        self.assertEqual(record.dr_positive[160], ("P60707", "ACTB_TRIVU"))
        self.assertEqual(record.dr_positive[161], ("P15475", "ACTB_XENBO"))
        self.assertEqual(record.dr_positive[162], ("O93400", "ACTB_XENLA"))
        self.assertEqual(record.dr_positive[163], ("Q964E3", "ACTC_BIOAL"))
        self.assertEqual(record.dr_positive[164], ("P92179", "ACTC_BIOGL"))
        self.assertEqual(record.dr_positive[165], ("Q964E1", "ACTC_BIOOB"))
        self.assertEqual(record.dr_positive[166], ("Q964E2", "ACTC_BIOPF"))
        self.assertEqual(record.dr_positive[167], ("Q964E0", "ACTC_BIOTE"))
        self.assertEqual(record.dr_positive[168], ("Q93129", "ACTC_BRABE"))
        self.assertEqual(record.dr_positive[169], ("Q93131", "ACTC_BRAFL"))
        self.assertEqual(record.dr_positive[170], ("O17503", "ACTC_BRALA"))
        self.assertEqual(record.dr_positive[171], ("P68034", "ACTC_CHICK"))
        self.assertEqual(record.dr_positive[172], ("P53480", "ACTC_FUGRU"))
        self.assertEqual(record.dr_positive[173], ("P53461", "ACTC_HALRO"))
        self.assertEqual(record.dr_positive[174], ("Q964D9", "ACTC_HELTI"))
        self.assertEqual(record.dr_positive[175], ("P68032", "ACTC_HUMAN"))
        self.assertEqual(record.dr_positive[176], ("P68033", "ACTC_MOUSE"))
        self.assertEqual(record.dr_positive[177], ("P12716", "ACTC_PISOC"))
        self.assertEqual(record.dr_positive[178], ("P68035", "ACTC_RAT"))
        self.assertEqual(record.dr_positive[179], ("Q07903", "ACTC_STRPU"))
        self.assertEqual(record.dr_positive[180], ("Q00215", "ACTC_STYPL"))
        self.assertEqual(record.dr_positive[181], ("P69005", "ACTD_STRPU"))
        self.assertEqual(record.dr_positive[182], ("P53474", "ACTE_STRPU"))
        self.assertEqual(record.dr_positive[183], ("P18499", "ACTF_STRPU"))
        self.assertEqual(record.dr_positive[184], ("P63256", "ACTG_ANSAN"))
        self.assertEqual(record.dr_positive[185], ("P63258", "ACTG_BOVIN"))
        self.assertEqual(record.dr_positive[186], ("Q9UVW9", "ACTG_CEPAC"))
        self.assertEqual(record.dr_positive[187], ("P20359", "ACTG_EMENI"))
        self.assertEqual(record.dr_positive[188], ("P63261", "ACTG_HUMAN"))
        self.assertEqual(record.dr_positive[189], ("P63260", "ACTG_MOUSE"))
        self.assertEqual(record.dr_positive[190], ("Q9URS0", "ACTG_PENCH"))
        self.assertEqual(record.dr_positive[191], ("P63259", "ACTG_RAT"))
        self.assertEqual(record.dr_positive[192], ("P63257", "ACTG_TRIVU"))
        self.assertEqual(record.dr_positive[193], ("P63270", "ACTH_CHICK"))
        self.assertEqual(record.dr_positive[194], ("P63267", "ACTH_HUMAN"))
        self.assertEqual(record.dr_positive[195], ("P63268", "ACTH_MOUSE"))
        self.assertEqual(record.dr_positive[196], ("P63269", "ACTH_RAT"))
        self.assertEqual(record.dr_positive[197], ("P17304", "ACTM_APLCA"))
        self.assertEqual(record.dr_positive[198], ("Q93130", "ACTM_BRABE"))
        self.assertEqual(record.dr_positive[199], ("Q93132", "ACTM_BRAFL"))
        self.assertEqual(record.dr_positive[200], ("O17502", "ACTM_BRALA"))
        self.assertEqual(record.dr_positive[201], ("O15998", "ACTM_CIOSA"))
        self.assertEqual(record.dr_positive[202], ("P53463", "ACTM_HELER"))
        self.assertEqual(record.dr_positive[203], ("P53464", "ACTM_HELTB"))
        self.assertEqual(record.dr_positive[204], ("Q25381", "ACTM_LYTPI"))
        self.assertEqual(record.dr_positive[205], ("P53467", "ACTM_MOLOC"))
        self.assertEqual(record.dr_positive[206], ("P12717", "ACTM_PISOC"))
        self.assertEqual(record.dr_positive[207], ("P26198", "ACTM_STYCL"))
        self.assertEqual(record.dr_positive[208], ("Q00214", "ACTM_STYPL"))
        self.assertEqual(record.dr_positive[209], ("P53475", "ACTN_STYCL"))
        self.assertEqual(record.dr_positive[210], ("Q90X97", "ACTS_ATRMM"))
        self.assertEqual(record.dr_positive[211], ("P68138", "ACTS_BOVIN"))
        self.assertEqual(record.dr_positive[212], ("P49055", "ACTS_CARAU"))
        self.assertEqual(record.dr_positive[213], ("P68139", "ACTS_CHICK"))
        self.assertEqual(record.dr_positive[214], ("P53479", "ACTS_CYPCA"))
        self.assertEqual(record.dr_positive[215], ("P68140", "ACTS_FUGRU"))
        self.assertEqual(record.dr_positive[216], ("P68133", "ACTS_HUMAN"))
        self.assertEqual(record.dr_positive[217], ("P68134", "ACTS_MOUSE"))
        self.assertEqual(record.dr_positive[218], ("P68264", "ACTS_OREMO"))
        self.assertEqual(record.dr_positive[219], ("P68137", "ACTS_PIG"))
        self.assertEqual(record.dr_positive[220], ("P10994", "ACTS_PLEWA"))
        self.assertEqual(record.dr_positive[221], ("P68135", "ACTS_RABIT"))
        self.assertEqual(record.dr_positive[222], ("P68136", "ACTS_RAT"))
        self.assertEqual(record.dr_positive[223], ("P53482", "ACTT_FUGRU"))
        self.assertEqual(record.dr_positive[224], ("P53483", "ACTX_FUGRU"))
        self.assertEqual(record.dr_positive[225], ("P42025", "ACTY_HUMAN"))
        self.assertEqual(record.dr_positive[226], ("P41341", "ACTY_LIMPO"))
        self.assertEqual(record.dr_positive[227], ("Q8R5C5", "ACTY_MOUSE"))
        self.assertEqual(record.dr_positive[228], ("P61162", "ACTZ_CANFA"))
        self.assertEqual(record.dr_positive[229], ("P45889", "ACTZ_DROME"))
        self.assertEqual(record.dr_positive[230], ("P61163", "ACTZ_HUMAN"))
        self.assertEqual(record.dr_positive[231], ("P61164", "ACTZ_MOUSE"))
        self.assertEqual(record.dr_positive[232], ("P53491", "ACT_ACECL"))
        self.assertEqual(record.dr_positive[233], ("P26182", "ACT_ACHBI"))
        self.assertEqual(record.dr_positive[234], ("P53455", "ACT_AJECA"))
        self.assertEqual(record.dr_positive[235], ("Q75D00", "ACT_ASHGO"))
        self.assertEqual(record.dr_positive[236], ("O13419", "ACT_BOTCI"))
        self.assertEqual(record.dr_positive[237], ("P90689", "ACT_BRUMA"))
        self.assertEqual(record.dr_positive[238], ("P14235", "ACT_CANAL"))
        self.assertEqual(record.dr_positive[239], ("Q9UVZ8", "ACT_CANDU"))
        self.assertEqual(record.dr_positive[240], ("P60009", "ACT_CANGA"))
        self.assertEqual(record.dr_positive[241], ("P53498", "ACT_CHLRE"))
        self.assertEqual(record.dr_positive[242], ("P53499", "ACT_CHOCR"))
        self.assertEqual(record.dr_positive[243], ("O65315", "ACT_COLSC"))
        self.assertEqual(record.dr_positive[244], ("Q9UVX4", "ACT_COPCI"))
        self.assertEqual(record.dr_positive[245], ("P30161", "ACT_COSCS"))
        self.assertEqual(record.dr_positive[246], ("O17320", "ACT_CRAGI"))
        self.assertEqual(record.dr_positive[247], ("P48465", "ACT_CRYNV"))
        self.assertEqual(record.dr_positive[248], ("P26183", "ACT_CRYPV"))
        self.assertEqual(record.dr_positive[249], ("P53500", "ACT_CYAME"))
        self.assertEqual(record.dr_positive[250], ("Q24733", "ACT_DICVI"))
        self.assertEqual(record.dr_positive[251], ("P11426", "ACT_ENTHI"))
        self.assertEqual(record.dr_positive[252], ("Q8X119", "ACT_EXODE"))
        self.assertEqual(record.dr_positive[253], ("P53502", "ACT_FUCDI"))
        self.assertEqual(record.dr_positive[254], ("Q39758", "ACT_FUCVE"))
        self.assertEqual(record.dr_positive[255], ("Q6TCF2", "ACT_GAEGA"))
        self.assertEqual(record.dr_positive[256], ("O81221", "ACT_GOSHI"))
        self.assertEqual(record.dr_positive[257], ("O00937", "ACT_HISCA"))
        self.assertEqual(record.dr_positive[258], ("P17126", "ACT_HYDAT"))
        self.assertEqual(record.dr_positive[259], ("P17128", "ACT_KLULA"))
        self.assertEqual(record.dr_positive[260], ("P45520", "ACT_LEIMA"))
        self.assertEqual(record.dr_positive[261], ("P91754", "ACT_LUMRU"))
        self.assertEqual(record.dr_positive[262], ("P49871", "ACT_MANSE"))
        self.assertEqual(record.dr_positive[263], ("O16808", "ACT_MAYDE"))
        self.assertEqual(record.dr_positive[264], ("O65316", "ACT_MESVI"))
        self.assertEqual(record.dr_positive[265], ("P78711", "ACT_NEUCR"))
        self.assertEqual(record.dr_positive[266], ("P53689", "ACT_PHARH"))
        self.assertEqual(record.dr_positive[267], ("P13363", "ACT_PHYME"))
        self.assertEqual(record.dr_positive[268], ("O74258", "ACT_PICAN"))
        self.assertEqual(record.dr_positive[269], ("Q9P4D1", "ACT_PICPA"))
        self.assertEqual(record.dr_positive[270], ("P24902", "ACT_PINCO"))
        self.assertEqual(record.dr_positive[271], ("Q26065", "ACT_PLAMG"))
        self.assertEqual(record.dr_positive[272], ("P45521", "ACT_PROCL"))
        self.assertEqual(record.dr_positive[273], ("P50138", "ACT_PUCGR"))
        self.assertEqual(record.dr_positive[274], ("P60011", "ACT_SACBA"))
        self.assertEqual(record.dr_positive[275], ("O65314", "ACT_SCHDU"))
        self.assertEqual(record.dr_positive[276], ("P10989", "ACT_SCHPO"))
        self.assertEqual(record.dr_positive[277], ("P68555", "ACT_TAESO"))
        self.assertEqual(record.dr_positive[278], ("P10365", "ACT_THELA"))
        self.assertEqual(record.dr_positive[279], ("P53476", "ACT_TOXGO"))
        self.assertEqual(record.dr_positive[280], ("Q99023", "ACT_TRIRE"))
        self.assertEqual(record.dr_positive[281], ("P53477", "ACT_TRYCR"))
        self.assertEqual(record.dr_positive[282], ("P20904", "ACT_VOLCA"))
        self.assertEqual(record.dr_positive[283], ("Q9UVF3", "ACT_YARLI"))
        self.assertEqual(record.dr_positive[284], ("P60010", "ACT_YEAST"))
        self.assertEqual(record.dr_positive[285], ("Q61JZ2", "ARP2_CAEBR"))
        self.assertEqual(record.dr_positive[286], ("P53489", "ARP2_CAEEL"))
        self.assertEqual(record.dr_positive[287], ("Q4WHA3", "ARP4_ASPFU"))
        self.assertEqual(record.dr_positive[288], ("Q5AC48", "ARP4_CANAL"))
        self.assertEqual(record.dr_positive[289], ("Q6BXN0", "ARP4_DEBHA"))
        self.assertEqual(record.dr_positive[290], ("Q4IPI4", "ARP4_GIBZE"))
        self.assertEqual(record.dr_positive[291], ("Q7SHR0", "ARP4_NEUCR"))
        self.assertEqual(record.dr_positive[292], ("Q9P7X7", "ARP4_SCHPO"))
        self.assertEqual(record.dr_positive[293], ("Q4P2E8", "ARP4_USTMA"))
        self.assertEqual(record.dr_positive[294], ("Q09849", "YAE9_SCHPO"))

    def read4_false_etc(self, record):
        self.assertEqual(len(record.dr_false_neg), 22)
        self.assertEqual(record.dr_false_neg[0], ("P30169", "ACT7_SOLTU"))
        self.assertEqual(record.dr_false_neg[1], ("P24263", "ACTD_PHYPO"))
        self.assertEqual(record.dr_false_neg[2], ("P12431", "ACTM_STRPU"))
        self.assertEqual(record.dr_false_neg[3], ("P20360", "ACT_EUPCR"))
        self.assertEqual(record.dr_false_neg[4], ("P51775", "ACT_GIALA"))
        self.assertEqual(record.dr_false_neg[5], ("Q754G5", "ARP4_ASHGO"))
        self.assertEqual(record.dr_false_neg[6], ("Q6FJV8", "ARP4_CANGA"))
        self.assertEqual(record.dr_false_neg[7], ("Q5AW89", "ARP4_EMENI"))
        self.assertEqual(record.dr_false_neg[8], ("Q6CSB9", "ARP4_KLULA"))
        self.assertEqual(record.dr_false_neg[9], ("Q6C061", "ARP4_YARLI"))
        self.assertEqual(record.dr_false_neg[10], ("P80428", "ARP4_YEAST"))
        self.assertEqual(record.dr_false_neg[11], ("Q74ZV8", "ARP6_ASHGO"))
        self.assertEqual(record.dr_false_neg[12], ("Q4W9M3", "ARP6_ASPFU"))
        self.assertEqual(record.dr_false_neg[13], ("Q5AP59", "ARP6_CANAL"))
        self.assertEqual(record.dr_false_neg[14], ("Q6FKE7", "ARP6_CANGA"))
        self.assertEqual(record.dr_false_neg[15], ("Q5KAQ4", "ARP6_CRYNE"))
        self.assertEqual(record.dr_false_neg[16], ("Q6BML9", "ARP6_DEBHA"))
        self.assertEqual(record.dr_false_neg[17], ("Q5AXH1", "ARP6_EMENI"))
        self.assertEqual(record.dr_false_neg[18], ("Q6CJF4", "ARP6_KLULA"))
        self.assertEqual(record.dr_false_neg[19], ("Q7S6X6", "ARP6_NEUCR"))
        self.assertEqual(record.dr_false_neg[20], ("O94241", "ARP6_SCHPO"))
        self.assertEqual(record.dr_false_neg[21], ("Q6C982", "ARP6_YARLI"))
        self.assertEqual(len(record.dr_false_pos), 0)
        self.assertEqual(len(record.dr_potential), 26)
        self.assertEqual(record.dr_potential[0], ("P30170", "ACT10_SOLTU"))
        self.assertEqual(record.dr_potential[1], ("P10982", "ACT1_ABSGL"))
        self.assertEqual(record.dr_potential[2], ("Q96482", "ACT1_LYCES"))
        self.assertEqual(record.dr_potential[3], ("P93587", "ACT1_SOLTU"))
        self.assertEqual(record.dr_potential[4], ("Q96483", "ACT2_LYCES"))
        self.assertEqual(record.dr_potential[5], ("P93586", "ACT2_SOLTU"))
        self.assertEqual(record.dr_potential[6], ("P93374", "ACT2_TOBAC"))
        self.assertEqual(record.dr_potential[7], ("Q96484", "ACT3_LYCES"))
        self.assertEqual(record.dr_potential[8], ("P93373", "ACT3_TOBAC"))
        self.assertEqual(record.dr_potential[9], ("Q96481", "ACT4_LYCES"))
        self.assertEqual(record.dr_potential[10], ("P93585", "ACT4_SOLTU"))
        self.assertEqual(record.dr_potential[11], ("P93372", "ACT4_TOBAC"))
        self.assertEqual(record.dr_potential[12], ("P81228", "ACT5_SOLTU"))
        self.assertEqual(record.dr_potential[13], ("P93371", "ACT5_TOBAC"))
        self.assertEqual(record.dr_potential[14], ("P15986", "ACT6_SOYBN"))
        self.assertEqual(record.dr_potential[15], ("P93376", "ACT6_TOBAC"))
        self.assertEqual(record.dr_potential[16], ("P15987", "ACT7_SOYBN"))
        self.assertEqual(record.dr_potential[17], ("P93375", "ACT7_TOBAC"))
        self.assertEqual(record.dr_potential[18], ("P81229", "ACT8_SOLTU"))
        self.assertEqual(record.dr_potential[19], ("P93584", "ACT9_SOLTU"))
        self.assertEqual(record.dr_potential[20], ("Q92192", "ACT_CALFI"))
        self.assertEqual(record.dr_potential[21], ("P80709", "ACT_CARMA"))
        self.assertEqual(record.dr_potential[22], ("Q39596", "ACT_CHERU"))
        self.assertEqual(record.dr_potential[23], ("Q92193", "ACT_CRAVI"))
        self.assertEqual(record.dr_potential[24], ("P81085", "ACT_PINPS"))
        self.assertEqual(record.dr_potential[25], ("Q11212", "ACT_SPOLI"))
        self.assertEqual(len(record.dr_unknown), 0)
        self.assertEqual(len(record.pdb_structs), 60)
        self.assertEqual(record.pdb_structs[0], "1ALM")
        self.assertEqual(record.pdb_structs[1], "1ATN")
        self.assertEqual(record.pdb_structs[2], "1C0F")
        self.assertEqual(record.pdb_structs[3], "1C0G")
        self.assertEqual(record.pdb_structs[4], "1D4X")
        self.assertEqual(record.pdb_structs[5], "1DEJ")
        self.assertEqual(record.pdb_structs[6], "1EQY")
        self.assertEqual(record.pdb_structs[7], "1ESV")
        self.assertEqual(record.pdb_structs[8], "1H1V")
        self.assertEqual(record.pdb_structs[9], "1HLU")
        self.assertEqual(record.pdb_structs[10], "1IJJ")
        self.assertEqual(record.pdb_structs[11], "1J6Z")
        self.assertEqual(record.pdb_structs[12], "1KXP")
        self.assertEqual(record.pdb_structs[13], "1LCU")
        self.assertEqual(record.pdb_structs[14], "1LOT")
        self.assertEqual(record.pdb_structs[15], "1LVT")
        self.assertEqual(record.pdb_structs[16], "1M8Q")
        self.assertEqual(record.pdb_structs[17], "1M8Q0")
        self.assertEqual(record.pdb_structs[18], "1MA9")
        self.assertEqual(record.pdb_structs[19], "1MDU")
        self.assertEqual(record.pdb_structs[20], "1MVW")
        self.assertEqual(record.pdb_structs[21], "1N58")
        self.assertEqual(record.pdb_structs[22], "1NLV")
        self.assertEqual(record.pdb_structs[23], "1NM1")
        self.assertEqual(record.pdb_structs[24], "1NMD")
        self.assertEqual(record.pdb_structs[25], "1NWK")
        self.assertEqual(record.pdb_structs[26], "1O18")
        self.assertEqual(record.pdb_structs[27], "1O19")
        self.assertEqual(record.pdb_structs[28], "1O1A")
        self.assertEqual(record.pdb_structs[29], "1O1B")
        self.assertEqual(record.pdb_structs[30], "1O1B0")
        self.assertEqual(record.pdb_structs[31], "1O1C")
        self.assertEqual(record.pdb_structs[32], "1O1C0")
        self.assertEqual(record.pdb_structs[33], "1O1D")
        self.assertEqual(record.pdb_structs[34], "1O1D0")
        self.assertEqual(record.pdb_structs[35], "1O1E")
        self.assertEqual(record.pdb_structs[36], "1O1F")
        self.assertEqual(record.pdb_structs[37], "1O1F0")
        self.assertEqual(record.pdb_structs[38], "1O1G")
        self.assertEqual(record.pdb_structs[39], "1P8Z")
        self.assertEqual(record.pdb_structs[40], "1QZ5")
        self.assertEqual(record.pdb_structs[41], "1QZ6")
        self.assertEqual(record.pdb_structs[42], "1RDW")
        self.assertEqual(record.pdb_structs[43], "1RFQ")
        self.assertEqual(record.pdb_structs[44], "1RGI")
        self.assertEqual(record.pdb_structs[45], "1S22")
        self.assertEqual(record.pdb_structs[46], "1SQK")
        self.assertEqual(record.pdb_structs[47], "1T44")
        self.assertEqual(record.pdb_structs[48], "1Y64")
        self.assertEqual(record.pdb_structs[49], "1YAG")
        self.assertEqual(record.pdb_structs[50], "1YVN")
        self.assertEqual(record.pdb_structs[51], "2A3Z")
        self.assertEqual(record.pdb_structs[52], "2A40")
        self.assertEqual(record.pdb_structs[53], "2A41")
        self.assertEqual(record.pdb_structs[54], "2A42")
        self.assertEqual(record.pdb_structs[55], "2A5X")
        self.assertEqual(record.pdb_structs[56], "2ASM")
        self.assertEqual(record.pdb_structs[57], "2ASO")
        self.assertEqual(record.pdb_structs[58], "2ASP")
        self.assertEqual(record.pdb_structs[59], "2BTF")

    def test_read5(self):
        """Parsing Prosite record ps00488.txt."""
        filename = os.path.join("Prosite", "ps00488.txt")
        with open(filename) as handle:
            record = Prosite.read(handle)
        self.assertEqual(record.name, "PAL_HISTIDASE")
        self.assertEqual(record.type, "PATTERN")
        self.assertEqual(record.accession, "PS00488")
        self.assertEqual(record.created, "MAY-1991")
        self.assertEqual(record.data_update, "DEC-2004")
        self.assertEqual(record.info_update, "MAR-2006")
        self.assertEqual(record.pdoc, "PDOC00424")
        self.assertEqual(record.description, "Phenylalanine and histidine ammonia-lyases signature.")
        self.assertEqual(record.pattern, "[GS]-[STG]-[LIVM]-[STG]-[SAC]-S-G-[DH]-L-x-P-L-[SA]-x(2,3)-[SAGVT].")
        self.assertEqual(record.matrix, [])
        self.assertEqual(record.rules, [])
        self.assertEqual(record.nr_sp_release, "49.3")
        self.assertEqual(record.nr_sp_seqs, 212425)
        self.assertEqual(record.cc_taxo_range, "A?EP?")
        self.assertEqual(record.cc_max_repeat, "1")
        self.assertEqual(len(record.cc_site), 1)
        self.assertEqual(record.cc_site[0], (6, "active_site"))
        self.assertEqual(len(record.dr_positive), 127)
        self.assertEqual(record.dr_positive[0], ("Q8RFC2", "HUTH1_FUSNN"))
        self.assertEqual(record.dr_positive[1], ("Q8RDU4", "HUTH2_FUSNN"))
        self.assertEqual(record.dr_positive[2], ("Q9KWE4", "HUTH_AGRRH"))
        self.assertEqual(record.dr_positive[3], ("Q8U8Z7", "HUTH_AGRT5"))
        self.assertEqual(record.dr_positive[4], ("Q5NZX8", "HUTH_AZOSE"))
        self.assertEqual(record.dr_positive[5], ("Q81Y45", "HUTH_BACAN"))
        self.assertEqual(record.dr_positive[6], ("Q733H8", "HUTH_BACC1"))
        self.assertEqual(record.dr_positive[7], ("Q81AC6", "HUTH_BACCR"))
        self.assertEqual(record.dr_positive[8], ("Q9KBE6", "HUTH_BACHD"))
        self.assertEqual(record.dr_positive[9], ("Q6HFE9", "HUTH_BACHK"))
        self.assertEqual(record.dr_positive[10], ("Q5WAZ6", "HUTH_BACSK"))
        self.assertEqual(record.dr_positive[11], ("P10944", "HUTH_BACSU"))
        self.assertEqual(record.dr_positive[12], ("Q6G3U8", "HUTH_BARHE"))
        self.assertEqual(record.dr_positive[13], ("Q6FZP9", "HUTH_BARQU"))
        self.assertEqual(record.dr_positive[14], ("Q89GV3", "HUTH_BRAJA"))
        self.assertEqual(record.dr_positive[15], ("Q579E8", "HUTH_BRUAB"))
        self.assertEqual(record.dr_positive[16], ("Q8FVB4", "HUTH_BRUSU"))
        self.assertEqual(record.dr_positive[17], ("Q62LJ6", "HUTH_BURMA"))
        self.assertEqual(record.dr_positive[18], ("Q63SH6", "HUTH_BURPS"))
        self.assertEqual(record.dr_positive[19], ("Q20502", "HUTH_CAEEL"))
        self.assertEqual(record.dr_positive[20], ("P58082", "HUTH_CAUCR"))
        self.assertEqual(record.dr_positive[21], ("Q7P188", "HUTH_CHRVO"))
        self.assertEqual(record.dr_positive[22], ("Q891Q1", "HUTH_CLOTE"))
        self.assertEqual(record.dr_positive[23], ("Q9RZ06", "HUTH_DEIRA"))
        self.assertEqual(record.dr_positive[24], ("Q6AKP3", "HUTH_DESPS"))
        self.assertEqual(record.dr_positive[25], ("Q5L310", "HUTH_GEOKA"))
        self.assertEqual(record.dr_positive[26], ("Q7NCB3", "HUTH_GLOVI"))
        self.assertEqual(record.dr_positive[27], ("Q5FRR8", "HUTH_GLUOX"))
        self.assertEqual(record.dr_positive[28], ("Q9HQD5", "HUTH_HALSA"))
        self.assertEqual(record.dr_positive[29], ("P42357", "HUTH_HUMAN"))
        self.assertEqual(record.dr_positive[30], ("Q5QV30", "HUTH_IDILO"))
        self.assertEqual(record.dr_positive[31], ("Q5X5I5", "HUTH_LEGPA"))
        self.assertEqual(record.dr_positive[32], ("Q5ZVR0", "HUTH_LEGPH"))
        self.assertEqual(record.dr_positive[33], ("Q5WWW8", "HUTH_LEGPL"))
        self.assertEqual(record.dr_positive[34], ("P35492", "HUTH_MOUSE"))
        self.assertEqual(record.dr_positive[35], ("Q7N296", "HUTH_PHOLL"))
        self.assertEqual(record.dr_positive[36], ("Q6LQ56", "HUTH_PHOPR"))
        self.assertEqual(record.dr_positive[37], ("Q9HU85", "HUTH_PSEAE"))
        self.assertEqual(record.dr_positive[38], ("Q8VMR3", "HUTH_PSEFL"))
        self.assertEqual(record.dr_positive[39], ("Q88CZ7", "HUTH_PSEPK"))
        self.assertEqual(record.dr_positive[40], ("P21310", "HUTH_PSEPU"))
        self.assertEqual(record.dr_positive[41], ("Q87UM1", "HUTH_PSESM"))
        self.assertEqual(record.dr_positive[42], ("Q8XW29", "HUTH_RALSO"))
        self.assertEqual(record.dr_positive[43], ("P21213", "HUTH_RAT"))
        self.assertEqual(record.dr_positive[44], ("Q983I0", "HUTH_RHILO"))
        self.assertEqual(record.dr_positive[45], ("O31197", "HUTH_RHIME"))
        self.assertEqual(record.dr_positive[46], ("Q57RG6", "HUTH_SALCH"))
        self.assertEqual(record.dr_positive[47], ("Q5PG61", "HUTH_SALPA"))
        self.assertEqual(record.dr_positive[48], ("Q8Z896", "HUTH_SALTI"))
        self.assertEqual(record.dr_positive[49], ("Q8ZQQ9", "HUTH_SALTY"))
        self.assertEqual(record.dr_positive[50], ("Q5LRD8", "HUTH_SILPO"))
        self.assertEqual(record.dr_positive[51], ("Q5HJY8", "HUTH_STAAC"))
        self.assertEqual(record.dr_positive[52], ("P64415", "HUTH_STAAM"))
        self.assertEqual(record.dr_positive[53], ("P64416", "HUTH_STAAN"))
        self.assertEqual(record.dr_positive[54], ("Q6GKT7", "HUTH_STAAR"))
        self.assertEqual(record.dr_positive[55], ("Q6GD82", "HUTH_STAAS"))
        self.assertEqual(record.dr_positive[56], ("Q8NYY3", "HUTH_STAAW"))
        self.assertEqual(record.dr_positive[57], ("Q93TX3", "HUTH_STIAU"))
        self.assertEqual(record.dr_positive[58], ("Q82I33", "HUTH_STRAW"))
        self.assertEqual(record.dr_positive[59], ("Q9EWW1", "HUTH_STRCO"))
        self.assertEqual(record.dr_positive[60], ("P24221", "HUTH_STRGR"))
        self.assertEqual(record.dr_positive[61], ("P58083", "HUTH_STRP1"))
        self.assertEqual(record.dr_positive[62], ("Q8K5L5", "HUTH_STRP3"))
        self.assertEqual(record.dr_positive[63], ("Q5X9K4", "HUTH_STRP6"))
        self.assertEqual(record.dr_positive[64], ("Q8NZ46", "HUTH_STRP8"))
        self.assertEqual(record.dr_positive[65], ("Q67JH4", "HUTH_SYMTH"))
        self.assertEqual(record.dr_positive[66], ("Q9HLI6", "HUTH_THEAC"))
        self.assertEqual(record.dr_positive[67], ("Q8RBH4", "HUTH_THETN"))
        self.assertEqual(record.dr_positive[68], ("Q978N8", "HUTH_THEVO"))
        self.assertEqual(record.dr_positive[69], ("Q9KSQ4", "HUTH_VIBCH"))
        self.assertEqual(record.dr_positive[70], ("Q5E0C6", "HUTH_VIBF1"))
        self.assertEqual(record.dr_positive[71], ("Q87Q77", "HUTH_VIBPA"))
        self.assertEqual(record.dr_positive[72], ("Q8DA21", "HUTH_VIBVU"))
        self.assertEqual(record.dr_positive[73], ("Q7MK58", "HUTH_VIBVY"))
        self.assertEqual(record.dr_positive[74], ("Q8PLZ8", "HUTH_XANAC"))
        self.assertEqual(record.dr_positive[75], ("Q8PAA7", "HUTH_XANCP"))
        self.assertEqual(record.dr_positive[76], ("Q8ZA10", "HUTH_YERPE"))
        self.assertEqual(record.dr_positive[77], ("Q664B8", "HUTH_YERPS"))
        self.assertEqual(record.dr_positive[78], ("Q8YD09", "HUTIH_BRUME"))
        self.assertEqual(record.dr_positive[79], ("P35510", "PAL1_ARATH"))
        self.assertEqual(record.dr_positive[80], ("O23865", "PAL1_DAUCA"))
        self.assertEqual(record.dr_positive[81], ("P14166", "PAL1_IPOBA"))
        self.assertEqual(record.dr_positive[82], ("O49835", "PAL1_LITER"))
        self.assertEqual(record.dr_positive[83], ("P35511", "PAL1_LYCES"))
        self.assertEqual(record.dr_positive[84], ("P14717", "PAL1_ORYSA"))
        self.assertEqual(record.dr_positive[85], ("Q01861", "PAL1_PEA"))
        self.assertEqual(record.dr_positive[86], ("P24481", "PAL1_PETCR"))
        self.assertEqual(record.dr_positive[87], ("P45731", "PAL1_POPKI"))
        self.assertEqual(record.dr_positive[88], ("O64963", "PAL1_PRUAV"))
        self.assertEqual(record.dr_positive[89], ("Q9M568", "PAL1_RUBID"))
        self.assertEqual(record.dr_positive[90], ("P31425", "PAL1_SOLTU"))
        self.assertEqual(record.dr_positive[91], ("P27991", "PAL1_SOYBN"))
        self.assertEqual(record.dr_positive[92], ("P25872", "PAL1_TOBAC"))
        self.assertEqual(record.dr_positive[93], ("P45724", "PAL2_ARATH"))
        self.assertEqual(record.dr_positive[94], ("Q9SMK9", "PAL2_CICAR"))
        self.assertEqual(record.dr_positive[95], ("Q42858", "PAL2_IPOBA"))
        self.assertEqual(record.dr_positive[96], ("O49836", "PAL2_LITER"))
        self.assertEqual(record.dr_positive[97], ("P53443", "PAL2_ORYSA"))
        self.assertEqual(record.dr_positive[98], ("Q04593", "PAL2_PEA"))
        self.assertEqual(record.dr_positive[99], ("P45728", "PAL2_PETCR"))
        self.assertEqual(record.dr_positive[100], ("P19142", "PAL2_PHAVU"))
        self.assertEqual(record.dr_positive[101], ("Q43052", "PAL2_POPKI"))
        self.assertEqual(record.dr_positive[102], ("P31426", "PAL2_SOLTU"))
        self.assertEqual(record.dr_positive[103], ("P35513", "PAL2_TOBAC"))
        self.assertEqual(record.dr_positive[104], ("P45725", "PAL3_ARATH"))
        self.assertEqual(record.dr_positive[105], ("P45729", "PAL3_PETCR"))
        self.assertEqual(record.dr_positive[106], ("P19143", "PAL3_PHAVU"))
        self.assertEqual(record.dr_positive[107], ("P45733", "PAL3_TOBAC"))
        self.assertEqual(record.dr_positive[108], ("Q9SS45", "PAL4_ARATH"))
        self.assertEqual(record.dr_positive[109], ("Q40910", "PAL4_POPKI"))
        self.assertEqual(record.dr_positive[110], ("P26600", "PAL5_LYCES"))
        self.assertEqual(record.dr_positive[111], ("O93967", "PALY_AMAMU"))
        self.assertEqual(record.dr_positive[112], ("Q42609", "PALY_BROFI"))
        self.assertEqual(record.dr_positive[113], ("P45726", "PALY_CAMSI"))
        self.assertEqual(record.dr_positive[114], ("Q42667", "PALY_CITLI"))
        self.assertEqual(record.dr_positive[115], ("O23924", "PALY_DIGLA"))
        self.assertEqual(record.dr_positive[116], ("O04058", "PALY_HELAN"))
        self.assertEqual(record.dr_positive[117], ("P27990", "PALY_MEDSA"))
        self.assertEqual(record.dr_positive[118], ("P45727", "PALY_PERAE"))
        self.assertEqual(record.dr_positive[119], ("P52777", "PALY_PINTA"))
        self.assertEqual(record.dr_positive[120], ("P45730", "PALY_POPTR"))
        self.assertEqual(record.dr_positive[121], ("P10248", "PALY_RHORB"))
        self.assertEqual(record.dr_positive[122], ("P11544", "PALY_RHOTO"))
        self.assertEqual(record.dr_positive[123], ("P45732", "PALY_STYHU"))
        self.assertEqual(record.dr_positive[124], ("P45734", "PALY_TRISU"))
        self.assertEqual(record.dr_positive[125], ("Q96V77", "PALY_USTMA"))
        self.assertEqual(record.dr_positive[126], ("Q43210", "PALY_WHEAT"))
        self.assertEqual(len(record.dr_false_neg), 2)
        self.assertEqual(record.dr_false_neg[0], ("Q8EKJ4", "HUTH_SHEON"))
        self.assertEqual(record.dr_false_neg[1], ("Q73Q56", "HUTH_TREDE"))
        self.assertEqual(len(record.dr_false_pos), 0)
        self.assertEqual(len(record.dr_potential), 4)
        self.assertEqual(record.dr_potential[0], ("P07218", "PAL1_PHAVU"))
        self.assertEqual(record.dr_potential[1], ("Q92195", "PALY_AGABI"))
        self.assertEqual(record.dr_potential[2], ("P35512", "PALY_MALDO"))
        self.assertEqual(record.dr_potential[3], ("P45735", "PALY_VITVI"))
        self.assertEqual(len(record.dr_unknown), 0)
        self.assertEqual(record.pdb_structs[0], "1B8F")
        self.assertEqual(record.pdb_structs[1], "1EB4")
        self.assertEqual(record.pdb_structs[2], "1GK2")
        self.assertEqual(record.pdb_structs[3], "1GKJ")
        self.assertEqual(record.pdb_structs[4], "1GKM")
        self.assertEqual(record.pdb_structs[5], "1Y2M")

    def test_read6(self):
        """Parsing Prosite record ps00546.txt."""
        filename = os.path.join("Prosite", "ps00546.txt")
        with open(filename) as handle:
            record = Prosite.read(handle)
        self.assertEqual(record.name, "CYSTEINE_SWITCH")
        self.assertEqual(record.type, "PATTERN")
        self.assertEqual(record.accession, "PS00546")
        self.assertEqual(record.created, "DEC-1991")
        self.assertEqual(record.data_update, "NOV-1997")
        self.assertEqual(record.info_update, "MAR-2006")
        self.assertEqual(record.pdoc, "PDOC00472")
        self.assertEqual(record.description, "Matrixins cysteine switch.")
        self.assertEqual(record.pattern, "P-R-C-[GN]-x-P-[DR]-[LIVSAPKQ].")
        self.assertEqual(record.matrix, [])
        self.assertEqual(record.rules, [])
        self.assertEqual(record.nr_sp_release, "49.3")
        self.assertEqual(record.nr_sp_seqs, 212425)
        self.assertEqual(record.cc_taxo_range, "??E??")
        self.assertEqual(record.cc_max_repeat, "1")
        self.assertEqual(len(record.cc_site), 1)
        self.assertEqual(record.cc_site[0], (3, "chelates_zinc"))
        self.assertEqual(len(record.dr_positive), 62)
        self.assertEqual(record.dr_positive[0], ("P31178", "GLE_CHLRE"))
        self.assertEqual(record.dr_positive[1], ("P91953", "HE_HEMPU"))
        self.assertEqual(record.dr_positive[2], ("P22757", "HE_PARLI"))
        self.assertEqual(record.dr_positive[3], ("P29136", "MEP1_SOYBN"))
        self.assertEqual(record.dr_positive[4], ("P09238", "MMP10_HUMAN"))
        self.assertEqual(record.dr_positive[5], ("O55123", "MMP10_MOUSE"))
        self.assertEqual(record.dr_positive[6], ("P07152", "MMP10_RAT"))
        self.assertEqual(record.dr_positive[7], ("P24347", "MMP11_HUMAN"))
        self.assertEqual(record.dr_positive[8], ("Q11005", "MMP11_XENLA"))
        self.assertEqual(record.dr_positive[9], ("P39900", "MMP12_HUMAN"))
        self.assertEqual(record.dr_positive[10], ("P34960", "MMP12_MOUSE"))
        self.assertEqual(record.dr_positive[11], ("P79227", "MMP12_RABIT"))
        self.assertEqual(record.dr_positive[12], ("O77656", "MMP13_BOVIN"))
        self.assertEqual(record.dr_positive[13], ("O18927", "MMP13_HORSE"))
        self.assertEqual(record.dr_positive[14], ("P45452", "MMP13_HUMAN"))
        self.assertEqual(record.dr_positive[15], ("P33435", "MMP13_MOUSE"))
        self.assertEqual(record.dr_positive[16], ("O62806", "MMP13_RABIT"))
        self.assertEqual(record.dr_positive[17], ("P23097", "MMP13_RAT"))
        self.assertEqual(record.dr_positive[18], ("Q10835", "MMP13_XENLA"))
        self.assertEqual(record.dr_positive[19], ("P50281", "MMP14_HUMAN"))
        self.assertEqual(record.dr_positive[20], ("P53690", "MMP14_MOUSE"))
        self.assertEqual(record.dr_positive[21], ("Q9XT90", "MMP14_PIG"))
        self.assertEqual(record.dr_positive[22], ("Q95220", "MMP14_RABIT"))
        self.assertEqual(record.dr_positive[23], ("Q10739", "MMP14_RAT"))
        self.assertEqual(record.dr_positive[24], ("P51511", "MMP15_HUMAN"))
        self.assertEqual(record.dr_positive[25], ("O54732", "MMP15_MOUSE"))
        self.assertEqual(record.dr_positive[26], ("P51512", "MMP16_HUMAN"))
        self.assertEqual(record.dr_positive[27], ("Q9WTR0", "MMP16_MOUSE"))
        self.assertEqual(record.dr_positive[28], ("O35548", "MMP16_RAT"))
        self.assertEqual(record.dr_positive[29], ("Q9EPL5", "MMP1A_MOUSE"))
        self.assertEqual(record.dr_positive[30], ("Q9EPL6", "MMP1B_MOUSE"))
        self.assertEqual(record.dr_positive[31], ("P28053", "MMP1_BOVIN"))
        self.assertEqual(record.dr_positive[32], ("Q9XSZ5", "MMP1_HORSE"))
        self.assertEqual(record.dr_positive[33], ("P03956", "MMP1_HUMAN"))
        self.assertEqual(record.dr_positive[34], ("P21692", "MMP1_PIG"))
        self.assertEqual(record.dr_positive[35], ("P13943", "MMP1_RABIT"))
        self.assertEqual(record.dr_positive[36], ("O18767", "MMP20_BOVIN"))
        self.assertEqual(record.dr_positive[37], ("O60882", "MMP20_HUMAN"))
        self.assertEqual(record.dr_positive[38], ("P57748", "MMP20_MOUSE"))
        self.assertEqual(record.dr_positive[39], ("P79287", "MMP20_PIG"))
        self.assertEqual(record.dr_positive[40], ("Q90611", "MMP2_CHICK"))
        self.assertEqual(record.dr_positive[41], ("P08253", "MMP2_HUMAN"))
        self.assertEqual(record.dr_positive[42], ("P33434", "MMP2_MOUSE"))
        self.assertEqual(record.dr_positive[43], ("P50757", "MMP2_RABIT"))
        self.assertEqual(record.dr_positive[44], ("P33436", "MMP2_RAT"))
        self.assertEqual(record.dr_positive[45], ("Q6Y4Q5", "MMP3_CANFA"))
        self.assertEqual(record.dr_positive[46], ("P08254", "MMP3_HUMAN"))
        self.assertEqual(record.dr_positive[47], ("P28862", "MMP3_MOUSE"))
        self.assertEqual(record.dr_positive[48], ("P28863", "MMP3_RABIT"))
        self.assertEqual(record.dr_positive[49], ("P03957", "MMP3_RAT"))
        self.assertEqual(record.dr_positive[50], ("P09237", "MMP7_HUMAN"))
        self.assertEqual(record.dr_positive[51], ("Q10738", "MMP7_MOUSE"))
        self.assertEqual(record.dr_positive[52], ("P50280", "MMP7_RAT"))
        self.assertEqual(record.dr_positive[53], ("P22894", "MMP8_HUMAN"))
        self.assertEqual(record.dr_positive[54], ("O70138", "MMP8_MOUSE"))
        self.assertEqual(record.dr_positive[55], ("O88766", "MMP8_RAT"))
        self.assertEqual(record.dr_positive[56], ("P52176", "MMP9_BOVIN"))
        self.assertEqual(record.dr_positive[57], ("O18733", "MMP9_CANFA"))
        self.assertEqual(record.dr_positive[58], ("P14780", "MMP9_HUMAN"))
        self.assertEqual(record.dr_positive[59], ("P41245", "MMP9_MOUSE"))
        self.assertEqual(record.dr_positive[60], ("P41246", "MMP9_RABIT"))
        self.assertEqual(record.dr_positive[61], ("P50282", "MMP9_RAT"))
        self.assertEqual(len(record.dr_false_neg), 59)
        self.assertEqual(record.dr_false_neg[0], ("Q9R158", "AD26A_MOUSE"))
        self.assertEqual(record.dr_false_neg[1], ("Q10741", "ADA10_BOVIN"))
        self.assertEqual(record.dr_false_neg[2], ("O14672", "ADA10_HUMAN"))
        self.assertEqual(record.dr_false_neg[3], ("O35598", "ADA10_MOUSE"))
        self.assertEqual(record.dr_false_neg[4], ("Q10743", "ADA10_RAT"))
        self.assertEqual(record.dr_false_neg[5], ("Q8JIY1", "ADA10_XENLA"))
        self.assertEqual(record.dr_false_neg[6], ("Q61824", "ADA12_MOUSE"))
        self.assertEqual(record.dr_false_neg[7], ("Q13444", "ADA15_HUMAN"))
        self.assertEqual(record.dr_false_neg[8], ("O88839", "ADA15_MOUSE"))
        self.assertEqual(record.dr_false_neg[9], ("Q9QYV0", "ADA15_RAT"))
        self.assertEqual(record.dr_false_neg[10], ("P78536", "ADA17_HUMAN"))
        self.assertEqual(record.dr_false_neg[11], ("Q9Z0F8", "ADA17_MOUSE"))
        self.assertEqual(record.dr_false_neg[12], ("Q9Z1K9", "ADA17_RAT"))
        self.assertEqual(record.dr_false_neg[13], ("Q9H013", "ADA19_HUMAN"))
        self.assertEqual(record.dr_false_neg[14], ("O35674", "ADA19_MOUSE"))
        self.assertEqual(record.dr_false_neg[15], ("O43506", "ADA20_HUMAN"))
        self.assertEqual(record.dr_false_neg[16], ("Q9UKJ8", "ADA21_HUMAN"))
        self.assertEqual(record.dr_false_neg[17], ("Q9JI76", "ADA21_MOUSE"))
        self.assertEqual(record.dr_false_neg[18], ("Q9R160", "ADA24_MOUSE"))
        self.assertEqual(record.dr_false_neg[19], ("Q9R159", "ADA25_MOUSE"))
        self.assertEqual(record.dr_false_neg[20], ("Q9UKQ2", "ADA28_HUMAN"))
        self.assertEqual(record.dr_false_neg[21], ("Q9XSL6", "ADA28_MACFA"))
        self.assertEqual(record.dr_false_neg[22], ("Q9JLN6", "ADA28_MOUSE"))
        self.assertEqual(record.dr_false_neg[23], ("Q9UKF2", "ADA30_HUMAN"))
        self.assertEqual(record.dr_false_neg[24], ("Q9BZ11", "ADA33_HUMAN"))
        self.assertEqual(record.dr_false_neg[25], ("Q923W9", "ADA33_MOUSE"))
        self.assertEqual(record.dr_false_neg[26], ("O15204", "ADEC1_HUMAN"))
        self.assertEqual(record.dr_false_neg[27], ("P58397", "ATS12_HUMAN"))
        self.assertEqual(record.dr_false_neg[28], ("Q811B3", "ATS12_MOUSE"))
        self.assertEqual(record.dr_false_neg[29], ("Q8WXS8", "ATS14_HUMAN"))
        self.assertEqual(record.dr_false_neg[30], ("Q8TE58", "ATS15_HUMAN"))
        self.assertEqual(record.dr_false_neg[31], ("P59384", "ATS15_MOUSE"))
        self.assertEqual(record.dr_false_neg[32], ("Q8TE57", "ATS16_HUMAN"))
        self.assertEqual(record.dr_false_neg[33], ("Q69Z28", "ATS16_MOUSE"))
        self.assertEqual(record.dr_false_neg[34], ("Q8TE56", "ATS17_HUMAN"))
        self.assertEqual(record.dr_false_neg[35], ("Q8TE60", "ATS18_HUMAN"))
        self.assertEqual(record.dr_false_neg[36], ("Q4VC17", "ATS18_MOUSE"))
        self.assertEqual(record.dr_false_neg[37], ("Q8TE59", "ATS19_HUMAN"))
        self.assertEqual(record.dr_false_neg[38], ("P59509", "ATS19_MOUSE"))
        self.assertEqual(record.dr_false_neg[39], ("Q9UKP4", "ATS7_HUMAN"))
        self.assertEqual(record.dr_false_neg[40], ("Q9P2N4", "ATS9_HUMAN"))
        self.assertEqual(record.dr_false_neg[41], ("Q02853", "MMP11_MOUSE"))
        self.assertEqual(record.dr_false_neg[42], ("Q63341", "MMP12_RAT"))
        self.assertEqual(record.dr_false_neg[43], ("Q9ULZ9", "MMP17_HUMAN"))
        self.assertEqual(record.dr_false_neg[44], ("Q9R0S3", "MMP17_MOUSE"))
        self.assertEqual(record.dr_false_neg[45], ("O13065", "MMP18_XENLA"))
        self.assertEqual(record.dr_false_neg[46], ("Q99542", "MMP19_HUMAN"))
        self.assertEqual(record.dr_false_neg[47], ("Q9JHI0", "MMP19_MOUSE"))
        self.assertEqual(record.dr_false_neg[48], ("Q11133", "MMP1_RANCA"))
        self.assertEqual(record.dr_false_neg[49], ("Q90YC2", "MMP21_CYNPY"))
        self.assertEqual(record.dr_false_neg[50], ("Q8N119", "MMP21_HUMAN"))
        self.assertEqual(record.dr_false_neg[51], ("Q8K3F2", "MMP21_MOUSE"))
        self.assertEqual(record.dr_false_neg[52], ("O93470", "MMP21_XENLA"))
        self.assertEqual(record.dr_false_neg[53], ("Q9Y5R2", "MMP24_HUMAN"))
        self.assertEqual(record.dr_false_neg[54], ("Q9R0S2", "MMP24_MOUSE"))
        self.assertEqual(record.dr_false_neg[55], ("Q9NRE1", "MMP26_HUMAN"))
        self.assertEqual(record.dr_false_neg[56], ("Q9H239", "MMP28_HUMAN"))
        self.assertEqual(record.dr_false_neg[57], ("Q28397", "MMP3_HORSE"))
        self.assertEqual(record.dr_false_neg[58], ("P55032", "MMP7_FELCA"))
        self.assertEqual(len(record.dr_false_pos), 1)
        self.assertEqual(record.dr_false_pos[0], ("Q9YC75", "RGYR2_AERPE"))
        self.assertEqual(len(record.dr_potential), 0)
        self.assertEqual(len(record.dr_unknown), 0)
        self.assertEqual(len(record.pdb_structs), 8)
        self.assertEqual(record.pdb_structs[0], "1CK7")
        self.assertEqual(record.pdb_structs[1], "1EAK")
        self.assertEqual(record.pdb_structs[2], "1GXD")
        self.assertEqual(record.pdb_structs[3], "1L6J")
        self.assertEqual(record.pdb_structs[4], "1LKG")
        self.assertEqual(record.pdb_structs[5], "1SLM")
        self.assertEqual(record.pdb_structs[6], "1SU3")
        self.assertEqual(record.pdb_structs[7], "1UC1")


if __name__ == "__main__":
    runner = unittest.TextTestRunner(verbosity=2)
    unittest.main(testRunner=runner)
