#!/usr/bin/env python
# File created on 14 Jul 2012
from __future__ import division

__author__ = "Greg Caporaso"
__copyright__ = "Copyright 2011, The QIIME project"
__credits__ = ["Greg Caporaso"]
__license__ = "GPL"
__version__ = "1.8.0"
__maintainer__ = "Greg Caporaso"
__email__ = "gregcaporaso@gmail.com"

from glob import glob
from shutil import rmtree
from os.path import exists, join
from cogent.util.unit_test import TestCase, main
from cogent.util.misc import remove_files, create_dir
from biom.parse import parse_biom_table
from qiime.util import (get_qiime_temp_dir, 
                        get_tmp_filename)
from qiime.parse import parse_distmat
from qiime.test import initiate_timeout, disable_timeout
from qiime.parallel.multiple_rarefactions import ParallelMultipleRarefactions

class ParallelMultipleRarefactionsTests(TestCase):
    
    def setUp(self):
        
        self.files_to_remove = []
        self.dirs_to_remove = []
        
        # Create example output directory
        tmp_dir = get_qiime_temp_dir()
        self.test_out = get_tmp_filename(tmp_dir=tmp_dir,
                                         prefix='qiime_parallel_tests_',
                                         suffix='',
                                         result_constructor=str)
        self.dirs_to_remove.append(self.test_out)
        create_dir(self.test_out)
        
        self.input1_fp = get_tmp_filename(tmp_dir=self.test_out,
                                            prefix='qiime_inseqs',
                                            suffix='.fasta')
        input1_f = open(self.input1_fp,'w')
        input1_f.write(input1)
        input1_f.close()
        self.files_to_remove.append(self.input1_fp)
        
        # Define number of seconds a test can run for before timing out 
        # and failing
        initiate_timeout(60)

    
    def tearDown(self):
        
        disable_timeout()
        remove_files(self.files_to_remove)
        # remove directories last, so we don't get errors
        # trying to remove files which may be in the directories
        for d in self.dirs_to_remove:
            if exists(d):
                rmtree(d)
    
    def test_parallel_mutliple_rarefactions(self):
        """ ParallelMultipleRarefacitons functions as expected """
        r = ParallelMultipleRarefactions()
        params = {'min':1,
                  'max':100,
                  'step':10,
                  'num_reps':2,
                  'jobs_to_start':2,
                  'suppress_lineages_included':False,
                  'subsample_multinomial':False}
        r(self.input1_fp,
          self.test_out,
          params,
          job_prefix='RARIFTEST',
          poll_directly=True,
          suppress_submit_jobs=False)
        biom_tables = glob('%s/*biom' % self.test_out)
        self.assertEqual(len(biom_tables),20)
        biom_tables.sort()
        input_table = parse_biom_table(open(self.input1_fp))
        # sanity checks on first table (sampled at 11 seqs/sample)
        output_table = parse_biom_table(open(biom_tables[0]))
        self.assertEqual(output_table.SampleIds,input_table.SampleIds)
        self.assertEqual(output_table.sum(),99)
        # sanity checks on first table (sampled at 91 seqs/sample)
        output_table = parse_biom_table(open(biom_tables[-1]))
        self.assertEqual(output_table.SampleIds,input_table.SampleIds)
        self.assertEqual(output_table.sum(),819)
        
        
input1 = """{"rows": [{"id": "0", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "1", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Bacillales", "f__Staphylococcaceae"]}}, {"id": "2", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "3", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "4", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "5", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "6", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "7", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "8", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "9", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "10", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "11", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "12", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "13", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Streptococcaceae"]}}, {"id": "14", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "15", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "16", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "17", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "18", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "19", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "20", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "21", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "22", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Deltaproteobacteria", "o__Desulfovibrionales", "f__Desulfovibrionaceae"]}}, {"id": "23", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "24", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "25", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "26", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "27", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__TM7", "c__TM7-3", "o__CW040", "f__F16"]}}, {"id": "28", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "29", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "30", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "31", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "32", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "33", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "34", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "35", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "36", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "37", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "38", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "39", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "40", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "41", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "42", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "43", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "44", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "45", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Deferribacteres", "c__Deferribacteres", "o__Deferribacterales", "f__Deferribacteraceae"]}}, {"id": "46", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "47", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "48", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "49", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "50", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "51", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "52", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "53", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "54", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "55", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "56", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "57", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "58", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "59", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "60", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "61", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "62", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "63", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "64", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "65", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "66", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "67", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "68", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "69", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "70", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "71", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "72", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "73", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "74", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "75", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "76", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "77", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "78", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "79", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "80", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "81", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "82", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "83", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "84", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "85", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "86", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "87", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Mollicutes", "o__RF39", "f__"]}}, {"id": "88", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "89", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "90", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "91", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "92", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "93", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "94", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "95", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "96", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "97", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "98", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "99", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "100", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "101", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "102", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "103", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "104", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "105", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "106", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "107", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "108", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Deferribacteres", "c__Deferribacteres", "o__Deferribacterales", "f__Deferribacteraceae"]}}, {"id": "109", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "110", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "111", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "112", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "113", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "114", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "115", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "116", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "117", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "118", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "119", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "120", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "121", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "122", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "123", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Mollicutes", "o__RF39", "f__"]}}, {"id": "124", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "125", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "126", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "127", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "128", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "129", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "130", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Peptococcaceae"]}}, {"id": "131", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "132", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "133", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "134", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "135", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "136", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "137", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "138", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "139", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "140", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "141", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "142", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "143", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "144", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "145", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "146", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "147", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "148", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "149", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "150", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales"]}}, {"id": "151", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "152", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "153", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "154", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "155", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "156", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "157", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "158", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "159", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "160", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "161", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "162", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "163", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "164", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "165", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "166", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "167", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "168", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "169", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "170", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "171", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "172", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "173", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "174", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "175", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "176", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "177", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "178", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "179", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "180", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "181", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "182", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "183", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "184", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "185", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "186", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "187", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "188", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "189", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "190", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "191", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "192", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "193", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "194", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "195", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "196", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "197", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "198", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "199", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Peptococcaceae"]}}, {"id": "200", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "201", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "202", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "203", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "204", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "205", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "206", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "207", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "208", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "209", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "210", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "211", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "212", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "213", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "214", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "215", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "216", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "217", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "218", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "219", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "220", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "221", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "222", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "223", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "224", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "225", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "226", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "227", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "228", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Peptococcaceae"]}}, {"id": "229", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "230", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "231", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "232", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "233", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "234", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "235", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "236", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "237", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "238", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "239", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "240", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "241", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "242", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "243", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "244", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "245", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "246", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "247", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "248", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "249", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "250", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "251", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "252", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "253", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "254", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "255", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "256", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "257", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "258", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Prevotellaceae"]}}, {"id": "259", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "260", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "261", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "262", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "263", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "264", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "265", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "266", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "267", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "268", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "269", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "270", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "271", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "272", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "273", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "274", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "275", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "276", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "277", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "278", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "279", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "280", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "281", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "282", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Deltaproteobacteria", "o__Desulfovibrionales", "f__Desulfovibrionaceae"]}}, {"id": "283", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "284", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "285", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "286", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "287", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "288", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "289", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "290", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "291", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Bacillales", "f__Staphylococcaceae"]}}, {"id": "292", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "293", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "294", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "295", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "296", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "297", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "298", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "299", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "300", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "301", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "302", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "303", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "304", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "305", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "306", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "307", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "308", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Epsilonproteobacteria", "o__Campylobacterales", "f__Helicobacteraceae"]}}, {"id": "309", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "310", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "311", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "312", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "313", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "314", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "315", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "316", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "317", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "318", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "319", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "320", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "321", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "322", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "323", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "324", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "325", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "326", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "327", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "328", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "329", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "330", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "331", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "332", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "333", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "334", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "335", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "336", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "337", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "338", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "339", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "340", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "341", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "342", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "343", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "344", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "345", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "346", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "347", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "348", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "349", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "350", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "351", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "352", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "353", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "354", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Clostridiales Family XIII. Incertae Sedis"]}}, {"id": "355", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "356", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "357", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "358", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "359", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "360", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Clostridiaceae"]}}, {"id": "361", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "362", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "363", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "364", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "365", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "366", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "367", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "368", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "369", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "370", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "371", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "372", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "373", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "374", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "375", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "376", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "377", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "378", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "379", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "380", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "381", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "382", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "383", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "384", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "385", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "386", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "387", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "388", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "389", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "390", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "391", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "392", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "393", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "394", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "395", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Deltaproteobacteria", "o__Desulfovibrionales", "f__Desulfovibrionaceae"]}}, {"id": "396", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "397", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "398", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "399", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "400", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "401", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "402", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "403", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "404", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "405", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "406", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "407", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "408", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "409", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Turicibacterales", "f__Turicibacteraceae"]}}, {"id": "410", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "411", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "412", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "413", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "414", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "415", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "416", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "417", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}], "format": "Biological Observation Matrix 0.9.1-dev", "data": [[0, 0, 1.0], [1, 0, 1.0], [2, 1, 1.0], [3, 2, 1.0], [4, 1, 1.0], [4, 3, 1.0], [5, 4, 1.0], [6, 0, 1.0], [6, 2, 1.0], [6, 3, 1.0], [6, 5, 1.0], [7, 1, 1.0], [8, 4, 1.0], [9, 4, 1.0], [10, 2, 2.0], [10, 6, 1.0], [11, 3, 1.0], [11, 5, 1.0], [12, 3, 1.0], [13, 7, 1.0], [14, 8, 1.0], [15, 5, 1.0], [16, 8, 1.0], [17, 8, 1.0], [18, 1, 1.0], [18, 2, 1.0], [19, 7, 1.0], [20, 7, 1.0], [21, 3, 2.0], [22, 3, 1.0], [23, 1, 1.0], [24, 1, 1.0], [24, 3, 1.0], [25, 7, 2.0], [26, 8, 1.0], [27, 8, 2.0], [28, 1, 1.0], [28, 3, 1.0], [29, 5, 2.0], [30, 3, 1.0], [30, 5, 1.0], [31, 3, 3.0], [31, 7, 6.0], [32, 1, 3.0], [33, 3, 1.0], [34, 7, 2.0], [35, 7, 1.0], [36, 7, 1.0], [37, 2, 1.0], [37, 5, 1.0], [38, 7, 5.0], [39, 7, 1.0], [40, 1, 1.0], [40, 7, 2.0], [41, 7, 2.0], [42, 2, 16.0], [42, 4, 12.0], [43, 4, 1.0], [44, 1, 1.0], [45, 0, 6.0], [45, 3, 2.0], [45, 7, 3.0], [45, 8, 5.0], [46, 4, 1.0], [47, 1, 1.0], [48, 4, 1.0], [49, 2, 1.0], [50, 8, 1.0], [51, 7, 1.0], [52, 7, 1.0], [53, 0, 37.0], [53, 1, 1.0], [53, 3, 10.0], [53, 8, 4.0], [54, 6, 1.0], [55, 2, 1.0], [56, 0, 5.0], [56, 1, 2.0], [56, 2, 1.0], [56, 3, 4.0], [56, 4, 1.0], [56, 5, 1.0], [56, 6, 3.0], [56, 7, 9.0], [56, 8, 2.0], [57, 7, 1.0], [58, 0, 1.0], [59, 7, 1.0], [60, 0, 1.0], [60, 1, 2.0], [60, 2, 2.0], [60, 3, 1.0], [60, 4, 24.0], [60, 5, 10.0], [60, 6, 1.0], [61, 6, 1.0], [62, 1, 1.0], [63, 1, 1.0], [64, 6, 1.0], [65, 7, 1.0], [66, 7, 1.0], [67, 4, 1.0], [68, 0, 2.0], [68, 3, 3.0], [68, 8, 2.0], [69, 0, 1.0], [69, 3, 4.0], [69, 4, 2.0], [69, 7, 2.0], [69, 8, 1.0], [70, 6, 1.0], [71, 0, 2.0], [71, 3, 2.0], [72, 2, 1.0], [73, 3, 1.0], [74, 5, 1.0], [75, 5, 1.0], [76, 5, 1.0], [77, 6, 1.0], [78, 0, 19.0], [78, 1, 1.0], [78, 3, 2.0], [78, 8, 1.0], [79, 1, 1.0], [80, 1, 1.0], [81, 1, 1.0], [82, 3, 3.0], [82, 8, 1.0], [83, 3, 1.0], [84, 3, 1.0], [85, 4, 1.0], [86, 0, 1.0], [86, 3, 2.0], [86, 4, 1.0], [86, 5, 4.0], [86, 6, 4.0], [86, 7, 2.0], [87, 5, 1.0], [88, 1, 6.0], [88, 2, 1.0], [88, 5, 2.0], [88, 6, 4.0], [89, 2, 2.0], [90, 0, 1.0], [91, 5, 1.0], [92, 2, 1.0], [92, 5, 2.0], [93, 8, 1.0], [94, 1, 1.0], [94, 3, 1.0], [94, 8, 3.0], [95, 3, 1.0], [96, 1, 1.0], [97, 7, 1.0], [98, 7, 1.0], [99, 0, 1.0], [100, 6, 1.0], [101, 6, 1.0], [102, 1, 2.0], [102, 2, 1.0], [102, 4, 4.0], [102, 6, 1.0], [103, 8, 1.0], [104, 7, 1.0], [105, 8, 5.0], [106, 7, 1.0], [107, 4, 1.0], [108, 0, 1.0], [109, 2, 1.0], [110, 1, 1.0], [110, 2, 1.0], [110, 7, 1.0], [111, 1, 1.0], [111, 5, 1.0], [112, 6, 1.0], [113, 5, 1.0], [114, 3, 1.0], [114, 8, 2.0], [115, 2, 1.0], [116, 0, 1.0], [116, 6, 1.0], [117, 6, 1.0], [118, 2, 1.0], [118, 5, 1.0], [118, 6, 3.0], [119, 3, 1.0], [120, 3, 1.0], [120, 5, 1.0], [121, 2, 1.0], [122, 2, 1.0], [123, 7, 1.0], [124, 2, 2.0], [124, 5, 2.0], [124, 6, 2.0], [125, 7, 1.0], [126, 1, 1.0], [127, 4, 1.0], [128, 4, 1.0], [128, 5, 1.0], [128, 6, 1.0], [129, 2, 1.0], [130, 2, 1.0], [130, 3, 1.0], [131, 2, 1.0], [132, 8, 1.0], [133, 8, 1.0], [134, 1, 1.0], [134, 2, 1.0], [134, 6, 4.0], [135, 2, 1.0], [136, 2, 1.0], [137, 7, 1.0], [138, 7, 1.0], [139, 3, 1.0], [139, 5, 2.0], [140, 3, 1.0], [141, 5, 1.0], [142, 1, 6.0], [142, 3, 1.0], [143, 7, 1.0], [144, 0, 1.0], [145, 1, 2.0], [145, 2, 4.0], [145, 3, 2.0], [145, 6, 9.0], [146, 1, 1.0], [147, 3, 1.0], [148, 0, 1.0], [149, 0, 2.0], [149, 1, 1.0], [149, 3, 5.0], [149, 8, 1.0], [150, 0, 1.0], [151, 8, 1.0], [152, 8, 1.0], [153, 3, 1.0], [154, 7, 1.0], [155, 1, 1.0], [155, 2, 1.0], [156, 6, 1.0], [157, 6, 2.0], [158, 0, 1.0], [158, 3, 1.0], [159, 0, 1.0], [159, 3, 1.0], [160, 5, 1.0], [161, 2, 6.0], [161, 4, 2.0], [161, 5, 4.0], [162, 2, 1.0], [162, 5, 1.0], [163, 1, 1.0], [163, 2, 1.0], [163, 5, 1.0], [164, 2, 2.0], [164, 4, 1.0], [164, 5, 8.0], [164, 6, 3.0], [165, 3, 1.0], [166, 1, 1.0], [167, 3, 1.0], [168, 4, 1.0], [169, 0, 1.0], [170, 0, 1.0], [170, 8, 1.0], [171, 2, 1.0], [171, 4, 10.0], [172, 8, 2.0], [173, 3, 1.0], [173, 5, 1.0], [173, 7, 2.0], [174, 7, 1.0], [175, 4, 9.0], [175, 7, 1.0], [176, 1, 1.0], [176, 2, 14.0], [176, 5, 14.0], [176, 6, 1.0], [177, 1, 1.0], [178, 0, 2.0], [178, 4, 4.0], [179, 1, 1.0], [180, 1, 1.0], [181, 7, 2.0], [182, 3, 1.0], [183, 5, 2.0], [184, 6, 1.0], [185, 0, 1.0], [185, 1, 1.0], [185, 3, 1.0], [185, 7, 1.0], [186, 3, 1.0], [187, 2, 1.0], [187, 4, 1.0], [188, 5, 1.0], [189, 5, 1.0], [190, 5, 1.0], [191, 5, 1.0], [192, 5, 1.0], [193, 8, 1.0], [194, 5, 1.0], [195, 4, 1.0], [196, 1, 1.0], [197, 3, 1.0], [198, 5, 1.0], [198, 6, 2.0], [199, 7, 1.0], [200, 7, 1.0], [201, 1, 5.0], [201, 2, 2.0], [201, 6, 1.0], [201, 7, 1.0], [202, 2, 29.0], [202, 5, 10.0], [202, 6, 1.0], [203, 3, 1.0], [204, 5, 1.0], [205, 7, 1.0], [206, 0, 1.0], [207, 0, 1.0], [208, 5, 2.0], [209, 1, 1.0], [210, 8, 13.0], [211, 1, 2.0], [212, 7, 1.0], [213, 5, 1.0], [214, 3, 1.0], [214, 7, 2.0], [215, 5, 1.0], [216, 3, 5.0], [216, 4, 2.0], [217, 1, 1.0], [217, 4, 4.0], [218, 2, 1.0], [218, 7, 4.0], [219, 1, 1.0], [219, 2, 2.0], [219, 5, 2.0], [220, 1, 4.0], [221, 5, 1.0], [222, 1, 1.0], [223, 0, 1.0], [224, 2, 1.0], [225, 8, 1.0], [226, 1, 1.0], [226, 2, 2.0], [226, 6, 2.0], [227, 2, 1.0], [228, 4, 1.0], [229, 2, 1.0], [229, 5, 1.0], [230, 7, 2.0], [231, 1, 5.0], [231, 2, 1.0], [231, 4, 17.0], [231, 7, 20.0], [232, 7, 1.0], [233, 3, 1.0], [234, 3, 1.0], [235, 0, 1.0], [236, 3, 1.0], [237, 8, 1.0], [238, 3, 1.0], [239, 6, 1.0], [240, 1, 1.0], [240, 2, 1.0], [241, 2, 1.0], [242, 8, 1.0], [243, 3, 1.0], [244, 4, 1.0], [245, 7, 1.0], [246, 0, 2.0], [246, 3, 2.0], [246, 7, 7.0], [247, 1, 1.0], [248, 3, 1.0], [248, 7, 1.0], [249, 7, 1.0], [250, 7, 1.0], [250, 8, 1.0], [251, 4, 1.0], [252, 2, 1.0], [252, 5, 1.0], [253, 1, 1.0], [254, 1, 1.0], [254, 7, 1.0], [255, 1, 2.0], [256, 3, 1.0], [257, 5, 1.0], [258, 3, 1.0], [259, 1, 1.0], [260, 2, 1.0], [261, 5, 1.0], [262, 7, 1.0], [263, 0, 1.0], [263, 8, 1.0], [264, 5, 1.0], [265, 4, 1.0], [265, 6, 1.0], [266, 5, 1.0], [267, 0, 1.0], [268, 2, 1.0], [268, 3, 2.0], [269, 1, 1.0], [270, 0, 1.0], [270, 1, 2.0], [270, 2, 1.0], [270, 5, 3.0], [270, 6, 5.0], [271, 0, 1.0], [272, 5, 2.0], [273, 6, 1.0], [274, 7, 1.0], [275, 4, 1.0], [276, 1, 1.0], [277, 1, 2.0], [277, 3, 1.0], [277, 6, 2.0], [278, 7, 1.0], [279, 3, 1.0], [280, 6, 1.0], [281, 4, 1.0], [282, 0, 1.0], [282, 8, 2.0], [283, 8, 5.0], [284, 4, 1.0], [285, 6, 1.0], [285, 8, 1.0], [286, 8, 1.0], [287, 1, 1.0], [288, 0, 1.0], [289, 0, 1.0], [289, 6, 1.0], [290, 3, 1.0], [291, 0, 2.0], [292, 3, 1.0], [293, 7, 1.0], [294, 1, 1.0], [295, 6, 1.0], [296, 3, 1.0], [297, 3, 1.0], [297, 7, 1.0], [298, 6, 1.0], [299, 5, 1.0], [300, 5, 2.0], [301, 7, 1.0], [302, 3, 1.0], [303, 7, 1.0], [304, 2, 1.0], [304, 5, 2.0], [304, 7, 6.0], [305, 3, 1.0], [306, 0, 1.0], [307, 3, 1.0], [308, 4, 5.0], [308, 7, 2.0], [309, 1, 13.0], [309, 2, 11.0], [309, 4, 2.0], [309, 5, 5.0], [309, 6, 12.0], [310, 0, 1.0], [310, 3, 1.0], [310, 8, 1.0], [311, 8, 1.0], [312, 8, 1.0], [313, 1, 1.0], [314, 8, 1.0], [315, 1, 2.0], [316, 5, 1.0], [317, 0, 1.0], [317, 6, 1.0], [318, 4, 1.0], [319, 0, 1.0], [320, 2, 1.0], [321, 0, 1.0], [322, 6, 1.0], [323, 7, 1.0], [324, 5, 1.0], [324, 6, 2.0], [325, 6, 1.0], [326, 2, 1.0], [326, 5, 1.0], [327, 5, 1.0], [328, 7, 1.0], [329, 2, 3.0], [330, 7, 1.0], [331, 3, 1.0], [332, 7, 1.0], [333, 3, 1.0], [334, 4, 2.0], [335, 0, 1.0], [336, 8, 1.0], [337, 0, 2.0], [337, 3, 1.0], [337, 5, 2.0], [338, 3, 1.0], [339, 0, 3.0], [339, 2, 1.0], [339, 4, 2.0], [339, 8, 2.0], [340, 0, 1.0], [341, 7, 1.0], [342, 3, 1.0], [342, 8, 2.0], [343, 0, 3.0], [343, 1, 1.0], [343, 4, 1.0], [343, 6, 1.0], [344, 5, 1.0], [345, 4, 3.0], [346, 6, 1.0], [347, 1, 1.0], [348, 1, 7.0], [348, 3, 2.0], [348, 6, 2.0], [349, 1, 1.0], [350, 5, 1.0], [351, 5, 1.0], [352, 8, 1.0], [353, 5, 1.0], [354, 7, 1.0], [354, 8, 1.0], [355, 3, 1.0], [356, 2, 1.0], [356, 6, 1.0], [357, 4, 2.0], [357, 8, 1.0], [358, 3, 1.0], [359, 3, 1.0], [360, 6, 1.0], [360, 8, 3.0], [361, 7, 1.0], [362, 8, 1.0], [363, 8, 1.0], [364, 6, 1.0], [365, 7, 1.0], [366, 6, 1.0], [367, 0, 1.0], [367, 3, 1.0], [368, 7, 1.0], [369, 5, 1.0], [370, 3, 1.0], [371, 1, 2.0], [372, 1, 4.0], [372, 3, 5.0], [372, 5, 2.0], [372, 6, 2.0], [372, 7, 5.0], [372, 8, 1.0], [373, 3, 1.0], [374, 3, 1.0], [375, 8, 1.0], [376, 0, 1.0], [377, 4, 1.0], [378, 0, 3.0], [378, 3, 5.0], [378, 7, 5.0], [378, 8, 9.0], [379, 0, 4.0], [379, 1, 1.0], [379, 2, 2.0], [379, 3, 4.0], [379, 6, 18.0], [379, 8, 21.0], [380, 1, 1.0], [381, 6, 2.0], [381, 7, 1.0], [382, 5, 1.0], [383, 2, 1.0], [383, 5, 1.0], [384, 5, 1.0], [385, 3, 1.0], [385, 5, 1.0], [386, 0, 1.0], [386, 5, 1.0], [387, 8, 1.0], [388, 4, 2.0], [388, 5, 2.0], [388, 6, 19.0], [388, 8, 3.0], [389, 3, 1.0], [390, 5, 1.0], [391, 5, 1.0], [392, 7, 1.0], [393, 5, 1.0], [394, 1, 1.0], [395, 7, 1.0], [396, 2, 1.0], [396, 5, 1.0], [396, 6, 1.0], [397, 0, 3.0], [397, 3, 1.0], [397, 4, 9.0], [397, 5, 2.0], [397, 6, 1.0], [397, 7, 1.0], [397, 8, 1.0], [398, 8, 2.0], [399, 0, 4.0], [399, 1, 1.0], [399, 4, 4.0], [399, 5, 3.0], [399, 6, 2.0], [399, 8, 5.0], [400, 8, 1.0], [401, 0, 1.0], [402, 6, 1.0], [403, 6, 1.0], [404, 0, 1.0], [404, 1, 2.0], [404, 3, 1.0], [404, 5, 1.0], [404, 6, 2.0], [404, 8, 1.0], [405, 1, 1.0], [406, 8, 1.0], [407, 8, 4.0], [408, 0, 1.0], [409, 1, 9.0], [409, 8, 3.0], [410, 4, 1.0], [411, 0, 1.0], [412, 2, 1.0], [413, 0, 2.0], [413, 3, 10.0], [413, 8, 8.0], [414, 1, 3.0], [414, 5, 1.0], [415, 6, 1.0], [416, 1, 1.0], [417, 1, 1.0]], "columns": [{"id": "PC.636", "metadata": null}, {"id": "PC.481", "metadata": null}, {"id": "PC.354", "metadata": null}, {"id": "PC.635", "metadata": null}, {"id": "PC.593", "metadata": null}, {"id": "PC.356", "metadata": null}, {"id": "PC.355", "metadata": null}, {"id": "PC.607", "metadata": null}, {"id": "PC.634", "metadata": null}], "generated_by": "QIIME 1.4.0-dev, svn revision 2835", "matrix_type": "sparse", "shape": [418, 9], "format_url": "http://biom-format.org", "date": "2012-03-14T20:48:29.112457", "type": "OTU table", "id": null, "matrix_element_type": "int"}"""




if __name__ == "__main__":
    main()