// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "platform/graphics/gpu/WebGLImageConversion.h"

#include "testing/gtest/include/gtest/gtest.h"

namespace blink {

class WebGLImageConversionTest : public testing::Test {
 protected:
  void unpackPixels(const uint16_t* sourceData,
                    WebGLImageConversion::DataFormat sourceDataFormat,
                    unsigned pixelsPerRow,
                    uint8_t* destinationData) {
    WebGLImageConversion::unpackPixels(sourceData, sourceDataFormat,
                                       pixelsPerRow, destinationData);
  }
  void packPixels(const uint8_t* sourceData,
                  WebGLImageConversion::DataFormat sourceDataFormat,
                  unsigned pixelsPerRow,
                  uint8_t* destinationData) {
    WebGLImageConversion::packPixels(sourceData, sourceDataFormat, pixelsPerRow,
                                     destinationData);
  }
};

TEST_F(WebGLImageConversionTest, ConvertRGBA4444toRGBA8) {
  uint16_t sourceData[9] = {0x1234, 0x3456, 0x1234, 0x3456, 0x1234,
                            0x3456, 0x1234, 0x3456, 0x1234};
  uint8_t expectedData[36] = {
      0x11, 0x22, 0x33, 0x44, 0x33, 0x44, 0x55, 0x66, 0x11, 0x22, 0x33, 0x44,
      0x33, 0x44, 0x55, 0x66, 0x11, 0x22, 0x33, 0x44, 0x33, 0x44, 0x55, 0x66,
      0x11, 0x22, 0x33, 0x44, 0x33, 0x44, 0x55, 0x66, 0x11, 0x22, 0x33, 0x44};
  uint8_t destinationData[36];
  unpackPixels(sourceData, WebGLImageConversion::DataFormatRGBA4444, 9,
               destinationData);
  EXPECT_EQ(0, memcmp(expectedData, destinationData, sizeof(destinationData)));
}

TEST_F(WebGLImageConversionTest, ConvertRGBA5551toRGBA8) {
  uint16_t sourceData[9] = {0x1234, 0x3456, 0x1234, 0x3456, 0x1234,
                            0x3456, 0x1234, 0x3456, 0x1234};
  uint8_t expectedData[36] = {
      0x12, 0x40, 0xd2, 0x0, 0x36, 0x89, 0x5b, 0x0, 0x12, 0x40, 0xd2, 0x0,
      0x36, 0x89, 0x5b, 0x0, 0x12, 0x40, 0xd2, 0x0, 0x36, 0x89, 0x5b, 0x0,
      0x12, 0x40, 0xd2, 0x0, 0x36, 0x89, 0x5b, 0x0, 0x12, 0x40, 0xd2, 0x0};
  uint8_t destinationData[36];
  unpackPixels(sourceData, WebGLImageConversion::DataFormatRGBA5551, 9,
               destinationData);
  EXPECT_EQ(0, memcmp(expectedData, destinationData, sizeof(destinationData)));
}

TEST_F(WebGLImageConversionTest, ConvertRGBA8toRA8) {
  uint8_t sourceData[40] = {0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56};
  uint8_t expectedData[20] = {0x9a, 0x56, 0x9a, 0x56, 0x9a, 0x56, 0x9a,
                              0x56, 0x9a, 0x56, 0x9a, 0x56, 0x9a, 0x56,
                              0x9a, 0x56, 0x9a, 0x56, 0x9a, 0x56};
  uint8_t destinationData[20];
  packPixels(sourceData, WebGLImageConversion::DataFormatRA8, 10,
             destinationData);
  EXPECT_EQ(0, memcmp(expectedData, destinationData, sizeof(destinationData)));
}

TEST_F(WebGLImageConversionTest, convertBGRA8toRGBA8) {
  uint32_t sourceData[9] = {0x12345678, 0x34567888, 0x12345678,
                            0x34567888, 0x12345678, 0x34567888,
                            0x12345678, 0x34567888, 0x12345678};
#if CPU(BIG_ENDIAN)
  uint32_t expectedData[9] = {0x56341278, 0x78563488, 0x56341278,
                              0x78563488, 0x56341278, 0x78563488,
                              0x56341278, 0x78563488, 0x56341278};
#else
  uint32_t expectedData[9] = {0x12785634, 0x34887856, 0x12785634,
                              0x34887856, 0x12785634, 0x34887856,
                              0x12785634, 0x34887856, 0x12785634};
#endif
  uint32_t destinationData[9];
  unpackPixels(reinterpret_cast<uint16_t*>(&sourceData[0]),
               WebGLImageConversion::DataFormatBGRA8, 9,
               reinterpret_cast<uint8_t*>(&destinationData[0]));
  EXPECT_EQ(0, memcmp(expectedData, destinationData, sizeof(destinationData)));
}

TEST_F(WebGLImageConversionTest, ConvertRGBA8toR8) {
  uint8_t sourceData[40] = {0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56};
  uint8_t expectedData[10] = {0x9a, 0x9a, 0x9a, 0x9a, 0x9a,
                              0x9a, 0x9a, 0x9a, 0x9a, 0x9a};
  uint8_t destinationData[10];
  packPixels(sourceData, WebGLImageConversion::DataFormatR8, 10,
             destinationData);
  EXPECT_EQ(0, memcmp(expectedData, destinationData, sizeof(destinationData)));
}

TEST_F(WebGLImageConversionTest, ConvertRGBA8toRGBA8) {
  uint8_t sourceData[40] = {0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56};
  uint8_t expectedData[40] = {0x9a, 0xff, 0x9a, 0x56, 0x9a, 0xff, 0x9a, 0x56,
                              0x9a, 0xff, 0x9a, 0x56, 0x9a, 0xff, 0x9a, 0x56,
                              0x9a, 0xff, 0x9a, 0x56, 0x9a, 0xff, 0x9a, 0x56,
                              0x9a, 0xff, 0x9a, 0x56, 0x9a, 0xff, 0x9a, 0x56,
                              0x9a, 0xff, 0x9a, 0x56, 0x9a, 0xff, 0x9a, 0x56};
  uint8_t destinationData[40];
  packPixels(sourceData, WebGLImageConversion::DataFormatRGBA8, 10,
             destinationData);
  EXPECT_EQ(0, memcmp(expectedData, destinationData, sizeof(destinationData)));
}

TEST_F(WebGLImageConversionTest, ConvertRGBA8ToUnsignedShort4444) {
  uint8_t sourceData[40] = {0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56};
  uint16_t expectedData[10] = {0x3535, 0x3535, 0x3535, 0x3535, 0x3535,
                               0x3535, 0x3535, 0x3535, 0x3535, 0x3535};
  uint16_t destinationData[10];
  packPixels(sourceData, WebGLImageConversion::DataFormatRGBA4444, 10,
             reinterpret_cast<uint8_t*>(&destinationData[0]));
  EXPECT_EQ(0, memcmp(expectedData, destinationData, sizeof(destinationData)));
}

TEST_F(WebGLImageConversionTest, ConvertRGBA8ToRGBA5551) {
  uint8_t sourceData[40] = {0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56};
  uint16_t expectedData[10] = {0x328c, 0x328c, 0x328c, 0x328c, 0x328c,
                               0x328c, 0x328c, 0x328c, 0x328c, 0x328c};
  uint16_t destinationData[10];
  packPixels(sourceData, WebGLImageConversion::DataFormatRGBA5551, 10,
             reinterpret_cast<uint8_t*>(&destinationData[0]));
  EXPECT_EQ(0, memcmp(expectedData, destinationData, sizeof(destinationData)));
}

TEST_F(WebGLImageConversionTest, ConvertRGBA8ToRGB565) {
  uint8_t sourceData[40] = {0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56,
                            0x34, 0x56, 0x34, 0x56, 0x34, 0x56, 0x34, 0x56};
  uint16_t expectedData[10] = {0x32a6, 0x32a6, 0x32a6, 0x32a6, 0x32a6,
                               0x32a6, 0x32a6, 0x32a6, 0x32a6, 0x32a6};
  uint16_t destinationData[10];
  packPixels(sourceData, WebGLImageConversion::DataFormatRGB565, 10,
             reinterpret_cast<uint8_t*>(&destinationData[0]));
  EXPECT_EQ(0, memcmp(expectedData, destinationData, sizeof(destinationData)));
}

}  // namespace blink
