/******************************************************************************
*   This file is part of TinTin++                                             *
*                                                                             *
*   Copyright 2004-2020 Igor van den Hoven                                    *
*                                                                             *
*   TinTin++ is free software; you can redistribute it and/or modify          *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 3 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with TinTin++.  If not, see https://www.gnu.org/licenses.           *
******************************************************************************/

/******************************************************************************
*                               T I N T I N + +                               *
*                                                                             *
*                      coded by Igor van den Hoven 2018                       *
******************************************************************************/

#include "tintin.h"

int boundless_binary_index_search16(unsigned short *table, unsigned short size, unsigned short key)
{
	unsigned short bot, mid, top;

	bot = 0;
	top = size;

	while (top > 1)
	{
		mid = top / 2;

		if (key >= table[bot + mid])
		{
			bot += mid;
		}
		top -= mid;
	}

	if (key == table[bot])
	{
		return bot;
	}
	return -1;
}

struct interval_type
{
	int head;
	int tail;
};

int boundless_binary_interval_search(struct interval_type *table, unsigned int size, int key)
{
	unsigned int bot, mid, top;

	bot = 0;
	top = size;

	while (top > 1)
	{
		mid = top / 2;

		if (key >= table[bot + mid].head)
		{
			bot += mid;
		}
		top -= mid;
	}

	return key >= table[bot].head && key <= table[bot].tail;
}

int boundless_binary_interval_index_search(struct interval_type *table, int size, int key)
{
	unsigned int bot, mid, top;

	bot = 0;
	top = size;

	while (top > 1)
	{
		mid = top / 2;

		if (key >= table[bot + mid].head)
		{
			bot += mid;
		}
		top -= mid;
	}

	if (key == table[bot].head)
	{
		return table[bot].tail;
	}
	return 0;
}

int linear_search(struct interval_type *table, int size, int key)
{
	int i;

	for (i = 0 ; i < size ; i++)
	{
		if (table[i].head > table[i].tail)
		{
			print_stdout(0, 0, "table[%d].head < table[%d].tail\n", i, i);
		}
		if (i < size - 1)
		{
			if (table[i].head >= table[i+1].head)
			{
				print_stdout(0, 0, "table[%d].head >= table[%d].head\n", i, i+1);
			}
		}
	}
	return 0;
}

// Special thanks to mintty for the data tables

static int unicode_width(int index)
{
	static struct interval_type zero_length_table[] =
	{
		{ 0x0300, 0x036F }, { 0x0483, 0x0489 }, { 0x0591, 0x05BD }, { 0x05BF, 0x05BF }, { 0x05C1, 0x05C2 }, { 0x05C4, 0x05C5 },
		{ 0x05C7, 0x05C7 }, { 0x0600, 0x0605 }, { 0x0610, 0x061A }, { 0x061C, 0x061C }, { 0x064B, 0x065F }, { 0x0670, 0x0670 },
		{ 0x06D6, 0x06DD }, { 0x06DF, 0x06E4 }, { 0x06E7, 0x06E8 }, { 0x06EA, 0x06ED }, { 0x070F, 0x070F }, { 0x0711, 0x0711 },
		{ 0x0730, 0x074A }, { 0x07A6, 0x07B0 }, { 0x07EB, 0x07F3 }, { 0x07FD, 0x07FD }, { 0x0816, 0x0819 }, { 0x081B, 0x0823 },
		{ 0x0825, 0x0827 }, { 0x0829, 0x082D }, { 0x0859, 0x085B }, { 0x08D3, 0x0902 }, { 0x093A, 0x093A }, { 0x093C, 0x093C },
		{ 0x0941, 0x0948 }, { 0x094D, 0x094D }, { 0x0951, 0x0957 }, { 0x0962, 0x0963 }, { 0x0981, 0x0981 }, { 0x09BC, 0x09BC },
		{ 0x09C1, 0x09C4 }, { 0x09CD, 0x09CD }, { 0x09E2, 0x09E3 }, { 0x09FE, 0x09FE }, { 0x0A01, 0x0A02 }, { 0x0A3C, 0x0A3C },
		{ 0x0A41, 0x0A42 }, { 0x0A47, 0x0A48 }, { 0x0A4B, 0x0A4D }, { 0x0A51, 0x0A51 }, { 0x0A70, 0x0A71 }, { 0x0A75, 0x0A75 },
		{ 0x0A81, 0x0A82 }, { 0x0ABC, 0x0ABC }, { 0x0AC1, 0x0AC5 }, { 0x0AC7, 0x0AC8 }, { 0x0ACD, 0x0ACD }, { 0x0AE2, 0x0AE3 },
		{ 0x0AFA, 0x0AFF }, { 0x0B01, 0x0B01 }, { 0x0B3C, 0x0B3C }, { 0x0B3F, 0x0B3F }, { 0x0B41, 0x0B44 }, { 0x0B4D, 0x0B4D },
		{ 0x0B56, 0x0B56 }, { 0x0B62, 0x0B63 }, { 0x0B82, 0x0B82 }, { 0x0BC0, 0x0BC0 }, { 0x0BCD, 0x0BCD }, { 0x0C00, 0x0C00 },
		{ 0x0C04, 0x0C04 }, { 0x0C3E, 0x0C40 }, { 0x0C46, 0x0C48 }, { 0x0C4A, 0x0C4D }, { 0x0C55, 0x0C56 }, { 0x0C62, 0x0C63 },
		{ 0x0C81, 0x0C81 }, { 0x0CBC, 0x0CBC }, { 0x0CBF, 0x0CBF }, { 0x0CC6, 0x0CC6 }, { 0x0CCC, 0x0CCD }, { 0x0CE2, 0x0CE3 },
		{ 0x0D00, 0x0D01 }, { 0x0D3B, 0x0D3C }, { 0x0D41, 0x0D44 }, { 0x0D4D, 0x0D4D }, { 0x0D62, 0x0D63 }, { 0x0DCA, 0x0DCA },
		{ 0x0DD2, 0x0DD4 }, { 0x0DD6, 0x0DD6 }, { 0x0E31, 0x0E31 }, { 0x0E34, 0x0E3A }, { 0x0E47, 0x0E4E }, { 0x0EB1, 0x0EB1 },
		{ 0x0EB4, 0x0EB9 }, { 0x0EBB, 0x0EBC }, { 0x0EC8, 0x0ECD }, { 0x0F18, 0x0F19 }, { 0x0F35, 0x0F35 }, { 0x0F37, 0x0F37 },
		{ 0x0F39, 0x0F39 }, { 0x0F71, 0x0F7E }, { 0x0F80, 0x0F84 }, { 0x0F86, 0x0F87 }, { 0x0F8D, 0x0F97 }, { 0x0F99, 0x0FBC },
		{ 0x0FC6, 0x0FC6 }, { 0x102D, 0x1030 }, { 0x1032, 0x1037 }, { 0x1039, 0x103A }, { 0x103D, 0x103E }, { 0x1058, 0x1059 },
		{ 0x105E, 0x1060 }, { 0x1071, 0x1074 }, { 0x1082, 0x1082 }, { 0x1085, 0x1086 }, { 0x108D, 0x108D }, { 0x109D, 0x109D },
		{ 0x1160, 0x11FF }, { 0x135D, 0x135F }, { 0x1712, 0x1714 }, { 0x1732, 0x1734 }, { 0x1752, 0x1753 }, { 0x1772, 0x1773 },
		{ 0x17B4, 0x17B5 }, { 0x17B7, 0x17BD }, { 0x17C6, 0x17C6 }, { 0x17C9, 0x17D3 }, { 0x17DD, 0x17DD }, { 0x180B, 0x180E },
		{ 0x1885, 0x1886 }, { 0x18A9, 0x18A9 }, { 0x1920, 0x1922 }, { 0x1927, 0x1928 }, { 0x1932, 0x1932 }, { 0x1939, 0x193B },
		{ 0x1A17, 0x1A18 }, { 0x1A1B, 0x1A1B }, { 0x1A56, 0x1A56 }, { 0x1A58, 0x1A5E }, { 0x1A60, 0x1A60 }, { 0x1A62, 0x1A62 },
		{ 0x1A65, 0x1A6C }, { 0x1A73, 0x1A7C }, { 0x1A7F, 0x1A7F }, { 0x1AB0, 0x1ABE }, { 0x1B00, 0x1B03 }, { 0x1B34, 0x1B34 },
		{ 0x1B36, 0x1B3A }, { 0x1B3C, 0x1B3C }, { 0x1B42, 0x1B42 }, { 0x1B6B, 0x1B73 }, { 0x1B80, 0x1B81 }, { 0x1BA2, 0x1BA5 },
		{ 0x1BA8, 0x1BA9 }, { 0x1BAB, 0x1BAD }, { 0x1BE6, 0x1BE6 }, { 0x1BE8, 0x1BE9 }, { 0x1BED, 0x1BED }, { 0x1BEF, 0x1BF1 },
		{ 0x1C2C, 0x1C33 }, { 0x1C36, 0x1C37 }, { 0x1CD0, 0x1CD2 }, { 0x1CD4, 0x1CE0 }, { 0x1CE2, 0x1CE8 }, { 0x1CED, 0x1CED },
		{ 0x1CF4, 0x1CF4 }, { 0x1CF8, 0x1CF9 }, { 0x1DC0, 0x1DF9 }, { 0x1DFB, 0x1DFF }, { 0x200B, 0x200F }, { 0x202A, 0x202E },
		{ 0x2060, 0x2064 }, { 0x2066, 0x206F }, { 0x20D0, 0x20F0 }, { 0x2CEF, 0x2CF1 }, { 0x2D7F, 0x2D7F }, { 0x2DE0, 0x2DFF },
		{ 0x302A, 0x302D }, { 0x3099, 0x309A }, { 0xA66F, 0xA672 }, { 0xA674, 0xA67D }, { 0xA69E, 0xA69F }, { 0xA6F0, 0xA6F1 },
		{ 0xA802, 0xA802 }, { 0xA806, 0xA806 }, { 0xA80B, 0xA80B }, { 0xA825, 0xA826 }, { 0xA8C4, 0xA8C5 }, { 0xA8E0, 0xA8F1 },
		{ 0xA8FF, 0xA8FF }, { 0xA926, 0xA92D }, { 0xA947, 0xA951 }, { 0xA980, 0xA982 }, { 0xA9B3, 0xA9B3 }, { 0xA9B6, 0xA9B9 },
		{ 0xA9BC, 0xA9BC }, { 0xA9E5, 0xA9E5 }, { 0xAA29, 0xAA2E }, { 0xAA31, 0xAA32 }, { 0xAA35, 0xAA36 }, { 0xAA43, 0xAA43 },
		{ 0xAA4C, 0xAA4C }, { 0xAA7C, 0xAA7C }, { 0xAAB0, 0xAAB0 }, { 0xAAB2, 0xAAB4 }, { 0xAAB7, 0xAAB8 }, { 0xAABE, 0xAABF },
		{ 0xAAC1, 0xAAC1 }, { 0xAAEC, 0xAAED }, { 0xAAF6, 0xAAF6 }, { 0xABE5, 0xABE5 }, { 0xABE8, 0xABE8 }, { 0xABED, 0xABED },
		{ 0xD7B0, 0xD7C6 }, { 0xD7CB, 0xD7FB }, { 0xFB1E, 0xFB1E }, { 0xFE00, 0xFE0F }, { 0xFE20, 0xFE2F }, { 0xFEFF, 0xFEFF },
		{ 0xFFF9, 0xFFFB }, {0x101FD, 0x101FD}, {0x102E0, 0x102E0}, {0x10376, 0x1037A}, {0x10A01, 0x10A03}, {0x10A05, 0x10A06},
		{0x10A0C, 0x10A0F}, {0x10A38, 0x10A3A}, {0x10A3F, 0x10A3F}, {0x10AE5, 0x10AE6}, {0x10D24, 0x10D27}, {0x10F46, 0x10F50},
		{0x11001, 0x11001}, {0x11038, 0x11046}, {0x1107F, 0x11081}, {0x110B3, 0x110B6}, {0x110B9, 0x110BA}, {0x110BD, 0x110BD},
		{0x110CD, 0x110CD}, {0x11100, 0x11102}, {0x11127, 0x1112B}, {0x1112D, 0x11134}, {0x11173, 0x11173}, {0x11180, 0x11181},
		{0x111B6, 0x111BE}, {0x111C9, 0x111CC}, {0x1122F, 0x11231}, {0x11234, 0x11234}, {0x11236, 0x11237}, {0x1123E, 0x1123E},
		{0x112DF, 0x112DF}, {0x112E3, 0x112EA}, {0x11300, 0x11301}, {0x1133B, 0x1133C}, {0x11340, 0x11340}, {0x11366, 0x1136C},
		{0x11370, 0x11374}, {0x11438, 0x1143F}, {0x11442, 0x11444}, {0x11446, 0x11446}, {0x1145E, 0x1145E}, {0x114B3, 0x114B8},
		{0x114BA, 0x114BA}, {0x114BF, 0x114C0}, {0x114C2, 0x114C3}, {0x115B2, 0x115B5}, {0x115BC, 0x115BD}, {0x115BF, 0x115C0},
		{0x115DC, 0x115DD}, {0x11633, 0x1163A}, {0x1163D, 0x1163D}, {0x1163F, 0x11640}, {0x116AB, 0x116AB}, {0x116AD, 0x116AD},
		{0x116B0, 0x116B5}, {0x116B7, 0x116B7}, {0x1171D, 0x1171F}, {0x11722, 0x11725}, {0x11727, 0x1172B}, {0x1182F, 0x11837},
		{0x11839, 0x1183A}, {0x11A01, 0x11A0A}, {0x11A33, 0x11A38}, {0x11A3B, 0x11A3E}, {0x11A47, 0x11A47}, {0x11A51, 0x11A56},
		{0x11A59, 0x11A5B}, {0x11A8A, 0x11A96}, {0x11A98, 0x11A99}, {0x11C30, 0x11C36}, {0x11C38, 0x11C3D}, {0x11C3F, 0x11C3F},
		{0x11C92, 0x11CA7}, {0x11CAA, 0x11CB0}, {0x11CB2, 0x11CB3}, {0x11CB5, 0x11CB6}, {0x11D31, 0x11D36}, {0x11D3A, 0x11D3A},
		{0x11D3C, 0x11D3D}, {0x11D3F, 0x11D45}, {0x11D47, 0x11D47}, {0x11D90, 0x11D91}, {0x11D95, 0x11D95}, {0x11D97, 0x11D97},
		{0x11EF3, 0x11EF4}, {0x16AF0, 0x16AF4}, {0x16B30, 0x16B36}, {0x16F8F, 0x16F92}, {0x1BC9D, 0x1BC9E}, {0x1BCA0, 0x1BCA3},
		{0x1D167, 0x1D169}, {0x1D173, 0x1D182}, {0x1D185, 0x1D18B}, {0x1D1AA, 0x1D1AD}, {0x1D242, 0x1D244}, {0x1DA00, 0x1DA36},
		{0x1DA3B, 0x1DA6C}, {0x1DA75, 0x1DA75}, {0x1DA84, 0x1DA84}, {0x1DA9B, 0x1DA9F}, {0x1DAA1, 0x1DAAF}, {0x1E000, 0x1E006},
		{0x1E008, 0x1E018}, {0x1E01B, 0x1E021}, {0x1E023, 0x1E024}, {0x1E026, 0x1E02A}, {0x1E8D0, 0x1E8D6}, {0x1E944, 0x1E94A},
		{0xE0001, 0xE0001}, {0xE0020, 0xE007F}, {0xE0100, 0xE01EF }
	};

	static struct interval_type double_length_table[] =
	{
		{ 0x1100, 0x115F}, { 0x231A, 0x231B}, { 0x2329, 0x232A}, { 0x23E9, 0x23EC}, { 0x23F0, 0x23F0},
		{ 0x23F3, 0x23F3}, { 0x25FD, 0x25FE}, { 0x2614, 0x2615}, { 0x2648, 0x2653}, { 0x267F, 0x267F},
		{ 0x2693, 0x2693}, { 0x26A1, 0x26A1}, { 0x26AA, 0x26AB}, { 0x26BD, 0x26BE}, { 0x26C4, 0x26C5},
		{ 0x26CE, 0x26CE}, { 0x26D4, 0x26D4}, { 0x26EA, 0x26EA}, { 0x26F2, 0x26F3}, { 0x26F5, 0x26F5},
		{ 0x26FA, 0x26FA}, { 0x26FD, 0x26FD}, { 0x2705, 0x2705}, { 0x270A, 0x270B}, { 0x2728, 0x2728},
		{ 0x274C, 0x274C}, { 0x274E, 0x274E}, { 0x2753, 0x2755}, { 0x2757, 0x2757}, { 0x2795, 0x2797},
		{ 0x27B0, 0x27B0}, { 0x27BF, 0x27BF}, { 0x2B1B, 0x2B1C}, { 0x2B50, 0x2B50}, { 0x2B55, 0x2B55},
		{ 0x2E80, 0x303E}, { 0x3040, 0x321E}, { 0x3220, 0x3247}, { 0x3250, 0x32FE}, { 0x3300, 0x4DBF},
		{ 0x4E00, 0xA4CF}, { 0xA960, 0xA97F}, { 0xAC00, 0xD7AF}, { 0xF900, 0xFAFF}, { 0xFE10, 0xFE1F},
		{ 0xFE30, 0xFE6F}, { 0xFF01, 0xFF60}, { 0xFFE0, 0xFFE6}, {0x16FE0,0x18AFF}, {0x1B000,0x1B12F},
		{0x1B170,0x1B2FF}, {0x1F004,0x1F004}, {0x1F0CF,0x1F0CF}, {0x1F18E,0x1F18E}, {0x1F191,0x1F19A},
		{0x1F200,0x1F320}, {0x1F32D,0x1F335}, {0x1F337,0x1F37C}, {0x1F37E,0x1F393}, {0x1F3A0,0x1F3CA},
		{0x1F3CF,0x1F3D3}, {0x1F3E0,0x1F3F0}, {0x1F3F4,0x1F3F4}, {0x1F3F8,0x1F43E}, {0x1F440,0x1F440},
		{0x1F442,0x1F4FC}, {0x1F4FF,0x1F53D}, {0x1F54B,0x1F54E}, {0x1F550,0x1F567}, {0x1F57A,0x1F57A},
		{0x1F595,0x1F596}, {0x1F5A4,0x1F5A4}, {0x1F5FB,0x1F64F}, {0x1F680,0x1F6C5}, {0x1F6CC,0x1F6CC},
		{0x1F6D0,0x1F6D2}, {0x1F6EB,0x1F6EC}, {0x1F6F4,0x1F6F9}, {0x1F910,0x1F93E}, {0x1F940,0x1F970},
		{0x1F973,0x1F976}, {0x1F97A,0x1F97A}, {0x1F97C,0x1F9A2}, {0x1F9B0,0x1F9B9}, {0x1F9C0,0x1F9C2},
		{0x1F9D0,0x1F9FF}, {0x20000,0x2FFFD}, {0x30000,0x3FFFD }
	};

	if (boundless_binary_interval_search(zero_length_table, sizeof(zero_length_table) / sizeof(struct interval_type), index))
	{
		return 0;
	}

	if (boundless_binary_interval_search(double_length_table, sizeof(double_length_table) / sizeof(struct interval_type), index))
	{
		return 2;
	}

	return 1;
}

int is_utf8_head(char *str)
{
	static int is_utf8[256] =
	{
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
		3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,0,0,0,0,0,0,0,0
	};

	return is_utf8[(unsigned char) *str];
}

int is_utf8_tail(char *str)
{
	static char utf8_tail[256] =
	{
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
	};
	return utf8_tail[(unsigned char) *str];
}

int get_utf8_size(char *str)
{
	static char utf8_size[256] =
	{
		1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
		3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,5,5,5,5,6,6,1,1
	};
	unsigned char *ptu = (unsigned char *) str;

	if (*ptu < 128)
	{
		return 1;
	}

	switch (utf8_size[*ptu])
	{
		case 2:
			if (utf8_size[ptu[1]])
			{
				return 1;
			}
			return 2;

		case 3:
			if (utf8_size[ptu[1]] || utf8_size[ptu[2]])
			{
				return 1;
			}
			return 3;

		case 4:
			if (utf8_size[ptu[1]] || utf8_size[ptu[2]] || utf8_size[ptu[3]])
			{
				return 1;
			}
			return 4;
		default:
			return 1;
	}
}

unsigned char utf8_width_table[256] =
{
	0,0,0,0,0,0,0,0,0,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
	3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,5,5,5,5,6,6,1,1
};

int get_ascii_width(char *str, int *width)
{
	*width = utf8_width_table[(unsigned char) *str] ? 1 : 0;

	return 1;
}

int get_utf8_width(char *str, int *width, int *index)
{
	int size, temp;
	unsigned char *ptu;

	if (index == NULL)
	{
		index = &temp;
	}

	*index = (unsigned char) *str;

	*width = size = utf8_width_table[*index];

	if (size <= 1)
	{
		return 1;
	}

	ptu = (unsigned char *) str;

	switch (*width)
	{
		case 2:
			if (get_utf8_size(str) != 2)
			{
				return 1;
			}
			*index = (int) ptu[1] - 128 + 64 * (ptu[0] - 192);
			break;
		case 3:
			if (get_utf8_size(str) != 3)
			{
				return 1;
			}
			*index = (int) ptu[2] - 128 + 64 * (ptu[1] - 128) + 4096 * (ptu[0] - 224);
			break;
		case 4:
			if (get_utf8_size(str) != 4)
			{
				return 1;
			}
			*index = (int) ptu[3] - 128 + 64 * (ptu[2] - 128) + 4096 * (ptu[1] - 128) + 262144 * (ptu[0] - 240);
			break;
		default:
			return 1;
	}

	*width = unicode_width(*index);

	return size;
}

int get_utf8_index(char *str, int *index)
{
	int size, width;
	unsigned char *ptu = (unsigned char *) str;

	*index = (int) *ptu;

	width = size = utf8_width_table[*index];

	switch (width)
	{
		default:
			return 1;

		case 2:
			if (get_utf8_size(str) != 2)
			{
				return 1;
			}
			*index = (int) ptu[1] - 128 + 64 * (ptu[0] - 192);
			break;
		case 3:
			if (get_utf8_size(str) != 3)
			{
				return 1;
			}
			*index = (int) ptu[2] - 128 + 64 * (ptu[1] - 128) + 4096 * (ptu[0] - 224);
			break;
		case 4:
			if (get_utf8_size(str) != 4)
			{
				return 1;
			}
			*index = (int) ptu[3] - 128 + 64 * (ptu[2] - 128) + 4096 * (ptu[1] - 128) + 262144 * (ptu[0] - 240);
			break;
	}
	return size;
}

int unicode_to_utf8(int index, char *out)
{
	unsigned char *pto = (unsigned char *) out;

	if (index < 128)
	{
		*pto++ = index;
		*pto++ = 0;
		return 1;
	}
	else if (index < 4096)
	{
		*pto++ = 192 + index / 64;
		*pto++ = 128 + index % 64;
		*pto++ = 0;
		return 2;
	}
	else if (index < 65536)
	{
		*pto++ = 224 + index / 4096;
		*pto++ = 128 + index / 64 % 64;
		*pto++ = 128 + index % 64;
		*pto++ = 0;
		return 3;
	}
	else if (index < 1114112)
	{
		*pto++ = 240 + index / 262144;
		*pto++ = 128 + index / 4096 % 64;
		*pto++ = 128 + index / 64 % 64;
	        *pto++ = 128 + index % 64;
	        *pto++ = 0;
	        return 4;
	}
	else // �
	{
		*pto++ = 239;
		*pto++ = 191;
		*pto++ = 189;
		*pto++ = 0;
		return 3;
	}
}

int utf8_strlen(char *str, int *str_len)
{
	int raw_len, size, width;

	raw_len = *str_len = 0;

	while (*str)
	{
		size = get_utf8_width(str, &width, NULL);

		*str_len += width;
		raw_len += size;
	}

	return raw_len;
}

int utf8_to_all(struct session *ses, char *in, char *out)
{
	switch (HAS_BIT(ses->charset, CHARSET_FLAG_ALL_TOUTF8))
	{
		case CHARSET_FLAG_BIG5TOUTF8:
			return utf8_to_big5(in, out);

		case CHARSET_FLAG_CP1251TOUTF8:
			return utf8_to_cp1251(in, out);

		case CHARSET_FLAG_CP949TOUTF8:
			return utf8_to_cp949(in, out);

		case CHARSET_FLAG_GBK1TOUTF8:
			return utf8_to_gbk1(in, out);

		case CHARSET_FLAG_FANSITOUTF8:
			return sprintf(out, "%s", in);

		case CHARSET_FLAG_ISO1TOUTF8:
			return utf8_to_iso1(in, out);

		case CHARSET_FLAG_ISO2TOUTF8:
			return utf8_to_iso2(in, out);

		case CHARSET_FLAG_KOI8TOUTF8:
			return utf8_to_koi8(in, out);

	}
	*out = 0;

	tintin_printf2(ses, "debug: utf8_to_all: unknown error");

	return 0;
}

int all_to_utf8(struct session *ses, char *in, char *out)
{
	switch (HAS_BIT(ses->charset, CHARSET_FLAG_ALL_TOUTF8))
	{
		case CHARSET_FLAG_BIG5TOUTF8:
			return big5_to_utf8(in, out);

		case CHARSET_FLAG_CP1251TOUTF8:
			return cp1251_to_utf8(in, out);

		case CHARSET_FLAG_CP949TOUTF8:
			return cp949_to_utf8(in, out);

		case CHARSET_FLAG_FANSITOUTF8:
			return fansi_to_utf8(in, out);

		case CHARSET_FLAG_GBK1TOUTF8:
			return gbk1_to_utf8(in, out);

		case CHARSET_FLAG_ISO1TOUTF8:
			return iso1_to_utf8(in, out);

		case CHARSET_FLAG_ISO2TOUTF8:
			return iso2_to_utf8(in, out);

		case CHARSET_FLAG_KOI8TOUTF8:
			return koi8_to_utf8(in, out);


	}
	tintin_printf2(ses, "debug: utf8_to_all: unknown error");
	return 0;
}

// ISO-1

int utf8_to_iso1(char *input, char *output)
{
	char *pti, *pto;
	int size, index;

	pti = input;
	pto = output;

	while (*pti)
	{
		size = get_utf8_index(pti, &index);

		if (size > 1)
		{
			if (index > 127 && index < 255)
			{
				*pto++ = index;
			}
			else
			{
				tintin_printf(gtd->ses, "<118>utf8_to_iso1: did not find unicode index '0x%04x'", index);

				*pto++ = '?';
			}
			pti += size;
		}
		else
		{
			*pto++ = *pti++;
		}
	}
	*pto = 0;

	return pto - output;
}

int iso1_to_utf8(char *input, char *output)
{
	unsigned char *pti, *pto;
	int index;

	pti = (unsigned char *) input;
	pto = (unsigned char *) output;

	while (*pti)
	{
		index = *pti;

		if (index < 128)
		{
			*pto++ = index;
		}
		else
		{
			*pto++ = 192 + index / 64;
			*pto++ = 128 + index % 64;
		}
		pti++;
	}
	*pto = 0;

	return pto - (unsigned char *) output;
}

// ISO-2

struct interval_type unicode_to_iso2_table[] =
{
	{0x0080,0x80},{0x0081,0x81},{0x0082,0x82},{0x0083,0x83},{0x0084,0x84},{0x0085,0x85},{0x0086,0x86},{0x0087,0x87},
	{0x0088,0x88},{0x0089,0x89},{0x008A,0x8A},{0x008B,0x8B},{0x008C,0x8C},{0x008D,0x8D},{0x008E,0x8E},{0x008F,0x8F},
	{0x0090,0x90},{0x0091,0x91},{0x0092,0x92},{0x0093,0x93},{0x0094,0x94},{0x0095,0x95},{0x0096,0x96},{0x0097,0x97},
	{0x0098,0x98},{0x0099,0x99},{0x009A,0x9A},{0x009B,0x9B},{0x009C,0x9C},{0x009D,0x9D},{0x009E,0x9E},{0x009F,0x9F},
	{0x00A0,0xA0},{0x00A4,0xA4},{0x00A7,0xA7},{0x00A8,0xA8},{0x00AD,0xAD},{0x00B0,0xB0},{0x00B4,0xB4},{0x00B8,0xB8},
	{0x00C1,0xC1},{0x00C2,0xC2},{0x00C4,0xC4},{0x00C7,0xC7},{0x00C9,0xC9},{0x00CB,0xCB},{0x00CD,0xCD},{0x00CE,0xCE},
	{0x00D3,0xD3},{0x00D4,0xD4},{0x00D6,0xD6},{0x00D7,0xD7},{0x00DA,0xDA},{0x00DC,0xDC},{0x00DD,0xDD},{0x00DF,0xDF},
	{0x00E1,0xE1},{0x00E2,0xE2},{0x00E4,0xE4},{0x00E7,0xE7},{0x00E9,0xE9},{0x00EB,0xEB},{0x00ED,0xED},{0x00EE,0xEE},
	{0x00F3,0xF3},{0x00F4,0xF4},{0x00F6,0xF6},{0x00F7,0xF7},{0x00FA,0xFA},{0x00FC,0xFC},{0x00FD,0xFD},{0x0102,0xC3},
	{0x0103,0xE3},{0x0104,0xA1},{0x0105,0xB1},{0x0106,0xC6},{0x0107,0xE6},{0x010C,0xC8},{0x010D,0xE8},{0x010E,0xCF},
	{0x010F,0xEF},{0x0110,0xD0},{0x0111,0xF0},{0x0118,0xCA},{0x0119,0xEA},{0x011A,0xCC},{0x011B,0xEC},{0x0139,0xC5},
	{0x013A,0xE5},{0x013D,0xA5},{0x013E,0xB5},{0x0141,0xA3},{0x0142,0xB3},{0x0143,0xD1},{0x0144,0xF1},{0x0147,0xD2},
	{0x0148,0xF2},{0x0150,0xD5},{0x0151,0xF5},{0x0154,0xC0},{0x0155,0xE0},{0x0158,0xD8},{0x0159,0xF8},{0x015A,0xA6},
	{0x015B,0xB6},{0x015E,0xAA},{0x015F,0xBA},{0x0160,0xA9},{0x0161,0xB9},{0x0162,0xDE},{0x0163,0xFE},{0x0164,0xAB},
	{0x0165,0xBB},{0x016E,0xD9},{0x016F,0xF9},{0x0170,0xDB},{0x0171,0xFB},{0x0179,0xAC},{0x017A,0xBC},{0x017B,0xAF},
	{0x017C,0xBF},{0x017D,0xAE},{0x017E,0xBE},{0x02C7,0xB7},{0x02D8,0xA2},{0x02D9,0xFF},{0x02DB,0xB2},{0x02DD,0xBD}
};

int utf8_to_iso2(char *input, char *output)
{
	char *pti, *pto;
	int size, index, result;

	pti = input;
	pto = output;

	while (*pti)
	{
		size = get_utf8_index(pti, &index);

		if (size > 1)
		{
			result = boundless_binary_interval_index_search(unicode_to_iso2_table, sizeof(unicode_to_iso2_table) / sizeof(struct interval_type), index);

			if (result)
			{
				*pto++ = result;
			}
			else
			{
				tintin_printf(gtd->ses, "<118>utf8_to_iso2: did not find unicode index '0x%04x'", index);

				*pto++ = '?';
			}
			pti += size;
		}
		else
		{
			*pto++ = *pti++;
		}
	}
	*pto = 0;

	return pto - output;
}

int iso2_to_utf8(char *input, char *output)
{
	unsigned char *pti, *pto;
	int index;

	static int iso2[256] =
	{
		        0,      1,      2,      3,      4,      5,      6,      7,
		        8,      9,     10,     11,     12,     13,     14,     15,
		       16,     17,     18,     19,     20,     21,     22,     23,
		       24,     25,     26,     27,     28,     29,     30,     31,
		       32,     33,     34,     35,     36,     37,     38,     39,
		       40,     41,     42,     43,     44,     45,     46,     47,
		       48,     49,     50,     51,     52,     53,     54,     55,
		       56,     57,     58,     59,     60,     61,     62,     63,
		       64,     65,     66,     67,     68,     69,     70,     71,
		       72,     73,     74,     75,     76,     77,     78,     79,
		       80,     81,     82,     83,     84,     85,     86,     87,
		       88,     89,     90,     91,     92,     93,     94,     95,
		       96,     97,     98,     99,    100,    101,    102,    103,
		      104,    105,    106,    107,    108,    109,    110,    111,
		      112,    113,    114,    115,    116,    117,    118,    119,
		      120,    121,    122,    123,    124,    125,    126,    127,
		   0x0080, 0x0081, 0x0082, 0x0083, 0x0084, 0x0085, 0x0086, 0x0087,
		   0x0088, 0x0089, 0x008A, 0x008B, 0x008C, 0x008D, 0x008E, 0x008F,
		   0x0090, 0x0091, 0x0092, 0x0093, 0x0094, 0x0095, 0x0096, 0x0097,
		   0x0098, 0x0099, 0x009A, 0x009B, 0x009C, 0x009D, 0x009E, 0x009F,
		   0x00A0, 0x0104, 0x02D8, 0x0141, 0x00A4, 0x013D, 0x015A, 0x00A7,
		   0x00A8, 0x0160, 0x015E, 0x0164, 0x0179, 0x00AD, 0x017D, 0x017B,
		   0x00B0, 0x0105, 0x02DB, 0x0142, 0x00B4, 0x013E, 0x015B, 0x02C7,
		   0x00B8, 0x0161, 0x015F, 0x0165, 0x017A, 0x02DD, 0x017E, 0x017C,
		   0x0154, 0x00C1, 0x00C2, 0x0102, 0x00C4, 0x0139, 0x0106, 0x00C7,
		   0x010C, 0x00C9, 0x0118, 0x00CB, 0x011A, 0x00CD, 0x00CE, 0x010E,
		   0x0110, 0x0143, 0x0147, 0x00D3, 0x00D4, 0x0150, 0x00D6, 0x00D7,
		   0x0158, 0x016E, 0x00DA, 0x0170, 0x00DC, 0x00DD, 0x0162, 0x00DF,
		   0x0155, 0x00E1, 0x00E2, 0x0103, 0x00E4, 0x013A, 0x0107, 0x00E7,
		   0x010D, 0x00E9, 0x0119, 0x00EB, 0x011B, 0x00ED, 0x00EE, 0x010F,
		   0x0111, 0x0144, 0x0148, 0x00F3, 0x00F4, 0x0151, 0x00F6, 0x00F7,
		   0x0159, 0x016F, 0x00FA, 0x0171, 0x00FC, 0x00FD, 0x0163, 0x02D9
	};

	pti = (unsigned char *) input;
	pto = (unsigned char *) output;

	while (*pti)
	{
		index = iso2[*pti];

		if (index < 128)
		{
			*pto++ = index;
		}
		else if (index < 4096)
		{
			*pto++ = 192 + index / 64;
			*pto++ = 128 + index % 64;
		}
		else
		{
			*pto++ = 224 + index / 4096;
			*pto++ = 128 + index / 64 % 64;
			*pto++ = 128 + index % 64;
		}
		pti++;
	}
	*pto = 0;

	return pto - (unsigned char *) output;
}

// CP1251 Special thanks to tpuctah for the data table

struct interval_type unicode_to_cp1251_table[] =
{
	{0x00a0, 0x00a0}, {0x00a4, 0x00a4}, {0x00a6, 0x00a6}, {0x00a7, 0x00a7}, {0x00a9, 0x00a9}, {0x00ab, 0x00ab}, {0x00ac, 0x00ac}, {0x00ad, 0x00ad},
	{0x00ae, 0x00ae}, {0x00b0, 0x00b0}, {0x00b1, 0x00b1}, {0x00b5, 0x00b5}, {0x00b6, 0x00b6}, {0x00b7, 0x00b7}, {0x00bb, 0x00bb}, {0x0401, 0x00a8},
	{0x0402, 0x0080}, {0x0403, 0x0081}, {0x0404, 0x00aa}, {0x0405, 0x00bd}, {0x0406, 0x00b2}, {0x0407, 0x00af}, {0x0408, 0x00a3}, {0x0409, 0x008a},
	{0x040a, 0x008c}, {0x040b, 0x008e}, {0x040c, 0x008d}, {0x040e, 0x00a1}, {0x040f, 0x008f}, {0x0410, 0x00c0}, {0x0411, 0x00c1}, {0x0412, 0x00c2},
	{0x0413, 0x00c3}, {0x0414, 0x00c4}, {0x0415, 0x00c5}, {0x0416, 0x00c6}, {0x0417, 0x00c7}, {0x0418, 0x00c8}, {0x0419, 0x00c9}, {0x041a, 0x00ca},
	{0x041b, 0x00cb}, {0x041c, 0x00cc}, {0x041d, 0x00cd}, {0x041e, 0x00ce}, {0x041f, 0x00cf}, {0x0420, 0x00d0}, {0x0421, 0x00d1}, {0x0422, 0x00d2},
	{0x0423, 0x00d3}, {0x0424, 0x00d4}, {0x0425, 0x00d5}, {0x0426, 0x00d6}, {0x0427, 0x00d7}, {0x0428, 0x00d8}, {0x0429, 0x00d9}, {0x042a, 0x00da},
	{0x042b, 0x00db}, {0x042c, 0x00dc}, {0x042d, 0x00dd}, {0x042e, 0x00de}, {0x042f, 0x00df}, {0x0430, 0x00e0}, {0x0431, 0x00e1}, {0x0432, 0x00e2},
	{0x0433, 0x00e3}, {0x0434, 0x00e4}, {0x0435, 0x00e5}, {0x0436, 0x00e6}, {0x0437, 0x00e7}, {0x0438, 0x00e8}, {0x0439, 0x00e9}, {0x043a, 0x00ea},
	{0x043b, 0x00eb}, {0x043c, 0x00ec}, {0x043d, 0x00ed}, {0x043e, 0x00ee}, {0x043f, 0x00ef}, {0x0440, 0x00f0}, {0x0441, 0x00f1}, {0x0442, 0x00f2},
	{0x0443, 0x00f3}, {0x0444, 0x00f4}, {0x0445, 0x00f5}, {0x0446, 0x00f6}, {0x0447, 0x00f7}, {0x0448, 0x00f8}, {0x0449, 0x00f9}, {0x044a, 0x00fa},
	{0x044b, 0x00fb}, {0x044c, 0x00fc}, {0x044d, 0x00fd}, {0x044e, 0x00fe}, {0x044f, 0x00ff}, {0x0451, 0x00b8}, {0x0452, 0x0090}, {0x0453, 0x0083},
	{0x0454, 0x00ba}, {0x0455, 0x00be}, {0x0456, 0x00b3}, {0x0457, 0x00bf}, {0x0458, 0x00bc}, {0x0459, 0x009a}, {0x045a, 0x009c}, {0x045b, 0x009e},
	{0x045c, 0x009d}, {0x045e, 0x00a2}, {0x045f, 0x009f}, {0x0490, 0x00a5}, {0x0491, 0x00b4}, {0x2013, 0x0096}, {0x2014, 0x0097}, {0x2018, 0x0091},
	{0x2019, 0x0092}, {0x201a, 0x0082}, {0x201c, 0x0093}, {0x201d, 0x0094}, {0x201e, 0x0084}, {0x2020, 0x0086}, {0x2021, 0x0087}, {0x2022, 0x0095},
	{0x2026, 0x0085}, {0x2030, 0x0089}, {0x2039, 0x008b}, {0x203a, 0x009b}, {0x20ac, 0x0088}, {0x2116, 0x00b9}, {0x2122, 0x0099}, {0xfffe, 0x0098}
};

int utf8_to_cp1251(char *input, char *output)
{
	char *pti, *pto;
	int size, index, result;

	pti = input;
	pto = output;

	while (*pti)
	{
		size = get_utf8_index(pti, &index);

		if (size > 1)
		{
			result = boundless_binary_interval_index_search(unicode_to_cp1251_table, sizeof(unicode_to_cp1251_table) / sizeof(struct interval_type), index);

			if (result)
			{
				*pto++ = result;
			}
			else
			{
				tintin_printf(gtd->ses, "<118>utf8_to_cp1251: did not find unicode index '0x%04x'", index);

				*pto++ = '?';
			}
			pti += size;
		}
		else
		{
			*pto++ = *pti++;
		}
	}
	*pto = 0;

	return pto - output;
}

int cp1251_to_utf8(char *input, char *output)
{
	unsigned char *pti, *pto;
	int index;

	static int cp1251[256] =
	{
		0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007,
		0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
		0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
		0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
		0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
		0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
		0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
		0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
		0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
		0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
		0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
		0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
		0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
		0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
		0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
		0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x007f,

		0x0402, 0x0403, 0x201a, 0x0453, 0x201e, 0x2026, 0x2020, 0x2021,
		0x20ac, 0x2030, 0x0409, 0x2039, 0x040a, 0x040c, 0x040b, 0x040f,
		0x0452, 0x2018, 0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014,
		0xfffe, 0x2122, 0x0459, 0x203a, 0x045a, 0x045c, 0x045b, 0x045f,
		0x00a0, 0x040e, 0x045e, 0x0408, 0x00a4, 0x0490, 0x00a6, 0x00a7,
		0x0401, 0x00a9, 0x0404, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x0407,
		0x00b0, 0x00b1, 0x0406, 0x0456, 0x0491, 0x00b5, 0x00b6, 0x00b7,
		0x0451, 0x2116, 0x0454, 0x00bb, 0x0458, 0x0405, 0x0455, 0x0457,
		0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417,
		0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f,
		0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427,
		0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f,
		0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437,
		0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f,
		0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447,
		0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f
	};

	pti = (unsigned char *) input;
	pto = (unsigned char *) output;

	while (*pti)
	{
		index = cp1251[*pti];

		if (index < 128)
		{
			*pto++ = index;
		}
		else if (index < 4096)
		{
			*pto++ = 192 + index / 64;
			*pto++ = 128 + index % 64;
		}
		else
		{
			*pto++ = 224 + index / 4096;
			*pto++ = 128 + index / 64 % 64;
			*pto++ = 128 + index % 64;
		}
		pti++;
	}
	*pto = 0;

	return pto - (unsigned char *) output;
}

// KOI-8

struct interval_type unicode_to_koi8_table[] =
{
	{ 0x00A0, 0x9A }, { 0x00A9, 0xBF }, { 0x00B0, 0x9C }, { 0x00B2, 0x9D }, { 0x00B7, 0x9E }, { 0x00F7, 0x9F }, { 0x0401, 0xB3 }, { 0x0410, 0xE1 },
	{ 0x0411, 0xE2 }, { 0x0412, 0xF7 }, { 0x0413, 0xE7 }, { 0x0414, 0xE4 }, { 0x0415, 0xE5 }, { 0x0416, 0xF6 }, { 0x0417, 0xFA }, { 0x0418, 0xE9 },
	{ 0x0419, 0xEA }, { 0x041A, 0xEB }, { 0x041B, 0xEC }, { 0x041C, 0xED }, { 0x041D, 0xEE }, { 0x041E, 0xEF }, { 0x041F, 0xF0 }, { 0x0420, 0xF2 },
	{ 0x0421, 0xF3 }, { 0x0422, 0xF4 }, { 0x0423, 0xF5 }, { 0x0424, 0xE6 }, { 0x0425, 0xE8 }, { 0x0426, 0xE3 }, { 0x0427, 0xFE }, { 0x0428, 0xFB },
	{ 0x0429, 0xFD }, { 0x042A, 0xFF }, { 0x042B, 0xF9 }, { 0x042C, 0xF8 }, { 0x042D, 0xFC }, { 0x042E, 0xE0 }, { 0x042F, 0xF1 }, { 0x0430, 0xC1 },
	{ 0x0431, 0xC2 }, { 0x0432, 0xD7 }, { 0x0433, 0xC7 }, { 0x0434, 0xC4 }, { 0x0435, 0xC5 }, { 0x0436, 0xD6 }, { 0x0437, 0xDA }, { 0x0438, 0xC9 },
	{ 0x0439, 0xCA }, { 0x043A, 0xCB }, { 0x043B, 0xCC }, { 0x043C, 0xCD }, { 0x043D, 0xCE }, { 0x043E, 0xCF }, { 0x043F, 0xD0 }, { 0x0440, 0xD2 },
	{ 0x0441, 0xD3 }, { 0x0442, 0xD4 }, { 0x0443, 0xD5 }, { 0x0444, 0xC6 }, { 0x0445, 0xC8 }, { 0x0446, 0xC3 }, { 0x0447, 0xDE }, { 0x0448, 0xDB },
	{ 0x0449, 0xDD }, { 0x044A, 0xDF }, { 0x044B, 0xD9 }, { 0x044C, 0xD8 }, { 0x044D, 0xDC }, { 0x044E, 0xC0 }, { 0x044F, 0xD1 }, { 0x0451, 0xA3 },
	{ 0x2219, 0x95 }, { 0x221A, 0x96 }, { 0x2248, 0x97 }, { 0x2264, 0x98 }, { 0x2265, 0x99 }, { 0x2320, 0x93 }, { 0x2321, 0x9B }, { 0x2500, 0x80 },
	{ 0x2502, 0x81 }, { 0x250C, 0x82 }, { 0x2510, 0x83 }, { 0x2514, 0x84 }, { 0x2518, 0x85 }, { 0x251C, 0x86 }, { 0x2524, 0x87 }, { 0x252C, 0x88 },
	{ 0x2534, 0x89 }, { 0x253C, 0x8A }, { 0x2550, 0xA0 }, { 0x2551, 0xA1 }, { 0x2552, 0xA2 }, { 0x2553, 0xA4 }, { 0x2554, 0xA5 }, { 0x2555, 0xA6 },
	{ 0x2556, 0xA7 }, { 0x2557, 0xA8 }, { 0x2558, 0xA9 }, { 0x2559, 0xAA }, { 0x255A, 0xAB }, { 0x255B, 0xAC }, { 0x255C, 0xAD }, { 0x255D, 0xAE },
	{ 0x255E, 0xAF }, { 0x255F, 0xB0 }, { 0x2560, 0xB1 }, { 0x2561, 0xB2 }, { 0x2562, 0xB4 }, { 0x2563, 0xB5 }, { 0x2564, 0xB6 }, { 0x2565, 0xB7 },
	{ 0x2566, 0xB8 }, { 0x2567, 0xB9 }, { 0x2568, 0xBA }, { 0x2569, 0xBB }, { 0x256A, 0xBC }, { 0x256B, 0xBD }, { 0x256C, 0xBE }, { 0x2580, 0x8B },
	{ 0x2584, 0x8C }, { 0x2588, 0x8D }, { 0x258C, 0x8E }, { 0x2590, 0x8F }, { 0x2591, 0x90 }, { 0x2592, 0x91 }, { 0x2593, 0x92 }, { 0x25A0, 0x94 }
};

int utf8_to_koi8(char *input, char *output)
{
	char *pti, *pto;
	int size, index, result;

	pti = input;
	pto = output;

	while (*pti)
	{
		size = get_utf8_index(pti, &index);

		if (size > 1)
		{
			result = boundless_binary_interval_index_search(unicode_to_koi8_table, sizeof(unicode_to_koi8_table) / sizeof(struct interval_type), index);

			if (result)
			{
				*pto++ = result;
			}
			else
			{
				tintin_printf(gtd->ses, "<118>utf8_to_koi8: did not find unicode index '0x%04x'", index);

				*pto++ = '?';
			}
			pti += size;
		}
		else
		{
			*pto++ = *pti++;
		}
	}
	*pto = 0;

	return pto - output;
}

int koi8_to_utf8(char *input, char *output)
{
	unsigned char *pti, *pto;
	int index;

	static int koi8[256] =
	{
		        0,      1,      2,      3,      4,      5,      6,      7,
		        8,      9,     10,     11,     12,     13,     14,     15,
		       16,     17,     18,     19,     20,     21,     22,     23,
		       24,     25,     26,     27,     28,     29,     30,     31,
		       32,     33,     34,     35,     36,     37,     38,     39,
		       40,     41,     42,     43,     44,     45,     46,     47,
		       48,     49,     50,     51,     52,     53,     54,     55,
		       56,     57,     58,     59,     60,     61,     62,     63,
		       64,     65,     66,     67,     68,     69,     70,     71,
		       72,     73,     74,     75,     76,     77,     78,     79,
		       80,     81,     82,     83,     84,     85,     86,     87,
		       88,     89,     90,     91,     92,     93,     94,     95,
		       96,     97,     98,     99,    100,    101,    102,    103,
		      104,    105,    106,    107,    108,    109,    110,    111,
		      112,    113,    114,    115,    116,    117,    118,    119,
		      120,    121,    122,    123,    124,    125,    126,    127,
		   0x2500, 0x2502, 0x250C, 0x2510, 0x2514, 0x2518, 0x251C, 0x2524,
		   0x252C, 0x2534, 0x253C, 0x2580, 0x2584, 0x2588, 0x258C, 0x2590,
		   0x2591, 0x2592, 0x2593, 0x2320, 0x25A0, 0x2219, 0x221A, 0x2248,
		   0x2264, 0x2265, 0x00A0, 0x2321, 0x00B0, 0x00B2, 0x00B7, 0x00F7,
		   0x2550, 0x2551, 0x2552, 0x0451, 0x2553, 0x2554, 0x2555, 0x2556,
		   0x2557, 0x2558, 0x2559, 0x255A, 0x255B, 0x255C, 0x255D, 0x255E,
		   0x255F, 0x2560, 0x2561, 0x0401, 0x2562, 0x2563, 0x2564, 0x2565,
		   0x2566, 0x2567, 0x2568, 0x2569, 0x256A, 0x256B, 0x256C, 0x00A9,
		   0x044E, 0x0430, 0x0431, 0x0446, 0x0434, 0x0435, 0x0444, 0x0433,
		   0x0445, 0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E,
		   0x043F, 0x044F, 0x0440, 0x0441, 0x0442, 0x0443, 0x0436, 0x0432,
		   0x044C, 0x044B, 0x0437, 0x0448, 0x044D, 0x0449, 0x0447, 0x044A,
		   0x042E, 0x0410, 0x0411, 0x0426, 0x0414, 0x0415, 0x0424, 0x0413,
		   0x0425, 0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D, 0x041E,
		   0x041F, 0x042F, 0x0420, 0x0421, 0x0422, 0x0423, 0x0416, 0x0412,
		   0x042C, 0x042B, 0x0417, 0x0428, 0x042D, 0x0429, 0x0427, 0x042A
	};

	pti = (unsigned char *) input;
	pto = (unsigned char *) output;

	while (*pti)
	{
		index = koi8[*pti];

		if (index < 128)
		{
			*pto++ = index;
		}
		else if (index < 4096)
		{
			*pto++ = 192 + index / 64;
			*pto++ = 128 + index % 64;
		}
		else
		{
			*pto++ = 224 + index / 4096;
			*pto++ = 128 + index / 64 % 64;
			*pto++ = 128 + index % 64;
		}
		pti++;
	}
	*pto = 0;

	return pto - (unsigned char *) output;
}


// FANSI Special thanks to Tom Dwaggy for the data table

int fansi_to_utf8(char *input, char *output)
{
	unsigned char *pti, *pto;
	int index;

	static int fansi[256] =
	{
		     0, 0x263a,      2, 0x2665, 0x2666, 0x2663, 0x2660,      7,
		     8,      9,     10,     11,     12,     13, 0x266b, 0x263c,
		0x25ba, 0x25c4, 0x2195, 0x203c, 0x00b6, 0x00a7, 0x25ac, 0x21a8,
		0x2191, 0x2193, 0x2192,     27, 0x221f, 0x2194, 0x25b2, 0x25bc,
		    32,     33,     34,     35,     36,     37,     38,     39,
		    40,     41,     42,     43,     44,     45,     46,     47,
		    48,     49,     50,     51,     52,     53,     54,     55,
		    56,     57,     58,     59,     60,     61,     62,     63,
		    64,     65,     66,     67,     68,     69,     70,     71,
		    72,     73,     74,     75,     76,     77,     78,     79,
		    80,     81,     82,     83,     84,     85,     86,     87,
		    88,     89,     90,     91,     92,     93,     94,     95,
		    96,     97,     98,     99,    100,    101,    102,    103,
		   104,    105,    106,    107,    108,    109,    110,    111,
		   112,    113,    114,    115,    116,    117,    118,    119,
		   120,    121,    122,    123,    124,    125,    126, 0x2302,
	        0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7,
	        0x00ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee,	0x00ec, 0x00c4, 0x00c5,
	        0x00c9, 0x00e6, 0x00c6, 0x00f4, 0x00f6,	0x00f2, 0x00fb, 0x00f9,
	        0x00ff, 0x00d6, 0x00dc, 0x00a2, 0x00a3,	0x00a5, 0x20a7, 0x0192,
	        0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x00f1,	0x00d1, 0x00aa, 0x00ba,
	        0x00bf, 0x2310, 0x00ac, 0x00bd, 0x00bc,	0x00a1, 0x00ab, 0x00bb,
	        0x2591, 0x2592, 0x2593, 0x2502, 0x2524,	0x2561, 0x2562, 0x2556,
	        0x2555, 0x2563, 0x2551, 0x2557, 0x255d,	0x255c, 0x255b, 0x2510,
	        0x2514, 0x2534, 0x252c, 0x251c, 0x2500,	0x253c, 0x255e, 0x255f,
	        0x255a, 0x2554, 0x2569, 0x2566, 0x2560,	0x2550, 0x256c, 0x2567,
	        0x2568, 0x2564, 0x2565, 0x2559, 0x2558,	0x2552, 0x2553, 0x256b,
	        0x256a, 0x2518, 0x250c, 0x2588, 0x2584,	0x258c, 0x2590, 0x2580,
	        0x03b1, 0x00df, 0x0393, 0x03c0, 0x03a3,	0x03c3, 0x00b5, 0x03c4,
	        0x03a6, 0x0398, 0x03a9, 0x03b4, 0x221e,	0x03c6, 0x03b5, 0x2229,
	        0x2261, 0x00b1, 0x2265, 0x2264, 0x2320,	0x2321, 0x00f7, 0x2248,
	        0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f,	0x00b2, 0x25a0,    255
	};

	pti = (unsigned char *) input;
	pto = (unsigned char *) output;

	while (*pti)
	{
		index = fansi[*pti];

		if (index < 128)
		{
			*pto++ = index;
		}
		else if (index < 4096)
		{
			*pto++ = 192 + index / 64;
			*pto++ = 128 + index % 64;
		}
		else
		{
			*pto++ = 224 + index / 4096;
			*pto++ = 128 + index / 64 % 64;
			*pto++ = 128 + index % 64;
		}
		pti++;
	}
	*pto = 0;

	return pto - (unsigned char *) output;
}

// EUC - Extended Unix Codes

int is_euc_head(struct session *ses, char *str)
{
	unsigned char *ptu = (unsigned char *) str;

	if (ptu[0] > 128 && ptu[0] < 255)
	{
		return 1;
	}
	return 0;
}

int get_euc_size(struct session *ses, char *str)
{
	unsigned char *ptu = (unsigned char *) str;

	if (HAS_BIT(ses->charset, CHARSET_FLAG_BIG5))
	{
		if (ptu[0] > 128 && ptu[0] < 255 && ptu[1] >= 64 && (ptu[1] < 127 || ptu[1] >= 160) && ptu[1] < 255)
		{
			return 2;
		}
		return 1;
	}

	if (HAS_BIT(ses->charset, CHARSET_FLAG_CP949))
	{
		if (ptu[0] > 128 && ptu[0] < 255 && ptu[1] > 64 && ptu[1] < 255)
		{
			return 2;
		}
		return 1;
	}

	if (HAS_BIT(ses->charset, CHARSET_FLAG_GBK1))
	{
		if (ptu[0] > 128 && ptu[0] < 255)
		{
			if (ptu[1] >= 64 && ptu[1] < 255)
			{
				return 2;
			}
			if (ptu[1] < 48)
			{
				return 1;
			}
			if (ptu[2] > 128 && ptu[2] < 255 && ptu[3] >= 48 && ptu[3] < 64)
			{
				return 4;
			}
			return 2;
		}
		return 1;
	}

	tintin_printf2(ses, "debug: unknown charset");
	return 1;
}

int get_euc_width(struct session *ses, char *str, int *width)
{
	unsigned char *ptu = (unsigned char *) str;

	if (HAS_BIT(ses->charset, CHARSET_FLAG_BIG5))
	{
		if (ptu[0] > 128 && ptu[0] < 255 && ptu[1] >= 64 && (ptu[1] < 127 || ptu[1] >= 160) && ptu[1] < 255)
		{
			*width = 2;
			return 2;
		}
		*width = 1;
		return 1;
	}

	if (HAS_BIT(ses->charset, CHARSET_FLAG_CP949))
	{
		if (ptu[0] > 128 && ptu[0] < 255 && ptu[1] > 64 && ptu[1] < 255)
		{
			*width = 2;
			return 2;
		}
		*width = 1;
		return 1;
	}

	if (HAS_BIT(ses->charset, CHARSET_FLAG_GBK1))
	{
		if (ptu[0] > 128 && ptu[0] < 255)
		{
			*width = 2;
			if (ptu[1] >= 64 && ptu[1] < 255)
			{
				return 2;
			}
			if (ptu[1] < 48)
			{
				return 1;
			}
			if (ptu[2] > 128 && ptu[2] < 255 && ptu[3] >= 48 && ptu[3] < 64)
			{
				return 4;
			}
			return 2;
		}
		*width = 1;
		return 1;
	}

	tintin_printf2(ses, "debug: unknown charset");
	*width = 1;
	return 1;
}

// BIG-5

unsigned short unicode_to_big5_keys[] = {   129,  161,  162,  163,  164,  165,  166,  167,  168,  169,  170,  173,  174,  175,  176,  177,  178,  179,  180,  181,  182,  183,  184,  185,  186,  188,  189,  190,  191,  192,  193,  194,  195,  196,  197,  198,  199,  200,  201,  202,  203,  204,  205,  206,  207,  208,  209,  210,  211,  212,  213,  214,  215,  216,  217,  218,  219,  220,  221,  222,  223,  224,  225,  226,  227,  228,  229,  230,  231,  232,  233,  234,  235,  236,  237,  238,  239,  240,  241,  242,  243,  244,  245,  246,  247,  248,  249,  250,  251,  252,  253,  254,  255,  257,  275,  283,  299,  330,  333,  339,  363,  402,  462,  464,  466,  468,  470,  472,  474,  476,  593,  594,  596,  601,  602,  603,  604,  605,  609,  618,  623,  629,  632,  643,  647,  650,  652,  658,  676,  679,  710,  711,  713,  714,  715,  717,  719,  729,  913,  914,  915,  916,  917,  918,  919,  920,  921,  922,  923,  924,  925,  926,  927,  928,  929,  931,  932,  933,  934,  935,  936,  937,  945,  946,  947,  948,  949,  950,  951,  952,  953,  954,  955,  956,  957,  958,  959,  960,  961,  963,  964,  965,  966,  967,  968,  969, 1025, 1040, 1041, 1042, 1043, 1044, 1045, 1046, 1047, 1048, 1049, 1050, 1051, 1052, 1053, 1054, 1055, 1056, 1057, 1058, 1059, 1060, 1061, 1062, 1063, 1064, 1065, 1066, 1067, 1068, 1069, 1070, 1071, 1072, 1073, 1074, 1075, 1076, 1077, 1078, 1079, 1080, 1081, 1082, 1083, 1084, 1085, 1086, 1087, 1088, 1089, 1090, 1091, 1092, 1093, 1094, 1095, 1096, 1097, 1098, 1099, 1100, 1101, 1102, 1103, 1105, 7739, 7747, 7753, 7922, 7923, 8208, 8211, 8212, 8213, 8214, 8216, 8217, 8220, 8221, 8224, 8225, 8226, 8228, 8229, 8230, 8231, 8240, 8242, 8243, 8245, 8251, 8252, 8254, 8364, 8451, 8453, 8457, 8470, 8478, 8481, 8482, 8491, 8507, 8531, 8532, 8533, 8534, 8535, 8536, 8537, 8538, 8539, 8540, 8541, 8542, 8544, 8545, 8546, 8547, 8548, 8549, 8550, 8551, 8552, 8553, 8560, 8561, 8562, 8563, 8564, 8565, 8566, 8567, 8568, 8569, 8592, 8593, 8594, 8595, 8596, 8597, 8598, 8599, 8600, 8601, 8616, 8626, 8632, 8633, 8651, 8660, 8678, 8679, 8680, 8681, 8704, 8706, 8707, 8711, 8712, 8715, 8719, 8721, 8725, 8726, 8728, 8730, 8734, 8735, 8736, 8739, 8741, 8743, 8744, 8745, 8746, 8747, 8748, 8750, 8756, 8757, 8758, 8759, 8786, 8800, 8801, 8803, 8806, 8807, 8834, 8835, 8838, 8839, 8853, 8857, 8869, 8895, 8962, 8967, 8978, 9216, 9217, 9218, 9219, 9220, 9221, 9222, 9223, 9224, 9225, 9226, 9227, 9228, 9229, 9230, 9231, 9232, 9233, 9234, 9235, 9236, 9237, 9238, 9239, 9240, 9241, 9242, 9243, 9244, 9245, 9246, 9247, 9249, 9312, 9313, 9314, 9315, 9316, 9317, 9318, 9319, 9320, 9321, 9322, 9323, 9324, 9325, 9326, 9327, 9328, 9329, 9330, 9331, 9332, 9333, 9334, 9335, 9336, 9337, 9338, 9339, 9340, 9341, 9342, 9343, 9344, 9345, 9346, 9347, 9348, 9349, 9350, 9351, 9352, 9353, 9354, 9355, 9356, 9357, 9358, 9359, 9360, 9361, 9362, 9363, 9364, 9365, 9366, 9367, 9368, 9369, 9370, 9371, 9372, 9373, 9374, 9375, 9376, 9377, 9378, 9379, 9380, 9381, 9382, 9383, 9384, 9385, 9386, 9387, 9388, 9389, 9390, 9391, 9392, 9393, 9394, 9395, 9396, 9397, 9398, 9399, 9400, 9401, 9402, 9403, 9404, 9405, 9406, 9407, 9408, 9409, 9410, 9411, 9412, 9413, 9414, 9415, 9416, 9417, 9418, 9419, 9420, 9421, 9422, 9423, 9424, 9425, 9426, 9427, 9428, 9429, 9430, 9431, 9432, 9433, 9434, 9435, 9436, 9437, 9438, 9439, 9440, 9441, 9442, 9443, 9444, 9445, 9446, 9447, 9448, 9449, 9450, 9472, 9473, 9474, 9475, 9476, 9477, 9478, 9479, 9480, 9481, 9482, 9483, 9484, 9485, 9486, 9487, 9488, 9489, 9490, 9491, 9492, 9493, 9494, 9495, 9496, 9497, 9498, 9499, 9500, 9501, 9502, 9503, 9504, 9505, 9506, 9507, 9508, 9509, 9510, 9511, 9512, 9513, 9514, 9515, 9516, 9517, 9518, 9519, 9520, 9521, 9522, 9523, 9524, 9525, 9526, 9527, 9528, 9529, 9530, 9531, 9532, 9533, 9534, 9535, 9536, 9537, 9538, 9539, 9540, 9541, 9542, 9543, 9544, 9545, 9546, 9547, 9550, 9552, 9553, 9554, 9555, 9556, 9557, 9558, 9559, 9560, 9561, 9562, 9563, 9564, 9565, 9566, 9567, 9568, 9569, 9570, 9571, 9572, 9573, 9574, 9575, 9576, 9577, 9578, 9579, 9580, 9581, 9582, 9583, 9584, 9585, 9586, 9587, 9588, 9601, 9602, 9603, 9604, 9605, 9606, 9607, 9608, 9609, 9610, 9611, 9612, 9613, 9614, 9615, 9619, 9620, 9621, 9632, 9633, 9650, 9651, 9654, 9660, 9661, 9668, 9670, 9671, 9675, 9678, 9679, 9688, 9689, 9698, 9699, 9700, 9701, 9711, 9729, 9730, 9733, 9737, 9742, 9745, 9746, 9755, 9756, 9758, 9759, 9760, 9775, 9776, 9777, 9778, 9779, 9780, 9781, 9782, 9783, 9785, 9786, 9787, 9788, 9789, 9792, 9793, 9794, 9824, 9825, 9826, 9827, 9828, 9829, 9830, 9831, 9834, 9836, 9837, 9839, 9986, 9992, 9993,10003,10005,10017,10025,10045,10063,10102,10103,10104,10105,10106,10107,10108,10109,10110,10111,10112,10113,10114,10115,10116,10117,10118,10119,10120,10121,10122,10123,10124,10125,10126,10127,10128,10129,10130,10131,11088,11904,11911,11914,11916,11925,11927,11940,11950,12034,12035,12037,12039,12044,12045,12046,12051,12054,12057,12059,12066,12071,12078,12083,12084,12085,12089,12090,12097,12102,12135,12136,12193,12202,12288,12289,12290,12291,12293,12294,12295,12296,12297,12298,12299,12300,12301,12302,12303,12304,12305,12306,12307,12308,12309,12310,12311,12316,12317,12318,12321,12322,12323,12324,12325,12326,12327,12328,12329,12336,12344,12345,12346,12353,12354,12355,12356,12357,12358,12359,12360,12361,12362,12363,12364,12365,12366,12367,12368,12369,12370,12371,12372,12373,12374,12375,12376,12377,12378,12379,12380,12381,12382,12383,12384,12385,12386,12387,12388,12389,12390,12391,12392,12393,12394,12395,12396,12397,12398,12399,12400,12401,12402,12403,12404,12405,12406,12407,12408,12409,12410,12411,12412,12413,12414,12415,12416,12417,12418,12419,12420,12421,12422,12423,12424,12425,12426,12427,12428,12429,12430,12431,12432,12433,12434,12435,12436,12443,12444,12445,12446,12449,12450,12451,12452,12453,12454,12455,12456,12457,12458,12459,12460,12461,12462,12463,12464,12465,12466,12467,12468,12469,12470,12471,12472,12473,12474,12475,12476,12477,12478,12479,12480,12481,12482,12483,12484,12485,12486,12487,12488,12489,12490,12491,12492,12493,12494,12495,12496,12497,12498,12499,12500,12501,12502,12503,12504,12505,12506,12507,12508,12509,12510,12511,12512,12513,12514,12515,12516,12517,12518,12519,12520,12521,12522,12523,12524,12525,12526,12527,12528,12529,12530,12531,12532,12533,12534,12535,12536,12537,12538,12539,12540,12541,12542,12549,12550,12551,12552,12553,12554,12555,12556,12557,12558,12559,12560,12561,12562,12563,12564,12565,12566,12567,12568,12569,12570,12571,12572,12573,12574,12575,12576,12577,12578,12579,12580,12581,12582,12583,12584,12585,12690,12691,12692,12693,12694,12695,12696,12697,12698,12699,12700,12701,12702,12703,12751,12832,12833,12834,12835,12836,12837,12838,12839,12840,12841,12842,12843,12844,12845,12846,12847,12848,12849,12850,12851,12852,12853,12854,12855,12856,12857,12858,12859,12860,12861,12862,12863,12864,12865,12866,12867,12928,12929,12930,12931,12932,12933,12934,12935,12936,12937,12938,12939,12940,12941,12942,12943,12944,12945,12946,12947,12948,12949,12950,12951,12952,12953,12954,12955,12956,12957,12958,12959,12960,12961,12962,12963,12964,12965,12966,12967,12968,12969,12970,12971,12972,12973,12974,12975,12976,13004,13006,13170,13171,13173,13174,13175,13178,13184,13185,13187,13188,13189,13190,13191,13194,13195,13198,13199,13200,13209,13210,13212,13213,13214,13217,13221,13225,13232,13233,13235,13236,13237,13239,13240,13241,13242,13243,13245,13246,13247,13250,13251,13252,13253,13256,13257,13258,13259,13260,13261,13262,13263,13264,13265,13266,13267,13268,13269,13271,13272,13274,13275,13276,13277,13316,13393,13535,13649,13774,13778,13782,13811,13822,13869,13908,14023,14776,14890,14935,14940,15341,15569,15820,16089,16485,16607,17462,17553,17591,17642,17659,17935,18825,19326,19460,19463,19515,19581,19968,19969,19971,19972,19974,19975,19976,19977,19978,19979,19980,19981,19982,19983,19984,19985,19987,19988,19989,19990,19991,19992,19993,19994,19995,19996,19997,19998,19999,20001,20002,20004,20005,20006,20007,20008,20010,20011,20012,20013,20014,20015,20016,20017,20018,20019,20020,20022,20023,20024,20025,20026,20027,20028,20029,20030,20031,20034,20035,20036,20037,20039,20040,20041,20043,20044,20045,20046,20047,20048,20050,20051,20052,20053,20054,20055,20056,20057,20058,20059,20060,20061,20062,20063,20064,20065,20066,20070,20073,20074,20080,20081,20083,20088,20094,20095,20096,20097,20098,20099,20100,20101,20102,20103,20104,20105,20106,20107,20108,20109,20110,20111,20113,20114,20115,20116,20117,20120,20121,20122,20123,20124,20126,20127,20128,20129,20130,20131,20132,20133,20134,20135,20136,20137,20139,20140,20141,20142,20144,20146,20147,20149,20150,20151,20153,20154,20155,20156,20159,20160,20161,20162,20163,20164,20165,20166,20167,20168,20169,20170,20171,20173,20174,20175,20177,20179,20180,20181,20182,20183,20184,20185,20186,20188,20189,20190,20191,20193,20195,20196,20197,20200,20201,20202,20203,20204,20205,20206,20208,20209,20210,20211,20212,20213,20214,20215,20216,20219,20220,20221,20223,20224,20225,20226,20227,20228,20229,20232,20233,20234,20235,20237,20238,20239,20240,20241,20242,20243,20244,20245,20247,20248,20249,20250,20251,20252,20253,20254,20255,20256,20258,20260,20261,20262,20263,20264,20265,20266,20267,20268,20269,20271,20272,20274,20275,20276,20278,20279,20280,20281,20282,20283,20284,20285,20286,20287,20289,20290,20291,20293,20294,20295,20296,20297,20299,20300,20301,20302,20303,20304,20305,20306,20307,20308,20309,20310,20311,20312,20313,20314,20315,20316,20317,20318,20319,20320,20321,20322,20323,20324,20325,20327,20329,20330,20331,20332,20334,20335,20336,20338,20339,20340,20341,20342,20343,20344,20345,20346,20347,20348,20349,20350,20351,20352,20353,20354,20355,20356,20357,20358,20359,20360,20361,20362,20363,20365,20367,20368,20369,20370,20372,20373,20374,20375,20376,20378,20379,20380,20381,20382,20384,20385,20386,20387,20389,20390,20391,20392,20393,20394,20395,20396,20397,20398,20399,20400,20402,20403,20404,20405,20406,20407,20409,20410,20411,20413,20415,20416,20417,20418,20419,20420,20421,20423,20424,20425,20426,20427,20428,20429,20430,20431,20432,20433,20435,20436,20438,20439,20440,20441,20442,20443,20444,20445,20446,20447,20448,20449,20451,20452,20453,20454,20456,20457,20458,20460,20461,20462,20463,20464,20465,20466,20467,20468,20469,20470,20471,20472,20473,20474,20477,20478,20479,20480,20483,20485,20486,20487,20488,20489,20491,20492,20493,20494,20495,20497,20498,20499,20500,20501,20502,20503,20504,20505,20506,20507,20508,20510,20511,20512,20513,20514,20515,20516,20517,20518,20519,20520,20521,20522,20523,20524,20525,20526,20527,20528,20529,20531,20532,20533,20534,20535,20537,20538,20539,20540,20542,20544,20545,20546,20547,20549,20550,20551,20552,20553,20554,20555,20556,20557,20558,20559,20560,20561,20563,20565,20566,20567,20568,20570,20571,20572,20573,20574,20575,20576,20577,20578,20579,20580,20581,20582,20584,20585,20586,20587,20588,20589,20590,20591,20592,20594,20595,20596,20597,20598,20599,20600,20602,20603,20605,20606,20607,20608,20609,20610,20611,20613,20615,20616,20619,20620,20621,20622,20624,20625,20626,20628,20629,20630,20632,20633,20634,20635,20636,20637,20638,20642,20643,20645,20646,20647,20648,20649,20652,20653,20654,20655,20656,20657,20658,20659,20660,20661,20662,20663,20664,20666,20667,20668,20669,20670,20671,20673,20674,20676,20677,20678,20679,20680,20681,20682,20683,20685,20686,20687,20688,20689,20691,20692,20693,20694,20695,20696,20697,20698,20699,20701,20702,20703,20704,20705,20707,20708,20709,20710,20711,20712,20713,20714,20716,20717,20718,20719,20720,20721,20723,20724,20725,20726,20728,20729,20731,20732,20733,20734,20735,20736,20737,20738,20739,20740,20741,20742,20743,20744,20745,20746,20747,20748,20749,20750,20752,20753,20754,20755,20756,20757,20758,20759,20760,20762,20764,20767,20768,20769,20770,20772,20773,20774,20777,20778,20779,20781,20782,20784,20785,20786,20787,20788,20789,20791,20792,20793,20794,20795,20796,20797,20799,20800,20801,20803,20804,20805,20806,20807,20808,20809,20810,20811,20812,20813,20814,20816,20817,20818,20820,20821,20822,20823,20825,20826,20827,20828,20829,20830,20831,20832,20833,20834,20835,20836,20837,20839,20840,20841,20842,20843,20844,20845,20846,20848,20849,20851,20852,20853,20854,20855,20856,20857,20859,20860,20861,20864,20865,20866,20869,20870,20871,20872,20873,20874,20876,20877,20879,20880,20881,20882,20883,20884,20885,20886,20887,20888,20889,20890,20891,20892,20893,20894,20896,20898,20900,20901,20902,20903,20904,20905,20906,20907,20908,20910,20911,20912,20913,20914,20915,20916,20917,20918,20919,20920,20921,20923,20924,20925,20926,20928,20931,20932,20933,20934,20935,20936,20937,20938,20939,20940,20941,20942,20943,20944,20945,20946,20947,20948,20950,20951,20952,20955,20956,20957,20958,20959,20960,20961,20962,20964,20966,20967,20969,20970,20971,20972,20973,20975,20976,20977,20979,20980,20981,20982,20984,20985,20986,20987,20988,20989,20990,20991,20992,20993,20994,20995,20996,20997,20998,20999,21000,21001,21002,21003,21004,21005,21006,21008,21009,21010,21011,21012,21013,21014,21015,21016,21017,21018,21019,21020,21021,21022,21023,21024,21025,21027,21028,21029,21030,21031,21032,21033,21034,21035,21037,21038,21040,21041,21042,21043,21044,21045,21046,21047,21048,21049,21050,21051,21052,21053,21055,21056,21057,21058,21059,21060,21062,21063,21065,21066,21067,21068,21069,21070,21071,21072,21073,21074,21076,21077,21078,21079,21081,21082,21083,21084,21085,21086,21087,21088,21089,21090,21091,21092,21093,21095,21096,21097,21098,21099,21100,21101,21102,21103,21104,21105,21106,21107,21108,21109,21111,21112,21113,21114,21115,21116,21117,21119,21120,21121,21122,21123,21124,21127,21128,21129,21130,21131,21132,21133,21135,21136,21137,21138,21139,21140,21142,21143,21144,21145,21146,21147,21148,21149,21150,21151,21152,21153,21154,21155,21156,21158,21160,21161,21162,21163,21164,21165,21166,21167,21169,21170,21171,21172,21173,21177,21179,21180,21182,21183,21184,21185,21186,21187,21189,21191,21193,21195,21196,21197,21200,21201,21202,21203,21205,21206,21207,21208,21209,21211,21213,21214,21215,21216,21217,21218,21219,21220,21222,21223,21225,21227,21231,21232,21233,21234,21235,21236,21237,21239,21240,21241,21242,21243,21244,21246,21247,21248,21249,21250,21251,21253,21254,21255,21256,21257,21258,21259,21261,21262,21263,21264,21265,21266,21269,21270,21271,21273,21274,21276,21277,21279,21280,21281,21282,21283,21284,21286,21287,21290,21292,21293,21294,21295,21296,21297,21298,21299,21300,21303,21304,21305,21306,21307,21308,21309,21310,21311,21312,21313,21314,21315,21316,21317,21318,21319,21320,21321,21322,21324,21325,21326,21327,21329,21330,21331,21332,21333,21334,21335,21336,21338,21340,21341,21342,21343,21344,21345,21346,21347,21348,21350,21351,21353,21355,21356,21357,21358,21359,21360,21361,21362,21363,21364,21365,21367,21368,21369,21370,21371,21372,21373,21374,21375,21378,21380,21381,21382,21385,21386,21387,21388,21389,21390,21391,21394,21395,21396,21397,21398,21399,21400,21401,21402,21404,21405,21406,21407,21408,21410,21411,21412,21413,21414,21415,21416,21417,21418,21419,21420,21421,21422,21424,21426,21427,21428,21430,21433,21435,21439,21441,21442,21443,21445,21448,21449,21450,21451,21452,21453,21454,21456,21457,21458,21460,21462,21463,21464,21465,21466,21467,21469,21471,21472,21473,21474,21475,21476,21477,21478,21480,21481,21482,21483,21484,21485,21486,21487,21488,21489,21490,21491,21493,21494,21495,21496,21497,21498,21499,21500,21501,21502,21505,21507,21508,21510,21511,21512,21513,21514,21515,21516,21517,21518,21519,21520,21521,21522,21523,21524,21525,21526,21527,21528,21529,21530,21531,21532,21533,21534,21535,21536,21537,21539,21540,21541,21542,21543,21544,21545,21546,21547,21548,21549,21550,21551,21552,21553,21554,21555,21556,21557,21558,21559,21560,21561,21563,21564,21565,21566,21568,21569,21570,21571,21573,21574,21575,21576,21577,21578,21579,21581,21582,21583,21584,21585,21586,21587,21588,21589,21590,21591,21592,21593,21595,21596,21599,21600,21601,21602,21603,21604,21605,21606,21607,21608,21609,21610,21611,21612,21613,21615,21616,21617,21618,21619,21620,21621,21622,21623,21624,21626,21627,21628,21629,21630,21631,21632,21633,21634,21636,21637,21638,21639,21640,21642,21643,21644,21645,21646,21647,21648,21649,21650,21651,21652,21653,21654,21655,21656,21657,21658,21659,21660,21661,21662,21664,21665,21666,21667,21668,21669,21670,21671,21672,21673,21674,21675,21676,21677,21678,21679,21680,21681,21682,21683,21684,21686,21687,21688,21689,21690,21691,21692,21693,21694,21695,21696,21697,21698,21699,21700,21701,21702,21703,21704,21705,21707,21708,21709,21710,21711,21712,21713,21714,21715,21716,21717,21718,21719,21720,21721,21722,21724,21725,21726,21727,21728,21729,21730,21731,21732,21733,21734,21735,21736,21737,21738,21739,21741,21742,21743,21745,21746,21747,21751,21752,21754,21755,21756,21757,21759,21761,21762,21763,21764,21765,21766,21767,21768,21769,21770,21771,21772,21773,21774,21775,21776,21777,21778,21779,21780,21782,21783,21784,21786,21787,21790,21792,21794,21795,21796,21797,21798,21799,21800,21802,21803,21804,21805,21806,21807,21808,21809,21810,21811,21812,21813,21814,21815,21816,21817,21819,21820,21822,21823,21824,21825,21827,21828,21829,21830,21831,21832,21833,21834,21835,21836,21837,21838,21839,21840,21841,21842,21843,21845,21846,21847,21852,21853,21854,21855,21857,21858,21859,21860,21861,21862,21863,21865,21866,21867,21868,21869,21870,21873,21874,21875,21877,21878,21879,21880,21881,21883,21884,21885,21886,21887,21888,21889,21890,21891,21892,21894,21895,21896,21897,21898,21899,21900,21901,21902,21903,21904,21905,21906,21907,21908,21909,21912,21913,21914,21916,21917,21918,21919,21921,21922,21923,21924,21925,21926,21927,21928,21929,21930,21931,21932,21933,21934,21936,21937,21938,21939,21940,21941,21942,21943,21945,21946,21947,21948,21949,21950,21951,21952,21953,21954,21955,21956,21957,21958,21959,21960,21961,21962,21963,21964,21965,21966,21967,21968,21969,21970,21971,21972,21973,21974,21975,21976,21977,21978,21979,21980,21981,21982,21983,21985,21986,21987,21988,21989,21990,21991,21992,21993,21994,21995,21996,21999,22000,22001,22002,22003,22005,22006,22007,22009,22010,22011,22012,22013,22014,22015,22016,22017,22018,22020,22021,22022,22024,22025,22028,22029,22030,22031,22032,22033,22034,22035,22036,22037,22038,22039,22040,22043,22044,22045,22046,22047,22048,22049,22050,22051,22052,22053,22055,22057,22058,22060,22061,22062,22063,22064,22065,22066,22067,22068,22069,22070,22071,22072,22073,22074,22075,22077,22078,22079,22080,22081,22082,22083,22085,22086,22088,22089,22090,22092,22093,22094,22095,22096,22098,22099,22100,22103,22104,22105,22106,22107,22108,22109,22110,22112,22113,22114,22115,22116,22117,22118,22120,22121,22122,22123,22124,22125,22126,22127,22128,22129,22130,22131,22132,22134,22135,22136,22137,22138,22139,22140,22142,22143,22144,22145,22146,22147,22148,22149,22150,22151,22153,22154,22155,22156,22157,22158,22159,22160,22162,22163,22164,22165,22167,22168,22169,22170,22172,22173,22174,22175,22176,22177,22178,22179,22180,22181,22182,22183,22184,22186,22187,22188,22189,22190,22191,22193,22194,22195,22196,22197,22198,22199,22201,22204,22205,22206,22207,22208,22209,22210,22211,22213,22214,22216,22217,22218,22219,22220,22221,22222,22225,22227,22228,22230,22231,22232,22234,22235,22237,22238,22239,22240,22241,22242,22243,22244,22245,22247,22250,22251,22253,22254,22255,22256,22257,22258,22259,22260,22261,22263,22265,22266,22267,22269,22270,22271,22272,22273,22274,22275,22276,22278,22279,22280,22281,22282,22283,22284,22285,22287,22290,22291,22292,22293,22294,22296,22298,22299,22300,22301,22302,22303,22304,22306,22307,22310,22311,22312,22313,22314,22316,22317,22318,22319,22320,22322,22323,22324,22327,22328,22329,22330,22331,22333,22334,22335,22336,22337,22338,22339,22341,22342,22343,22345,22346,22347,22348,22349,22350,22351,22352,22353,22354,22356,22359,22361,22362,22363,22364,22365,22366,22367,22368,22369,22370,22372,22373,22374,22375,22376,22377,22378,22379,22381,22383,22384,22385,22386,22387,22388,22389,22390,22391,22394,22395,22396,22397,22398,22399,22400,22402,22403,22404,22405,22406,22408,22409,22410,22411,22412,22413,22415,22416,22418,22419,22420,22421,22423,22424,22425,22426,22427,22428,22429,22430,22431,22432,22433,22434,22435,22436,22437,22438,22439,22441,22442,22443,22444,22445,22446,22448,22450,22451,22452,22453,22454,22456,22457,22458,22459,22460,22461,22462,22463,22464,22465,22466,22467,22468,22470,22471,22472,22475,22476,22478,22479,22480,22482,22483,22484,22485,22486,22487,22488,22489,22490,22492,22493,22494,22495,22496,22497,22498,22499,22500,22501,22502,22503,22505,22508,22509,22510,22511,22512,22513,22514,22515,22516,22517,22518,22519,22520,22521,22522,22523,22524,22525,22526,22528,22529,22530,22531,22532,22533,22534,22535,22536,22537,22538,22539,22540,22541,22542,22544,22545,22546,22548,22549,22553,22555,22556,22557,22558,22560,22561,22562,22563,22564,22565,22566,22567,22568,22569,22570,22572,22573,22574,22575,22576,22577,22578,22579,22580,22581,22582,22583,22584,22585,22586,22587,22589,22591,22592,22593,22596,22599,22600,22601,22602,22603,22604,22605,22606,22607,22609,22610,22611,22612,22613,22615,22616,22617,22618,22619,22620,22621,22622,22623,22626,22627,22628,22629,22632,22633,22635,22636,22637,22639,22640,22641,22642,22643,22644,22645,22646,22649,22650,22651,22652,22653,22654,22655,22656,22657,22658,22659,22661,22662,22663,22664,22665,22666,22667,22670,22671,22672,22673,22674,22675,22676,22679,22680,22681,22682,22684,22685,22686,22687,22688,22689,22691,22693,22694,22695,22696,22697,22698,22699,22700,22702,22703,22704,22705,22706,22707,22709,22710,22712,22713,22714,22715,22716,22717,22718,22719,22721,22722,22725,22726,22727,22728,22729,22730,22731,22732,22734,22735,22736,22737,22738,22739,22740,22741,22742,22743,22744,22745,22746,22747,22748,22749,22750,22751,22752,22754,22755,22756,22757,22759,22760,22761,22763,22764,22766,22767,22768,22769,22770,22771,22772,22774,22775,22777,22778,22779,22780,22781,22782,22783,22786,22787,22788,22789,22790,22791,22793,22794,22795,22796,22797,22798,22799,22800,22801,22802,22804,22805,22806,22807,22808,22809,22810,22811,22812,22813,22815,22816,22818,22820,22821,22823,22825,22826,22827,22828,22829,22830,22831,22833,22834,22836,22839,22840,22841,22842,22844,22846,22848,22849,22850,22852,22853,22855,22856,22857,22858,22859,22862,22863,22864,22865,22867,22868,22869,22870,22871,22872,22874,22875,22876,22877,22880,22881,22882,22883,22885,22887,22888,22889,22890,22891,22892,22893,22894,22896,22897,22898,22899,22900,22901,22902,22903,22904,22905,22907,22908,22909,22910,22911,22912,22913,22914,22915,22916,22917,22918,22919,22920,22921,22922,22925,22926,22927,22928,22930,22931,22932,22934,22935,22936,22937,22938,22939,22941,22942,22943,22944,22945,22946,22947,22948,22949,22950,22951,22952,22953,22954,22955,22956,22958,22959,22960,22961,22962,22963,22964,22965,22966,22967,22968,22969,22970,22971,22972,22973,22974,22975,22976,22977,22979,22980,22981,22982,22983,22984,22985,22986,22987,22988,22989,22990,22991,22992,22993,22994,22995,22996,22998,22999,23000,23001,23002,23003,23004,23005,23006,23008,23009,23011,23012,23013,23014,23016,23017,23018,23019,23020,23021,23022,23023,23024,23025,23026,23027,23028,23029,23030,23031,23032,23033,23034,23035,23036,23037,23038,23039,23040,23041,23042,23043,23044,23045,23046,23047,23048,23049,23050,23051,23052,23053,23055,23057,23058,23059,23061,23062,23063,23064,23065,23066,23067,23068,23070,23071,23072,23073,23075,23076,23077,23079,23081,23082,23083,23084,23085,23086,23087,23089,23090,23091,23092,23093,23094,23095,23096,23097,23100,23101,23102,23104,23105,23106,23107,23108,23109,23110,23111,23112,23113,23114,23116,23117,23120,23121,23122,23123,23124,23125,23126,23127,23128,23129,23130,23131,23132,23133,23134,23135,23136,23137,23138,23139,23140,23141,23142,23143,23144,23145,23146,23147,23148,23149,23150,23152,23153,23156,23157,23158,23159,23160,23161,23162,23163,23164,23165,23166,23167,23169,23170,23171,23172,23174,23176,23178,23179,23180,23182,23183,23184,23185,23186,23187,23188,23189,23190,23191,23193,23194,23195,23196,23197,23198,23199,23200,23201,23202,23204,23205,23206,23207,23209,23210,23211,23212,23214,23215,23216,23217,23218,23219,23220,23221,23222,23223,23224,23225,23226,23227,23228,23229,23230,23231,23232,23233,23234,23235,23236,23238,23239,23240,23241,23242,23243,23244,23245,23246,23247,23248,23249,23250,23251,23252,23253,23254,23255,23256,23257,23258,23259,23260,23261,23262,23263,23264,23265,23266,23267,23268,23269,23270,23272,23273,23274,23275,23276,23277,23278,23280,23281,23282,23283,23284,23285,23286,23287,23288,23289,23290,23291,23293,23294,23295,23297,23298,23299,23301,23303,23304,23305,23307,23308,23309,23311,23312,23313,23315,23316,23317,23318,23319,23321,23322,23323,23325,23326,23327,23328,23329,23330,23331,23332,23333,23334,23335,23336,23338,23339,23340,23341,23342,23343,23344,23346,23348,23350,23351,23352,23356,23357,23358,23359,23360,23361,23363,23364,23365,23366,23367,23368,23370,23371,23372,23373,23374,23375,23376,23377,23379,23380,23381,23382,23383,23384,23385,23386,23387,23388,23389,23391,23394,23395,23396,23397,23398,23400,23401,23402,23403,23404,23405,23406,23408,23409,23410,23411,23412,23413,23414,23415,23416,23418,23419,23420,23421,23423,23424,23425,23426,23427,23428,23429,23431,23432,23433,23435,23436,23437,23438,23439,23440,23442,23443,23445,23446,23447,23448,23449,23450,23451,23452,23453,23454,23455,23456,23457,23458,23459,23460,23461,23462,23463,23464,23466,23467,23468,23469,23470,23472,23475,23476,23477,23478,23479,23480,23481,23485,23486,23487,23488,23489,23490,23491,23492,23493,23494,23495,23497,23498,23499,23500,23501,23502,23504,23505,23506,23507,23508,23509,23510,23511,23512,23513,23515,23517,23518,23519,23520,23521,23522,23523,23524,23525,23526,23527,23528,23529,23530,23531,23532,23534,23535,23536,23537,23538,23539,23541,23542,23544,23545,23546,23547,23548,23550,23551,23553,23554,23555,23556,23557,23558,23559,23560,23561,23562,23563,23564,23565,23566,23567,23568,23569,23570,23571,23572,23573,23574,23576,23578,23580,23581,23582,23583,23584,23586,23587,23588,23589,23591,23592,23594,23596,23597,23600,23601,23603,23604,23607,23608,23609,23610,23611,23612,23613,23614,23615,23616,23617,23618,23620,23621,23622,23623,23624,23625,23626,23627,23628,23629,23630,23631,23632,23633,23635,23636,23637,23638,23640,23641,23644,23645,23646,23648,23649,23650,23651,23652,23653,23654,23655,23656,23657,23658,23660,23661,23662,23663,23665,23667,23668,23670,23672,23673,23674,23675,23676,23678,23679,23681,23682,23685,23686,23688,23689,23690,23691,23692,23693,23695,23696,23697,23698,23699,23700,23701,23702,23703,23704,23705,23706,23707,23708,23709,23710,23711,23712,23713,23714,23715,23716,23717,23718,23719,23720,23721,23722,23723,23724,23725,23726,23727,23728,23729,23731,23733,23734,23735,23736,23738,23739,23740,23741,23742,23743,23745,23746,23748,23749,23750,23751,23752,23753,23754,23755,23756,23758,23759,23760,23761,23762,23763,23764,23765,23766,23767,23768,23769,23770,23771,23774,23775,23776,23777,23780,23781,23782,23784,23785,23786,23788,23789,23790,23791,23792,23793,23796,23797,23798,23799,23800,23801,23802,23803,23804,23805,23807,23808,23809,23810,23811,23814,23815,23819,23820,23821,23822,23823,23824,23825,23826,23828,23829,23830,23831,23832,23833,23834,23835,23837,23838,23839,23840,23842,23843,23844,23845,23846,23847,23848,23849,23852,23853,23854,23855,23856,23857,23858,23859,23860,23861,23862,23863,23864,23865,23866,23868,23869,23870,23871,23872,23873,23874,23875,23877,23878,23879,23880,23881,23882,23883,23884,23886,23888,23889,23890,23891,23893,23894,23895,23896,23897,23899,23900,23901,23902,23906,23907,23909,23911,23912,23913,23915,23916,23917,23919,23920,23921,23922,23923,23924,23925,23926,23927,23928,23929,23930,23931,23932,23933,23934,23935,23936,23937,23938,23940,23941,23942,23943,23944,23945,23946,23947,23948,23949,23950,23952,23954,23955,23956,23957,23959,23961,23962,23964,23965,23966,23967,23968,23969,23970,23972,23975,23976,23977,23978,23979,23980,23981,23982,23983,23984,23985,23986,23988,23989,23990,23991,23992,23993,23994,23996,23997,24000,24001,24002,24003,24005,24006,24007,24009,24011,24012,24013,24015,24016,24017,24018,24019,24020,24021,24022,24023,24024,24027,24029,24030,24031,24032,24033,24034,24035,24037,24038,24039,24040,24041,24043,24046,24047,24048,24049,24050,24051,24052,24053,24055,24057,24059,24061,24062,24063,24065,24066,24067,24068,24069,24070,24072,24073,24074,24075,24076,24078,24079,24080,24081,24082,24084,24085,24086,24087,24088,24089,24090,24091,24092,24093,24095,24096,24097,24098,24099,24100,24101,24102,24103,24104,24105,24107,24109,24110,24111,24112,24113,24115,24116,24118,24119,24120,24123,24124,24125,24126,24128,24129,24130,24131,24132,24133,24135,24136,24138,24139,24140,24141,24142,24143,24147,24148,24149,24151,24152,24153,24155,24156,24157,24158,24159,24160,24161,24162,24163,24164,24166,24167,24168,24169,24170,24171,24172,24173,24174,24175,24176,24178,24179,24180,24181,24182,24184,24185,24186,24187,24188,24189,24190,24191,24192,24193,24194,24195,24196,24198,24199,24200,24201,24202,24203,24204,24205,24207,24208,24209,24210,24211,24212,24213,24214,24215,24217,24218,24219,24220,24222,24223,24224,24226,24227,24228,24229,24230,24231,24232,24234,24235,24236,24237,24238,24240,24241,24242,24243,24244,24245,24246,24247,24248,24249,24253,24254,24257,24258,24259,24260,24261,24262,24263,24264,24265,24266,24267,24268,24269,24270,24271,24272,24273,24274,24275,24276,24277,24278,24279,24280,24281,24282,24283,24284,24285,24286,24287,24288,24289,24290,24291,24293,24294,24295,24296,24297,24298,24300,24302,24303,24304,24305,24306,24307,24308,24310,24311,24312,24313,24314,24315,24316,24318,24319,24320,24321,24322,24323,24324,24325,24327,24328,24329,24330,24331,24332,24333,24334,24335,24336,24337,24338,24339,24340,24341,24342,24343,24344,24346,24347,24349,24351,24352,24353,24354,24355,24356,24357,24358,24359,24360,24361,24362,24365,24366,24367,24368,24369,24371,24372,24373,24374,24375,24376,24377,24378,24379,24380,24382,24384,24385,24387,24388,24389,24390,24392,24393,24394,24395,24396,24397,24398,24399,24400,24401,24402,24403,24404,24405,24406,24407,24408,24409,24412,24413,24414,24417,24418,24419,24420,24421,24422,24423,24425,24426,24427,24428,24429,24431,24432,24433,24434,24435,24436,24438,24439,24440,24441,24443,24444,24445,24446,24447,24448,24449,24450,24451,24452,24453,24454,24455,24456,24457,24458,24459,24460,24464,24465,24466,24467,24469,24470,24471,24472,24473,24475,24476,24478,24479,24480,24481,24484,24485,24486,24487,24488,24489,24490,24491,24492,24493,24494,24495,24497,24498,24499,24500,24501,24502,24503,24505,24506,24507,24508,24509,24510,24511,24512,24513,24514,24515,24516,24517,24518,24521,24524,24525,24527,24528,24529,24530,24532,24533,24534,24535,24536,24537,24539,24540,24541,24542,24543,24544,24545,24547,24548,24549,24551,24552,24554,24555,24557,24558,24559,24560,24561,24563,24564,24565,24567,24568,24570,24571,24573,24574,24575,24576,24577,24578,24579,24580,24581,24582,24585,24586,24587,24588,24589,24590,24591,24592,24593,24594,24595,24596,24597,24598,24599,24601,24602,24603,24604,24605,24606,24608,24609,24610,24611,24612,24613,24614,24615,24616,24617,24618,24619,24620,24621,24622,24623,24625,24626,24627,24628,24629,24631,24633,24634,24635,24636,24639,24640,24641,24642,24643,24644,24645,24646,24647,24649,24650,24651,24652,24653,24656,24658,24659,24660,24661,24664,24665,24666,24667,24669,24670,24671,24672,24674,24675,24676,24677,24678,24679,24680,24681,24682,24683,24684,24685,24686,24687,24688,24690,24691,24693,24694,24695,24696,24697,24698,24699,24700,24701,24702,24703,24704,24705,24707,24708,24709,24710,24711,24712,24713,24714,24715,24716,24717,24718,24720,24722,24724,24725,24726,24727,24730,24731,24732,24733,24734,24735,24736,24738,24739,24740,24742,24743,24744,24745,24746,24747,24748,24749,24751,24752,24753,24754,24755,24756,24757,24758,24759,24760,24761,24762,24763,24764,24765,24766,24767,24768,24769,24771,24772,24773,24774,24775,24776,24777,24778,24779,24780,24781,24782,24783,24785,24787,24788,24789,24791,24792,24793,24794,24795,24796,24797,24798,24799,24800,24801,24802,24803,24804,24806,24807,24808,24809,24811,24812,24813,24814,24815,24816,24817,24818,24819,24820,24821,24822,24823,24824,24825,24826,24827,24828,24829,24830,24831,24832,24833,24835,24836,24837,24838,24839,24840,24841,24842,24843,24844,24845,24846,24847,24848,24849,24850,24851,24852,24853,24854,24856,24857,24858,24859,24860,24861,24863,24864,24865,24866,24867,24868,24870,24871,24872,24873,24875,24876,24878,24879,24880,24882,24884,24886,24887,24891,24892,24893,24894,24895,24896,24897,24898,24900,24901,24902,24903,24904,24905,24906,24907,24908,24909,24910,24911,24912,24913,24914,24915,24916,24917,24918,24920,24921,24922,24923,24924,24925,24926,24927,24928,24929,24930,24931,24932,24933,24934,24935,24936,24938,24939,24940,24942,24943,24944,24945,24946,24947,24948,24949,24950,24951,24953,24954,24956,24957,24958,24960,24961,24962,24963,24967,24969,24970,24971,24972,24973,24974,24976,24977,24978,24979,24980,24981,24982,24984,24985,24986,24987,24988,24989,24991,24993,24994,24996,24999,25000,25001,25002,25003,25004,25005,25006,25007,25008,25009,25010,25011,25012,25013,25014,25015,25016,25017,25018,25020,25022,25023,25024,25025,25026,25027,25029,25030,25031,25032,25033,25034,25035,25036,25037,25039,25040,25041,25042,25043,25044,25046,25048,25050,25054,25055,25056,25058,25059,25060,25061,25062,25063,25064,25065,25066,25067,25069,25070,25072,25073,25074,25076,25077,25078,25079,25080,25081,25082,25083,25084,25085,25086,25087,25088,25089,25091,25092,25094,25095,25096,25097,25098,25099,25100,25101,25102,25103,25104,25105,25106,25107,25108,25109,25110,25111,25112,25113,25114,25115,25117,25118,25119,25120,25121,25122,25123,25124,25125,25126,25127,25129,25130,25131,25132,25133,25134,25135,25136,25138,25139,25140,25142,25143,25144,25145,25146,25147,25149,25150,25151,25152,25153,25154,25155,25158,25159,25160,25161,25162,25163,25164,25165,25166,25168,25169,25170,25171,25172,25173,25176,25177,25178,25179,25180,25182,25184,25185,25186,25187,25188,25189,25190,25191,25192,25193,25194,25195,25196,25197,25198,25199,25200,25201,25202,25203,25204,25206,25207,25209,25210,25211,25212,25213,25214,25215,25216,25217,25218,25219,25220,25221,25222,25223,25224,25225,25226,25228,25230,25231,25232,25233,25234,25235,25236,25237,25238,25239,25240,25242,25243,25244,25245,25246,25247,25248,25249,25250,25252,25253,25254,25256,25257,25258,25259,25260,25261,25262,25263,25264,25265,25267,25268,25269,25270,25272,25273,25275,25276,25277,25278,25279,25282,25283,25284,25285,25286,25287,25288,25289,25290,25291,25292,25293,25294,25295,25296,25297,25298,25299,25300,25301,25302,25303,25304,25305,25306,25307,25308,25309,25311,25312,25313,25314,25315,25317,25318,25319,25320,25321,25323,25324,25325,25326,25327,25328,25329,25330,25331,25332,25333,25334,25335,25336,25337,25338,25339,25340,25341,25342,25343,25344,25345,25346,25347,25351,25352,25353,25355,25356,25357,25358,25359,25360,25361,25363,25364,25365,25366,25368,25369,25370,25371,25373,25374,25375,25376,25377,25378,25379,25380,25381,25383,25384,25385,25386,25387,25388,25389,25390,25391,25393,25394,25395,25396,25397,25398,25399,25400,25401,25402,25403,25404,25405,25406,25407,25408,25409,25410,25411,25412,25413,25414,25415,25416,25417,25418,25419,25420,25421,25422,25423,25424,25425,25428,25429,25430,25431,25432,25433,25434,25436,25438,25439,25441,25442,25443,25444,25445,25447,25448,25449,25451,25452,25453,25454,25455,25456,25457,25458,25461,25462,25463,25464,25465,25466,25467,25468,25469,25471,25472,25473,25474,25475,25476,25477,25479,25480,25481,25482,25483,25484,25485,25486,25487,25488,25489,25490,25492,25494,25495,25496,25497,25499,25500,25501,25502,25503,25504,25505,25506,25507,25508,25509,25511,25512,25513,25514,25515,25516,25517,25518,25519,25520,25521,25522,25523,25524,25525,25527,25528,25529,25530,25531,25532,25533,25534,25536,25537,25538,25539,25540,25541,25542,25543,25544,25545,25546,25547,25548,25549,25550,25551,25552,25553,25554,25555,25557,25558,25559,25560,25561,25562,25563,25564,25565,25566,25567,25568,25569,25570,25571,25572,25573,25574,25575,25576,25577,25578,25579,25581,25582,25583,25584,25585,25586,25587,25588,25589,25590,25592,25593,25594,25595,25596,25597,25598,25599,25600,25601,25602,25605,25606,25607,25609,25610,25611,25612,25613,25614,25615,25616,25618,25619,25620,25621,25622,25623,25624,25626,25627,25628,25630,25631,25632,25633,25634,25635,25636,25637,25638,25639,25640,25642,25643,25644,25645,25646,25647,25648,25650,25651,25652,25653,25654,25655,25656,25657,25658,25659,25661,25662,25663,25664,25665,25666,25667,25668,25669,25670,25671,25672,25674,25675,25677,25678,25680,25681,25682,25683,25684,25688,25689,25690,25691,25692,25693,25694,25695,25696,25697,25701,25702,25703,25704,25705,25707,25708,25709,25710,25711,25712,25713,25714,25715,25716,25717,25718,25719,25720,25721,25722,25723,25724,25725,25727,25730,25731,25732,25733,25735,25736,25737,25738,25739,25740,25741,25743,25744,25745,25746,25747,25749,25750,25751,25752,25753,25754,25756,25757,25758,25759,25760,25762,25763,25764,25765,25766,25769,25771,25772,25773,25774,25775,25776,25777,25778,25779,25780,25781,25782,25783,25784,25785,25786,25787,25788,25789,25790,25791,25792,25793,25794,25795,25796,25797,25799,25801,25802,25803,25805,25806,25807,25808,25810,25811,25812,25814,25815,25816,25817,25818,25819,25821,25822,25824,25825,25826,25827,25828,25829,25830,25831,25832,25833,25834,25835,25836,25837,25839,25840,25841,25842,25843,25844,25846,25847,25848,25850,25851,25852,25853,25854,25855,25856,25857,25859,25860,25861,25862,25863,25865,25866,25868,25869,25870,25871,25872,25873,25874,25875,25876,25877,25878,25879,25880,25881,25883,25884,25885,25886,25887,25888,25889,25890,25891,25892,25893,25894,25897,25898,25899,25900,25901,25902,25903,25904,25906,25907,25908,25909,25910,25911,25912,25913,25915,25917,25918,25919,25921,25923,25925,25926,25928,25929,25930,25932,25933,25934,25935,25937,25939,25940,25941,25942,25943,25944,25945,25947,25948,25949,25950,25951,25954,25955,25956,25957,25958,25959,25960,25962,25963,25964,25965,25967,25968,25970,25971,25972,25973,25974,25975,25976,25977,25978,25979,25980,25983,25984,25985,25986,25987,25988,25989,25990,25991,25992,25993,25995,25996,25998,26000,26001,26002,26003,26004,26005,26006,26007,26009,26011,26012,26013,26014,26015,26016,26017,26018,26020,26021,26023,26024,26025,26026,26027,26028,26029,26030,26031,26032,26034,26035,26037,26038,26039,26040,26041,26043,26044,26045,26046,26047,26049,26050,26051,26052,26053,26054,26059,26060,26061,26062,26063,26064,26065,26066,26067,26068,26070,26071,26073,26074,26075,26077,26078,26079,26080,26081,26082,26083,26085,26086,26087,26088,26089,26092,26093,26094,26095,26096,26097,26098,26099,26100,26101,26102,26103,26104,26106,26107,26108,26109,26111,26112,26114,26115,26116,26117,26118,26119,26120,26121,26122,26123,26124,26125,26126,26127,26128,26129,26130,26131,26132,26133,26136,26137,26140,26141,26142,26143,26144,26145,26146,26147,26148,26149,26150,26151,26152,26155,26157,26158,26159,26160,26161,26162,26163,26164,26165,26166,26169,26170,26171,26172,26174,26175,26177,26178,26179,26180,26181,26183,26184,26185,26186,26187,26188,26189,26191,26193,26194,26195,26196,26197,26198,26199,26201,26202,26203,26204,26205,26206,26207,26209,26210,26211,26212,26213,26214,26215,26216,26217,26218,26219,26220,26222,26223,26224,26225,26226,26227,26228,26230,26231,26232,26233,26234,26235,26236,26237,26238,26240,26241,26242,26243,26244,26245,26246,26247,26248,26249,26250,26251,26252,26253,26254,26256,26257,26258,26260,26261,26262,26263,26264,26265,26266,26269,26271,26272,26273,26274,26276,26278,26279,26280,26281,26282,26283,26285,26286,26287,26288,26289,26290,26291,26292,26293,26294,26295,26296,26297,26298,26299,26300,26301,26302,26303,26304,26305,26308,26310,26311,26312,26313,26314,26315,26316,26317,26318,26319,26322,26326,26328,26329,26330,26331,26332,26333,26334,26336,26339,26340,26342,26343,26344,26345,26347,26348,26349,26350,26352,26353,26354,26355,26356,26357,26358,26359,26360,26361,26362,26363,26364,26365,26366,26367,26368,26369,26370,26371,26372,26373,26376,26377,26378,26379,26380,26381,26382,26383,26384,26386,26387,26388,26389,26390,26391,26392,26393,26395,26397,26398,26399,26400,26401,26402,26403,26405,26406,26407,26408,26409,26410,26411,26412,26413,26414,26415,26417,26419,26420,26421,26422,26423,26424,26425,26426,26427,26428,26429,26430,26431,26432,26433,26434,26435,26436,26437,26438,26439,26440,26441,26443,26444,26445,26446,26447,26448,26449,26451,26453,26454,26455,26457,26458,26460,26461,26462,26463,26464,26465,26466,26467,26468,26469,26470,26471,26472,26473,26474,26475,26476,26477,26478,26479,26480,26481,26482,26483,26484,26485,26486,26487,26488,26489,26490,26491,26492,26493,26494,26495,26497,26499,26500,26501,26502,26503,26505,26507,26508,26509,26510,26511,26512,26513,26514,26515,26516,26517,26519,26520,26521,26522,26524,26525,26526,26527,26528,26529,26530,26531,26532,26533,26534,26535,26536,26537,26538,26539,26540,26541,26542,26543,26544,26545,26546,26547,26548,26549,26550,26551,26552,26553,26554,26555,26559,26560,26561,26562,26563,26564,26565,26566,26568,26569,26570,26571,26572,26573,26574,26575,26576,26577,26578,26579,26580,26582,26583,26584,26585,26586,26587,26588,26589,26590,26591,26592,26594,26595,26596,26597,26598,26599,26601,26602,26603,26604,26605,26606,26607,26608,26609,26610,26611,26612,26613,26614,26615,26616,26617,26618,26619,26620,26621,26622,26623,26624,26625,26626,26627,26628,26629,26631,26632,26633,26634,26635,26636,26637,26638,26639,26640,26641,26642,26643,26644,26646,26647,26648,26650,26651,26652,26653,26654,26655,26656,26657,26658,26661,26662,26664,26665,26666,26667,26669,26670,26671,26673,26674,26675,26676,26677,26678,26679,26680,26681,26682,26683,26684,26685,26686,26688,26689,26690,26691,26692,26693,26694,26695,26696,26697,26698,26699,26700,26701,26702,26703,26704,26705,26706,26707,26708,26709,26710,26713,26716,26717,26719,26720,26721,26722,26723,26724,26725,26726,26727,26728,26729,26731,26733,26734,26735,26737,26738,26740,26741,26742,26743,26744,26745,26747,26748,26749,26750,26751,26752,26753,26754,26755,26756,26757,26758,26759,26760,26761,26762,26763,26764,26765,26767,26768,26769,26770,26771,26772,26774,26775,26776,26779,26780,26781,26783,26784,26785,26786,26787,26788,26790,26791,26792,26793,26794,26795,26796,26797,26798,26799,26800,26801,26802,26803,26804,26805,26806,26809,26810,26812,26813,26816,26818,26819,26820,26821,26822,26823,26824,26825,26826,26827,26828,26829,26830,26831,26832,26833,26834,26835,26836,26837,26838,26839,26840,26842,26844,26845,26846,26847,26848,26849,26851,26852,26854,26855,26856,26857,26858,26859,26860,26862,26863,26864,26865,26866,26867,26868,26869,26870,26871,26872,26873,26874,26875,26876,26877,26880,26881,26882,26883,26884,26885,26886,26887,26888,26889,26890,26891,26892,26893,26894,26895,26896,26897,26898,26899,26900,26901,26903,26904,26905,26906,26907,26908,26911,26912,26913,26914,26915,26916,26917,26918,26920,26922,26924,26925,26927,26928,26930,26931,26932,26933,26934,26935,26936,26937,26939,26940,26941,26942,26943,26944,26945,26946,26947,26948,26949,26950,26952,26953,26954,26955,26956,26958,26959,26961,26962,26963,26964,26965,26966,26967,26968,26969,26970,26971,26972,26973,26974,26975,26976,26977,26978,26979,26980,26981,26982,26983,26984,26985,26986,26987,26988,26989,26990,26991,26992,26993,26994,26995,26996,26997,26998,26999,27000,27001,27002,27003,27004,27005,27006,27008,27009,27010,27011,27012,27013,27014,27015,27016,27017,27018,27021,27022,27024,27025,27027,27028,27029,27030,27031,27032,27033,27034,27035,27036,27038,27039,27040,27041,27042,27043,27044,27045,27046,27047,27048,27049,27050,27051,27052,27053,27054,27055,27056,27057,27058,27059,27060,27061,27062,27063,27065,27067,27068,27069,27070,27071,27072,27073,27074,27075,27076,27078,27079,27081,27082,27083,27084,27085,27086,27087,27088,27089,27091,27092,27093,27094,27096,27097,27099,27101,27102,27103,27104,27105,27106,27108,27109,27110,27111,27112,27113,27115,27116,27117,27118,27121,27122,27123,27124,27126,27127,27128,27129,27130,27131,27132,27133,27134,27135,27136,27137,27138,27139,27140,27141,27142,27143,27144,27145,27146,27147,27148,27149,27151,27153,27154,27155,27156,27157,27158,27159,27160,27161,27162,27163,27164,27165,27166,27167,27168,27169,27170,27171,27173,27174,27175,27176,27177,27178,27179,27180,27181,27182,27183,27184,27185,27186,27187,27188,27189,27190,27192,27193,27194,27195,27196,27197,27198,27199,27200,27201,27203,27204,27205,27206,27207,27208,27209,27211,27212,27213,27214,27215,27216,27217,27218,27219,27220,27221,27222,27223,27224,27225,27226,27227,27229,27230,27231,27232,27233,27234,27235,27236,27237,27238,27239,27240,27241,27242,27243,27245,27247,27249,27250,27251,27252,27254,27256,27257,27258,27260,27262,27263,27264,27265,27266,27267,27268,27269,27271,27273,27274,27276,27277,27278,27279,27280,27281,27282,27283,27284,27285,27286,27287,27289,27290,27291,27292,27293,27294,27295,27296,27297,27298,27299,27300,27301,27302,27303,27304,27305,27306,27307,27308,27309,27310,27311,27313,27314,27315,27316,27317,27318,27319,27320,27321,27322,27323,27325,27326,27329,27330,27331,27333,27334,27335,27336,27337,27338,27339,27340,27341,27343,27344,27345,27347,27348,27352,27353,27354,27355,27356,27357,27358,27359,27360,27361,27362,27364,27365,27367,27368,27370,27371,27372,27374,27375,27376,27377,27379,27382,27384,27385,27386,27387,27388,27392,27394,27395,27396,27397,27400,27401,27402,27403,27407,27408,27409,27410,27411,27414,27415,27416,27417,27418,27421,27422,27423,27424,27425,27426,27427,27428,27429,27431,27432,27436,27437,27439,27441,27442,27443,27444,27445,27446,27447,27448,27449,27450,27451,27452,27453,27454,27455,27457,27458,27459,27461,27462,27463,27464,27465,27466,27467,27468,27469,27470,27472,27473,27474,27475,27476,27477,27478,27479,27481,27483,27484,27486,27487,27488,27489,27490,27491,27492,27493,27494,27495,27497,27498,27501,27503,27506,27507,27508,27510,27511,27512,27513,27514,27515,27516,27518,27519,27520,27521,27522,27523,27524,27526,27527,27528,27529,27530,27531,27532,27533,27534,27535,27537,27538,27539,27540,27541,27542,27543,27544,27545,27546,27547,27550,27551,27552,27553,27554,27555,27556,27557,27558,27559,27562,27563,27565,27566,27567,27568,27569,27570,27571,27572,27573,27574,27575,27578,27579,27580,27581,27583,27584,27585,27586,27587,27588,27589,27590,27591,27592,27593,27594,27595,27596,27597,27598,27599,27600,27602,27603,27604,27605,27606,27607,27608,27609,27610,27611,27612,27614,27615,27616,27617,27618,27619,27620,27622,27623,27624,27626,27627,27628,27631,27632,27634,27635,27637,27639,27640,27641,27642,27643,27644,27645,27646,27647,27648,27649,27650,27651,27652,27653,27654,27655,27656,27657,27659,27660,27661,27663,27664,27665,27667,27668,27669,27670,27671,27672,27673,27674,27675,27676,27677,27679,27680,27681,27682,27683,27684,27685,27686,27687,27688,27689,27690,27691,27692,27694,27695,27696,27697,27698,27699,27700,27701,27702,27703,27704,27705,27706,27707,27709,27710,27711,27712,27713,27714,27715,27718,27719,27721,27722,27723,27724,27725,27726,27727,27728,27730,27732,27733,27735,27736,27737,27738,27739,27740,27741,27742,27743,27744,27745,27746,27748,27749,27750,27751,27752,27753,27754,27755,27757,27758,27759,27760,27761,27762,27763,27764,27765,27766,27768,27769,27770,27771,27773,27774,27775,27776,27777,27778,27779,27780,27781,27782,27783,27784,27785,27786,27787,27788,27789,27790,27791,27792,27794,27795,27796,27797,27798,27800,27801,27802,27803,27804,27805,27807,27809,27810,27811,27812,27813,27814,27815,27817,27818,27819,27820,27821,27822,27823,27824,27825,27826,27827,27828,27830,27831,27832,27833,27834,27835,27836,27837,27838,27839,27840,27841,27842,27843,27844,27845,27846,27847,27849,27850,27851,27852,27853,27854,27855,27856,27857,27858,27859,27860,27861,27862,27863,27865,27866,27867,27868,27869,27870,27871,27872,27873,27874,27875,27877,27879,27880,27881,27882,27883,27884,27885,27886,27887,27888,27889,27890,27891,27893,27894,27895,27896,27897,27898,27899,27900,27901,27902,27903,27904,27905,27906,27907,27908,27909,27910,27911,27912,27913,27914,27915,27916,27917,27918,27919,27920,27921,27922,27926,27927,27928,27929,27930,27931,27933,27934,27935,27936,27938,27940,27941,27942,27943,27944,27945,27946,27947,27948,27949,27950,27951,27952,27953,27954,27955,27956,27957,27958,27959,27960,27961,27962,27963,27964,27965,27966,27967,27968,27969,27970,27971,27972,27973,27974,27975,27976,27978,27979,27981,27982,27983,27985,27986,27987,27988,27991,27992,27993,27994,27995,27996,27998,27999,28000,28001,28002,28003,28004,28005,28006,28007,28008,28009,28010,28012,28013,28014,28015,28016,28017,28018,28020,28021,28022,28023,28024,28025,28026,28027,28028,28029,28030,28031,28032,28033,28034,28035,28036,28037,28038,28039,28040,28041,28042,28043,28044,28045,28046,28047,28048,28049,28050,28051,28052,28053,28054,28055,28056,28057,28059,28060,28061,28062,28063,28064,28065,28067,28068,28069,28070,28071,28072,28073,28074,28075,28076,28078,28079,28081,28082,28083,28084,28085,28087,28088,28089,28090,28091,28092,28093,28094,28095,28096,28098,28100,28101,28102,28103,28104,28105,28106,28107,28108,28109,28111,28112,28113,28114,28115,28116,28117,28118,28119,28120,28121,28122,28123,28124,28125,28126,28127,28128,28129,28130,28131,28132,28133,28134,28136,28137,28138,28139,28140,28141,28142,28143,28144,28145,28146,28147,28148,28149,28150,28151,28152,28153,28154,28155,28156,28157,28158,28160,28162,28163,28164,28165,28167,28168,28169,28170,28171,28172,28173,28174,28175,28176,28177,28179,28180,28181,28182,28183,28184,28185,28186,28187,28188,28189,28191,28192,28193,28194,28195,28196,28197,28198,28199,28200,28201,28202,28203,28204,28205,28206,28207,28208,28209,28210,28211,28212,28213,28214,28216,28217,28218,28219,28220,28221,28222,28223,28224,28225,28227,28228,28229,28230,28231,28233,28234,28235,28237,28238,28239,28240,28241,28242,28243,28244,28245,28246,28247,28248,28249,28250,28251,28252,28253,28254,28255,28256,28257,28258,28259,28260,28261,28262,28263,28264,28265,28267,28270,28271,28273,28274,28275,28276,28278,28279,28280,28281,28284,28286,28287,28288,28290,28291,28293,28294,28296,28297,28299,28300,28301,28302,28303,28304,28306,28307,28308,28310,28311,28312,28313,28314,28315,28316,28317,28318,28319,28320,28321,28322,28323,28324,28325,28326,28327,28330,28331,28334,28335,28336,28337,28338,28339,28340,28341,28342,28343,28344,28345,28346,28347,28348,28349,28350,28351,28352,28353,28354,28355,28356,28357,28358,28359,28360,28361,28362,28363,28364,28365,28366,28367,28368,28369,28370,28371,28372,28373,28374,28375,28376,28377,28378,28379,28380,28381,28382,28383,28384,28385,28386,28388,28389,28390,28392,28393,28395,28396,28397,28398,28399,28401,28402,28404,28405,28406,28407,28408,28409,28410,28411,28412,28413,28414,28415,28416,28417,28418,28419,28420,28421,28422,28423,28424,28425,28426,28427,28428,28429,28430,28431,28433,28434,28435,28436,28437,28438,28439,28440,28441,28442,28444,28446,28447,28448,28449,28450,28451,28452,28453,28454,28455,28457,28458,28459,28460,28461,28462,28463,28464,28465,28466,28467,28468,28469,28470,28471,28472,28473,28474,28475,28476,28477,28478,28479,28480,28481,28483,28484,28485,28486,28487,28491,28493,28494,28495,28496,28497,28498,28499,28500,28501,28502,28503,28504,28506,28507,28508,28509,28510,28511,28512,28513,28514,28515,28516,28518,28519,28521,28522,28523,28524,28525,28526,28527,28528,28530,28531,28532,28534,28535,28536,28537,28538,28539,28540,28541,28542,28543,28544,28545,28546,28548,28549,28550,28551,28552,28553,28554,28555,28556,28557,28558,28560,28561,28562,28563,28564,28565,28566,28567,28572,28573,28574,28575,28576,28577,28578,28579,28580,28581,28582,28583,28584,28585,28586,28587,28588,28589,28590,28591,28592,28593,28594,28595,28596,28597,28598,28600,28601,28602,28603,28604,28605,28606,28607,28608,28609,28610,28611,28612,28614,28615,28616,28617,28618,28619,28620,28621,28622,28623,28625,28626,28627,28628,28629,28632,28633,28634,28635,28636,28637,28638,28639,28640,28641,28642,28643,28644,28646,28647,28648,28649,28651,28652,28653,28654,28655,28656,28657,28658,28659,28660,28661,28662,28663,28664,28666,28667,28668,28670,28671,28672,28673,28675,28676,28677,28678,28679,28681,28682,28683,28684,28685,28686,28687,28689,28692,28693,28694,28695,28696,28697,28698,28699,28700,28701,28702,28703,28704,28705,28706,28707,28708,28710,28711,28712,28713,28714,28715,28716,28719,28720,28721,28722,28723,28724,28725,28727,28728,28729,28730,28731,28732,28734,28735,28736,28737,28738,28739,28740,28741,28742,28744,28745,28746,28747,28748,28749,28751,28752,28753,28754,28756,28757,28758,28759,28760,28762,28763,28764,28765,28766,28767,28768,28769,28770,28771,28772,28773,28774,28775,28776,28777,28778,28779,28780,28781,28783,28784,28785,28788,28789,28790,28791,28792,28793,28794,28796,28797,28798,28799,28800,28801,28802,28803,28804,28805,28806,28809,28810,28811,28813,28814,28815,28817,28818,28819,28820,28821,28822,28824,28825,28826,28828,28829,28831,28832,28833,28835,28836,28837,28838,28839,28841,28843,28844,28845,28846,28847,28848,28849,28851,28852,28853,28855,28856,28857,28858,28859,28860,28861,28862,28864,28865,28866,28867,28868,28869,28870,28871,28872,28874,28875,28876,28877,28878,28879,28880,28881,28882,28883,28884,28885,28886,28887,28888,28889,28890,28891,28892,28893,28894,28895,28896,28897,28898,28900,28902,28903,28904,28905,28907,28908,28909,28911,28912,28913,28915,28916,28917,28918,28919,28920,28921,28922,28923,28924,28925,28926,28927,28928,28930,28932,28933,28934,28937,28938,28939,28940,28941,28942,28943,28944,28947,28948,28949,28950,28951,28952,28953,28954,28955,28956,28957,28958,28959,28960,28961,28962,28963,28965,28966,28968,28969,28971,28972,28974,28975,28976,28977,28978,28979,28980,28981,28982,28986,28987,28988,28990,28991,28992,28993,28994,28995,28996,28997,28998,28999,29001,29002,29003,29004,29005,29006,29007,29008,29009,29010,29011,29012,29013,29014,29015,29016,29017,29018,29020,29021,29022,29023,29024,29025,29026,29027,29028,29029,29030,29031,29032,29033,29034,29035,29036,29038,29040,29041,29042,29043,29045,29046,29047,29048,29050,29051,29052,29053,29054,29056,29057,29058,29060,29061,29062,29063,29064,29065,29066,29068,29070,29071,29072,29073,29074,29076,29077,29078,29079,29080,29081,29082,29083,29084,29085,29086,29087,29088,29089,29090,29091,29092,29093,29095,29096,29097,29098,29100,29101,29103,29104,29105,29106,29107,29108,29109,29111,29112,29113,29114,29116,29117,29118,29119,29120,29121,29122,29123,29124,29125,29126,29127,29128,29129,29130,29131,29134,29135,29136,29137,29138,29140,29141,29142,29143,29144,29145,29146,29147,29148,29149,29151,29152,29153,29154,29156,29157,29158,29159,29160,29163,29164,29165,29166,29168,29169,29170,29172,29173,29174,29176,29177,29179,29180,29181,29182,29183,29184,29185,29186,29187,29189,29190,29191,29193,29194,29196,29197,29198,29199,29200,29203,29204,29205,29206,29207,29209,29210,29211,29213,29214,29215,29218,29219,29220,29221,29222,29223,29224,29225,29226,29227,29228,29229,29230,29232,29233,29234,29237,29238,29239,29240,29241,29242,29243,29244,29245,29246,29247,29248,29249,29250,29252,29254,29255,29256,29257,29258,29259,29260,29261,29263,29264,29266,29267,29269,29270,29271,29272,29273,29274,29275,29276,29277,29278,29279,29280,29281,29282,29283,29286,29287,29289,29290,29292,29294,29295,29296,29298,29299,29300,29301,29302,29303,29304,29305,29306,29307,29308,29309,29310,29311,29312,29313,29314,29316,29317,29318,29319,29320,29321,29322,29323,29324,29325,29326,29327,29328,29329,29330,29331,29332,29333,29334,29335,29336,29338,29339,29341,29342,29343,29344,29345,29346,29347,29348,29349,29350,29351,29352,29353,29354,29356,29357,29358,29359,29360,29361,29362,29364,29365,29366,29367,29368,29369,29370,29373,29374,29375,29376,29377,29378,29379,29380,29381,29382,29384,29385,29386,29387,29388,29389,29390,29392,29393,29394,29396,29398,29399,29400,29401,29402,29403,29404,29406,29407,29408,29409,29410,29411,29412,29414,29416,29417,29418,29419,29420,29421,29422,29423,29424,29425,29426,29427,29428,29430,29431,29432,29433,29434,29435,29436,29437,29438,29439,29440,29441,29442,29443,29444,29447,29448,29450,29451,29452,29454,29455,29456,29457,29458,29459,29461,29462,29463,29464,29465,29467,29468,29469,29470,29471,29473,29474,29475,29476,29477,29478,29479,29480,29481,29482,29483,29484,29485,29486,29487,29488,29489,29490,29491,29492,29493,29494,29495,29496,29497,29498,29499,29500,29502,29503,29504,29505,29506,29507,29508,29509,29512,29513,29514,29516,29517,29518,29519,29520,29521,29522,29527,29528,29529,29530,29531,29533,29534,29535,29536,29537,29538,29539,29541,29542,29543,29544,29545,29546,29547,29548,29549,29550,29551,29552,29553,29554,29555,29556,29557,29558,29559,29560,29562,29563,29564,29565,29566,29567,29568,29569,29570,29571,29572,29573,29574,29575,29576,29577,29578,29579,29580,29582,29583,29585,29586,29587,29588,29589,29590,29591,29592,29595,29596,29597,29598,29599,29600,29601,29602,29604,29605,29606,29607,29608,29609,29610,29611,29612,29613,29614,29615,29616,29618,29619,29620,29621,29622,29623,29624,29625,29626,29627,29628,29629,29630,29631,29632,29634,29635,29637,29638,29639,29640,29641,29642,29643,29644,29645,29646,29647,29648,29649,29650,29651,29652,29653,29654,29655,29656,29657,29658,29659,29660,29661,29662,29664,29665,29666,29667,29668,29669,29670,29671,29672,29673,29674,29675,29677,29678,29679,29681,29682,29683,29684,29685,29686,29687,29688,29689,29690,29691,29692,29693,29694,29695,29696,29697,29698,29699,29700,29701,29702,29703,29704,29705,29706,29707,29708,29709,29711,29712,29713,29714,29716,29717,29718,29719,29721,29722,29723,29724,29725,29726,29727,29728,29729,29730,29731,29732,29733,29734,29736,29737,29738,29739,29740,29741,29742,29743,29744,29745,29746,29747,29748,29749,29750,29751,29752,29753,29754,29756,29759,29760,29761,29762,29763,29764,29765,29766,29767,29768,29769,29770,29771,29773,29774,29775,29776,29777,29778,29779,29780,29781,29782,29783,29785,29786,29787,29788,29789,29790,29791,29792,29793,29794,29795,29796,29797,29799,29800,29801,29802,29803,29804,29805,29806,29807,29808,29809,29810,29811,29812,29813,29814,29815,29817,29818,29820,29821,29822,29823,29824,29825,29826,29827,29829,29830,29831,29832,29833,29834,29835,29836,29837,29838,29840,29842,29844,29845,29847,29848,29849,29850,29851,29852,29853,29854,29855,29856,29857,29858,29859,29860,29861,29862,29863,29864,29865,29866,29867,29869,29871,29872,29873,29874,29876,29877,29878,29879,29880,29881,29882,29883,29885,29886,29887,29888,29889,29890,29891,29893,29896,29898,29899,29900,29903,29904,29906,29907,29908,29909,29910,29911,29912,29913,29914,29915,29916,29917,29918,29919,29920,29921,29922,29923,29924,29925,29926,29927,29928,29929,29932,29934,29935,29936,29937,29938,29940,29941,29942,29943,29944,29947,29949,29950,29951,29952,29953,29954,29955,29956,29957,29959,29960,29963,29964,29965,29966,29967,29968,29969,29970,29971,29972,29973,29974,29975,29976,29977,29978,29980,29981,29982,29983,29985,29986,29987,29989,29990,29992,29993,29994,29995,29996,29997,29998,29999,30000,30001,30002,30003,30004,30005,30007,30008,30009,30010,30011,30012,30013,30014,30015,30016,30018,30020,30021,30022,30023,30024,30025,30026,30027,30028,30029,30030,30031,30033,30035,30036,30037,30041,30042,30043,30044,30045,30047,30048,30050,30051,30052,30053,30054,30055,30057,30058,30059,30060,30061,30062,30063,30064,30066,30067,30068,30070,30071,30072,30073,30074,30077,30078,30079,30080,30082,30083,30084,30086,30087,30089,30090,30091,30092,30093,30094,30095,30096,30097,30098,30100,30101,30102,30103,30104,30105,30106,30109,30110,30111,30112,30113,30114,30115,30116,30117,30119,30122,30123,30124,30126,30127,30128,30129,30130,30131,30132,30133,30134,30136,30137,30138,30139,30140,30141,30142,30143,30144,30145,30146,30147,30148,30149,30151,30152,30153,30154,30155,30156,30157,30158,30159,30160,30161,30162,30164,30165,30166,30167,30168,30169,30170,30171,30172,30173,30174,30175,30176,30177,30178,30179,30180,30182,30183,30184,30185,30186,30187,30189,30191,30192,30193,30194,30195,30196,30197,30198,30199,30200,30201,30202,30203,30204,30205,30206,30207,30208,30209,30210,30211,30213,30215,30216,30217,30218,30219,30220,30221,30223,30224,30225,30227,30228,30229,30230,30231,30232,30233,30234,30235,30236,30237,30238,30239,30240,30241,30242,30243,30244,30245,30246,30247,30248,30249,30250,30251,30252,30253,30255,30256,30257,30258,30259,30260,30261,30264,30266,30267,30268,30269,30270,30271,30272,30274,30275,30278,30279,30280,30281,30284,30285,30286,30287,30288,30289,30290,30291,30292,30294,30295,30296,30297,30298,30300,30302,30303,30304,30305,30306,30307,30308,30309,30310,30311,30312,30313,30314,30315,30316,30317,30318,30319,30320,30321,30322,30323,30324,30325,30326,30328,30329,30330,30331,30332,30333,30334,30335,30336,30337,30338,30339,30340,30342,30343,30344,30345,30346,30347,30350,30351,30352,30353,30354,30355,30357,30358,30361,30362,30363,30364,30365,30366,30369,30372,30373,30374,30378,30379,30381,30382,30383,30384,30385,30386,30388,30389,30391,30392,30393,30394,30395,30397,30398,30399,30402,30403,30404,30405,30406,30408,30409,30410,30412,30413,30414,30415,30416,30417,30418,30419,30420,30422,30423,30424,30425,30426,30427,30428,30429,30430,30431,30433,30435,30436,30437,30438,30439,30441,30442,30444,30445,30446,30447,30448,30449,30450,30451,30452,30453,30455,30456,30457,30458,30459,30460,30462,30465,30467,30468,30469,30471,30472,30473,30474,30475,30476,30477,30478,30479,30480,30481,30482,30483,30485,30489,30490,30491,30493,30494,30495,30496,30498,30499,30500,30501,30502,30503,30504,30505,30507,30509,30511,30513,30514,30515,30516,30517,30518,30519,30520,30521,30522,30523,30524,30525,30526,30528,30529,30531,30532,30533,30534,30535,30538,30539,30540,30541,30542,30543,30544,30545,30546,30548,30549,30550,30552,30553,30554,30555,30556,30558,30559,30560,30561,30562,30563,30565,30566,30567,30568,30569,30570,30571,30572,30573,30574,30575,30578,30583,30584,30585,30586,30587,30588,30589,30590,30591,30592,30593,30594,30595,30596,30597,30599,30600,30601,30603,30604,30605,30606,30607,30609,30610,30611,30613,30615,30616,30617,30618,30619,30620,30621,30622,30623,30624,30625,30626,30627,30629,30631,30632,30633,30634,30635,30636,30637,30639,30640,30641,30642,30643,30644,30645,30646,30647,30649,30650,30651,30652,30653,30654,30655,30658,30659,30660,30661,30663,30665,30666,30667,30668,30669,30670,30671,30672,30675,30676,30677,30679,30680,30681,30682,30683,30684,30685,30686,30688,30690,30691,30693,30694,30695,30696,30697,30699,30700,30701,30702,30703,30704,30705,30706,30707,30708,30710,30711,30712,30713,30714,30715,30716,30717,30718,30719,30720,30721,30722,30723,30725,30726,30728,30729,30732,30733,30734,30735,30736,30737,30738,30739,30740,30741,30742,30743,30744,30746,30748,30749,30750,30751,30752,30753,30754,30755,30757,30758,30759,30760,30761,30762,30763,30764,30765,30766,30767,30768,30769,30770,30771,30772,30773,30775,30776,30777,30778,30779,30780,30781,30782,30783,30784,30786,30787,30788,30789,30791,30792,30793,30794,30795,30796,30797,30798,30800,30801,30802,30803,30804,30805,30806,30807,30812,30813,30814,30816,30818,30820,30821,30822,30824,30825,30826,30827,30828,30829,30830,30831,30832,30833,30834,30836,30839,30841,30842,30843,30844,30846,30847,30848,30849,30851,30852,30853,30854,30855,30856,30857,30860,30861,30862,30863,30865,30867,30868,30869,30870,30871,30872,30873,30874,30875,30876,30878,30879,30880,30881,30882,30883,30884,30885,30887,30888,30889,30890,30891,30892,30893,30895,30896,30897,30898,30899,30900,30901,30902,30904,30905,30906,30907,30908,30910,30913,30915,30916,30917,30918,30919,30920,30921,30922,30923,30924,30925,30926,30927,30928,30929,30930,30931,30932,30933,30935,30936,30937,30938,30939,30941,30942,30943,30944,30945,30946,30947,30948,30949,30951,30952,30953,30954,30956,30957,30958,30959,30960,30961,30962,30963,30964,30965,30967,30969,30970,30971,30972,30973,30974,30975,30977,30978,30980,30981,30982,30983,30985,30988,30990,30992,30993,30994,30995,30996,30999,31001,31003,31004,31005,31006,31009,31011,31012,31013,31014,31015,31016,31017,31018,31019,31020,31021,31022,31023,31024,31025,31026,31027,31028,31029,31030,31032,31033,31034,31035,31036,31037,31038,31039,31040,31041,31042,31044,31045,31046,31047,31048,31049,31050,31051,31052,31055,31056,31057,31058,31059,31060,31061,31062,31063,31064,31065,31066,31067,31068,31069,31070,31071,31072,31073,31074,31075,31076,31077,31079,31080,31081,31082,31083,31085,31087,31088,31089,31090,31091,31092,31095,31096,31097,31098,31100,31101,31102,31103,31104,31105,31106,31107,31108,31109,31110,31111,31112,31114,31115,31117,31118,31119,31120,31121,31122,31123,31124,31125,31126,31127,31128,31129,31130,31131,31132,31133,31135,31136,31137,31138,31140,31141,31142,31143,31144,31145,31146,31147,31148,31149,31150,31152,31153,31154,31155,31156,31158,31159,31160,31161,31162,31163,31165,31166,31167,31168,31169,31171,31172,31173,31174,31176,31177,31178,31179,31180,31181,31182,31183,31185,31186,31188,31189,31190,31192,31196,31197,31198,31199,31200,31201,31202,31203,31204,31206,31207,31209,31210,31211,31212,31213,31214,31215,31216,31217,31220,31222,31223,31224,31226,31227,31229,31232,31234,31235,31236,31237,31238,31240,31242,31243,31244,31245,31246,31248,31249,31250,31251,31252,31253,31255,31256,31257,31258,31259,31260,31262,31263,31264,31266,31267,31270,31272,31274,31275,31277,31278,31279,31280,31281,31282,31283,31287,31289,31290,31291,31292,31293,31294,31295,31296,31298,31299,31300,31301,31302,31303,31304,31305,31306,31307,31308,31309,31310,31311,31312,31313,31316,31318,31319,31320,31322,31323,31324,31327,31328,31329,31330,31331,31333,31335,31336,31337,31339,31340,31341,31342,31344,31345,31348,31349,31350,31351,31352,31353,31354,31355,31357,31358,31359,31360,31361,31363,31364,31365,31366,31367,31368,31369,31370,31371,31372,31373,31375,31376,31377,31378,31379,31380,31381,31382,31383,31384,31385,31388,31389,31390,31391,31392,31394,31395,31397,31398,31400,31401,31402,31403,31404,31405,31406,31407,31408,31409,31410,31411,31412,31413,31414,31415,31416,31418,31419,31420,31422,31423,31424,31425,31426,31427,31428,31429,31431,31432,31433,31434,31435,31437,31439,31441,31442,31443,31445,31446,31448,31449,31450,31451,31452,31453,31454,31455,31456,31457,31458,31459,31460,31461,31462,31463,31465,31466,31467,31469,31470,31471,31472,31478,31479,31480,31481,31482,31483,31485,31486,31487,31488,31489,31490,31491,31492,31493,31494,31496,31497,31498,31499,31500,31502,31503,31504,31505,31506,31507,31508,31509,31512,31513,31514,31515,31517,31518,31519,31520,31522,31523,31524,31525,31526,31527,31528,31529,31530,31531,31532,31533,31534,31535,31536,31537,31538,31539,31540,31541,31542,31544,31545,31546,31547,31548,31550,31552,31554,31555,31556,31557,31558,31559,31560,31561,31562,31563,31564,31565,31566,31567,31568,31569,31570,31572,31573,31574,31576,31578,31579,31581,31584,31585,31586,31587,31588,31589,31590,31591,31593,31596,31597,31598,31599,31600,31601,31602,31603,31604,31605,31606,31607,31608,31609,31610,31611,31614,31616,31618,31620,31621,31622,31623,31624,31626,31627,31628,31629,31630,31631,31632,31633,31634,31636,31637,31638,31639,31640,31641,31642,31643,31644,31645,31646,31647,31648,31649,31650,31651,31652,31654,31655,31656,31657,31658,31659,31660,31661,31663,31665,31666,31668,31669,31671,31672,31673,31678,31680,31681,31684,31686,31687,31689,31690,31691,31692,31694,31695,31696,31697,31699,31700,31701,31704,31705,31706,31707,31708,31709,31710,31711,31712,31713,31714,31715,31716,31717,31718,31719,31720,31721,31722,31723,31724,31725,31726,31728,31729,31730,31731,31732,31734,31735,31736,31737,31738,31739,31740,31741,31742,31743,31744,31745,31746,31747,31749,31750,31751,31753,31754,31755,31756,31757,31758,31759,31760,31761,31762,31763,31764,31765,31766,31767,31769,31771,31772,31773,31774,31775,31776,31777,31778,31779,31781,31782,31783,31784,31785,31786,31787,31788,31789,31792,31795,31797,31799,31800,31801,31803,31804,31805,31806,31807,31808,31809,31810,31811,31812,31813,31815,31816,31817,31818,31820,31821,31823,31824,31825,31827,31828,31830,31831,31832,31833,31834,31835,31836,31837,31839,31840,31843,31844,31845,31846,31847,31849,31850,31851,31852,31853,31854,31855,31856,31858,31859,31860,31861,31864,31865,31866,31867,31868,31869,31870,31871,31872,31873,31874,31875,31876,31877,31878,31880,31881,31882,31883,31884,31885,31886,31888,31889,31890,31892,31893,31894,31895,31896,31899,31900,31901,31902,31903,31905,31906,31907,31908,31909,31910,31911,31912,31914,31915,31916,31917,31918,31919,31921,31922,31923,31924,31925,31928,31929,31930,31931,31932,31933,31934,31935,31936,31937,31938,31939,31941,31943,31944,31945,31946,31947,31948,31949,31950,31952,31953,31954,31955,31956,31957,31958,31959,31960,31961,31962,31964,31965,31966,31967,31968,31970,31974,31975,31976,31978,31980,31981,31982,31983,31984,31985,31986,31987,31988,31989,31990,31991,31992,31993,31994,31995,31996,31997,31998,32000,32001,32002,32003,32004,32005,32006,32007,32008,32009,32010,32011,32012,32013,32014,32015,32016,32017,32018,32019,32020,32021,32022,32023,32024,32025,32026,32027,32028,32029,32030,32031,32032,32033,32034,32037,32039,32040,32041,32043,32044,32046,32047,32048,32049,32050,32051,32053,32054,32056,32057,32058,32059,32060,32061,32062,32063,32064,32065,32066,32067,32068,32069,32070,32071,32072,32074,32075,32076,32077,32078,32079,32080,32081,32082,32083,32084,32085,32086,32088,32090,32091,32092,32093,32094,32095,32097,32098,32099,32102,32103,32104,32105,32106,32107,32109,32110,32111,32112,32113,32114,32115,32117,32118,32119,32121,32122,32123,32124,32125,32127,32128,32129,32131,32132,32133,32134,32136,32137,32139,32140,32141,32142,32143,32145,32146,32147,32148,32149,32150,32151,32153,32154,32155,32156,32157,32158,32159,32160,32161,32162,32163,32164,32166,32167,32168,32169,32170,32171,32172,32173,32174,32175,32176,32177,32178,32179,32180,32181,32183,32184,32185,32186,32187,32188,32189,32190,32191,32192,32193,32194,32196,32197,32198,32199,32201,32202,32203,32204,32205,32206,32207,32208,32209,32210,32211,32212,32213,32214,32215,32216,32217,32218,32219,32220,32221,32222,32223,32224,32225,32227,32228,32229,32230,32231,32232,32233,32234,32236,32238,32239,32240,32241,32242,32243,32244,32245,32246,32247,32249,32250,32251,32253,32254,32257,32259,32260,32261,32263,32264,32265,32266,32267,32268,32269,32270,32271,32272,32273,32274,32275,32276,32277,32278,32279,32282,32283,32284,32285,32286,32287,32288,32289,32290,32291,32292,32293,32294,32295,32297,32298,32299,32301,32302,32303,32304,32305,32306,32307,32308,32309,32310,32311,32312,32313,32314,32315,32316,32317,32318,32319,32320,32321,32322,32323,32324,32325,32326,32327,32328,32329,32330,32331,32332,32333,32336,32337,32338,32339,32340,32341,32342,32343,32344,32345,32346,32347,32348,32349,32350,32351,32352,32353,32354,32355,32357,32358,32359,32360,32361,32362,32363,32364,32365,32366,32367,32368,32370,32371,32372,32373,32374,32375,32376,32377,32378,32379,32380,32381,32382,32383,32384,32385,32386,32387,32390,32391,32392,32393,32394,32395,32396,32397,32398,32399,32400,32401,32402,32403,32404,32405,32406,32407,32408,32409,32410,32411,32412,32415,32416,32417,32418,32419,32420,32421,32422,32423,32424,32425,32426,32427,32428,32429,32431,32432,32433,32434,32435,32437,32438,32439,32440,32441,32442,32445,32446,32447,32448,32449,32450,32451,32452,32453,32454,32455,32456,32457,32458,32459,32460,32461,32462,32463,32464,32465,32466,32467,32468,32469,32470,32471,32472,32473,32474,32475,32476,32477,32478,32479,32480,32481,32482,32483,32485,32486,32487,32488,32489,32490,32491,32493,32494,32495,32496,32497,32498,32499,32500,32501,32502,32503,32504,32506,32507,32508,32509,32510,32511,32512,32513,32514,32515,32516,32517,32518,32519,32520,32521,32523,32524,32525,32526,32527,32529,32530,32531,32532,32533,32534,32535,32536,32537,32538,32539,32540,32541,32543,32544,32545,32546,32547,32548,32549,32550,32551,32552,32553,32554,32555,32556,32557,32558,32559,32560,32561,32562,32563,32564,32565,32566,32567,32568,32569,32570,32573,32574,32575,32576,32577,32578,32579,32580,32581,32583,32584,32585,32586,32587,32588,32589,32590,32591,32592,32593,32594,32595,32596,32597,32599,32600,32602,32603,32604,32605,32606,32607,32608,32609,32610,32611,32613,32614,32615,32616,32617,32618,32619,32620,32621,32622,32624,32625,32626,32627,32628,32629,32630,32631,32632,32633,32634,32635,32636,32637,32638,32639,32641,32642,32643,32645,32646,32647,32648,32649,32650,32651,32652,32653,32654,32655,32657,32658,32659,32660,32661,32662,32663,32666,32667,32668,32669,32670,32671,32672,32673,32674,32675,32676,32677,32678,32679,32680,32681,32684,32685,32686,32687,32688,32689,32690,32691,32692,32693,32694,32695,32696,32697,32698,32699,32700,32701,32702,32703,32704,32705,32706,32707,32709,32710,32711,32713,32714,32715,32716,32717,32718,32719,32720,32721,32722,32724,32725,32727,32728,32731,32732,32733,32734,32735,32736,32737,32738,32739,32741,32742,32743,32744,32745,32746,32747,32748,32749,32750,32751,32752,32753,32754,32755,32756,32757,32759,32760,32761,32762,32763,32764,32765,32766,32767,32768,32769,32770,32771,32772,32773,32774,32775,32776,32779,32780,32781,32782,32783,32784,32785,32786,32788,32789,32790,32791,32792,32793,32795,32796,32797,32798,32799,32800,32801,32802,32804,32805,32806,32807,32808,32809,32810,32812,32814,32815,32816,32817,32819,32820,32821,32822,32823,32824,32825,32827,32828,32829,32830,32831,32834,32835,32838,32839,32840,32842,32843,32844,32845,32847,32848,32849,32850,32852,32854,32856,32858,32859,32860,32861,32862,32863,32865,32866,32867,32868,32870,32871,32872,32873,32874,32876,32879,32880,32881,32882,32883,32884,32885,32886,32887,32888,32889,32893,32894,32895,32896,32898,32899,32900,32901,32902,32903,32905,32906,32907,32908,32911,32912,32914,32915,32917,32918,32920,32921,32922,32923,32924,32925,32927,32928,32929,32930,32931,32932,32933,32934,32935,32937,32938,32939,32940,32941,32942,32943,32945,32946,32948,32949,32950,32951,32952,32954,32956,32957,32958,32959,32960,32961,32962,32963,32964,32965,32966,32967,32968,32969,32970,32972,32973,32974,32975,32976,32977,32980,32981,32982,32983,32984,32985,32986,32987,32988,32989,32990,32992,32993,32995,32996,32997,32998,32999,33000,33001,33002,33003,33004,33005,33007,33008,33009,33010,33011,33012,33013,33014,33016,33017,33018,33019,33020,33021,33022,33024,33025,33026,33027,33029,33030,33031,33032,33033,33034,33036,33037,33038,33039,33040,33041,33042,33043,33044,33045,33046,33047,33048,33049,33050,33051,33053,33054,33055,33057,33058,33059,33060,33061,33063,33065,33066,33067,33068,33069,33071,33072,33073,33074,33075,33076,33078,33079,33080,33081,33082,33085,33086,33090,33091,33092,33094,33095,33096,33098,33099,33100,33101,33102,33103,33104,33105,33106,33107,33108,33109,33110,33113,33114,33115,33116,33118,33119,33120,33121,33122,33124,33125,33126,33127,33129,33131,33132,33133,33134,33135,33136,33137,33138,33139,33140,33142,33143,33144,33145,33146,33147,33148,33149,33150,33151,33152,33154,33155,33156,33158,33159,33160,33161,33162,33163,33164,33165,33167,33169,33171,33173,33175,33176,33177,33178,33179,33180,33181,33182,33183,33184,33186,33187,33188,33189,33190,33191,33192,33193,33194,33195,33196,33198,33200,33201,33202,33203,33204,33205,33206,33207,33208,33209,33210,33211,33212,33213,33214,33215,33216,33217,33218,33219,33220,33221,33222,33223,33224,33225,33226,33228,33229,33231,33232,33233,33234,33235,33237,33239,33240,33241,33242,33243,33245,33246,33247,33248,33249,33250,33251,33252,33253,33254,33255,33256,33257,33258,33260,33261,33262,33263,33266,33267,33268,33270,33271,33272,33273,33274,33275,33276,33278,33279,33280,33281,33282,33284,33285,33286,33287,33288,33289,33290,33291,33292,33293,33294,33296,33297,33298,33300,33301,33302,33303,33304,33306,33307,33308,33309,33310,33311,33312,33313,33314,33315,33317,33318,33320,33321,33322,33323,33324,33325,33326,33327,33328,33329,33330,33331,33332,33333,33334,33335,33336,33337,33338,33339,33340,33341,33342,33343,33344,33346,33348,33349,33351,33353,33355,33358,33359,33360,33361,33362,33363,33364,33365,33366,33367,33368,33369,33370,33371,33372,33373,33374,33375,33377,33378,33379,33380,33381,33382,33384,33385,33386,33387,33388,33389,33390,33391,33392,33393,33394,33395,33396,33397,33398,33399,33400,33401,33402,33403,33404,33405,33406,33407,33408,33410,33411,33412,33413,33415,33416,33418,33419,33421,33422,33423,33424,33425,33426,33427,33428,33431,33432,33433,33434,33435,33436,33437,33438,33439,33440,33441,33442,33443,33444,33445,33446,33447,33448,33449,33450,33451,33452,33453,33454,33455,33456,33457,33459,33460,33461,33462,33463,33464,33465,33466,33467,33468,33469,33470,33471,33472,33473,33474,33475,33476,33477,33479,33480,33482,33483,33484,33485,33486,33487,33488,33489,33490,33491,33492,33493,33494,33495,33496,33497,33499,33500,33502,33503,33504,33505,33506,33507,33508,33509,33510,33511,33512,33514,33515,33516,33517,33518,33519,33520,33521,33522,33523,33524,33525,33526,33527,33528,33529,33530,33531,33532,33533,33534,33535,33536,33537,33538,33539,33540,33541,33542,33543,33544,33545,33547,33548,33549,33550,33551,33553,33556,33557,33558,33559,33560,33561,33562,33563,33564,33565,33566,33568,33570,33571,33572,33573,33574,33575,33576,33577,33578,33579,33580,33581,33583,33585,33586,33587,33588,33589,33590,33591,33592,33593,33594,33595,33596,33597,33599,33600,33601,33602,33603,33604,33605,33606,33607,33608,33609,33610,33611,33612,33613,33614,33615,33616,33617,33618,33619,33620,33622,33623,33624,33626,33627,33628,33630,33631,33632,33633,33634,33635,33636,33637,33638,33639,33640,33641,33642,33643,33644,33645,33646,33647,33651,33652,33653,33654,33655,33656,33658,33659,33660,33661,33662,33663,33665,33667,33669,33670,33671,33672,33673,33674,33675,33676,33677,33678,33679,33680,33681,33682,33683,33684,33685,33686,33687,33688,33689,33690,33691,33692,33693,33694,33695,33696,33698,33699,33700,33701,33702,33703,33704,33705,33706,33707,33708,33710,33711,33712,33713,33714,33715,33716,33717,33718,33719,33720,33721,33722,33724,33725,33726,33727,33728,33729,33730,33731,33732,33733,33734,33735,33736,33737,33738,33739,33740,33741,33742,33743,33745,33747,33748,33749,33750,33751,33752,33753,33755,33756,33757,33758,33759,33760,33761,33762,33763,33764,33765,33767,33768,33769,33770,33771,33772,33773,33774,33775,33776,33777,33778,33779,33780,33781,33782,33783,33784,33785,33786,33787,33788,33789,33790,33791,33793,33795,33796,33797,33798,33799,33801,33802,33803,33804,33805,33806,33807,33808,33809,33810,33811,33812,33814,33816,33819,33820,33821,33824,33825,33826,33827,33828,33829,33830,33831,33832,33833,33834,33835,33836,33837,33838,33839,33840,33841,33842,33843,33844,33845,33846,33847,33848,33849,33850,33851,33852,33853,33854,33855,33856,33858,33859,33860,33861,33862,33863,33864,33865,33866,33867,33868,33869,33870,33872,33873,33874,33875,33876,33877,33878,33879,33880,33881,33882,33883,33884,33885,33886,33887,33888,33889,33890,33891,33892,33893,33894,33895,33896,33897,33899,33900,33901,33902,33903,33904,33905,33906,33907,33908,33909,33910,33911,33912,33913,33914,33915,33917,33918,33919,33920,33922,33924,33926,33927,33928,33929,33931,33932,33933,33934,33935,33936,33937,33938,33939,33940,33942,33943,33944,33945,33946,33947,33948,33949,33950,33951,33952,33953,33954,33955,33956,33959,33960,33961,33962,33963,33964,33965,33966,33967,33968,33969,33970,33972,33974,33976,33977,33978,33979,33980,33981,33982,33983,33984,33985,33986,33988,33989,33990,33991,33993,33994,33995,33996,33997,33998,33999,34000,34001,34002,34003,34004,34006,34007,34009,34010,34011,34012,34013,34014,34015,34016,34017,34018,34019,34020,34021,34022,34023,34024,34025,34026,34027,34028,34030,34031,34032,34033,34034,34035,34036,34038,34039,34040,34041,34042,34043,34044,34045,34046,34047,34048,34050,34051,34053,34054,34055,34056,34057,34058,34059,34060,34061,34062,34063,34064,34065,34066,34067,34068,34069,34070,34071,34072,34073,34074,34076,34077,34078,34079,34080,34081,34083,34084,34085,34086,34087,34088,34089,34090,34091,34092,34093,34094,34095,34096,34097,34099,34100,34101,34103,34104,34105,34106,34107,34108,34109,34110,34112,34113,34114,34115,34116,34117,34118,34119,34120,34121,34122,34123,34124,34125,34126,34129,34130,34131,34132,34133,34134,34135,34136,34137,34138,34139,34141,34142,34143,34144,34145,34146,34147,34148,34149,34150,34151,34152,34153,34154,34155,34156,34157,34158,34159,34160,34161,34162,34163,34164,34165,34166,34167,34168,34169,34170,34171,34172,34174,34176,34177,34178,34179,34180,34181,34182,34183,34184,34185,34186,34187,34188,34189,34190,34191,34192,34193,34195,34196,34197,34198,34199,34200,34201,34202,34203,34204,34205,34206,34207,34208,34209,34210,34211,34212,34214,34215,34216,34217,34218,34219,34220,34222,34223,34224,34225,34227,34228,34229,34230,34231,34232,34233,34234,34237,34238,34239,34240,34241,34242,34243,34244,34245,34246,34247,34248,34249,34251,34253,34254,34255,34256,34257,34258,34259,34261,34262,34263,34264,34265,34266,34268,34269,34270,34271,34272,34273,34274,34275,34276,34277,34278,34280,34281,34282,34283,34284,34285,34286,34287,34288,34289,34290,34294,34295,34296,34297,34298,34299,34300,34301,34302,34303,34304,34305,34306,34308,34309,34310,34311,34313,34314,34315,34316,34317,34319,34320,34321,34323,34324,34326,34327,34328,34329,34330,34331,34332,34334,34335,34336,34337,34338,34339,34340,34341,34342,34343,34344,34345,34346,34348,34349,34350,34351,34352,34353,34354,34355,34356,34357,34358,34360,34361,34362,34363,34364,34366,34367,34368,34370,34371,34373,34374,34375,34376,34379,34380,34381,34382,34383,34384,34385,34386,34387,34388,34389,34390,34393,34394,34395,34396,34398,34399,34401,34402,34403,34404,34405,34407,34408,34409,34410,34411,34412,34413,34414,34415,34416,34417,34418,34419,34420,34423,34425,34426,34427,34428,34429,34430,34431,34432,34433,34434,34437,34438,34439,34442,34443,34444,34445,34446,34448,34449,34450,34451,34452,34453,34454,34455,34456,34457,34458,34460,34461,34462,34464,34465,34466,34467,34468,34469,34471,34472,34473,34474,34475,34476,34477,34479,34480,34481,34482,34483,34484,34485,34486,34487,34488,34489,34490,34491,34492,34493,34494,34495,34496,34497,34498,34499,34500,34501,34502,34503,34504,34505,34506,34507,34508,34509,34510,34511,34512,34513,34515,34516,34518,34519,34520,34521,34522,34523,34524,34525,34526,34527,34530,34531,34532,34534,34536,34537,34538,34539,34540,34541,34542,34543,34544,34545,34546,34547,34548,34549,34550,34551,34552,34553,34554,34555,34558,34560,34561,34562,34563,34564,34565,34566,34567,34568,34569,34570,34571,34572,34573,34574,34577,34578,34579,34581,34583,34584,34585,34586,34587,34588,34590,34592,34593,34594,34595,34596,34597,34598,34599,34600,34601,34602,34604,34605,34606,34608,34609,34610,34611,34612,34613,34615,34616,34618,34619,34620,34622,34623,34624,34625,34626,34627,34630,34631,34632,34633,34635,34636,34637,34638,34639,34640,34641,34642,34643,34644,34645,34646,34647,34648,34649,34650,34651,34652,34653,34654,34655,34656,34657,34658,34659,34660,34661,34662,34663,34664,34665,34666,34667,34668,34669,34670,34671,34672,34673,34675,34676,34677,34678,34679,34680,34681,34682,34683,34684,34685,34686,34687,34689,34690,34691,34692,34693,34694,34695,34696,34697,34699,34700,34701,34703,34704,34705,34706,34707,34708,34710,34711,34712,34714,34715,34716,34717,34718,34719,34722,34723,34724,34725,34728,34729,34730,34731,34732,34733,34734,34735,34736,34737,34738,34739,34740,34741,34742,34743,34744,34745,34746,34747,34748,34749,34750,34751,34752,34753,34754,34755,34756,34757,34758,34759,34760,34761,34762,34763,34764,34766,34768,34769,34770,34771,34772,34774,34775,34776,34777,34778,34779,34780,34781,34782,34783,34784,34785,34786,34787,34788,34789,34790,34791,34792,34794,34795,34796,34797,34798,34799,34802,34803,34804,34805,34806,34807,34809,34810,34811,34812,34814,34815,34816,34817,34818,34819,34820,34821,34822,34823,34824,34825,34826,34827,34828,34829,34830,34831,34832,34833,34835,34836,34837,34838,34839,34840,34841,34843,34844,34845,34847,34848,34849,34850,34851,34852,34853,34854,34855,34856,34857,34858,34859,34860,34861,34862,34863,34864,34865,34866,34867,34869,34870,34871,34872,34873,34875,34876,34877,34878,34879,34880,34881,34882,34883,34884,34885,34886,34888,34890,34891,34892,34893,34894,34895,34898,34899,34900,34901,34902,34903,34905,34906,34907,34909,34910,34912,34913,34914,34915,34916,34917,34919,34920,34921,34922,34923,34924,34925,34926,34927,34928,34929,34930,34932,34933,34934,34935,34937,34940,34941,34942,34943,34944,34945,34946,34947,34948,34949,34951,34952,34953,34955,34956,34957,34958,34959,34960,34961,34962,34963,34965,34966,34967,34968,34969,34970,34971,34972,34974,34975,34976,34977,34978,34980,34983,34984,34986,34987,34988,34989,34990,34992,34993,34994,34996,34997,34998,34999,35000,35001,35002,35004,35005,35006,35007,35008,35009,35010,35011,35012,35013,35014,35015,35017,35018,35019,35020,35021,35022,35023,35024,35026,35028,35029,35030,35031,35032,35033,35034,35035,35036,35037,35038,35039,35041,35042,35043,35044,35045,35046,35047,35048,35051,35052,35054,35055,35056,35057,35058,35059,35060,35061,35062,35063,35064,35065,35066,35067,35068,35069,35070,35071,35072,35073,35074,35076,35077,35078,35079,35081,35082,35083,35084,35086,35088,35089,35090,35091,35092,35093,35094,35095,35096,35097,35098,35099,35100,35101,35102,35103,35105,35106,35107,35108,35109,35110,35111,35113,35114,35115,35116,35117,35118,35119,35120,35121,35122,35123,35124,35125,35126,35127,35128,35131,35132,35133,35134,35137,35138,35139,35140,35142,35143,35145,35147,35148,35149,35151,35152,35153,35154,35155,35156,35158,35159,35160,35161,35162,35163,35164,35165,35166,35167,35168,35169,35170,35171,35172,35173,35174,35177,35178,35179,35180,35181,35182,35183,35185,35186,35187,35188,35190,35191,35193,35194,35195,35196,35198,35199,35200,35201,35202,35203,35205,35206,35207,35208,35209,35210,35211,35215,35217,35219,35220,35221,35222,35223,35224,35226,35227,35228,35229,35230,35231,35233,35234,35235,35236,35237,35238,35239,35241,35242,35244,35245,35246,35247,35250,35251,35254,35255,35257,35258,35260,35261,35262,35263,35264,35265,35266,35268,35269,35270,35271,35272,35273,35274,35275,35276,35278,35279,35280,35281,35282,35283,35284,35285,35286,35289,35290,35291,35292,35293,35294,35295,35296,35297,35298,35299,35300,35301,35302,35303,35304,35305,35307,35308,35309,35311,35312,35313,35314,35315,35316,35318,35319,35320,35322,35323,35324,35326,35327,35328,35330,35331,35332,35335,35336,35338,35340,35342,35343,35344,35345,35346,35347,35349,35350,35351,35352,35355,35356,35357,35358,35359,35362,35363,35365,35367,35369,35370,35371,35372,35373,35376,35377,35379,35380,35382,35383,35384,35385,35386,35387,35388,35389,35390,35391,35392,35393,35396,35397,35398,35400,35401,35402,35404,35405,35406,35407,35408,35409,35410,35412,35413,35414,35415,35416,35417,35419,35422,35424,35425,35426,35427,35430,35431,35432,35433,35435,35436,35437,35438,35440,35441,35442,35443,35444,35445,35446,35447,35449,35450,35451,35452,35454,35455,35457,35458,35459,35460,35461,35462,35463,35465,35466,35467,35468,35469,35471,35472,35473,35474,35475,35476,35477,35478,35480,35481,35482,35484,35486,35488,35489,35491,35492,35493,35494,35495,35496,35497,35498,35499,35500,35501,35503,35504,35506,35508,35510,35512,35513,35514,35515,35516,35517,35518,35519,35520,35522,35523,35524,35525,35526,35527,35528,35529,35531,35532,35533,35535,35537,35538,35539,35540,35541,35542,35543,35544,35545,35546,35547,35548,35549,35550,35551,35552,35553,35554,35556,35558,35559,35560,35562,35563,35565,35566,35567,35568,35569,35570,35571,35572,35573,35574,35575,35576,35577,35578,35579,35580,35582,35583,35584,35585,35586,35588,35589,35590,35591,35592,35594,35595,35596,35597,35598,35599,35600,35601,35602,35603,35604,35605,35606,35607,35608,35609,35610,35611,35612,35613,35614,35615,35616,35617,35618,35619,35620,35621,35622,35623,35624,35626,35627,35628,35629,35630,35631,35632,35633,35635,35637,35638,35639,35641,35642,35643,35644,35645,35646,35647,35648,35649,35650,35651,35653,35654,35655,35656,35657,35658,35659,35660,35661,35662,35663,35664,35665,35666,35667,35668,35669,35670,35671,35672,35673,35674,35675,35676,35677,35678,35679,35680,35682,35683,35685,35686,35687,35688,35689,35690,35691,35692,35693,35695,35696,35697,35698,35700,35703,35704,35705,35706,35707,35709,35710,35711,35712,35713,35714,35715,35716,35717,35718,35720,35722,35723,35724,35726,35728,35730,35731,35732,35733,35734,35736,35737,35738,35739,35740,35742,35743,35744,35745,35746,35747,35748,35749,35750,35751,35752,35753,35754,35755,35757,35758,35759,35760,35762,35763,35764,35765,35766,35767,35768,35769,35770,35772,35773,35774,35775,35776,35777,35778,35779,35780,35781,35782,35784,35785,35786,35787,35788,35789,35790,35791,35793,35794,35795,35796,35797,35798,35799,35800,35801,35802,35803,35804,35805,35806,35807,35808,35809,35810,35811,35812,35813,35814,35815,35816,35817,35819,35820,35821,35822,35823,35824,35825,35826,35827,35828,35829,35830,35831,35832,35833,35834,35835,35836,35837,35838,35839,35840,35841,35842,35843,35844,35845,35846,35847,35848,35850,35851,35852,35853,35854,35855,35856,35857,35858,35859,35860,35861,35862,35863,35864,35865,35866,35867,35868,35869,35871,35872,35873,35874,35875,35876,35877,35878,35879,35880,35881,35882,35883,35884,35885,35886,35887,35888,35889,35890,35891,35892,35893,35894,35895,35897,35898,35899,35900,35901,35902,35903,35905,35906,35907,35909,35910,35911,35912,35913,35914,35915,35916,35917,35918,35919,35920,35921,35924,35925,35926,35927,35930,35932,35933,35935,35937,35938,35940,35941,35942,35944,35945,35946,35947,35948,35949,35951,35952,35953,35954,35955,35957,35958,35959,35960,35961,35962,35963,35964,35965,35968,35969,35970,35972,35973,35974,35977,35978,35980,35981,35982,35983,35984,35985,35986,35987,35988,35989,35991,35992,35993,35994,35995,35996,35997,35998,35999,36000,36001,36002,36003,36004,36005,36007,36008,36009,36010,36011,36012,36013,36014,36015,36016,36018,36019,36020,36021,36022,36023,36024,36025,36026,36027,36028,36029,36030,36031,36032,36033,36034,36035,36036,36037,36039,36040,36042,36044,36045,36046,36047,36049,36050,36051,36052,36053,36054,36055,36057,36058,36059,36060,36061,36062,36063,36064,36065,36066,36067,36068,36069,36070,36071,36072,36073,36074,36075,36076,36077,36078,36080,36081,36082,36083,36084,36085,36087,36088,36089,36090,36091,36092,36093,36094,36096,36098,36099,36100,36101,36102,36103,36104,36105,36106,36107,36108,36109,36111,36112,36113,36114,36115,36116,36117,36118,36119,36120,36121,36123,36124,36125,36126,36127,36129,36130,36131,36132,36133,36134,36135,36136,36137,36138,36139,36140,36141,36142,36143,36144,36145,36146,36147,36148,36149,36150,36151,36152,36153,36154,36155,36156,36157,36158,36159,36160,36161,36162,36163,36164,36165,36166,36167,36168,36169,36170,36171,36172,36173,36174,36175,36176,36179,36180,36181,36182,36184,36185,36186,36187,36188,36189,36190,36192,36193,36194,36195,36196,36198,36199,36200,36201,36203,36204,36205,36206,36207,36208,36209,36210,36211,36212,36213,36214,36215,36216,36217,36218,36219,36221,36224,36225,36226,36228,36229,36233,36234,36235,36236,36237,36238,36239,36240,36241,36242,36243,36244,36245,36246,36249,36251,36252,36255,36256,36257,36259,36261,36262,36263,36264,36265,36266,36267,36268,36269,36270,36271,36273,36274,36275,36276,36277,36278,36279,36280,36281,36282,36284,36286,36287,36288,36289,36290,36291,36292,36293,36294,36295,36296,36299,36300,36301,36302,36303,36304,36305,36307,36308,36309,36310,36311,36312,36313,36314,36315,36316,36317,36318,36319,36320,36321,36322,36323,36324,36326,36327,36328,36329,36330,36331,36332,36334,36335,36336,36337,36338,36339,36340,36341,36343,36344,36345,36346,36347,36348,36349,36350,36351,36352,36353,36354,36355,36356,36357,36358,36359,36360,36361,36362,36364,36365,36366,36367,36368,36369,36370,36371,36372,36373,36374,36375,36376,36377,36378,36379,36380,36381,36382,36383,36384,36385,36386,36387,36388,36389,36390,36391,36392,36393,36394,36395,36396,36397,36398,36399,36400,36401,36403,36404,36405,36406,36408,36409,36410,36412,36413,36414,36415,36416,36417,36418,36420,36421,36422,36423,36424,36425,36426,36427,36428,36429,36430,36431,36432,36433,36434,36435,36436,36437,36438,36439,36441,36442,36443,36444,36445,36446,36447,36448,36449,36450,36451,36452,36453,36454,36455,36456,36457,36458,36460,36461,36463,36464,36465,36466,36467,36468,36469,36470,36471,36472,36474,36475,36476,36478,36479,36480,36481,36482,36484,36485,36486,36487,36488,36489,36490,36491,36492,36493,36494,36495,36496,36497,36498,36499,36500,36501,36502,36503,36504,36505,36506,36508,36509,36510,36511,36512,36513,36515,36516,36517,36518,36519,36520,36521,36522,36523,36524,36525,36527,36528,36529,36530,36534,36537,36538,36540,36541,36542,36544,36546,36547,36549,36550,36552,36553,36554,36555,36556,36557,36558,36559,36561,36562,36563,36564,36567,36568,36570,36571,36572,36573,36574,36575,36576,36577,36578,36579,36580,36581,36582,36583,36584,36585,36587,36588,36589,36590,36591,36593,36594,36596,36597,36598,36599,36600,36601,36602,36603,36604,36605,36606,36607,36608,36609,36610,36611,36613,36614,36615,36616,36617,36618,36619,36620,36621,36622,36624,36625,36626,36627,36628,36629,36630,36631,36632,36633,36634,36635,36636,36637,36638,36639,36640,36643,36644,36645,36646,36649,36650,36652,36653,36654,36655,36656,36658,36659,36660,36661,36662,36663,36664,36665,36667,36670,36671,36672,36673,36674,36675,36676,36677,36678,36679,36680,36681,36682,36683,36684,36685,36686,36687,36688,36689,36690,36691,36692,36693,36694,36695,36696,36697,36698,36699,36700,36701,36702,36703,36704,36705,36706,36707,36708,36710,36711,36712,36713,36715,36716,36717,36718,36719,36720,36721,36722,36723,36724,36725,36726,36727,36728,36729,36730,36731,36732,36733,36734,36735,36737,36738,36739,36740,36741,36742,36743,36744,36745,36746,36747,36749,36750,36751,36752,36753,36755,36756,36757,36758,36759,36760,36761,36762,36763,36764,36766,36767,36768,36771,36773,36774,36775,36776,36777,36779,36781,36782,36783,36784,36785,36786,36787,36788,36789,36790,36791,36792,36793,36794,36795,36796,36797,36798,36799,36801,36802,36804,36805,36806,36807,36808,36809,36810,36811,36812,36813,36814,36815,36816,36817,36818,36819,36820,36821,36822,36823,36824,36825,36826,36827,36828,36829,36830,36831,36832,36833,36834,36835,36836,36837,36838,36840,36841,36842,36843,36845,36846,36847,36848,36851,36852,36853,36854,36855,36856,36857,36858,36859,36860,36861,36862,36863,36864,36865,36866,36867,36868,36869,36870,36872,36873,36874,36875,36876,36877,36878,36879,36880,36881,36882,36883,36884,36885,36886,36887,36889,36890,36891,36892,36893,36894,36895,36896,36897,36898,36899,36900,36902,36903,36909,36910,36911,36913,36914,36915,36916,36917,36918,36919,36920,36921,36923,36924,36925,36926,36927,36929,36930,36932,36933,36935,36937,36938,36939,36940,36941,36942,36943,36944,36945,36946,36947,36948,36949,36950,36951,36952,36953,36955,36956,36957,36958,36960,36961,36962,36963,36964,36965,36967,36968,36969,36971,36972,36973,36974,36975,36976,36978,36979,36980,36981,36982,36983,36984,36985,36986,36987,36988,36989,36990,36991,36992,36993,36994,36995,36996,36997,36998,36999,37000,37001,37002,37003,37005,37007,37008,37009,37011,37012,37013,37015,37016,37017,37019,37021,37022,37023,37024,37025,37026,37027,37029,37030,37031,37032,37034,37036,37038,37039,37040,37041,37042,37043,37044,37045,37046,37048,37049,37050,37051,37053,37054,37055,37057,37059,37060,37061,37063,37064,37066,37067,37070,37071,37072,37073,37075,37076,37077,37078,37079,37080,37081,37082,37083,37084,37085,37086,37087,37088,37089,37090,37091,37092,37093,37094,37095,37096,37097,37098,37099,37100,37101,37103,37104,37105,37106,37107,37108,37109,37111,37112,37113,37114,37115,37116,37117,37118,37119,37120,37121,37122,37123,37124,37125,37126,37127,37128,37129,37131,37133,37134,37135,37136,37137,37138,37140,37141,37142,37143,37144,37145,37146,37147,37148,37149,37150,37151,37152,37153,37154,37155,37156,37158,37159,37160,37161,37162,37163,37164,37165,37166,37167,37168,37169,37170,37171,37172,37173,37174,37176,37177,37178,37179,37182,37183,37184,37185,37187,37188,37189,37190,37191,37192,37193,37194,37195,37196,37197,37198,37199,37200,37201,37202,37203,37204,37205,37206,37207,37208,37209,37210,37212,37213,37214,37215,37216,37217,37218,37219,37220,37221,37223,37224,37225,37226,37228,37230,37231,37232,37233,37234,37235,37236,37237,37238,37239,37240,37241,37242,37244,37245,37246,37247,37248,37249,37250,37251,37252,37253,37254,37255,37257,37258,37259,37260,37261,37262,37263,37264,37265,37266,37267,37271,37273,37274,37275,37276,37277,37278,37279,37280,37281,37282,37283,37284,37285,37287,37288,37289,37290,37291,37292,37293,37294,37295,37296,37297,37298,37299,37300,37301,37302,37303,37304,37305,37306,37307,37308,37309,37310,37312,37313,37314,37315,37316,37317,37318,37319,37320,37321,37322,37323,37324,37325,37326,37327,37328,37329,37331,37332,37333,37334,37335,37336,37337,37338,37339,37340,37341,37342,37343,37345,37346,37347,37348,37349,37350,37351,37352,37353,37354,37355,37356,37357,37358,37361,37363,37364,37365,37366,37367,37368,37369,37370,37372,37373,37374,37375,37376,37377,37378,37379,37380,37381,37382,37383,37384,37385,37386,37388,37389,37390,37391,37392,37393,37394,37395,37396,37397,37398,37399,37401,37402,37404,37406,37409,37411,37412,37413,37414,37415,37416,37417,37418,37419,37420,37421,37422,37424,37425,37426,37427,37428,37429,37430,37431,37432,37433,37434,37436,37437,37438,37439,37440,37441,37444,37445,37446,37448,37449,37450,37451,37452,37453,37454,37455,37456,37457,37458,37459,37460,37461,37462,37463,37464,37465,37466,37467,37469,37470,37471,37472,37473,37474,37475,37476,37477,37478,37479,37483,37484,37485,37486,37487,37488,37489,37490,37491,37492,37494,37495,37496,37497,37498,37499,37500,37501,37502,37503,37504,37505,37506,37507,37508,37509,37510,37511,37512,37513,37514,37515,37516,37517,37518,37519,37521,37523,37524,37525,37526,37527,37528,37529,37530,37531,37532,37533,37536,37537,37538,37539,37540,37541,37542,37543,37544,37545,37546,37547,37548,37549,37550,37553,37554,37555,37556,37557,37558,37559,37561,37562,37563,37564,37566,37567,37568,37569,37570,37571,37572,37573,37574,37575,37576,37577,37578,37579,37580,37581,37582,37583,37584,37585,37586,37587,37588,37589,37591,37592,37593,37595,37597,37598,37599,37600,37601,37603,37604,37605,37606,37607,37608,37609,37610,37611,37612,37613,37614,37615,37616,37617,37618,37619,37620,37622,37623,37624,37625,37626,37627,37628,37629,37630,37631,37632,37633,37634,37635,37636,37638,37639,37640,37641,37643,37644,37645,37646,37647,37648,37650,37651,37652,37653,37654,37656,37657,37658,37659,37661,37662,37663,37664,37665,37666,37667,37668,37669,37670,37671,37672,37673,37674,37675,37676,37677,37678,37679,37680,37681,37682,37683,37684,37685,37686,37688,37689,37690,37691,37692,37694,37696,37697,37698,37699,37700,37701,37702,37703,37704,37705,37706,37707,37708,37709,37710,37711,37712,37713,37714,37716,37717,37718,37719,37720,37721,37722,37723,37724,37726,37727,37728,37729,37730,37731,37732,37733,37734,37735,37736,37737,37738,37739,37740,37741,37742,37744,37745,37747,37748,37749,37750,37751,37752,37753,37754,37755,37756,37757,37758,37760,37761,37762,37763,37764,37765,37766,37767,37768,37769,37770,37772,37773,37774,37775,37776,37777,37778,37780,37781,37782,37783,37784,37785,37786,37787,37788,37789,37790,37791,37792,37793,37794,37795,37796,37797,37798,37799,37800,37801,37802,37804,37805,37806,37807,37808,37809,37810,37811,37812,37813,37815,37816,37817,37818,37819,37821,37823,37824,37826,37827,37828,37830,37831,37832,37834,37835,37836,37837,37838,37839,37840,37841,37842,37843,37844,37845,37846,37847,37848,37849,37850,37851,37852,37853,37854,37855,37856,37857,37858,37859,37860,37861,37862,37863,37864,37868,37870,37872,37873,37875,37876,37877,37878,37879,37880,37881,37882,37883,37884,37885,37886,37887,37888,37889,37891,37892,37894,37895,37896,37897,37898,37899,37900,37901,37902,37903,37904,37905,37906,37907,37908,37909,37910,37911,37912,37913,37914,37915,37917,37920,37921,37924,37925,37926,37927,37928,37929,37930,37931,37932,37933,37934,37936,37937,37938,37939,37941,37942,37943,37944,37945,37946,37947,37948,37949,37950,37951,37952,37953,37954,37955,37956,37957,37958,37959,37960,37961,37962,37963,37964,37965,37967,37968,37969,37970,37971,37972,37973,37975,37976,37978,37979,37981,37982,37984,37986,37987,37988,37989,37991,37992,37993,37994,37995,37996,37997,37998,37999,38000,38001,38002,38003,38004,38005,38006,38007,38008,38009,38011,38012,38013,38014,38015,38016,38017,38018,38019,38021,38022,38023,38024,38025,38026,38027,38028,38029,38030,38031,38032,38034,38035,38036,38037,38039,38041,38042,38043,38044,38045,38046,38047,38048,38049,38050,38051,38052,38053,38054,38055,38056,38057,38058,38059,38060,38061,38062,38063,38064,38065,38066,38067,38068,38069,38070,38071,38072,38073,38074,38075,38076,38077,38078,38079,38080,38081,38082,38083,38084,38085,38086,38088,38089,38090,38091,38092,38093,38094,38096,38097,38098,38101,38102,38103,38104,38105,38107,38108,38109,38110,38111,38112,38113,38114,38115,38116,38117,38119,38120,38121,38122,38123,38124,38125,38126,38127,38128,38129,38130,38131,38132,38133,38134,38135,38136,38137,38138,38140,38141,38142,38143,38144,38145,38146,38147,38148,38149,38150,38151,38152,38153,38154,38155,38156,38157,38158,38159,38160,38161,38162,38163,38164,38165,38166,38167,38168,38169,38170,38171,38173,38174,38175,38177,38178,38179,38180,38181,38182,38184,38185,38186,38187,38188,38189,38190,38191,38192,38193,38194,38196,38197,38198,38199,38200,38201,38202,38203,38204,38206,38207,38208,38209,38210,38212,38213,38214,38215,38217,38218,38220,38221,38222,38223,38224,38225,38226,38227,38228,38230,38231,38232,38233,38235,38236,38237,38238,38239,38241,38242,38243,38244,38245,38246,38247,38248,38249,38250,38251,38252,38253,38255,38256,38257,38258,38259,38262,38263,38264,38266,38267,38268,38269,38271,38272,38274,38275,38277,38278,38279,38280,38281,38282,38283,38284,38285,38286,38287,38288,38289,38290,38291,38292,38294,38295,38296,38297,38299,38300,38302,38303,38304,38305,38306,38307,38308,38309,38310,38311,38312,38313,38314,38315,38316,38317,38318,38320,38321,38322,38325,38326,38327,38329,38330,38331,38332,38333,38334,38335,38336,38339,38341,38342,38343,38344,38345,38346,38347,38348,38349,38352,38353,38354,38355,38356,38357,38358,38360,38362,38363,38364,38366,38367,38368,38369,38370,38371,38372,38373,38376,38377,38378,38379,38381,38382,38383,38384,38385,38386,38387,38388,38389,38390,38391,38392,38393,38394,38395,38396,38397,38398,38400,38401,38402,38403,38404,38405,38406,38408,38409,38410,38411,38412,38413,38414,38415,38416,38417,38418,38420,38421,38422,38423,38425,38426,38428,38429,38430,38431,38432,38433,38434,38435,38436,38440,38442,38444,38445,38446,38447,38448,38449,38450,38451,38452,38453,38454,38456,38457,38458,38459,38460,38461,38463,38464,38465,38466,38467,38468,38469,38470,38471,38472,38473,38474,38475,38476,38477,38478,38479,38480,38481,38483,38484,38485,38486,38488,38491,38492,38493,38494,38495,38497,38498,38499,38500,38501,38502,38503,38504,38505,38506,38507,38508,38509,38511,38512,38513,38514,38515,38516,38517,38518,38519,38520,38522,38523,38524,38525,38526,38528,38529,38531,38532,38533,38534,38535,38536,38537,38538,38539,38541,38542,38543,38544,38545,38546,38547,38548,38549,38550,38551,38552,38553,38555,38556,38557,38558,38560,38561,38562,38563,38564,38565,38567,38568,38569,38570,38572,38574,38575,38576,38577,38578,38579,38580,38582,38583,38584,38585,38587,38588,38589,38590,38591,38592,38593,38594,38595,38596,38597,38598,38599,38600,38601,38602,38603,38604,38605,38606,38607,38609,38610,38611,38612,38613,38614,38615,38616,38617,38618,38619,38620,38621,38622,38623,38624,38625,38626,38627,38629,38632,38633,38634,38635,38639,38640,38641,38642,38643,38644,38645,38646,38647,38648,38649,38650,38651,38653,38654,38655,38656,38657,38658,38659,38660,38661,38662,38663,38664,38665,38666,38667,38669,38670,38671,38672,38673,38674,38675,38678,38680,38681,38683,38684,38685,38686,38687,38688,38689,38690,38691,38692,38693,38694,38695,38696,38697,38698,38699,38700,38701,38702,38703,38704,38705,38706,38707,38708,38709,38710,38712,38713,38714,38715,38717,38718,38719,38720,38721,38722,38723,38724,38726,38727,38728,38729,38731,38733,38735,38737,38738,38739,38741,38742,38743,38744,38745,38746,38747,38748,38749,38750,38751,38752,38753,38754,38756,38757,38758,38760,38761,38762,38763,38764,38765,38766,38768,38769,38770,38771,38772,38774,38775,38776,38777,38778,38779,38780,38781,38782,38783,38784,38785,38786,38787,38788,38789,38790,38791,38792,38793,38794,38795,38797,38798,38799,38800,38801,38802,38804,38807,38808,38809,38810,38811,38812,38813,38814,38815,38816,38817,38818,38819,38820,38821,38822,38824,38826,38827,38828,38829,38830,38831,38833,38834,38835,38836,38838,38839,38840,38841,38842,38843,38845,38846,38847,38848,38849,38850,38851,38852,38853,38854,38855,38856,38857,38859,38860,38861,38862,38863,38864,38866,38867,38868,38869,38870,38871,38872,38873,38876,38877,38878,38879,38880,38881,38883,38885,38886,38887,38889,38890,38891,38892,38893,38894,38896,38897,38898,38899,38901,38902,38904,38905,38906,38907,38909,38910,38911,38912,38913,38914,38915,38916,38917,38918,38919,38920,38922,38924,38925,38926,38927,38928,38929,38930,38931,38932,38933,38934,38935,38936,38938,38939,38940,38941,38942,38943,38944,38945,38947,38948,38950,38951,38952,38953,38955,38956,38957,38958,38959,38960,38962,38963,38964,38965,38967,38968,38969,38971,38972,38973,38977,38979,38980,38981,38982,38983,38984,38985,38986,38987,38988,38989,38990,38991,38992,38993,38994,38995,38996,38997,38998,38999,39000,39001,39003,39004,39005,39006,39007,39008,39010,39011,39012,39013,39014,39015,39016,39017,39018,39019,39020,39023,39024,39025,39026,39027,39028,39029,39030,39031,39032,39033,39034,39035,39036,39037,39038,39039,39040,39041,39042,39043,39044,39045,39046,39047,39048,39049,39050,39052,39053,39055,39056,39057,39059,39060,39062,39063,39064,39066,39067,39068,39069,39070,39071,39072,39073,39074,39076,39077,39078,39079,39080,39081,39082,39084,39085,39086,39087,39089,39090,39091,39092,39094,39095,39096,39097,39098,39099,39100,39101,39102,39103,39104,39105,39106,39107,39108,39110,39111,39112,39113,39114,39115,39116,39118,39121,39122,39123,39125,39128,39129,39130,39131,39132,39134,39135,39136,39137,39138,39139,39141,39142,39143,39144,39145,39146,39147,39148,39149,39150,39151,39153,39154,39156,39157,39158,39161,39162,39164,39165,39166,39168,39170,39171,39173,39175,39176,39177,39178,39180,39181,39182,39184,39185,39186,39187,39188,39189,39190,39191,39192,39193,39194,39195,39196,39197,39198,39199,39200,39201,39204,39205,39207,39208,39209,39210,39211,39212,39213,39214,39215,39216,39217,39218,39219,39221,39223,39224,39225,39226,39227,39228,39229,39230,39231,39232,39233,39234,39235,39237,39239,39240,39241,39242,39243,39244,39245,39246,39248,39249,39250,39251,39252,39253,39254,39255,39256,39257,39259,39260,39261,39262,39263,39265,39266,39267,39269,39271,39272,39273,39274,39275,39276,39277,39278,39279,39280,39281,39282,39284,39285,39286,39287,39290,39292,39293,39295,39296,39297,39300,39301,39302,39303,39304,39306,39307,39309,39311,39312,39313,39314,39315,39316,39317,39318,39319,39320,39321,39323,39324,39325,39326,39329,39331,39332,39333,39334,39335,39336,39338,39339,39340,39341,39342,39343,39344,39345,39346,39347,39348,39349,39352,39353,39354,39355,39356,39357,39361,39362,39363,39364,39365,39366,39367,39368,39369,39371,39372,39373,39374,39375,39376,39377,39378,39379,39380,39381,39382,39383,39384,39385,39386,39387,39388,39389,39391,39392,39393,39394,39395,39396,39397,39398,39399,39401,39402,39404,39405,39406,39408,39409,39410,39412,39413,39414,39415,39416,39417,39418,39419,39420,39421,39422,39423,39425,39426,39427,39428,39429,39430,39431,39433,39434,39435,39436,39437,39438,39439,39440,39441,39442,39443,39444,39445,39446,39449,39450,39451,39452,39453,39454,39455,39456,39457,39458,39459,39460,39461,39462,39463,39464,39465,39466,39467,39468,39469,39470,39471,39472,39473,39474,39476,39477,39478,39479,39480,39481,39482,39483,39484,39485,39486,39487,39488,39489,39490,39491,39492,39493,39494,39496,39497,39498,39500,39501,39502,39503,39504,39506,39507,39508,39509,39510,39511,39512,39513,39514,39515,39516,39518,39519,39520,39522,39523,39524,39525,39526,39527,39528,39529,39530,39531,39532,39533,39534,39535,39536,39537,39539,39540,39541,39542,39543,39544,39545,39546,39547,39548,39549,39550,39551,39552,39553,39554,39556,39557,39558,39559,39560,39562,39563,39564,39567,39568,39569,39570,39571,39574,39575,39576,39578,39579,39580,39581,39582,39583,39584,39585,39586,39587,39588,39589,39591,39592,39595,39597,39599,39600,39601,39602,39603,39604,39606,39607,39608,39609,39610,39611,39612,39613,39614,39615,39616,39617,39618,39620,39621,39622,39623,39626,39627,39628,39629,39631,39632,39633,39634,39635,39636,39637,39638,39639,39640,39641,39644,39646,39647,39648,39649,39650,39651,39654,39655,39658,39659,39660,39661,39662,39663,39665,39666,39667,39668,39670,39671,39673,39674,39675,39676,39677,39678,39679,39681,39682,39683,39684,39685,39686,39688,39689,39690,39691,39692,39693,39694,39695,39696,39697,39698,39699,39700,39701,39702,39703,39704,39705,39706,39710,39711,39712,39714,39715,39716,39717,39719,39720,39721,39722,39723,39725,39726,39727,39729,39730,39731,39732,39733,39735,39737,39738,39739,39740,39742,39743,39744,39745,39746,39747,39748,39749,39750,39751,39752,39753,39754,39755,39756,39757,39758,39759,39760,39761,39762,39764,39765,39766,39768,39769,39770,39771,39775,39776,39777,39780,39782,39783,39784,39785,39788,39791,39792,39793,39794,39796,39797,39798,39799,39802,39803,39804,39805,39806,39807,39808,39809,39810,39811,39813,39814,39815,39816,39819,39821,39822,39823,39824,39825,39826,39827,39829,39830,39831,39834,39835,39837,39838,39839,39840,39841,39842,39844,39845,39846,39848,39850,39851,39853,39854,39855,39856,39857,39860,39861,39862,39864,39865,39867,39869,39871,39872,39873,39875,39876,39878,39879,39880,39881,39882,39887,39889,39890,39891,39892,39893,39894,39895,39897,39898,39899,39900,39901,39902,39904,39905,39906,39907,39908,39909,39910,39911,39912,39913,39914,39915,39916,39917,39920,39921,39922,39924,39925,39927,39928,39933,39935,39936,39938,39940,39941,39942,39943,39944,39945,39946,39947,39948,39949,39950,39952,39954,39955,39956,39957,39959,39963,39964,39965,39967,39968,39969,39971,39972,39973,39974,39976,39977,39979,39980,39981,39982,39983,39984,39985,39986,39987,39988,39989,39990,39991,39993,39994,39995,39996,39997,39998,39999,40000,40001,40004,40005,40006,40007,40008,40009,40010,40011,40012,40013,40014,40015,40016,40018,40019,40020,40021,40022,40023,40024,40025,40026,40029,40030,40031,40032,40034,40035,40038,40039,40040,40041,40045,40046,40049,40050,40051,40052,40053,40054,40055,40056,40057,40058,40059,40060,40063,40065,40066,40069,40070,40071,40072,40075,40077,40078,40080,40081,40082,40084,40085,40087,40090,40091,40092,40094,40095,40096,40097,40098,40099,40100,40101,40102,40103,40104,40105,40107,40109,40110,40112,40113,40114,40115,40116,40117,40118,40119,40120,40122,40123,40124,40125,40131,40132,40133,40134,40135,40138,40139,40140,40141,40142,40143,40144,40147,40148,40149,40150,40151,40152,40153,40156,40157,40158,40159,40162,40165,40166,40167,40169,40170,40171,40172,40173,40176,40177,40178,40179,40180,40181,40182,40183,40185,40186,40187,40188,40189,40191,40192,40194,40195,40196,40197,40198,40199,40200,40201,40204,40206,40208,40210,40212,40213,40214,40215,40216,40217,40219,40221,40222,40223,40224,40225,40226,40227,40229,40230,40232,40233,40234,40235,40236,40237,40238,40239,40240,40241,40243,40244,40246,40247,40248,40249,40251,40253,40254,40255,40256,40257,40258,40259,40260,40261,40262,40263,40264,40265,40266,40267,40268,40270,40271,40272,40273,40274,40275,40276,40278,40279,40280,40281,40282,40283,40284,40285,40286,40287,40288,40289,40292,40293,40295,40296,40297,40298,40299,40300,40301,40302,40303,40304,40305,40306,40307,40308,40309,40311,40312,40313,40314,40315,40316,40317,40318,40319,40320,40321,40322,40323,40324,40325,40326,40327,40328,40329,40330,40331,40332,40335,40336,40338,40339,40340,40342,40343,40344,40345,40346,40347,40348,40349,40350,40351,40352,40353,40354,40355,40356,40357,40358,40359,40360,40361,40362,40363,40364,40365,40367,40369,40370,40371,40372,40373,40374,40375,40376,40377,40378,40379,40380,40381,40382,40383,40384,40385,40386,40387,40388,40389,40390,40391,40392,40393,40394,40395,40396,40397,40398,40399,40400,40401,40402,40403,40404,40405,40406,40407,40408,40409,40410,40411,40412,40413,40414,40415,40417,40418,40419,40420,40421,40422,40424,40425,40427,40428,40429,40430,40431,40432,40434,40435,40436,40437,40438,40439,40440,40441,40442,40443,40444,40445,40446,40447,40448,40449,40450,40451,40452,40453,40454,40455,40457,40458,40459,40460,40461,40462,40463,40464,40465,40466,40467,40468,40469,40471,40472,40473,40474,40475,40476,40477,40478,40479,40480,40481,40482,40483,40485,40486,40488,40489,40490,40491,40492,40493,40495,40497,40498,40499,40501,40502,40503,40504,40505,40506,40509,40510,40511,40513,40514,40515,40516,40517,40518,40519,40520,40521,40522,40523,40524,40526,40527,40529,40533,40535,40536,40538,40539,40540,40542,40547,40548,40550,40551,40552,40553,40554,40555,40556,40557,40560,40561,40563,40565,40568,40569,40570,40571,40572,40573,40574,40575,40576,40577,40578,40579,40580,40581,40582,40583,40584,40585,40586,40587,40588,40589,40590,40592,40593,40594,40595,40596,40597,40598,40599,40600,40601,40602,40603,40604,40605,40606,40607,40608,40609,40610,40612,40613,40614,40615,40616,40617,40618,40619,40620,40621,40622,40623,40624,40625,40628,40629,40630,40631,40632,40633,40634,40635,40636,40637,40638,40639,40640,40641,40642,40643,40644,40646,40647,40648,40649,40652,40653,40654,40655,40656,40657,40658,40659,40660,40661,40662,40664,40665,40666,40667,40668,40669,40670,40671,40672,40674,40676,40677,40678,40679,40680,40681,40682,40683,40685,40686,40687,40688,40689,40690,40691,40692,40693,40694,40695,40696,40697,40698,40699,40700,40701,40702,40703,40704,40705,40706,40710,40711,40712,40713,40714,40715,40717,40718,40719,40720,40722,40723,40725,40726,40727,40728,40729,40730,40731,40732,40734,40736,40737,40738,40739,40740,40741,40742,40743,40744,40745,40746,40747,40748,40749,40750,40751,40752,40753,40754,40755,40756,40757,40758,40759,40760,40761,40763,40765,40766,40768,40769,40770,40771,40772,40773,40774,40775,40776,40777,40778,40779,40780,40781,40782,40783,40784,40785,40786,40787,40788,40789,40790,40791,40792,40793,40794,40795,40796,40797,40798,40799,40800,40801,40802,40803,40804,40805,40806,40807,40809,40810,40811,40812,40814,40815,40816,40817,40818,40820,40821,40822,40823,40824,40825,40826,40827,40830,40831,40832,40835,40836,40837,40838,40839,40840,40841,40842,40843,40844,40845,40846,40848,40849,40850,40852,40853,40856,40857,40858,40859,40860,40861,40863,40864,40866,40868,40869,40880,40881,57344,57345,57346,57347,57348,57349,57350,57351,57352,57353,57354,57355,57356,57357,57358,57359,57360,57361,57362,57363,57364,57365,57366,57367,57368,57369,57370,57371,57372,57373,57374,57375,57376,57377,57378,57379,57380,57381,57382,57383,57384,57385,57386,57387,57388,57389,57390,57391,57392,57393,57394,57395,57396,57397,57398,57399,57400,57401,57402,57403,57404,57405,57406,57407,57408,57409,57410,57411,57412,57413,57414,57415,57416,57417,57418,57419,57420,57421,57422,57423,57424,57425,57426,57427,57428,57429,57430,57431,57432,57433,57434,57435,57436,57437,57438,57439,57440,57441,57442,57443,57444,57445,57446,57447,57448,57449,57450,57451,57452,57453,57454,57455,57456,57457,57458,57459,57460,57461,57462,57463,57464,57465,57466,57467,57468,57469,57470,57471,57472,57473,57474,57475,57476,57477,57478,57479,57480,57481,57482,57483,57484,57485,57486,57487,57488,57489,57490,57491,57492,57493,57494,57495,57496,57497,57498,57499,57500,57501,57502,57503,57504,57505,57506,57507,57508,57509,57510,57511,57512,57513,57514,57515,57516,57517,57518,57519,57520,57521,57522,57523,57524,57525,57526,57527,57528,57529,57530,57531,57532,57533,57534,57535,57536,57537,57538,57539,57540,57541,57542,57543,57544,57545,57546,57547,57548,57549,57550,57551,57552,57553,57554,57555,57556,57557,57558,57559,57560,57561,57562,57563,57564,57565,57566,57567,57568,57569,57570,57571,57572,57573,57574,57575,57576,57577,57578,57579,57580,57581,57582,57583,57584,57585,57586,57587,57588,57589,57590,57591,57592,57593,57594,57595,57596,57597,57598,57599,57600,57601,57602,57603,57604,57605,57606,57607,57608,57609,57610,57611,57612,57613,57614,57615,57616,57617,57618,57619,57620,57621,57622,57623,57624,57625,57626,57627,57628,57629,57630,57631,57632,57633,57634,57635,57636,57637,57638,57639,57640,57641,57642,57643,57644,57645,57646,57647,57648,57649,57650,57651,57652,57653,57654,57655,57656,57657,57658,57659,57660,57661,57662,57663,57664,57665,57666,57667,57668,57669,57670,57671,57672,57673,57674,57675,57676,57677,57678,57679,57680,57681,57682,57683,57684,57685,57686,57687,57688,57689,57690,57691,57692,57693,57694,57695,57696,57697,57698,57699,57700,57701,57702,57703,57704,57705,57706,57707,57708,57709,57710,57711,57712,57713,57714,57715,57716,57717,57718,57719,57720,57721,57722,57723,57724,57725,57726,57727,57728,57729,57730,57731,57732,57733,57734,57735,57736,57737,57738,57739,57740,57741,57742,57743,57744,57745,57746,57747,57748,57749,57750,57751,57752,57753,57754,57755,57756,57757,57758,57759,57760,57761,57762,57763,57764,57765,57766,57767,57768,57769,57770,57771,57772,57773,57774,57775,57776,57777,57778,57779,57780,57781,57782,57783,57784,57785,57786,57787,57788,57789,57790,57791,57792,57793,57794,57795,57796,57797,57798,57799,57800,57801,57802,57803,57804,57805,57806,57807,57808,57809,57810,57811,57812,57813,57814,57815,57816,57817,57818,57819,57820,57821,57822,57823,57824,57825,57826,57827,57828,57829,57830,57831,57832,57833,57834,57835,57836,57837,57838,57839,57840,57841,57842,57843,57844,57845,57846,57847,57848,57849,57850,57851,57852,57853,57854,57855,57856,57857,57858,57859,57860,57861,57862,57863,57864,57865,57866,57867,57868,57869,57870,57871,57872,57873,57874,57875,57876,57877,57878,57879,57880,57881,57882,57883,57884,57885,57886,57887,57888,57889,57890,57891,57892,57893,57894,57895,57896,57897,57898,57899,57900,57901,57902,57903,57904,57905,57906,57907,57908,57909,57910,57911,57912,57913,57914,57915,57916,57917,57918,57919,57920,57921,57922,57923,57924,57925,57926,57927,57928,57929,57930,57931,57932,57933,57934,57935,57936,57937,57938,57939,57940,57941,57942,57943,57944,57945,57946,57947,57948,57949,57950,57951,57952,57953,57954,57955,57956,57957,57958,57959,57960,57961,57962,57963,57964,57965,57966,57967,57968,57969,57970,57971,57972,57973,57974,57975,57976,57977,57978,57979,57980,57981,57982,57983,57984,57985,57986,57987,57988,57989,57990,57991,57992,57993,57994,57995,57996,57997,57998,57999,58000,58001,58002,58003,58004,58005,58006,58007,58008,58009,58010,58011,58012,58013,58014,58015,58016,58017,58018,58019,58020,58021,58022,58023,58024,58025,58026,58027,58028,58029,58030,58031,58032,58033,58034,58035,58036,58037,58038,58039,58040,58041,58042,58043,58044,58045,58046,58047,58048,58049,58050,58051,58052,58053,58054,58055,58056,58057,58058,58059,58060,58061,58062,58063,58064,58065,58066,58067,58068,58069,58070,58071,58072,58073,58074,58075,58076,58077,58078,58079,58080,58081,58082,58083,58084,58085,58086,58087,58088,58089,58090,58091,58092,58093,58094,58095,58096,58097,58098,58099,58100,58101,58102,58103,58104,58105,58106,58107,58108,58109,58110,58111,58112,58113,58114,58115,58116,58117,58118,58119,58120,58121,58122,58123,58124,58125,58126,58127,58128,58129,58130,58131,58132,58133,58134,58135,58136,58137,58138,58139,58140,58141,58142,58143,58144,58145,58146,58147,58148,58149,58150,58151,58152,58153,58154,58155,58156,58157,58158,58159,58160,58161,58162,58163,58164,58165,58166,58167,58168,58169,58170,58171,58172,58173,58174,58175,58176,58177,58178,58179,58180,58181,58182,58183,58184,58185,58186,58187,58188,58189,58190,58191,58192,58193,58194,58195,58196,58197,58198,58199,58200,58201,58202,58203,58204,58205,58206,58207,58208,58209,58210,58211,58212,58213,58214,58215,58216,58217,58218,58219,58220,58221,58222,58223,58224,58225,58226,58227,58228,58229,58230,58231,58232,58233,58234,58235,58236,58237,58238,58239,58240,58241,58242,58243,58244,58245,58246,58247,58248,58249,58250,58251,58252,58253,58254,58255,58256,58257,58258,58259,58260,58261,58262,58263,58264,58265,58266,58267,58268,58269,58270,58271,58272,58273,58274,58275,58276,58277,58278,58279,58280,58281,58282,58283,58284,58285,58286,58287,58288,58289,58290,58291,58292,58293,58294,58295,58296,58297,58298,58299,58300,58301,58302,58303,58304,58305,58306,58307,58308,58309,58310,58311,58312,58313,58314,58315,58316,58317,58318,58319,58320,58321,58322,58323,58324,58325,58326,58327,58328,58329,58330,58331,58332,58333,58334,58335,58336,58337,58338,58339,58340,58341,58342,58343,58344,58345,58346,58347,58348,58349,58350,58351,58352,58353,58354,58355,58356,58357,58358,58359,58360,58361,58362,58363,58364,58365,58366,58367,58368,58369,58370,58371,58372,58373,58374,58375,58376,58377,58378,58379,58380,58381,58382,58383,58384,58385,58386,58387,58388,58389,58390,58391,58392,58393,58394,58395,58396,58397,58398,58399,58400,58401,58402,58403,58404,58405,58406,58407,58408,58409,58410,58411,58412,58413,58414,58415,58416,58417,58418,58419,58420,58421,58422,58423,58424,58425,58426,58427,58428,58429,58430,58431,58432,58433,58434,58435,58436,58437,58438,58439,58440,58441,58442,58443,58444,58445,58446,58447,58448,58449,58450,58451,58452,58453,58454,58455,58456,58457,58458,58459,58460,58461,58462,58463,58464,58465,58466,58467,58468,58469,58470,58471,58472,58473,58474,58475,58476,58477,58478,58479,58480,58481,58482,58483,58484,58485,58486,58487,58488,58489,58490,58491,58492,58493,58494,58495,58496,58497,58498,58499,58500,58501,58502,58503,58504,58505,58506,58507,58508,58509,58510,58511,58512,58513,58514,58515,58516,58517,58518,58519,58520,58521,58522,58523,58524,58525,58526,58527,58528,58529,58530,58531,58532,58533,58534,58535,58536,58537,58538,58539,58540,58541,58542,58543,58544,58545,58546,58547,58548,58549,58550,58551,58552,58553,58554,58555,58556,58557,58558,58559,58560,58561,58562,58563,58564,58565,58566,58567,58568,58569,58570,58571,58572,58573,58574,58575,58576,58577,58578,58579,58580,58581,58582,58583,58584,58585,58586,58587,58588,58589,58590,58591,58592,58593,58594,58595,58596,58597,58598,58599,58600,58601,58602,58603,58604,58605,58606,58607,58608,58609,58610,58611,58612,58613,58614,58615,58616,58617,58618,58619,58620,58621,58622,58623,58624,58625,58626,58627,58628,58629,58630,58631,58632,58633,58634,58635,58636,58637,58638,58639,58640,58641,58642,58643,58644,58645,58646,58647,58648,58649,58650,58651,58652,58653,58654,58655,58656,58657,58658,58659,58660,58661,58662,58663,58664,58665,58666,58667,58668,58669,58670,58671,58672,58673,58674,58675,58676,58677,58678,58679,58680,58681,58682,58683,58684,58685,58686,58687,58688,58689,58690,58691,58692,58693,58694,58695,58696,58697,58698,58699,58700,58701,58702,58703,58704,58705,58706,58707,58708,58709,58710,58711,58712,58713,58714,58715,58716,58717,58718,58719,58720,58721,58722,58723,58724,58725,58726,58727,58728,58729,58730,58731,58732,58733,58734,58735,58736,58737,58738,58739,58740,58741,58742,58743,58744,58745,58746,58747,58748,58749,58750,58751,58752,58753,58754,58755,58756,58757,58758,58759,58760,58761,58762,58763,58764,58765,58766,58767,58768,58769,58770,58771,58772,58773,58774,58775,58776,58777,58778,58779,58780,58781,58782,58783,58784,58785,58786,58787,58788,58789,58790,58791,58792,58793,58794,58795,58796,58797,58798,58799,58800,58801,58802,58803,58804,58805,58806,58807,58808,58809,58810,58811,58812,58813,58814,58815,58816,58817,58818,58819,58820,58821,58822,58823,58824,58825,58826,58827,58828,58829,58830,58831,58832,58833,58834,58835,58836,58837,58838,58839,58840,58841,58842,58843,58844,58845,58846,58847,58848,58849,58850,58851,58852,58853,58854,58855,58856,58857,58858,58859,58860,58861,58862,58863,58864,58865,58866,58867,58868,58869,58870,58871,58872,58873,58874,58875,58876,58877,58878,58879,58880,58881,58882,58883,58884,58885,58886,58887,58888,58889,58890,58891,58892,58893,58894,58895,58896,58897,58898,58899,58900,58901,58902,58903,58904,58905,58906,58907,58908,58909,58910,58911,58912,58913,58914,58915,58916,58917,58918,58919,58920,58921,58922,58923,58924,58925,58926,58927,58928,58929,58930,58931,58932,58933,58934,58935,58936,58937,58938,58939,58940,58941,58942,58943,58944,58945,58946,58947,58948,58949,58950,58951,58952,58953,58954,58955,58956,58957,58958,58959,58960,58961,58962,58963,58964,58965,58966,58967,58968,58969,58970,58971,58972,58973,58974,58975,58976,58977,58978,58979,58980,58981,58982,58983,58984,58985,58986,58987,58988,58989,58990,58991,58992,58993,58994,58995,58996,58997,58998,58999,59000,59001,59002,59003,59004,59005,59006,59007,59008,59009,59010,59011,59012,59013,59014,59015,59016,59017,59018,59019,59020,59021,59022,59023,59024,59025,59026,59027,59028,59029,59030,59031,59032,59033,59034,59035,59036,59037,59038,59039,59040,59041,59042,59043,59044,59045,59046,59047,59048,59049,59050,59051,59052,59053,59054,59055,59056,59057,59058,59059,59060,59061,59062,59063,59064,59065,59066,59067,59068,59069,59070,59071,59072,59073,59074,59075,59076,59077,59078,59079,59080,59081,59082,59083,59084,59085,59086,59087,59088,59089,59090,59091,59092,59093,59094,59095,59096,59097,59098,59099,59100,59101,59102,59103,59104,59105,59106,59107,59108,59109,59110,59111,59112,59113,59114,59115,59116,59117,59118,59119,59120,59121,59122,59123,59124,59125,59126,59127,59128,59129,59130,59131,59132,59133,59134,59135,59136,59137,59138,59139,59140,59141,59142,59143,59144,59145,59146,59147,59148,59149,59150,59151,59152,59153,59154,59155,59156,59157,59158,59159,59160,59161,59162,59163,59164,59165,59166,59167,59168,59169,59170,59171,59172,59173,59174,59175,59176,59177,59178,59179,59180,59181,59182,59183,59184,59185,59186,59187,59188,59189,59190,59191,59192,59193,59194,59195,59196,59197,59198,59199,59200,59201,59202,59203,59204,59205,59206,59207,59208,59209,59210,59211,59212,59213,59214,59215,59216,59217,59218,59219,59220,59221,59222,59223,59224,59225,59226,59227,59228,59229,59230,59231,59232,59233,59234,59235,59236,59237,59238,59239,59240,59241,59242,59243,59244,59245,59246,59247,59248,59249,59250,59251,59252,59253,59254,59255,59256,59257,59258,59259,59260,59261,59262,59263,59264,59265,59266,59267,59268,59269,59270,59271,59272,59273,59274,59275,59276,59277,59278,59279,59280,59281,59282,59283,59284,59285,59286,59287,59288,59289,59290,59291,59292,59293,59294,59295,59296,59297,59298,59299,59300,59301,59302,59303,59304,59305,59306,59307,59308,59309,59310,59311,59312,59313,59314,59315,59316,59317,59318,59319,59320,59321,59322,59323,59324,59325,59326,59327,59328,59329,59330,59331,59332,59333,59334,59335,59336,59337,59338,59339,59340,59341,59342,59343,59344,59345,59346,59347,59348,59349,59350,59351,59352,59353,59354,59355,59356,59357,59358,59359,59360,59361,59362,59363,59364,59365,59366,59367,59368,59369,59370,59371,59372,59373,59374,59375,59376,59377,59378,59379,59380,59381,59382,59383,59384,59385,59386,59387,59388,59389,59390,59391,59392,59393,59394,59395,59396,59397,59398,59399,59400,59401,59402,59403,59404,59405,59406,59407,59408,59409,59410,59411,59412,59413,59414,59415,59416,59417,59418,59419,59420,59421,59422,59423,59424,59425,59426,59427,59428,59429,59430,59431,59432,59433,59434,59435,59436,59437,59438,59439,59440,59441,59442,59443,59444,59445,59446,59447,59448,59449,59450,59451,59452,59453,59454,59455,59456,59457,59458,59459,59460,59461,59462,59463,59464,59465,59466,59467,59468,59469,59470,59471,59472,59473,59474,59475,59476,59477,59478,59479,59480,59481,59482,59483,59484,59485,59486,59487,59488,59489,59490,59491,59492,59493,59494,59495,59496,59497,59498,59499,59500,59501,59502,59503,59504,59505,59506,59507,59508,59509,59510,59511,59512,59513,59514,59515,59516,59517,59518,59519,59520,59521,59522,59523,59524,59525,59526,59527,59528,59529,59530,59531,59532,59533,59534,59535,59536,59537,59538,59539,59540,59541,59542,59543,59544,59545,59546,59547,59548,59549,59550,59551,59552,59553,59554,59555,59556,59557,59558,59559,59560,59561,59562,59563,59564,59565,59566,59567,59568,59569,59570,59571,59572,59573,59574,59575,59576,59577,59578,59579,59580,59581,59582,59583,59584,59585,59586,59587,59588,59589,59590,59591,59592,59593,59594,59595,59596,59597,59598,59599,59600,59601,59602,59603,59604,59605,59606,59607,59608,59609,59610,59611,59612,59613,59614,59615,59616,59617,59618,59619,59620,59621,59622,59623,59624,59625,59626,59627,59628,59629,59630,59631,59632,59633,59634,59635,59636,59637,59638,59639,59640,59641,59642,59643,59644,59645,59646,59647,59648,59649,59650,59651,59652,59653,59654,59655,59656,59657,59658,59659,59660,59661,59662,59663,59664,59665,59666,59667,59668,59669,59670,59671,59672,59673,59674,59675,59676,59677,59678,59679,59680,59681,59682,59683,59684,59685,59686,59687,59688,59689,59690,59691,59692,59693,59694,59695,59696,59697,59698,59699,59700,59701,59702,59703,59704,59705,59706,59707,59708,59709,59710,59711,59712,59713,59714,59715,59716,59717,59718,59719,59720,59721,59722,59723,59724,59725,59726,59727,59728,59729,59730,59731,59732,59733,59734,59735,59736,59737,59738,59739,59740,59741,59742,59743,59744,59745,59746,59747,59748,59749,59750,59751,59752,59753,59754,59755,59756,59757,59758,59759,59760,59761,59762,59763,59764,59765,59766,59767,59768,59769,59770,59771,59772,59773,59774,59775,59776,59777,59778,59779,59780,59781,59782,59783,59784,59785,59786,59787,59788,59789,59790,59791,59792,59793,59794,59795,59796,59797,59798,59799,59800,59801,59802,59803,59804,59805,59806,59807,59808,59809,59810,59811,59812,59813,59814,59815,59816,59817,59818,59819,59820,59821,59822,59823,59824,59825,59826,59827,59828,59829,59830,59831,59832,59833,59834,59835,59836,59837,59838,59839,59840,59841,59842,59843,59844,59845,59846,59847,59848,59849,59850,59851,59852,59853,59854,59855,59856,59857,59858,59859,59860,59861,59862,59863,59864,59865,59866,59867,59868,59869,59870,59871,59872,59873,59874,59875,59876,59877,59878,59879,59880,59881,59882,59883,59884,59885,59886,59887,59888,59889,59890,59891,59892,59893,59894,59895,59896,59897,59898,59899,59900,59901,59902,59903,59904,59905,59906,59907,59908,59909,59910,59911,59912,59913,59914,59915,59916,59917,59918,59919,59920,59921,59922,59923,59924,59925,59926,59927,59928,59929,59930,59931,59932,59933,59934,59935,59936,59937,59938,59939,59940,59941,59942,59943,59944,59945,59946,59947,59948,59949,59950,59951,59952,59953,59954,59955,59956,59957,59958,59959,59960,59961,59962,59963,59964,59965,59966,59967,59968,59969,59970,59971,59972,59973,59974,59975,59976,59977,59978,59979,59980,59981,59982,59983,59984,59985,59986,59987,59988,59989,59990,59991,59992,59993,59994,59995,59996,59997,59998,59999,60000,60001,60002,60003,60004,60005,60006,60007,60008,60009,60010,60011,60012,60013,60014,60015,60016,60017,60018,60019,60020,60021,60022,60023,60024,60025,60026,60027,60028,60029,60030,60031,60032,60033,60034,60035,60036,60037,60038,60039,60040,60041,60042,60043,60044,60045,60046,60047,60048,60049,60050,60051,60052,60053,60054,60055,60056,60057,60058,60059,60060,60061,60062,60063,60064,60065,60066,60067,60068,60069,60070,60071,60072,60073,60074,60075,60076,60077,60078,60079,60080,60081,60082,60083,60084,60085,60086,60087,60088,60089,60090,60091,60092,60093,60094,60095,60096,60097,60098,60099,60100,60101,60102,60103,60104,60105,60106,60107,60108,60109,60110,60111,60112,60113,60114,60115,60116,60117,60118,60119,60120,60121,60122,60123,60124,60125,60126,60127,60128,60129,60130,60131,60132,60133,60134,60135,60136,60137,60138,60139,60140,60141,60142,60143,60144,60145,60146,60147,60148,60149,60150,60151,60152,60153,60154,60155,60156,60157,60158,60159,60160,60161,60162,60163,60164,60165,60166,60167,60168,60169,60170,60171,60172,60173,60174,60175,60176,60177,60178,60179,60180,60181,60182,60183,60184,60185,60186,60187,60188,60189,60190,60191,60192,60193,60194,60195,60196,60197,60198,60199,60200,60201,60202,60203,60204,60205,60206,60207,60208,60209,60210,60211,60212,60213,60214,60215,60216,60217,60218,60219,60220,60221,60222,60223,60224,60225,60226,60227,60228,60229,60230,60231,60232,60233,60234,60235,60236,60237,60238,60239,60240,60241,60242,60243,60244,60245,60246,60247,60248,60249,60250,60251,60252,60253,60254,60255,60256,60257,60258,60259,60260,60261,60262,60263,60264,60265,60266,60267,60268,60269,60270,60271,60272,60273,60274,60275,60276,60277,60278,60279,60280,60281,60282,60283,60284,60285,60286,60287,60288,60289,60290,60291,60292,60293,60294,60295,60296,60297,60298,60299,60300,60301,60302,60303,60304,60305,60306,60307,60308,60309,60310,60311,60312,60313,60314,60315,60316,60317,60318,60319,60320,60321,60322,60323,60324,60325,60326,60327,60328,60329,60330,60331,60332,60333,60334,60335,60336,60337,60338,60339,60340,60341,60342,60343,60344,60345,60346,60347,60348,60349,60350,60351,60352,60353,60354,60355,60356,60357,60358,60359,60360,60361,60362,60363,60364,60365,60366,60367,60368,60369,60370,60371,60372,60373,60374,60375,60376,60377,60378,60379,60380,60381,60382,60383,60384,60385,60386,60387,60388,60389,60390,60391,60392,60393,60394,60395,60396,60397,60398,60399,60400,60401,60402,60403,60404,60405,60406,60407,60408,60409,60410,60411,60412,60413,60414,60415,60416,60417,60418,60419,60420,60421,60422,60423,60424,60425,60426,60427,60428,60429,60430,60431,60432,60433,60434,60435,60436,60437,60438,60439,60440,60441,60442,60443,60444,60445,60446,60447,60448,60449,60450,60451,60452,60453,60454,60455,60456,60457,60458,60459,60460,60461,60462,60463,60464,60465,60466,60467,60468,60469,60470,60471,60472,60473,60474,60475,60476,60477,60478,60479,60480,60481,60482,60483,60484,60485,60486,60487,60488,60489,60490,60491,60492,60493,60494,60495,60496,60497,60498,60499,60500,60501,60502,60503,60504,60505,60506,60507,60508,60509,60510,60511,60512,60513,60514,60515,60516,60517,60518,60519,60520,60521,60522,60523,60524,60525,60526,60527,60528,60529,60530,60531,60532,60533,60534,60535,60536,60537,60538,60539,60540,60541,60542,60543,60544,60545,60546,60547,60548,60549,60550,60551,60552,60553,60554,60555,60556,60557,60558,60559,60560,60561,60562,60563,60564,60565,60566,60567,60568,60569,60570,60571,60572,60573,60574,60575,60576,60577,60578,60579,60580,60581,60582,60583,60584,60585,60586,60587,60588,60589,60590,60591,60592,60593,60594,60595,60596,60597,60598,60599,60600,60601,60602,60603,60604,60605,60606,60607,60608,60609,60610,60611,60612,60613,60614,60615,60616,60617,60618,60619,60620,60621,60622,60623,60624,60625,60626,60627,60628,60629,60630,60631,60632,60633,60634,60635,60636,60637,60638,60639,60640,60641,60642,60643,60644,60645,60646,60647,60648,60649,60650,60651,60652,60653,60654,60655,60656,60657,60658,60659,60660,60661,60662,60663,60664,60665,60666,60667,60668,60669,60670,60671,60672,60673,60674,60675,60676,60677,60678,60679,60680,60681,60682,60683,60684,60685,60686,60687,60688,60689,60690,60691,60692,60693,60694,60695,60696,60697,60698,60699,60700,60701,60702,60703,60704,60705,60706,60707,60708,60709,60710,60711,60712,60713,60714,60715,60716,60717,60718,60719,60720,60721,60722,60723,60724,60725,60726,60727,60728,60729,60730,60731,60732,60733,60734,60735,60736,60737,60738,60739,60740,60741,60742,60743,60744,60745,60746,60747,60748,60749,60750,60751,60752,60753,60754,60755,60756,60757,60758,60759,60760,60761,60762,60763,60764,60765,60766,60767,60768,60769,60770,60771,60772,60773,60774,60775,60776,60777,60778,60779,60780,60781,60782,60783,60784,60785,60786,60787,60788,60789,60790,60791,60792,60793,60794,60795,60796,60797,60798,60799,60800,60801,60802,60803,60804,60805,60806,60807,60808,60809,60810,60811,60812,60813,60814,60815,60816,60817,60818,60819,60820,60821,60822,60823,60824,60825,60826,60827,60828,60829,60830,60831,60832,60833,60834,60835,60836,60837,60838,60839,60840,60841,60842,60843,60844,60845,60846,60847,60848,60849,60850,60851,60852,60853,60854,60855,60856,60857,60858,60859,60860,60861,60862,60863,60864,60865,60866,60867,60868,60869,60870,60871,60872,60873,60874,60875,60876,60877,60878,60879,60880,60881,60882,60883,60884,60885,60886,60887,60888,60889,60890,60891,60892,60893,60894,60895,60896,60897,60898,60899,60900,60901,60902,60903,60904,60905,60906,60907,60908,60909,60910,60911,60912,60913,60914,60915,60916,60917,60918,60919,60920,60921,60922,60923,60924,60925,60926,60927,60928,60929,60930,60931,60932,60933,60934,60935,60936,60937,60938,60939,60940,60941,60942,60943,60944,60945,60946,60947,60948,60949,60950,60951,60952,60953,60954,60955,60956,60957,60958,60959,60960,60961,60962,60963,60964,60965,60966,60967,60968,60969,60970,60971,60972,60973,60974,60975,60976,60977,60978,60979,60980,60981,60982,60983,60984,60985,60986,60987,60988,60989,60990,60991,60992,60993,60994,60995,60996,60997,60998,60999,61000,61001,61002,61003,61004,61005,61006,61007,61008,61009,61010,61011,61012,61013,61014,61015,61016,61017,61018,61019,61020,61021,61022,61023,61024,61025,61026,61027,61028,61029,61030,61031,61032,61033,61034,61035,61036,61037,61038,61039,61040,61041,61042,61043,61044,61045,61046,61047,61048,61049,61050,61051,61052,61053,61054,61055,61056,61057,61058,61059,61060,61061,61062,61063,61064,61065,61066,61067,61068,61069,61070,61071,61072,61073,61074,61075,61076,61077,61078,61079,61080,61081,61082,61083,61084,61085,61086,61087,61088,61089,61090,61091,61092,61093,61094,61095,61096,61097,61098,61099,61100,61101,61102,61103,61104,61105,61106,61107,61108,61109,61110,61111,61112,61113,61114,61115,61116,61117,61118,61119,61120,61121,61122,61123,61124,61125,61126,61127,61128,61129,61130,61131,61132,61133,61134,61135,61136,61137,61138,61139,61140,61141,61142,61143,61144,61145,61146,61147,61148,61149,61150,61151,61152,61153,61154,61155,61156,61157,61158,61159,61160,61161,61162,61163,61164,61165,61166,61167,61168,61169,61170,61171,61172,61173,61174,61175,61176,61177,61178,61179,61180,61181,61182,61183,61184,61185,61186,61187,61188,61189,61190,61191,61192,61193,61194,61195,61196,61197,61198,61199,61200,61201,61202,61203,61204,61205,61206,61207,61208,61209,61210,61211,61212,61213,61214,61215,61216,61217,61218,61219,61220,61221,61222,61223,61224,61225,61226,61227,61228,61229,61230,61231,61232,61233,61234,61235,61236,61237,61238,61239,61240,61241,61242,61243,61244,61245,61246,61247,61248,61249,61250,61251,61252,61253,61254,61255,61256,61257,61258,61259,61260,61261,61262,61263,61264,61265,61266,61267,61268,61269,61270,61271,61272,61273,61274,61275,61276,61277,61278,61279,61280,61281,61282,61283,61284,61285,61286,61287,61288,61289,61290,61291,61292,61293,61294,61295,61296,61297,61298,61299,61300,61301,61302,61303,61304,61305,61306,61307,61308,61309,61310,61311,61312,61313,61314,61315,61316,61317,61318,61319,61320,61321,61322,61323,61324,61325,61326,61327,61328,61329,61330,61331,61332,61333,61334,61335,61336,61337,61338,61339,61340,61341,61342,61343,61344,61345,61346,61347,61348,61349,61350,61351,61352,61353,61354,61355,61356,61357,61358,61359,61360,61361,61362,61363,61364,61365,61366,61367,61368,61369,61370,61371,61372,61373,61374,61375,61376,61377,61378,61379,61380,61381,61382,61383,61384,61385,61386,61387,61388,61389,61390,61391,61392,61393,61394,61395,61396,61397,61398,61399,61400,61401,61402,61403,61404,61405,61406,61407,61408,61409,61410,61411,61412,61413,61414,61415,61416,61417,61418,61419,61420,61421,61422,61423,61424,61425,61426,61427,61428,61429,61430,61431,61432,61433,61434,61435,61436,61437,61438,61439,61440,61441,61442,61443,61444,61445,61446,61447,61448,61449,61450,61451,61452,61453,61454,61455,61456,61457,61458,61459,61460,61461,61462,61463,61464,61465,61466,61467,61468,61469,61470,61471,61472,61473,61474,61475,61476,61477,61478,61479,61480,61481,61482,61483,61484,61485,61486,61487,61488,61489,61490,61491,61492,61493,61494,61495,61496,61497,61498,61499,61500,61501,61502,61503,61504,61505,61506,61507,61508,61509,61510,61511,61512,61513,61514,61515,61516,61517,61518,61519,61520,61521,61522,61523,61524,61525,61526,61527,61528,61529,61530,61531,61532,61533,61534,61535,61536,61537,61538,61539,61540,61541,61542,61543,61544,61545,61546,61547,61548,61549,61550,61551,61552,61553,61554,61555,61556,61557,61558,61559,61560,61561,61562,61563,61564,61565,61566,61567,61568,61569,61570,61571,61572,61573,61574,61575,61576,61577,61578,61579,61580,61581,61582,61583,61584,61585,61586,61587,61588,61589,61590,61591,61592,61593,61594,61595,61596,61597,61598,61599,61600,61601,61602,61603,61604,61605,61606,61607,61608,61609,61610,61611,61612,61613,61614,61615,61616,61617,61618,61619,61620,61621,61622,61623,61624,61625,61626,61627,61628,61629,61630,61631,61632,61633,61634,61635,61636,61637,61638,61639,61640,61641,61642,61643,61644,61645,61646,61647,61648,61649,61650,61651,61652,61653,61654,61655,61656,61657,61658,61659,61660,61661,61662,61663,61664,61665,61666,61667,61668,61669,61670,61671,61672,61673,61674,61675,61676,61677,61678,61679,61680,61681,61682,61683,61684,61685,61686,61687,61688,61689,61690,61691,61692,61693,61694,61695,61696,61697,61698,61699,61700,61701,61702,61703,61704,61705,61706,61707,61708,61709,61710,61711,61712,61713,61714,61715,61716,61717,61718,61719,61720,61721,61722,61723,61724,61725,61726,61727,61728,61729,61730,61731,61732,61733,61734,61735,61736,61737,61738,61739,61740,61741,61742,61743,61744,61745,61746,61747,61748,61749,61750,61751,61752,61753,61754,61755,61756,61757,61758,61759,61760,61761,61762,61763,61764,61765,61766,61767,61768,61769,61770,61771,61772,61773,61774,61775,61776,61777,61778,61779,61780,61781,61782,61783,61784,61785,61786,61787,61788,61789,61790,61791,61792,61793,61794,61795,61796,61797,61798,61799,61800,61801,61802,61803,61804,61805,61806,61807,61808,61809,61810,61811,61812,61813,61814,61815,61816,61817,61818,61819,61820,61821,61822,61823,61824,61825,61826,61827,61828,61829,61830,61831,61832,61833,61834,61835,61836,61837,61838,61839,61840,61841,61842,61843,61844,61845,61846,61847,61848,61849,61850,61851,61852,61853,61854,61855,61856,61857,61858,61859,61860,61861,61862,61863,61864,61865,61866,61867,61868,61869,61870,61871,61872,61873,61874,61875,61876,61877,61878,61879,61880,61881,61882,61883,61884,61885,61886,61887,61888,61889,61890,61891,61892,61893,61894,61895,61896,61897,61898,61899,61900,61901,61902,61903,61904,61905,61906,61907,61908,61909,61910,61911,61912,61913,61914,61915,61916,61917,61918,61919,61920,61921,61922,61923,61924,61925,61926,61927,61928,61929,61930,61931,61932,61933,61934,61935,61936,61937,61938,61939,61940,61941,61942,61943,61944,61945,61946,61947,61948,61949,61950,61951,61952,61953,61954,61955,61956,61957,61958,61959,61960,61961,61962,61963,61964,61965,61966,61967,61968,61969,61970,61971,61972,61973,61974,61975,61976,61977,61978,61979,61980,61981,61982,61983,61984,61985,61986,61987,61988,61989,61990,61991,61992,61993,61994,61995,61996,61997,61998,61999,62000,62001,62002,62003,62004,62005,62006,62007,62008,62009,62010,62011,62012,62013,62014,62015,62016,62017,62018,62019,62020,62021,62022,62023,62024,62025,62026,62027,62028,62029,62030,62031,62032,62033,62034,62035,62036,62037,62038,62039,62040,62041,62042,62043,62044,62045,62046,62047,62048,62049,62050,62051,62052,62053,62054,62055,62056,62057,62058,62059,62060,62061,62062,62063,62064,62065,62066,62067,62068,62069,62070,62071,62072,62073,62074,62075,62076,62077,62078,62079,62080,62081,62082,62083,62084,62085,62086,62087,62088,62089,62090,62091,62092,62093,62094,62095,62096,62097,62098,62099,62100,62101,62102,62103,62104,62105,62106,62107,62108,62109,62110,62111,62112,62113,62114,62115,62116,62117,62118,62119,62120,62121,62122,62123,62124,62125,62126,62127,62128,62129,62130,62131,62132,62133,62134,62135,62136,62137,62138,62139,62140,62141,62142,62143,62144,62145,62146,62147,62148,62149,62150,62151,62152,62153,62154,62155,62156,62157,62158,62159,62160,62161,62162,62163,62164,62165,62166,62167,62168,62169,62170,62171,62172,62173,62174,62175,62176,62177,62178,62179,62180,62181,62182,62183,62184,62185,62186,62187,62188,62189,62190,62191,62192,62193,62194,62195,62196,62197,62198,62199,62200,62201,62202,62203,62204,62205,62206,62207,62208,62209,62210,62211,62212,62213,62214,62215,62216,62217,62218,62219,62220,62221,62222,62223,62224,62225,62226,62227,62228,62229,62230,62231,62232,62233,62234,62235,62236,62237,62238,62239,62240,62241,62242,62243,62244,62245,62246,62247,62248,62249,62250,62251,62252,62253,62254,62255,62256,62257,62258,62259,62260,62261,62262,62263,62264,62265,62266,62267,62268,62269,62270,62271,62272,62273,62274,62275,62276,62277,62278,62279,62280,62281,62282,62283,62284,62285,62286,62287,62288,62289,62290,62291,62292,62293,62294,62295,62296,62297,62298,62299,62300,62301,62302,62303,62304,62305,62306,62307,62308,62309,62310,62311,62312,62313,62314,62315,62316,62317,62318,62319,62320,62321,62322,62323,62324,62325,62326,62327,62328,62329,62330,62331,62332,62333,62334,62335,62336,62337,62338,62339,62340,62341,62342,62343,62344,62345,62346,62347,62348,62349,62350,62351,62352,62353,62354,62355,62356,62357,62358,62359,62360,62361,62362,62363,62364,62365,62366,62367,62368,62369,62370,62371,62372,62373,62374,62375,62376,62377,62378,62379,62380,62381,62382,62383,62384,62385,62386,62387,62388,62389,62390,62391,62392,62393,62394,62395,62396,62397,62398,62399,62400,62401,62402,62403,62404,62405,62406,62407,62408,62409,62410,62411,62412,62413,62414,62415,62416,62417,62418,62419,62420,62421,62422,62423,62424,62425,62426,62427,62428,62429,62430,62431,62432,62433,62434,62435,62436,62437,62438,62439,62440,62441,62442,62443,62444,62445,62446,62447,62448,62449,62450,62451,62452,62453,62454,62455,62456,62457,62458,62459,62460,62461,62462,62463,62464,62465,62466,62467,62468,62469,62470,62471,62472,62473,62474,62475,62476,62477,62478,62479,62480,62481,62482,62483,62484,62485,62486,62487,62488,62489,62490,62491,62492,62493,62494,62495,62496,62497,62498,62499,62500,62501,62502,62503,62504,62505,62506,62507,62508,62509,62510,62511,62512,62513,62514,62515,62516,62517,62518,62519,62520,62521,62522,62523,62524,62525,62526,62527,62528,62529,62530,62531,62532,62533,62534,62535,62536,62537,62538,62539,62540,62541,62542,62543,62544,62545,62546,62547,62548,62549,62550,62551,62552,62553,62554,62555,62556,62557,62558,62559,62560,62561,62562,62563,62564,62565,62566,62567,62568,62569,62570,62571,62572,62573,62574,62575,62576,62577,62578,62579,62580,62581,62582,62583,62584,62585,62586,62587,62588,62589,62590,62591,62592,62593,62594,62595,62596,62597,62598,62599,62600,62601,62602,62603,62604,62605,62606,62607,62608,62609,62610,62611,62612,62613,62614,62615,62616,62617,62618,62619,62620,62621,62622,62623,62624,62625,62626,62627,62628,62629,62630,62631,62632,62633,62634,62635,62636,62637,62638,62639,62640,62641,62642,62643,62644,62645,62646,62647,62648,62649,62650,62651,62652,62653,62654,62655,62656,62657,62658,62659,62660,62661,62662,62663,62664,62665,62666,62667,62668,62669,62670,62671,62672,62673,62674,62675,62676,62677,62678,62679,62680,62681,62682,62683,62684,62685,62686,62687,62688,62689,62690,62691,62692,62693,62694,62695,62696,62697,62698,62699,62700,62701,62702,62703,62704,62705,62706,62707,62708,62709,62710,62711,62712,62713,62714,62715,62716,62717,62718,62719,62720,62721,62722,62723,62724,62725,62726,62727,62728,62729,62730,62731,62732,62733,62734,62735,62736,62737,62738,62739,62740,62741,62742,62743,62744,62745,62746,62747,62748,62749,62750,62751,62752,62753,62754,62755,62756,62757,62758,62759,62760,62761,62762,62763,62764,62765,62766,62767,62768,62769,62770,62771,62772,62773,62774,62775,62776,62777,62778,62779,62780,62781,62782,62783,62784,62785,62786,62787,62788,62789,62790,62791,62792,62793,62794,62795,62796,62797,62798,62799,62800,62801,62802,62803,62804,62805,62806,62807,62808,62809,62810,62811,62812,62813,62814,62815,62816,62817,62818,62819,62820,62821,62822,62823,62824,62825,62826,62827,62828,62829,62830,62831,62832,62833,62834,62835,62836,62837,62838,62839,62840,62841,62842,62843,62844,62845,62846,62847,62848,62849,62850,62851,62852,62853,62854,62855,62856,62857,62858,62859,62860,62861,62862,62863,62864,62865,62866,62867,62868,62869,62870,62871,62872,62873,62874,62875,62876,62877,62878,62879,62880,62881,62882,62883,62884,62885,62886,62887,62888,62889,62890,62891,62892,62893,62894,62895,62896,62897,62898,62899,62900,62901,62902,62903,62904,62905,62906,62907,62908,62909,62910,62911,62912,62913,62914,62915,62916,62917,62918,62919,62920,62921,62922,62923,62924,62925,62926,62927,62928,62929,62930,62931,62932,62933,62934,62935,62936,62937,62938,62939,62940,62941,62942,62943,62944,62945,62946,62947,62948,62949,62950,62951,62952,62953,62954,62955,62956,62957,62958,62959,62960,62961,62962,62963,62964,62965,62966,62967,62968,62969,62970,62971,62972,62973,62974,62975,62976,62977,62978,62979,62980,62981,62982,62983,62984,62985,62986,62987,62988,62989,62990,62991,62992,62993,62994,62995,62996,62997,62998,62999,63000,63001,63002,63003,63004,63005,63006,63007,63008,63009,63010,63011,63012,63013,63014,63015,63016,63017,63018,63019,63020,63021,63022,63023,63024,63025,63026,63027,63028,63029,63030,63031,63032,63033,63034,63035,63036,63037,63038,63039,63040,63041,63042,63043,63044,63045,63046,63047,63048,63049,63050,63051,63052,63053,63054,63055,63056,63057,63058,63059,63060,63061,63062,63063,63064,63065,63066,63067,63068,63069,63070,63071,63072,63073,63074,63075,63076,63077,63078,63079,63080,63081,63082,63083,63084,63085,63086,63087,63088,63089,63090,63091,63092,63093,63094,63095,63096,63097,63098,63099,63100,63101,63102,63103,63104,63105,63106,63107,63108,63109,63110,63111,63112,63113,63114,63115,63116,63117,63118,63119,63120,63121,63122,63123,63124,63125,63126,63127,63128,63129,63130,63131,63132,63133,63134,63135,63136,63137,63138,63139,63140,63141,63142,63143,63144,63145,63146,63147,63148,63149,63150,63151,63152,63153,63154,63155,63156,63157,63158,63159,63160,63161,63162,63163,63164,63165,63166,63167,63168,63169,63170,63171,63172,63173,63174,63175,63176,63177,63178,63179,63180,63181,63182,63183,63184,63185,63186,63187,63188,63189,63190,63191,63192,63193,63194,63195,63196,63197,63198,63199,63200,63201,63202,63203,63204,63205,63206,63207,63208,63209,63210,63211,63212,63213,63214,63215,63216,63217,63218,63219,63220,63221,63222,63223,63224,63225,63226,63227,63228,63229,63230,63231,63232,63233,63234,63235,63236,63237,63238,63239,63240,63241,63242,63243,63244,63245,63246,63247,63248,63249,63250,63251,63252,63253,63254,63255,63256,63257,63258,63259,63260,63261,63262,63263,63264,63265,63266,63267,63268,63269,63270,63271,63272,63273,63274,63275,63276,63277,63278,63279,63280,63281,63282,63283,63284,63285,63286,63287,63288,63289,63290,63291,63292,63293,63294,63295,63296,63297,63298,63299,63300,63301,63302,63303,63304,63305,63306,63307,63308,63309,63310,63311,63312,63313,63314,63315,63316,63317,63318,63319,63320,63321,63322,63323,63324,63325,63326,63327,63328,63329,63330,63331,63332,63333,63334,63335,63336,63337,63338,63339,63340,63341,63342,63343,63344,63345,63346,63347,63348,63349,63350,63351,63352,63353,63354,63355,63356,63357,63358,63359,63360,63361,63362,63363,63364,63365,63366,63367,63368,63369,63370,63371,63372,63373,63374,63375,63376,63377,63378,63379,63380,63381,63382,63383,63384,63385,63386,63387,63388,63389,63390,63391,63392,63393,63394,63395,63396,63397,63398,63399,63400,63401,63402,63403,63404,63405,63406,63407,63408,63409,63410,63411,63412,63413,63414,63415,63416,63417,63418,63419,63420,63421,63422,63423,63424,63425,63426,63427,63428,63429,63430,63431,63432,63433,63434,63435,63436,63437,63438,63439,63440,63441,63442,63443,63444,63445,63446,63447,63448,63449,63450,63451,63452,63453,63454,63455,63456,63457,63458,63459,63460,63461,63462,63463,63464,63465,63466,63467,63468,63469,63470,63471,63472,63473,63474,63475,63476,63477,63478,63479,63480,63481,63482,63483,63484,63485,63486,63487,63488,63489,63490,63491,63492,63493,63494,63495,63496,63497,63498,63499,63500,63501,63502,63503,63504,63505,63506,63507,63508,63509,63510,63511,63512,63513,63514,63515,63516,63517,63518,63519,63520,63521,63522,63523,63524,63525,63526,63527,63528,63529,63530,63531,63532,63533,63534,63535,63536,63537,63538,63539,63540,63541,63542,63543,63544,63545,63546,63547,63548,63549,63550,63551,63552,63553,63554,63555,63556,63557,63558,63559,63560,63561,63562,63563,63564,63565,63566,63567,63568,63569,63570,63571,63572,63573,63574,63575,63576,63577,63578,63579,63580,63581,63582,63583,63584,63585,63586,63587,63588,63589,63590,63591,63736,63744,63745,63746,63747,63748,63749,63750,63751,63752,63753,63754,63755,63756,63757,63758,63759,63760,63761,63762,63763,63764,63765,63766,63767,63768,63769,63770,63771,63772,63773,63774,63775,63776,63777,63778,63779,63780,63781,63782,63783,63784,63785,63786,63787,63788,63789,63790,63791,63792,63793,63794,63795,63796,63797,63798,63799,63800,63801,63802,63803,63804,63805,63806,63807,63808,63809,63810,63811,63812,63813,63814,63815,63816,63817,63818,63819,63820,63821,63822,63823,63824,63825,63826,63827,63828,63829,63830,63831,63832,63833,63834,63835,63836,63837,63838,63839,63840,63841,63842,63843,63844,63845,63846,63847,63848,63849,63850,63851,63852,63853,63854,63855,63856,63857,63858,63859,63860,63861,63862,63863,63864,63865,63866,63867,63868,63869,63870,63871,63872,63873,63874,63875,63876,63877,63878,63879,63880,63881,63882,63883,63884,63885,63886,63887,63888,63889,63890,63891,63892,63893,63894,63895,63896,63897,63898,63899,63900,63901,63902,63903,63904,63905,63906,63907,63908,63909,63910,63911,63912,63913,63914,63915,63916,63917,63918,63919,63920,63921,63922,63923,63924,63925,63926,63927,63928,63929,63930,63931,63932,63933,63934,63935,63936,63937,63938,63939,63940,63941,63942,63943,63944,63945,63946,63947,63948,63949,63950,63951,63952,63953,63954,63955,63956,63957,63958,63959,63960,63961,63962,63963,63964,63965,63966,63967,63968,63969,63970,63971,63972,63973,63974,63975,63976,63977,63978,63979,63980,63981,63982,63983,63984,63985,63986,63987,63988,63989,63990,63991,63992,63993,63994,63995,63996,63997,63998,63999,64000,64001,64002,64003,64004,64005,64006,64007,64008,64009,64010,64011,64012,64013,64014,64015,64016,64017,64018,64019,64020,64021,64022,64023,64024,64025,64026,64027,64028,64029,64030,64031,64032,64033,64034,64035,64036,64037,64038,64039,64040,64041,64042,64043,64044,64045,64048,64049,64050,64051,64052,64053,64054,64055,64056,64057,64058,64059,64060,64061,64062,64063,64064,64065,64066,64067,64068,64069,64070,64071,64072,64073,64074,64075,64076,64077,64078,64079,64080,64081,64082,64083,64084,64085,64086,64087,64088,64089,64090,64091,64092,64093,64094,64095,64096,64097,64098,64099,64100,64101,64102,64103,64104,64105,64106,65072,65073,65075,65076,65077,65078,65079,65080,65081,65082,65083,65084,65085,65086,65087,65088,65089,65090,65091,65092,65097,65098,65099,65100,65101,65102,65103,65104,65105,65106,65108,65109,65110,65111,65113,65114,65115,65116,65117,65118,65119,65120,65121,65122,65123,65124,65125,65126,65128,65129,65130,65131,65281,65282,65283,65284,65285,65286,65287,65288,65289,65290,65291,65292,65293,65294,65295,65296,65297,65298,65299,65300,65301,65302,65303,65304,65305,65306,65307,65308,65309,65310,65311,65312,65313,65314,65315,65316,65317,65318,65319,65320,65321,65322,65323,65324,65325,65326,65327,65328,65329,65330,65331,65332,65333,65334,65335,65336,65337,65338,65339,65340,65341,65342,65343,65344,65345,65346,65347,65348,65349,65350,65351,65352,65353,65354,65355,65356,65357,65358,65359,65360,65361,65362,65363,65364,65365,65366,65367,65368,65369,65370,65371,65372,65373,65374,65377,65378,65379,65380,65381,65382,65383,65384,65385,65386,65387,65388,65389,65390,65391,65392,65393,65394,65395,65396,65397,65398,65399,65400,65401,65402,65403,65404,65405,65406,65407,65408,65409,65410,65411,65412,65413,65414,65415,65416,65417,65418,65419,65420,65421,65422,65423,65424,65425,65426,65427,65428,65429,65430,65431,65432,65433,65434,65435,65436,65437,65438,65439,65504,65505,65506,65507,65508,65509,65512,65513,65514,65515,65516,65517,65518 };
unsigned short unicode_to_big5_vals[] = { 65533,41183,41542,41543,41212,41540,31776,41393,50904,25376,41182,11552,38907,41410,41560,41427,12832,13088,41388,41831,37368,41296,41211,12576,41198,65203,65200,65204,41199,41127,41132,41206,41150,41142,41174,41147,41148,41128,41175,41207,41143,41129,41133,41208,41144,41178,41149,41130,41134,41209,41151,41145,41425,41177,41131,41135,41210,41146,22816,41180,41181,41184,41152,41200,41166,41158,41190,41163,41164,41185,41153,41201,41159,41186,41154,41202,41160,41194,41165,41187,41155,41203,41167,41161,41426,41193,41188,41156,41204,41162,31008,41196,41197,41136,41137,41169,41138,39893,41139,39909,41140,41214,41168,41170,41171,41172,41141,41157,41173,41189,39883,39910,39884,39887,39888,39880,39889,39886,39890,39879,39904,39895,39905,39897,39906,39885,39882,39898,39900,39899,50905,41918,41916,41917,41919,41413,39913,41915,41796,41797,41798,41799,41800,41801,41802,41803,41804,41805,41806,41807,41808,41809,41810,41811,41812,41813,41814,41815,41816,41817,41818,41819,41820,41821,41822,41823,41824,41825,41826,41827,41828,41829,41830,41831,41832,41833,41834,41835,41836,41837,41838,41839,41840,41841,41842,41843,51193,51187,51188,51189,51190,51191,51192,51194,51195,51196,51197,51198,51264,51265,51266,51267,51268,51269,51270,51271,51272,51273,51274,51275,51276,51277,51278,51279,51280,51281,51282,51283,51284,51285,51286,51287,51288,51289,51290,51292,51293,51294,51295,51296,51297,51298,51299,51300,51301,51302,51303,51304,51305,51306,51307,51308,51309,51310,51311,51312,51313,51314,51315,51316,51317,51291,39894,39891,39892,41179,41195,40940,41302,41304,41591,41468,41381,41382,41383,41384,41423,37342,41285,41915,41292,41291,41285,65250,41388,41394,41387,41392,37367,41411,41953,41546,41409,41547,40399,40418,40393,38908,38482,40392,65201,65202,65205,65206,65207,65208,65209,65210,65217,65218,65219,65220,41657,41658,41659,41660,41661,41662,41663,41664,41665,41666,50869,50870,50871,50872,50873,50874,50875,50876,50877,50878,41462,41460,41463,41461,37371,37366,41464,41465,41467,41466,37370,40445,51319,51320,38483,39269,40266,51318,40265,40268,37369,41081,39161,41406,37227,37215,41811,41813,41537,41538,41560,41428,41435,41448,41447,41469,41468,41806,41700,41444,41445,41452,40780,41453,41455,41454,41287,40774,41436,41434,41437,41437,41432,41433,39648,39590,40050,40106,41458,41459,41446,41449,37372,41307,65241,41920,41921,41922,41923,41924,41925,41926,41927,41928,41929,41930,41931,41932,41933,41934,41935,41936,41937,41938,41939,41940,41941,41942,41943,41944,41945,41946,41947,41948,41949,41950,41951,41952,50849,50850,50851,50852,50853,50854,50855,50856,50857,50858,38731,38732,38733,38734,38735,38736,38737,38738,38739,38740,50859,50860,50861,50862,50863,50864,50865,50866,50867,50868,39243,39244,39245,39246,39247,39248,39249,39250,39251,39252,39489,39490,39491,39492,39493,39494,39495,39496,39497,39498,39499,39500,39501,39502,39503,39504,39505,39506,39507,39508,39393,39394,39395,39396,39397,39398,39399,39400,39401,39402,39403,39404,39405,39406,39407,39408,39409,39410,39411,39412,39413,39414,39415,39416,39417,39418,38855,38856,38857,38858,38859,38860,38861,38862,38863,38864,38865,38866,38867,38868,38869,38870,38871,38872,38873,38874,38875,38876,38877,38878,38879,38880,38881,38882,38883,38884,38885,38886,38887,38888,38889,38890,38891,38892,38893,38894,38895,38896,38897,38898,38899,38900,38901,38902,38903,38904,38905,38906,38720,41591,40317,41592,40318,41292,41292,41290,41290,41291,41291,41301,41301,41594,41594,41594,41594,41595,41595,41595,41595,41596,41596,41596,41596,41597,41597,41597,41597,41589,41589,41589,41589,41589,41589,41589,41589,41588,41588,41588,41588,41588,41588,41588,41588,41587,41587,41587,41587,41587,41587,41587,41587,41586,41586,41586,41586,41586,41586,41586,41586,41585,41585,41585,41585,41585,41585,41585,41585,41585,41585,41585,41585,41585,41585,41585,41585,37360,63993,63992,63974,63983,63965,63976,63985,63967,63980,63989,63971,63982,63991,63973,63977,63986,63968,63979,63988,63970,63975,63984,63966,63981,63990,63972,63978,63987,63969,41598,41633,41635,41634,41644,41645,41646,41306,41570,41571,41572,41573,41574,41575,41576,41577,41584,41583,41582,41581,41580,41579,41578,63998,41590,41593,41405,41404,41398,41397,37364,41407,41406,37365,41403,41402,41395,41399,41396,37361,37362,41640,41641,41643,41642,41395,40378,40380,41401,41459,40256,40429,40145,40271,40270,40269,40272,40405,40789,40790,40797,40795,40792,40793,40794,40796,40791,40286,40285,39164,40376,40279,41456,41458,41457,40408,40413,40411,40410,40407,40414,40412,40409,40294,40293,41706,41389,40300,40384,40299,40277,40276,40284,40283,50918,40311,38977,38978,38979,38980,38981,38982,38983,38984,38985,38986,50849,50850,50851,50852,50853,50854,50855,50856,50857,50858,38977,38978,38979,38980,38981,38982,38983,38984,38985,38986,41400,37357,40142,40149,40148,40155,37339,37321,40183,50879,50880,50881,50882,50883,50884,50885,50886,50887,50888,50889,50890,50891,50892,50893,50894,50895,50896,50897,50898,50899,50900,50901,50902,50903,41280,41282,41283,41394,50912,50913,50914,41329,41330,41325,41326,41333,41334,41337,41338,41321,41322,41541,38493,41317,41318,40273,40274,41443,41385,41386,41667,41668,41669,41670,41671,41672,41673,41674,41675,41418,41676,41677,41678,50919,50920,50921,50922,50923,50924,50925,50926,50927,50928,50929,50930,50931,50932,50933,50934,50935,50936,50937,50938,50939,50940,50941,50942,51008,51009,51010,51011,51012,51013,51014,51015,51016,51017,51018,51019,51020,51021,51022,51023,51024,51025,51026,51027,51028,51029,51030,51031,51032,51033,51034,51035,51036,51037,51038,51039,51040,51041,51042,51043,51044,51045,51046,51047,51048,51049,51050,51051,51052,51053,51054,51055,51056,51057,51058,51059,51060,51061,51062,51063,51064,51065,51066,51379,51377,51378,50908,50909,51067,51068,51069,51070,51105,51106,51107,51108,51109,51110,51111,51112,51113,51114,51115,51116,51117,51118,51119,51120,51121,51122,51123,51124,51125,51126,51127,51128,51129,51130,51131,51132,51133,51134,51135,51136,51137,51138,51139,51140,51141,51142,51143,51144,51145,51146,51147,51148,51149,51150,51151,51152,51153,51154,51155,51156,51157,51158,51159,51160,51161,51162,51163,51164,51165,51166,51167,51168,51169,51170,51171,51172,51173,51174,51175,51176,51177,51178,51179,51180,51181,51182,51183,51184,51185,51186,51380,51381,51382,51383,51384,50915,50906,50907,41844,41845,41846,41847,41848,41849,41850,41851,41852,41853,41854,41889,41890,41891,41892,41893,41894,41895,41896,41897,41898,41899,41900,41901,41902,41903,41904,41905,41906,41907,41908,41909,41910,41911,41912,41913,41914,42048,42055,42068,42364,42071,42148,42069,42450,42049,42238,42050,42193,42593,42056,51321,40001,40002,40003,40004,40005,40006,40007,40008,40009,40010,42219,42229,42228,42220,43767,42087,42217,39421,39420,43712,42583,44883,45149,44972,46034,42318,43337,48839,47818,42488,47338,43251,45757,42480,42715,42716,42048,42055,42068,42364,42157,42171,42051,42059,42053,42065,42219,42229,42228,42220,43767,42087,42217,44776,42675,43712,42583,44883,45149,44972,46034,44981,43115,42091,48705,49269,42572,43616,46773,42480,48231,41408,42071,42148,42069,42410,42347,49893,43382,48839,47818,42488,47338,43251,43357,18535,25942,25697,16725,28502,28771,25709,18773,28737,28225,27969,27457,19266,19778,18242,28742,28230,41557,41558,18554,26221,28269,41552,41553,41554,41556,40423,20577,28787,28275,28019,28758,28246,27990,27478,19798,28759,28247,27991,27479,19799,40419,17009,41559,25444,25666,18297,26721,18512,26990,19275,41555,27508,27757,41451,41450,27768,28002,41551,20552,40420,20562,29554,21366,22370,37344,41122,64324,38328,41032,41036,64479,64485,64367,38210,65149,65127,38238,38366,38343,64609,38345,64636,64680,65137,64706,65101,38573,40147,64862,65098,64871,64432,64381,64995,65009,65007,65010,64325,42048,42050,42051,39079,37335,51525,42070,42068,42071,42069,51526,42147,51535,51533,42146,42145,33789,42306,42305,42304,33088,42307,42238,34258,35517,33627,34118,42464,42465,37861,33089,33621,35826,43203,33994,40137,33706,42072,33090,42148,51536,39080,42149,51555,42730,52145,35418,50879,37324,42073,42150,33647,42308,51556,35821,35419,50880,51520,42052,38130,42075,51527,42076,34291,42151,33727,42309,42311,42310,34788,42466,42467,33998,33091,43204,37609,44476,42049,40159,40168,51521,42053,42078,42077,33908,34160,33092,33724,42468,65089,34151,34223,43205,41057,45230,54347,37882,39081,46787,56497,56498,50881,42054,39082,42153,33628,33093,43206,42055,51528,42079,35423,42154,42156,51537,42157,42155,65122,42469,33618,43207,37440,43208,43845,50882,42080,42158,39083,42470,42472,42471,33892,42731,33730,43209,43210,43846,43847,33094,35165,44477,35426,56499,39084,63190,42056,40180,39085,34723,42160,42159,51538,42161,42167,34226,42162,42163,51540,51539,42165,42166,42164,33863,37826,33620,33708,42314,42315,42316,42317,42313,42320,51562,51558,51561,42321,42337,51560,42318,42319,42312,51557,51559,34724,33095,33704,33096,37460,42485,51632,42482,42486,51642,51630,42483,51634,39086,42484,33097,42487,42473,51633,42488,51637,64442,51641,51638,51635,42474,42476,42489,42478,51627,42481,42479,42480,51643,51640,51631,42477,33895,51628,42475,34257,36349,33098,51636,33988,33772,34225,51639,34228,33705,33707,33987,39087,39088,33771,33603,51629,51814,42818,42740,64445,51815,42737,42820,39089,42745,64444,42744,51803,42748,42743,51808,51816,51812,39090,42746,39091,42749,42734,42823,51805,39092,52157,42732,42819,42733,42741,42742,51810,51806,42747,42739,51802,42735,51813,42821,42824,42738,42816,42822,42736,51811,42817,51817,51804,42750,51807,36416,51809,43224,52159,52171,43216,52172,43211,43221,39093,43214,52153,43222,52152,52156,52163,52161,43230,43225,52147,52149,43227,43215,52150,52162,52169,43220,52155,52148,43219,52151,43223,52154,33099,43218,43213,43228,52164,43229,52168,52166,52170,43226,52158,52146,52160,43217,52165,43212,52167,33637,37461,39094,33638,34475,33773,33774,34476,34727,35018,64446,34726,33405,43862,43850,39095,52704,52712,39159,43849,43857,43869,52718,52716,52711,39096,43851,52717,52707,43865,43856,43864,52702,52714,39097,52705,43860,52706,39098,52701,43867,43854,43863,43853,52703,52708,52715,43861,43858,52710,43866,52713,52709,43855,43868,43859,43852,43848,36281,37983,37984,35016,36040,33703,35948,52719,34728,44503,44481,39099,44497,39100,44502,53456,53455,53460,53461,44484,39101,44493,65147,44506,33100,44494,39102,53449,44487,53450,39103,44508,44499,44478,44479,53469,45247,44492,44491,53451,44495,54363,44486,53462,44501,44500,44490,53454,53463,53448,44489,53464,44498,53452,44480,33449,44483,44482,53465,44496,44485,44505,44507,53459,44504,64449,53467,53453,53468,53457,39104,53466,33443,53458,37554,34224,39105,44488,34227,54371,54359,33101,45235,54364,54370,45234,54357,45238,54361,54354,45236,54358,45241,45246,33102,54375,54353,45242,64447,54374,39106,45237,54360,45233,54355,54351,54365,54352,54350,54362,54368,54369,45239,39107,55387,54366,54349,54367,34229,45249,54372,45248,54348,54356,54373,45244,45243,45240,45245,33103,45231,36350,45232,36453,35272,46024,39108,55390,55383,46021,55391,37104,55381,55384,46020,55385,39109,46023,55389,55379,55378,46025,46026,46022,46027,55377,55388,55386,55380,46019,55382,36041,64450,35017,35273,36859,46794,46788,56503,46797,56509,56512,46790,46791,56506,46789,56515,46795,56516,56511,46796,39110,56500,46793,56501,56510,56508,56504,46792,56502,46798,56507,56514,56505,56513,37967,47542,47539,39111,47540,57593,57585,47538,47535,57586,33104,64451,47537,57589,57591,33105,39112,57598,39113,57597,57592,47534,57584,47532,57587,47543,57590,57594,47536,47533,57596,57595,47541,57588,33106,48120,58604,58601,48121,48119,39114,58608,58605,58598,48118,64454,48122,58599,48117,48125,58602,58603,48123,48124,58609,58606,58607,39115,64452,48810,59640,48807,59637,48809,48811,33107,59638,48808,59639,59636,49270,60605,49271,60603,60604,60602,60601,60606,49269,39116,61368,61369,58600,61367,49272,50015,61931,61932,50391,50392,62913,62912,50540,50539,63440,42057,42081,42169,42168,42323,42322,42492,42491,42493,42490,33108,42826,42825,42827,38396,37585,33109,43232,43231,43233,33639,43870,41561,53470,41562,45250,41564,41563,55392,39117,41565,47544,41566,33110,42058,42170,42494,43234,33111,42059,42173,42171,42172,36003,42560,35799,35157,42828,43236,43235,43237,37052,34997,44509,35750,48812,36948,50883,33112,37452,51534,33624,42324,42325,38372,42561,51818,33113,43872,43871,53472,53471,45251,50884,42174,51541,34752,64321,33662,34401,33114,52173,43873,44512,44510,44511,33115,64334,33116,33117,48813,50885,42326,39118,34221,42562,51644,33612,33611,38139,38357,42829,42830,39119,51819,33709,52174,43238,52175,33888,38387,53474,53475,44515,33118,53476,33883,53473,44516,44514,44513,53477,37046,54376,34030,39120,64463,55393,33119,56517,57664,33120,48126,48814,59641,64464,42060,42074,39121,34721,37601,37964,37985,37970,36451,33121,35031,33989,45252,46029,47545,64455,51522,42175,42329,42327,42328,35291,33122,43239,64466,36269,42061,42062,51325,42082,38351,39122,42176,42177,42178,51646,42330,33123,51563,33746,42566,51647,42564,42565,51645,33124,33125,42567,42563,34730,33640,33710,33991,51820,43756,51821,64456,33126,51822,41086,42832,42831,38267,38373,42835,42833,42834,33127,36282,43245,43244,52180,52177,52178,39123,52176,43246,43242,43241,33128,43243,43240,39124,34732,36527,33990,43247,35020,43875,52720,52179,43880,52721,43876,43879,43878,43877,43874,33129,36296,34731,53480,44519,53483,44517,64458,39125,53479,44520,44518,44521,53481,53482,64457,53478,53484,37555,37577,33398,34729,39126,46033,45253,54377,54379,54378,54380,45254,37611,33130,46030,64757,46031,46032,46800,56519,39127,56518,56520,56521,46801,46799,57665,57666,47547,47546,58202,48192,48193,48194,48196,58610,58611,48195,64361,33131,48815,33132,48816,33133,61933,62915,62914,63441,64459,42063,33134,35823,35188,42332,42331,33775,36528,42568,39128,51648,33776,42837,42838,42836,42839,51823,51824,33135,35274,33641,33992,37877,33136,38306,43249,52181,43248,34231,52722,43884,52723,43883,38212,43881,43882,33993,39129,53485,33137,64460,45255,54382,45258,54381,45541,45257,45256,46036,46035,46034,46802,64461,64462,46805,46806,46804,46803,37480,57667,57668,58613,48197,58612,37540,48817,60607,49273,61934,50261,50886,42083,42179,51542,42180,42181,33138,37982,37971,40144,42333,42334,33139,42569,51825,52182,52183,43885,53486,45260,45259,55395,55394,42064,42182,42335,45261,51523,51564,42336,51650,42571,42570,51649,42840,33140,36297,39130,44522,64468,54383,34477,46807,57669,47548,64467,64469,59642,62461,50887,42183,33777,35659,52184,52724,45264,45262,45263,42065,39131,42084,41677,42186,33141,42185,42184,42339,42338,51565,51651,34127,33622,43253,43250,43252,43251,33995,34927,43886,37741,46037,42066,39132,42187,33142,42341,42340,35137,51826,33984,43254,33632,50888,34901,51543,38386,42343,42342,42572,42573,51827,42841,33642,42842,43255,43256,43257,36285,43887,52725,39133,39134,44523,51524,42188,36205,35045,34733,51652,35277,34478,36283,51828,51829,52185,33143,52186,34010,52727,52726,52729,52728,43888,54384,44525,53487,44524,38376,38377,36790,55396,46038,38266,55397,34757,34011,64471,64470,57670,47549,34759,33144,48198,37562,61935,50889,51544,42344,35151,38216,33778,45265,39135,42067,42085,42190,42189,35674,42191,33602,39136,34042,39137,43259,43258,43260,36160,38323,64475,43889,33145,44526,36052,59643,49743,42086,42346,42361,42356,42351,42350,42357,42355,42348,42362,42349,42345,42360,42359,42358,42347,42354,34300,34369,42353,34481,37986,42363,42352,34740,39138,42579,42585,42581,42587,51653,42584,42574,42577,42580,42576,42583,42586,42575,42578,42582,42588,35276,64323,37048,64322,34234,51838,51835,41045,42855,51836,42843,42845,42869,42864,38578,33146,51877,51837,42847,42849,51876,42856,51832,42868,42870,42844,42861,33873,51830,42867,38580,42852,33605,42862,42863,51831,42860,42858,42859,42865,51873,42846,42866,51875,42854,42851,51834,42850,51878,42853,37563,42857,38564,41050,42848,51874,33147,33148,36530,36042,51833,34480,36791,36290,33712,36284,34236,34235,33455,52203,52202,43343,52205,52207,52196,52199,52206,43344,64473,38140,52193,52197,39139,52201,52809,43339,52813,43261,52198,43262,43340,43333,43329,52194,43332,43337,43346,52195,52188,43331,52189,52191,43334,39140,43336,52187,52192,33149,43345,43341,52200,43347,38307,43338,52190,43335,39141,38582,43330,43328,41047,52204,35824,43342,35275,64474,33150,39142,52808,52731,52811,33185,38265,52733,43896,43944,43892,43943,43901,43940,43890,52732,52803,43939,52815,43941,37958,43897,33186,52805,52802,43895,39143,52730,43942,52810,43900,52812,43945,43891,43902,43899,52800,43937,52806,52807,43898,43938,43894,64330,38607,36023,43893,52734,33187,33779,36578,36529,34484,36579,52804,34736,37987,36580,38579,36687,35022,52814,33996,53572,44539,53489,39144,53494,44532,44608,53492,44527,44537,44542,53499,44538,44541,39145,53502,44533,53493,53570,53571,44535,53569,44531,44611,53496,44529,64346,53574,53497,53501,44534,44610,53498,44540,53568,53575,54433,39146,53573,44612,44528,53500,53491,64338,44536,33344,53490,39147,53495,36454,64341,34734,36417,33188,33997,39148,53488,44609,39149,54391,39150,45284,54439,45282,45279,54396,45275,54434,45286,54390,54395,54394,44530,45281,54437,54440,54387,46056,33189,54441,45287,45273,45270,54398,45267,39151,54438,38577,45274,54442,33190,54388,54436,45277,54389,54392,54397,33191,45278,45276,45288,45283,33192,45271,45522,45272,54393,45285,45280,54435,45269,34483,64347,45268,41030,33193,35950,36039,41033,41029,64477,54385,54386,55402,34739,39152,46039,46042,55413,46062,55416,46040,55409,46046,46052,46525,38381,46050,55406,46063,46043,46051,55414,56535,55419,55407,39153,55398,55411,55405,46049,55417,46045,46065,46058,46047,46044,33194,46055,55418,55404,55410,55412,55400,55415,46041,55399,33195,46048,46064,46060,55401,46054,37954,46061,46057,46053,39154,55408,37449,34738,38581,41034,46059,64332,34482,36825,56533,56529,39155,56544,56522,56531,46821,46822,46814,56540,46824,56527,56526,56524,56542,46812,56536,56525,46815,56534,46810,56530,56537,56539,39156,39157,56543,46819,56523,46813,56528,64342,46808,46820,56538,46816,46817,46823,46811,41567,46809,56532,38601,35949,33196,46818,41037,41035,56541,35023,33197,47565,47560,57685,57681,39158,57675,47554,47550,57684,47551,57678,57680,57683,41038,47556,47563,47557,57673,47558,47559,57676,47564,64481,57674,57679,47555,57672,47561,47553,33459,47552,57677,57682,41042,47562,64729,39160,41039,33198,35825,64357,57671,48205,58695,58692,41040,48199,48211,48212,36177,48202,58690,48204,58617,48210,65117,58694,48201,58696,48200,58691,58693,48203,58689,58618,58615,41041,55403,58621,58614,58620,58619,58616,64482,48207,64362,33199,39162,48206,64484,48208,58622,48818,58688,33359,35518,39232,59717,59645,39253,48830,59714,48822,48826,59713,48825,48821,48824,48819,48829,59715,59646,48828,59644,48827,59716,59712,48209,48831,59718,48823,48820,37975,38602,33200,60614,60616,49275,60617,60615,60613,60612,49277,60611,49278,39254,33201,39255,60609,60610,49274,49313,49276,41043,60608,33202,49744,61372,61370,61375,61373,61371,61374,39256,41052,33203,39257,33396,35951,41051,50016,61938,61939,50262,61940,61936,61941,61937,49745,33204,39258,62462,62529,50265,62528,50264,50263,39259,50266,62917,62918,39260,50394,50393,50395,62916,63192,63191,50541,50543,50542,63193,50632,63654,37988,50673,63653,63726,39261,51529,33205,42365,42364,42591,42590,51655,42589,51654,34485,37743,42873,51881,51880,42871,42874,34237,51879,39262,42872,33606,37443,37622,33999,36298,52208,52209,43348,40278,33782,34486,43946,33206,53576,53577,44613,44614,34238,54444,45289,45291,54443,45290,55420,46066,37557,46825,46826,56545,39263,47567,47566,58697,59720,59719,39264,63851,42087,51545,51566,51567,37989,37442,42594,42598,51657,42596,42595,51656,42597,42593,39265,42592,51658,37990,37991,35519,34001,42918,39266,42915,39267,42877,51882,38325,39268,51883,64487,42913,51885,42875,51886,51884,42878,42914,42917,42916,42876,51887,65135,34240,33207,33783,35025,35695,36178,34241,34742,34743,43353,52222,43355,33208,43354,64489,52288,43352,43351,52213,52212,52210,52215,52214,52211,52220,52221,52218,52216,43350,64488,52219,43356,52289,39270,33209,52217,43947,43349,35694,33210,36792,37992,33211,39271,43948,52820,39272,52826,39273,35520,43954,52824,52830,52821,52825,52827,52829,52823,39274,52822,52817,52818,43949,33212,43951,43950,52819,52828,35024,36278,33784,37994,34488,33213,36299,43953,37995,36418,37993,33214,52816,53587,53586,53591,53582,39275,53585,53584,39276,53588,33215,53592,44615,44618,39277,53583,53589,33216,44617,53578,43952,54458,53590,53581,33217,44616,53580,37996,65097,34242,35278,34003,54449,64492,39278,45292,45296,54465,54447,54461,45297,54463,64491,54469,54473,54464,54452,54460,33218,54474,54472,54462,54457,54450,55462,54448,45301,54455,45302,45298,54445,54467,54453,39279,54451,54470,45299,38218,54476,45293,45295,54459,54454,44619,45294,54456,54471,54475,54466,54468,34489,39280,54446,34744,55457,55466,55465,46074,55458,46075,46073,39281,55460,46070,55464,64495,55459,55461,55421,46068,55474,55473,55470,46067,46071,46072,53579,55467,46069,45300,55469,55422,55472,55471,64490,55475,56559,55468,37977,37865,33219,39282,55463,56551,46836,46839,46834,56550,56554,56549,46828,46838,56546,46832,56553,46830,46829,56556,46831,56558,39283,56555,46827,39284,46837,56560,56548,56557,56547,38349,46833,33220,46835,56552,37997,56561,38242,39285,57693,47568,57699,47573,57695,57702,57687,47575,47569,57692,48213,57691,57700,47570,47574,57690,57696,57701,57686,47572,57694,57698,57704,57688,57697,33221,47571,57703,37726,57689,35317,33222,48217,58699,48215,48214,58701,58706,58702,58705,48220,39286,48805,48219,39287,58698,58704,48218,58703,39288,58700,33223,48216,39289,39290,33224,37998,59725,63961,59727,59722,48833,59724,48832,59726,48835,59728,48834,59721,59723,37468,39291,37614,49317,60620,39292,49316,60621,49315,60619,49314,60618,37999,49747,49746,61942,61944,64496,61943,50017,50018,39293,50019,62530,50267,33225,63443,63442,50674,42088,42192,33607,42919,35410,37445,37811,34026,52831,34004,33345,46076,46077,33226,56562,47576,57705,58707,33227,51546,33914,39294,51888,33986,37828,50890,33228,52290,52832,53593,44620,33229,39329,61945,50396,42089,42366,51568,33230,42599,42600,33231,43357,40280,33786,45303,47578,47579,47577,42090,42193,42195,42194,51547,42196,42401,51569,42402,33461,35251,42601,42602,33608,34492,51659,42920,51889,34493,33643,43361,52291,43359,43360,43358,53594,35026,43958,43957,43959,43956,52833,43362,43955,34802,44621,44622,44623,33462,54477,33463,46078,55476,45304,33464,34244,46840,37606,47581,47580,57706,33465,48221,48836,61376,63194,63444,42091,42403,65126,42404,51665,42604,42607,51663,51661,42606,51664,51666,51660,42609,42608,42605,42603,51662,33609,33788,34245,39330,42931,42928,51894,51897,51896,42922,42930,39331,42927,51893,51891,42926,39332,33466,42921,42924,39333,51892,51899,51895,42925,42929,42932,51890,51898,42923,34747,34494,36315,38309,43367,43375,39334,52303,52296,43376,52307,52292,52299,64497,39335,43366,52293,43364,52300,52304,43363,39744,52305,52298,52301,39336,43378,43369,52308,52306,37884,43374,43372,52297,43371,52295,52294,43370,43368,43377,43373,43365,52302,33467,43961,64498,43968,52847,43960,52839,52835,52851,52834,43963,52844,43966,43969,43964,52848,43967,33468,44630,52854,52836,39337,39338,52838,52845,52849,52853,52850,52843,52846,39339,38336,52840,43971,52842,52841,52852,43962,52837,43970,39340,43965,33787,36300,34749,34748,36949,44636,53602,39341,44635,39342,53600,44624,39343,44629,53599,53596,53601,44625,53595,33469,44628,44626,53603,44627,44631,39344,44632,39345,44634,39346,44633,39347,39348,39349,53597,53598,37564,33713,34005,53604,34745,54484,45305,55490,54483,54502,45376,39350,54500,45310,45306,54509,54493,54496,39351,45379,54506,54498,45307,45380,54503,54501,54486,54507,54495,54490,39352,54480,54508,54492,54479,39353,45378,54497,54510,54494,54482,54487,54478,39354,45377,38243,54491,54488,45308,54481,39355,54505,45309,39356,54489,54485,39357,54504,39358,35279,34746,35521,46144,55483,39359,55480,55497,55485,55498,39360,46146,39361,39362,55494,55491,39363,39364,55492,55495,55499,54499,55501,56647,39365,46147,55502,55478,55488,65144,55493,39366,46145,46148,55500,55503,55482,55479,39367,39368,55481,40048,55486,55484,46149,55496,33470,39369,55487,55489,55477,56570,56568,46914,46912,56643,56569,56644,56640,56567,56646,56566,56573,46846,46845,46844,56571,56641,46841,46913,39370,56564,56574,56563,56572,46842,56642,56565,46843,56645,39371,39372,33471,40295,36581,39373,35280,57710,47586,47585,47587,57722,57712,57718,57707,57721,57720,57724,57717,47582,57716,47588,39374,57709,47583,57723,47584,57711,57714,57719,57713,57708,39375,33472,64359,57715,58709,48225,58712,58711,58714,58716,63964,48223,58710,39376,58708,58717,58715,58713,58719,58718,48227,48222,48224,48226,39377,58720,59735,39378,59734,59733,39379,59736,59729,59730,59738,59731,48837,59740,39380,59739,59732,37615,60625,49320,60623,60628,60627,59737,49319,39381,60626,60622,60630,60629,49318,60624,48838,33473,35281,49748,61377,61946,61947,61948,50268,39382,50269,39383,62531,39384,62920,62919,39385,63195,63196,63445,63655,39386,42092,42093,42094,42197,42405,51667,42610,42611,33714,42935,42936,42934,42933,43379,52309,43381,43380,52310,35028,39387,43972,36043,44637,53605,41025,54512,45381,46151,54511,46150,39388,47589,39389,57725,48839,49321,60631,39390,50270,50544,50891,51570,39391,42406,51571,42614,42612,42613,42615,42938,42937,37885,51900,42939,39392,51901,52311,52312,39488,43382,43384,43386,43383,43387,43385,35827,34495,37588,35696,34751,43976,43973,43975,43977,43974,53606,52855,35030,37043,53608,53607,44643,44639,44640,44642,44644,44641,40296,44646,44645,34750,34649,45386,54514,54513,45385,64579,45384,45383,45387,45382,33474,55509,55506,46153,55505,55510,46155,55508,46152,46154,55507,39509,56648,39510,56649,56650,37481,34496,47590,47598,57726,47592,47596,57761,47597,47593,47594,47591,47595,48230,55504,48231,48229,48228,59741,48840,60632,60633,33475,50020,50271,42095,34497,42616,34006,35029,37734,34490,43978,37703,53609,44647,38235,37602,45390,45389,45388,46156,46157,55511,47599,48841,42096,51548,42198,51572,33476,34528,51668,42617,34487,43388,33477,34479,33478,56651,64499,42097,39511,42199,51669,34000,51902,51903,42940,33369,55512,46158,56652,35282,49322,42098,42152,42200,51573,42407,34537,42944,42943,42941,42942,34753,52313,43390,43425,52314,43389,33856,33623,43982,52856,43981,43979,43980,44650,44648,33479,53611,44649,53610,44638,54515,45392,45393,35952,45391,34498,47600,57762,48232,48233,35283,58721,49323,61378,61379,50397,63656,51531,42201,42099,51575,51574,38000,40292,42618,51671,51672,51670,51673,35284,34263,33755,39512,51911,51906,51908,51910,51907,42948,51904,51905,42945,42946,51909,51912,42947,51913,34500,33858,33717,33480,34007,33716,33481,52328,39513,52322,52317,43427,52325,52323,52316,52329,52332,52327,52320,43429,52326,43430,52321,52324,52315,52319,52331,43431,43432,52318,52330,43426,43428,33482,33483,38001,36301,38003,36860,33484,39514,36582,38002,52907,52900,52906,52899,52901,52861,52859,52908,52905,52857,64501,43984,52903,52904,40305,52902,52860,52858,43983,52898,52862,52897,52909,37968,33715,36532,33857,36045,44655,64502,44654,53612,44651,53614,38237,44656,53615,44659,33485,44657,53616,52910,53618,33486,44653,40297,44652,53613,53617,44658,36531,36302,45395,45394,54517,54521,54523,45396,54526,39515,45400,54593,45402,33487,45398,45406,33488,45403,54519,45397,54518,54516,54595,54520,45399,54594,45404,54525,54524,45405,54522,45401,39516,34499,54596,39517,54592,55527,55534,55523,46161,55519,55535,55513,55532,55530,55524,55533,55526,33489,55518,55536,55516,55529,55514,55537,64503,46162,39518,55531,56655,55517,46159,55521,46160,55520,55525,33490,55522,39519,38245,35285,55528,36316,33491,36455,56659,56662,56654,56656,56661,56660,46915,55515,56658,33492,46916,39520,56653,56657,33493,33494,65150,38004,57769,40314,57776,57767,39521,57774,57765,57773,57777,57764,57768,57763,47601,39522,57766,47602,57772,57771,57770,37886,33495,57775,64504,33496,58725,58727,48235,58728,58723,58722,58732,58730,48234,58733,58724,58729,58731,58726,39523,59745,59750,59744,59749,39524,59742,59752,59748,59753,59747,59743,59751,59754,59746,39525,60634,49327,33497,49325,49324,49326,61380,39526,61810,61949,36044,62532,62533,50272,62921,33363,50398,62922,33498,63198,50546,33499,50545,63197,50633,39527,63446,50892,42100,42619,51674,51914,43189,45407,37719,42101,42410,42409,42408,34994,42949,44660,36419,56663,42102,42103,42104,42202,64506,43985,52911,37478,46163,42105,51549,34501,42411,42412,51576,33644,42620,33718,39528,51915,64507,42950,51916,34009,33860,43438,64508,52334,43436,43435,52333,43433,52335,43434,43437,34755,43986,43988,52915,52912,52913,52914,52916,43987,33859,34008,53620,53619,44662,44661,35286,37735,37487,36688,45410,54598,45409,45411,45408,36456,34502,46165,54597,46166,55539,35019,46167,55538,46164,33500,39529,56666,56668,46917,56667,56665,56664,57780,47607,47605,47606,57778,57779,47603,58737,58735,64510,48237,58736,48238,48236,47604,33501,59757,59755,59756,58734,60636,49328,60635,61381,61382,59758,61950,42106,42413,42622,51675,42621,43439,46918,50893,42203,42414,43989,46168,50894,51577,37749,51578,37566,51676,34762,42952,51920,51918,42953,51917,51919,51921,42951,35698,36533,39530,33719,35288,43443,43444,43441,34758,43440,52920,43442,35697,34756,43990,52919,52921,52918,52922,43991,44665,53621,53623,44663,53624,44664,53622,52917,54599,54602,54603,54600,45415,45414,45412,45413,54601,39531,45416,46170,46171,37810,46172,56669,56671,56673,46920,46919,46169,56672,56670,65188,57784,33502,33503,57782,57788,47608,57789,57786,47609,57783,57781,57787,48240,58739,57785,48242,58740,48241,48244,58741,48239,48243,59763,59761,59760,59762,59759,33504,50022,62534,62535,33505,62923,63199,50773,50895,43445,42954,38308,38305,43992,38382,38394,42107,42204,34171,42415,51677,33878,42955,51922,52923,43993,33506,47610,42108,33507,38314,39532,42657,37806,33508,46921,42109,42205,42206,38005,42417,42416,51678,42658,51923,42956,33861,33509,52337,52338,52339,37881,43446,43447,52336,43448,36291,43994,52924,36046,53626,44666,53625,33510,45417,54604,45418,54605,34761,33862,40315,46173,37745,56674,33511,57791,57790,33512,47611,48245,58742,48842,59764,49329,64577,50547,63448,50896,33513,35536,34277,52340,35235,52925,45419,55540,46922,33514,49749,49749,50897,42958,65192,42957,43995,33515,53627,45421,45891,45422,45420,46174,57792,47612,48246,39533,51532,51679,51925,42959,51924,42960,34503,43452,52343,52342,43451,43449,43450,52341,33516,37542,43997,52926,44000,43996,44002,43998,43999,44001,44669,44668,44667,37596,33864,54607,45423,45426,45424,54606,45429,45425,54608,45428,45427,64583,55542,55541,38234,46177,46175,46176,55543,46923,56676,46924,56675,39534,58743,37800,37750,48248,57793,48247,47613,64584,60638,59765,49330,60637,62016,62536,62537,39535,42207,40181,42418,35034,51579,42962,42964,51682,51928,51927,51926,51681,51680,42660,42963,42961,42659,39536,37454,43453,52344,39537,43454,51933,51935,51934,52345,34764,51930,42968,42966,51929,51931,51937,33517,42965,51932,51941,43456,51938,42967,51936,51939,43455,34247,43457,51940,34504,34765,34769,36459,33866,34248,52399,52386,52350,52398,52393,44007,43458,52394,52397,44003,52396,43459,43464,43462,52387,52348,52389,43469,52400,44004,52390,44005,43465,52392,39538,52941,44006,52347,43466,44008,43467,43463,43468,52391,52346,52395,43460,38310,52349,52388,52385,43461,52927,52928,38006,35403,36724,36583,52938,53665,52939,44014,52942,52932,44013,52934,52935,33518,36086,52937,44009,44707,63962,52933,52929,44708,52943,44670,53629,52936,53628,52931,52940,33519,44012,44705,44018,44706,52944,53630,44011,44710,44017,44016,44015,44709,52945,44711,44010,52930,35289,37544,34012,64587,34506,36689,34249,34015,34016,36317,39539,45430,53668,53670,53672,44712,44718,54611,53676,53667,45432,54609,33520,44717,44715,53678,54610,53669,44716,53673,44719,53675,44714,53674,53677,53671,64356,44713,45433,53666,45431,64588,33521,65190,45434,37807,37441,36457,35828,36458,34766,54613,54622,46180,64589,45436,45475,46181,54624,45482,55545,54614,45474,45477,45438,54612,54626,54629,55625,54627,55549,45473,45480,45484,54621,55544,54625,45435,55546,54628,55548,54617,46178,54615,54616,45479,39540,45478,54619,45483,54623,45476,54620,33522,45481,46182,46179,55547,33453,54618,45437,35953,34508,35699,35032,34014,34507,34767,34505,46187,46191,55616,46929,46189,55620,46193,56677,55622,46931,46185,46188,55623,64590,55624,55630,46195,46932,55626,55631,55619,46942,39541,46933,46194,55617,55632,39542,46941,46192,46926,55629,33523,46196,55621,55550,46186,55618,55627,39543,46925,46930,46183,55628,46928,33524,33525,39544,46184,34768,36534,46940,57795,56688,56680,57794,56684,56686,64591,56683,46939,56682,46943,57810,33526,33527,46938,47680,56689,57796,64713,46936,56681,56685,47614,46927,56678,56679,47681,46935,46937,46934,56687,39545,35954,57800,57801,57806,48253,57813,47687,64594,47686,57808,64596,48252,57797,47685,39546,57812,47683,47684,38317,57809,58794,48250,46190,57811,48291,57803,48251,38246,48290,57798,57802,57799,57805,47688,48249,47682,58746,57807,48289,64595,48292,57804,39547,48254,58745,38353,58750,48846,58744,59811,58793,48296,48294,48844,58790,58786,48300,39548,59768,33528,65140,48298,58785,64593,59766,58789,58792,58749,48299,48293,59767,48845,58791,48295,48297,58788,48301,58787,58748,58747,48843,58795,59770,60640,48848,33529,59810,39549,59774,60641,48849,59809,39550,59772,49332,60639,59769,59771,49333,48851,49331,48850,49335,59773,48847,39585,37469,35522,35701,39586,35033,61391,61383,39587,60647,61384,60643,64598,49750,60645,60644,49334,60642,60646,61392,61388,61390,61385,61386,61389,61387,50023,33530,50026,50025,50024,50273,62538,50274,62017,50399,62924,50400,50548,50634,63449,63450,63451,37035,63930,42208,51580,42419,36280,42662,42663,42661,35290,42664,42970,42969,33531,52401,43471,43470,36461,35035,53679,45485,45486,33532,33533,46197,56690,46944,46945,56692,56694,56693,37705,57815,57814,47689,57816,36460,58796,48302,37488,48852,49336,49751,49337,42209,33534,37053,64600,51942,37868,52402,43473,43472,43474,44019,52946,52947,53680,44720,45487,46198,55633,42210,40178,42110,42211,51581,42423,42422,42420,42421,37825,42667,51689,51691,42666,51683,51684,51690,51686,51688,42665,51685,51692,51687,33785,64601,35523,33869,33867,34020,42977,42986,42984,51952,51949,51957,42982,51958,42975,51955,42981,51951,51950,42979,51956,42980,43475,42974,51953,64603,51943,42971,64602,42990,51948,51954,42976,42978,51944,51945,51946,39588,42989,42983,42988,51947,42987,42973,42972,42985,34775,37054,37819,64855,37738,36462,36463,33870,34251,36008,34002,64604,43489,52414,52407,43484,43503,52403,52410,52412,52415,43498,52411,52404,43496,52408,52416,43481,52413,43491,43490,52406,43479,43480,64326,43478,35043,43502,43494,43488,43476,52409,43487,43477,43495,43504,52948,43492,52405,43482,43485,43486,64605,43500,43501,43499,43493,43497,43483,44020,37809,35294,37493,37470,35702,34017,35036,35830,35293,34773,35041,52954,44097,44024,44026,44096,52966,44029,53681,44721,44099,52951,52959,44030,52958,52955,52963,52965,44023,44027,44098,44723,52960,44025,44101,52953,44028,44722,44022,52950,52957,52949,52952,52956,53682,44100,52961,52962,52964,44021,38132,37494,34770,36087,35044,35038,33721,34774,35037,36535,33871,35292,34018,37055,44737,53694,44735,44736,53684,53700,41054,44726,39589,54630,53702,53696,64607,53687,39591,53705,53690,44732,54653,53693,44734,44725,37718,53707,53695,44728,53688,53685,53686,44729,53701,53708,44731,53692,53691,44739,44738,44724,44730,44733,53704,53698,44727,53683,53706,53697,53699,53703,37717,34772,34250,35042,34019,34252,64606,54631,45495,45515,45514,45503,38233,54649,54645,54642,54694,45498,45490,54647,46248,45494,54689,39592,45516,45513,54651,54634,64597,45512,54691,54633,45501,45505,54690,54643,45506,45500,54632,39593,46200,54693,54641,45511,54644,54692,45510,55634,45491,54639,45496,45507,45502,54648,54638,54636,54654,45488,45508,45492,46199,54652,45493,45489,45504,45499,45497,54640,45509,54637,54650,54646,55636,55635,37545,35040,34510,38007,35524,34776,64343,34511,37056,36464,54635,55652,46202,39594,55658,55641,55655,56695,46205,55659,55662,46204,55644,55661,55660,46206,55637,46201,46243,64876,46241,55657,55647,46245,55664,55656,55665,46253,46251,55654,55653,37057,55651,55645,46244,39595,46242,53689,55638,39596,56759,55639,46203,46250,56697,46246,46247,55640,55663,56696,55648,55643,46249,55649,55646,41055,46254,37058,39597,64349,36179,39598,34777,35831,35295,34509,36088,46960,39599,56700,56753,56758,56746,46956,56763,46953,56698,56699,46946,46955,56740,46958,46959,56741,56754,56760,46954,46948,56739,56701,56762,56744,56745,56702,56756,56747,56757,56749,46949,57817,46952,46950,56761,56752,56748,39600,56737,47699,56751,46957,56743,39601,56742,35956,39602,46951,46947,57838,56755,56750,37701,56738,35955,36725,35527,37850,36690,36047,57833,57818,57829,57836,47697,46252,57834,47692,47691,57841,39603,57819,57832,57820,57831,47695,57835,55650,57842,57827,47698,58810,48303,57840,57839,47700,58797,48304,58798,39604,57823,57824,57821,57826,57822,57843,47694,48305,47696,47701,39605,57825,57837,57830,37552,36826,58801,47690,48308,59818,58806,58805,58807,39606,58804,48309,37059,48315,48312,48313,58799,58802,58812,48321,48319,58803,55642,48306,58809,58800,48322,58808,47693,48311,57828,48314,48318,48320,48317,48316,39607,48310,58811,48307,48323,39608,35526,39609,35529,36726,33357,36862,48856,48857,59817,48866,48863,37060,48854,48861,59819,48859,48853,48860,59816,49339,48855,48862,49338,59815,59814,48864,41053,48865,59813,59812,49340,59822,48858,59820,39610,35528,49341,37061,49346,60650,60652,39611,49343,64419,60653,60649,38344,60651,49344,49347,60648,49342,49345,49753,59821,49752,37062,49758,61396,49756,49757,61399,61395,49754,61393,50027,61397,61398,61394,37063,49755,62018,62021,39612,62022,62020,62023,50028,62019,39613,36951,62542,50276,62541,62540,62539,50275,50277,62925,50402,50401,41056,39614,63201,63200,63203,50635,50549,63453,63202,63452,50637,50636,50675,63657,63727,42212,65115,55666,59823,50898,37064,42668,51959,42993,42991,42992,52417,43505,44102,52967,52968,44103,53710,44740,44741,53709,34778,38324,37065,45523,45519,54695,45526,45525,45518,45521,45524,45520,35296,55670,45517,46255,39615,46257,46258,55669,55672,46256,55667,55671,55668,37066,46961,39616,56764,34779,47702,57844,48867,48324,58813,48325,48326,58815,58814,58816,59825,59824,60655,60654,49348,49349,62024,64695,39617,42213,37067,37698,35516,55673,37575,46260,46259,56765,35957,61400,50403,63454,42214,44742,45528,45527,55674,55675,46962,57845,47703,59826,42215,42424,43506,52418,33875,52969,44104,45529,35530,55676,46261,46963,58817,58818,39618,60656,49759,63728,42216,52419,43507,44105,39619,52970,44743,53714,53712,53713,44744,53711,45531,45532,54696,45533,45530,55677,39620,55678,56766,39621,47705,47704,37068,60657,61401,62026,62025,62543,50899,51550,44106,39622,42217,42425,35574,42670,42669,42671,42672,51694,51693,51960,42994,51963,51962,51961,51964,33722,35703,34256,43508,52425,52421,52430,39623,43515,43513,52426,52422,52429,43512,43584,52424,52420,43518,52427,43511,52428,43514,43516,52432,52431,52423,43510,43509,43517,39624,35048,52975,52981,37069,44112,44109,52972,52977,39625,44115,44107,52976,44110,44113,52979,44108,52984,44111,38398,44114,52973,52978,52982,52974,52971,52983,52980,37070,33876,36166,37071,44752,44745,44748,37072,44751,53717,64610,44746,53715,33723,44750,40281,44747,53718,44749,35046,35047,34255,34254,37073,54700,45535,54699,54701,45534,45539,53716,54698,54702,39626,45536,54697,45538,64611,45537,37074,55719,39627,55714,46262,46266,46263,55717,55720,37075,46264,46265,46270,56775,55718,46268,55715,55713,39628,46269,55716,37501,34780,37076,46969,39629,56767,46966,46967,46965,56772,56771,56768,46971,65146,56770,46267,39630,56774,56769,46968,46964,46970,56773,39631,47708,57848,57847,57846,47706,39632,37871,35298,47707,58821,58824,48328,39633,48327,58825,58820,48330,58822,39634,48329,58819,39635,58823,48873,48870,59835,59834,37077,59833,59828,37078,59829,37079,48871,48868,48872,59827,48869,59830,59831,59836,39636,39637,59832,60658,49351,61404,49350,61402,61403,49760,50030,62027,50029,62545,62546,50278,40394,62544,50404,63455,50638,63658,63659,42218,41048,42673,42674,42995,37080,52433,44116,44753,45540,37081,37082,45266,37883,46271,46272,46028,55721,39638,46972,57850,57849,42219,42675,52434,43586,39639,43585,37083,52985,52986,53719,53720,44754,44755,37084,44756,54703,39640,45542,46274,64612,46273,56776,57210,57851,59837,65118,49761,50279,42220,37329,42428,42429,42427,42430,42426,33915,42678,51702,42677,42679,38268,37085,51697,51696,51699,51698,51701,42676,51695,51700,33881,38008,35673,36048,39641,51965,43005,51966,52035,43004,52039,52034,52037,42997,42998,42999,43000,43072,52033,43002,43073,52032,52038,43001,52036,43003,42996,43006,33879,37980,38009,37086,33619,37087,64422,34260,36466,43607,39642,52436,43587,40306,43597,43598,43590,43608,43592,52444,43603,52439,43593,52454,52455,52447,52440,43606,52452,43601,43599,52453,52451,52443,52435,52442,43594,43600,43588,52446,52445,52437,39643,43602,52449,52438,43605,52456,43589,43596,52441,52450,43604,43591,43595,34789,52448,38141,38010,34784,34021,39644,36828,37088,37089,36319,37090,34514,34262,64614,33880,53083,44124,44137,38213,53078,53068,44130,53066,44123,53061,44133,53074,52990,53057,39645,53060,52987,53073,53089,44128,53062,53080,52989,53087,53088,53091,53082,53067,53075,44134,44121,44129,44141,44118,44120,39646,39647,53059,44138,44131,53085,53056,44140,44135,53065,35534,44139,53072,53064,44132,53084,53076,44126,53090,53063,44122,53081,53071,44127,53077,44119,52988,44136,44771,44125,53070,53069,53058,64615,53086,37091,53079,37092,37978,44117,37093,37094,37965,37969,37448,37095,34785,34023,35302,36827,34022,36829,39649,36794,35959,39650,35051,53740,44778,53741,53729,44767,44779,53722,39651,53731,53739,37096,53721,53748,44757,65093,53747,53742,53743,44765,44776,53733,53734,53744,53735,53730,53724,53725,53738,53732,39652,34782,44758,44762,53746,53726,44774,44770,36970,44773,44780,44763,44775,53737,44777,44760,39653,44759,53723,37097,53727,44768,53745,53736,53728,44772,44769,37098,44761,44764,37099,39654,37100,37457,33404,37580,36318,35055,34261,35299,36465,35053,35049,35301,34787,34783,54724,54708,54709,54713,54728,54725,54718,54717,45549,54721,54736,54704,54737,54723,54741,54729,45548,54727,45543,45564,45554,39655,45558,45557,54705,39656,54734,54740,54732,54739,37101,54720,54706,54738,54722,45546,45559,54731,45552,64344,54730,54707,45560,45562,54733,45563,45545,54714,54735,34491,45551,45561,54716,54726,54711,54715,45556,54710,45544,45553,45550,54719,44766,55744,45547,64616,37102,38011,33391,39657,35300,35960,39658,45555,39659,55747,55769,55758,46294,64620,46289,55741,46290,55757,37103,55750,55763,46286,55723,55765,46276,55731,46279,46278,46295,55725,55759,55760,46281,46277,55739,46288,55734,55761,46284,55753,55766,55728,55733,55727,46283,55746,56798,55729,46287,55738,55762,46282,55735,55732,55749,46285,46275,46297,55752,55751,64619,64618,39660,39661,55724,46280,55764,55740,55742,39662,55755,55754,55722,46291,46293,55730,55737,55745,46292,55736,55748,55767,55756,39663,37961,65103,37953,37558,35705,36536,38050,37107,38049,36971,55768,37108,38013,55726,64617,35052,56818,47014,56816,56795,56800,56793,37117,56812,56779,56786,56810,56820,56796,64623,56783,56802,56807,56787,39664,56804,56784,39665,56791,56792,47016,56811,56809,56780,56814,56815,56817,47020,47012,33388,54712,56788,56806,56789,47009,47025,56813,47023,47019,56778,47011,37118,56781,47024,39666,56797,56777,39667,47017,56801,56785,47018,56794,46974,46296,56803,55743,56782,39668,64621,56808,47013,56805,47010,56799,47021,56790,56819,34786,37474,38051,64622,38052,47015,57030,36206,39669,47022,35833,35706,36928,37957,57930,57928,57950,57926,57944,46973,47711,57922,57949,38247,57927,57941,47716,47709,57947,39670,57920,57946,39671,47727,57937,57953,47725,57929,47710,57931,57945,47719,57924,47723,47713,57933,57923,57852,64625,57943,47720,57952,57853,47717,57939,47718,57925,57936,57932,57934,64626,47712,57951,47726,57935,57954,37358,57854,57940,47715,47724,47722,57921,57942,47721,39672,47714,57938,39673,39674,37851,57948,35533,38053,38525,35531,36793,39745,58837,58833,58829,58849,58846,48333,64629,58853,58836,48344,58843,58832,58842,48341,58862,58859,58845,58830,38526,64628,58850,58852,48337,58840,58835,58826,48334,48342,39746,58855,48343,58827,58861,58848,58854,48340,65108,38606,58851,58858,48345,38635,48339,58844,58831,58863,58828,58856,48336,39747,58838,39748,58839,48335,48332,58834,48338,38636,48331,58857,58860,58841,59850,37559,35050,37952,39749,39750,38054,35303,38637,35958,59842,39751,59838,48886,38638,48875,48880,48876,59852,59863,48874,59844,59853,58847,59854,39752,48881,39753,59869,48885,48888,59840,48884,39754,59867,59868,59858,59857,59849,64631,39755,59859,59866,59865,39756,48879,48877,59851,59848,59845,59864,48887,59862,48883,48882,39757,59856,38639,59839,59841,59843,59861,59855,48878,59846,59860,35704,38055,38648,39758,59847,38649,35707,38014,38652,49359,60741,49352,60661,39759,60737,49354,60744,60668,60663,39760,60745,60659,60670,39761,49361,60740,60746,60669,49353,60736,60660,49360,39762,60743,60665,49356,64632,60667,60664,49362,60666,49355,49358,60739,60662,60742,39763,60738,38653,38654,39422,49763,61415,49768,49769,39764,64630,49762,61414,39765,61411,61412,49766,61406,61410,49765,61407,39766,39678,49767,49764,61405,61409,61413,38371,39767,64476,62033,62030,62039,62038,62036,62031,50034,39768,39769,62032,50033,49357,62035,50032,62040,62034,62029,61408,40185,40417,50031,62028,62550,62549,62037,50280,62553,62554,62548,62552,62547,39770,62929,62551,50407,50405,62927,62930,62926,62928,50406,40446,63205,63206,50550,63204,63458,50639,63456,63457,63660,50774,63731,63729,63730,63732,64633,63931,38056,42221,42680,36049,43609,35535,52457,34790,53092,53749,53751,53750,53752,45565,54743,53753,38329,54742,54744,54745,55770,46299,55771,55773,46300,46298,55772,56826,56824,56823,56822,56821,47026,56825,47728,57955,57957,47729,57956,48347,48346,58864,64634,37484,59871,59870,59872,39771,48889,60747,49363,61416,49770,62041,50551,42222,42431,42681,43074,43610,43611,37829,44142,53754,37584,47027,37576,37872,59089,48890,49771,42223,39772,42682,35961,52459,43612,52458,40782,53093,44143,53094,44144,34791,53756,44782,44781,37583,54750,54748,54749,54747,54746,36277,35304,55774,55777,46302,55775,46301,55776,36584,56827,57958,57959,57960,35537,58867,58866,48348,58865,58868,59873,59874,59875,60748,49364,49772,62042,40783,50408,51551,34792,44145,53095,44783,45566,37471,46303,55778,47029,47028,37045,35445,57961,57962,48349,48350,59877,59876,61417,63459,42224,51552,42432,37841,43075,52040,44146,47030,42225,33893,53096,44147,53097,35297,49365,42226,39773,52460,38057,53098,35305,53826,53825,53758,53757,53827,53824,41058,45632,45633,46304,55779,55780,55781,36537,56897,56898,56896,64635,56829,56830,47031,57963,58871,58870,58869,58872,59879,59878,48891,59880,36795,49366,60749,61418,62043,63207,42227,42434,42433,43613,51553,51582,42683,37486,51703,52041,52042,43614,39774,52461,44148,53099,53100,33882,44784,44788,53828,44787,44785,44786,34028,54751,45634,46307,46305,46306,55782,39775,41059,47730,42228,40179,51617,38379,42435,65088,37330,51620,41060,42438,51619,42437,42436,43076,51618,51704,34233,34517,51708,51710,51776,42693,42694,51707,42689,51705,51709,42690,42685,39776,42686,38378,42692,51706,42684,43077,42687,42688,42691,41061,34033,52059,52057,52044,43089,52051,43084,52045,52053,39777,52050,43087,52049,43094,52058,43096,39778,43098,52043,33646,43085,52060,43092,43095,39779,52549,43079,43102,43093,52046,43082,43097,52054,43080,43081,52547,52047,43088,43099,52061,52048,43086,43091,52462,43100,52055,43090,43101,43078,52052,43083,52056,52548,34264,38346,37739,35060,36469,35710,33887,34268,36320,34523,43626,43642,52469,43633,39780,52555,43618,41062,43621,52546,52467,52471,43629,43631,52474,43638,43624,43622,43623,43637,52551,43632,52473,52475,43630,43635,52476,52554,44149,43641,39781,43619,52553,64637,52557,52472,52559,52544,43628,52468,43627,43645,43634,52466,53109,43640,43644,52545,52550,39782,43646,43639,43625,43615,43620,52470,43616,52558,33886,52464,52463,52477,52465,43643,44789,43636,52478,43617,44198,36692,36796,35746,52556,36727,35538,34794,35058,33725,39783,53116,53153,39784,53156,53111,39785,39786,53159,53162,53164,53108,44150,44155,53833,44205,53157,53165,53115,53107,53860,44158,53154,53112,53114,44197,53117,44157,53104,53160,53163,38320,44154,39787,44200,53101,44202,44152,44206,53161,53103,44203,53854,52552,44156,44151,53110,53102,44204,44196,53155,44201,44199,53113,44193,53105,44194,44195,53106,53158,44153,53118,33726,37610,37704,34793,34796,36321,35059,34034,36585,35308,35543,34035,36468,35057,34800,39788,53836,44797,44867,39789,37821,53845,53851,53847,53834,53837,53830,53831,44874,44794,53846,53855,44869,44790,44864,53838,44866,53839,53849,39790,64423,44868,53864,53832,44796,44795,44872,53829,53862,53850,53863,53857,53843,53858,39791,53852,53861,53859,44873,53844,44793,44792,44865,44871,53856,44870,53841,45635,39792,53865,53840,53835,44798,44875,44791,65116,53848,53853,37864,35309,33393,36539,36303,34522,36541,34032,34036,34526,39793,34798,34799,34521,35311,45669,54753,54757,45650,45648,39794,45639,54755,54754,45659,54760,45653,64673,54778,54855,45636,54775,54768,45671,54752,54780,45668,45656,45667,45646,54764,54782,54774,45647,45641,54853,54781,54848,45649,45657,54850,54762,54779,54767,54852,45662,45638,45660,54772,54770,54771,45651,54766,54765,45640,54759,54854,45642,54769,45672,45666,54758,45663,45661,45670,54776,45665,53842,54777,45664,54849,45637,54773,45655,54761,45654,41063,45652,45644,45643,55783,54851,39795,54763,39796,55804,39797,45645,37476,37574,37604,33885,37597,36304,34516,35541,39798,34520,35062,37546,34524,38058,35539,34525,39799,46401,45658,46318,55798,46332,55786,46315,46311,55881,46317,46321,46316,46325,55885,55876,41064,39800,55793,46330,46324,55805,46308,55882,55875,46312,55799,46327,55893,55894,46309,55880,46329,55803,55789,55790,46333,55794,55801,55795,46331,46404,55791,55784,55785,55787,46314,55800,46328,46402,64638,39801,55802,55891,55883,46310,55889,46322,40395,46320,39802,55895,46319,55873,55796,55806,46407,55877,55874,55792,46403,55887,55884,55892,46313,55872,46406,55879,46323,46326,55878,46405,55797,54756,41065,55888,55886,55890,38347,36207,34266,37842,41066,34797,35542,41067,55788,46400,65130,33400,56929,56928,56902,47037,56927,56905,56906,47047,56936,47042,56926,39803,56899,47048,47038,56914,56904,56907,56931,47032,56938,56930,47041,56919,47052,47051,47045,56937,47033,56917,56908,56921,56933,47053,39804,47035,56916,39805,56909,47044,41068,47043,56912,56922,56932,56903,56913,47036,56923,47049,47040,56910,47039,56901,56915,56935,46334,47792,56918,57964,56920,56934,47046,56911,47034,47050,48368,56900,36542,56925,38224,34515,39806,56924,37737,34519,41069,36929,34518,41070,35540,35310,36219,35307,36050,58026,47789,57981,58020,47778,57966,47791,47735,57965,58032,47793,57969,58019,39841,57971,58035,58031,47733,47777,58963,47790,47741,57967,39842,58030,47779,58027,58040,57973,57982,39843,39844,58038,58028,47740,41071,57980,47734,47732,47784,39845,39846,57978,57975,57976,58034,58039,58037,47738,58041,47742,47783,41072,57968,58874,57977,47736,47788,47785,47739,58021,57972,47786,58023,47780,47782,47731,39847,58025,58017,57970,47781,58033,58036,57979,58024,39848,47737,48351,58022,58873,58029,64674,33360,36797,35708,35835,35312,57974,58948,58958,48354,58957,58969,48356,58955,39849,58959,48367,58950,48359,37049,58962,59888,48371,48370,58964,58947,58974,48365,48355,58967,58971,58976,58965,58953,48358,48361,48369,48364,58956,58018,41073,58952,58975,48360,39850,48363,58977,48352,58966,58875,58972,49375,38251,58954,48353,58949,48357,58876,47787,58945,39851,58970,58946,58944,48362,58968,41074,58878,58961,58960,58973,58951,48366,35834,39852,59891,39853,48969,48894,59968,59883,48961,59895,48968,48963,59893,60751,59899,59970,59898,59881,59896,59972,59974,48893,59973,48964,48970,41075,48967,59902,48966,59897,39854,59885,59890,39855,59901,48965,48962,48892,48960,59889,58877,59884,59887,59969,59892,59882,60750,59971,59886,59900,35709,35745,39856,60753,49379,49367,39857,64676,49371,60755,60761,60759,49369,49370,49377,60762,60754,49372,60758,60757,60763,49378,49373,49376,60756,49380,49374,49381,49368,60760,41076,60752,41077,38342,61431,39858,49777,61428,61430,49775,61426,61427,61422,39859,59894,61423,49776,61419,49773,61432,49774,61420,61421,61425,49779,49778,61424,50040,62047,62053,50041,62044,50038,50035,62055,50039,65185,50036,62046,62049,62050,62051,62054,61429,62045,50037,62052,62056,62048,37696,62557,50282,62560,50283,62568,62559,62556,62558,62562,62565,62564,62567,62555,50281,62563,62566,62569,62561,62931,62932,62936,62937,62934,62935,62933,39860,50409,39861,36982,39862,50552,63211,39863,63208,63209,63210,50553,63461,63460,39864,63663,50676,63661,63664,63662,63733,50775,50789,63907,63852,39865,63906,63952,63953,42229,40182,34265,35065,42695,51777,52062,36191,43103,39866,43106,39867,52063,43104,43105,33613,35314,34271,64677,52568,52570,52565,52562,52564,37876,43684,39868,65187,43682,39869,52566,43683,52563,52560,43681,52567,52561,43685,52569,36421,36470,53167,39870,53171,39871,44215,39872,39873,39874,53174,44207,44210,44212,44214,44211,53170,53169,44209,53172,53173,53166,44213,35515,44208,41078,34270,35063,53168,41079,35747,35962,36305,39875,53879,53880,53881,44880,44876,53870,39876,53878,53883,44881,39877,53868,53874,53867,53877,64599,39878,53873,44877,44879,53882,35315,53866,53869,53875,38319,53876,53884,53872,44878,34269,35064,36586,35316,35066,35544,34801,45677,54862,38217,54864,54860,40000,54872,54858,54871,45673,54856,55899,54866,45676,40011,54867,54870,54874,54863,40012,54868,45674,45675,54873,54861,54857,54875,41080,54865,54869,33348,34272,35067,54859,35545,46408,46409,55909,46415,40013,55897,55906,55896,46412,55904,55902,55903,46410,55907,40014,40015,40016,55900,55898,46411,55901,55905,40017,64679,40018,46413,55908,40019,37605,40020,65142,40021,56944,56951,56953,56993,64358,47066,56939,47058,38223,56954,47063,56994,47054,40022,56957,38239,56941,56958,56940,41082,47068,40397,56952,47055,56995,47060,56945,47065,56956,56943,56950,56946,56942,47057,47064,47062,47059,47067,47056,56949,40023,47061,46414,56955,65107,56947,41083,40024,64681,40025,56948,41084,58049,40026,47796,40027,58045,58051,58047,47798,58046,58050,58042,41176,58044,47797,40028,40029,58048,58043,40030,47799,47794,38059,40031,58052,41191,47795,58983,58980,58992,58986,58988,48372,58982,58990,40032,40033,58989,58987,58993,48375,58984,58991,48373,40034,58979,58981,48374,58978,58994,40035,58985,40036,59978,48977,40037,59989,59987,48971,59977,59980,59981,59976,48981,48982,59975,59990,59985,48975,48972,59984,59982,48978,59986,48973,40038,48974,59983,48976,59979,41192,59988,48979,59991,59992,48980,40039,49383,49390,60764,60770,60768,49386,49385,49382,60766,40040,49388,49387,49384,60769,60765,60767,49389,41205,64368,49783,61435,49780,49781,61437,49782,61434,40041,61433,62060,61436,62061,50042,62059,40042,62058,62057,50043,40043,40044,50284,62570,62571,64684,40045,40046,62940,62939,50410,62938,63212,63213,63462,63665,40047,40282,63734,63932,50809,63942,42230,37321,43686,43687,40049,44216,34246,64100,49391,42231,34273,43688,44882,47069,42232,64101,45678,47800,51554,38311,53175,53885,58053,49392,42233,43689,53176,53177,55910,46416,35748,56996,64685,47070,58054,38254,48376,40051,50044,42234,55911,42235,40052,42697,51778,42696,43109,43108,43107,52064,64102,43690,43691,52571,53178,53181,44218,53179,44217,53180,44219,33889,53922,53921,53886,44883,35963,54877,54878,45679,54876,54879,46418,45680,64103,46417,55915,55914,64686,55912,55913,35749,55916,56998,56997,57001,64104,57000,56999,47801,58057,40053,58056,47802,58055,58995,58996,48377,59993,59994,64105,37489,62066,50045,62065,62064,62062,62063,50411,62572,63214,63735,42236,64106,51621,42439,51622,64107,64688,51779,51780,33629,35546,64108,34038,52070,52066,64109,52065,43692,52069,43111,52067,43110,52071,52068,33729,52575,53182,52573,52580,64110,43693,43696,52581,52577,52578,52572,43695,52574,43694,52579,64111,52576,35318,53186,44221,44222,64689,53189,53183,53188,53184,44220,53187,53185,35068,33728,34274,36588,33890,34530,36423,53928,53925,53927,44888,44887,44885,53924,53929,44884,44886,53926,54887,53923,53930,64690,36587,40054,54882,54886,54885,55918,55929,35547,54888,40055,54883,55917,45684,36693,45683,54881,54884,45685,45682,45681,54880,54889,37862,36051,55920,55927,64112,46420,55926,55923,64692,46422,38333,38332,64687,55925,38226,64113,55919,55921,55924,55922,46421,55928,46419,47071,64114,64115,57005,57004,57002,47074,47073,57006,40056,57003,58058,47803,47072,64691,57008,57007,58061,58059,48378,64693,47804,58060,58998,48379,58997,59006,59005,59003,59002,58999,59000,59001,59004,59041,37598,59999,59996,59997,48983,59995,60001,60000,59998,35751,60772,60773,49393,64694,49394,60771,40057,49785,61438,49784,50046,50081,50285,62574,62573,62941,63215,50554,63464,63463,63465,42440,53190,44889,45686,54890,42441,51623,42237,40058,51781,40059,35069,52076,52074,52075,52072,43112,52073,40060,34532,40061,52589,40062,43699,52587,52583,52586,52582,43701,52585,40097,43698,43697,40098,43700,52588,52584,36424,35320,33891,44226,44229,53198,53197,53196,44223,53205,53195,35752,44225,53935,64116,53202,53200,44228,53192,53203,53194,53204,53201,53193,38334,44224,53206,53191,44227,64117,38326,37042,35837,53940,53931,53942,40099,53934,53945,53946,53932,53944,53941,53939,53943,44895,44893,40100,40101,53937,40102,53933,40103,53936,53947,53938,44894,53199,44890,44892,40104,64118,34276,40105,54904,54893,54891,65194,54892,40107,54899,40108,54900,54896,45691,54901,54898,54895,40109,45689,54894,45687,45690,54897,54905,44891,45688,54903,54902,45692,36543,34531,40110,40111,40112,40113,55934,40114,40115,55969,46432,40116,55975,40117,40118,55977,55970,46426,55974,55973,46427,46433,46434,55976,46424,55933,55931,55971,55930,46431,55932,55972,55978,46425,46430,46428,46429,40119,40120,40121,46423,35753,47081,57015,47080,57019,40122,57009,40123,57020,40124,40125,40126,57010,57011,57021,57018,57016,57017,57013,57012,40127,57022,47077,40128,57014,47082,47076,47083,47084,64697,47079,47078,65141,40307,58062,47806,47805,40129,58067,65196,48380,47807,40130,40131,47809,58068,47075,47808,58064,58066,58063,40132,58065,64119,35321,59051,40133,59050,59047,48448,60002,48449,59046,40134,48382,59048,59045,59042,59049,59043,59044,48381,33232,33233,35965,60777,60006,60005,60007,60774,48986,33234,60003,33235,48984,40308,48988,48987,60004,60008,64120,48985,33236,60781,49397,49786,49398,49395,60778,60776,60779,60782,49396,60780,60775,33237,61506,61509,62069,61504,33238,62575,61510,50082,61508,49787,61505,61507,61511,62070,62068,33239,50083,62067,33240,50286,33241,36089,33242,50413,63217,50412,63219,63216,63218,50640,63666,42442,52590,53948,53949,45693,57023,48989,50084,50555,63667,42443,38060,52591,41568,53207,53208,35070,38062,38063,38061,53950,53951,45694,45729,38064,55979,57026,57025,57024,58069,64121,58070,58071,47810,38065,59053,59052,60009,48990,48991,38338,60786,60783,60784,60785,61513,61512,49788,62071,62942,42444,64122,44230,45730,57027,64699,42445,53952,45731,64123,46435,46436,42446,42447,51782,43114,43113,44231,53209,55980,42448,42449,42450,42451,41049,34426,43115,43116,52078,52077,34040,64124,43702,52594,52592,52593,33243,64125,34512,64465,53210,53211,64126,38335,44235,44233,64161,44234,44232,37973,64700,44896,33244,44900,44899,53953,44898,44897,53954,37974,45734,54907,54906,45732,45733,38348,38066,46438,46437,55982,64162,33245,55981,45735,36322,37612,35754,47085,57029,47086,57028,64701,58072,59054,48450,60010,64163,64164,60787,50086,50085,64165,50556,42452,52595,33246,37713,45736,58073,47811,50900,52079,52080,36728,35323,52596,43704,43705,43703,33247,34533,35549,34534,44239,44240,44237,44238,53212,53213,44236,64166,34804,34535,53955,65105,44904,44905,64348,45739,53961,44910,44908,53962,53957,44907,44906,44901,53960,53959,53956,44909,64723,53958,44902,44903,36861,34041,45740,54945,54946,45741,54908,54910,54948,54947,54909,45737,45738,64167,55990,46443,46442,55984,46440,33248,55987,46444,55988,46445,55985,46439,46441,55989,55986,55983,35322,64168,34536,36696,57042,57031,47088,47091,47090,47095,47094,57043,57041,57034,57038,57037,47092,57040,57036,57044,57035,47093,47087,47089,64725,57033,36695,64724,58075,47815,58079,47814,58076,47813,57032,57039,58078,47816,58080,58077,58074,36545,35136,59057,59061,59063,59059,59058,59056,48453,48451,48456,48457,59060,48454,59055,48455,47812,59062,48452,35755,36180,33249,60012,60011,60019,60013,60018,60015,48992,60017,48993,48994,38313,60016,60014,36053,36952,64169,49400,60788,49399,60791,60789,60790,49401,64170,38255,33250,61517,33251,49825,61518,64171,49789,61519,49790,61516,61520,61514,35966,50087,62072,50088,50287,36054,61515,50288,64374,64727,64172,50414,62943,36181,50558,63220,50557,64728,63466,50677,50678,33252,33253,63948,50901,44241,53214,37817,46446,46447,42453,42698,51783,64173,52081,43117,64174,43706,44242,44243,44244,54950,53963,44911,45742,54949,64175,34805,55992,46449,55991,46448,57045,48458,59067,59064,59065,59066,64702,60792,33254,61521,62577,62576,63221,42454,52597,44912,34806,36471,46450,57046,64176,58081,64177,48459,60020,61522,62578,42455,43707,44247,53215,44248,44246,44245,53964,44913,64703,44914,44915,34278,36201,34538,45744,54951,45743,34633,34043,34807,33255,55993,45745,46451,57047,47096,47097,47817,47818,48460,48996,60021,48995,60793,49402,61523,62579,42456,43118,52600,52599,43708,52598,43709,52601,53221,44251,44250,53223,53222,44255,44254,44249,53217,53218,53219,44256,53216,44252,53220,44253,37553,64178,33256,33257,53967,53971,53969,53968,53972,53973,53974,53966,53965,64179,44917,44918,53975,53970,64704,54960,36306,53976,44919,44916,64705,54954,54953,54955,54956,54958,54957,54962,45749,45746,45750,54952,45751,54961,45748,54959,45747,38221,34280,64180,55996,55998,55994,55995,55999,56001,56002,55997,56000,46452,34279,35550,57051,57056,57048,57052,33258,57057,57053,47098,47171,47101,57049,57050,47822,47174,47102,47172,47100,57055,47173,57054,47169,47099,47170,57058,58086,58088,33259,33260,33261,47168,33262,33263,58083,47820,58089,47821,58087,58082,58085,58090,47819,58084,48462,59071,59070,48465,48463,59068,48461,59069,48464,35138,64363,60029,60065,33264,60030,60022,60026,60025,60023,48998,48999,48997,60024,60027,60028,49000,49472,60835,36220,49404,60795,49406,49473,33265,49405,60834,60796,49403,60833,60794,60798,60797,65114,61525,49828,49829,49826,33266,49827,61524,33267,62075,33268,50089,62073,62074,33269,62580,62583,62581,62582,62944,50415,63467,63668,50679,63736,63737,50790,42457,44257,33270,56003,57059,42458,43119,43710,33271,53224,53225,44920,35324,56004,46453,47175,49474,60836,62076,62584,42459,33272,35326,52641,52602,52604,52606,52605,52603,43711,35905,35838,36472,34809,44258,53234,53229,53226,38465,53233,44260,44261,53232,53231,53230,53227,53228,53235,44259,37573,35139,36323,64181,34044,64182,44924,33273,44964,44963,53985,53979,53977,44961,54969,44922,53982,53986,53988,53984,53978,44962,53983,53981,44921,53989,44965,53987,44925,53980,44926,44923,64183,35904,36930,64184,33274,35906,33375,35551,33275,45753,33276,54970,64185,54963,54965,54967,33277,54968,54966,45754,54971,33278,54964,64707,38475,34539,64186,36697,56008,46454,56016,56005,56017,56006,56007,38476,56015,56014,56011,45752,46455,56009,56012,46456,56013,56010,37972,38068,36200,57070,64187,57074,47182,58096,47185,57072,63958,57069,57064,57066,57067,57060,38477,47181,47180,35756,47176,57063,47183,47184,57062,57065,57073,47178,47179,57071,57061,35140,36589,58098,47824,58100,57068,58102,47828,58103,58099,47825,58095,47827,58092,58097,58101,58094,64711,47177,38384,58091,47826,58093,64188,38478,38605,64189,48468,59073,48472,48470,47823,59080,59081,48467,64190,38479,59079,59082,48469,48466,59075,59072,59077,59074,48473,59076,38480,64708,59078,48471,38481,64709,64191,49002,60072,60066,60070,60076,60077,60073,60074,60071,38484,60068,49004,49001,60067,60069,49003,60075,38485,49478,38486,38487,60842,60837,49477,38488,49475,60844,49476,60840,60841,60838,60845,61526,49479,60839,60846,60843,64424,64192,61530,61527,49830,61531,61533,61532,61528,61529,62115,50090,62078,62114,62077,62116,62113,62586,62589,62585,50289,62587,62588,62590,50290,50292,50291,62945,65143,62947,64193,62946,38489,38490,64194,63222,38491,63669,63738,42460,64195,35552,52082,43712,52643,43713,43714,52642,53240,53239,44262,44265,44264,44263,53236,53238,53237,53992,44967,53996,53995,53994,53990,44966,44970,44973,38492,38494,44974,53991,53993,44972,44971,44969,44968,54978,36798,54976,54972,45755,54973,45756,54974,54975,54977,45757,34540,56021,38495,56020,56019,56018,33392,34541,57078,47186,57075,57077,38496,47187,64196,47188,57076,38497,33399,37712,38498,38499,58177,58105,58106,47831,47829,47830,58179,38500,58178,58110,58109,58108,58107,58176,58104,38501,59083,59088,59086,64197,38502,59085,59084,59087,60078,38503,49005,49480,60848,64714,49481,60847,61535,61534,49831,62117,50091,62625,50593,63223,63671,63670,51624,44266,44267,54979,47190,42461,43122,43121,43120,64198,64425,52644,34045,43716,43715,40398,44270,64678,53242,53245,53243,44268,44269,64716,53241,53244,44981,54003,54005,54004,44978,53999,64199,38504,44976,44975,44979,44977,44980,54002,53997,53998,54001,54000,34232,34543,38505,38506,54982,54983,54981,54980,45758,35554,46461,56022,56024,56026,46460,64200,46458,56023,46459,56025,46457,64201,57153,57079,57082,57086,47194,57084,57083,57080,57081,47192,57152,47191,47196,47195,47193,57085,35143,58185,58184,38507,58180,64718,47832,58183,58182,47833,37798,35758,48478,59090,38508,48479,48475,48477,64719,48474,48476,37625,64202,60079,38509,49008,60081,60080,64720,58181,49010,49009,49006,49007,37459,64203,35553,60853,60851,49482,60852,60854,60850,60849,61536,49834,49832,49833,33383,38510,62118,62119,50093,50092,62627,62628,62626,63224,63225,42462,51784,43123,34811,52645,43718,43717,52646,64578,53312,44271,53246,44272,36090,44982,54008,54006,54012,44983,54007,54011,54009,54010,35556,54984,54986,34810,45759,37720,54985,45760,46498,46497,46462,56027,35555,34545,57156,47197,47198,57155,57154,35144,35907,58186,47835,47834,58187,58188,36590,48481,48480,65191,60085,59091,59093,59092,60084,60082,60086,60083,49011,64580,38511,60855,49483,60856,60857,38512,64204,49835,49836,50293,64997,38513,50641,42463,38069,38070,53313,64205,38072,38071,34919,54013,44984,64722,38514,37858,64721,35908,46010,46009,38073,64206,46500,56029,46499,56028,64207,38515,33389,57157,47836,58189,47837,38074,50294,62629,64208,42699,43719,52647,44274,38516,44273,53314,53315,38075,35145,54080,54082,44985,54084,54087,54085,34113,38517,54086,54083,54014,44986,54088,54081,34112,34284,54995,45766,55004,45763,54997,45767,64732,45761,54992,55005,54993,54990,45765,38518,45762,38519,54996,54999,45764,55000,45768,55001,54991,54998,55002,54994,54989,54987,64209,55003,65124,34546,56031,36055,36986,56036,38520,38521,56032,56038,46503,54988,56033,46501,56030,46508,56034,46507,56035,46509,46504,46510,46505,46506,38522,46502,56037,36699,35146,34547,47201,57168,64210,57171,57159,57164,57158,47203,57162,64211,57160,47202,38523,57167,57166,57163,57165,57161,47841,57170,47199,57169,35909,64212,64213,35759,35561,58205,47848,58200,33402,47847,58190,58192,47840,58197,58196,58199,47845,58194,58193,64733,47844,47839,58195,47842,58201,58203,64214,58198,58191,47843,64215,64216,48489,47838,64217,64734,58204,36698,64218,36953,36209,35559,35557,59097,48482,59099,48483,38524,48485,59102,59094,47846,59100,59096,47200,48488,48484,48486,48487,49014,59101,59095,48490,59098,64735,38608,35560,35392,60096,60091,60101,49012,60093,49016,60099,60090,60087,60102,49489,49017,60098,60088,49015,60092,49019,60089,60094,49018,60097,60100,38609,33441,35910,60875,60876,60860,60867,60865,64219,49487,60872,60095,38610,60863,64737,60873,49486,60862,60861,60871,60868,60870,60858,60874,49484,60869,60878,60866,49488,49485,60864,60859,60877,49013,64220,38076,64221,38611,36983,38077,61539,38612,61537,61543,49840,61541,61540,49842,61546,49841,61547,61544,49838,61545,61538,49839,49837,62123,61542,61548,62120,38613,50098,50096,62122,62124,62121,50097,50094,50095,50099,36056,64738,50296,38614,62634,62633,62631,62630,62632,50295,50297,64222,50416,64739,62949,62948,64223,63226,64224,63228,63230,63229,63227,64740,50595,50594,50643,50642,50644,63469,63468,63739,63672,63740,50776,38615,50777,63853,64741,50814,42700,36954,52648,53317,53318,53316,35805,33630,44275,64742,53319,53320,53321,37955,64744,54089,54095,64743,54093,44987,54091,37623,54092,54094,38616,38078,54090,45769,55006,45771,55008,45770,55007,37600,36987,36931,56040,46511,56042,56039,55009,34285,46512,65123,63963,56041,35393,64225,64745,38079,35562,57174,47204,57172,47205,57173,47206,38617,47849,58209,58206,58208,47850,47851,58207,38080,35395,64747,38618,59103,64746,59104,38322,48491,59106,59105,41044,41569,60106,60107,60103,64382,60104,49020,49021,60105,38081,49495,64749,49491,49496,49492,49494,49490,49493,38619,49843,60879,62126,62125,38620,62635,50298,50299,63297,62950,38621,63296,38622,63741,63908,42701,38623,64226,43124,38625,52649,43720,44278,53324,44276,53322,44281,44277,44282,44280,53323,44279,44991,44990,54106,44999,54099,54105,44995,54098,54104,54102,44994,44996,54101,44989,54100,45000,44997,45001,44998,54097,54096,54103,44992,44988,44993,38370,34554,55024,55017,46517,55016,45775,45782,45779,45785,45784,45780,55010,55013,55012,45776,55014,55023,45777,55011,55020,55021,45778,55018,45783,45773,45781,55015,45772,55019,64227,55022,64228,37547,64751,56059,56050,46514,56057,56054,56046,56055,46516,56047,56043,64752,47212,56052,65120,46513,56058,46520,46522,56045,46521,56048,46515,56056,56049,56053,56051,46518,56044,46523,45774,46519,46524,37467,37702,64229,47208,57181,57183,57185,57189,57179,57177,47210,57184,57188,57180,57176,57175,38248,38626,57186,57178,57182,47211,47209,57190,47207,57187,40377,58226,38627,37550,37732,38082,47854,58218,48504,58228,47857,58232,47863,58213,38628,58229,58210,38629,58231,58214,38230,47870,47867,58230,58224,47853,47861,47860,38250,47859,47865,58211,47866,58225,47862,47852,58227,47855,47856,47864,58216,58215,58212,58220,58217,58221,47869,58233,47858,58222,58223,38630,58219,37724,38631,37863,47868,38632,64230,64231,64232,59111,48496,48505,48501,59108,38339,48498,48502,59120,48492,59112,48500,64233,38633,59115,59110,48499,48503,59109,48497,59119,48494,59118,59117,48506,58738,48493,64753,59116,59107,48507,59114,48495,38634,40830,37453,59113,37613,38083,40865,49058,49063,49022,60120,60111,60123,60115,60121,49064,49057,60108,60114,60124,60117,60122,60110,60118,49059,60116,49062,49061,60112,60113,60109,60119,49060,60126,60125,37599,64754,60890,60886,49503,60880,49497,49513,60892,49505,49501,60883,49508,49511,60894,49500,60885,49509,60896,60893,60881,49504,49498,49512,60888,49507,60882,49502,60895,49506,49499,60889,49510,60887,37711,33370,60891,64234,61550,61556,49849,61559,49844,49845,61551,61558,61553,49850,49847,40866,61549,64235,49846,61555,61557,49848,61554,61552,40867,35408,38084,62136,50103,50104,50100,40868,50101,36072,62132,62130,62134,50106,62135,62128,62127,62131,62129,50102,62133,62636,50302,50301,62637,64236,62639,62638,50337,38085,62955,62952,62953,64237,62951,62954,50418,62956,64238,50417,38086,63298,64239,50645,50647,63470,50646,63673,63808,63810,63742,63809,50796,40171,33631,33654,33649,33648,35406,33652,33653,33739,36286,36932,33650,33651,34814,33740,33736,33738,33734,34556,33741,35404,34625,33743,33742,33735,33733,33737,36293,34885,36307,33905,33906,34813,33903,33902,33901,35563,33904,35155,33896,33899,33900,33898,35761,34286,36308,34288,34116,34115,64240,35565,34122,36324,35763,34120,34121,34123,34119,34117,34114,33897,36426,36427,34287,35567,34289,34290,35913,36425,35397,34627,34552,36002,34557,36475,34551,64241,36474,35762,34626,34624,34629,35401,34558,36476,36473,34550,34548,34549,34553,34555,34628,34889,36546,34880,34881,36259,34891,34890,34882,36729,36548,34888,34886,36549,36550,64242,34887,34812,35399,34883,36547,34884,35154,35150,36592,35153,35402,35152,36001,36701,35147,35148,35912,35407,35409,35400,36091,35396,35398,35405,36933,35566,35564,35568,64243,36830,36799,36800,35765,36988,42702,40869,44283,53871,45002,45786,56060,56061,40870,40871,35914,60127,49514,60897,38260,49851,40872,62138,62137,50338,62957,64426,63299,50680,51785,40873,40309,43721,43125,35764,53325,34630,54112,54107,54111,54109,45003,54110,54108,34893,55025,56062,56128,57193,57194,47214,47215,57192,57195,57191,47213,47936,64244,47216,58234,36801,48508,59121,48509,38244,49065,60130,60128,60129,60900,60899,60898,36182,64245,62139,50105,62140,63300,50681,63674,42703,43723,43722,53327,44284,40874,53326,54114,40875,45004,55026,54113,38368,45788,55029,55027,55028,45787,36428,56130,56131,56129,38229,47219,57197,57196,57198,47218,47217,59122,59124,64246,48510,59123,60131,49066,61561,40876,61560,50107,62141,50109,50108,62640,62958,50419,42704,53328,44285,54117,45006,54116,54115,45005,64247,55035,55037,55030,55031,45789,55032,45790,55036,55033,55034,45791,46526,46527,56132,35569,57199,57200,40877,58238,47939,47937,47938,58235,58236,58237,59129,40878,59130,48545,59127,59126,59128,59125,49069,60132,49067,49068,60902,49515,60901,61352,61562,61563,49852,40879,49853,49516,62142,62143,62641,50339,42705,40880,42706,44286,43724,45007,53329,40881,46528,42707,44353,53330,53331,44352,44354,42708,53332,45009,54118,45011,45008,45010,55105,45792,64758,55104,55038,64248,57201,64249,58273,64250,48546,36702,49070,60134,60133,60903,40882,40883,62959,64251,42709,52083,52650,44355,53333,35413,54120,38236,40884,45012,54119,45013,35571,55107,45794,55106,55108,45793,36057,35570,36831,56134,56135,56133,46529,35412,47220,47221,47941,64762,58275,58274,47940,64252,33387,64759,64760,59131,64761,59132,64253,36730,35411,60135,49520,49519,49517,49518,49521,37751,61564,49855,49854,62144,62642,50597,50596,42710,37333,53755,34124,47223,46530,47222,47942,42711,51625,42712,42713,52651,52086,52087,43127,52084,43126,43129,52085,43131,43130,52088,43128,38575,34294,43729,43727,52653,34895,43726,64763,40885,43731,43733,43730,64254,52656,52652,43734,43728,43132,43732,52655,64333,64320,52654,43725,38603,38566,34125,34296,34126,33744,53339,44359,44360,53341,35415,53335,53338,53347,53345,44361,53351,44364,53348,53340,53337,56137,53346,44356,53349,53334,53343,44358,44363,53344,44367,44365,53336,44362,53342,44366,44357,53350,35832,65019,65020,35915,36309,65021,45018,45027,45016,45014,54122,45022,45019,54124,34894,45021,54123,54121,54126,45026,45024,56136,54127,54125,45015,64764,45017,45020,65096,45023,33745,45025,40886,35417,65022,36058,35572,34297,65262,35414,36989,55118,45796,38312,55109,55111,34295,55112,55120,55116,55114,55117,55121,45797,45801,55110,55119,45799,40887,45798,55115,55113,45795,45800,65263,38565,37808,40888,65264,41026,35416,46536,56145,56143,46538,40889,56138,57249,46537,56142,38571,56139,46533,46539,56144,46535,56141,47943,46534,56140,46540,46532,46531,40890,38574,38562,57207,57205,57211,65265,57203,57250,57208,57202,47227,47267,57213,57206,47230,40400,65266,47228,57214,47225,47224,57209,47229,46541,57212,57204,47226,47265,47266,35156,35246,65267,35941,47948,47944,47949,58278,40891,58277,58279,47946,58276,47947,58282,58281,58280,47945,35573,65268,59201,59204,48552,59203,48551,48547,48548,48549,59200,59134,48550,59202,59133,38087,40892,60137,60147,49073,49072,65269,60141,60143,60138,60142,60136,60145,49071,60144,60140,64366,60146,65270,60139,49524,60904,60910,49528,49530,49527,49526,65271,49525,49523,60905,60908,49522,60909,64766,49529,60907,60906,49856,49857,61601,61565,61566,37730,62146,62145,50110,62644,50340,62643,62960,63301,50598,63811,63812,50648,42714,40893,43735,56146,47950,49531,60911,42715,45029,45028,56147,38355,60148,42716,44368,64834,56148,56149,56150,47951,49074,42717,43736,53352,45030,54128,45802,56151,47268,35447,47952,49075,49532,49858,62645,42718,43737,37590,45031,55122,46542,47953,58283,59205,37830,38380,64835,42719,46543,57251,47954,42720,52657,53353,44369,36802,54130,64339,45034,65272,45032,45033,45035,65273,65274,54129,35916,35158,55127,55124,55126,45803,45805,45804,55123,45806,55125,36955,56152,56153,65275,56154,57254,57255,57253,57256,47269,57252,47955,59210,59206,59209,59211,59208,59207,40894,60149,60150,60151,49076,49077,60913,60912,60914,38088,61603,61602,62148,65276,62149,62147,40895,50341,62646,62647,65277,63302,63471,63675,42721,43133,35420,49533,42722,36187,55128,56155,33349,50753,51786,40146,35767,40896,51787,51789,42723,51790,51788,34283,52130,52131,52091,40897,65278,52129,43169,43170,52092,52090,52089,52093,43134,52094,53354,36598,52662,43740,52661,52663,35161,43739,52668,43743,52658,52672,52678,43750,52675,43747,35919,52665,52671,52673,64421,52660,43746,43741,52666,43748,43751,43745,43738,52670,52664,52677,43753,43749,43744,52669,45036,52667,43742,43752,40898,52659,36554,52674,52676,38590,37184,34299,36833,37185,33913,34129,34635,33656,34039,64863,44386,44380,44388,44385,53361,53364,44381,37186,53355,44374,44384,44387,44389,53410,53367,40899,44373,53409,44377,44375,44370,53359,53374,53363,53366,53413,40900,44390,53373,44382,53368,53412,53365,53369,53372,38563,40901,53357,53411,53371,40902,64861,53356,40903,53360,44383,44378,44371,44376,44372,44391,53358,54181,44379,40904,53370,52801,37543,36832,36553,34243,36422,54184,45050,37187,54134,37188,54179,54141,40905,54194,54186,54142,37189,54185,54136,54140,54197,45053,54189,54180,45037,54195,54132,54188,45052,45047,54131,45045,45044,45049,54187,45041,45048,53362,56156,54182,40906,54138,45051,54139,54177,45054,54133,54191,37190,54190,54198,45043,45040,54196,54192,54183,54178,45046,45042,54135,45038,54193,45039,54137,40907,37716,33909,37191,36551,36596,36703,35577,35159,37192,34513,34298,36467,34529,34527,36594,36731,34634,34897,37193,36429,37194,35769,55134,55136,55141,55161,45820,45810,55133,45821,45822,55144,55151,55157,55138,55145,34632,45888,55159,55154,45818,45816,55150,55146,55132,45807,55137,55129,40908,45815,45817,55142,55139,45812,55155,45809,55140,55162,55148,37195,55147,45808,37196,45819,45811,55130,55135,55152,55158,45889,55131,55143,55149,45814,40909,55160,55153,55156,34128,34896,36477,34301,37197,36704,35319,36597,34803,36034,37198,45813,64429,56172,56160,46551,56189,56231,56234,46549,56168,56227,56169,56183,46562,56179,46559,40910,56180,56157,56228,65106,46568,56225,56181,56236,56176,57288,56239,46566,56174,56186,46569,46548,56178,56237,56171,56164,56175,56163,56161,46544,56229,56170,56232,40911,56233,46552,46557,46553,46561,56190,46554,56182,56166,37199,46546,56158,56226,56235,56165,46560,56240,56177,56173,46545,46565,40912,56188,46567,56184,46556,46550,46558,46547,46564,56185,56167,56187,56162,56230,40913,40914,38567,56238,38589,36093,64867,40915,38089,56159,35163,35313,35149,35160,35575,57287,37200,57309,47189,57292,40916,57290,57269,47273,57285,57305,57281,47281,57304,57279,46563,57295,57280,57302,47280,47272,40917,57258,57266,57291,57283,57308,57286,47286,57303,37201,47277,40918,57289,57297,57270,57296,57313,57265,57298,40919,57311,40920,57259,46555,40921,57273,57272,47279,37202,57276,57278,57293,57310,47282,37203,47283,40922,57264,47275,57268,57306,47284,47276,47278,47285,57312,57299,57294,38389,40923,57275,57274,47274,57260,47271,57284,57261,57282,38592,57271,57307,40924,40925,47270,37204,57267,36593,37205,36595,34898,36552,57263,57301,57262,47968,58323,40926,40927,58306,40928,58284,58314,47960,58299,58309,47963,58302,47961,58287,58317,58286,58305,40929,58285,58303,58312,58310,58298,58293,58291,64870,58292,58311,58322,58300,47962,58295,58315,47965,58294,58288,58304,47969,38588,40930,47957,47966,58296,58290,47959,57300,47958,58307,47956,47971,47964,58308,58297,58289,58316,58301,47970,58320,47967,58319,58313,58318,38090,64869,58321,37206,35576,40931,35422,36972,40932,40933,37207,40934,36778,36029,59251,59252,59239,59238,59234,48564,48556,59254,59253,57257,59231,59235,59229,59248,59233,40935,59255,59226,59224,59236,59246,59241,48566,59215,59245,40936,40937,48567,57277,59227,59218,59221,59259,59228,59219,59217,59214,40938,48560,59237,48559,48563,59232,59240,48553,59256,59260,48555,59223,59243,59247,59220,59257,48562,48561,59212,48565,59250,59222,59242,59216,59230,59225,48557,48558,59244,59261,59258,59249,40939,65100,37728,35421,37208,36934,35918,59213,35917,48554,60233,60224,60227,40941,49083,60229,60153,60225,60231,49080,49084,49078,64872,40942,60155,60236,60230,40943,60156,60245,60239,60152,60998,60158,49079,37209,60234,60244,49087,40944,60241,60157,60228,60232,60226,60246,60243,60240,49081,49082,49086,60154,60247,49085,60237,40945,40946,60235,36834,40947,35921,60238,61011,60992,60997,61010,60996,60923,60993,49570,60916,61005,61007,60915,49569,61009,61001,49576,61008,60994,49578,60921,60242,61002,60999,60917,61013,49572,40310,49573,60919,61000,37210,61012,61003,60925,49575,49571,61004,60926,61014,60920,60995,61006,60922,60924,49867,60918,49577,49860,49534,37211,37846,33406,49574,49864,61619,61609,61604,61610,61620,61624,61623,49866,49865,61611,61625,61614,61606,37212,61608,61607,61613,61618,61605,61612,61617,49863,61615,49861,61616,49859,49862,62165,61621,36004,50114,40948,62157,62161,62153,62156,62164,50112,62169,62162,40949,62154,62170,62163,50115,50116,62167,62155,50111,50113,62150,62158,62152,40950,62168,62166,62151,62159,62654,50117,62160,50343,50345,50342,40951,62659,62651,62649,62653,62650,64874,62655,62657,50346,50348,62656,50349,50347,62658,40952,40953,64875,50344,37213,40954,37214,50420,62961,62967,50422,62652,62966,62973,62964,62971,62970,62648,62965,61622,62974,62963,62968,40955,62972,62962,63306,50421,62969,64430,38091,63476,63307,63305,63303,63304,63308,50649,63474,63472,63477,63475,63478,50650,63473,40956,63676,40957,63813,63814,63815,63943,63933,51791,43754,34368,44392,34763,54200,54199,45120,45890,55164,55163,34130,46570,47288,47287,47289,58324,59262,60248,60250,60249,49579,61015,61626,63909,42724,38264,52681,52682,52680,52679,43755,40958,53417,53415,53414,44393,44395,44394,53416,35503,34899,36804,34636,35770,35162,54212,54209,54207,45121,54210,45126,54204,54219,54221,54205,65211,45123,54222,54217,54203,54208,54218,54214,54211,45128,54220,54206,65212,54215,54201,45127,45124,54213,54216,54202,45125,45122,38092,34371,65213,45900,55205,45899,65214,55208,55211,45896,45894,55166,55209,55207,55204,55212,55213,55215,55216,55165,45893,55202,55201,55214,45895,55203,45897,45892,55206,45901,36094,45898,55210,37548,33356,36803,46577,56255,56244,46574,57319,56253,56241,46572,56246,46575,56250,56248,46578,46571,56242,56245,46576,56243,56254,56252,56247,56249,56251,46573,36210,38093,36705,36430,35578,36599,36835,57320,57326,57316,57322,47290,57318,47296,47295,47294,57325,47297,47298,57315,57328,47299,47293,47292,57324,47300,57314,57317,57327,57323,58356,58345,47291,37216,34900,47978,58333,58354,58334,47973,58331,58340,58332,47975,58326,58353,47976,58350,58351,58327,47981,58342,58336,58343,58330,58355,58347,58341,58325,47977,58348,47980,58352,58346,47974,58344,58338,47972,58329,58337,58349,58335,58339,35771,35425,35579,33442,48577,57321,59314,59323,59313,59309,59306,48578,59304,47979,59297,48576,59303,48575,59308,59305,59321,59316,59310,59315,48571,59307,59326,59298,59299,59322,48572,59327,48574,59328,59312,58328,59318,59311,59320,59317,37217,64837,59302,48569,59325,48570,59300,48573,60260,59319,59324,35424,37218,36836,33397,60257,48568,49088,60267,60263,64364,60261,60256,60271,37219,64839,49092,60252,60264,60265,60255,60254,60268,60258,60253,60259,60270,60251,60269,60266,49090,49089,49091,60262,61643,65215,37220,64838,61017,49585,61021,61018,61025,61031,61020,65216,61040,49582,61034,61023,61035,61030,61037,61022,49587,49586,61024,61038,61016,61036,49580,64841,61028,61027,61032,61019,49584,37221,49588,61026,61033,49589,61029,38599,37222,49581,49583,61639,61637,64844,61644,61641,61645,65221,61630,61638,61649,61039,61634,49871,59301,61629,61642,61636,61633,61628,61627,61648,61632,61631,49869,61640,64427,49868,49870,61635,61647,64845,62174,62175,50121,62172,50118,62180,50122,62182,62171,61646,62184,62173,64370,50119,62179,37223,62181,62176,62183,62178,62177,50120,37224,64846,62661,62662,62664,50350,50351,62665,62663,64847,62660,63042,63045,63041,50426,63043,50425,50424,50423,63044,63313,63311,37225,63310,63040,63312,63046,63309,64848,63481,63447,63479,50651,63480,63482,63679,50682,63678,63677,50683,50778,63854,63911,63910,63912,42725,53418,64849,54223,54224,36213,37595,56256,63047,63680,42726,44396,53419,65222,55217,45902,33781,56258,56257,46579,47301,59329,48579,48580,64859,65223,49093,50684,42727,37334,34372,53420,43757,53422,53421,44397,36060,54225,33916,54232,45129,54230,54228,54235,54226,54227,45130,45134,54236,45133,54234,54231,54229,45131,45132,54233,35773,35027,65224,45904,55218,45909,55234,45908,55236,65225,65226,55224,45906,55235,55219,45907,55231,55227,55229,55223,55230,37050,45903,55226,64854,55225,55221,55232,55228,55220,55222,45905,55233,36059,37226,38094,46582,56269,38365,37228,56265,56267,56262,56261,56259,56266,56268,56264,37229,56263,46580,46581,38095,38096,34373,35772,64355,56271,47309,57330,57336,57331,57332,63960,57337,47311,47303,47310,57329,56260,47306,47304,57335,57334,47305,47307,57333,47302,47308,36706,36805,35164,37230,65227,58358,47988,58434,58433,58363,47990,58432,58359,58360,47982,47984,37231,58365,58357,47986,47985,58361,58366,58364,47987,58362,65228,65229,56270,47983,38097,59330,59337,48582,59341,48586,59333,59331,59340,48581,59339,48583,48584,59332,48585,59338,59334,59335,59336,47989,35427,37232,37233,60272,60284,49098,60279,60281,65230,49096,60273,60277,60280,49094,49097,60283,60275,60276,60282,60274,60278,49095,61042,35923,61041,49591,61047,49593,49590,61043,49594,61044,61045,61048,64856,49592,61654,65231,61657,61651,61653,64857,61652,61655,61656,61046,61650,65232,50125,62188,62191,62193,62186,62187,62190,62192,50126,50124,50123,62189,62185,62666,50352,65233,62667,63049,50427,63051,50428,63048,63050,50600,63314,50599,63485,63484,63483,37234,63816,63817,63819,63818,51792,42728,65234,44398,55237,46583,57338,49872,38390,62194,64877,64878,43171,45911,65235,45910,64879,56272,46584,56274,56273,37472,57339,47312,58435,58438,58437,58436,59342,59344,59343,64431,49100,37857,38341,49099,49595,61049,61051,61050,49873,37485,62196,62195,62668,50353,64881,50429,63316,63315,50779,33614,36211,33918,33917,34131,36326,36006,35924,35428,36478,36956,35166,36707,35580,36806,43172,53423,44399,55240,55238,55239,56276,56277,57411,56275,35581,57340,57409,57408,57410,47313,57342,57341,57412,37235,58441,58439,58440,59347,59345,36807,59346,60285,61052,61053,49874,62197,62669,50354,63052,63317,50601,63486,63820,43173,44401,44402,53424,53425,44400,45140,45138,45137,45144,45136,45145,54237,45142,45139,45143,45141,45135,45919,65236,45913,55244,45918,45920,45914,45915,55242,65237,45912,65238,55243,45917,55241,45916,37845,46660,46662,37236,65239,56280,46661,46585,46589,65240,47332,57417,56282,46590,56285,56286,46659,56288,64883,56290,56291,56279,56278,56292,46658,56289,56287,46656,46587,46663,56283,56284,56281,46657,46588,46586,57416,47327,47322,47317,64730,47333,47318,47314,47329,47326,47328,47319,47324,47315,47316,57424,57421,57413,57418,47330,57425,47331,47321,64882,57415,57423,57419,57422,57420,47325,57414,47320,36009,35436,58444,47992,47995,58446,65242,48037,58445,47997,65243,48591,58447,48036,58443,48038,65244,47993,47323,47996,47994,47998,48034,47991,48039,48035,65245,48033,58442,37237,37803,65246,48598,48594,65247,48601,59350,48602,59362,59355,48587,59363,59357,48597,59358,48596,59361,48590,59359,59349,48589,60330,48595,48592,33361,48600,59348,59352,48588,59351,59353,59354,48599,59356,59360,59364,37238,48603,49106,60325,60331,60328,60286,60332,60321,60327,49101,49107,60333,65248,49103,49113,49108,60335,60329,49104,60322,49114,60323,60324,49115,49112,48593,40401,49102,60336,49116,49109,60334,49105,49110,49111,49603,61092,61101,61098,61100,49600,61093,64884,61099,49596,61095,49604,61091,61096,61103,60326,61097,61090,49597,61089,49598,61104,49599,61102,49602,61054,65249,49601,37852,61094,61660,61674,61669,61671,61659,49875,61658,49878,49877,38593,61673,61665,61662,61668,61661,61663,61672,61670,49876,61677,61675,61666,61676,61667,65251,62201,50127,62273,64885,63055,50134,61664,62199,50130,62200,62205,64887,65252,50132,50133,62198,62272,62274,62202,62204,62206,62203,62275,50129,50135,50131,37239,50128,62672,65253,50359,62670,65254,62674,62675,50357,62676,62673,65255,62671,50360,50356,62677,50358,50355,37240,37616,50430,50496,63054,63053,63056,63057,50497,63318,63323,50602,64886,63320,33379,63319,63322,63321,63555,50652,63554,63552,63553,38392,50686,50685,63681,63682,50752,63821,63822,50791,65256,50797,63913,63944,40172,33657,33658,33659,34640,35778,33748,33750,33749,36184,33751,33754,33753,35926,33752,33747,35432,35170,36734,36327,33954,33956,33957,33960,34914,33959,35178,33958,33953,33955,34134,34135,34142,34132,34137,35775,34138,34140,34141,34139,36709,34133,36328,34136,35927,36329,36431,36434,34377,36432,34378,34379,34380,34382,34384,36435,34383,34906,34388,34387,34385,34386,34381,34915,34375,34376,34374,36600,36433,34641,34639,34638,34648,34642,34644,34646,34645,34647,34643,34637,36513,34907,34908,36555,35175,36061,36556,34917,34909,34910,34918,34913,36557,34912,34911,34903,34905,34916,34904,34902,35171,35181,35172,35433,35169,35173,36601,35176,35177,36602,35179,35182,36185,35174,35180,35168,35167,35429,36603,35582,36260,36708,35435,35434,35430,36710,35431,35437,35648,36769,35650,37241,35649,35776,36808,35779,36984,35774,35777,37030,36007,35928,36186,43174,55245,37242,55246,57426,58448,59365,49606,49605,61678,62276,63556,43175,54238,45146,45921,57428,57427,48604,59366,48605,61105,49879,65257,50806,43176,52683,54239,45922,55247,55248,56293,46664,47334,57430,57429,57431,58449,58450,48040,49117,48606,49118,61109,61106,61108,61107,49607,61679,62278,62277,52132,45148,45147,54240,37243,55249,56295,56294,46665,57433,57434,57432,47336,47335,48042,48041,37244,59367,60339,60337,60338,49119,61111,61110,61682,61681,61680,62279,65258,63914,43177,44403,65259,44404,45149,45150,54242,54241,55250,45928,45926,45923,45927,45925,45924,64889,37245,46666,56298,47341,46668,46673,56300,46675,46674,46677,56299,56296,46671,46667,46669,56297,46676,46672,46670,47343,47342,47340,47344,47338,47339,47337,57435,38240,33378,58452,48044,48045,48043,65260,58451,65261,58453,59370,59372,37582,48615,59373,48608,59369,48607,48617,48613,48614,48610,59368,48609,59374,59371,40512,48616,64890,48611,48612,60341,60343,60342,40513,60344,49120,60340,64891,49611,61112,49608,49612,49610,49609,61683,61686,61685,40514,61684,49880,62280,62281,50136,62282,50137,33367,64892,50362,50361,63058,40515,37246,50498,63059,63324,50603,50604,38262,63557,50754,40516,33616,33660,33661,33757,33756,33962,34926,33872,34924,33964,34929,33961,33965,33966,34152,35440,34143,33963,34145,34923,34147,36331,34144,34150,36330,34154,34153,34148,34149,34146,34389,36771,34391,34392,34393,34394,34395,36010,34390,34396,36935,36514,34650,37281,34651,34922,34925,37282,36062,34921,34928,34930,34920,36558,35183,35651,35441,35438,35439,36770,36063,35781,35780,35930,35929,36188,43178,45930,45929,57436,57437,48046,60345,48618,60346,61113,43179,37283,53426,44406,44405,34652,54243,45151,54244,55253,40517,55252,55251,56302,46680,64893,56301,46679,56303,46678,35442,57439,57442,57440,57441,57445,57438,57446,57443,57444,48048,58454,48047,59378,59376,48619,59375,59377,48620,60347,64894,60348,49613,40518,62284,62286,62283,62285,62678,63060,37027,63855,43180,44407,54245,54247,54246,35931,55256,45932,55254,45931,55257,40519,55258,55255,36012,36711,56315,46688,56307,56313,46683,46686,56306,46681,56310,57452,46685,56305,64929,56311,56308,56314,56304,56312,46684,46687,56309,46682,36958,47346,57448,47345,57455,57454,47352,47353,57456,47347,57453,47351,57458,57449,57451,47348,57447,57450,57457,47349,57459,34931,35782,35654,36957,47350,36013,48049,58459,58465,58457,58466,40520,58456,58461,58467,58464,58463,58462,37284,58455,58460,36011,58458,40521,48625,48622,59387,59457,59459,59456,59384,59386,59461,59458,59388,59462,59385,59460,48623,48629,48627,59379,48628,48624,59380,59382,59381,59389,59390,64930,48626,38231,48621,36065,64365,59383,36064,60358,49122,60349,49123,49126,60354,60351,49125,37285,60355,60356,60350,60359,60352,60357,49124,49121,60353,40522,61119,49616,49614,49617,49615,61118,61115,61114,64931,61117,36212,35652,61116,61765,49886,61691,61690,49881,61761,61760,61687,61763,61692,49885,61689,61762,61688,49882,49884,61693,49883,61694,40523,61764,62290,50142,62287,62291,37286,40524,50139,62289,50144,64933,50141,64932,62288,50143,62292,50138,40525,36990,40526,50364,50366,62681,50365,62679,50140,62680,50363,50499,50501,63062,50500,63061,36261,63329,50605,63328,50606,63326,63325,63330,63331,63558,37287,63327,37028,63686,63683,63684,63685,50780,63825,63824,63823,63856,64934,63934,63915,50798,43181,45152,38367,33353,37288,37289,47354,40527,40528,48630,40529,60360,38099,49887,62293,40530,37290,37291,38100,63916,43182,43758,44409,44408,37292,45155,54248,45153,54249,45154,55263,55259,64417,45933,55262,55261,55260,45934,55264,55265,37757,37293,40531,56387,56385,56389,56390,56396,56392,56394,40532,56386,56316,56393,40533,56395,56388,56391,56317,46690,56384,56318,46689,46691,37549,47357,57461,57463,57462,57467,47355,57464,57460,57465,57466,47356,47358,57468,37294,58471,58470,58468,58469,48051,48053,48050,48052,59469,59470,59465,64935,59466,48632,48637,48631,48638,48633,59467,59468,59464,48704,48635,48634,48636,59463,38256,60362,49128,40534,60364,49130,60367,60363,60361,60366,49129,60365,49127,49619,49622,61121,40535,49620,61120,49618,49621,61766,61767,61768,49888,40536,61769,38101,49889,50146,62296,62297,62295,62294,62298,50145,62685,62683,62684,62686,62682,62687,63064,64937,63065,63063,50502,63332,50607,63333,63560,63559,33617,33633,33697,33758,33759,35656,33967,34937,33968,36014,37295,34155,36974,34156,36333,34157,37296,36334,36959,36332,34656,34399,34398,34400,35784,36438,36437,34397,34655,34654,34933,34936,34935,34932,34940,34939,34934,34938,35187,35184,35185,35186,36066,35446,35443,35444,36772,35657,35783,43183,46692,35785,47424,64833,48054,40537,49132,37297,49131,36015,35911,50147,50300,50503,43184,45156,47425,40538,62299,50902,40160,38102,40539,35786,37827,37966,37956,35191,34403,52134,34942,43185,43188,43187,43186,34405,35449,52133,40540,52685,41124,52687,43759,40541,34402,43761,52684,52686,43760,52689,52688,52690,35448,33969,38103,34158,34657,34404,33970,35190,53430,53428,44412,53427,44451,44414,44411,44452,35658,44413,44450,44449,53429,37298,44410,33971,45162,54251,54257,45159,45166,38361,45161,54254,54256,45164,54250,54253,45160,45157,54252,45163,54255,45165,45158,40542,35189,34658,55267,55270,45936,37299,45946,45942,55268,34659,37754,45950,45943,45948,45938,45935,45937,45949,55269,45941,45944,45940,45945,55271,45947,45939,55266,36161,38104,56397,46693,56399,46695,46697,40543,56398,46694,46698,40544,46696,37300,36162,47431,57507,47439,57470,47440,47429,57505,37746,47434,57506,47427,47426,64950,47437,47436,47435,47433,47438,57469,47428,47430,47432,64949,35192,48056,48059,48063,48057,48062,48060,48055,65121,48061,48058,40545,37301,59474,48707,48705,59475,40546,48708,48706,59473,59472,49136,59471,49134,49133,60368,48709,49135,60369,49138,60370,49137,49624,61123,49623,49628,49626,49627,49891,49625,61122,60371,49890,49892,37302,50148,50149,62688,50654,50653,43190,34979,51797,45167,51794,51795,51793,51796,36773,52138,52135,52140,52136,43191,43194,52137,43193,52139,38208,43192,36439,34159,52693,52695,43764,52691,52694,52692,43762,43765,43763,34406,35193,34977,53432,53436,53433,44455,44456,64939,53435,53437,53439,44453,53438,44454,36294,36604,34978,36336,55278,53434,54258,54267,54265,54260,54261,54266,54268,45169,37303,54263,54259,45168,45170,54262,54269,54264,36067,37051,45985,55281,55273,55279,55280,45986,55272,55274,53431,55276,55277,55275,46700,37499,36559,56406,60372,56407,56404,45987,46702,56403,56409,56408,46699,56412,56402,56411,56400,56410,56405,46701,57514,57509,57515,57510,57508,57511,47441,57513,37304,57512,47442,48065,48064,58478,58481,58473,58477,48066,58476,58474,58480,58475,58472,58479,59481,48712,61770,59478,59479,59477,56401,48711,59482,59476,48710,48713,59480,60373,49139,60374,60375,61124,49629,61771,61772,61773,62301,62300,62690,62689,63067,63068,63066,63334,50608,43195,44458,44457,45173,45172,54336,54337,54270,64940,45171,55285,37624,55286,55282,45988,55283,64943,55284,64942,35450,38568,56415,56417,56413,56416,46703,56414,46704,64944,56691,47445,47444,47443,57516,57517,38569,35660,58483,58485,48070,48067,38561,48069,48068,58484,58482,64945,36262,37029,36189,59489,59486,59487,48717,59488,59483,59484,48714,48715,59485,48716,38570,60379,64946,60380,60377,60378,49140,60376,33401,61128,61125,61127,49632,61131,49631,61129,61132,61130,61126,49630,33382,61775,61776,61774,40547,61778,49893,49894,62303,50151,61777,62302,50150,62693,62694,50367,62692,40548,62691,37617,63069,50504,64947,63561,63688,63687,50755,50781,63689,63857,40549,50799,43196,43766,37592,47446,35932,50368,43197,44459,45989,46705,49895,43767,53441,53440,54338,37305,45176,45174,45178,54340,37306,45177,45175,37307,38473,37308,54339,45992,55292,37309,45991,45993,55362,45995,55294,55360,55287,45994,55363,55289,55290,55288,45990,37310,55361,55291,55293,40550,37311,56429,64951,56428,56426,56418,56433,56421,56431,56438,56430,46713,40551,46709,56419,56425,46711,38220,56424,46712,46714,56427,40552,46706,46707,56439,56437,56436,56422,56434,46710,40553,46708,56435,56420,56423,56432,40554,37312,64351,40555,37313,58554,57527,57520,57539,57548,57523,47457,40556,57536,47447,47449,47461,57521,40557,47450,47452,47462,47451,40558,37755,47460,57529,57518,47458,57528,47454,57546,47459,57544,57532,57542,47456,57519,57545,57540,38464,57547,47448,40559,37315,47463,47453,40560,57525,40561,57533,57537,38385,57541,47455,57524,57522,57534,40562,57531,57530,40563,57535,57538,37568,57543,38467,36069,58488,37316,48071,58532,58490,48076,48080,58541,58549,58534,48072,40564,58538,57526,40565,48073,58545,58550,58542,40566,58544,58553,58546,58494,58537,40567,48081,48077,58492,58539,48075,58533,48074,58547,58530,58489,48078,58552,58491,58543,58540,58535,58487,58486,58529,58548,48079,58551,58493,58531,48722,37707,36264,40568,48730,48725,59556,59553,59495,48720,63959,40569,48719,48726,64953,40570,59493,48724,59505,59491,59492,48718,59555,48728,59508,59513,59507,60398,59503,59511,59509,59496,59490,59517,48727,59518,40571,59512,59501,59499,59494,40572,59502,59515,59498,59514,59554,40573,48723,40574,59510,59516,59506,59500,48721,40609,40610,37317,58536,59504,48729,59497,37976,37748,40611,40612,60404,49143,60403,60400,60484,49147,40613,60481,60408,60483,60393,60406,40614,49149,60385,40615,60383,60482,60480,60414,60397,60396,60386,49216,60392,60402,60413,49219,60485,49640,49221,49150,60390,60399,60382,60384,49141,49218,49146,60391,60407,60401,49217,60381,49635,60409,60412,49148,37875,60395,49220,49145,40616,40617,37879,49144,60405,60411,49142,60388,60410,37318,38107,60389,35798,40618,65148,40619,40620,38741,38466,60394,61138,65145,61143,49637,49639,61149,49633,61164,61155,61144,61145,61154,49646,61153,61137,61152,61140,61165,49645,49643,61141,61160,40621,61146,61159,40622,61161,61136,49638,40623,61162,40624,38468,61150,64954,49642,61147,64955,49644,61156,40625,40626,49636,61142,61157,40627,61151,60387,61158,61139,40628,49641,61163,40629,49634,61134,40630,64371,40631,40632,61792,61785,49897,61780,61795,61787,61148,40633,61797,61781,49896,61791,49898,49906,49904,61793,49905,61783,40634,61784,61789,61794,40635,61133,49899,61802,61799,61803,61790,61786,61800,62314,61788,49902,38742,49901,61135,49903,61796,61798,49900,61801,61779,61782,40636,38108,64956,40637,40638,40639,62323,62307,50155,62321,38743,62305,50156,62316,40640,62312,50161,62322,62306,62309,50153,62324,40641,62317,62320,50159,50164,50162,62313,62308,40642,50157,50158,62304,50154,40643,50152,50160,62319,50163,38744,62315,62325,50165,62311,62318,40644,40645,40646,40647,62707,62786,62709,62716,62310,62714,62697,62784,50371,62701,62718,62708,40648,50370,40649,62788,62710,40650,62715,62717,62695,62785,62706,62711,62699,62703,62787,62713,62696,62700,62702,62712,40651,50369,62705,38745,40652,40653,62698,38746,40654,40655,40656,64957,62704,63073,63078,50511,63080,40657,50505,63076,63082,50510,50506,50507,63072,63079,50509,63077,50508,63071,63075,63074,38747,63070,63081,38748,40658,40659,50609,63341,63344,63340,63342,63343,63337,63338,63335,40660,63339,63336,50610,50611,38749,40661,63563,63565,40662,38750,38232,63564,63566,50656,63562,50655,50657,40663,40664,63691,63692,50756,63690,40665,63827,63826,63828,50783,63829,50782,63830,63858,63861,63860,50792,63859,40666,40667,50802,50800,50801,50807,63936,63937,63935,63945,40176,36289,36288,33761,33760,33762,33763,36295,36313,36310,33974,36311,33975,33973,36678,36312,33972,34163,38751,36340,34419,34166,34161,35933,34164,34990,35233,34168,34167,36214,34024,34165,35663,34414,36343,36338,36339,36342,34162,36341,36442,35198,36443,34408,34407,33907,36444,38752,34410,34409,36274,36263,36448,36447,34411,34421,34412,36018,34415,34416,36164,34413,34422,36441,34417,36440,34418,36446,34420,36020,34984,36517,36516,36518,36524,36561,38753,35934,36019,34662,34986,38754,36519,36775,36714,34664,34668,36565,36525,36844,34666,36813,36522,36520,34665,34663,35236,36521,34660,36515,34230,38755,36837,34667,34661,36523,36068,36846,34983,36674,36562,35790,35797,34982,35662,34991,38756,34985,35490,36566,36567,35935,34988,36564,34989,34980,36445,38757,38758,34987,34981,36560,36563,36568,36676,36672,36673,38759,35197,35453,36675,38760,36677,35239,35234,35240,36265,35491,35237,35238,37105,36605,38761,38762,36606,35194,35454,35195,35196,35241,35489,36713,35795,36715,35496,36712,35493,35492,35494,36776,35495,38763,35451,35452,35796,36842,38764,35394,35665,35668,36221,36022,35664,38765,36777,35666,35661,35667,36774,36960,35793,35791,35792,35794,36809,35787,35788,35789,36810,36840,38766,36841,36845,38767,36838,37025,36839,38768,36811,36843,36961,36017,36021,36016,36936,38769,36163,35964,43768,40668,55364,56440,59557,62326,33635,43769,44460,45179,40669,55365,38770,55366,45996,38109,46717,56442,56441,46755,46716,56443,46718,46754,46753,46715,38110,64958,47464,38771,57552,57550,57551,57549,38772,48082,37483,48085,48087,48086,40670,64959,48083,48084,40671,59559,59558,48731,59560,59561,48732,38773,60493,60491,61171,60489,60490,49222,60486,60494,60488,60492,61167,61169,61170,49651,61166,49650,61168,49647,49648,49649,60487,49909,61806,61804,61805,49907,49910,49908,37799,62327,62328,50166,62789,62791,62790,50372,50512,63085,63084,63083,33636,33698,33764,36314,33976,33780,35670,34170,35500,34172,34174,34173,34169,36344,34013,34423,35009,34670,34631,36937,34671,34992,34673,34672,34669,36570,37036,34993,36569,36680,36681,36682,36784,36683,36684,35242,36679,35936,35499,36716,35497,35498,35669,35671,35672,36847,43770,40177,51626,34209,51800,42729,51798,51801,51799,52142,43201,43202,52144,43199,52143,52141,43200,43198,34211,33979,33765,34210,64961,52696,52699,43773,52698,52697,43772,43771,65090,43840,52700,43774,34674,33978,35800,33977,33767,53446,44462,44463,44464,53447,53443,44461,53444,53445,53442,33766,40672,45220,45217,54341,45218,45221,54342,45182,45180,45181,45219,37479,38774,36345,34424,35244,45997,55369,46005,55368,55371,46001,55370,46763,45999,46002,45998,46003,46004,46000,37556,40673,55367,46759,56445,56483,64965,56482,46764,46760,46761,56444,56446,56481,46756,46758,46762,46757,37626,35501,57555,57553,57554,47466,47467,64964,57556,47465,48088,48090,48089,38775,58555,37447,58556,59563,38393,59562,40674,49223,49224,60495,49225,61174,61172,38776,61173,49652,38777,61807,50167,50903,38395,49653,43841,45222,54343,34425,35801,55372,46006,46765,56484,56486,46767,46766,46768,46769,56485,47470,47471,47469,48091,47468,57557,35675,37578,48092,59564,60496,49226,49654,61808,61812,49913,61809,49914,49912,61813,49915,61811,36973,62329,49911,50168,63693,43842,46008,46007,37960,46770,56488,56487,46771,36190,40675,57561,47475,47472,57560,47474,57558,47473,57559,35802,58557,48093,36070,59567,64967,48733,59565,48734,48735,59566,48736,37869,60497,49230,49227,49232,60499,49228,60498,49231,49229,61177,61179,40676,49655,61178,49656,61176,61175,64968,61815,61814,49916,61816,62334,50170,62333,62330,50169,62331,62332,36192,62792,62793,50373,64969,50515,38778,40379,63086,40677,50513,50514,63087,38779,50612,50613,63345,40678,40679,50757,63695,50759,63694,63696,50758,63831,63917,38780,38781,38782,43843,36571,38817,47476,40680,58558,37699,59568,49233,49234,65133,43844,64970,48737,50171,44465,38818,36165,49235,50658,44466,55373,38819,56489,38391,56491,56490,38820,57565,57562,47477,47478,57563,57564,38111,58560,58565,48094,58559,58561,58568,58563,58567,58564,58562,58566,48095,38112,40681,59571,40682,59569,48739,48738,59570,48740,38113,36071,38821,60502,60501,49236,60500,61180,40683,61182,61249,61248,64971,49657,61181,61857,49917,61821,61858,49918,61819,61822,61820,61817,49984,61818,36222,40684,38822,62369,64972,62371,62370,64973,62794,64974,62795,40685,64975,63088,40686,50615,40687,50614,63567,63568,50760,63697,64976,50793,44467,46772,58570,58569,59573,59572,40688,49658,61251,61250,61861,61859,61862,61860,50172,62372,62373,62374,40689,63089,63346,63698,33699,34212,35504,36779,36814,35803,44468,38241,60503,61252,38823,44469,35804,48096,60504,49985,61863,50173,62796,62797,50516,63569,44470,46011,46012,55374,46773,46774,56492,46775,47482,47484,57567,57568,57566,47479,47480,47483,47481,40690,40691,58571,48097,48098,33371,59580,48743,59575,59574,64978,59579,48741,40692,49243,59576,59581,59578,59577,48742,33403,49241,64979,60506,49237,60507,40693,33346,60505,49240,49238,49242,49239,37855,38824,61253,61258,61254,61257,49659,64981,60884,61256,61255,64982,49988,49986,49989,49987,61864,61865,61866,49990,38825,37560,40402,62378,50240,62376,50241,62375,62377,50174,62801,62798,62799,62800,63090,50518,64983,50517,40694,63348,63347,50616,64984,50659,50761,50784,63832,63918,63919,33700,33980,33981,36346,34213,34214,34215,34430,34428,36024,34429,36450,34465,34466,36449,34427,36215,34676,34675,35248,34995,35247,34996,36538,36572,35253,35249,35252,35250,35142,35505,35679,35680,35681,35678,35677,37031,36848,35806,36815,36025,36073,38826,36193,37032,44471,56493,38114,57569,58572,58573,48099,48100,59582,48744,64985,49660,40695,61867,40696,49991,62381,50242,62380,62382,62379,63093,62802,62803,38827,50374,63092,40697,64434,63091,40698,63349,63920,33701,36526,34677,35506,35807,35937,36938,38828,44472,64987,33702,44473,40699,40700,45223,54344,55375,64988,46776,36026,46779,46777,56494,40701,46781,38829,46778,38362,46780,47486,40702,57570,57571,59584,47485,47521,47522,58575,58574,48101,48102,58576,59583,48104,48745,48103,36167,40816,49244,59585,48747,48746,59586,59589,59587,59588,48748,40817,49249,49247,40818,38830,49246,60509,38831,49248,60508,61259,60510,49245,60511,61262,61260,61261,61266,49995,61265,61268,61267,61264,61263,49661,40819,41028,64990,61870,64991,61869,49994,49992,49993,64435,61868,64989,62385,50243,62384,62383,50244,64993,62808,62807,64436,62805,62804,50376,50375,62809,63350,50617,63095,50519,63094,62806,63351,50660,40820,50785,63833,63921,40821,38832,33768,36717,34218,36780,34216,36347,34220,34217,34219,35255,34470,34469,34467,34468,34680,35938,34681,34678,34679,36573,34998,38833,34999,35256,35257,35254,38834,35684,35683,36216,38835,35682,36816,35809,35808,38836,36849,38837,44474,55376,61269,44475,40822,58578,58577,60512,61271,61270,40823,49996,62386,62387,50377,40824,63922,45224,46783,46782,57572,57574,47524,57573,47523,47525,57575,40825,58580,58582,58581,38838,58584,48105,58583,58579,37963,37450,37539,58585,38839,59596,59599,59601,59591,59595,59592,48750,48753,48755,59593,59594,48754,59597,59600,59598,48756,64996,48752,59590,48749,48751,40826,38340,49251,60518,60516,60515,40827,60521,60520,60519,60514,49250,60513,60517,49252,38840,61274,40828,61278,61275,61277,61276,61273,61279,61282,61280,61281,49728,49662,61272,61283,61875,61878,61880,61879,61873,61877,61872,64437,61874,49997,61871,38369,61876,37725,37561,62400,62389,50245,50246,62388,62393,62399,62391,62398,40829,62395,39675,62394,62397,62392,62390,39676,62396,33365,62816,62814,50378,62813,62819,62817,39677,50379,62812,62810,62811,50381,62815,50380,62818,63096,63102,40186,40187,63097,50523,63137,50522,63101,63100,50521,63099,50520,63098,63357,63393,63358,63355,50619,63352,63356,63395,63394,63353,63354,50618,63570,50663,40188,63571,50661,50662,40189,63699,50762,63862,50794,40190,63923,50795,63924,63925,63939,63938,50810,63949,33769,34222,34472,34473,34471,36027,34682,36257,36574,35003,35002,35007,35006,36850,35001,35000,35004,35005,36169,36575,36075,34892,36074,36963,35260,35258,35259,36271,35507,36170,35508,36781,35685,36782,36783,36940,35811,36853,35942,35810,36941,36851,35940,36852,36028,36030,36939,36962,36194,36258,37033,45225,57577,57576,48106,48107,58586,40368,59602,60524,38841,48757,49253,60522,64998,60525,49254,64999,61284,60523,61881,49998,62401,37627,36031,62822,62820,62821,36217,36985,63138,50524,63396,50666,50620,50664,50665,63700,50786,64438,45226,38842,61882,38843,54345,40369,47526,65004,58587,48108,58588,37818,59604,59603,49256,48758,48759,59607,59606,59605,65111,60526,60529,60528,60527,49255,61288,61286,61285,65006,61287,65005,49999,61884,61885,50000,61883,65008,62403,62402,62405,50247,62404,38844,62823,62825,62824,36195,40370,63139,63142,63140,63141,63397,50621,63572,63573,63574,50763,50787,63926,45227,48760,49257,61886,65001,63398,65003,63940,54346,50811,45228,60530,40371,61887,62406,40372,63143,63399,45229,58589,58590,40373,48109,48110,59609,48762,48761,59608,36196,61289,38845,61888,61890,61889,50003,50002,50001,40374,50526,63144,50525,63401,63400,50764,63701,46013,57578,58593,58591,58592,59618,59613,59610,59617,65199,59619,48764,59616,59612,38846,59611,59615,59614,48763,60541,60536,60534,60577,60535,40375,60531,40430,60537,38847,60532,61298,60533,60578,40431,40432,38142,65011,60540,49258,60539,60538,60542,38115,38116,61290,61293,65012,61292,38848,61300,61295,61299,61297,61296,61294,61291,49731,49730,49732,49729,61301,65013,38133,38117,61896,61899,61897,61901,38849,61902,61894,50008,61895,61893,61900,61892,61891,50007,50005,50004,38118,38119,38120,61898,62415,62421,50250,62416,62419,62423,50251,62418,40433,62410,62409,62422,62413,38850,62411,62420,62412,50249,50248,40434,62407,62408,62417,40435,62414,38122,38121,40436,38851,62828,62831,50006,65017,38852,65104,38853,62829,62835,62833,62827,62838,65016,62826,65014,50383,62834,38397,62830,50382,62837,65015,62836,38854,63147,63146,40437,40438,63153,63149,63152,50528,40439,63150,63151,63145,63148,50527,65018,38137,38123,50623,63412,63407,63411,64851,63414,63410,63406,64715,50625,63409,63413,50624,63404,62832,63408,63405,64880,63402,38124,63403,50622,63578,63580,63583,63579,63584,40784,63577,63575,40440,50667,63581,50669,50668,63576,63582,38125,40441,63706,50765,63707,63705,63702,63704,63703,63834,38134,63836,63835,63865,64376,63864,63863,63866,38976,50803,50804,63946,63950,40442,33982,35010,35011,36576,38987,38988,38989,36267,38990,35263,36197,38991,35512,36718,35511,36719,37106,37026,35510,35509,38992,36976,36786,37037,36077,36965,35688,36942,36966,35690,35687,36785,35689,35814,38993,38994,36821,38995,35813,38996,36818,36819,35815,35812,38997,36820,38998,38999,35943,36266,35944,39000,39001,39002,39003,36202,36032,36943,36033,36944,39004,36078,36964,36076,36079,39005,39006,36175,36173,36174,36975,39007,46014,56495,57581,47527,57579,39008,39009,57580,38126,58594,58595,48113,48111,58596,48112,59624,59627,59621,59628,59620,59622,59623,59626,64865,48801,59631,59630,48765,59625,59629,48766,40443,33352,60588,49263,60583,49259,40444,60580,60586,60589,49264,60585,60582,60590,60581,40768,60587,49260,60579,49261,49262,60584,39010,37959,33351,61353,61306,61307,61310,61308,61302,40769,61305,61349,61309,40770,49733,61351,61348,49734,61350,61303,61346,61347,64938,61345,38127,38135,39011,40771,61906,61908,61911,40772,61905,39012,50009,61913,61904,61914,61910,61912,61916,61909,61917,61907,61903,50010,38258,61915,50011,50253,38128,38136,61304,62449,62440,50255,62436,50256,40773,40775,62445,62439,62429,50254,62442,62437,62438,62424,62431,62446,39013,62443,40776,62435,64710,62447,62430,62425,62444,40777,62427,62441,62432,62448,62428,50252,62426,62433,62434,37551,62845,62843,40778,62882,62894,62885,62844,62840,62887,62846,62883,62842,62890,62839,62881,62886,62888,62891,62841,40779,62895,62896,62889,62893,62884,38129,63169,63172,50529,63171,63176,63174,50530,63165,63155,63154,50532,63167,63168,63164,63156,40781,63161,62892,40786,63157,50531,63163,38261,63162,39014,63158,63170,40798,63159,63419,63173,63175,63166,63160,63420,63422,63416,50626,40799,63429,63427,50627,63426,63425,63418,63415,63421,63430,63417,63423,63593,63598,63588,63591,50670,63595,63602,63424,63589,63599,63603,63594,63587,63597,63596,63601,63600,63428,63592,63586,63590,50766,50767,63585,40800,63718,63709,63717,63714,63715,63708,63711,63719,63713,63712,63710,63716,40801,63837,40802,63838,40803,63840,63839,63842,63841,63868,63867,63927,63928,40804,63941,50808,50812,64842,63951,50813,33983,34474,35676,34686,34685,36081,35013,36577,35012,35265,36823,37034,35267,35266,36686,36685,35268,35264,36176,36968,36720,36721,36722,35514,36273,35513,36788,37038,35691,35693,35692,36789,36981,36787,35817,35818,39015,35819,36822,35820,35816,39016,36854,36946,36855,39017,36856,36036,36945,36035,36270,36080,36967,36977,36979,36980,36978,36199,36198,39018,36272,46015,33374,50384,63177,40805,50768,50769,36947,46016,57582,64765,47528,59632,64350,40806,60592,60593,60591,61355,61354,49735,61919,61356,61918,40785,62451,50257,50259,62450,64756,64380,50258,64439,62897,62899,62898,63178,50533,64840,50671,63720,63843,40807,63954,46017,33985,58597,64948,48802,39019,64418,40808,60595,60594,38259,61357,40809,50260,50385,63431,63947,35014,33376,33377,46018,48114,39020,48803,38138,62452,40810,63604,46784,39021,40811,40812,61358,36218,50788,46785,48804,49736,63605,46786,37571,59633,49266,60596,60597,49265,37844,61359,49740,49738,49739,49737,61920,50012,39022,62901,62900,62903,62902,50386,36268,39023,63179,63181,63180,50534,63432,40813,63606,63607,50672,63844,63869,50805,40814,56496,60598,61360,62453,57583,36452,61361,61922,61921,65000,63608,50770,39024,63845,63870,39025,39026,47529,59634,59635,60599,47530,50013,61923,64731,63183,50535,63184,63182,63609,63721,47531,39027,61364,61363,61362,61924,64440,40815,61928,61927,61926,61925,50014,62454,62905,50387,62904,63185,63435,63434,50628,63433,63612,63611,63610,36082,48115,60600,49741,62455,62456,63436,63613,39028,64994,63722,63846,63929,63956,48116,49742,61929,62457,63186,63614,34722,39029,48806,64992,61365,61930,62458,62459,62460,62910,64986,62906,50536,62909,62908,50388,62907,50390,37870,50389,63188,63187,50537,50538,64866,50630,63437,50629,63651,63652,63650,63649,50772,63723,63724,63725,50771,63847,63850,63849,63848,63955,35015,36723,35946,35947,36858,36857,36037,36038,36083,36084,36204,36203,49267,64472,50021,62911,63189,50631,63438,63957,35269,36085,36969,49268,39030,35270,61366,63439,63905,64980,51361,51363,64064,64065,64066,64067,64068,64069,64070,64071,64072,64073,64074,64075,64076,64077,64078,64079,64080,64081,64082,64083,64084,64085,64086,64087,64088,64089,64090,64091,64092,64093,64094,64095,64096,64097,64098,64099,64100,64101,64102,64103,64104,64105,64106,64107,64108,64109,64110,64111,64112,64113,64114,64115,64116,64117,64118,64119,64120,64121,64122,64123,64124,64125,64126,64161,64162,64163,64164,64165,64166,64167,64168,64169,64170,64171,64172,64173,64174,64175,64176,64177,64178,64179,64180,64181,64182,64183,64184,64185,64186,64187,64188,64189,64190,64191,64192,64193,64194,64195,64196,64197,64198,64199,64200,64201,64202,64203,64204,64205,64206,64207,64208,64209,64210,64211,64212,64213,64214,64215,64216,64217,64218,64219,64220,64221,64222,64223,64224,64225,64226,64227,64228,64229,64230,64231,64232,64233,64234,64235,64236,64237,64238,64239,64240,64241,64242,64243,64244,64245,64246,64247,64248,64249,64250,64251,64252,64253,64254,64320,64321,64322,64323,64324,64325,64326,64327,64328,64329,64330,64331,64332,64333,64334,64335,64336,64337,64338,64339,64340,64341,64342,64343,64344,64345,64346,64347,64348,64349,64350,64351,64352,64353,64354,64355,64356,64357,64358,64359,64360,64361,64362,64363,64364,64365,64366,64367,64368,64369,64370,64371,64372,64373,64374,64375,64376,64377,64378,64379,64380,64381,64382,64417,64418,64419,64420,64421,64422,64423,64424,64425,64426,64427,64428,64429,64430,64431,64432,64433,64434,64435,64436,64437,64438,64439,64440,64441,64442,64443,64444,64445,64446,64447,64448,64449,64450,64451,64452,64453,64454,64455,64456,64457,64458,64459,64460,64461,64462,64463,64464,64465,64466,64467,64468,64469,64470,64471,64472,64473,64474,64475,64476,64477,64478,64479,64480,64481,64482,64483,64484,64485,64486,64487,64488,64489,64490,64491,64492,64493,64494,64495,64496,64497,64498,64499,64500,64501,64502,64503,64504,64505,64506,64507,64508,64509,64510,64576,64577,64578,64579,64580,64581,64582,64583,64584,64585,64586,64587,64588,64589,64590,64591,64592,64593,64594,64595,64596,64597,64598,64599,64600,64601,64602,64603,64604,64605,64606,64607,64608,64609,64610,64611,64612,64613,64614,64615,64616,64617,64618,64619,64620,64621,64622,64623,64624,64625,64626,64627,64628,64629,64630,64631,64632,64633,64634,64635,64636,64637,64638,64673,64674,64675,64676,64677,64678,64679,64680,64681,64682,64683,64684,64685,64686,64687,64688,64689,64690,64691,64692,64693,64694,64695,64696,64697,64698,64699,64700,64701,64702,64703,64704,64705,64706,64707,64708,64709,64710,64711,64712,64713,64714,64715,64716,64717,64718,64719,64720,64721,64722,64723,64724,64725,64726,64727,64728,64729,64730,64731,64732,64733,64734,64735,64736,64737,64738,64739,64740,64741,64742,64743,64744,64745,64746,64747,64748,64749,64750,64751,64752,64753,64754,64755,64756,64757,64758,64759,64760,64761,64762,64763,64764,64765,64766,64832,64833,64834,64835,64836,64837,64838,64839,64840,64841,64842,64843,64844,64845,64846,64847,64848,64849,64850,64851,64852,64853,64854,64855,64856,64857,64858,64859,64860,64861,64862,64863,64864,64865,64866,64867,64868,64869,64870,64871,64872,64873,64874,64875,64876,64877,64878,64879,64880,64881,64882,64883,64884,64885,64886,64887,64888,64889,64890,64891,64892,64893,64894,64929,64930,64931,64932,64933,64934,64935,64936,64937,64938,64939,64940,64941,64942,64943,64944,64945,64946,64947,64948,64949,64950,64951,64952,64953,64954,64955,64956,64957,64958,64959,64960,64961,64962,64963,64964,64965,64966,64967,64968,64969,64970,64971,64972,64973,64974,64975,64976,64977,64978,64979,64980,64981,64982,64983,64984,64985,64986,64987,64988,64989,64990,64991,64992,64993,64994,64995,64996,64997,64998,64999,65000,65001,65002,65003,65004,65005,65006,65007,65008,65009,65010,65011,65012,65013,65014,65015,65016,65017,65018,65019,65020,65021,65022,65088,65089,65090,65091,65092,65093,65094,65095,65096,65097,65098,65099,65100,65101,65102,65103,65104,65105,65106,65107,65108,65109,65110,65111,65112,65113,65114,65115,65116,65117,65118,65119,65120,65121,65122,65123,65124,65125,65126,65127,65128,65129,65130,65131,65132,65133,65134,65135,65136,65137,65138,65139,65140,65141,65142,65143,65144,65145,65146,65147,65148,65149,65150,65185,65186,65187,65188,65189,65190,65191,65192,65193,65194,65195,65196,65197,65198,65199,65200,65201,65202,65203,65204,65205,65206,65207,65208,65209,65210,65211,65212,65213,65214,65215,65216,65217,65218,65219,65220,65221,65222,65223,65224,65225,65226,65227,65228,65229,65230,65231,65232,65233,65234,65235,65236,65237,65238,65239,65240,65241,65242,65243,65244,65245,65246,65247,65248,65249,65250,65251,65252,65253,65254,65255,65256,65257,65258,65259,65260,65261,65262,65263,65264,65265,65266,65267,65268,65269,65270,65271,65272,65273,65274,65275,65276,65277,65278,36416,36417,36418,36419,36420,36421,36422,36423,36424,36425,36426,36427,36428,36429,36430,36431,36432,36433,36434,36435,36436,36437,36438,36439,36440,36441,36442,36443,36444,36445,36446,36447,36448,36449,36450,36451,36452,36453,36454,36455,36456,36457,36458,36459,36460,36461,36462,36463,36464,36465,36466,36467,36468,36469,36470,36471,36472,36473,36474,36475,36476,36477,36478,36513,36514,36515,36516,36517,36518,36519,36520,36521,36522,36523,36524,36525,36526,36527,36528,36529,36530,36531,36532,36533,36534,36535,36536,36537,36538,36539,36540,36541,36542,36543,36544,36545,36546,36547,36548,36549,36550,36551,36552,36553,36554,36555,36556,36557,36558,36559,36560,36561,36562,36563,36564,36565,36566,36567,36568,36569,36570,36571,36572,36573,36574,36575,36576,36577,36578,36579,36580,36581,36582,36583,36584,36585,36586,36587,36588,36589,36590,36591,36592,36593,36594,36595,36596,36597,36598,36599,36600,36601,36602,36603,36604,36605,36606,36672,36673,36674,36675,36676,36677,36678,36679,36680,36681,36682,36683,36684,36685,36686,36687,36688,36689,36690,36691,36692,36693,36694,36695,36696,36697,36698,36699,36700,36701,36702,36703,36704,36705,36706,36707,36708,36709,36710,36711,36712,36713,36714,36715,36716,36717,36718,36719,36720,36721,36722,36723,36724,36725,36726,36727,36728,36729,36730,36731,36732,36733,36734,36769,36770,36771,36772,36773,36774,36775,36776,36777,36778,36779,36780,36781,36782,36783,36784,36785,36786,36787,36788,36789,36790,36791,36792,36793,36794,36795,36796,36797,36798,36799,36800,36801,36802,36803,36804,36805,36806,36807,36808,36809,36810,36811,36812,36813,36814,36815,36816,36817,36818,36819,36820,36821,36822,36823,36824,36825,36826,36827,36828,36829,36830,36831,36832,36833,36834,36835,36836,36837,36838,36839,36840,36841,36842,36843,36844,36845,36846,36847,36848,36849,36850,36851,36852,36853,36854,36855,36856,36857,36858,36859,36860,36861,36862,36928,36929,36930,36931,36932,36933,36934,36935,36936,36937,36938,36939,36940,36941,36942,36943,36944,36945,36946,36947,36948,36949,36950,36951,36952,36953,36954,36955,36956,36957,36958,36959,36960,36961,36962,36963,36964,36965,36966,36967,36968,36969,36970,36971,36972,36973,36974,36975,36976,36977,36978,36979,36980,36981,36982,36983,36984,36985,36986,36987,36988,36989,36990,37025,37026,37027,37028,37029,37030,37031,37032,37033,37034,37035,37036,37037,37038,37039,37040,37041,37042,37043,37044,37045,37046,37047,37048,37049,37050,37051,37052,37053,37054,37055,37056,37057,37058,37059,37060,37061,37062,37063,37064,37065,37066,37067,37068,37069,37070,37071,37072,37073,37074,37075,37076,37077,37078,37079,37080,37081,37082,37083,37084,37085,37086,37087,37088,37089,37090,37091,37092,37093,37094,37095,37096,37097,37098,37099,37100,37101,37102,37103,37104,37105,37106,37107,37108,37109,37110,37111,37112,37113,37114,37115,37116,37117,37118,37184,37185,37186,37187,37188,37189,37190,37191,37192,37193,37194,37195,37196,37197,37198,37199,37200,37201,37202,37203,37204,37205,37206,37207,37208,37209,37210,37211,37212,37213,37214,37215,37216,37217,37218,37219,37220,37221,37222,37223,37224,37225,37226,37227,37228,37229,37230,37231,37232,37233,37234,37235,37236,37237,37238,37239,37240,37241,37242,37243,37244,37245,37246,37281,37282,37283,37284,37285,37286,37287,37288,37289,37290,37291,37292,37293,37294,37295,37296,37297,37298,37299,37300,37301,37302,37303,37304,37305,37306,37307,37308,37309,37310,37311,37312,37313,37314,37315,37316,37317,37318,37319,37320,37321,37322,37323,37324,37325,37326,37327,37328,37329,37330,37331,37332,37333,37334,37335,37336,37337,37338,37339,37340,37341,37342,37343,37344,37345,37346,37347,37348,37349,37350,37351,37352,37353,37354,37355,37356,37357,37358,37359,37360,37361,37362,37363,37364,37365,37366,37367,37368,37369,37370,37371,37372,37373,37374,37440,37441,37442,37443,37444,37445,37446,37447,37448,37449,37450,37451,37452,37453,37454,37455,37456,37457,37458,37459,37460,37461,37462,37463,37464,37465,37466,37467,37468,37469,37470,37471,37472,37473,37474,37475,37476,37477,37478,37479,37480,37481,37482,37483,37484,37485,37486,37487,37488,37489,37490,37491,37492,37493,37494,37495,37496,37497,37498,37499,37500,37501,37502,37537,37538,37539,37540,37541,37542,37543,37544,37545,37546,37547,37548,37549,37550,37551,37552,37553,37554,37555,37556,37557,37558,37559,37560,37561,37562,37563,37564,37565,37566,37567,37568,37569,37570,37571,37572,37573,37574,37575,37576,37577,37578,37579,37580,37581,37582,37583,37584,37585,37586,37587,37588,37589,37590,37591,37592,37593,37594,37595,37596,37597,37598,37599,37600,37601,37602,37603,37604,37605,37606,37607,37608,37609,37610,37611,37612,37613,37614,37615,37616,37617,37618,37619,37620,37621,37622,37623,37624,37625,37626,37627,37628,37629,37630,37696,37697,37698,37699,37700,37701,37702,37703,37704,37705,37706,37707,37708,37709,37710,37711,37712,37713,37714,37715,37716,37717,37718,37719,37720,37721,37722,37723,37724,37725,37726,37727,37728,37729,37730,37731,37732,37733,37734,37735,37736,37737,37738,37739,37740,37741,37742,37743,37744,37745,37746,37747,37748,37749,37750,37751,37752,37753,37754,37755,37756,37757,37758,37793,37794,37795,37796,37797,37798,37799,37800,37801,37802,37803,37804,37805,37806,37807,37808,37809,37810,37811,37812,37813,37814,37815,37816,37817,37818,37819,37820,37821,37822,37823,37824,37825,37826,37827,37828,37829,37830,37831,37832,37833,37834,37835,37836,37837,37838,37839,37840,37841,37842,37843,37844,37845,37846,37847,37848,37849,37850,37851,37852,37853,37854,37855,37856,37857,37858,37859,37860,37861,37862,37863,37864,37865,37866,37867,37868,37869,37870,37871,37872,37873,37874,37875,37876,37877,37878,37879,37880,37881,37882,37883,37884,37885,37886,37952,37953,37954,37955,37956,37957,37958,37959,37960,37961,37962,37963,37964,37965,37966,37967,37968,37969,37970,37971,37972,37973,37974,37975,37976,37977,37978,37979,37980,37981,37982,37983,37984,37985,37986,37987,37988,37989,37990,37991,37992,37993,37994,37995,37996,37997,37998,37999,38000,38001,38002,38003,38004,38005,38006,38007,38008,38009,38010,38011,38012,38013,38014,38049,38050,38051,38052,38053,38054,38055,38056,38057,38058,38059,38060,38061,38062,38063,38064,38065,38066,38067,38068,38069,38070,38071,38072,38073,38074,38075,38076,38077,38078,38079,38080,38081,38082,38083,38084,38085,38086,38087,38088,38089,38090,38091,38092,38093,38094,38095,38096,38097,38098,38099,38100,38101,38102,38103,38104,38105,38106,38107,38108,38109,38110,38111,38112,38113,38114,38115,38116,38117,38118,38119,38120,38121,38122,38123,38124,38125,38126,38127,38128,38129,38130,38131,38132,38133,38134,38135,38136,38137,38138,38139,38140,38141,38142,38208,38209,38210,38211,38212,38213,38214,38215,38216,38217,38218,38219,38220,38221,38222,38223,38224,38225,38226,38227,38228,38229,38230,38231,38232,38233,38234,38235,38236,38237,38238,38239,38240,38241,38242,38243,38244,38245,38246,38247,38248,38249,38250,38251,38252,38253,38254,38255,38256,38257,38258,38259,38260,38261,38262,38263,38264,38265,38266,38267,38268,38269,38270,38305,38306,38307,38308,38309,38310,38311,38312,38313,38314,38315,38316,38317,38318,38319,38320,38321,38322,38323,38324,38325,38326,38327,38328,38329,38330,38331,38332,38333,38334,38335,38336,38337,38338,38339,38340,38341,38342,38343,38344,38345,38346,38347,38348,38349,38350,38351,38352,38353,38354,38355,38356,38357,38358,38359,38360,38361,38362,38363,38364,38365,38366,38367,38368,38369,38370,38371,38372,38373,38374,38375,38376,38377,38378,38379,38380,38381,38382,38383,38384,38385,38386,38387,38388,38389,38390,38391,38392,38393,38394,38395,38396,38397,38398,38464,38465,38466,38467,38468,38469,38470,38471,38472,38473,38474,38475,38476,38477,38478,38479,38480,38481,38482,38483,38484,38485,38486,38487,38488,38489,38490,38491,38492,38493,38494,38495,38496,38497,38498,38499,38500,38501,38502,38503,38504,38505,38506,38507,38508,38509,38510,38511,38512,38513,38514,38515,38516,38517,38518,38519,38520,38521,38522,38523,38524,38525,38526,38561,38562,38563,38564,38565,38566,38567,38568,38569,38570,38571,38572,38573,38574,38575,38576,38577,38578,38579,38580,38581,38582,38583,38584,38585,38586,38587,38588,38589,38590,38591,38592,38593,38594,38595,38596,38597,38598,38599,38600,38601,38602,38603,38604,38605,38606,38607,38608,38609,38610,38611,38612,38613,38614,38615,38616,38617,38618,38619,38620,38621,38622,38623,38624,38625,38626,38627,38628,38629,38630,38631,38632,38633,38634,38635,38636,38637,38638,38639,38640,38641,38642,38643,38644,38645,38646,38647,38648,38649,38650,38651,38652,38653,38654,38720,38721,38722,38723,38724,38725,38726,38727,38728,38729,38730,38731,38732,38733,38734,38735,38736,38737,38738,38739,38740,38741,38742,38743,38744,38745,38746,38747,38748,38749,38750,38751,38752,38753,38754,38755,38756,38757,38758,38759,38760,38761,38762,38763,38764,38765,38766,38767,38768,38769,38770,38771,38772,38773,38774,38775,38776,38777,38778,38779,38780,38781,38782,38817,38818,38819,38820,38821,38822,38823,38824,38825,38826,38827,38828,38829,38830,38831,38832,38833,38834,38835,38836,38837,38838,38839,38840,38841,38842,38843,38844,38845,38846,38847,38848,38849,38850,38851,38852,38853,38854,38855,38856,38857,38858,38859,38860,38861,38862,38863,38864,38865,38866,38867,38868,38869,38870,38871,38872,38873,38874,38875,38876,38877,38878,38879,38880,38881,38882,38883,38884,38885,38886,38887,38888,38889,38890,38891,38892,38893,38894,38895,38896,38897,38898,38899,38900,38901,38902,38903,38904,38905,38906,38907,38908,38909,38910,38976,38977,38978,38979,38980,38981,38982,38983,38984,38985,38986,38987,38988,38989,38990,38991,38992,38993,38994,38995,38996,38997,38998,38999,39000,39001,39002,39003,39004,39005,39006,39007,39008,39009,39010,39011,39012,39013,39014,39015,39016,39017,39018,39019,39020,39021,39022,39023,39024,39025,39026,39027,39028,39029,39030,39031,39032,39033,39034,39035,39036,39037,39038,39073,39074,39075,39076,39077,39078,39079,39080,39081,39082,39083,39084,39085,39086,39087,39088,39089,39090,39091,39092,39093,39094,39095,39096,39097,39098,39099,39100,39101,39102,39103,39104,39105,39106,39107,39108,39109,39110,39111,39112,39113,39114,39115,39116,39117,39118,39119,39120,39121,39122,39123,39124,39125,39126,39127,39128,39129,39130,39131,39132,39133,39134,39135,39136,39137,39138,39139,39140,39141,39142,39143,39144,39145,39146,39147,39148,39149,39150,39151,39152,39153,39154,39155,39156,39157,39158,39159,39160,39161,39162,39163,39164,39165,39166,39232,39233,39234,39235,39236,39237,39238,39239,39240,39241,39242,39243,39244,39245,39246,39247,39248,39249,39250,39251,39252,39253,39254,39255,39256,39257,39258,39259,39260,39261,39262,39263,39264,39265,39266,39267,39268,39269,39270,39271,39272,39273,39274,39275,39276,39277,39278,39279,39280,39281,39282,39283,39284,39285,39286,39287,39288,39289,39290,39291,39292,39293,39294,39329,39330,39331,39332,39333,39334,39335,39336,39337,39338,39339,39340,39341,39342,39343,39344,39345,39346,39347,39348,39349,39350,39351,39352,39353,39354,39355,39356,39357,39358,39359,39360,39361,39362,39363,39364,39365,39366,39367,39368,39369,39370,39371,39372,39373,39374,39375,39376,39377,39378,39379,39380,39381,39382,39383,39384,39385,39386,39387,39388,39389,39390,39391,39392,39393,39394,39395,39396,39397,39398,39399,39400,39401,39402,39403,39404,39405,39406,39407,39408,39409,39410,39411,39412,39413,39414,39415,39416,39417,39418,39419,39420,39421,39422,39488,39489,39490,39491,39492,39493,39494,39495,39496,39497,39498,39499,39500,39501,39502,39503,39504,39505,39506,39507,39508,39509,39510,39511,39512,39513,39514,39515,39516,39517,39518,39519,39520,39521,39522,39523,39524,39525,39526,39527,39528,39529,39530,39531,39532,39533,39534,39535,39536,39537,39538,39539,39540,39541,39542,39543,39544,39545,39546,39547,39548,39549,39550,39585,39586,39587,39588,39589,39590,39591,39592,39593,39594,39595,39596,39597,39598,39599,39600,39601,39602,39603,39604,39605,39606,39607,39608,39609,39610,39611,39612,39613,39614,39615,39616,39617,39618,39619,39620,39621,39622,39623,39624,39625,39626,39627,39628,39629,39630,39631,39632,39633,39634,39635,39636,39637,39638,39639,39640,39641,39642,39643,39644,39645,39646,39647,39648,39649,39650,39651,39652,39653,39654,39655,39656,39657,39658,39659,39660,39661,39662,39663,39664,39665,39666,39667,39668,39669,39670,39671,39672,39673,39674,39675,39676,39677,39678,39744,39745,39746,39747,39748,39749,39750,39751,39752,39753,39754,39755,39756,39757,39758,39759,39760,39761,39762,39763,39764,39765,39766,39767,39768,39769,39770,39771,39772,39773,39774,39775,39776,39777,39778,39779,39780,39781,39782,39783,39784,39785,39786,39787,39788,39789,39790,39791,39792,39793,39794,39795,39796,39797,39798,39799,39800,39801,39802,39803,39804,39805,39806,39841,39842,39843,39844,39845,39846,39847,39848,39849,39850,39851,39852,39853,39854,39855,39856,39857,39858,39859,39860,39861,39862,39863,39864,39865,39866,39867,39868,39869,39870,39871,39872,39873,39874,39875,39876,39877,39878,39879,39880,39881,39882,39883,39884,39885,39886,39887,39888,39889,39890,39891,39892,39893,39894,39895,39896,39897,39898,39899,39900,39901,39902,39903,39904,39905,39906,39907,39908,39909,39910,39911,39912,39913,39914,39915,39916,39917,39918,39919,39920,39921,39922,39923,39924,39925,39926,39927,39928,39929,39930,39931,39932,39933,39934,40000,40001,40002,40003,40004,40005,40006,40007,40008,40009,40010,40011,40012,40013,40014,40015,40016,40017,40018,40019,40020,40021,40022,40023,40024,40025,40026,40027,40028,40029,40030,40031,40032,40033,40034,40035,40036,40037,40038,40039,40040,40041,40042,40043,40044,40045,40046,40047,40048,40049,40050,40051,40052,40053,40054,40055,40056,40057,40058,40059,40060,40061,40062,40097,40098,40099,40100,40101,40102,40103,40104,40105,40106,40107,40108,40109,40110,40111,40112,40113,40114,40115,40116,40117,40118,40119,40120,40121,40122,40123,40124,40125,40126,40127,40128,40129,40130,40131,40132,40133,40134,40135,40136,40137,40138,40139,40140,40141,40142,40143,40144,40145,40146,40147,40148,40149,40150,40151,40152,40153,40154,40155,40156,40157,40158,40159,40160,40161,40162,40163,40164,40165,40166,40167,40168,40169,40170,40171,40172,40173,40174,40175,40176,40177,40178,40179,40180,40181,40182,40183,40184,40185,40186,40187,40188,40189,40190,40256,40257,40258,40259,40260,40261,40262,40263,40264,40265,40266,40267,40268,40269,40270,40271,40272,40273,40274,40275,40276,40277,40278,40279,40280,40281,40282,40283,40284,40285,40286,40287,40288,40289,40290,40291,40292,40293,40294,40295,40296,40297,40298,40299,40300,40301,40302,40303,40304,40305,40306,40307,40308,40309,40310,40311,40312,40313,40314,40315,40316,40317,40318,40353,40354,40355,40356,40357,40358,40359,40360,40361,40362,40363,40364,40365,40366,40367,40368,40369,40370,40371,40372,40373,40374,40375,40376,40377,40378,40379,40380,40381,40382,40383,40384,40385,40386,40387,40388,40389,40390,40391,40392,40393,40394,40395,40396,40397,40398,40399,40400,40401,40402,40403,40404,40405,40406,40407,40408,40409,40410,40411,40412,40413,40414,40415,40416,40417,40418,40419,40420,40421,40422,40423,40424,40425,40426,40427,40428,40429,40430,40431,40432,40433,40434,40435,40436,40437,40438,40439,40440,40441,40442,40443,40444,40445,40446,40512,40513,40514,40515,40516,40517,40518,40519,40520,40521,40522,40523,40524,40525,40526,40527,40528,40529,40530,40531,40532,40533,40534,40535,40536,40537,40538,40539,40540,40541,40542,40543,40544,40545,40546,40547,40548,40549,40550,40551,40552,40553,40554,40555,40556,40557,40558,40559,40560,40561,40562,40563,40564,40565,40566,40567,40568,40569,40570,40571,40572,40573,40574,40609,40610,40611,40612,40613,40614,40615,40616,40617,40618,40619,40620,40621,40622,40623,40624,40625,40626,40627,40628,40629,40630,40631,40632,40633,40634,40635,40636,40637,40638,40639,40640,40641,40642,40643,40644,40645,40646,40647,40648,40649,40650,40651,40652,40653,40654,40655,40656,40657,40658,40659,40660,40661,40662,40663,40664,40665,40666,40667,40668,40669,40670,40671,40672,40673,40674,40675,40676,40677,40678,40679,40680,40681,40682,40683,40684,40685,40686,40687,40688,40689,40690,40691,40692,40693,40694,40695,40696,40697,40698,40699,40700,40701,40702,40768,40769,40770,40771,40772,40773,40774,40775,40776,40777,40778,40779,40780,40781,40782,40783,40784,40785,40786,40787,40788,40789,40790,40791,40792,40793,40794,40795,40796,40797,40798,40799,40800,40801,40802,40803,40804,40805,40806,40807,40808,40809,40810,40811,40812,40813,40814,40815,40816,40817,40818,40819,40820,40821,40822,40823,40824,40825,40826,40827,40828,40829,40830,40865,40866,40867,40868,40869,40870,40871,40872,40873,40874,40875,40876,40877,40878,40879,40880,40881,40882,40883,40884,40885,40886,40887,40888,40889,40890,40891,40892,40893,40894,40895,40896,40897,40898,40899,40900,40901,40902,40903,40904,40905,40906,40907,40908,40909,40910,40911,40912,40913,40914,40915,40916,40917,40918,40919,40920,40921,40922,40923,40924,40925,40926,40927,40928,40929,40930,40931,40932,40933,40934,40935,40936,40937,40938,40939,40940,40941,40942,40943,40944,40945,40946,40947,40948,40949,40950,40951,40952,40953,40954,40955,40956,40957,40958,41024,41025,41026,41027,41028,41029,41030,41031,41032,41033,41034,41035,41036,41037,41038,41039,41040,41041,41042,41043,41044,41045,41046,41047,41048,41049,41050,41051,41052,41053,41054,41055,41056,41057,41058,41059,41060,41061,41062,41063,41064,41065,41066,41067,41068,41069,41070,41071,41072,41073,41074,41075,41076,41077,41078,41079,41080,41081,41082,41083,41084,41085,41086,41121,41122,41123,41124,41125,41126,41127,41128,41129,41130,41131,41132,41133,41134,41135,41136,41137,41138,41139,41140,41141,41142,41143,41144,41145,41146,41147,41148,41149,41150,41151,41152,41153,41154,41155,41156,41157,41158,41159,41160,41161,41162,41163,41164,41165,41166,41167,41168,41169,41170,41171,41172,41173,41174,41175,41176,41177,41178,41179,41180,41181,41182,41183,41184,41185,41186,41187,41188,41189,41190,41191,41192,41193,41194,41195,41196,41197,41198,41199,41200,41201,41202,41203,41204,41205,41206,41207,41208,41209,41210,41211,41212,41213,41214,33088,33089,33090,33091,33092,33093,33094,33095,33096,33097,33098,33099,33100,33101,33102,33103,33104,33105,33106,33107,33108,33109,33110,33111,33112,33113,33114,33115,33116,33117,33118,33119,33120,33121,33122,33123,33124,33125,33126,33127,33128,33129,33130,33131,33132,33133,33134,33135,33136,33137,33138,33139,33140,33141,33142,33143,33144,33145,33146,33147,33148,33149,33150,33185,33186,33187,33188,33189,33190,33191,33192,33193,33194,33195,33196,33197,33198,33199,33200,33201,33202,33203,33204,33205,33206,33207,33208,33209,33210,33211,33212,33213,33214,33215,33216,33217,33218,33219,33220,33221,33222,33223,33224,33225,33226,33227,33228,33229,33230,33231,33232,33233,33234,33235,33236,33237,33238,33239,33240,33241,33242,33243,33244,33245,33246,33247,33248,33249,33250,33251,33252,33253,33254,33255,33256,33257,33258,33259,33260,33261,33262,33263,33264,33265,33266,33267,33268,33269,33270,33271,33272,33273,33274,33275,33276,33277,33278,33344,33345,33346,33347,33348,33349,33350,33351,33352,33353,33354,33355,33356,33357,33358,33359,33360,33361,33362,33363,33364,33365,33366,33367,33368,33369,33370,33371,33372,33373,33374,33375,33376,33377,33378,33379,33380,33381,33382,33383,33384,33385,33386,33387,33388,33389,33390,33391,33392,33393,33394,33395,33396,33397,33398,33399,33400,33401,33402,33403,33404,33405,33406,33441,33442,33443,33444,33445,33446,33447,33448,33449,33450,33451,33452,33453,33454,33455,33456,33457,33458,33459,33460,33461,33462,33463,33464,33465,33466,33467,33468,33469,33470,33471,33472,33473,33474,33475,33476,33477,33478,33479,33480,33481,33482,33483,33484,33485,33486,33487,33488,33489,33490,33491,33492,33493,33494,33495,33496,33497,33498,33499,33500,33501,33502,33503,33504,33505,33506,33507,33508,33509,33510,33511,33512,33513,33514,33515,33516,33517,33518,33519,33520,33521,33522,33523,33524,33525,33526,33527,33528,33529,33530,33531,33532,33533,33534,33600,33601,33602,33603,33604,33605,33606,33607,33608,33609,33610,33611,33612,33613,33614,33615,33616,33617,33618,33619,33620,33621,33622,33623,33624,33625,33626,33627,33628,33629,33630,33631,33632,33633,33634,33635,33636,33637,33638,33639,33640,33641,33642,33643,33644,33645,33646,33647,33648,33649,33650,33651,33652,33653,33654,33655,33656,33657,33658,33659,33660,33661,33662,33697,33698,33699,33700,33701,33702,33703,33704,33705,33706,33707,33708,33709,33710,33711,33712,33713,33714,33715,33716,33717,33718,33719,33720,33721,33722,33723,33724,33725,33726,33727,33728,33729,33730,33731,33732,33733,33734,33735,33736,33737,33738,33739,33740,33741,33742,33743,33744,33745,33746,33747,33748,33749,33750,33751,33752,33753,33754,33755,33756,33757,33758,33759,33760,33761,33762,33763,33764,33765,33766,33767,33768,33769,33770,33771,33772,33773,33774,33775,33776,33777,33778,33779,33780,33781,33782,33783,33784,33785,33786,33787,33788,33789,33790,33856,33857,33858,33859,33860,33861,33862,33863,33864,33865,33866,33867,33868,33869,33870,33871,33872,33873,33874,33875,33876,33877,33878,33879,33880,33881,33882,33883,33884,33885,33886,33887,33888,33889,33890,33891,33892,33893,33894,33895,33896,33897,33898,33899,33900,33901,33902,33903,33904,33905,33906,33907,33908,33909,33910,33911,33912,33913,33914,33915,33916,33917,33918,33953,33954,33955,33956,33957,33958,33959,33960,33961,33962,33963,33964,33965,33966,33967,33968,33969,33970,33971,33972,33973,33974,33975,33976,33977,33978,33979,33980,33981,33982,33983,33984,33985,33986,33987,33988,33989,33990,33991,33992,33993,33994,33995,33996,33997,33998,33999,34000,34001,34002,34003,34004,34005,34006,34007,34008,34009,34010,34011,34012,34013,34014,34015,34016,34017,34018,34019,34020,34021,34022,34023,34024,34025,34026,34027,34028,34029,34030,34031,34032,34033,34034,34035,34036,34037,34038,34039,34040,34041,34042,34043,34044,34045,34046,34112,34113,34114,34115,34116,34117,34118,34119,34120,34121,34122,34123,34124,34125,34126,34127,34128,34129,34130,34131,34132,34133,34134,34135,34136,34137,34138,34139,34140,34141,34142,34143,34144,34145,34146,34147,34148,34149,34150,34151,34152,34153,34154,34155,34156,34157,34158,34159,34160,34161,34162,34163,34164,34165,34166,34167,34168,34169,34170,34171,34172,34173,34174,34209,34210,34211,34212,34213,34214,34215,34216,34217,34218,34219,34220,34221,34222,34223,34224,34225,34226,34227,34228,34229,34230,34231,34232,34233,34234,34235,34236,34237,34238,34239,34240,34241,34242,34243,34244,34245,34246,34247,34248,34249,34250,34251,34252,34253,34254,34255,34256,34257,34258,34259,34260,34261,34262,34263,34264,34265,34266,34267,34268,34269,34270,34271,34272,34273,34274,34275,34276,34277,34278,34279,34280,34281,34282,34283,34284,34285,34286,34287,34288,34289,34290,34291,34292,34293,34294,34295,34296,34297,34298,34299,34300,34301,34302,34368,34369,34370,34371,34372,34373,34374,34375,34376,34377,34378,34379,34380,34381,34382,34383,34384,34385,34386,34387,34388,34389,34390,34391,34392,34393,34394,34395,34396,34397,34398,34399,34400,34401,34402,34403,34404,34405,34406,34407,34408,34409,34410,34411,34412,34413,34414,34415,34416,34417,34418,34419,34420,34421,34422,34423,34424,34425,34426,34427,34428,34429,34430,34465,34466,34467,34468,34469,34470,34471,34472,34473,34474,34475,34476,34477,34478,34479,34480,34481,34482,34483,34484,34485,34486,34487,34488,34489,34490,34491,34492,34493,34494,34495,34496,34497,34498,34499,34500,34501,34502,34503,34504,34505,34506,34507,34508,34509,34510,34511,34512,34513,34514,34515,34516,34517,34518,34519,34520,34521,34522,34523,34524,34525,34526,34527,34528,34529,34530,34531,34532,34533,34534,34535,34536,34537,34538,34539,34540,34541,34542,34543,34544,34545,34546,34547,34548,34549,34550,34551,34552,34553,34554,34555,34556,34557,34558,34624,34625,34626,34627,34628,34629,34630,34631,34632,34633,34634,34635,34636,34637,34638,34639,34640,34641,34642,34643,34644,34645,34646,34647,34648,34649,34650,34651,34652,34653,34654,34655,34656,34657,34658,34659,34660,34661,34662,34663,34664,34665,34666,34667,34668,34669,34670,34671,34672,34673,34674,34675,34676,34677,34678,34679,34680,34681,34682,34683,34684,34685,34686,34721,34722,34723,34724,34725,34726,34727,34728,34729,34730,34731,34732,34733,34734,34735,34736,34737,34738,34739,34740,34741,34742,34743,34744,34745,34746,34747,34748,34749,34750,34751,34752,34753,34754,34755,34756,34757,34758,34759,34760,34761,34762,34763,34764,34765,34766,34767,34768,34769,34770,34771,34772,34773,34774,34775,34776,34777,34778,34779,34780,34781,34782,34783,34784,34785,34786,34787,34788,34789,34790,34791,34792,34793,34794,34795,34796,34797,34798,34799,34800,34801,34802,34803,34804,34805,34806,34807,34808,34809,34810,34811,34812,34813,34814,34880,34881,34882,34883,34884,34885,34886,34887,34888,34889,34890,34891,34892,34893,34894,34895,34896,34897,34898,34899,34900,34901,34902,34903,34904,34905,34906,34907,34908,34909,34910,34911,34912,34913,34914,34915,34916,34917,34918,34919,34920,34921,34922,34923,34924,34925,34926,34927,34928,34929,34930,34931,34932,34933,34934,34935,34936,34937,34938,34939,34940,34941,34942,34977,34978,34979,34980,34981,34982,34983,34984,34985,34986,34987,34988,34989,34990,34991,34992,34993,34994,34995,34996,34997,34998,34999,35000,35001,35002,35003,35004,35005,35006,35007,35008,35009,35010,35011,35012,35013,35014,35015,35016,35017,35018,35019,35020,35021,35022,35023,35024,35025,35026,35027,35028,35029,35030,35031,35032,35033,35034,35035,35036,35037,35038,35039,35040,35041,35042,35043,35044,35045,35046,35047,35048,35049,35050,35051,35052,35053,35054,35055,35056,35057,35058,35059,35060,35061,35062,35063,35064,35065,35066,35067,35068,35069,35070,35136,35137,35138,35139,35140,35141,35142,35143,35144,35145,35146,35147,35148,35149,35150,35151,35152,35153,35154,35155,35156,35157,35158,35159,35160,35161,35162,35163,35164,35165,35166,35167,35168,35169,35170,35171,35172,35173,35174,35175,35176,35177,35178,35179,35180,35181,35182,35183,35184,35185,35186,35187,35188,35189,35190,35191,35192,35193,35194,35195,35196,35197,35198,35233,35234,35235,35236,35237,35238,35239,35240,35241,35242,35243,35244,35245,35246,35247,35248,35249,35250,35251,35252,35253,35254,35255,35256,35257,35258,35259,35260,35261,35262,35263,35264,35265,35266,35267,35268,35269,35270,35271,35272,35273,35274,35275,35276,35277,35278,35279,35280,35281,35282,35283,35284,35285,35286,35287,35288,35289,35290,35291,35292,35293,35294,35295,35296,35297,35298,35299,35300,35301,35302,35303,35304,35305,35306,35307,35308,35309,35310,35311,35312,35313,35314,35315,35316,35317,35318,35319,35320,35321,35322,35323,35324,35325,35326,35392,35393,35394,35395,35396,35397,35398,35399,35400,35401,35402,35403,35404,35405,35406,35407,35408,35409,35410,35411,35412,35413,35414,35415,35416,35417,35418,35419,35420,35421,35422,35423,35424,35425,35426,35427,35428,35429,35430,35431,35432,35433,35434,35435,35436,35437,35438,35439,35440,35441,35442,35443,35444,35445,35446,35447,35448,35449,35450,35451,35452,35453,35454,35489,35490,35491,35492,35493,35494,35495,35496,35497,35498,35499,35500,35501,35502,35503,35504,35505,35506,35507,35508,35509,35510,35511,35512,35513,35514,35515,35516,35517,35518,35519,35520,35521,35522,35523,35524,35525,35526,35527,35528,35529,35530,35531,35532,35533,35534,35535,35536,35537,35538,35539,35540,35541,35542,35543,35544,35545,35546,35547,35548,35549,35550,35551,35552,35553,35554,35555,35556,35557,35558,35559,35560,35561,35562,35563,35564,35565,35566,35567,35568,35569,35570,35571,35572,35573,35574,35575,35576,35577,35578,35579,35580,35581,35582,35648,35649,35650,35651,35652,35653,35654,35655,35656,35657,35658,35659,35660,35661,35662,35663,35664,35665,35666,35667,35668,35669,35670,35671,35672,35673,35674,35675,35676,35677,35678,35679,35680,35681,35682,35683,35684,35685,35686,35687,35688,35689,35690,35691,35692,35693,35694,35695,35696,35697,35698,35699,35700,35701,35702,35703,35704,35705,35706,35707,35708,35709,35710,35745,35746,35747,35748,35749,35750,35751,35752,35753,35754,35755,35756,35757,35758,35759,35760,35761,35762,35763,35764,35765,35766,35767,35768,35769,35770,35771,35772,35773,35774,35775,35776,35777,35778,35779,35780,35781,35782,35783,35784,35785,35786,35787,35788,35789,35790,35791,35792,35793,35794,35795,35796,35797,35798,35799,35800,35801,35802,35803,35804,35805,35806,35807,35808,35809,35810,35811,35812,35813,35814,35815,35816,35817,35818,35819,35820,35821,35822,35823,35824,35825,35826,35827,35828,35829,35830,35831,35832,35833,35834,35835,35836,35837,35838,35904,35905,35906,35907,35908,35909,35910,35911,35912,35913,35914,35915,35916,35917,35918,35919,35920,35921,35922,35923,35924,35925,35926,35927,35928,35929,35930,35931,35932,35933,35934,35935,35936,35937,35938,35939,35940,35941,35942,35943,35944,35945,35946,35947,35948,35949,35950,35951,35952,35953,35954,35955,35956,35957,35958,35959,35960,35961,35962,35963,35964,35965,35966,36001,36002,36003,36004,36005,36006,36007,36008,36009,36010,36011,36012,36013,36014,36015,36016,36017,36018,36019,36020,36021,36022,36023,36024,36025,36026,36027,36028,36029,36030,36031,36032,36033,36034,36035,36036,36037,36038,36039,36040,36041,36042,36043,36044,36045,36046,36047,36048,36049,36050,36051,36052,36053,36054,36055,36056,36057,36058,36059,36060,36061,36062,36063,36064,36065,36066,36067,36068,36069,36070,36071,36072,36073,36074,36075,36076,36077,36078,36079,36080,36081,36082,36083,36084,36085,36086,36087,36088,36089,36090,36091,36092,36093,36094,36160,36161,36162,36163,36164,36165,36166,36167,36168,36169,36170,36171,36172,36173,36174,36175,36176,36177,36178,36179,36180,36181,36182,36183,36184,36185,36186,36187,36188,36189,36190,36191,36192,36193,36194,36195,36196,36197,36198,36199,36200,36201,36202,36203,36204,36205,36206,36207,36208,36209,36210,36211,36212,36213,36214,36215,36216,36217,36218,36219,36220,36221,36222,36257,36258,36259,36260,36261,36262,36263,36264,36265,36266,36267,36268,36269,36270,36271,36272,36273,36274,36275,36276,36277,36278,36279,36280,36281,36282,36283,36284,36285,36286,36287,36288,36289,36290,36291,36292,36293,36294,36295,36296,36297,36298,36299,36300,36301,36302,36303,36304,36305,36306,36307,36308,36309,36310,36311,36312,36313,36314,36315,36316,36317,36318,36319,36320,36321,36322,36323,36324,36325,36326,36327,36328,36329,36330,36331,36332,36333,36334,36335,36336,36337,36338,36339,36340,36341,36342,36343,36344,36345,36346,36347,36348,36349,36350,50849,50850,50851,50852,50853,50854,50855,50856,50857,50858,50859,50860,50861,50862,50863,50864,50865,50866,50867,50868,50869,50870,50871,50872,50873,50874,50875,50876,50877,50878,50879,50880,50881,50882,50883,50884,50885,50886,50887,50888,50889,50890,50891,50892,50893,50894,50895,50896,50897,50898,50899,50900,50901,50902,50903,50904,50905,50906,50907,50908,50909,50910,50911,50912,50913,50914,50915,50916,50917,50918,50919,50920,50921,50922,50923,50924,50925,50926,50927,50928,50929,50930,50931,50932,50933,50934,50935,50936,50937,50938,50939,50940,50941,50942,51008,51009,51010,51011,51012,51013,51014,51015,51016,51017,51018,51019,51020,51021,51022,51023,51024,51025,51026,51027,51028,51029,51030,51031,51032,51033,51034,51035,51036,51037,51038,51039,51040,51041,51042,51043,51044,51045,51046,51047,51048,51049,51050,51051,51052,51053,51054,51055,51056,51057,51058,51059,51060,51061,51062,51063,51064,51065,51066,51067,51068,51069,51070,51105,51106,51107,51108,51109,51110,51111,51112,51113,51114,51115,51116,51117,51118,51119,51120,51121,51122,51123,51124,51125,51126,51127,51128,51129,51130,51131,51132,51133,51134,51135,51136,51137,51138,51139,51140,51141,51142,51143,51144,51145,51146,51147,51148,51149,51150,51151,51152,51153,51154,51155,51156,51157,51158,51159,51160,51161,51162,51163,51164,51165,51166,51167,51168,51169,51170,51171,51172,51173,51174,51175,51176,51177,51178,51179,51180,51181,51182,51183,51184,51185,51186,51187,51188,51189,51190,51191,51192,51193,51194,51195,51196,51197,51198,51264,51265,51266,51267,51268,51269,51270,51271,51272,51273,51274,51275,51276,51277,51278,51279,51280,51281,51282,51283,51284,51285,51286,51287,51288,51289,51290,51291,51292,51293,51294,51295,51296,51297,51298,51299,51300,51301,51302,51303,51304,51305,51306,51307,51308,51309,51310,51311,51312,51313,51314,51315,51316,51317,51318,51319,51320,51321,51322,51323,51324,51325,51326,51361,51362,51363,51364,51365,51366,51367,51368,51369,51370,51371,51372,51373,51374,51375,51376,51377,51378,51379,51380,51381,51382,51383,51384,51385,51386,51387,51388,51389,51390,51391,51392,51393,51394,51395,51396,51397,51398,51399,51400,51401,51402,51403,51404,51405,51406,51407,51408,51409,51410,51411,51412,51413,51414,51415,51416,51417,51418,51419,51420,51421,51422,51423,51424,51425,51426,51427,51428,51429,51430,51431,51432,51433,51434,51435,51436,51437,51438,51439,51440,51441,51442,51443,51444,51445,51446,51447,51448,51449,51450,51451,51452,51453,51454,41954,41955,41956,41957,41958,41959,41960,41961,41962,41963,41964,41965,41966,41967,41968,41969,41970,41971,41972,41973,41974,41975,41976,41977,41978,41979,41980,41981,41982,50910,50911,  255,45146,42995,43182,47339,47046,42730,42361,49268,49268,43956,43767,46050,43360,50025,50414,50105,50650,49587,47986,50654,48342,44197,44879,44895,47272,47444,49252,46787,42842,50406,50410,50421,50813,46160,49373,49861,50352,43476,50110,50426,46169,44756,44790,44884,44454,43219,42830,46034,48859,50034,50284,48995,42705,50346,47288,47348,50515,48764,50767,47180,47187,47857,56183,49149,46016,48599,50018,42955,50594,50596,43107,48469,47343,47472,49747,47600,48339,45660,47740,45782,49500,44462,45255,42712,48126,44514,47191,47856,46553,45998,50602,52948,48342,49109,42150,47591,44003,45686,45735,42335,60840,43851,46175,42147,43619,48326,44993,45265,46827,44249,47277,48033,45566,43184,43080,44098,44377,45488,45732,43847,43234,33883,45543,49843,43133,48588,46705,49273,42854,42091,50022,44744,49775,50290,48731,50810,50258,48804,42063,48868,48890,63333,42622,48294,50634,48319,47783,47058,59043,40398,48493,49520,48635,48556,45939,49637,42563,42568,43900,44880,46581,48033,46919,43456,45513,49364,50094,49785,42319,52209,47591,49325,52400,44226,48380,45788,45794,47457,47475,50758,48098,43218,49831,50367,49653,46179,42054,47537,48228,42943,44742,48342,48978,49400,59236,49137,49267,46967,43199,48194,52440,44136,44153,47048,44891,44900,45752,44995,50174,42171,48302,46000,44507,45403,45663,48636,43999,46936,44767,45686,46761,42833,42575,48233,43510,42997,45561,43620,45690,46439,49065,63960,47308,43197,49911,45262,47044,42843,48973,48986,50345,38393,50668,50671,43596,45647,49531,42463,45761,45769,43692,43685,50129,42160,45049,43243,42177,43991,43485,49021,42614,44157,48329,49127,42726,44464,43171,47608,51530,56828,39031,39032,46831,39033,46264,39034,39035,59641,38333,44913,39036,44971,45755,47830,47476,47851,42704,39037,39038,39073,48593,39074,39075,46696,45987,39076,39077,39078,46778,47485,49245,50530,43862,47532,42827,43882,46804,43253,46044,47556,48825,37977,48805,48232,51531,44716,46190,48296,50023,45523,44106,46267,45558,44796,45658,47742,46414,40048,46426,47183,43712,44263,44264,44967,44970,44972,47831,47829,48476,44272,47200,48493,49063,49507,47216,43724,45028,40146,40146,46555,48581,46584,49110,49876,48043,50136,40160,46696,50168,50516,49239,41290,41303,41305,41307,41311,41312,41315,41316,41319,41320,41323,41324,41327,41328,41331,41332,41335,41336,41339,41340,41414,41415,41418,41419,41416,41417,41308,41293,41294,41295,41297,41298,41299,41300,41341,41342,41377,41378,41379,41380,41420,41421,41422,41438,41439,41440,41441,41442,41538,41548,41549,41550,41289,51408,41389,41539,41544,41390,37314,41309,41310,41391,41423,41281,41424,41284,41470,41647,41648,41649,41650,41651,41652,41653,41654,41655,41656,41287,41286,41429,41431,41430,41288,41545,41679,41680,41681,41682,41683,41684,41685,41686,41687,41688,41689,41690,41691,41692,41693,41694,41695,41696,41697,41698,41699,41700,41701,41702,41703,41704,50916,41536,50917,24096,41412,41387,41705,41706,41707,41708,41709,41710,41711,41712,41713,41714,41715,41716,41717,41718,41719,41720,41721,41722,41723,41724,41725,41726,41792,41793,41794,41795,41313,41301,41314,41443,51385,51386,51387,51388,51389,51390,51391,51392,51393,51394,51395,51396,51397,51398,51399,51400,51401,51402,51403,51404,51412,51413,51414,51415,51416,51417,51418,51419,51420,51421,51422,51423,51424,51425,51426,51427,51428,51429,51430,51431,51432,51433,51434,51435,51436,51437,51438,51439,51440,51441,51442,51443,51444,51445,51446,51447,51448,51449,51450,51451,51452,51453,51454,41542,41543,51405,41411,51406,41540,41592,41462,41460,41463,41461,41405,41395 };

unsigned short big5_to_unicode_keys[] = { 33088,33089,33090,33091,33092,33093,33094,33095,33096,33097,33098,33099,33100,33101,33102,33103,33104,33105,33106,33107,33108,33109,33110,33111,33112,33113,33114,33115,33116,33117,33118,33119,33120,33121,33122,33123,33124,33125,33126,33127,33128,33129,33130,33131,33132,33133,33134,33135,33136,33137,33138,33139,33140,33141,33142,33143,33144,33145,33146,33147,33148,33149,33150,33185,33186,33187,33188,33189,33190,33191,33192,33193,33194,33195,33196,33197,33198,33199,33200,33201,33202,33203,33204,33205,33206,33207,33208,33209,33210,33211,33212,33213,33214,33215,33216,33217,33218,33219,33220,33221,33222,33223,33224,33225,33226,33227,33228,33229,33230,33231,33232,33233,33234,33235,33236,33237,33238,33239,33240,33241,33242,33243,33244,33245,33246,33247,33248,33249,33250,33251,33252,33253,33254,33255,33256,33257,33258,33259,33260,33261,33262,33263,33264,33265,33266,33267,33268,33269,33270,33271,33272,33273,33274,33275,33276,33277,33278,33344,33345,33346,33347,33348,33349,33350,33351,33352,33353,33354,33355,33356,33357,33358,33359,33360,33361,33362,33363,33364,33365,33366,33367,33368,33369,33370,33371,33372,33373,33374,33375,33376,33377,33378,33379,33380,33381,33382,33383,33384,33385,33386,33387,33388,33389,33390,33391,33392,33393,33394,33395,33396,33397,33398,33399,33400,33401,33402,33403,33404,33405,33406,33441,33442,33443,33444,33445,33446,33447,33448,33449,33450,33451,33452,33453,33454,33455,33456,33457,33458,33459,33460,33461,33462,33463,33464,33465,33466,33467,33468,33469,33470,33471,33472,33473,33474,33475,33476,33477,33478,33479,33480,33481,33482,33483,33484,33485,33486,33487,33488,33489,33490,33491,33492,33493,33494,33495,33496,33497,33498,33499,33500,33501,33502,33503,33504,33505,33506,33507,33508,33509,33510,33511,33512,33513,33514,33515,33516,33517,33518,33519,33520,33521,33522,33523,33524,33525,33526,33527,33528,33529,33530,33531,33532,33533,33534,33600,33601,33602,33603,33604,33605,33606,33607,33608,33609,33610,33611,33612,33613,33614,33615,33616,33617,33618,33619,33620,33621,33622,33623,33624,33625,33626,33627,33628,33629,33630,33631,33632,33633,33634,33635,33636,33637,33638,33639,33640,33641,33642,33643,33644,33645,33646,33647,33648,33649,33650,33651,33652,33653,33654,33655,33656,33657,33658,33659,33660,33661,33662,33697,33698,33699,33700,33701,33702,33703,33704,33705,33706,33707,33708,33709,33710,33711,33712,33713,33714,33715,33716,33717,33718,33719,33720,33721,33722,33723,33724,33725,33726,33727,33728,33729,33730,33731,33732,33733,33734,33735,33736,33737,33738,33739,33740,33741,33742,33743,33744,33745,33746,33747,33748,33749,33750,33751,33752,33753,33754,33755,33756,33757,33758,33759,33760,33761,33762,33763,33764,33765,33766,33767,33768,33769,33770,33771,33772,33773,33774,33775,33776,33777,33778,33779,33780,33781,33782,33783,33784,33785,33786,33787,33788,33789,33790,33856,33857,33858,33859,33860,33861,33862,33863,33864,33865,33866,33867,33868,33869,33870,33871,33872,33873,33874,33875,33876,33877,33878,33879,33880,33881,33882,33883,33884,33885,33886,33887,33888,33889,33890,33891,33892,33893,33894,33895,33896,33897,33898,33899,33900,33901,33902,33903,33904,33905,33906,33907,33908,33909,33910,33911,33912,33913,33914,33915,33916,33917,33918,33953,33954,33955,33956,33957,33958,33959,33960,33961,33962,33963,33964,33965,33966,33967,33968,33969,33970,33971,33972,33973,33974,33975,33976,33977,33978,33979,33980,33981,33982,33983,33984,33985,33986,33987,33988,33989,33990,33991,33992,33993,33994,33995,33996,33997,33998,33999,34000,34001,34002,34003,34004,34005,34006,34007,34008,34009,34010,34011,34012,34013,34014,34015,34016,34017,34018,34019,34020,34021,34022,34023,34024,34025,34026,34027,34028,34029,34030,34031,34032,34033,34034,34035,34036,34037,34038,34039,34040,34041,34042,34043,34044,34045,34046,34112,34113,34114,34115,34116,34117,34118,34119,34120,34121,34122,34123,34124,34125,34126,34127,34128,34129,34130,34131,34132,34133,34134,34135,34136,34137,34138,34139,34140,34141,34142,34143,34144,34145,34146,34147,34148,34149,34150,34151,34152,34153,34154,34155,34156,34157,34158,34159,34160,34161,34162,34163,34164,34165,34166,34167,34168,34169,34170,34171,34172,34173,34174,34209,34210,34211,34212,34213,34214,34215,34216,34217,34218,34219,34220,34221,34222,34223,34224,34225,34226,34227,34228,34229,34230,34231,34232,34233,34234,34235,34236,34237,34238,34239,34240,34241,34242,34243,34244,34245,34246,34247,34248,34249,34250,34251,34252,34253,34254,34255,34256,34257,34258,34259,34260,34261,34262,34263,34264,34265,34266,34267,34268,34269,34270,34271,34272,34273,34274,34275,34276,34277,34278,34279,34280,34281,34282,34283,34284,34285,34286,34287,34288,34289,34290,34291,34292,34293,34294,34295,34296,34297,34298,34299,34300,34301,34302,34368,34369,34370,34371,34372,34373,34374,34375,34376,34377,34378,34379,34380,34381,34382,34383,34384,34385,34386,34387,34388,34389,34390,34391,34392,34393,34394,34395,34396,34397,34398,34399,34400,34401,34402,34403,34404,34405,34406,34407,34408,34409,34410,34411,34412,34413,34414,34415,34416,34417,34418,34419,34420,34421,34422,34423,34424,34425,34426,34427,34428,34429,34430,34465,34466,34467,34468,34469,34470,34471,34472,34473,34474,34475,34476,34477,34478,34479,34480,34481,34482,34483,34484,34485,34486,34487,34488,34489,34490,34491,34492,34493,34494,34495,34496,34497,34498,34499,34500,34501,34502,34503,34504,34505,34506,34507,34508,34509,34510,34511,34512,34513,34514,34515,34516,34517,34518,34519,34520,34521,34522,34523,34524,34525,34526,34527,34528,34529,34530,34531,34532,34533,34534,34535,34536,34537,34538,34539,34540,34541,34542,34543,34544,34545,34546,34547,34548,34549,34550,34551,34552,34553,34554,34555,34556,34557,34558,34624,34625,34626,34627,34628,34629,34630,34631,34632,34633,34634,34635,34636,34637,34638,34639,34640,34641,34642,34643,34644,34645,34646,34647,34648,34649,34650,34651,34652,34653,34654,34655,34656,34657,34658,34659,34660,34661,34662,34663,34664,34665,34666,34667,34668,34669,34670,34671,34672,34673,34674,34675,34676,34677,34678,34679,34680,34681,34682,34683,34684,34685,34686,34721,34722,34723,34724,34725,34726,34727,34728,34729,34730,34731,34732,34733,34734,34735,34736,34737,34738,34739,34740,34741,34742,34743,34744,34745,34746,34747,34748,34749,34750,34751,34752,34753,34754,34755,34756,34757,34758,34759,34760,34761,34762,34763,34764,34765,34766,34767,34768,34769,34770,34771,34772,34773,34774,34775,34776,34777,34778,34779,34780,34781,34782,34783,34784,34785,34786,34787,34788,34789,34790,34791,34792,34793,34794,34795,34796,34797,34798,34799,34800,34801,34802,34803,34804,34805,34806,34807,34808,34809,34810,34811,34812,34813,34814,34880,34881,34882,34883,34884,34885,34886,34887,34888,34889,34890,34891,34892,34893,34894,34895,34896,34897,34898,34899,34900,34901,34902,34903,34904,34905,34906,34907,34908,34909,34910,34911,34912,34913,34914,34915,34916,34917,34918,34919,34920,34921,34922,34923,34924,34925,34926,34927,34928,34929,34930,34931,34932,34933,34934,34935,34936,34937,34938,34939,34940,34941,34942,34977,34978,34979,34980,34981,34982,34983,34984,34985,34986,34987,34988,34989,34990,34991,34992,34993,34994,34995,34996,34997,34998,34999,35000,35001,35002,35003,35004,35005,35006,35007,35008,35009,35010,35011,35012,35013,35014,35015,35016,35017,35018,35019,35020,35021,35022,35023,35024,35025,35026,35027,35028,35029,35030,35031,35032,35033,35034,35035,35036,35037,35038,35039,35040,35041,35042,35043,35044,35045,35046,35047,35048,35049,35050,35051,35052,35053,35054,35055,35056,35057,35058,35059,35060,35061,35062,35063,35064,35065,35066,35067,35068,35069,35070,35136,35137,35138,35139,35140,35141,35142,35143,35144,35145,35146,35147,35148,35149,35150,35151,35152,35153,35154,35155,35156,35157,35158,35159,35160,35161,35162,35163,35164,35165,35166,35167,35168,35169,35170,35171,35172,35173,35174,35175,35176,35177,35178,35179,35180,35181,35182,35183,35184,35185,35186,35187,35188,35189,35190,35191,35192,35193,35194,35195,35196,35197,35198,35233,35234,35235,35236,35237,35238,35239,35240,35241,35242,35243,35244,35245,35246,35247,35248,35249,35250,35251,35252,35253,35254,35255,35256,35257,35258,35259,35260,35261,35262,35263,35264,35265,35266,35267,35268,35269,35270,35271,35272,35273,35274,35275,35276,35277,35278,35279,35280,35281,35282,35283,35284,35285,35286,35287,35288,35289,35290,35291,35292,35293,35294,35295,35296,35297,35298,35299,35300,35301,35302,35303,35304,35305,35306,35307,35308,35309,35310,35311,35312,35313,35314,35315,35316,35317,35318,35319,35320,35321,35322,35323,35324,35325,35326,35392,35393,35394,35395,35396,35397,35398,35399,35400,35401,35402,35403,35404,35405,35406,35407,35408,35409,35410,35411,35412,35413,35414,35415,35416,35417,35418,35419,35420,35421,35422,35423,35424,35425,35426,35427,35428,35429,35430,35431,35432,35433,35434,35435,35436,35437,35438,35439,35440,35441,35442,35443,35444,35445,35446,35447,35448,35449,35450,35451,35452,35453,35454,35489,35490,35491,35492,35493,35494,35495,35496,35497,35498,35499,35500,35501,35502,35503,35504,35505,35506,35507,35508,35509,35510,35511,35512,35513,35514,35515,35516,35517,35518,35519,35520,35521,35522,35523,35524,35525,35526,35527,35528,35529,35530,35531,35532,35533,35534,35535,35536,35537,35538,35539,35540,35541,35542,35543,35544,35545,35546,35547,35548,35549,35550,35551,35552,35553,35554,35555,35556,35557,35558,35559,35560,35561,35562,35563,35564,35565,35566,35567,35568,35569,35570,35571,35572,35573,35574,35575,35576,35577,35578,35579,35580,35581,35582,35648,35649,35650,35651,35652,35653,35654,35655,35656,35657,35658,35659,35660,35661,35662,35663,35664,35665,35666,35667,35668,35669,35670,35671,35672,35673,35674,35675,35676,35677,35678,35679,35680,35681,35682,35683,35684,35685,35686,35687,35688,35689,35690,35691,35692,35693,35694,35695,35696,35697,35698,35699,35700,35701,35702,35703,35704,35705,35706,35707,35708,35709,35710,35745,35746,35747,35748,35749,35750,35751,35752,35753,35754,35755,35756,35757,35758,35759,35760,35761,35762,35763,35764,35765,35766,35767,35768,35769,35770,35771,35772,35773,35774,35775,35776,35777,35778,35779,35780,35781,35782,35783,35784,35785,35786,35787,35788,35789,35790,35791,35792,35793,35794,35795,35796,35797,35798,35799,35800,35801,35802,35803,35804,35805,35806,35807,35808,35809,35810,35811,35812,35813,35814,35815,35816,35817,35818,35819,35820,35821,35822,35823,35824,35825,35826,35827,35828,35829,35830,35831,35832,35833,35834,35835,35836,35837,35838,35904,35905,35906,35907,35908,35909,35910,35911,35912,35913,35914,35915,35916,35917,35918,35919,35920,35921,35922,35923,35924,35925,35926,35927,35928,35929,35930,35931,35932,35933,35934,35935,35936,35937,35938,35939,35940,35941,35942,35943,35944,35945,35946,35947,35948,35949,35950,35951,35952,35953,35954,35955,35956,35957,35958,35959,35960,35961,35962,35963,35964,35965,35966,36001,36002,36003,36004,36005,36006,36007,36008,36009,36010,36011,36012,36013,36014,36015,36016,36017,36018,36019,36020,36021,36022,36023,36024,36025,36026,36027,36028,36029,36030,36031,36032,36033,36034,36035,36036,36037,36038,36039,36040,36041,36042,36043,36044,36045,36046,36047,36048,36049,36050,36051,36052,36053,36054,36055,36056,36057,36058,36059,36060,36061,36062,36063,36064,36065,36066,36067,36068,36069,36070,36071,36072,36073,36074,36075,36076,36077,36078,36079,36080,36081,36082,36083,36084,36085,36086,36087,36088,36089,36090,36091,36092,36093,36094,36160,36161,36162,36163,36164,36165,36166,36167,36168,36169,36170,36171,36172,36173,36174,36175,36176,36177,36178,36179,36180,36181,36182,36183,36184,36185,36186,36187,36188,36189,36190,36191,36192,36193,36194,36195,36196,36197,36198,36199,36200,36201,36202,36203,36204,36205,36206,36207,36208,36209,36210,36211,36212,36213,36214,36215,36216,36217,36218,36219,36220,36221,36222,36257,36258,36259,36260,36261,36262,36263,36264,36265,36266,36267,36268,36269,36270,36271,36272,36273,36274,36275,36276,36277,36278,36279,36280,36281,36282,36283,36284,36285,36286,36287,36288,36289,36290,36291,36292,36293,36294,36295,36296,36297,36298,36299,36300,36301,36302,36303,36304,36305,36306,36307,36308,36309,36310,36311,36312,36313,36314,36315,36316,36317,36318,36319,36320,36321,36322,36323,36324,36325,36326,36327,36328,36329,36330,36331,36332,36333,36334,36335,36336,36337,36338,36339,36340,36341,36342,36343,36344,36345,36346,36347,36348,36349,36350,36416,36417,36418,36419,36420,36421,36422,36423,36424,36425,36426,36427,36428,36429,36430,36431,36432,36433,36434,36435,36436,36437,36438,36439,36440,36441,36442,36443,36444,36445,36446,36447,36448,36449,36450,36451,36452,36453,36454,36455,36456,36457,36458,36459,36460,36461,36462,36463,36464,36465,36466,36467,36468,36469,36470,36471,36472,36473,36474,36475,36476,36477,36478,36513,36514,36515,36516,36517,36518,36519,36520,36521,36522,36523,36524,36525,36526,36527,36528,36529,36530,36531,36532,36533,36534,36535,36536,36537,36538,36539,36540,36541,36542,36543,36544,36545,36546,36547,36548,36549,36550,36551,36552,36553,36554,36555,36556,36557,36558,36559,36560,36561,36562,36563,36564,36565,36566,36567,36568,36569,36570,36571,36572,36573,36574,36575,36576,36577,36578,36579,36580,36581,36582,36583,36584,36585,36586,36587,36588,36589,36590,36591,36592,36593,36594,36595,36596,36597,36598,36599,36600,36601,36602,36603,36604,36605,36606,36672,36673,36674,36675,36676,36677,36678,36679,36680,36681,36682,36683,36684,36685,36686,36687,36688,36689,36690,36691,36692,36693,36694,36695,36696,36697,36698,36699,36700,36701,36702,36703,36704,36705,36706,36707,36708,36709,36710,36711,36712,36713,36714,36715,36716,36717,36718,36719,36720,36721,36722,36723,36724,36725,36726,36727,36728,36729,36730,36731,36732,36733,36734,36769,36770,36771,36772,36773,36774,36775,36776,36777,36778,36779,36780,36781,36782,36783,36784,36785,36786,36787,36788,36789,36790,36791,36792,36793,36794,36795,36796,36797,36798,36799,36800,36801,36802,36803,36804,36805,36806,36807,36808,36809,36810,36811,36812,36813,36814,36815,36816,36817,36818,36819,36820,36821,36822,36823,36824,36825,36826,36827,36828,36829,36830,36831,36832,36833,36834,36835,36836,36837,36838,36839,36840,36841,36842,36843,36844,36845,36846,36847,36848,36849,36850,36851,36852,36853,36854,36855,36856,36857,36858,36859,36860,36861,36862,36928,36929,36930,36931,36932,36933,36934,36935,36936,36937,36938,36939,36940,36941,36942,36943,36944,36945,36946,36947,36948,36949,36950,36951,36952,36953,36954,36955,36956,36957,36958,36959,36960,36961,36962,36963,36964,36965,36966,36967,36968,36969,36970,36971,36972,36973,36974,36975,36976,36977,36978,36979,36980,36981,36982,36983,36984,36985,36986,36987,36988,36989,36990,37025,37026,37027,37028,37029,37030,37031,37032,37033,37034,37035,37036,37037,37038,37039,37040,37041,37042,37043,37044,37045,37046,37047,37048,37049,37050,37051,37052,37053,37054,37055,37056,37057,37058,37059,37060,37061,37062,37063,37064,37065,37066,37067,37068,37069,37070,37071,37072,37073,37074,37075,37076,37077,37078,37079,37080,37081,37082,37083,37084,37085,37086,37087,37088,37089,37090,37091,37092,37093,37094,37095,37096,37097,37098,37099,37100,37101,37102,37103,37104,37105,37106,37107,37108,37109,37110,37111,37112,37113,37114,37115,37116,37117,37118,37184,37185,37186,37187,37188,37189,37190,37191,37192,37193,37194,37195,37196,37197,37198,37199,37200,37201,37202,37203,37204,37205,37206,37207,37208,37209,37210,37211,37212,37213,37214,37215,37216,37217,37218,37219,37220,37221,37222,37223,37224,37225,37226,37227,37228,37229,37230,37231,37232,37233,37234,37235,37236,37237,37238,37239,37240,37241,37242,37243,37244,37245,37246,37281,37282,37283,37284,37285,37286,37287,37288,37289,37290,37291,37292,37293,37294,37295,37296,37297,37298,37299,37300,37301,37302,37303,37304,37305,37306,37307,37308,37309,37310,37311,37312,37313,37314,37315,37316,37317,37318,37319,37320,37321,37322,37323,37324,37325,37326,37327,37328,37329,37330,37331,37332,37333,37334,37335,37336,37337,37338,37339,37340,37341,37342,37343,37344,37345,37346,37347,37348,37349,37350,37351,37352,37353,37354,37355,37356,37357,37358,37359,37360,37361,37362,37363,37364,37365,37366,37367,37368,37369,37370,37371,37372,37373,37374,37440,37441,37442,37443,37444,37445,37446,37447,37448,37449,37450,37451,37452,37453,37454,37455,37456,37457,37458,37459,37460,37461,37462,37463,37464,37465,37466,37467,37468,37469,37470,37471,37472,37473,37474,37475,37476,37477,37478,37479,37480,37481,37482,37483,37484,37485,37486,37487,37488,37489,37490,37491,37492,37493,37494,37495,37496,37497,37498,37499,37500,37501,37502,37537,37538,37539,37540,37541,37542,37543,37544,37545,37546,37547,37548,37549,37550,37551,37552,37553,37554,37555,37556,37557,37558,37559,37560,37561,37562,37563,37564,37565,37566,37567,37568,37569,37570,37571,37572,37573,37574,37575,37576,37577,37578,37579,37580,37581,37582,37583,37584,37585,37586,37587,37588,37589,37590,37591,37592,37593,37594,37595,37596,37597,37598,37599,37600,37601,37602,37603,37604,37605,37606,37607,37608,37609,37610,37611,37612,37613,37614,37615,37616,37617,37618,37619,37620,37621,37622,37623,37624,37625,37626,37627,37628,37629,37630,37696,37697,37698,37699,37700,37701,37702,37703,37704,37705,37706,37707,37708,37709,37710,37711,37712,37713,37714,37715,37716,37717,37718,37719,37720,37721,37722,37723,37724,37725,37726,37727,37728,37729,37730,37731,37732,37733,37734,37735,37736,37737,37738,37739,37740,37741,37742,37743,37744,37745,37746,37747,37748,37749,37750,37751,37752,37753,37754,37755,37756,37757,37758,37793,37794,37795,37796,37797,37798,37799,37800,37801,37802,37803,37804,37805,37806,37807,37808,37809,37810,37811,37812,37813,37814,37815,37816,37817,37818,37819,37820,37821,37822,37823,37824,37825,37826,37827,37828,37829,37830,37831,37832,37833,37834,37835,37836,37837,37838,37839,37840,37841,37842,37843,37844,37845,37846,37847,37848,37849,37850,37851,37852,37853,37854,37855,37856,37857,37858,37859,37860,37861,37862,37863,37864,37865,37866,37867,37868,37869,37870,37871,37872,37873,37874,37875,37876,37877,37878,37879,37880,37881,37882,37883,37884,37885,37886,37952,37953,37954,37955,37956,37957,37958,37959,37960,37961,37962,37963,37964,37965,37966,37967,37968,37969,37970,37971,37972,37973,37974,37975,37976,37977,37978,37979,37980,37981,37982,37983,37984,37985,37986,37987,37988,37989,37990,37991,37992,37993,37994,37995,37996,37997,37998,37999,38000,38001,38002,38003,38004,38005,38006,38007,38008,38009,38010,38011,38012,38013,38014,38049,38050,38051,38052,38053,38054,38055,38056,38057,38058,38059,38060,38061,38062,38063,38064,38065,38066,38067,38068,38069,38070,38071,38072,38073,38074,38075,38076,38077,38078,38079,38080,38081,38082,38083,38084,38085,38086,38087,38088,38089,38090,38091,38092,38093,38094,38095,38096,38097,38098,38099,38100,38101,38102,38103,38104,38105,38106,38107,38108,38109,38110,38111,38112,38113,38114,38115,38116,38117,38118,38119,38120,38121,38122,38123,38124,38125,38126,38127,38128,38129,38130,38131,38132,38133,38134,38135,38136,38137,38138,38139,38140,38141,38142,38208,38209,38210,38211,38212,38213,38214,38215,38216,38217,38218,38219,38220,38221,38222,38223,38224,38225,38226,38227,38228,38229,38230,38231,38232,38233,38234,38235,38236,38237,38238,38239,38240,38241,38242,38243,38244,38245,38246,38247,38248,38249,38250,38251,38252,38253,38254,38255,38256,38257,38258,38259,38260,38261,38262,38263,38264,38265,38266,38267,38268,38269,38270,38305,38306,38307,38308,38309,38310,38311,38312,38313,38314,38315,38316,38317,38318,38319,38320,38321,38322,38323,38324,38325,38326,38327,38328,38329,38330,38331,38332,38333,38334,38335,38336,38337,38338,38339,38340,38341,38342,38343,38344,38345,38346,38347,38348,38349,38350,38351,38352,38353,38354,38355,38356,38357,38358,38359,38360,38361,38362,38363,38364,38365,38366,38367,38368,38369,38370,38371,38372,38373,38374,38375,38376,38377,38378,38379,38380,38381,38382,38383,38384,38385,38386,38387,38388,38389,38390,38391,38392,38393,38394,38395,38396,38397,38398,38464,38465,38466,38467,38468,38469,38470,38471,38472,38473,38474,38475,38476,38477,38478,38479,38480,38481,38482,38483,38484,38485,38486,38487,38488,38489,38490,38491,38492,38493,38494,38495,38496,38497,38498,38499,38500,38501,38502,38503,38504,38505,38506,38507,38508,38509,38510,38511,38512,38513,38514,38515,38516,38517,38518,38519,38520,38521,38522,38523,38524,38525,38526,38561,38562,38563,38564,38565,38566,38567,38568,38569,38570,38571,38572,38573,38574,38575,38576,38577,38578,38579,38580,38581,38582,38583,38584,38585,38586,38587,38588,38589,38590,38591,38592,38593,38594,38595,38596,38597,38598,38599,38600,38601,38602,38603,38604,38605,38606,38607,38608,38609,38610,38611,38612,38613,38614,38615,38616,38617,38618,38619,38620,38621,38622,38623,38624,38625,38626,38627,38628,38629,38630,38631,38632,38633,38634,38635,38636,38637,38638,38639,38640,38641,38642,38643,38644,38645,38646,38647,38648,38649,38650,38651,38652,38653,38654,38720,38721,38722,38723,38724,38725,38726,38727,38728,38729,38730,38731,38732,38733,38734,38735,38736,38737,38738,38739,38740,38741,38742,38743,38744,38745,38746,38747,38748,38749,38750,38751,38752,38753,38754,38755,38756,38757,38758,38759,38760,38761,38762,38763,38764,38765,38766,38767,38768,38769,38770,38771,38772,38773,38774,38775,38776,38777,38778,38779,38780,38781,38782,38817,38818,38819,38820,38821,38822,38823,38824,38825,38826,38827,38828,38829,38830,38831,38832,38833,38834,38835,38836,38837,38838,38839,38840,38841,38842,38843,38844,38845,38846,38847,38848,38849,38850,38851,38852,38853,38854,38855,38856,38857,38858,38859,38860,38861,38862,38863,38864,38865,38866,38867,38868,38869,38870,38871,38872,38873,38874,38875,38876,38877,38878,38879,38880,38881,38882,38883,38884,38885,38886,38887,38888,38889,38890,38891,38892,38893,38894,38895,38896,38897,38898,38899,38900,38901,38902,38903,38904,38905,38906,38907,38908,38909,38910,38976,38977,38978,38979,38980,38981,38982,38983,38984,38985,38986,38987,38988,38989,38990,38991,38992,38993,38994,38995,38996,38997,38998,38999,39000,39001,39002,39003,39004,39005,39006,39007,39008,39009,39010,39011,39012,39013,39014,39015,39016,39017,39018,39019,39020,39021,39022,39023,39024,39025,39026,39027,39028,39029,39030,39031,39032,39033,39034,39035,39036,39037,39038,39073,39074,39075,39076,39077,39078,39079,39080,39081,39082,39083,39084,39085,39086,39087,39088,39089,39090,39091,39092,39093,39094,39095,39096,39097,39098,39099,39100,39101,39102,39103,39104,39105,39106,39107,39108,39109,39110,39111,39112,39113,39114,39115,39116,39117,39118,39119,39120,39121,39122,39123,39124,39125,39126,39127,39128,39129,39130,39131,39132,39133,39134,39135,39136,39137,39138,39139,39140,39141,39142,39143,39144,39145,39146,39147,39148,39149,39150,39151,39152,39153,39154,39155,39156,39157,39158,39159,39160,39161,39162,39163,39164,39165,39166,39232,39233,39234,39235,39236,39237,39238,39239,39240,39241,39242,39243,39244,39245,39246,39247,39248,39249,39250,39251,39252,39253,39254,39255,39256,39257,39258,39259,39260,39261,39262,39263,39264,39265,39266,39267,39268,39269,39270,39271,39272,39273,39274,39275,39276,39277,39278,39279,39280,39281,39282,39283,39284,39285,39286,39287,39288,39289,39290,39291,39292,39293,39294,39329,39330,39331,39332,39333,39334,39335,39336,39337,39338,39339,39340,39341,39342,39343,39344,39345,39346,39347,39348,39349,39350,39351,39352,39353,39354,39355,39356,39357,39358,39359,39360,39361,39362,39363,39364,39365,39366,39367,39368,39369,39370,39371,39372,39373,39374,39375,39376,39377,39378,39379,39380,39381,39382,39383,39384,39385,39386,39387,39388,39389,39390,39391,39392,39393,39394,39395,39396,39397,39398,39399,39400,39401,39402,39403,39404,39405,39406,39407,39408,39409,39410,39411,39412,39413,39414,39415,39416,39417,39418,39419,39420,39421,39422,39488,39489,39490,39491,39492,39493,39494,39495,39496,39497,39498,39499,39500,39501,39502,39503,39504,39505,39506,39507,39508,39509,39510,39511,39512,39513,39514,39515,39516,39517,39518,39519,39520,39521,39522,39523,39524,39525,39526,39527,39528,39529,39530,39531,39532,39533,39534,39535,39536,39537,39538,39539,39540,39541,39542,39543,39544,39545,39546,39547,39548,39549,39550,39585,39586,39587,39588,39589,39590,39591,39592,39593,39594,39595,39596,39597,39598,39599,39600,39601,39602,39603,39604,39605,39606,39607,39608,39609,39610,39611,39612,39613,39614,39615,39616,39617,39618,39619,39620,39621,39622,39623,39624,39625,39626,39627,39628,39629,39630,39631,39632,39633,39634,39635,39636,39637,39638,39639,39640,39641,39642,39643,39644,39645,39646,39647,39648,39649,39650,39651,39652,39653,39654,39655,39656,39657,39658,39659,39660,39661,39662,39663,39664,39665,39666,39667,39668,39669,39670,39671,39672,39673,39674,39675,39676,39677,39678,39744,39745,39746,39747,39748,39749,39750,39751,39752,39753,39754,39755,39756,39757,39758,39759,39760,39761,39762,39763,39764,39765,39766,39767,39768,39769,39770,39771,39772,39773,39774,39775,39776,39777,39778,39779,39780,39781,39782,39783,39784,39785,39786,39787,39788,39789,39790,39791,39792,39793,39794,39795,39796,39797,39798,39799,39800,39801,39802,39803,39804,39805,39806,39841,39842,39843,39844,39845,39846,39847,39848,39849,39850,39851,39852,39853,39854,39855,39856,39857,39858,39859,39860,39861,39862,39863,39864,39865,39866,39867,39868,39869,39870,39871,39872,39873,39874,39875,39876,39877,39878,39879,39880,39881,39882,39883,39884,39885,39886,39887,39888,39889,39890,39891,39892,39893,39894,39895,39896,39897,39898,39899,39900,39901,39902,39903,39904,39905,39906,39907,39908,39909,39910,39911,39912,39913,39914,39915,39916,39917,39918,39919,39920,39921,39922,39923,39924,39925,39926,39927,39928,39929,39930,39931,39932,39933,39934,40000,40001,40002,40003,40004,40005,40006,40007,40008,40009,40010,40011,40012,40013,40014,40015,40016,40017,40018,40019,40020,40021,40022,40023,40024,40025,40026,40027,40028,40029,40030,40031,40032,40033,40034,40035,40036,40037,40038,40039,40040,40041,40042,40043,40044,40045,40046,40047,40048,40049,40050,40051,40052,40053,40054,40055,40056,40057,40058,40059,40060,40061,40062,40097,40098,40099,40100,40101,40102,40103,40104,40105,40106,40107,40108,40109,40110,40111,40112,40113,40114,40115,40116,40117,40118,40119,40120,40121,40122,40123,40124,40125,40126,40127,40128,40129,40130,40131,40132,40133,40134,40135,40136,40137,40138,40139,40140,40141,40142,40143,40144,40145,40146,40147,40148,40149,40150,40151,40152,40153,40154,40155,40156,40157,40158,40159,40160,40161,40162,40163,40164,40165,40166,40167,40168,40169,40170,40171,40172,40173,40174,40175,40176,40177,40178,40179,40180,40181,40182,40183,40184,40185,40186,40187,40188,40189,40190,40256,40257,40258,40259,40260,40261,40262,40263,40264,40265,40266,40267,40268,40269,40270,40271,40272,40273,40274,40275,40276,40277,40278,40279,40280,40281,40282,40283,40284,40285,40286,40287,40288,40289,40290,40291,40292,40293,40294,40295,40296,40297,40298,40299,40300,40301,40302,40303,40304,40305,40306,40307,40308,40309,40310,40311,40312,40313,40314,40315,40316,40317,40318,40353,40354,40355,40356,40357,40358,40359,40360,40361,40362,40363,40364,40365,40366,40367,40368,40369,40370,40371,40372,40373,40374,40375,40376,40377,40378,40379,40380,40381,40382,40383,40384,40385,40386,40387,40388,40389,40390,40391,40392,40393,40394,40395,40396,40397,40398,40399,40400,40401,40402,40403,40404,40405,40406,40407,40408,40409,40410,40411,40412,40413,40414,40415,40416,40417,40418,40419,40420,40421,40422,40423,40424,40425,40426,40427,40428,40429,40430,40431,40432,40433,40434,40435,40436,40437,40438,40439,40440,40441,40442,40443,40444,40445,40446,40512,40513,40514,40515,40516,40517,40518,40519,40520,40521,40522,40523,40524,40525,40526,40527,40528,40529,40530,40531,40532,40533,40534,40535,40536,40537,40538,40539,40540,40541,40542,40543,40544,40545,40546,40547,40548,40549,40550,40551,40552,40553,40554,40555,40556,40557,40558,40559,40560,40561,40562,40563,40564,40565,40566,40567,40568,40569,40570,40571,40572,40573,40574,40609,40610,40611,40612,40613,40614,40615,40616,40617,40618,40619,40620,40621,40622,40623,40624,40625,40626,40627,40628,40629,40630,40631,40632,40633,40634,40635,40636,40637,40638,40639,40640,40641,40642,40643,40644,40645,40646,40647,40648,40649,40650,40651,40652,40653,40654,40655,40656,40657,40658,40659,40660,40661,40662,40663,40664,40665,40666,40667,40668,40669,40670,40671,40672,40673,40674,40675,40676,40677,40678,40679,40680,40681,40682,40683,40684,40685,40686,40687,40688,40689,40690,40691,40692,40693,40694,40695,40696,40697,40698,40699,40700,40701,40702,40768,40769,40770,40771,40772,40773,40774,40775,40776,40777,40778,40779,40780,40781,40782,40783,40784,40785,40786,40787,40788,40789,40790,40791,40792,40793,40794,40795,40796,40797,40798,40799,40800,40801,40802,40803,40804,40805,40806,40807,40808,40809,40810,40811,40812,40813,40814,40815,40816,40817,40818,40819,40820,40821,40822,40823,40824,40825,40826,40827,40828,40829,40830,40865,40866,40867,40868,40869,40870,40871,40872,40873,40874,40875,40876,40877,40878,40879,40880,40881,40882,40883,40884,40885,40886,40887,40888,40889,40890,40891,40892,40893,40894,40895,40896,40897,40898,40899,40900,40901,40902,40903,40904,40905,40906,40907,40908,40909,40910,40911,40912,40913,40914,40915,40916,40917,40918,40919,40920,40921,40922,40923,40924,40925,40926,40927,40928,40929,40930,40931,40932,40933,40934,40935,40936,40937,40938,40939,40940,40941,40942,40943,40944,40945,40946,40947,40948,40949,40950,40951,40952,40953,40954,40955,40956,40957,40958,41024,41025,41026,41027,41028,41029,41030,41031,41032,41033,41034,41035,41036,41037,41038,41039,41040,41041,41042,41043,41044,41045,41046,41047,41048,41049,41050,41051,41052,41053,41054,41055,41056,41057,41058,41059,41060,41061,41062,41063,41064,41065,41066,41067,41068,41069,41070,41071,41072,41073,41074,41075,41076,41077,41078,41079,41080,41081,41082,41083,41084,41085,41086,41121,41122,41123,41124,41125,41126,41127,41128,41129,41130,41131,41132,41133,41134,41135,41136,41137,41138,41139,41140,41141,41142,41143,41144,41145,41146,41147,41148,41149,41150,41151,41152,41153,41154,41155,41156,41157,41158,41159,41160,41161,41162,41163,41164,41165,41166,41167,41168,41169,41170,41171,41172,41173,41174,41175,41176,41177,41178,41179,41180,41181,41182,41183,41184,41185,41186,41187,41188,41189,41190,41191,41192,41193,41194,41195,41196,41197,41198,41199,41200,41201,41202,41203,41204,41205,41206,41207,41208,41209,41210,41211,41212,41213,41214,41280,41281,41282,41283,41284,41285,41286,41287,41288,41289,41290,41291,41292,41293,41294,41295,41296,41297,41298,41299,41300,41301,41302,41303,41304,41305,41306,41307,41308,41309,41310,41311,41312,41313,41314,41315,41316,41317,41318,41319,41320,41321,41322,41323,41324,41325,41326,41327,41328,41329,41330,41331,41332,41333,41334,41335,41336,41337,41338,41339,41340,41341,41342,41377,41378,41379,41380,41381,41382,41383,41384,41385,41386,41387,41388,41389,41390,41391,41392,41393,41394,41395,41396,41397,41398,41399,41400,41401,41402,41403,41404,41405,41406,41407,41408,41409,41410,41411,41412,41413,41414,41415,41416,41417,41418,41419,41420,41421,41422,41423,41424,41425,41426,41427,41428,41429,41430,41431,41432,41433,41434,41435,41436,41437,41438,41439,41440,41441,41442,41443,41444,41445,41446,41447,41448,41449,41450,41451,41452,41453,41454,41455,41456,41457,41458,41459,41460,41461,41462,41463,41464,41465,41466,41467,41468,41469,41470,41536,41537,41538,41539,41540,41541,41542,41543,41544,41545,41546,41547,41548,41549,41550,41551,41552,41553,41554,41555,41556,41557,41558,41559,41560,41561,41562,41563,41564,41565,41566,41567,41568,41569,41570,41571,41572,41573,41574,41575,41576,41577,41578,41579,41580,41581,41582,41583,41584,41585,41586,41587,41588,41589,41590,41591,41592,41593,41594,41595,41596,41597,41598,41633,41634,41635,41640,41641,41642,41643,41644,41645,41646,41647,41648,41649,41650,41651,41652,41653,41654,41655,41656,41657,41658,41659,41660,41661,41662,41663,41664,41665,41666,41667,41668,41669,41670,41671,41672,41673,41674,41675,41677,41679,41680,41681,41682,41683,41684,41685,41686,41687,41688,41689,41690,41691,41692,41693,41694,41695,41696,41697,41698,41699,41700,41701,41702,41703,41704,41705,41706,41707,41708,41709,41710,41711,41712,41713,41714,41715,41716,41717,41718,41719,41720,41721,41722,41723,41724,41725,41726,41792,41793,41794,41795,41796,41797,41798,41799,41800,41801,41802,41803,41804,41805,41806,41807,41808,41809,41810,41811,41812,41813,41814,41815,41816,41817,41818,41819,41820,41821,41822,41823,41824,41825,41826,41827,41828,41829,41830,41831,41832,41833,41834,41835,41836,41837,41838,41839,41840,41841,41842,41843,41844,41845,41846,41847,41848,41849,41850,41851,41852,41853,41854,41889,41890,41891,41892,41893,41894,41895,41896,41897,41898,41899,41900,41901,41902,41903,41904,41905,41906,41907,41908,41909,41910,41911,41912,41913,41914,41915,41916,41917,41918,41919,41953,42048,42049,42050,42051,42052,42053,42054,42055,42056,42057,42058,42059,42060,42061,42062,42063,42064,42065,42066,42067,42068,42069,42070,42071,42072,42073,42074,42075,42076,42077,42078,42079,42080,42081,42082,42083,42084,42085,42086,42087,42088,42089,42090,42091,42092,42093,42094,42095,42096,42097,42098,42099,42100,42101,42102,42103,42104,42105,42106,42107,42108,42109,42110,42145,42146,42147,42148,42149,42150,42151,42152,42153,42154,42155,42156,42157,42158,42159,42160,42161,42162,42163,42164,42165,42166,42167,42168,42169,42170,42171,42172,42173,42174,42175,42176,42177,42178,42179,42180,42181,42182,42183,42184,42185,42186,42187,42188,42189,42190,42191,42192,42193,42194,42195,42196,42197,42198,42199,42200,42201,42202,42203,42204,42205,42206,42207,42208,42209,42210,42211,42212,42213,42214,42215,42216,42217,42218,42219,42220,42221,42222,42223,42224,42225,42226,42227,42228,42229,42230,42231,42232,42233,42234,42235,42236,42237,42238,42304,42305,42306,42307,42308,42309,42310,42311,42312,42313,42314,42315,42316,42317,42318,42319,42320,42321,42322,42323,42324,42325,42326,42327,42328,42329,42330,42331,42332,42333,42334,42335,42336,42337,42338,42339,42340,42341,42342,42343,42344,42345,42346,42347,42348,42349,42350,42351,42352,42353,42354,42355,42356,42357,42358,42359,42360,42361,42362,42363,42364,42365,42366,42401,42402,42403,42404,42405,42406,42407,42408,42409,42410,42411,42412,42413,42414,42415,42416,42417,42418,42419,42420,42421,42422,42423,42424,42425,42426,42427,42428,42429,42430,42431,42432,42433,42434,42435,42436,42437,42438,42439,42440,42441,42442,42443,42444,42445,42446,42447,42448,42449,42450,42451,42452,42453,42454,42455,42456,42457,42458,42459,42460,42461,42462,42463,42464,42465,42466,42467,42468,42469,42470,42471,42472,42473,42474,42475,42476,42477,42478,42479,42480,42481,42482,42483,42484,42485,42486,42487,42488,42489,42490,42491,42492,42493,42494,42560,42561,42562,42563,42564,42565,42566,42567,42568,42569,42570,42571,42572,42573,42574,42575,42576,42577,42578,42579,42580,42581,42582,42583,42584,42585,42586,42587,42588,42589,42590,42591,42592,42593,42594,42595,42596,42597,42598,42599,42600,42601,42602,42603,42604,42605,42606,42607,42608,42609,42610,42611,42612,42613,42614,42615,42616,42617,42618,42619,42620,42621,42622,42657,42658,42659,42660,42661,42662,42663,42664,42665,42666,42667,42668,42669,42670,42671,42672,42673,42674,42675,42676,42677,42678,42679,42680,42681,42682,42683,42684,42685,42686,42687,42688,42689,42690,42691,42692,42693,42694,42695,42696,42697,42698,42699,42700,42701,42702,42703,42704,42705,42706,42707,42708,42709,42710,42711,42712,42713,42714,42715,42716,42717,42718,42719,42720,42721,42722,42723,42724,42725,42726,42727,42728,42729,42730,42731,42732,42733,42734,42735,42736,42737,42738,42739,42740,42741,42742,42743,42744,42745,42746,42747,42748,42749,42750,42816,42817,42818,42819,42820,42821,42822,42823,42824,42825,42826,42827,42828,42829,42830,42831,42832,42833,42834,42835,42836,42837,42838,42839,42840,42841,42842,42843,42844,42845,42846,42847,42848,42849,42850,42851,42852,42853,42854,42855,42856,42857,42858,42859,42860,42861,42862,42863,42864,42865,42866,42867,42868,42869,42870,42871,42872,42873,42874,42875,42876,42877,42878,42913,42914,42915,42916,42917,42918,42919,42920,42921,42922,42923,42924,42925,42926,42927,42928,42929,42930,42931,42932,42933,42934,42935,42936,42937,42938,42939,42940,42941,42942,42943,42944,42945,42946,42947,42948,42949,42950,42951,42952,42953,42954,42955,42956,42957,42958,42959,42960,42961,42962,42963,42964,42965,42966,42967,42968,42969,42970,42971,42972,42973,42974,42975,42976,42977,42978,42979,42980,42981,42982,42983,42984,42985,42986,42987,42988,42989,42990,42991,42992,42993,42994,42995,42996,42997,42998,42999,43000,43001,43002,43003,43004,43005,43006,43072,43073,43074,43075,43076,43077,43078,43079,43080,43081,43082,43083,43084,43085,43086,43087,43088,43089,43090,43091,43092,43093,43094,43095,43096,43097,43098,43099,43100,43101,43102,43103,43104,43105,43106,43107,43108,43109,43110,43111,43112,43113,43114,43115,43116,43117,43118,43119,43120,43121,43122,43123,43124,43125,43126,43127,43128,43129,43130,43131,43132,43133,43134,43169,43170,43171,43172,43173,43174,43175,43176,43177,43178,43179,43180,43181,43182,43183,43184,43185,43186,43187,43188,43189,43190,43191,43192,43193,43194,43195,43196,43197,43198,43199,43200,43201,43202,43203,43204,43205,43206,43207,43208,43209,43210,43211,43212,43213,43214,43215,43216,43217,43218,43219,43220,43221,43222,43223,43224,43225,43226,43227,43228,43229,43230,43231,43232,43233,43234,43235,43236,43237,43238,43239,43240,43241,43242,43243,43244,43245,43246,43247,43248,43249,43250,43251,43252,43253,43254,43255,43256,43257,43258,43259,43260,43261,43262,43328,43329,43330,43331,43332,43333,43334,43335,43336,43337,43338,43339,43340,43341,43342,43343,43344,43345,43346,43347,43348,43349,43350,43351,43352,43353,43354,43355,43356,43357,43358,43359,43360,43361,43362,43363,43364,43365,43366,43367,43368,43369,43370,43371,43372,43373,43374,43375,43376,43377,43378,43379,43380,43381,43382,43383,43384,43385,43386,43387,43388,43389,43390,43425,43426,43427,43428,43429,43430,43431,43432,43433,43434,43435,43436,43437,43438,43439,43440,43441,43442,43443,43444,43445,43446,43447,43448,43449,43450,43451,43452,43453,43454,43455,43456,43457,43458,43459,43460,43461,43462,43463,43464,43465,43466,43467,43468,43469,43470,43471,43472,43473,43474,43475,43476,43477,43478,43479,43480,43481,43482,43483,43484,43485,43486,43487,43488,43489,43490,43491,43492,43493,43494,43495,43496,43497,43498,43499,43500,43501,43502,43503,43504,43505,43506,43507,43508,43509,43510,43511,43512,43513,43514,43515,43516,43517,43518,43584,43585,43586,43587,43588,43589,43590,43591,43592,43593,43594,43595,43596,43597,43598,43599,43600,43601,43602,43603,43604,43605,43606,43607,43608,43609,43610,43611,43612,43613,43614,43615,43616,43617,43618,43619,43620,43621,43622,43623,43624,43625,43626,43627,43628,43629,43630,43631,43632,43633,43634,43635,43636,43637,43638,43639,43640,43641,43642,43643,43644,43645,43646,43681,43682,43683,43684,43685,43686,43687,43688,43689,43690,43691,43692,43693,43694,43695,43696,43697,43698,43699,43700,43701,43702,43703,43704,43705,43706,43707,43708,43709,43710,43711,43712,43713,43714,43715,43716,43717,43718,43719,43720,43721,43722,43723,43724,43725,43726,43727,43728,43729,43730,43731,43732,43733,43734,43735,43736,43737,43738,43739,43740,43741,43742,43743,43744,43745,43746,43747,43748,43749,43750,43751,43752,43753,43754,43755,43756,43757,43758,43759,43760,43761,43762,43763,43764,43765,43766,43767,43768,43769,43770,43771,43772,43773,43774,43840,43841,43842,43843,43844,43845,43846,43847,43848,43849,43850,43851,43852,43853,43854,43855,43856,43857,43858,43859,43860,43861,43862,43863,43864,43865,43866,43867,43868,43869,43870,43871,43872,43873,43874,43875,43876,43877,43878,43879,43880,43881,43882,43883,43884,43885,43886,43887,43888,43889,43890,43891,43892,43893,43894,43895,43896,43897,43898,43899,43900,43901,43902,43937,43938,43939,43940,43941,43942,43943,43944,43945,43946,43947,43948,43949,43950,43951,43952,43953,43954,43955,43956,43957,43958,43959,43960,43961,43962,43963,43964,43965,43966,43967,43968,43969,43970,43971,43972,43973,43974,43975,43976,43977,43978,43979,43980,43981,43982,43983,43984,43985,43986,43987,43988,43989,43990,43991,43992,43993,43994,43995,43996,43997,43998,43999,44000,44001,44002,44003,44004,44005,44006,44007,44008,44009,44010,44011,44012,44013,44014,44015,44016,44017,44018,44019,44020,44021,44022,44023,44024,44025,44026,44027,44028,44029,44030,44096,44097,44098,44099,44100,44101,44102,44103,44104,44105,44106,44107,44108,44109,44110,44111,44112,44113,44114,44115,44116,44117,44118,44119,44120,44121,44122,44123,44124,44125,44126,44127,44128,44129,44130,44131,44132,44133,44134,44135,44136,44137,44138,44139,44140,44141,44142,44143,44144,44145,44146,44147,44148,44149,44150,44151,44152,44153,44154,44155,44156,44157,44158,44193,44194,44195,44196,44197,44198,44199,44200,44201,44202,44203,44204,44205,44206,44207,44208,44209,44210,44211,44212,44213,44214,44215,44216,44217,44218,44219,44220,44221,44222,44223,44224,44225,44226,44227,44228,44229,44230,44231,44232,44233,44234,44235,44236,44237,44238,44239,44240,44241,44242,44243,44244,44245,44246,44247,44248,44249,44250,44251,44252,44253,44254,44255,44256,44257,44258,44259,44260,44261,44262,44263,44264,44265,44266,44267,44268,44269,44270,44271,44272,44273,44274,44275,44276,44277,44278,44279,44280,44281,44282,44283,44284,44285,44286,44352,44353,44354,44355,44356,44357,44358,44359,44360,44361,44362,44363,44364,44365,44366,44367,44368,44369,44370,44371,44372,44373,44374,44375,44376,44377,44378,44379,44380,44381,44382,44383,44384,44385,44386,44387,44388,44389,44390,44391,44392,44393,44394,44395,44396,44397,44398,44399,44400,44401,44402,44403,44404,44405,44406,44407,44408,44409,44410,44411,44412,44413,44414,44449,44450,44451,44452,44453,44454,44455,44456,44457,44458,44459,44460,44461,44462,44463,44464,44465,44466,44467,44468,44469,44470,44471,44472,44473,44474,44475,44476,44477,44478,44479,44480,44481,44482,44483,44484,44485,44486,44487,44488,44489,44490,44491,44492,44493,44494,44495,44496,44497,44498,44499,44500,44501,44502,44503,44504,44505,44506,44507,44508,44509,44510,44511,44512,44513,44514,44515,44516,44517,44518,44519,44520,44521,44522,44523,44524,44525,44526,44527,44528,44529,44530,44531,44532,44533,44534,44535,44536,44537,44538,44539,44540,44541,44542,44608,44609,44610,44611,44612,44613,44614,44615,44616,44617,44618,44619,44620,44621,44622,44623,44624,44625,44626,44627,44628,44629,44630,44631,44632,44633,44634,44635,44636,44637,44638,44639,44640,44641,44642,44643,44644,44645,44646,44647,44648,44649,44650,44651,44652,44653,44654,44655,44656,44657,44658,44659,44660,44661,44662,44663,44664,44665,44666,44667,44668,44669,44670,44705,44706,44707,44708,44709,44710,44711,44712,44713,44714,44715,44716,44717,44718,44719,44720,44721,44722,44723,44724,44725,44726,44727,44728,44729,44730,44731,44732,44733,44734,44735,44736,44737,44738,44739,44740,44741,44742,44743,44744,44745,44746,44747,44748,44749,44750,44751,44752,44753,44754,44755,44756,44757,44758,44759,44760,44761,44762,44763,44764,44765,44766,44767,44768,44769,44770,44771,44772,44773,44774,44775,44776,44777,44778,44779,44780,44781,44782,44783,44784,44785,44786,44787,44788,44789,44790,44791,44792,44793,44794,44795,44796,44797,44798,44864,44865,44866,44867,44868,44869,44870,44871,44872,44873,44874,44875,44876,44877,44878,44879,44880,44881,44882,44883,44884,44885,44886,44887,44888,44889,44890,44891,44892,44893,44894,44895,44896,44897,44898,44899,44900,44901,44902,44903,44904,44905,44906,44907,44908,44909,44910,44911,44912,44913,44914,44915,44916,44917,44918,44919,44920,44921,44922,44923,44924,44925,44926,44961,44962,44963,44964,44965,44966,44967,44968,44969,44970,44971,44972,44973,44974,44975,44976,44977,44978,44979,44980,44981,44982,44983,44984,44985,44986,44987,44988,44989,44990,44991,44992,44993,44994,44995,44996,44997,44998,44999,45000,45001,45002,45003,45004,45005,45006,45007,45008,45009,45010,45011,45012,45013,45014,45015,45016,45017,45018,45019,45020,45021,45022,45023,45024,45025,45026,45027,45028,45029,45030,45031,45032,45033,45034,45035,45036,45037,45038,45039,45040,45041,45042,45043,45044,45045,45046,45047,45048,45049,45050,45051,45052,45053,45054,45120,45121,45122,45123,45124,45125,45126,45127,45128,45129,45130,45131,45132,45133,45134,45135,45136,45137,45138,45139,45140,45141,45142,45143,45144,45145,45146,45147,45148,45149,45150,45151,45152,45153,45154,45155,45156,45157,45158,45159,45160,45161,45162,45163,45164,45165,45166,45167,45168,45169,45170,45171,45172,45173,45174,45175,45176,45177,45178,45179,45180,45181,45182,45217,45218,45219,45220,45221,45222,45223,45224,45225,45226,45227,45228,45229,45230,45231,45232,45233,45234,45235,45236,45237,45238,45239,45240,45241,45242,45243,45244,45245,45246,45247,45248,45249,45250,45251,45252,45253,45254,45255,45256,45257,45258,45259,45260,45261,45262,45263,45264,45265,45266,45267,45268,45269,45270,45271,45272,45273,45274,45275,45276,45277,45278,45279,45280,45281,45282,45283,45284,45285,45286,45287,45288,45289,45290,45291,45292,45293,45294,45295,45296,45297,45298,45299,45300,45301,45302,45303,45304,45305,45306,45307,45308,45309,45310,45376,45377,45378,45379,45380,45381,45382,45383,45384,45385,45386,45387,45388,45389,45390,45391,45392,45393,45394,45395,45396,45397,45398,45399,45400,45401,45402,45403,45404,45405,45406,45407,45408,45409,45410,45411,45412,45413,45414,45415,45416,45417,45418,45419,45420,45421,45422,45423,45424,45425,45426,45427,45428,45429,45430,45431,45432,45433,45434,45435,45436,45437,45438,45473,45474,45475,45476,45477,45478,45479,45480,45481,45482,45483,45484,45485,45486,45487,45488,45489,45490,45491,45492,45493,45494,45495,45496,45497,45498,45499,45500,45501,45502,45503,45504,45505,45506,45507,45508,45509,45510,45511,45512,45513,45514,45515,45516,45517,45518,45519,45520,45521,45522,45523,45524,45525,45526,45527,45528,45529,45530,45531,45532,45533,45534,45535,45536,45537,45538,45539,45540,45541,45542,45543,45544,45545,45546,45547,45548,45549,45550,45551,45552,45553,45554,45555,45556,45557,45558,45559,45560,45561,45562,45563,45564,45565,45566,45632,45633,45634,45635,45636,45637,45638,45639,45640,45641,45642,45643,45644,45645,45646,45647,45648,45649,45650,45651,45652,45653,45654,45655,45656,45657,45658,45659,45660,45661,45662,45663,45664,45665,45666,45667,45668,45669,45670,45671,45672,45673,45674,45675,45676,45677,45678,45679,45680,45681,45682,45683,45684,45685,45686,45687,45688,45689,45690,45691,45692,45693,45694,45729,45730,45731,45732,45733,45734,45735,45736,45737,45738,45739,45740,45741,45742,45743,45744,45745,45746,45747,45748,45749,45750,45751,45752,45753,45754,45755,45756,45757,45758,45759,45760,45761,45762,45763,45764,45765,45766,45767,45768,45769,45770,45771,45772,45773,45774,45775,45776,45777,45778,45779,45780,45781,45782,45783,45784,45785,45786,45787,45788,45789,45790,45791,45792,45793,45794,45795,45796,45797,45798,45799,45800,45801,45802,45803,45804,45805,45806,45807,45808,45809,45810,45811,45812,45813,45814,45815,45816,45817,45818,45819,45820,45821,45822,45888,45889,45890,45891,45892,45893,45894,45895,45896,45897,45898,45899,45900,45901,45902,45903,45904,45905,45906,45907,45908,45909,45910,45911,45912,45913,45914,45915,45916,45917,45918,45919,45920,45921,45922,45923,45924,45925,45926,45927,45928,45929,45930,45931,45932,45933,45934,45935,45936,45937,45938,45939,45940,45941,45942,45943,45944,45945,45946,45947,45948,45949,45950,45985,45986,45987,45988,45989,45990,45991,45992,45993,45994,45995,45996,45997,45998,45999,46000,46001,46002,46003,46004,46005,46006,46007,46008,46009,46010,46011,46012,46013,46014,46015,46016,46017,46018,46019,46020,46021,46022,46023,46024,46025,46026,46027,46028,46029,46030,46031,46032,46033,46034,46035,46036,46037,46038,46039,46040,46041,46042,46043,46044,46045,46046,46047,46048,46049,46050,46051,46052,46053,46054,46055,46056,46057,46058,46059,46060,46061,46062,46063,46064,46065,46066,46067,46068,46069,46070,46071,46072,46073,46074,46075,46076,46077,46078,46144,46145,46146,46147,46148,46149,46150,46151,46152,46153,46154,46155,46156,46157,46158,46159,46160,46161,46162,46163,46164,46165,46166,46167,46168,46169,46170,46171,46172,46173,46174,46175,46176,46177,46178,46179,46180,46181,46182,46183,46184,46185,46186,46187,46188,46189,46190,46191,46192,46193,46194,46195,46196,46197,46198,46199,46200,46201,46202,46203,46204,46205,46206,46241,46242,46243,46244,46245,46246,46247,46248,46249,46250,46251,46252,46253,46254,46255,46256,46257,46258,46259,46260,46261,46262,46263,46264,46265,46266,46267,46268,46269,46270,46271,46272,46273,46274,46275,46276,46277,46278,46279,46280,46281,46282,46283,46284,46285,46286,46287,46288,46289,46290,46291,46292,46293,46294,46295,46296,46297,46298,46299,46300,46301,46302,46303,46304,46305,46306,46307,46308,46309,46310,46311,46312,46313,46314,46315,46316,46317,46318,46319,46320,46321,46322,46323,46324,46325,46326,46327,46328,46329,46330,46331,46332,46333,46334,46400,46401,46402,46403,46404,46405,46406,46407,46408,46409,46410,46411,46412,46413,46414,46415,46416,46417,46418,46419,46420,46421,46422,46423,46424,46425,46426,46427,46428,46429,46430,46431,46432,46433,46434,46435,46436,46437,46438,46439,46440,46441,46442,46443,46444,46445,46446,46447,46448,46449,46450,46451,46452,46453,46454,46455,46456,46457,46458,46459,46460,46461,46462,46497,46498,46499,46500,46501,46502,46503,46504,46505,46506,46507,46508,46509,46510,46511,46512,46513,46514,46515,46516,46517,46518,46519,46520,46521,46522,46523,46524,46525,46526,46527,46528,46529,46530,46531,46532,46533,46534,46535,46536,46537,46538,46539,46540,46541,46542,46543,46544,46545,46546,46547,46548,46549,46550,46551,46552,46553,46554,46555,46556,46557,46558,46559,46560,46561,46562,46563,46564,46565,46566,46567,46568,46569,46570,46571,46572,46573,46574,46575,46576,46577,46578,46579,46580,46581,46582,46583,46584,46585,46586,46587,46588,46589,46590,46656,46657,46658,46659,46660,46661,46662,46663,46664,46665,46666,46667,46668,46669,46670,46671,46672,46673,46674,46675,46676,46677,46678,46679,46680,46681,46682,46683,46684,46685,46686,46687,46688,46689,46690,46691,46692,46693,46694,46695,46696,46697,46698,46699,46700,46701,46702,46703,46704,46705,46706,46707,46708,46709,46710,46711,46712,46713,46714,46715,46716,46717,46718,46753,46754,46755,46756,46757,46758,46759,46760,46761,46762,46763,46764,46765,46766,46767,46768,46769,46770,46771,46772,46773,46774,46775,46776,46777,46778,46779,46780,46781,46782,46783,46784,46785,46786,46787,46788,46789,46790,46791,46792,46793,46794,46795,46796,46797,46798,46799,46800,46801,46802,46803,46804,46805,46806,46807,46808,46809,46810,46811,46812,46813,46814,46815,46816,46817,46818,46819,46820,46821,46822,46823,46824,46825,46826,46827,46828,46829,46830,46831,46832,46833,46834,46835,46836,46837,46838,46839,46840,46841,46842,46843,46844,46845,46846,46912,46913,46914,46915,46916,46917,46918,46919,46920,46921,46922,46923,46924,46925,46926,46927,46928,46929,46930,46931,46932,46933,46934,46935,46936,46937,46938,46939,46940,46941,46942,46943,46944,46945,46946,46947,46948,46949,46950,46951,46952,46953,46954,46955,46956,46957,46958,46959,46960,46961,46962,46963,46964,46965,46966,46967,46968,46969,46970,46971,46972,46973,46974,47009,47010,47011,47012,47013,47014,47015,47016,47017,47018,47019,47020,47021,47022,47023,47024,47025,47026,47027,47028,47029,47030,47031,47032,47033,47034,47035,47036,47037,47038,47039,47040,47041,47042,47043,47044,47045,47046,47047,47048,47049,47050,47051,47052,47053,47054,47055,47056,47057,47058,47059,47060,47061,47062,47063,47064,47065,47066,47067,47068,47069,47070,47071,47072,47073,47074,47075,47076,47077,47078,47079,47080,47081,47082,47083,47084,47085,47086,47087,47088,47089,47090,47091,47092,47093,47094,47095,47096,47097,47098,47099,47100,47101,47102,47168,47169,47170,47171,47172,47173,47174,47175,47176,47177,47178,47179,47180,47181,47182,47183,47184,47185,47186,47187,47188,47189,47190,47191,47192,47193,47194,47195,47196,47197,47198,47199,47200,47201,47202,47203,47204,47205,47206,47207,47208,47209,47210,47211,47212,47213,47214,47215,47216,47217,47218,47219,47220,47221,47222,47223,47224,47225,47226,47227,47228,47229,47230,47265,47266,47267,47268,47269,47270,47271,47272,47273,47274,47275,47276,47277,47278,47279,47280,47281,47282,47283,47284,47285,47286,47287,47288,47289,47290,47291,47292,47293,47294,47295,47296,47297,47298,47299,47300,47301,47302,47303,47304,47305,47306,47307,47308,47309,47310,47311,47312,47313,47314,47315,47316,47317,47318,47319,47320,47321,47322,47323,47324,47325,47326,47327,47328,47329,47330,47331,47332,47333,47334,47335,47336,47337,47338,47339,47340,47341,47342,47343,47344,47345,47346,47347,47348,47349,47350,47351,47352,47353,47354,47355,47356,47357,47358,47424,47425,47426,47427,47428,47429,47430,47431,47432,47433,47434,47435,47436,47437,47438,47439,47440,47441,47442,47443,47444,47445,47446,47447,47448,47449,47450,47451,47452,47453,47454,47455,47456,47457,47458,47459,47460,47461,47462,47463,47464,47465,47466,47467,47468,47469,47470,47471,47472,47473,47474,47475,47476,47477,47478,47479,47480,47481,47482,47483,47484,47485,47486,47521,47522,47523,47524,47525,47526,47527,47528,47529,47530,47531,47532,47533,47534,47535,47536,47537,47538,47539,47540,47541,47542,47543,47544,47545,47546,47547,47548,47549,47550,47551,47552,47553,47554,47555,47556,47557,47558,47559,47560,47561,47562,47563,47564,47565,47566,47567,47568,47569,47570,47571,47572,47573,47574,47575,47576,47577,47578,47579,47580,47581,47582,47583,47584,47585,47586,47587,47588,47589,47590,47591,47592,47593,47594,47595,47596,47597,47598,47599,47600,47601,47602,47603,47604,47605,47606,47607,47608,47609,47610,47611,47612,47613,47614,47680,47681,47682,47683,47684,47685,47686,47687,47688,47689,47690,47691,47692,47693,47694,47695,47696,47697,47698,47699,47700,47701,47702,47703,47704,47705,47706,47707,47708,47709,47710,47711,47712,47713,47714,47715,47716,47717,47718,47719,47720,47721,47722,47723,47724,47725,47726,47727,47728,47729,47730,47731,47732,47733,47734,47735,47736,47737,47738,47739,47740,47741,47742,47777,47778,47779,47780,47781,47782,47783,47784,47785,47786,47787,47788,47789,47790,47791,47792,47793,47794,47795,47796,47797,47798,47799,47800,47801,47802,47803,47804,47805,47806,47807,47808,47809,47810,47811,47812,47813,47814,47815,47816,47817,47818,47819,47820,47821,47822,47823,47824,47825,47826,47827,47828,47829,47830,47831,47832,47833,47834,47835,47836,47837,47838,47839,47840,47841,47842,47843,47844,47845,47846,47847,47848,47849,47850,47851,47852,47853,47854,47855,47856,47857,47858,47859,47860,47861,47862,47863,47864,47865,47866,47867,47868,47869,47870,47936,47937,47938,47939,47940,47941,47942,47943,47944,47945,47946,47947,47948,47949,47950,47951,47952,47953,47954,47955,47956,47957,47958,47959,47960,47961,47962,47963,47964,47965,47966,47967,47968,47969,47970,47971,47972,47973,47974,47975,47976,47977,47978,47979,47980,47981,47982,47983,47984,47985,47986,47987,47988,47989,47990,47991,47992,47993,47994,47995,47996,47997,47998,48033,48034,48035,48036,48037,48038,48039,48040,48041,48042,48043,48044,48045,48046,48047,48048,48049,48050,48051,48052,48053,48054,48055,48056,48057,48058,48059,48060,48061,48062,48063,48064,48065,48066,48067,48068,48069,48070,48071,48072,48073,48074,48075,48076,48077,48078,48079,48080,48081,48082,48083,48084,48085,48086,48087,48088,48089,48090,48091,48092,48093,48094,48095,48096,48097,48098,48099,48100,48101,48102,48103,48104,48105,48106,48107,48108,48109,48110,48111,48112,48113,48114,48115,48116,48117,48118,48119,48120,48121,48122,48123,48124,48125,48126,48192,48193,48194,48195,48196,48197,48198,48199,48200,48201,48202,48203,48204,48205,48206,48207,48208,48209,48210,48211,48212,48213,48214,48215,48216,48217,48218,48219,48220,48221,48222,48223,48224,48225,48226,48227,48228,48229,48230,48231,48232,48233,48234,48235,48236,48237,48238,48239,48240,48241,48242,48243,48244,48245,48246,48247,48248,48249,48250,48251,48252,48253,48254,48289,48290,48291,48292,48293,48294,48295,48296,48297,48298,48299,48300,48301,48302,48303,48304,48305,48306,48307,48308,48309,48310,48311,48312,48313,48314,48315,48316,48317,48318,48319,48320,48321,48322,48323,48324,48325,48326,48327,48328,48329,48330,48331,48332,48333,48334,48335,48336,48337,48338,48339,48340,48341,48342,48343,48344,48345,48346,48347,48348,48349,48350,48351,48352,48353,48354,48355,48356,48357,48358,48359,48360,48361,48362,48363,48364,48365,48366,48367,48368,48369,48370,48371,48372,48373,48374,48375,48376,48377,48378,48379,48380,48381,48382,48448,48449,48450,48451,48452,48453,48454,48455,48456,48457,48458,48459,48460,48461,48462,48463,48464,48465,48466,48467,48468,48469,48470,48471,48472,48473,48474,48475,48476,48477,48478,48479,48480,48481,48482,48483,48484,48485,48486,48487,48488,48489,48490,48491,48492,48493,48494,48495,48496,48497,48498,48499,48500,48501,48502,48503,48504,48505,48506,48507,48508,48509,48510,48545,48546,48547,48548,48549,48550,48551,48552,48553,48554,48555,48556,48557,48558,48559,48560,48561,48562,48563,48564,48565,48566,48567,48568,48569,48570,48571,48572,48573,48574,48575,48576,48577,48578,48579,48580,48581,48582,48583,48584,48585,48586,48587,48588,48589,48590,48591,48592,48593,48594,48595,48596,48597,48598,48599,48600,48601,48602,48603,48604,48605,48606,48607,48608,48609,48610,48611,48612,48613,48614,48615,48616,48617,48618,48619,48620,48621,48622,48623,48624,48625,48626,48627,48628,48629,48630,48631,48632,48633,48634,48635,48636,48637,48638,48704,48705,48706,48707,48708,48709,48710,48711,48712,48713,48714,48715,48716,48717,48718,48719,48720,48721,48722,48723,48724,48725,48726,48727,48728,48729,48730,48731,48732,48733,48734,48735,48736,48737,48738,48739,48740,48741,48742,48743,48744,48745,48746,48747,48748,48749,48750,48751,48752,48753,48754,48755,48756,48757,48758,48759,48760,48761,48762,48763,48764,48765,48766,48801,48802,48803,48804,48805,48806,48807,48808,48809,48810,48811,48812,48813,48814,48815,48816,48817,48818,48819,48820,48821,48822,48823,48824,48825,48826,48827,48828,48829,48830,48831,48832,48833,48834,48835,48836,48837,48838,48839,48840,48841,48842,48843,48844,48845,48846,48847,48848,48849,48850,48851,48852,48853,48854,48855,48856,48857,48858,48859,48860,48861,48862,48863,48864,48865,48866,48867,48868,48869,48870,48871,48872,48873,48874,48875,48876,48877,48878,48879,48880,48881,48882,48883,48884,48885,48886,48887,48888,48889,48890,48891,48892,48893,48894,48960,48961,48962,48963,48964,48965,48966,48967,48968,48969,48970,48971,48972,48973,48974,48975,48976,48977,48978,48979,48980,48981,48982,48983,48984,48985,48986,48987,48988,48989,48990,48991,48992,48993,48994,48995,48996,48997,48998,48999,49000,49001,49002,49003,49004,49005,49006,49007,49008,49009,49010,49011,49012,49013,49014,49015,49016,49017,49018,49019,49020,49021,49022,49057,49058,49059,49060,49061,49062,49063,49064,49065,49066,49067,49068,49069,49070,49071,49072,49073,49074,49075,49076,49077,49078,49079,49080,49081,49082,49083,49084,49085,49086,49087,49088,49089,49090,49091,49092,49093,49094,49095,49096,49097,49098,49099,49100,49101,49102,49103,49104,49105,49106,49107,49108,49109,49110,49111,49112,49113,49114,49115,49116,49117,49118,49119,49120,49121,49122,49123,49124,49125,49126,49127,49128,49129,49130,49131,49132,49133,49134,49135,49136,49137,49138,49139,49140,49141,49142,49143,49144,49145,49146,49147,49148,49149,49150,49216,49217,49218,49219,49220,49221,49222,49223,49224,49225,49226,49227,49228,49229,49230,49231,49232,49233,49234,49235,49236,49237,49238,49239,49240,49241,49242,49243,49244,49245,49246,49247,49248,49249,49250,49251,49252,49253,49254,49255,49256,49257,49258,49259,49260,49261,49262,49263,49264,49265,49266,49267,49268,49269,49270,49271,49272,49273,49274,49275,49276,49277,49278,49313,49314,49315,49316,49317,49318,49319,49320,49321,49322,49323,49324,49325,49326,49327,49328,49329,49330,49331,49332,49333,49334,49335,49336,49337,49338,49339,49340,49341,49342,49343,49344,49345,49346,49347,49348,49349,49350,49351,49352,49353,49354,49355,49356,49357,49358,49359,49360,49361,49362,49363,49364,49365,49366,49367,49368,49369,49370,49371,49372,49373,49374,49375,49376,49377,49378,49379,49380,49381,49382,49383,49384,49385,49386,49387,49388,49389,49390,49391,49392,49393,49394,49395,49396,49397,49398,49399,49400,49401,49402,49403,49404,49405,49406,49472,49473,49474,49475,49476,49477,49478,49479,49480,49481,49482,49483,49484,49485,49486,49487,49488,49489,49490,49491,49492,49493,49494,49495,49496,49497,49498,49499,49500,49501,49502,49503,49504,49505,49506,49507,49508,49509,49510,49511,49512,49513,49514,49515,49516,49517,49518,49519,49520,49521,49522,49523,49524,49525,49526,49527,49528,49529,49530,49531,49532,49533,49534,49569,49570,49571,49572,49573,49574,49575,49576,49577,49578,49579,49580,49581,49582,49583,49584,49585,49586,49587,49588,49589,49590,49591,49592,49593,49594,49595,49596,49597,49598,49599,49600,49601,49602,49603,49604,49605,49606,49607,49608,49609,49610,49611,49612,49613,49614,49615,49616,49617,49618,49619,49620,49621,49622,49623,49624,49625,49626,49627,49628,49629,49630,49631,49632,49633,49634,49635,49636,49637,49638,49639,49640,49641,49642,49643,49644,49645,49646,49647,49648,49649,49650,49651,49652,49653,49654,49655,49656,49657,49658,49659,49660,49661,49662,49728,49729,49730,49731,49732,49733,49734,49735,49736,49737,49738,49739,49740,49741,49742,49743,49744,49745,49746,49747,49748,49749,49750,49751,49752,49753,49754,49755,49756,49757,49758,49759,49760,49761,49762,49763,49764,49765,49766,49767,49768,49769,49770,49771,49772,49773,49774,49775,49776,49777,49778,49779,49780,49781,49782,49783,49784,49785,49786,49787,49788,49789,49790,49825,49826,49827,49828,49829,49830,49831,49832,49833,49834,49835,49836,49837,49838,49839,49840,49841,49842,49843,49844,49845,49846,49847,49848,49849,49850,49851,49852,49853,49854,49855,49856,49857,49858,49859,49860,49861,49862,49863,49864,49865,49866,49867,49868,49869,49870,49871,49872,49873,49874,49875,49876,49877,49878,49879,49880,49881,49882,49883,49884,49885,49886,49887,49888,49889,49890,49891,49892,49893,49894,49895,49896,49897,49898,49899,49900,49901,49902,49903,49904,49905,49906,49907,49908,49909,49910,49911,49912,49913,49914,49915,49916,49917,49918,49984,49985,49986,49987,49988,49989,49990,49991,49992,49993,49994,49995,49996,49997,49998,49999,50000,50001,50002,50003,50004,50005,50006,50007,50008,50009,50010,50011,50012,50013,50014,50015,50016,50017,50018,50019,50020,50021,50022,50023,50024,50025,50026,50027,50028,50029,50030,50031,50032,50033,50034,50035,50036,50037,50038,50039,50040,50041,50042,50043,50044,50045,50046,50081,50082,50083,50084,50085,50086,50087,50088,50089,50090,50091,50092,50093,50094,50095,50096,50097,50098,50099,50100,50101,50102,50103,50104,50105,50106,50107,50108,50109,50110,50111,50112,50113,50114,50115,50116,50117,50118,50119,50120,50121,50122,50123,50124,50125,50126,50127,50128,50129,50130,50131,50132,50133,50134,50135,50136,50137,50138,50139,50140,50141,50142,50143,50144,50145,50146,50147,50148,50149,50150,50151,50152,50153,50154,50155,50156,50157,50158,50159,50160,50161,50162,50163,50164,50165,50166,50167,50168,50169,50170,50171,50172,50173,50174,50240,50241,50242,50243,50244,50245,50246,50247,50248,50249,50250,50251,50252,50253,50254,50255,50256,50257,50258,50259,50260,50261,50262,50263,50264,50265,50266,50267,50268,50269,50270,50271,50272,50273,50274,50275,50276,50277,50278,50279,50280,50281,50282,50283,50284,50285,50286,50287,50288,50289,50290,50291,50292,50293,50294,50295,50296,50297,50298,50299,50300,50301,50302,50337,50338,50339,50340,50341,50342,50343,50344,50345,50346,50347,50348,50349,50350,50351,50352,50353,50354,50355,50356,50357,50358,50359,50360,50361,50362,50363,50364,50365,50366,50367,50368,50369,50370,50371,50372,50373,50374,50375,50376,50377,50378,50379,50380,50381,50382,50383,50384,50385,50386,50387,50388,50389,50390,50391,50392,50393,50394,50395,50396,50397,50398,50399,50400,50401,50402,50403,50404,50405,50406,50407,50408,50409,50410,50411,50412,50413,50414,50415,50416,50417,50418,50419,50420,50421,50422,50423,50424,50425,50426,50427,50428,50429,50430,50496,50497,50498,50499,50500,50501,50502,50503,50504,50505,50506,50507,50508,50509,50510,50511,50512,50513,50514,50515,50516,50517,50518,50519,50520,50521,50522,50523,50524,50525,50526,50527,50528,50529,50530,50531,50532,50533,50534,50535,50536,50537,50538,50539,50540,50541,50542,50543,50544,50545,50546,50547,50548,50549,50550,50551,50552,50553,50554,50555,50556,50557,50558,50593,50594,50595,50596,50597,50598,50599,50600,50601,50602,50603,50604,50605,50606,50607,50608,50609,50610,50611,50612,50613,50614,50615,50616,50617,50618,50619,50620,50621,50622,50623,50624,50625,50626,50627,50628,50629,50630,50631,50632,50633,50634,50635,50636,50637,50638,50639,50640,50641,50642,50643,50644,50645,50646,50647,50648,50649,50650,50651,50652,50653,50654,50655,50656,50657,50658,50659,50660,50661,50662,50663,50664,50665,50666,50667,50668,50669,50670,50671,50672,50673,50674,50675,50676,50677,50678,50679,50680,50681,50682,50683,50684,50685,50686,50752,50753,50754,50755,50756,50757,50758,50759,50760,50761,50762,50763,50764,50765,50766,50767,50768,50769,50770,50771,50772,50773,50774,50775,50776,50777,50778,50779,50780,50781,50782,50783,50784,50785,50786,50787,50788,50789,50790,50791,50792,50793,50794,50795,50796,50797,50798,50799,50800,50801,50802,50803,50804,50805,50806,50807,50808,50809,50810,50811,50812,50813,50814,50849,50850,50851,50852,50853,50854,50855,50856,50857,50858,50859,50860,50861,50862,50863,50864,50865,50866,50867,50868,50869,50870,50871,50872,50873,50874,50875,50876,50877,50878,50879,50880,50881,50882,50883,50884,50885,50886,50887,50888,50889,50890,50891,50892,50893,50894,50895,50896,50897,50898,50899,50900,50901,50902,50903,50904,50905,50906,50907,50908,50909,50910,50911,50912,50913,50914,50915,50916,50917,50918,50919,50920,50921,50922,50923,50924,50925,50926,50927,50928,50929,50930,50931,50932,50933,50934,50935,50936,50937,50938,50939,50940,50941,50942,51008,51009,51010,51011,51012,51013,51014,51015,51016,51017,51018,51019,51020,51021,51022,51023,51024,51025,51026,51027,51028,51029,51030,51031,51032,51033,51034,51035,51036,51037,51038,51039,51040,51041,51042,51043,51044,51045,51046,51047,51048,51049,51050,51051,51052,51053,51054,51055,51056,51057,51058,51059,51060,51061,51062,51063,51064,51065,51066,51067,51068,51069,51070,51105,51106,51107,51108,51109,51110,51111,51112,51113,51114,51115,51116,51117,51118,51119,51120,51121,51122,51123,51124,51125,51126,51127,51128,51129,51130,51131,51132,51133,51134,51135,51136,51137,51138,51139,51140,51141,51142,51143,51144,51145,51146,51147,51148,51149,51150,51151,51152,51153,51154,51155,51156,51157,51158,51159,51160,51161,51162,51163,51164,51165,51166,51167,51168,51169,51170,51171,51172,51173,51174,51175,51176,51177,51178,51179,51180,51181,51182,51183,51184,51185,51186,51187,51188,51189,51190,51191,51192,51193,51194,51195,51196,51197,51198,51264,51265,51266,51267,51268,51269,51270,51271,51272,51273,51274,51275,51276,51277,51278,51279,51280,51281,51282,51283,51284,51285,51286,51287,51288,51289,51290,51291,51292,51293,51294,51295,51296,51297,51298,51299,51300,51301,51302,51303,51304,51305,51306,51307,51308,51309,51310,51311,51312,51313,51314,51315,51316,51317,51318,51319,51320,51321,51322,51323,51324,51325,51326,51361,51362,51363,51364,51365,51366,51367,51368,51369,51370,51371,51372,51373,51374,51375,51376,51377,51378,51379,51380,51381,51382,51383,51384,51385,51386,51387,51388,51389,51390,51391,51392,51393,51394,51395,51396,51397,51398,51399,51400,51401,51402,51403,51404,51405,51406,51407,51408,51409,51410,51411,51412,51413,51414,51415,51416,51417,51418,51419,51420,51421,51422,51423,51424,51425,51426,51427,51428,51429,51430,51431,51432,51433,51434,51435,51436,51437,51438,51439,51440,51441,51442,51443,51444,51445,51446,51447,51448,51449,51450,51451,51452,51453,51454,51520,51521,51522,51523,51524,51525,51526,51527,51528,51529,51530,51531,51532,51533,51534,51535,51536,51537,51538,51539,51540,51541,51542,51543,51544,51545,51546,51547,51548,51549,51550,51551,51552,51553,51554,51555,51556,51557,51558,51559,51560,51561,51562,51563,51564,51565,51566,51567,51568,51569,51570,51571,51572,51573,51574,51575,51576,51577,51578,51579,51580,51581,51582,51617,51618,51619,51620,51621,51622,51623,51624,51625,51626,51627,51628,51629,51630,51631,51632,51633,51634,51635,51636,51637,51638,51639,51640,51641,51642,51643,51644,51645,51646,51647,51648,51649,51650,51651,51652,51653,51654,51655,51656,51657,51658,51659,51660,51661,51662,51663,51664,51665,51666,51667,51668,51669,51670,51671,51672,51673,51674,51675,51676,51677,51678,51679,51680,51681,51682,51683,51684,51685,51686,51687,51688,51689,51690,51691,51692,51693,51694,51695,51696,51697,51698,51699,51700,51701,51702,51703,51704,51705,51706,51707,51708,51709,51710,51776,51777,51778,51779,51780,51781,51782,51783,51784,51785,51786,51787,51788,51789,51790,51791,51792,51793,51794,51795,51796,51797,51798,51799,51800,51801,51802,51803,51804,51805,51806,51807,51808,51809,51810,51811,51812,51813,51814,51815,51816,51817,51818,51819,51820,51821,51822,51823,51824,51825,51826,51827,51828,51829,51830,51831,51832,51833,51834,51835,51836,51837,51838,51873,51874,51875,51876,51877,51878,51879,51880,51881,51882,51883,51884,51885,51886,51887,51888,51889,51890,51891,51892,51893,51894,51895,51896,51897,51898,51899,51900,51901,51902,51903,51904,51905,51906,51907,51908,51909,51910,51911,51912,51913,51914,51915,51916,51917,51918,51919,51920,51921,51922,51923,51924,51925,51926,51927,51928,51929,51930,51931,51932,51933,51934,51935,51936,51937,51938,51939,51940,51941,51942,51943,51944,51945,51946,51947,51948,51949,51950,51951,51952,51953,51954,51955,51956,51957,51958,51959,51960,51961,51962,51963,51964,51965,51966,52032,52033,52034,52035,52036,52037,52038,52039,52040,52041,52042,52043,52044,52045,52046,52047,52048,52049,52050,52051,52052,52053,52054,52055,52056,52057,52058,52059,52060,52061,52062,52063,52064,52065,52066,52067,52068,52069,52070,52071,52072,52073,52074,52075,52076,52077,52078,52079,52080,52081,52082,52083,52084,52085,52086,52087,52088,52089,52090,52091,52092,52093,52094,52129,52130,52131,52132,52133,52134,52135,52136,52137,52138,52139,52140,52141,52142,52143,52144,52145,52146,52147,52148,52149,52150,52151,52152,52153,52154,52155,52156,52157,52158,52159,52160,52161,52162,52163,52164,52165,52166,52167,52168,52169,52170,52171,52172,52173,52174,52175,52176,52177,52178,52179,52180,52181,52182,52183,52184,52185,52186,52187,52188,52189,52190,52191,52192,52193,52194,52195,52196,52197,52198,52199,52200,52201,52202,52203,52204,52205,52206,52207,52208,52209,52210,52211,52212,52213,52214,52215,52216,52217,52218,52219,52220,52221,52222,52288,52289,52290,52291,52292,52293,52294,52295,52296,52297,52298,52299,52300,52301,52302,52303,52304,52305,52306,52307,52308,52309,52310,52311,52312,52313,52314,52315,52316,52317,52318,52319,52320,52321,52322,52323,52324,52325,52326,52327,52328,52329,52330,52331,52332,52333,52334,52335,52336,52337,52338,52339,52340,52341,52342,52343,52344,52345,52346,52347,52348,52349,52350,52385,52386,52387,52388,52389,52390,52391,52392,52393,52394,52395,52396,52397,52398,52399,52400,52401,52402,52403,52404,52405,52406,52407,52408,52409,52410,52411,52412,52413,52414,52415,52416,52417,52418,52419,52420,52421,52422,52423,52424,52425,52426,52427,52428,52429,52430,52431,52432,52433,52434,52435,52436,52437,52438,52439,52440,52441,52442,52443,52444,52445,52446,52447,52448,52449,52450,52451,52452,52453,52454,52455,52456,52457,52458,52459,52460,52461,52462,52463,52464,52465,52466,52467,52468,52469,52470,52471,52472,52473,52474,52475,52476,52477,52478,52544,52545,52546,52547,52548,52549,52550,52551,52552,52553,52554,52555,52556,52557,52558,52559,52560,52561,52562,52563,52564,52565,52566,52567,52568,52569,52570,52571,52572,52573,52574,52575,52576,52577,52578,52579,52580,52581,52582,52583,52584,52585,52586,52587,52588,52589,52590,52591,52592,52593,52594,52595,52596,52597,52598,52599,52600,52601,52602,52603,52604,52605,52606,52641,52642,52643,52644,52645,52646,52647,52648,52649,52650,52651,52652,52653,52654,52655,52656,52657,52658,52659,52660,52661,52662,52663,52664,52665,52666,52667,52668,52669,52670,52671,52672,52673,52674,52675,52676,52677,52678,52679,52680,52681,52682,52683,52684,52685,52686,52687,52688,52689,52690,52691,52692,52693,52694,52695,52696,52697,52698,52699,52700,52701,52702,52703,52704,52705,52706,52707,52708,52709,52710,52711,52712,52713,52714,52715,52716,52717,52718,52719,52720,52721,52722,52723,52724,52725,52726,52727,52728,52729,52730,52731,52732,52733,52734,52800,52801,52802,52803,52804,52805,52806,52807,52808,52809,52810,52811,52812,52813,52814,52815,52816,52817,52818,52819,52820,52821,52822,52823,52824,52825,52826,52827,52828,52829,52830,52831,52832,52833,52834,52835,52836,52837,52838,52839,52840,52841,52842,52843,52844,52845,52846,52847,52848,52849,52850,52851,52852,52853,52854,52855,52856,52857,52858,52859,52860,52861,52862,52897,52898,52899,52900,52901,52902,52903,52904,52905,52906,52907,52908,52909,52910,52911,52912,52913,52914,52915,52916,52917,52918,52919,52920,52921,52922,52923,52924,52925,52926,52927,52928,52929,52930,52931,52932,52933,52934,52935,52936,52937,52938,52939,52940,52941,52942,52943,52944,52945,52946,52947,52948,52949,52950,52951,52952,52953,52954,52955,52956,52957,52958,52959,52960,52961,52962,52963,52964,52965,52966,52967,52968,52969,52970,52971,52972,52973,52974,52975,52976,52977,52978,52979,52980,52981,52982,52983,52984,52985,52986,52987,52988,52989,52990,53056,53057,53058,53059,53060,53061,53062,53063,53064,53065,53066,53067,53068,53069,53070,53071,53072,53073,53074,53075,53076,53077,53078,53079,53080,53081,53082,53083,53084,53085,53086,53087,53088,53089,53090,53091,53092,53093,53094,53095,53096,53097,53098,53099,53100,53101,53102,53103,53104,53105,53106,53107,53108,53109,53110,53111,53112,53113,53114,53115,53116,53117,53118,53153,53154,53155,53156,53157,53158,53159,53160,53161,53162,53163,53164,53165,53166,53167,53168,53169,53170,53171,53172,53173,53174,53175,53176,53177,53178,53179,53180,53181,53182,53183,53184,53185,53186,53187,53188,53189,53190,53191,53192,53193,53194,53195,53196,53197,53198,53199,53200,53201,53202,53203,53204,53205,53206,53207,53208,53209,53210,53211,53212,53213,53214,53215,53216,53217,53218,53219,53220,53221,53222,53223,53224,53225,53226,53227,53228,53229,53230,53231,53232,53233,53234,53235,53236,53237,53238,53239,53240,53241,53242,53243,53244,53245,53246,53312,53313,53314,53315,53316,53317,53318,53319,53320,53321,53322,53323,53324,53325,53326,53327,53328,53329,53330,53331,53332,53333,53334,53335,53336,53337,53338,53339,53340,53341,53342,53343,53344,53345,53346,53347,53348,53349,53350,53351,53352,53353,53354,53355,53356,53357,53358,53359,53360,53361,53362,53363,53364,53365,53366,53367,53368,53369,53370,53371,53372,53373,53374,53409,53410,53411,53412,53413,53414,53415,53416,53417,53418,53419,53420,53421,53422,53423,53424,53425,53426,53427,53428,53429,53430,53431,53432,53433,53434,53435,53436,53437,53438,53439,53440,53441,53442,53443,53444,53445,53446,53447,53448,53449,53450,53451,53452,53453,53454,53455,53456,53457,53458,53459,53460,53461,53462,53463,53464,53465,53466,53467,53468,53469,53470,53471,53472,53473,53474,53475,53476,53477,53478,53479,53480,53481,53482,53483,53484,53485,53486,53487,53488,53489,53490,53491,53492,53493,53494,53495,53496,53497,53498,53499,53500,53501,53502,53568,53569,53570,53571,53572,53573,53574,53575,53576,53577,53578,53579,53580,53581,53582,53583,53584,53585,53586,53587,53588,53589,53590,53591,53592,53593,53594,53595,53596,53597,53598,53599,53600,53601,53602,53603,53604,53605,53606,53607,53608,53609,53610,53611,53612,53613,53614,53615,53616,53617,53618,53619,53620,53621,53622,53623,53624,53625,53626,53627,53628,53629,53630,53665,53666,53667,53668,53669,53670,53671,53672,53673,53674,53675,53676,53677,53678,53679,53680,53681,53682,53683,53684,53685,53686,53687,53688,53689,53690,53691,53692,53693,53694,53695,53696,53697,53698,53699,53700,53701,53702,53703,53704,53705,53706,53707,53708,53709,53710,53711,53712,53713,53714,53715,53716,53717,53718,53719,53720,53721,53722,53723,53724,53725,53726,53727,53728,53729,53730,53731,53732,53733,53734,53735,53736,53737,53738,53739,53740,53741,53742,53743,53744,53745,53746,53747,53748,53749,53750,53751,53752,53753,53754,53755,53756,53757,53758,53824,53825,53826,53827,53828,53829,53830,53831,53832,53833,53834,53835,53836,53837,53838,53839,53840,53841,53842,53843,53844,53845,53846,53847,53848,53849,53850,53851,53852,53853,53854,53855,53856,53857,53858,53859,53860,53861,53862,53863,53864,53865,53866,53867,53868,53869,53870,53871,53872,53873,53874,53875,53876,53877,53878,53879,53880,53881,53882,53883,53884,53885,53886,53921,53922,53923,53924,53925,53926,53927,53928,53929,53930,53931,53932,53933,53934,53935,53936,53937,53938,53939,53940,53941,53942,53943,53944,53945,53946,53947,53948,53949,53950,53951,53952,53953,53954,53955,53956,53957,53958,53959,53960,53961,53962,53963,53964,53965,53966,53967,53968,53969,53970,53971,53972,53973,53974,53975,53976,53977,53978,53979,53980,53981,53982,53983,53984,53985,53986,53987,53988,53989,53990,53991,53992,53993,53994,53995,53996,53997,53998,53999,54000,54001,54002,54003,54004,54005,54006,54007,54008,54009,54010,54011,54012,54013,54014,54080,54081,54082,54083,54084,54085,54086,54087,54088,54089,54090,54091,54092,54093,54094,54095,54096,54097,54098,54099,54100,54101,54102,54103,54104,54105,54106,54107,54108,54109,54110,54111,54112,54113,54114,54115,54116,54117,54118,54119,54120,54121,54122,54123,54124,54125,54126,54127,54128,54129,54130,54131,54132,54133,54134,54135,54136,54137,54138,54139,54140,54141,54142,54177,54178,54179,54180,54181,54182,54183,54184,54185,54186,54187,54188,54189,54190,54191,54192,54193,54194,54195,54196,54197,54198,54199,54200,54201,54202,54203,54204,54205,54206,54207,54208,54209,54210,54211,54212,54213,54214,54215,54216,54217,54218,54219,54220,54221,54222,54223,54224,54225,54226,54227,54228,54229,54230,54231,54232,54233,54234,54235,54236,54237,54238,54239,54240,54241,54242,54243,54244,54245,54246,54247,54248,54249,54250,54251,54252,54253,54254,54255,54256,54257,54258,54259,54260,54261,54262,54263,54264,54265,54266,54267,54268,54269,54270,54336,54337,54338,54339,54340,54341,54342,54343,54344,54345,54346,54347,54348,54349,54350,54351,54352,54353,54354,54355,54356,54357,54358,54359,54360,54361,54362,54363,54364,54365,54366,54367,54368,54369,54370,54371,54372,54373,54374,54375,54376,54377,54378,54379,54380,54381,54382,54383,54384,54385,54386,54387,54388,54389,54390,54391,54392,54393,54394,54395,54396,54397,54398,54433,54434,54435,54436,54437,54438,54439,54440,54441,54442,54443,54444,54445,54446,54447,54448,54449,54450,54451,54452,54453,54454,54455,54456,54457,54458,54459,54460,54461,54462,54463,54464,54465,54466,54467,54468,54469,54470,54471,54472,54473,54474,54475,54476,54477,54478,54479,54480,54481,54482,54483,54484,54485,54486,54487,54488,54489,54490,54491,54492,54493,54494,54495,54496,54497,54498,54499,54500,54501,54502,54503,54504,54505,54506,54507,54508,54509,54510,54511,54512,54513,54514,54515,54516,54517,54518,54519,54520,54521,54522,54523,54524,54525,54526,54592,54593,54594,54595,54596,54597,54598,54599,54600,54601,54602,54603,54604,54605,54606,54607,54608,54609,54610,54611,54612,54613,54614,54615,54616,54617,54618,54619,54620,54621,54622,54623,54624,54625,54626,54627,54628,54629,54630,54631,54632,54633,54634,54635,54636,54637,54638,54639,54640,54641,54642,54643,54644,54645,54646,54647,54648,54649,54650,54651,54652,54653,54654,54689,54690,54691,54692,54693,54694,54695,54696,54697,54698,54699,54700,54701,54702,54703,54704,54705,54706,54707,54708,54709,54710,54711,54712,54713,54714,54715,54716,54717,54718,54719,54720,54721,54722,54723,54724,54725,54726,54727,54728,54729,54730,54731,54732,54733,54734,54735,54736,54737,54738,54739,54740,54741,54742,54743,54744,54745,54746,54747,54748,54749,54750,54751,54752,54753,54754,54755,54756,54757,54758,54759,54760,54761,54762,54763,54764,54765,54766,54767,54768,54769,54770,54771,54772,54773,54774,54775,54776,54777,54778,54779,54780,54781,54782,54848,54849,54850,54851,54852,54853,54854,54855,54856,54857,54858,54859,54860,54861,54862,54863,54864,54865,54866,54867,54868,54869,54870,54871,54872,54873,54874,54875,54876,54877,54878,54879,54880,54881,54882,54883,54884,54885,54886,54887,54888,54889,54890,54891,54892,54893,54894,54895,54896,54897,54898,54899,54900,54901,54902,54903,54904,54905,54906,54907,54908,54909,54910,54945,54946,54947,54948,54949,54950,54951,54952,54953,54954,54955,54956,54957,54958,54959,54960,54961,54962,54963,54964,54965,54966,54967,54968,54969,54970,54971,54972,54973,54974,54975,54976,54977,54978,54979,54980,54981,54982,54983,54984,54985,54986,54987,54988,54989,54990,54991,54992,54993,54994,54995,54996,54997,54998,54999,55000,55001,55002,55003,55004,55005,55006,55007,55008,55009,55010,55011,55012,55013,55014,55015,55016,55017,55018,55019,55020,55021,55022,55023,55024,55025,55026,55027,55028,55029,55030,55031,55032,55033,55034,55035,55036,55037,55038,55104,55105,55106,55107,55108,55109,55110,55111,55112,55113,55114,55115,55116,55117,55118,55119,55120,55121,55122,55123,55124,55125,55126,55127,55128,55129,55130,55131,55132,55133,55134,55135,55136,55137,55138,55139,55140,55141,55142,55143,55144,55145,55146,55147,55148,55149,55150,55151,55152,55153,55154,55155,55156,55157,55158,55159,55160,55161,55162,55163,55164,55165,55166,55201,55202,55203,55204,55205,55206,55207,55208,55209,55210,55211,55212,55213,55214,55215,55216,55217,55218,55219,55220,55221,55222,55223,55224,55225,55226,55227,55228,55229,55230,55231,55232,55233,55234,55235,55236,55237,55238,55239,55240,55241,55242,55243,55244,55245,55246,55247,55248,55249,55250,55251,55252,55253,55254,55255,55256,55257,55258,55259,55260,55261,55262,55263,55264,55265,55266,55267,55268,55269,55270,55271,55272,55273,55274,55275,55276,55277,55278,55279,55280,55281,55282,55283,55284,55285,55286,55287,55288,55289,55290,55291,55292,55293,55294,55360,55361,55362,55363,55364,55365,55366,55367,55368,55369,55370,55371,55372,55373,55374,55375,55376,55377,55378,55379,55380,55381,55382,55383,55384,55385,55386,55387,55388,55389,55390,55391,55392,55393,55394,55395,55396,55397,55398,55399,55400,55401,55402,55403,55404,55405,55406,55407,55408,55409,55410,55411,55412,55413,55414,55415,55416,55417,55418,55419,55420,55421,55422,55457,55458,55459,55460,55461,55462,55463,55464,55465,55466,55467,55468,55469,55470,55471,55472,55473,55474,55475,55476,55477,55478,55479,55480,55481,55482,55483,55484,55485,55486,55487,55488,55489,55490,55491,55492,55493,55494,55495,55496,55497,55498,55499,55500,55501,55502,55503,55504,55505,55506,55507,55508,55509,55510,55511,55512,55513,55514,55515,55516,55517,55518,55519,55520,55521,55522,55523,55524,55525,55526,55527,55528,55529,55530,55531,55532,55533,55534,55535,55536,55537,55538,55539,55540,55541,55542,55543,55544,55545,55546,55547,55548,55549,55550,55616,55617,55618,55619,55620,55621,55622,55623,55624,55625,55626,55627,55628,55629,55630,55631,55632,55633,55634,55635,55636,55637,55638,55639,55640,55641,55642,55643,55644,55645,55646,55647,55648,55649,55650,55651,55652,55653,55654,55655,55656,55657,55658,55659,55660,55661,55662,55663,55664,55665,55666,55667,55668,55669,55670,55671,55672,55673,55674,55675,55676,55677,55678,55713,55714,55715,55716,55717,55718,55719,55720,55721,55722,55723,55724,55725,55726,55727,55728,55729,55730,55731,55732,55733,55734,55735,55736,55737,55738,55739,55740,55741,55742,55743,55744,55745,55746,55747,55748,55749,55750,55751,55752,55753,55754,55755,55756,55757,55758,55759,55760,55761,55762,55763,55764,55765,55766,55767,55768,55769,55770,55771,55772,55773,55774,55775,55776,55777,55778,55779,55780,55781,55782,55783,55784,55785,55786,55787,55788,55789,55790,55791,55792,55793,55794,55795,55796,55797,55798,55799,55800,55801,55802,55803,55804,55805,55806,55872,55873,55874,55875,55876,55877,55878,55879,55880,55881,55882,55883,55884,55885,55886,55887,55888,55889,55890,55891,55892,55893,55894,55895,55896,55897,55898,55899,55900,55901,55902,55903,55904,55905,55906,55907,55908,55909,55910,55911,55912,55913,55914,55915,55916,55917,55918,55919,55920,55921,55922,55923,55924,55925,55926,55927,55928,55929,55930,55931,55932,55933,55934,55969,55970,55971,55972,55973,55974,55975,55976,55977,55978,55979,55980,55981,55982,55983,55984,55985,55986,55987,55988,55989,55990,55991,55992,55993,55994,55995,55996,55997,55998,55999,56000,56001,56002,56003,56004,56005,56006,56007,56008,56009,56010,56011,56012,56013,56014,56015,56016,56017,56018,56019,56020,56021,56022,56023,56024,56025,56026,56027,56028,56029,56030,56031,56032,56033,56034,56035,56036,56037,56038,56039,56040,56041,56042,56043,56044,56045,56046,56047,56048,56049,56050,56051,56052,56053,56054,56055,56056,56057,56058,56059,56060,56061,56062,56128,56129,56130,56131,56132,56133,56134,56135,56136,56137,56138,56139,56140,56141,56142,56143,56144,56145,56146,56147,56148,56149,56150,56151,56152,56153,56154,56155,56156,56157,56158,56159,56160,56161,56162,56163,56164,56165,56166,56167,56168,56169,56170,56171,56172,56173,56174,56175,56176,56177,56178,56179,56180,56181,56182,56183,56184,56185,56186,56187,56188,56189,56190,56225,56226,56227,56228,56229,56230,56231,56232,56233,56234,56235,56236,56237,56238,56239,56240,56241,56242,56243,56244,56245,56246,56247,56248,56249,56250,56251,56252,56253,56254,56255,56256,56257,56258,56259,56260,56261,56262,56263,56264,56265,56266,56267,56268,56269,56270,56271,56272,56273,56274,56275,56276,56277,56278,56279,56280,56281,56282,56283,56284,56285,56286,56287,56288,56289,56290,56291,56292,56293,56294,56295,56296,56297,56298,56299,56300,56301,56302,56303,56304,56305,56306,56307,56308,56309,56310,56311,56312,56313,56314,56315,56316,56317,56318,56384,56385,56386,56387,56388,56389,56390,56391,56392,56393,56394,56395,56396,56397,56398,56399,56400,56401,56402,56403,56404,56405,56406,56407,56408,56409,56410,56411,56412,56413,56414,56415,56416,56417,56418,56419,56420,56421,56422,56423,56424,56425,56426,56427,56428,56429,56430,56431,56432,56433,56434,56435,56436,56437,56438,56439,56440,56441,56442,56443,56444,56445,56446,56481,56482,56483,56484,56485,56486,56487,56488,56489,56490,56491,56492,56493,56494,56495,56496,56497,56498,56499,56500,56501,56502,56503,56504,56505,56506,56507,56508,56509,56510,56511,56512,56513,56514,56515,56516,56517,56518,56519,56520,56521,56522,56523,56524,56525,56526,56527,56528,56529,56530,56531,56532,56533,56534,56535,56536,56537,56538,56539,56540,56541,56542,56543,56544,56545,56546,56547,56548,56549,56550,56551,56552,56553,56554,56555,56556,56557,56558,56559,56560,56561,56562,56563,56564,56565,56566,56567,56568,56569,56570,56571,56572,56573,56574,56640,56641,56642,56643,56644,56645,56646,56647,56648,56649,56650,56651,56652,56653,56654,56655,56656,56657,56658,56659,56660,56661,56662,56663,56664,56665,56666,56667,56668,56669,56670,56671,56672,56673,56674,56675,56676,56677,56678,56679,56680,56681,56682,56683,56684,56685,56686,56687,56688,56689,56690,56691,56692,56693,56694,56695,56696,56697,56698,56699,56700,56701,56702,56737,56738,56739,56740,56741,56742,56743,56744,56745,56746,56747,56748,56749,56750,56751,56752,56753,56754,56755,56756,56757,56758,56759,56760,56761,56762,56763,56764,56765,56766,56767,56768,56769,56770,56771,56772,56773,56774,56775,56776,56777,56778,56779,56780,56781,56782,56783,56784,56785,56786,56787,56788,56789,56790,56791,56792,56793,56794,56795,56796,56797,56798,56799,56800,56801,56802,56803,56804,56805,56806,56807,56808,56809,56810,56811,56812,56813,56814,56815,56816,56817,56818,56819,56820,56821,56822,56823,56824,56825,56826,56827,56828,56829,56830,56896,56897,56898,56899,56900,56901,56902,56903,56904,56905,56906,56907,56908,56909,56910,56911,56912,56913,56914,56915,56916,56917,56918,56919,56920,56921,56922,56923,56924,56925,56926,56927,56928,56929,56930,56931,56932,56933,56934,56935,56936,56937,56938,56939,56940,56941,56942,56943,56944,56945,56946,56947,56948,56949,56950,56951,56952,56953,56954,56955,56956,56957,56958,56993,56994,56995,56996,56997,56998,56999,57000,57001,57002,57003,57004,57005,57006,57007,57008,57009,57010,57011,57012,57013,57014,57015,57016,57017,57018,57019,57020,57021,57022,57023,57024,57025,57026,57027,57028,57029,57030,57031,57032,57033,57034,57035,57036,57037,57038,57039,57040,57041,57042,57043,57044,57045,57046,57047,57048,57049,57050,57051,57052,57053,57054,57055,57056,57057,57058,57059,57060,57061,57062,57063,57064,57065,57066,57067,57068,57069,57070,57071,57072,57073,57074,57075,57076,57077,57078,57079,57080,57081,57082,57083,57084,57085,57086,57152,57153,57154,57155,57156,57157,57158,57159,57160,57161,57162,57163,57164,57165,57166,57167,57168,57169,57170,57171,57172,57173,57174,57175,57176,57177,57178,57179,57180,57181,57182,57183,57184,57185,57186,57187,57188,57189,57190,57191,57192,57193,57194,57195,57196,57197,57198,57199,57200,57201,57202,57203,57204,57205,57206,57207,57208,57209,57210,57211,57212,57213,57214,57249,57250,57251,57252,57253,57254,57255,57256,57257,57258,57259,57260,57261,57262,57263,57264,57265,57266,57267,57268,57269,57270,57271,57272,57273,57274,57275,57276,57277,57278,57279,57280,57281,57282,57283,57284,57285,57286,57287,57288,57289,57290,57291,57292,57293,57294,57295,57296,57297,57298,57299,57300,57301,57302,57303,57304,57305,57306,57307,57308,57309,57310,57311,57312,57313,57314,57315,57316,57317,57318,57319,57320,57321,57322,57323,57324,57325,57326,57327,57328,57329,57330,57331,57332,57333,57334,57335,57336,57337,57338,57339,57340,57341,57342,57408,57409,57410,57411,57412,57413,57414,57415,57416,57417,57418,57419,57420,57421,57422,57423,57424,57425,57426,57427,57428,57429,57430,57431,57432,57433,57434,57435,57436,57437,57438,57439,57440,57441,57442,57443,57444,57445,57446,57447,57448,57449,57450,57451,57452,57453,57454,57455,57456,57457,57458,57459,57460,57461,57462,57463,57464,57465,57466,57467,57468,57469,57470,57505,57506,57507,57508,57509,57510,57511,57512,57513,57514,57515,57516,57517,57518,57519,57520,57521,57522,57523,57524,57525,57526,57527,57528,57529,57530,57531,57532,57533,57534,57535,57536,57537,57538,57539,57540,57541,57542,57543,57544,57545,57546,57547,57548,57549,57550,57551,57552,57553,57554,57555,57556,57557,57558,57559,57560,57561,57562,57563,57564,57565,57566,57567,57568,57569,57570,57571,57572,57573,57574,57575,57576,57577,57578,57579,57580,57581,57582,57583,57584,57585,57586,57587,57588,57589,57590,57591,57592,57593,57594,57595,57596,57597,57598,57664,57665,57666,57667,57668,57669,57670,57671,57672,57673,57674,57675,57676,57677,57678,57679,57680,57681,57682,57683,57684,57685,57686,57687,57688,57689,57690,57691,57692,57693,57694,57695,57696,57697,57698,57699,57700,57701,57702,57703,57704,57705,57706,57707,57708,57709,57710,57711,57712,57713,57714,57715,57716,57717,57718,57719,57720,57721,57722,57723,57724,57725,57726,57761,57762,57763,57764,57765,57766,57767,57768,57769,57770,57771,57772,57773,57774,57775,57776,57777,57778,57779,57780,57781,57782,57783,57784,57785,57786,57787,57788,57789,57790,57791,57792,57793,57794,57795,57796,57797,57798,57799,57800,57801,57802,57803,57804,57805,57806,57807,57808,57809,57810,57811,57812,57813,57814,57815,57816,57817,57818,57819,57820,57821,57822,57823,57824,57825,57826,57827,57828,57829,57830,57831,57832,57833,57834,57835,57836,57837,57838,57839,57840,57841,57842,57843,57844,57845,57846,57847,57848,57849,57850,57851,57852,57853,57854,57920,57921,57922,57923,57924,57925,57926,57927,57928,57929,57930,57931,57932,57933,57934,57935,57936,57937,57938,57939,57940,57941,57942,57943,57944,57945,57946,57947,57948,57949,57950,57951,57952,57953,57954,57955,57956,57957,57958,57959,57960,57961,57962,57963,57964,57965,57966,57967,57968,57969,57970,57971,57972,57973,57974,57975,57976,57977,57978,57979,57980,57981,57982,58017,58018,58019,58020,58021,58022,58023,58024,58025,58026,58027,58028,58029,58030,58031,58032,58033,58034,58035,58036,58037,58038,58039,58040,58041,58042,58043,58044,58045,58046,58047,58048,58049,58050,58051,58052,58053,58054,58055,58056,58057,58058,58059,58060,58061,58062,58063,58064,58065,58066,58067,58068,58069,58070,58071,58072,58073,58074,58075,58076,58077,58078,58079,58080,58081,58082,58083,58084,58085,58086,58087,58088,58089,58090,58091,58092,58093,58094,58095,58096,58097,58098,58099,58100,58101,58102,58103,58104,58105,58106,58107,58108,58109,58110,58176,58177,58178,58179,58180,58181,58182,58183,58184,58185,58186,58187,58188,58189,58190,58191,58192,58193,58194,58195,58196,58197,58198,58199,58200,58201,58202,58203,58204,58205,58206,58207,58208,58209,58210,58211,58212,58213,58214,58215,58216,58217,58218,58219,58220,58221,58222,58223,58224,58225,58226,58227,58228,58229,58230,58231,58232,58233,58234,58235,58236,58237,58238,58273,58274,58275,58276,58277,58278,58279,58280,58281,58282,58283,58284,58285,58286,58287,58288,58289,58290,58291,58292,58293,58294,58295,58296,58297,58298,58299,58300,58301,58302,58303,58304,58305,58306,58307,58308,58309,58310,58311,58312,58313,58314,58315,58316,58317,58318,58319,58320,58321,58322,58323,58324,58325,58326,58327,58328,58329,58330,58331,58332,58333,58334,58335,58336,58337,58338,58339,58340,58341,58342,58343,58344,58345,58346,58347,58348,58349,58350,58351,58352,58353,58354,58355,58356,58357,58358,58359,58360,58361,58362,58363,58364,58365,58366,58432,58433,58434,58435,58436,58437,58438,58439,58440,58441,58442,58443,58444,58445,58446,58447,58448,58449,58450,58451,58452,58453,58454,58455,58456,58457,58458,58459,58460,58461,58462,58463,58464,58465,58466,58467,58468,58469,58470,58471,58472,58473,58474,58475,58476,58477,58478,58479,58480,58481,58482,58483,58484,58485,58486,58487,58488,58489,58490,58491,58492,58493,58494,58529,58530,58531,58532,58533,58534,58535,58536,58537,58538,58539,58540,58541,58542,58543,58544,58545,58546,58547,58548,58549,58550,58551,58552,58553,58554,58555,58556,58557,58558,58559,58560,58561,58562,58563,58564,58565,58566,58567,58568,58569,58570,58571,58572,58573,58574,58575,58576,58577,58578,58579,58580,58581,58582,58583,58584,58585,58586,58587,58588,58589,58590,58591,58592,58593,58594,58595,58596,58597,58598,58599,58600,58601,58602,58603,58604,58605,58606,58607,58608,58609,58610,58611,58612,58613,58614,58615,58616,58617,58618,58619,58620,58621,58622,58688,58689,58690,58691,58692,58693,58694,58695,58696,58697,58698,58699,58700,58701,58702,58703,58704,58705,58706,58707,58708,58709,58710,58711,58712,58713,58714,58715,58716,58717,58718,58719,58720,58721,58722,58723,58724,58725,58726,58727,58728,58729,58730,58731,58732,58733,58734,58735,58736,58737,58738,58739,58740,58741,58742,58743,58744,58745,58746,58747,58748,58749,58750,58785,58786,58787,58788,58789,58790,58791,58792,58793,58794,58795,58796,58797,58798,58799,58800,58801,58802,58803,58804,58805,58806,58807,58808,58809,58810,58811,58812,58813,58814,58815,58816,58817,58818,58819,58820,58821,58822,58823,58824,58825,58826,58827,58828,58829,58830,58831,58832,58833,58834,58835,58836,58837,58838,58839,58840,58841,58842,58843,58844,58845,58846,58847,58848,58849,58850,58851,58852,58853,58854,58855,58856,58857,58858,58859,58860,58861,58862,58863,58864,58865,58866,58867,58868,58869,58870,58871,58872,58873,58874,58875,58876,58877,58878,58944,58945,58946,58947,58948,58949,58950,58951,58952,58953,58954,58955,58956,58957,58958,58959,58960,58961,58962,58963,58964,58965,58966,58967,58968,58969,58970,58971,58972,58973,58974,58975,58976,58977,58978,58979,58980,58981,58982,58983,58984,58985,58986,58987,58988,58989,58990,58991,58992,58993,58994,58995,58996,58997,58998,58999,59000,59001,59002,59003,59004,59005,59006,59041,59042,59043,59044,59045,59046,59047,59048,59049,59050,59051,59052,59053,59054,59055,59056,59057,59058,59059,59060,59061,59062,59063,59064,59065,59066,59067,59068,59069,59070,59071,59072,59073,59074,59075,59076,59077,59078,59079,59080,59081,59082,59083,59084,59085,59086,59087,59088,59089,59090,59091,59092,59093,59094,59095,59096,59097,59098,59099,59100,59101,59102,59103,59104,59105,59106,59107,59108,59109,59110,59111,59112,59113,59114,59115,59116,59117,59118,59119,59120,59121,59122,59123,59124,59125,59126,59127,59128,59129,59130,59131,59132,59133,59134,59200,59201,59202,59203,59204,59205,59206,59207,59208,59209,59210,59211,59212,59213,59214,59215,59216,59217,59218,59219,59220,59221,59222,59223,59224,59225,59226,59227,59228,59229,59230,59231,59232,59233,59234,59235,59236,59237,59238,59239,59240,59241,59242,59243,59244,59245,59246,59247,59248,59249,59250,59251,59252,59253,59254,59255,59256,59257,59258,59259,59260,59261,59262,59297,59298,59299,59300,59301,59302,59303,59304,59305,59306,59307,59308,59309,59310,59311,59312,59313,59314,59315,59316,59317,59318,59319,59320,59321,59322,59323,59324,59325,59326,59327,59328,59329,59330,59331,59332,59333,59334,59335,59336,59337,59338,59339,59340,59341,59342,59343,59344,59345,59346,59347,59348,59349,59350,59351,59352,59353,59354,59355,59356,59357,59358,59359,59360,59361,59362,59363,59364,59365,59366,59367,59368,59369,59370,59371,59372,59373,59374,59375,59376,59377,59378,59379,59380,59381,59382,59383,59384,59385,59386,59387,59388,59389,59390,59456,59457,59458,59459,59460,59461,59462,59463,59464,59465,59466,59467,59468,59469,59470,59471,59472,59473,59474,59475,59476,59477,59478,59479,59480,59481,59482,59483,59484,59485,59486,59487,59488,59489,59490,59491,59492,59493,59494,59495,59496,59497,59498,59499,59500,59501,59502,59503,59504,59505,59506,59507,59508,59509,59510,59511,59512,59513,59514,59515,59516,59517,59518,59553,59554,59555,59556,59557,59558,59559,59560,59561,59562,59563,59564,59565,59566,59567,59568,59569,59570,59571,59572,59573,59574,59575,59576,59577,59578,59579,59580,59581,59582,59583,59584,59585,59586,59587,59588,59589,59590,59591,59592,59593,59594,59595,59596,59597,59598,59599,59600,59601,59602,59603,59604,59605,59606,59607,59608,59609,59610,59611,59612,59613,59614,59615,59616,59617,59618,59619,59620,59621,59622,59623,59624,59625,59626,59627,59628,59629,59630,59631,59632,59633,59634,59635,59636,59637,59638,59639,59640,59641,59642,59643,59644,59645,59646,59712,59713,59714,59715,59716,59717,59718,59719,59720,59721,59722,59723,59724,59725,59726,59727,59728,59729,59730,59731,59732,59733,59734,59735,59736,59737,59738,59739,59740,59741,59742,59743,59744,59745,59746,59747,59748,59749,59750,59751,59752,59753,59754,59755,59756,59757,59758,59759,59760,59761,59762,59763,59764,59765,59766,59767,59768,59769,59770,59771,59772,59773,59774,59809,59810,59811,59812,59813,59814,59815,59816,59817,59818,59819,59820,59821,59822,59823,59824,59825,59826,59827,59828,59829,59830,59831,59832,59833,59834,59835,59836,59837,59838,59839,59840,59841,59842,59843,59844,59845,59846,59847,59848,59849,59850,59851,59852,59853,59854,59855,59856,59857,59858,59859,59860,59861,59862,59863,59864,59865,59866,59867,59868,59869,59870,59871,59872,59873,59874,59875,59876,59877,59878,59879,59880,59881,59882,59883,59884,59885,59886,59887,59888,59889,59890,59891,59892,59893,59894,59895,59896,59897,59898,59899,59900,59901,59902,59968,59969,59970,59971,59972,59973,59974,59975,59976,59977,59978,59979,59980,59981,59982,59983,59984,59985,59986,59987,59988,59989,59990,59991,59992,59993,59994,59995,59996,59997,59998,59999,60000,60001,60002,60003,60004,60005,60006,60007,60008,60009,60010,60011,60012,60013,60014,60015,60016,60017,60018,60019,60020,60021,60022,60023,60024,60025,60026,60027,60028,60029,60030,60065,60066,60067,60068,60069,60070,60071,60072,60073,60074,60075,60076,60077,60078,60079,60080,60081,60082,60083,60084,60085,60086,60087,60088,60089,60090,60091,60092,60093,60094,60095,60096,60097,60098,60099,60100,60101,60102,60103,60104,60105,60106,60107,60108,60109,60110,60111,60112,60113,60114,60115,60116,60117,60118,60119,60120,60121,60122,60123,60124,60125,60126,60127,60128,60129,60130,60131,60132,60133,60134,60135,60136,60137,60138,60139,60140,60141,60142,60143,60144,60145,60146,60147,60148,60149,60150,60151,60152,60153,60154,60155,60156,60157,60158,60224,60225,60226,60227,60228,60229,60230,60231,60232,60233,60234,60235,60236,60237,60238,60239,60240,60241,60242,60243,60244,60245,60246,60247,60248,60249,60250,60251,60252,60253,60254,60255,60256,60257,60258,60259,60260,60261,60262,60263,60264,60265,60266,60267,60268,60269,60270,60271,60272,60273,60274,60275,60276,60277,60278,60279,60280,60281,60282,60283,60284,60285,60286,60321,60322,60323,60324,60325,60326,60327,60328,60329,60330,60331,60332,60333,60334,60335,60336,60337,60338,60339,60340,60341,60342,60343,60344,60345,60346,60347,60348,60349,60350,60351,60352,60353,60354,60355,60356,60357,60358,60359,60360,60361,60362,60363,60364,60365,60366,60367,60368,60369,60370,60371,60372,60373,60374,60375,60376,60377,60378,60379,60380,60381,60382,60383,60384,60385,60386,60387,60388,60389,60390,60391,60392,60393,60394,60395,60396,60397,60398,60399,60400,60401,60402,60403,60404,60405,60406,60407,60408,60409,60410,60411,60412,60413,60414,60480,60481,60482,60483,60484,60485,60486,60487,60488,60489,60490,60491,60492,60493,60494,60495,60496,60497,60498,60499,60500,60501,60502,60503,60504,60505,60506,60507,60508,60509,60510,60511,60512,60513,60514,60515,60516,60517,60518,60519,60520,60521,60522,60523,60524,60525,60526,60527,60528,60529,60530,60531,60532,60533,60534,60535,60536,60537,60538,60539,60540,60541,60542,60577,60578,60579,60580,60581,60582,60583,60584,60585,60586,60587,60588,60589,60590,60591,60592,60593,60594,60595,60596,60597,60598,60599,60600,60601,60602,60603,60604,60605,60606,60607,60608,60609,60610,60611,60612,60613,60614,60615,60616,60617,60618,60619,60620,60621,60622,60623,60624,60625,60626,60627,60628,60629,60630,60631,60632,60633,60634,60635,60636,60637,60638,60639,60640,60641,60642,60643,60644,60645,60646,60647,60648,60649,60650,60651,60652,60653,60654,60655,60656,60657,60658,60659,60660,60661,60662,60663,60664,60665,60666,60667,60668,60669,60670,60736,60737,60738,60739,60740,60741,60742,60743,60744,60745,60746,60747,60748,60749,60750,60751,60752,60753,60754,60755,60756,60757,60758,60759,60760,60761,60762,60763,60764,60765,60766,60767,60768,60769,60770,60771,60772,60773,60774,60775,60776,60777,60778,60779,60780,60781,60782,60783,60784,60785,60786,60787,60788,60789,60790,60791,60792,60793,60794,60795,60796,60797,60798,60833,60834,60835,60836,60837,60838,60839,60840,60841,60842,60843,60844,60845,60846,60847,60848,60849,60850,60851,60852,60853,60854,60855,60856,60857,60858,60859,60860,60861,60862,60863,60864,60865,60866,60867,60868,60869,60870,60871,60872,60873,60874,60875,60876,60877,60878,60879,60880,60881,60882,60883,60884,60885,60886,60887,60888,60889,60890,60891,60892,60893,60894,60895,60896,60897,60898,60899,60900,60901,60902,60903,60904,60905,60906,60907,60908,60909,60910,60911,60912,60913,60914,60915,60916,60917,60918,60919,60920,60921,60922,60923,60924,60925,60926,60992,60993,60994,60995,60996,60997,60998,60999,61000,61001,61002,61003,61004,61005,61006,61007,61008,61009,61010,61011,61012,61013,61014,61015,61016,61017,61018,61019,61020,61021,61022,61023,61024,61025,61026,61027,61028,61029,61030,61031,61032,61033,61034,61035,61036,61037,61038,61039,61040,61041,61042,61043,61044,61045,61046,61047,61048,61049,61050,61051,61052,61053,61054,61089,61090,61091,61092,61093,61094,61095,61096,61097,61098,61099,61100,61101,61102,61103,61104,61105,61106,61107,61108,61109,61110,61111,61112,61113,61114,61115,61116,61117,61118,61119,61120,61121,61122,61123,61124,61125,61126,61127,61128,61129,61130,61131,61132,61133,61134,61135,61136,61137,61138,61139,61140,61141,61142,61143,61144,61145,61146,61147,61148,61149,61150,61151,61152,61153,61154,61155,61156,61157,61158,61159,61160,61161,61162,61163,61164,61165,61166,61167,61168,61169,61170,61171,61172,61173,61174,61175,61176,61177,61178,61179,61180,61181,61182,61248,61249,61250,61251,61252,61253,61254,61255,61256,61257,61258,61259,61260,61261,61262,61263,61264,61265,61266,61267,61268,61269,61270,61271,61272,61273,61274,61275,61276,61277,61278,61279,61280,61281,61282,61283,61284,61285,61286,61287,61288,61289,61290,61291,61292,61293,61294,61295,61296,61297,61298,61299,61300,61301,61302,61303,61304,61305,61306,61307,61308,61309,61310,61345,61346,61347,61348,61349,61350,61351,61352,61353,61354,61355,61356,61357,61358,61359,61360,61361,61362,61363,61364,61365,61366,61367,61368,61369,61370,61371,61372,61373,61374,61375,61376,61377,61378,61379,61380,61381,61382,61383,61384,61385,61386,61387,61388,61389,61390,61391,61392,61393,61394,61395,61396,61397,61398,61399,61400,61401,61402,61403,61404,61405,61406,61407,61408,61409,61410,61411,61412,61413,61414,61415,61416,61417,61418,61419,61420,61421,61422,61423,61424,61425,61426,61427,61428,61429,61430,61431,61432,61433,61434,61435,61436,61437,61438,61504,61505,61506,61507,61508,61509,61510,61511,61512,61513,61514,61515,61516,61517,61518,61519,61520,61521,61522,61523,61524,61525,61526,61527,61528,61529,61530,61531,61532,61533,61534,61535,61536,61537,61538,61539,61540,61541,61542,61543,61544,61545,61546,61547,61548,61549,61550,61551,61552,61553,61554,61555,61556,61557,61558,61559,61560,61561,61562,61563,61564,61565,61566,61601,61602,61603,61604,61605,61606,61607,61608,61609,61610,61611,61612,61613,61614,61615,61616,61617,61618,61619,61620,61621,61622,61623,61624,61625,61626,61627,61628,61629,61630,61631,61632,61633,61634,61635,61636,61637,61638,61639,61640,61641,61642,61643,61644,61645,61646,61647,61648,61649,61650,61651,61652,61653,61654,61655,61656,61657,61658,61659,61660,61661,61662,61663,61664,61665,61666,61667,61668,61669,61670,61671,61672,61673,61674,61675,61676,61677,61678,61679,61680,61681,61682,61683,61684,61685,61686,61687,61688,61689,61690,61691,61692,61693,61694,61760,61761,61762,61763,61764,61765,61766,61767,61768,61769,61770,61771,61772,61773,61774,61775,61776,61777,61778,61779,61780,61781,61782,61783,61784,61785,61786,61787,61788,61789,61790,61791,61792,61793,61794,61795,61796,61797,61798,61799,61800,61801,61802,61803,61804,61805,61806,61807,61808,61809,61810,61811,61812,61813,61814,61815,61816,61817,61818,61819,61820,61821,61822,61857,61858,61859,61860,61861,61862,61863,61864,61865,61866,61867,61868,61869,61870,61871,61872,61873,61874,61875,61876,61877,61878,61879,61880,61881,61882,61883,61884,61885,61886,61887,61888,61889,61890,61891,61892,61893,61894,61895,61896,61897,61898,61899,61900,61901,61902,61903,61904,61905,61906,61907,61908,61909,61910,61911,61912,61913,61914,61915,61916,61917,61918,61919,61920,61921,61922,61923,61924,61925,61926,61927,61928,61929,61930,61931,61932,61933,61934,61935,61936,61937,61938,61939,61940,61941,61942,61943,61944,61945,61946,61947,61948,61949,61950,62016,62017,62018,62019,62020,62021,62022,62023,62024,62025,62026,62027,62028,62029,62030,62031,62032,62033,62034,62035,62036,62037,62038,62039,62040,62041,62042,62043,62044,62045,62046,62047,62048,62049,62050,62051,62052,62053,62054,62055,62056,62057,62058,62059,62060,62061,62062,62063,62064,62065,62066,62067,62068,62069,62070,62071,62072,62073,62074,62075,62076,62077,62078,62113,62114,62115,62116,62117,62118,62119,62120,62121,62122,62123,62124,62125,62126,62127,62128,62129,62130,62131,62132,62133,62134,62135,62136,62137,62138,62139,62140,62141,62142,62143,62144,62145,62146,62147,62148,62149,62150,62151,62152,62153,62154,62155,62156,62157,62158,62159,62160,62161,62162,62163,62164,62165,62166,62167,62168,62169,62170,62171,62172,62173,62174,62175,62176,62177,62178,62179,62180,62181,62182,62183,62184,62185,62186,62187,62188,62189,62190,62191,62192,62193,62194,62195,62196,62197,62198,62199,62200,62201,62202,62203,62204,62205,62206,62272,62273,62274,62275,62276,62277,62278,62279,62280,62281,62282,62283,62284,62285,62286,62287,62288,62289,62290,62291,62292,62293,62294,62295,62296,62297,62298,62299,62300,62301,62302,62303,62304,62305,62306,62307,62308,62309,62310,62311,62312,62313,62314,62315,62316,62317,62318,62319,62320,62321,62322,62323,62324,62325,62326,62327,62328,62329,62330,62331,62332,62333,62334,62369,62370,62371,62372,62373,62374,62375,62376,62377,62378,62379,62380,62381,62382,62383,62384,62385,62386,62387,62388,62389,62390,62391,62392,62393,62394,62395,62396,62397,62398,62399,62400,62401,62402,62403,62404,62405,62406,62407,62408,62409,62410,62411,62412,62413,62414,62415,62416,62417,62418,62419,62420,62421,62422,62423,62424,62425,62426,62427,62428,62429,62430,62431,62432,62433,62434,62435,62436,62437,62438,62439,62440,62441,62442,62443,62444,62445,62446,62447,62448,62449,62450,62451,62452,62453,62454,62455,62456,62457,62458,62459,62460,62461,62462,62528,62529,62530,62531,62532,62533,62534,62535,62536,62537,62538,62539,62540,62541,62542,62543,62544,62545,62546,62547,62548,62549,62550,62551,62552,62553,62554,62555,62556,62557,62558,62559,62560,62561,62562,62563,62564,62565,62566,62567,62568,62569,62570,62571,62572,62573,62574,62575,62576,62577,62578,62579,62580,62581,62582,62583,62584,62585,62586,62587,62588,62589,62590,62625,62626,62627,62628,62629,62630,62631,62632,62633,62634,62635,62636,62637,62638,62639,62640,62641,62642,62643,62644,62645,62646,62647,62648,62649,62650,62651,62652,62653,62654,62655,62656,62657,62658,62659,62660,62661,62662,62663,62664,62665,62666,62667,62668,62669,62670,62671,62672,62673,62674,62675,62676,62677,62678,62679,62680,62681,62682,62683,62684,62685,62686,62687,62688,62689,62690,62691,62692,62693,62694,62695,62696,62697,62698,62699,62700,62701,62702,62703,62704,62705,62706,62707,62708,62709,62710,62711,62712,62713,62714,62715,62716,62717,62718,62784,62785,62786,62787,62788,62789,62790,62791,62792,62793,62794,62795,62796,62797,62798,62799,62800,62801,62802,62803,62804,62805,62806,62807,62808,62809,62810,62811,62812,62813,62814,62815,62816,62817,62818,62819,62820,62821,62822,62823,62824,62825,62826,62827,62828,62829,62830,62831,62832,62833,62834,62835,62836,62837,62838,62839,62840,62841,62842,62843,62844,62845,62846,62881,62882,62883,62884,62885,62886,62887,62888,62889,62890,62891,62892,62893,62894,62895,62896,62897,62898,62899,62900,62901,62902,62903,62904,62905,62906,62907,62908,62909,62910,62911,62912,62913,62914,62915,62916,62917,62918,62919,62920,62921,62922,62923,62924,62925,62926,62927,62928,62929,62930,62931,62932,62933,62934,62935,62936,62937,62938,62939,62940,62941,62942,62943,62944,62945,62946,62947,62948,62949,62950,62951,62952,62953,62954,62955,62956,62957,62958,62959,62960,62961,62962,62963,62964,62965,62966,62967,62968,62969,62970,62971,62972,62973,62974,63040,63041,63042,63043,63044,63045,63046,63047,63048,63049,63050,63051,63052,63053,63054,63055,63056,63057,63058,63059,63060,63061,63062,63063,63064,63065,63066,63067,63068,63069,63070,63071,63072,63073,63074,63075,63076,63077,63078,63079,63080,63081,63082,63083,63084,63085,63086,63087,63088,63089,63090,63091,63092,63093,63094,63095,63096,63097,63098,63099,63100,63101,63102,63137,63138,63139,63140,63141,63142,63143,63144,63145,63146,63147,63148,63149,63150,63151,63152,63153,63154,63155,63156,63157,63158,63159,63160,63161,63162,63163,63164,63165,63166,63167,63168,63169,63170,63171,63172,63173,63174,63175,63176,63177,63178,63179,63180,63181,63182,63183,63184,63185,63186,63187,63188,63189,63190,63191,63192,63193,63194,63195,63196,63197,63198,63199,63200,63201,63202,63203,63204,63205,63206,63207,63208,63209,63210,63211,63212,63213,63214,63215,63216,63217,63218,63219,63220,63221,63222,63223,63224,63225,63226,63227,63228,63229,63230,63296,63297,63298,63299,63300,63301,63302,63303,63304,63305,63306,63307,63308,63309,63310,63311,63312,63313,63314,63315,63316,63317,63318,63319,63320,63321,63322,63323,63324,63325,63326,63327,63328,63329,63330,63331,63332,63333,63334,63335,63336,63337,63338,63339,63340,63341,63342,63343,63344,63345,63346,63347,63348,63349,63350,63351,63352,63353,63354,63355,63356,63357,63358,63393,63394,63395,63396,63397,63398,63399,63400,63401,63402,63403,63404,63405,63406,63407,63408,63409,63410,63411,63412,63413,63414,63415,63416,63417,63418,63419,63420,63421,63422,63423,63424,63425,63426,63427,63428,63429,63430,63431,63432,63433,63434,63435,63436,63437,63438,63439,63440,63441,63442,63443,63444,63445,63446,63447,63448,63449,63450,63451,63452,63453,63454,63455,63456,63457,63458,63459,63460,63461,63462,63463,63464,63465,63466,63467,63468,63469,63470,63471,63472,63473,63474,63475,63476,63477,63478,63479,63480,63481,63482,63483,63484,63485,63486,63552,63553,63554,63555,63556,63557,63558,63559,63560,63561,63562,63563,63564,63565,63566,63567,63568,63569,63570,63571,63572,63573,63574,63575,63576,63577,63578,63579,63580,63581,63582,63583,63584,63585,63586,63587,63588,63589,63590,63591,63592,63593,63594,63595,63596,63597,63598,63599,63600,63601,63602,63603,63604,63605,63606,63607,63608,63609,63610,63611,63612,63613,63614,63649,63650,63651,63652,63653,63654,63655,63656,63657,63658,63659,63660,63661,63662,63663,63664,63665,63666,63667,63668,63669,63670,63671,63672,63673,63674,63675,63676,63677,63678,63679,63680,63681,63682,63683,63684,63685,63686,63687,63688,63689,63690,63691,63692,63693,63694,63695,63696,63697,63698,63699,63700,63701,63702,63703,63704,63705,63706,63707,63708,63709,63710,63711,63712,63713,63714,63715,63716,63717,63718,63719,63720,63721,63722,63723,63724,63725,63726,63727,63728,63729,63730,63731,63732,63733,63734,63735,63736,63737,63738,63739,63740,63741,63742,63808,63809,63810,63811,63812,63813,63814,63815,63816,63817,63818,63819,63820,63821,63822,63823,63824,63825,63826,63827,63828,63829,63830,63831,63832,63833,63834,63835,63836,63837,63838,63839,63840,63841,63842,63843,63844,63845,63846,63847,63848,63849,63850,63851,63852,63853,63854,63855,63856,63857,63858,63859,63860,63861,63862,63863,63864,63865,63866,63867,63868,63869,63870,63905,63906,63907,63908,63909,63910,63911,63912,63913,63914,63915,63916,63917,63918,63919,63920,63921,63922,63923,63924,63925,63926,63927,63928,63929,63930,63931,63932,63933,63934,63935,63936,63937,63938,63939,63940,63941,63942,63943,63944,63945,63946,63947,63948,63949,63950,63951,63952,63953,63954,63955,63956,63957,63958,63959,63960,63961,63962,63963,63964,63965,63966,63967,63968,63969,63970,63971,63972,63973,63974,63975,63976,63977,63978,63979,63980,63981,63982,63983,63984,63985,63986,63987,63988,63989,63990,63991,63992,63993,63994,63995,63996,63997,63998,64064,64065,64066,64067,64068,64069,64070,64071,64072,64073,64074,64075,64076,64077,64078,64079,64080,64081,64082,64083,64084,64085,64086,64087,64088,64089,64090,64091,64092,64093,64094,64095,64096,64097,64098,64099,64100,64101,64102,64103,64104,64105,64106,64107,64108,64109,64110,64111,64112,64113,64114,64115,64116,64117,64118,64119,64120,64121,64122,64123,64124,64125,64126,64161,64162,64163,64164,64165,64166,64167,64168,64169,64170,64171,64172,64173,64174,64175,64176,64177,64178,64179,64180,64181,64182,64183,64184,64185,64186,64187,64188,64189,64190,64191,64192,64193,64194,64195,64196,64197,64198,64199,64200,64201,64202,64203,64204,64205,64206,64207,64208,64209,64210,64211,64212,64213,64214,64215,64216,64217,64218,64219,64220,64221,64222,64223,64224,64225,64226,64227,64228,64229,64230,64231,64232,64233,64234,64235,64236,64237,64238,64239,64240,64241,64242,64243,64244,64245,64246,64247,64248,64249,64250,64251,64252,64253,64254,64320,64321,64322,64323,64324,64325,64326,64327,64328,64329,64330,64331,64332,64333,64334,64335,64336,64337,64338,64339,64340,64341,64342,64343,64344,64345,64346,64347,64348,64349,64350,64351,64352,64353,64354,64355,64356,64357,64358,64359,64360,64361,64362,64363,64364,64365,64366,64367,64368,64369,64370,64371,64372,64373,64374,64375,64376,64377,64378,64379,64380,64381,64382,64417,64418,64419,64420,64421,64422,64423,64424,64425,64426,64427,64428,64429,64430,64431,64432,64433,64434,64435,64436,64437,64438,64439,64440,64441,64442,64443,64444,64445,64446,64447,64448,64449,64450,64451,64452,64453,64454,64455,64456,64457,64458,64459,64460,64461,64462,64463,64464,64465,64466,64467,64468,64469,64470,64471,64472,64473,64474,64475,64476,64477,64478,64479,64480,64481,64482,64483,64484,64485,64486,64487,64488,64489,64490,64491,64492,64493,64494,64495,64496,64497,64498,64499,64500,64501,64502,64503,64504,64505,64506,64507,64508,64509,64510,64576,64577,64578,64579,64580,64581,64582,64583,64584,64585,64586,64587,64588,64589,64590,64591,64592,64593,64594,64595,64596,64597,64598,64599,64600,64601,64602,64603,64604,64605,64606,64607,64608,64609,64610,64611,64612,64613,64614,64615,64616,64617,64618,64619,64620,64621,64622,64623,64624,64625,64626,64627,64628,64629,64630,64631,64632,64633,64634,64635,64636,64637,64638,64673,64674,64675,64676,64677,64678,64679,64680,64681,64682,64683,64684,64685,64686,64687,64688,64689,64690,64691,64692,64693,64694,64695,64696,64697,64698,64699,64700,64701,64702,64703,64704,64705,64706,64707,64708,64709,64710,64711,64712,64713,64714,64715,64716,64717,64718,64719,64720,64721,64722,64723,64724,64725,64726,64727,64728,64729,64730,64731,64732,64733,64734,64735,64736,64737,64738,64739,64740,64741,64742,64743,64744,64745,64746,64747,64748,64749,64750,64751,64752,64753,64754,64755,64756,64757,64758,64759,64760,64761,64762,64763,64764,64765,64766,64832,64833,64834,64835,64836,64837,64838,64839,64840,64841,64842,64843,64844,64845,64846,64847,64848,64849,64850,64851,64852,64853,64854,64855,64856,64857,64858,64859,64860,64861,64862,64863,64864,64865,64866,64867,64868,64869,64870,64871,64872,64873,64874,64875,64876,64877,64878,64879,64880,64881,64882,64883,64884,64885,64886,64887,64888,64889,64890,64891,64892,64893,64894,64929,64930,64931,64932,64933,64934,64935,64936,64937,64938,64939,64940,64941,64942,64943,64944,64945,64946,64947,64948,64949,64950,64951,64952,64953,64954,64955,64956,64957,64958,64959,64960,64961,64962,64963,64964,64965,64966,64967,64968,64969,64970,64971,64972,64973,64974,64975,64976,64977,64978,64979,64980,64981,64982,64983,64984,64985,64986,64987,64988,64989,64990,64991,64992,64993,64994,64995,64996,64997,64998,64999,65000,65001,65002,65003,65004,65005,65006,65007,65008,65009,65010,65011,65012,65013,65014,65015,65016,65017,65018,65019,65020,65021,65022,65088,65089,65090,65091,65092,65093,65094,65095,65096,65097,65098,65099,65100,65101,65102,65103,65104,65105,65106,65107,65108,65109,65110,65111,65112,65113,65114,65115,65116,65117,65118,65119,65120,65121,65122,65123,65124,65125,65126,65127,65128,65129,65130,65131,65132,65133,65134,65135,65136,65137,65138,65139,65140,65141,65142,65143,65144,65145,65146,65147,65148,65149,65150,65185,65186,65187,65188,65189,65190,65191,65192,65193,65194,65195,65196,65197,65198,65199,65200,65201,65202,65203,65204,65205,65206,65207,65208,65209,65210,65211,65212,65213,65214,65215,65216,65217,65218,65219,65220,65221,65222,65223,65224,65225,65226,65227,65228,65229,65230,65231,65232,65233,65234,65235,65236,65237,65238,65239,65240,65241,65242,65243,65244,65245,65246,65247,65248,65249,65250,65251,65252,65253,65254,65255,65256,65257,65258,65259,65260,65261,65262,65263,65264,65265,65266,65267,65268,65269,65270,65271,65272,65273,65274,65275,65276,65277,65278 };
unsigned short big5_to_unicode_vals[] = { 61112,61113,61114,61115,61116,61117,61118,61119,61120,61121,61122,61123,61124,61125,61126,61127,61128,61129,61130,61131,61132,61133,61134,61135,61136,61137,61138,61139,61140,61141,61142,61143,61144,61145,61146,61147,61148,61149,61150,61151,61152,61153,61154,61155,61156,61157,61158,61159,61160,61161,61162,61163,61164,61165,61166,61167,61168,61169,61170,61171,61172,61173,61174,61175,61176,61177,61178,61179,61180,61181,61182,61183,61184,61185,61186,61187,61188,61189,61190,61191,61192,61193,61194,61195,61196,61197,61198,61199,61200,61201,61202,61203,61204,61205,61206,61207,61208,61209,61210,61211,61212,61213,61214,61215,61216,61217,61218,61219,61220,61221,61222,61223,61224,61225,61226,61227,61228,61229,61230,61231,61232,61233,61234,61235,61236,61237,61238,61239,61240,61241,61242,61243,61244,61245,61246,61247,61248,61249,61250,61251,61252,61253,61254,61255,61256,61257,61258,61259,61260,61261,61262,61263,61264,61265,61266,61267,61268,61269,61270,61271,61272,61273,61274,61275,61276,61277,61278,61279,61280,61281,61282,61283,61284,61285,61286,61287,61288,61289,61290,61291,61292,61293,61294,61295,61296,61297,61298,61299,61300,61301,61302,61303,61304,61305,61306,61307,61308,61309,61310,61311,61312,61313,61314,61315,61316,61317,61318,61319,61320,61321,61322,61323,61324,61325,61326,61327,61328,61329,61330,61331,61332,61333,61334,61335,61336,61337,61338,61339,61340,61341,61342,61343,61344,61345,61346,61347,61348,61349,61350,61351,61352,61353,61354,61355,61356,61357,61358,61359,61360,61361,61362,61363,61364,61365,61366,61367,61368,61369,61370,61371,61372,61373,61374,61375,61376,61377,61378,61379,61380,61381,61382,61383,61384,61385,61386,61387,61388,61389,61390,61391,61392,61393,61394,61395,61396,61397,61398,61399,61400,61401,61402,61403,61404,61405,61406,61407,61408,61409,61410,61411,61412,61413,61414,61415,61416,61417,61418,61419,61420,61421,61422,61423,61424,61425,61426,61427,61428,61429,61430,61431,61432,61433,61434,61435,61436,61437,61438,61439,61440,61441,61442,61443,61444,61445,61446,61447,61448,61449,61450,61451,61452,61453,61454,61455,61456,61457,61458,61459,61460,61461,61462,61463,61464,61465,61466,61467,61468,61469,61470,61471,61472,61473,61474,61475,61476,61477,61478,61479,61480,61481,61482,61483,61484,61485,61486,61487,61488,61489,61490,61491,61492,61493,61494,61495,61496,61497,61498,61499,61500,61501,61502,61503,61504,61505,61506,61507,61508,61509,61510,61511,61512,61513,61514,61515,61516,61517,61518,61519,61520,61521,61522,61523,61524,61525,61526,61527,61528,61529,61530,61531,61532,61533,61534,61535,61536,61537,61538,61539,61540,61541,61542,61543,61544,61545,61546,61547,61548,61549,61550,61551,61552,61553,61554,61555,61556,61557,61558,61559,61560,61561,61562,61563,61564,61565,61566,61567,61568,61569,61570,61571,61572,61573,61574,61575,61576,61577,61578,61579,61580,61581,61582,61583,61584,61585,61586,61587,61588,61589,61590,61591,61592,61593,61594,61595,61596,61597,61598,61599,61600,61601,61602,61603,61604,61605,61606,61607,61608,61609,61610,61611,61612,61613,61614,61615,61616,61617,61618,61619,61620,61621,61622,61623,61624,61625,61626,61627,61628,61629,61630,61631,61632,61633,61634,61635,61636,61637,61638,61639,61640,61641,61642,61643,61644,61645,61646,61647,61648,61649,61650,61651,61652,61653,61654,61655,61656,61657,61658,61659,61660,61661,61662,61663,61664,61665,61666,61667,61668,61669,61670,61671,61672,61673,61674,61675,61676,61677,61678,61679,61680,61681,61682,61683,61684,61685,61686,61687,61688,61689,61690,61691,61692,61693,61694,61695,61696,61697,61698,61699,61700,61701,61702,61703,61704,61705,61706,61707,61708,61709,61710,61711,61712,61713,61714,61715,61716,61717,61718,61719,61720,61721,61722,61723,61724,61725,61726,61727,61728,61729,61730,61731,61732,61733,61734,61735,61736,61737,61738,61739,61740,61741,61742,61743,61744,61745,61746,61747,61748,61749,61750,61751,61752,61753,61754,61755,61756,61757,61758,61759,61760,61761,61762,61763,61764,61765,61766,61767,61768,61769,61770,61771,61772,61773,61774,61775,61776,61777,61778,61779,61780,61781,61782,61783,61784,61785,61786,61787,61788,61789,61790,61791,61792,61793,61794,61795,61796,61797,61798,61799,61800,61801,61802,61803,61804,61805,61806,61807,61808,61809,61810,61811,61812,61813,61814,61815,61816,61817,61818,61819,61820,61821,61822,61823,61824,61825,61826,61827,61828,61829,61830,61831,61832,61833,61834,61835,61836,61837,61838,61839,61840,61841,61842,61843,61844,61845,61846,61847,61848,61849,61850,61851,61852,61853,61854,61855,61856,61857,61858,61859,61860,61861,61862,61863,61864,61865,61866,61867,61868,61869,61870,61871,61872,61873,61874,61875,61876,61877,61878,61879,61880,61881,61882,61883,61884,61885,61886,61887,61888,61889,61890,61891,61892,61893,61894,61895,61896,61897,61898,61899,61900,61901,61902,61903,61904,61905,61906,61907,61908,61909,61910,61911,61912,61913,61914,61915,61916,61917,61918,61919,61920,61921,61922,61923,61924,61925,61926,61927,61928,61929,61930,61931,61932,61933,61934,61935,61936,61937,61938,61939,61940,61941,61942,61943,61944,61945,61946,61947,61948,61949,61950,61951,61952,61953,61954,61955,61956,61957,61958,61959,61960,61961,61962,61963,61964,61965,61966,61967,61968,61969,61970,61971,61972,61973,61974,61975,61976,61977,61978,61979,61980,61981,61982,61983,61984,61985,61986,61987,61988,61989,61990,61991,61992,61993,61994,61995,61996,61997,61998,61999,62000,62001,62002,62003,62004,62005,62006,62007,62008,62009,62010,62011,62012,62013,62014,62015,62016,62017,62018,62019,62020,62021,62022,62023,62024,62025,62026,62027,62028,62029,62030,62031,62032,62033,62034,62035,62036,62037,62038,62039,62040,62041,62042,62043,62044,62045,62046,62047,62048,62049,62050,62051,62052,62053,62054,62055,62056,62057,62058,62059,62060,62061,62062,62063,62064,62065,62066,62067,62068,62069,62070,62071,62072,62073,62074,62075,62076,62077,62078,62079,62080,62081,62082,62083,62084,62085,62086,62087,62088,62089,62090,62091,62092,62093,62094,62095,62096,62097,62098,62099,62100,62101,62102,62103,62104,62105,62106,62107,62108,62109,62110,62111,62112,62113,62114,62115,62116,62117,62118,62119,62120,62121,62122,62123,62124,62125,62126,62127,62128,62129,62130,62131,62132,62133,62134,62135,62136,62137,62138,62139,62140,62141,62142,62143,62144,62145,62146,62147,62148,62149,62150,62151,62152,62153,62154,62155,62156,62157,62158,62159,62160,62161,62162,62163,62164,62165,62166,62167,62168,62169,62170,62171,62172,62173,62174,62175,62176,62177,62178,62179,62180,62181,62182,62183,62184,62185,62186,62187,62188,62189,62190,62191,62192,62193,62194,62195,62196,62197,62198,62199,62200,62201,62202,62203,62204,62205,62206,62207,62208,62209,62210,62211,62212,62213,62214,62215,62216,62217,62218,62219,62220,62221,62222,62223,62224,62225,62226,62227,62228,62229,62230,62231,62232,62233,62234,62235,62236,62237,62238,62239,62240,62241,62242,62243,62244,62245,62246,62247,62248,62249,62250,62251,62252,62253,62254,62255,62256,62257,62258,62259,62260,62261,62262,62263,62264,62265,62266,62267,62268,62269,62270,62271,62272,62273,62274,62275,62276,62277,62278,62279,62280,62281,62282,62283,62284,62285,62286,62287,62288,62289,62290,62291,62292,62293,62294,62295,62296,62297,62298,62299,62300,62301,62302,62303,62304,62305,62306,62307,62308,62309,62310,62311,62312,62313,62314,62315,62316,62317,62318,62319,62320,62321,62322,62323,62324,62325,62326,62327,62328,62329,62330,62331,62332,62333,62334,62335,62336,62337,62338,62339,62340,62341,62342,62343,62344,62345,62346,62347,62348,62349,62350,62351,62352,62353,62354,62355,62356,62357,62358,62359,62360,62361,62362,62363,62364,62365,62366,62367,62368,62369,62370,62371,62372,62373,62374,62375,62376,62377,62378,62379,62380,62381,62382,62383,62384,62385,62386,62387,62388,62389,62390,62391,62392,62393,62394,62395,62396,62397,62398,62399,62400,62401,62402,62403,62404,62405,62406,62407,62408,62409,62410,62411,62412,62413,62414,62415,62416,62417,62418,62419,62420,62421,62422,62423,62424,62425,62426,62427,62428,62429,62430,62431,62432,62433,62434,62435,62436,62437,62438,62439,62440,62441,62442,62443,62444,62445,62446,62447,62448,62449,62450,62451,62452,62453,62454,62455,62456,62457,62458,62459,62460,62461,62462,62463,62464,62465,62466,62467,62468,62469,62470,62471,62472,62473,62474,62475,62476,62477,62478,62479,62480,62481,62482,62483,62484,62485,62486,62487,62488,62489,62490,62491,62492,62493,62494,62495,62496,62497,62498,62499,62500,62501,62502,62503,62504,62505,62506,62507,62508,62509,62510,62511,62512,62513,62514,62515,62516,62517,62518,62519,62520,62521,62522,62523,62524,62525,62526,62527,62528,62529,62530,62531,62532,62533,62534,62535,62536,62537,62538,62539,62540,62541,62542,62543,62544,62545,62546,62547,62548,62549,62550,62551,62552,62553,62554,62555,62556,62557,62558,62559,62560,62561,62562,62563,62564,62565,62566,62567,62568,62569,62570,62571,62572,62573,62574,62575,62576,62577,62578,62579,62580,62581,62582,62583,62584,62585,62586,62587,62588,62589,62590,62591,62592,62593,62594,62595,62596,62597,62598,62599,62600,62601,62602,62603,62604,62605,62606,62607,62608,62609,62610,62611,62612,62613,62614,62615,62616,62617,62618,62619,62620,62621,62622,62623,62624,62625,62626,62627,62628,62629,62630,62631,62632,62633,62634,62635,62636,62637,62638,62639,62640,62641,62642,62643,62644,62645,62646,62647,62648,62649,62650,62651,62652,62653,62654,62655,62656,62657,62658,62659,62660,62661,62662,62663,62664,62665,62666,62667,62668,62669,62670,62671,62672,62673,62674,62675,62676,62677,62678,62679,62680,62681,62682,62683,62684,62685,62686,62687,62688,62689,62690,62691,62692,62693,62694,62695,62696,62697,62698,62699,62700,62701,62702,62703,62704,62705,62706,62707,62708,62709,62710,62711,62712,62713,62714,62715,62716,62717,62718,62719,62720,62721,62722,62723,62724,62725,62726,62727,62728,62729,62730,62731,62732,62733,62734,62735,62736,62737,62738,62739,62740,62741,62742,62743,62744,62745,62746,62747,62748,62749,62750,62751,62752,62753,62754,62755,62756,62757,62758,62759,62760,62761,62762,62763,62764,62765,62766,62767,62768,62769,62770,62771,62772,62773,62774,62775,62776,62777,62778,62779,62780,62781,62782,62783,62784,62785,62786,62787,62788,62789,62790,62791,62792,62793,62794,62795,62796,62797,62798,62799,62800,62801,62802,62803,62804,62805,62806,62807,62808,62809,62810,62811,62812,62813,62814,62815,62816,62817,62818,62819,62820,62821,62822,62823,62824,62825,62826,62827,62828,62829,62830,62831,62832,62833,62834,62835,62836,62837,62838,62839,62840,62841,62842,62843,62844,62845,62846,62847,62848,62849,62850,62851,62852,62853,62854,62855,62856,62857,62858,62859,62860,62861,62862,62863,62864,62865,62866,62867,62868,62869,62870,62871,62872,62873,62874,62875,62876,62877,62878,62879,62880,62881,62882,62883,62884,62885,62886,62887,62888,62889,62890,62891,62892,62893,62894,62895,62896,62897,62898,62899,62900,62901,62902,62903,62904,62905,62906,62907,62908,62909,62910,62911,62912,62913,62914,62915,62916,62917,62918,62919,62920,62921,62922,62923,62924,62925,62926,62927,62928,62929,62930,62931,62932,62933,62934,62935,62936,62937,62938,62939,62940,62941,62942,62943,62944,62945,62946,62947,62948,62949,62950,62951,62952,62953,62954,62955,62956,62957,62958,62959,62960,62961,62962,62963,62964,62965,62966,62967,62968,62969,62970,62971,62972,62973,62974,62975,62976,62977,62978,62979,62980,62981,62982,62983,62984,62985,62986,62987,62988,62989,62990,62991,62992,62993,62994,62995,62996,62997,62998,62999,63000,63001,63002,63003,63004,63005,63006,63007,63008,63009,63010,63011,63012,63013,63014,63015,63016,63017,63018,63019,63020,63021,63022,63023,63024,63025,63026,63027,63028,63029,63030,63031,63032,63033,63034,63035,63036,63037,63038,63039,63040,63041,63042,63043,63044,63045,63046,63047,63048,63049,63050,63051,63052,63053,63054,63055,63056,63057,63058,63059,63060,63061,63062,63063,63064,63065,63066,63067,63068,63069,63070,63071,63072,63073,63074,63075,63076,63077,63078,63079,63080,63081,63082,63083,63084,63085,63086,63087,63088,63089,63090,63091,63092,63093,63094,63095,63096,63097,63098,63099,63100,63101,63102,63103,63104,63105,63106,63107,63108,63109,63110,63111,63112,63113,63114,63115,63116,63117,63118,63119,63120,63121,63122,63123,63124,63125,63126,63127,63128,63129,63130,63131,63132,63133,63134,63135,63136,63137,63138,63139,63140,63141,63142,63143,63144,63145,63146,63147,63148,63149,63150,63151,63152,58129,58130,58131,58132,58133,58134,58135,58136,58137,58138,58139,58140,58141,58142,58143,58144,58145,58146,58147,58148,58149,58150,58151,58152,58153,58154,58155,58156,58157,58158,58159,58160,58161,58162,58163,58164,58165,58166,58167,58168,58169,58170,58171,58172,58173,58174,58175,58176,58177,58178,58179,58180,58181,58182,58183,58184,58185,58186,58187,58188,58189,58190,58191,58192,58193,58194,58195,58196,58197,58198,58199,58200,58201,58202,58203,58204,58205,58206,58207,58208,58209,58210,58211,58212,58213,58214,58215,58216,58217,58218,58219,58220,58221,58222,58223,58224,58225,58226,58227,58228,58229,58230,58231,58232,58233,58234,58235,58236,58237,58238,58239,58240,58241,58242,58243,58244,58245,58246,58247,58248,58249,58250,58251,58252,58253,58254,58255,58256,58257,58258,58259,58260,58261,58262,58263,58264,58265,58266,58267,58268,58269,58270,58271,58272,58273,58274,58275,58276,58277,58278,58279,58280,58281,58282,58283,58284,58285,58286,58287,58288,58289,58290,58291,58292,58293,58294,58295,58296,58297,58298,58299,58300,58301,58302,58303,58304,58305,58306,58307,58308,58309,58310,58311,58312,58313,58314,58315,58316,58317,58318,58319,58320,58321,58322,58323,58324,58325,58326,58327,58328,58329,58330,58331,58332,58333,58334,58335,58336,58337,58338,58339,58340,58341,58342,58343,58344,58345,58346,58347,58348,58349,58350,58351,58352,58353,58354,58355,58356,58357,58358,58359,58360,58361,58362,58363,58364,58365,58366,58367,58368,58369,58370,58371,58372,58373,58374,58375,58376,58377,58378,58379,58380,58381,58382,58383,58384,58385,58386,58387,58388,58389,58390,58391,58392,58393,58394,58395,58396,58397,58398,58399,58400,58401,58402,58403,58404,58405,58406,58407,58408,58409,58410,58411,58412,58413,58414,58415,58416,58417,58418,58419,58420,58421,58422,58423,58424,58425,58426,58427,58428,58429,58430,58431,58432,58433,58434,58435,58436,58437,58438,58439,58440,58441,58442,58443,58444,58445,58446,58447,58448,58449,58450,58451,58452,58453,58454,58455,58456,58457,58458,58459,58460,58461,58462,58463,58464,58465,58466,58467,58468,58469,58470,58471,58472,58473,58474,58475,58476,58477,58478,58479,58480,58481,58482,58483,58484,58485,58486,58487,58488,58489,58490,58491,58492,58493,58494,58495,58496,58497,58498,58499,58500,58501,58502,58503,58504,58505,58506,58507,58508,58509,58510,58511,58512,58513,58514,58515,58516,58517,58518,58519,58520,58521,58522,58523,58524,58525,58526,58527,58528,58529,58530,58531,58532,58533,58534,58535,58536,58537,58538,58539,58540,58541,58542,58543,58544,58545,58546,58547,58548,58549,58550,58551,58552,58553,58554,58555,58556,58557,58558,58559,58560,58561,58562,58563,58564,58565,58566,58567,58568,58569,58570,58571,58572,58573,58574,58575,58576,58577,58578,58579,58580,58581,58582,58583,58584,58585,58586,58587,58588,58589,58590,58591,58592,58593,58594,58595,58596,58597,58598,58599,58600,58601,58602,58603,58604,58605,58606,58607,58608,58609,58610,58611,58612,58613,58614,58615,58616,58617,58618,58619,58620,58621,58622,58623,58624,58625,58626,58627,58628,58629,58630,58631,58632,58633,58634,58635,58636,58637,58638,58639,58640,58641,58642,58643,58644,58645,58646,58647,58648,58649,58650,58651,58652,58653,58654,58655,58656,58657,58658,58659,58660,58661,58662,58663,58664,58665,58666,58667,58668,58669,58670,58671,58672,58673,58674,58675,58676,58677,58678,58679,58680,58681,58682,58683,58684,58685,58686,58687,58688,58689,58690,58691,58692,58693,58694,58695,58696,58697,58698,58699,58700,58701,58702,58703,58704,58705,58706,58707,58708,58709,58710,58711,58712,58713,58714,58715,58716,58717,58718,58719,58720,58721,58722,58723,58724,58725,58726,58727,58728,58729,58730,58731,58732,58733,58734,58735,58736,58737,58738,58739,58740,58741,58742,58743,58744,58745,58746,58747,58748,58749,58750,58751,58752,58753,58754,58755,58756,58757,58758,58759,58760,58761,58762,58763,58764,58765,58766,58767,58768,58769,58770,58771,58772,58773,58774,58775,58776,58777,58778,58779,58780,58781,58782,58783,58784,58785,58786,58787,58788,58789,58790,58791,58792,58793,58794,58795,58796,58797,58798,58799,58800,58801,58802,58803,58804,58805,58806,58807,58808,58809,58810,58811,58812,58813,58814,58815,58816,58817,58818,58819,58820,58821,58822,58823,58824,58825,58826,58827,58828,58829,58830,58831,58832,58833,58834,58835,58836,58837,58838,58839,58840,58841,58842,58843,58844,58845,58846,58847,58848,58849,58850,58851,58852,58853,58854,58855,58856,58857,58858,58859,58860,58861,58862,58863,58864,58865,58866,58867,58868,58869,58870,58871,58872,58873,58874,58875,58876,58877,58878,58879,58880,58881,58882,58883,58884,58885,58886,58887,58888,58889,58890,58891,58892,58893,58894,58895,58896,58897,58898,58899,58900,58901,58902,58903,58904,58905,58906,58907,58908,58909,58910,58911,58912,58913,58914,58915,58916,58917,58918,58919,58920,58921,58922,58923,58924,58925,58926,58927,58928,58929,58930,58931,58932,58933,58934,58935,58936,58937,58938,58939,58940,58941,58942,58943,58944,58945,58946,58947,58948,58949,58950,58951,58952,58953,58954,58955,58956,58957,58958,58959,58960,58961,58962,58963,58964,58965,58966,58967,58968,58969,58970,58971,58972,58973,58974,58975,58976,58977,58978,58979,58980,58981,58982,58983,58984,58985,58986,58987,58988,58989,58990,58991,58992,58993,58994,58995,58996,58997,58998,58999,59000,59001,59002,59003,59004,59005,59006,59007,59008,59009,59010,59011,59012,59013,59014,59015,59016,59017,59018,59019,59020,59021,59022,59023,59024,59025,59026,59027,59028,59029,59030,59031,59032,59033,59034,59035,59036,59037,59038,59039,59040,59041,59042,59043,59044,59045,59046,59047,59048,59049,59050,59051,59052,59053,59054,59055,59056,59057,59058,59059,59060,59061,59062,59063,59064,59065,59066,59067,59068,59069,59070,59071,59072,59073,59074,59075,59076,59077,59078,59079,59080,59081,59082,59083,59084,59085,59086,59087,59088,59089,59090,59091,59092,59093,59094,59095,59096,59097,59098,59099,59100,59101,59102,59103,59104,59105,59106,59107,59108,59109,59110,59111,59112,59113,59114,59115,59116,59117,59118,59119,59120,59121,59122,59123,59124,59125,59126,59127,59128,59129,59130,59131,59132,59133,59134,59135,59136,59137,59138,59139,59140,59141,59142,59143,59144,59145,59146,59147,59148,59149,59150,59151,59152,59153,59154,59155,59156,59157,59158,59159,59160,59161,59162,59163,59164,59165,59166,59167,59168,59169,59170,59171,59172,59173,59174,59175,59176,59177,59178,59179,59180,59181,59182,59183,59184,59185,59186,59187,59188,59189,59190,59191,59192,59193,59194,59195,59196,59197,59198,59199,59200,59201,59202,59203,59204,59205,59206,59207,59208,59209,59210,59211,59212,59213,59214,59215,59216,59217,59218,59219,59220,59221,59222,59223,59224,59225,59226,59227,59228,59229,59230,59231,59232,59233,59234,59235,59236,59237,59238,59239,59240,59241,59242,59243,59244,59245,59246,59247,59248,59249,59250,59251,59252,59253,59254,59255,59256,59257,59258,59259,59260,59261,59262,59263,59264,59265,59266,59267,59268,59269,59270,59271,59272,59273,59274,59275,59276,59277,59278,59279,59280,59281,59282,59283,59284,59285,59286,59287,59288,59289,59290,59291,59292,59293,59294,59295,59296,59297,59298,59299,59300,59301,59302,59303,59304,59305,59306,59307,59308,59309,59310,59311,59312,59313,59314,59315,59316,59317,59318,59319,59320,59321,59322,59323,59324,59325,59326,59327,59328,59329,59330,59331,59332,59333,59334,59335,59336,59337,59338,59339,59340,59341,59342,59343,59344,59345,59346,59347,59348,59349,59350,59351,59352,59353,59354,59355,59356,59357,59358,59359,59360,59361,59362,59363,59364,59365,59366,59367,59368,59369,59370,59371,59372,59373,59374,59375,59376,59377,59378,59379,59380,59381,59382,59383,59384,59385,59386,59387,59388,59389,59390,59391,59392,59393,59394,59395,59396,59397,59398,59399,59400,59401,59402,59403,59404,59405,59406,59407,59408,59409,59410,59411,59412,59413,59414,59415,59416,59417,59418,59419,59420,59421,59422,59423,59424,59425,59426,59427,59428,59429,59430,59431,59432,59433,59434,59435,59436,59437,59438,59439,59440,59441,59442,59443,59444,59445,59446,59447,59448,59449,59450,59451,59452,59453,59454,59455,59456,59457,59458,59459,59460,59461,59462,59463,59464,59465,59466,59467,59468,59469,59470,59471,59472,59473,59474,59475,59476,59477,59478,59479,59480,59481,59482,59483,59484,59485,59486,59487,59488,59489,59490,59491,59492,59493,59494,59495,59496,59497,59498,59499,59500,59501,59502,59503,59504,59505,59506,59507,59508,59509,59510,59511,59512,59513,59514,59515,59516,59517,59518,59519,59520,59521,59522,59523,59524,59525,59526,59527,59528,59529,59530,59531,59532,59533,59534,59535,59536,59537,59538,59539,59540,59541,59542,59543,59544,59545,59546,59547,59548,59549,59550,59551,59552,59553,59554,59555,59556,59557,59558,59559,59560,59561,59562,59563,59564,59565,59566,59567,59568,59569,59570,59571,59572,59573,59574,59575,59576,59577,59578,59579,59580,59581,59582,59583,59584,59585,59586,59587,59588,59589,59590,59591,59592,59593,59594,59595,59596,59597,59598,59599,59600,59601,59602,59603,59604,59605,59606,59607,59608,59609,59610,59611,59612,59613,59614,59615,59616,59617,59618,59619,59620,59621,59622,59623,59624,59625,59626,59627,59628,59629,59630,59631,59632,59633,59634,59635,59636,59637,59638,59639,59640,59641,59642,59643,59644,59645,59646,59647,59648,59649,59650,59651,59652,59653,59654,59655,59656,59657,59658,59659,59660,59661,59662,59663,59664,59665,59666,59667,59668,59669,59670,59671,59672,59673,59674,59675,59676,59677,59678,59679,59680,59681,59682,59683,59684,59685,59686,59687,59688,59689,59690,59691,59692,59693,59694,59695,59696,59697,59698,59699,59700,59701,59702,59703,59704,59705,59706,59707,59708,59709,59710,59711,59712,59713,59714,59715,59716,59717,59718,59719,59720,59721,59722,59723,59724,59725,59726,59727,59728,59729,59730,59731,59732,59733,59734,59735,59736,59737,59738,59739,59740,59741,59742,59743,59744,59745,59746,59747,59748,59749,59750,59751,59752,59753,59754,59755,59756,59757,59758,59759,59760,59761,59762,59763,59764,59765,59766,59767,59768,59769,59770,59771,59772,59773,59774,59775,59776,59777,59778,59779,59780,59781,59782,59783,59784,59785,59786,59787,59788,59789,59790,59791,59792,59793,59794,59795,59796,59797,59798,59799,59800,59801,59802,59803,59804,59805,59806,59807,59808,59809,59810,59811,59812,59813,59814,59815,59816,59817,59818,59819,59820,59821,59822,59823,59824,59825,59826,59827,59828,59829,59830,59831,59832,59833,59834,59835,59836,59837,59838,59839,59840,59841,59842,59843,59844,59845,59846,59847,59848,59849,59850,59851,59852,59853,59854,59855,59856,59857,59858,59859,59860,59861,59862,59863,59864,59865,59866,59867,59868,59869,59870,59871,59872,59873,59874,59875,59876,59877,59878,59879,59880,59881,59882,59883,59884,59885,59886,59887,59888,59889,59890,59891,59892,59893,59894,59895,59896,59897,59898,59899,59900,59901,59902,59903,59904,59905,59906,59907,59908,59909,59910,59911,59912,59913,59914,59915,59916,59917,59918,59919,59920,59921,59922,59923,59924,59925,59926,59927,59928,59929,59930,59931,59932,59933,59934,59935,59936,59937,59938,59939,59940,59941,59942,59943,59944,59945,59946,59947,59948,59949,59950,59951,59952,59953,59954,59955,59956,59957,59958,59959,59960,59961,59962,59963,59964,59965,59966,59967,59968,59969,59970,59971,59972,59973,59974,59975,59976,59977,59978,59979,59980,59981,59982,59983,59984,59985,59986,59987,59988,59989,59990,59991,59992,59993,59994,59995,59996,59997,59998,59999,60000,60001,60002,60003,60004,60005,60006,60007,60008,60009,60010,60011,60012,60013,60014,60015,60016,60017,60018,60019,60020,60021,60022,60023,60024,60025,60026,60027,60028,60029,60030,60031,60032,60033,60034,60035,60036,60037,60038,60039,60040,60041,60042,60043,60044,60045,60046,60047,60048,60049,60050,60051,60052,60053,60054,60055,60056,60057,60058,60059,60060,60061,60062,60063,60064,60065,60066,60067,60068,60069,60070,60071,60072,60073,60074,60075,60076,60077,60078,60079,60080,60081,60082,60083,60084,60085,60086,60087,60088,60089,60090,60091,60092,60093,60094,60095,60096,60097,60098,60099,60100,60101,60102,60103,60104,60105,60106,60107,60108,60109,60110,60111,60112,60113,60114,60115,60116,60117,60118,60119,60120,60121,60122,60123,60124,60125,60126,60127,60128,60129,60130,60131,60132,60133,60134,60135,60136,60137,60138,60139,60140,60141,60142,60143,60144,60145,60146,60147,60148,60149,60150,60151,60152,60153,60154,60155,60156,60157,60158,60159,60160,60161,60162,60163,60164,60165,60166,60167,60168,60169,60170,60171,60172,60173,60174,60175,60176,60177,60178,60179,60180,60181,60182,60183,60184,60185,60186,60187,60188,60189,60190,60191,60192,60193,60194,60195,60196,60197,60198,60199,60200,60201,60202,60203,60204,60205,60206,60207,60208,60209,60210,60211,60212,60213,60214,60215,60216,60217,60218,60219,60220,60221,60222,60223,60224,60225,60226,60227,60228,60229,60230,60231,60232,60233,60234,60235,60236,60237,60238,60239,60240,60241,60242,60243,60244,60245,60246,60247,60248,60249,60250,60251,60252,60253,60254,60255,60256,60257,60258,60259,60260,60261,60262,60263,60264,60265,60266,60267,60268,60269,60270,60271,60272,60273,60274,60275,60276,60277,60278,60279,60280,60281,60282,60283,60284,60285,60286,60287,60288,60289,60290,60291,60292,60293,60294,60295,60296,60297,60298,60299,60300,60301,60302,60303,60304,60305,60306,60307,60308,60309,60310,60311,60312,60313,60314,60315,60316,60317,60318,60319,60320,60321,60322,60323,60324,60325,60326,60327,60328,60329,60330,60331,60332,60333,60334,60335,60336,60337,60338,60339,60340,60341,60342,60343,60344,60345,60346,60347,60348,60349,60350,60351,60352,60353,60354,60355,60356,60357,60358,60359,60360,60361,60362,60363,60364,60365,60366,60367,60368,60369,60370,60371,60372,60373,60374,60375,60376,60377,60378,60379,60380,60381,60382,60383,60384,60385,60386,60387,60388,60389,60390,60391,60392,60393,60394,60395,60396,60397,60398,60399,60400,60401,60402,60403,60404,60405,60406,60407,60408,60409,60410,60411,60412,60413,60414,60415,60416,60417,60418,60419,60420,60421,60422,60423,60424,60425,60426,60427,60428,60429,60430,60431,60432,60433,60434,60435,60436,60437,60438,60439,60440,60441,60442,60443,60444,60445,60446,60447,60448,60449,60450,60451,60452,60453,60454,60455,60456,60457,60458,60459,60460,60461,60462,60463,60464,60465,60466,60467,60468,60469,60470,60471,60472,60473,60474,60475,60476,60477,60478,60479,60480,60481,60482,60483,60484,60485,60486,60487,60488,60489,60490,60491,60492,60493,60494,60495,60496,60497,60498,60499,60500,60501,60502,60503,60504,60505,60506,60507,60508,60509,60510,60511,60512,60513,60514,60515,60516,60517,60518,60519,60520,60521,60522,60523,60524,60525,60526,60527,60528,60529,60530,60531,60532,60533,60534,60535,60536,60537,60538,60539,60540,60541,60542,60543,60544,60545,60546,60547,60548,60549,60550,60551,60552,60553,60554,60555,60556,60557,60558,60559,60560,60561,60562,60563,60564,60565,60566,60567,60568,60569,60570,60571,60572,60573,60574,60575,60576,60577,60578,60579,60580,60581,60582,60583,60584,60585,60586,60587,60588,60589,60590,60591,60592,60593,60594,60595,60596,60597,60598,60599,60600,60601,60602,60603,60604,60605,60606,60607,60608,60609,60610,60611,60612,60613,60614,60615,60616,60617,60618,60619,60620,60621,60622,60623,60624,60625,60626,60627,60628,60629,60630,60631,60632,60633,60634,60635,60636,60637,60638,60639,60640,60641,60642,60643,60644,60645,60646,60647,60648,60649,60650,60651,60652,60653,60654,60655,60656,60657,60658,60659,60660,60661,60662,60663,60664,60665,60666,60667,60668,60669,60670,60671,60672,60673,60674,60675,60676,60677,60678,60679,60680,60681,60682,60683,60684,60685,60686,60687,60688,60689,60690,60691,60692,60693,60694,60695,60696,60697,60698,60699,60700,60701,60702,60703,60704,60705,60706,60707,60708,60709,60710,60711,60712,60713,60714,60715,60716,60717,60718,60719,60720,60721,60722,60723,60724,60725,60726,60727,60728,60729,60730,60731,60732,60733,60734,60735,60736,60737,60738,60739,60740,60741,60742,60743,60744,60745,60746,60747,60748,60749,60750,60751,60752,60753,60754,60755,60756,60757,60758,60759,60760,60761,60762,60763,60764,60765,60766,60767,60768,60769,60770,60771,60772,60773,60774,60775,60776,60777,60778,60779,60780,60781,60782,60783,60784,60785,60786,60787,60788,60789,60790,60791,60792,60793,60794,60795,60796,60797,60798,60799,60800,60801,60802,60803,60804,60805,60806,60807,60808,60809,60810,60811,60812,60813,60814,60815,60816,60817,60818,60819,60820,60821,60822,60823,60824,60825,60826,60827,60828,60829,60830,60831,60832,60833,60834,60835,60836,60837,60838,60839,60840,60841,60842,60843,60844,60845,60846,60847,60848,60849,60850,60851,60852,60853,60854,60855,60856,60857,60858,60859,60860,60861,60862,60863,60864,60865,60866,60867,60868,60869,60870,60871,60872,60873,60874,60875,60876,60877,60878,60879,60880,60881,60882,60883,60884,60885,60886,60887,60888,60889,60890,60891,60892,60893,60894,60895,60896,60897,60898,60899,60900,60901,60902,60903,60904,60905,60906,60907,60908,60909,60910,60911,60912,60913,60914,60915,60916,60917,60918,60919,60920,60921,60922,60923,60924,60925,60926,60927,60928,60929,60930,60931,60932,60933,60934,60935,60936,60937,60938,60939,60940,60941,60942,60943,60944,60945,60946,60947,60948,60949,60950,60951,60952,60953,60954,60955,60956,60957,60958,60959,60960,60961,60962,60963,60964,60965,60966,60967,60968,60969,60970,60971,60972,60973,60974,60975,60976,60977,60978,60979,60980,60981,60982,60983,60984,60985,60986,60987,60988,60989,60990,60991,60992,60993,60994,60995,60996,60997,60998,60999,61000,61001,61002,61003,61004,61005,61006,61007,61008,61009,61010,61011,61012,61013,61014,61015,61016,61017,61018,61019,61020,61021,61022,61023,61024,61025,61026,61027,61028,61029,61030,61031,61032,61033,61034,61035,61036,61037,61038,61039,61040,61041,61042,61043,61044,61045,61046,61047,61048,61049,61050,61051,61052,61053,61054,61055,61056,61057,61058,61059,61060,61061,61062,61063,61064,61065,61066,61067,61068,61069,61070,61071,61072,61073,61074,61075,61076,61077,61078,61079,61080,61081,61082,61083,61084,61085,61086,61087,61088,61089,61090,61091,61092,61093,61094,61095,61096,61097,61098,61099,61100,61101,61102,61103,61104,61105,61106,61107,61108,61109,61110,61111,12288,65292,12289,12290,65294, 8231,65307,65306,65311,65281,65072, 8230, 8229,65104,65380,65106,  183,65108,65109,65110,65111,65372, 8211,65073, 8212,65075, 9588,65076,65103,65288,65289,65077,65078,65371,65373,65079,65080,65339,65341,65081,65082,12304,12305,65083,65084,12298,12299,65085,65086,12296,12297,65342,65088,12300,12301,65089,65090,12302,12303,65091,65092,65113,65114,65115,65116,65117,65118,65344,65287, 8220,65282,12317,12318, 8245, 8242,65283,65286,65290, 8251,  167,12291, 9675, 9679, 9651, 9650, 9678,11088, 9733, 9671, 9670, 9633, 9632, 9661, 9660,12963, 8453,  773,65507,65343,  717,65097,65098,65101,65102,65099,65100,65119,65120,65121,65291,65293,  215,  247,  177, 8730,65308,65310,65309, 8806, 8807, 8800, 8734, 8786, 8803,65122,65123,65124,65125,65126,65374, 8745, 8746, 8869, 8736, 8735, 8895,13266,13265, 8747, 8750, 8757, 8756, 9792, 9794, 9793, 9737, 8593, 8595, 8592, 8594, 8598, 8599, 8601, 8600, 8741, 8739,65295,65340, 8725,65128,65284,65509,12306,65504,65505,65285,65312, 8451, 8457,65129,65130,65131,13269,13212,13213,13214,13262,13217,13198,13199,13252, 8728,20825,20827,20830,20829,20833,20835,21991,29929,31950, 9601, 9602, 9603, 9604, 9605, 9606, 9607, 9608, 9615, 9614, 9613, 9612, 9611, 9610, 9609, 9532, 9524, 9516, 9508, 9500, 9620, 9472, 9474, 9621, 9484, 9488, 9492, 9496, 9581, 9582, 9584, 9583, 9698, 9699, 9701, 9700, 9585, 9586, 9587,65296,65297,65298,65299,65300,65301,65302,65303,65304,65305, 8544, 8545, 8546, 8547, 8548, 8549, 8550, 8551, 8552, 8553,12321,12322,12323,12324,12325,12326,12327,12328,12329,21316,65313,65314,65315,65316,65317,65318,65319,65320,65321,65322,65323,65324,65325,65326,65327,65328,65329,65330,65331,65332,65333,65334,65335,65336,65337,65338,65345,65346,65347,65348,65349,65350,65351,65352,65353,65354,65355,65356,65357,65358,65359,65360,65361,65362,65363,65364,65365,65366,65367,65368,65369,65370,  913,  914,  915,  916,  917,  918,  919,  920,  921,  922,  923,  924,  925,  926,  927,  928,  929,  931,  932,  933,  934,  935,  936,  937,  945,  946,  947,  948,  949,  950,  951,  952,  953,  954,  955,  956,  957,  958,  959,  960,  961,  963,  964,  965,  966,  967,  968,  969,12549,12550,12551,12552,12553,12554,12555,12556,12557,12558,12559,12560,12561,12562,12563,12564,12565,12566,12567,12568,12569,12570,12571,12572,12573,12574,12575,12576,12577,12578,12579,12580,12581,12582,12583,12584,12585, 8228,  713,  714,  711,  715, 8364,19968,20057,19969,19971,20035,20061,63930,20108,20154,20799,20837,20843,20960,20992,20993,63882,21269,21313,21340,21448,19977,19979,19976,19978,20011,20024,20961,20037,20040,20063,20062,20110,20129,20800,20995,21242,21315,21449,21475,22303,22763,22805,22823,63873,23376,23377,23379,23544,23567,23586,23608,23665,24029,24037,24049,24050,24051,24062,24178,24318,24331,24339,25165,19985,19984,63847,20013,20016,63838,20043,23609,20104,20113,20117,20114,20116,20130,20161,63997,20163,20166,20167,20173,20170,20171,20164,20803,20801,20839,63953,20846,20844,20887,20982,20998,64000,21000,21243,21246,21247,21270,21305,21320,21319,21317,21342,21380,21451,21450,21453,22764,22825,22827,22826,22829,23380,23569,23588,23610,23663,24052,24187,24319,24340,24341,24515,25096,25142,25163,25166,25903,25991,26007,26020,26041,26085,26352,26376,26408,27424,27490,27513,27595,27604,27611,27663,27700,28779,29226,29238,29243,29255,29273,29275,29356,29579,19993,19990,19989,19988,19992,20027,20045,20047,20046,20197,20184,20180,20181,20182,20183,20195,63912,20185,20190,20805,20804,20873,20874,20908,20985,20986,20984,21002,21152,21151,21253,21254,63843,21277,20191,21322,21321,21345,21344,21359,21358,21435,21487,21476,21491,21484,21486,21481,21480,21500,21496,21493,21483,21478,21482,21490,21489,21488,63750,21485,21499,22235,22234,22806,22830,22833,22900,22902,23381,23427,23612,24040,24039,24038,24066,24067,24179,24188,24321,24344,24343,24517,25098,25171,25172,25170,25169,26021,26086,26414,26412,26410,26411,26413,27491,27597,27665,27664,27704,27713,27712,27710,29359,29572,29577,29916,29926,29976,29983,29992,29993,30000,30001,30002,30003,30091,30333,30382,30399,30446,30683,30690,30707,31034,31166,31348,63991,19998,19999,20050,20051,20073,20121,20132,20134,20133,20223,20233,20249,20234,20245,20237,20240,20241,20239,20210,20214,20219,20208,20211,20221,20225,20235,20809,20807,20806,20808,20840,20849,20877,20912,63900,21009,21010,21006,21014,63901,21256,21281,21280,21360,21361,21513,63966,21516,21514,21520,21505,21515,21508,21521,21517,21512,21507,21518,21510,21522,22240,22238,22237,22323,22320,22312,22317,22316,22319,22313,22809,22810,22839,22840,22916,22904,22915,22909,22905,22914,22913,23383,23384,23431,23432,64004,23433,23546,23574,23673,24030,24070,24182,63886,24335,24347,24537,24534,25102,25100,25101,25104,25187,25179,25176,25910,26089,26088,26092,26093,26354,26355,26377,26429,26420,26417,26421,27425,27492,27515,27670,27741,27735,27737,27743,27744,27728,27733,27745,27739,27725,27726,28784,29279,29277,30334,31481,31859,31992,32566,32650,64030,63796,32771,32780,32786,32819,32895,32905,63827,32908,33251,33258,33267,33276,33292,33307,33311,33390,33394,33406,34411,34880,64008,34915,35199,38433,63749,20136,20301,20303,20295,20311,20318,20276,20315,20309,20272,20304,20305,20285,20282,20280,20291,20308,20284,20294,20323,20316,20320,20271,20302,20278,20313,20317,20296,20314,20812,20811,20813,20853,20918,63790,21029,21028,63965,21034,21032,21163,21161,21162,21164,21283,21363,63772,63981,21549,21534,21566,21542,21582,21543,21574,21571,21555,21576,63872,21531,21545,21578,21561,21563,21560,21550,21557,21558,21536,21564,21568,21553,21547,21535,21548,22250,22256,22244,22251,22346,22353,22336,22349,22343,22350,22334,22352,22351,22331,22767,22846,22941,22930,22952,22942,22947,22937,22934,22925,22948,22931,22922,22949,23389,23388,23386,23387,23436,23435,23439,23596,23616,23617,63933,23614,23696,23697,23700,23692,24043,24076,24207,24199,24202,24311,63811,24351,24420,24418,24439,24441,24536,24524,24535,24525,24561,24555,24568,24554,25106,25105,25220,25239,25238,25216,25206,25225,25197,25226,25212,25214,25209,25203,25234,25199,25240,25198,25237,25235,25233,25222,25913,25915,25912,26097,63745,26463,63969,26447,26448,26449,26460,26454,26462,26441,26438,26464,26451,26455,27493,27599,27714,27742,27801,27777,63858,27785,27781,27803,27754,27770,27792,27760,27788,27752,27798,27794,27773,27779,27762,27774,27764,27782,27766,27789,27796,27800,27778,28790,28796,28797,28792,63814,29281,29280,29380,29378,29590,29996,29995,30007,30008,30338,30447,30691,31169,31168,31167,31350,31995,32597,32918,32915,32925,32920,32923,32922,32946,63868,33426,33419,33421,64010,35282,35328,35895,35910,35925,35997,36196,36208,36275,36523,63746,36763,63857,36802,36806,36805,36804,24033,37009,37026,37034,37030,37027,37193,37318,63977,38450,63942,38449,38442,38444,20006,20054,20083,20107,20123,20126,20139,20140,20335,20381,20365,20339,20351,20332,20379,63925,63789,20355,20336,20341,20360,20329,20347,20374,20350,20367,20369,20346,20820,20818,20821,63864,20855,20854,20856,20925,20989,21051,21048,21047,63999,21040,21038,21046,21057,21182,21179,21330,21332,21331,21329,21350,21367,21368,21369,21462,21460,21463,21619,21621,21654,21624,21653,21632,21627,21623,21636,21650,21638,21628,21648,21617,21622,21644,21658,21602,21608,21643,21629,21646,22266,22403,22391,22378,22377,22369,22374,22372,22396,22812,22857,22855,63756,22852,22868,22974,22971,22996,22969,22958,22993,22982,22992,22989,22987,22995,22986,22959,22963,22994,22981,23391,23396,23395,23447,23450,23448,23452,23449,23451,23578,23624,23621,23622,23735,23713,23736,23721,23723,23729,23731,24088,24090,24086,24085,24091,24081,24184,24218,24215,24220,24213,24214,24310,24358,24359,24361,24448,24449,24447,24444,24541,24544,24573,63907,24575,24591,24596,24623,24629,24598,24618,24597,24609,24615,24617,24619,24603,25110,25109,25151,25150,25152,25215,63781,25292,25284,25279,25282,25273,25298,25307,25259,64002,25300,25291,25288,25256,25277,25276,25296,25305,25287,25293,25269,25306,25265,25304,25302,25303,25286,25260,25294,25918,26023,26044,26106,26132,63968,26124,26118,26114,26126,26112,26127,26133,26122,26119,26381,26379,26477,26507,26517,26481,26524,26483,26487,26503,26525,63988,26479,26480,26495,26505,26494,26512,26485,26522,26515,26492,26474,26482,27427,27494,27495,27519,27667,27675,27875,27880,27891,27825,63848,63971,27827,27837,27838,27836,27874,27819,27861,27859,27832,27844,27833,27841,27822,27863,27845,27889,27839,27835,27873,27867,27850,27820,27887,27868,27862,27872,28821,28814,28818,28810,63995,29228,29229,29240,29256,29287,29289,63994,29390,29401,29399,29392,29609,29608,29599,29611,29605,30013,30109,30105,30106,30340,30402,30450,30452,30693,30717,31038,31040,31041,31177,31176,31354,31353,31482,31998,32596,32652,32651,32773,32954,32933,32930,32945,32929,32939,32937,32948,32938,32943,33253,33278,33293,33459,33437,33433,33453,33469,33439,33465,33457,33452,33445,33455,33464,33443,33456,33470,33463,34382,34417,21021,34920,36555,36814,36820,36817,37045,37048,37041,37046,37319,63754,38263,38272,38428,38464,38463,38459,38468,38466,38585,38632,38738,38750,20127,20141,63863,20449,20405,20399,63845,20448,20433,20431,20445,20419,20406,20440,20447,20426,20439,20398,20432,20420,20418,20442,20430,20446,20407,20823,20882,20881,20896,21070,21059,21066,21069,21068,21067,21063,21191,21193,21187,21185,21261,21335,21371,21402,21467,21676,21696,21672,21710,21705,21688,21670,21683,21703,21698,63902,21674,21697,21700,21704,21679,21675,21681,21691,21673,21671,21695,22271,22402,22411,22432,22435,22434,22478,22446,22419,22869,63753,22863,22862,22864,23004,23000,23039,23011,23016,23043,23013,23018,23002,23014,23041,23035,23401,23459,23462,23460,23458,23461,23553,23630,23631,23629,23627,23769,23762,24055,24093,24101,24095,24189,24224,64001,24314,24328,24365,24421,24456,24453,24458,63960,24455,24460,24457,63840,24605,24608,24613,24590,24616,24653,24688,24680,24674,24646,24643,24684,24683,24682,24676,25153,25308,25366,25353,25340,25325,25345,25326,25341,25351,25329,25335,25327,25324,63859,25332,25361,25346,25919,25925,26027,26045,26082,26149,26157,26144,26151,26159,26143,26152,26161,26148,26359,26623,26579,26609,26580,26576,26604,26550,26543,26613,26601,26607,26564,26577,26548,26586,26597,26552,26575,26590,63945,26544,26585,26594,26589,26578,27498,27523,27526,27573,27602,27607,27679,27849,27915,27954,27946,63946,27941,27916,27953,64005,27927,27963,27965,27966,27958,63765,27893,27961,27943,27960,27945,27950,27957,27918,27947,28843,28858,28851,28844,28847,28845,28856,28846,28836,29232,29298,29295,29300,29417,29408,29409,29623,29642,29627,63917,29645,29632,29619,29978,29997,30031,30028,30030,30027,30123,30116,30117,30114,30115,30328,30342,30343,30344,30408,30406,30403,30405,63853,30457,30456,30473,30475,30462,30460,30471,30684,30722,30740,30732,30733,31046,31049,31048,31047,31161,31162,31185,31186,31179,31359,31361,31487,31485,31869,32002,32005,32000,32009,32007,32004,32006,32568,32654,32703,32772,32784,32781,32785,32822,32982,32997,32986,32963,32964,32972,32993,32987,32974,32990,32996,32989,33268,33314,33511,33539,33541,33507,33499,33510,33540,63860,33538,33545,33490,33495,33521,33537,33500,33492,33489,33502,33491,33503,33519,33542,34384,34425,34427,34426,34893,34923,35201,35284,35336,35330,35331,35998,36000,36212,36211,36276,36557,36556,36848,36838,36834,36842,36837,36845,36843,36836,36840,37066,37070,37057,37059,37195,37194,37325,38274,38480,63825,38476,64009,38754,38761,38859,38893,38899,38913,39080,39131,39135,39318,39321,20056,20147,20492,20493,20515,20463,20518,20517,20472,20521,20502,20486,20540,20511,20506,20498,20497,20474,20480,20500,20520,20465,20513,20491,20505,20504,20467,20462,20525,20522,20478,63956,20489,20860,20900,20901,20898,20941,63829,20934,20939,21078,21084,21076,21083,21085,21290,21375,21407,21405,21471,21736,21776,21761,21815,21756,21733,21746,21766,21754,21780,21737,21741,21729,21769,21742,21738,21734,21799,21767,21757,21775,22275,22276,22466,22484,22475,22467,22537,22799,22871,22872,22874,23057,23064,23068,23071,23067,23059,23020,23072,23075,23081,23077,23052,23049,23403,23640,23472,23475,23478,23476,23470,23477,23481,23480,23556,23633,23637,23632,23789,23805,23803,23786,23784,23792,23798,23809,23796,24046,24109,24107,24235,24237,24231,24369,24466,24465,24464,24665,24675,24677,24656,24661,24685,24681,24687,24708,24735,24730,24717,24724,24716,24709,24726,25159,25331,25352,25343,25422,25406,25391,25429,25410,25414,25423,25417,25402,25424,25405,25386,25387,25384,25421,25420,25928,25929,63934,26049,63875,26178,26185,26191,26179,26194,26188,26181,26177,26360,26388,26389,63785,26657,26680,26696,26694,26707,26681,26690,26708,26665,26803,63962,26700,26705,26685,26612,26704,26688,26684,26691,26666,26693,26643,26648,26689,27530,27529,27575,27683,27687,27688,27686,27684,27888,63786,28053,28040,28039,28006,28024,28023,27993,28051,28012,28041,28014,27994,28020,28009,28044,28042,28025,28037,28005,28052,28874,28888,28900,63766,63903,28879,29241,29305,63787,29433,29437,29432,29431,29574,29677,63948,29678,29664,29674,63767,30036,30045,30044,30042,63949,30142,30149,30151,30130,30131,30141,30140,30137,30146,30136,30347,30384,64023,30413,30414,30505,30495,30496,30504,30697,30768,30759,30776,30749,30772,30775,30757,30765,30752,30751,30770,31061,31056,31072,31071,31062,64025,31069,31063,31066,31204,31203,31207,31199,31206,31209,31192,31364,31368,31449,31494,31505,31881,32033,32023,32011,32010,32032,63850,32020,63951,32021,32026,32028,32013,32025,32027,32570,32607,32660,32709,32705,32774,32792,32789,32793,32791,32829,32831,33009,33026,33008,33029,33005,33012,33030,33016,33011,33032,33021,33034,33020,33007,33261,33260,33280,33296,33322,33323,33320,33324,33467,33579,33618,33620,33610,33592,33616,33609,33589,33588,33615,33586,33593,63998,33559,33600,33585,33576,33603,34388,34442,34474,34451,34468,34473,34444,34467,34460,34928,34935,34945,34946,34941,34937,35352,35344,35342,35340,35349,35338,35351,35347,35350,35343,35345,63744,35962,35961,36001,36002,36215,36524,36562,36564,36559,36785,36865,36870,36855,36864,36858,36852,36867,36861,36869,36856,37013,37089,37085,37090,37202,37197,37196,37336,37341,37335,37340,37337,38275,38498,38499,38497,38491,38493,38500,38488,38494,38587,39138,39340,39592,39640,39717,39730,39740,20094,20602,20605,20572,20551,20547,20556,20570,20553,20581,20598,20558,20565,20597,20596,20599,20559,20495,20591,20589,20828,20885,20976,21098,21103,63826,21209,21208,21205,21264,21263,21273,63979,21312,21310,63851,26364,21830,21866,21862,21828,21854,21857,21827,21834,21809,21846,21839,21845,21807,21860,21816,21806,21852,21804,21859,21811,21825,21847,22280,22283,22281,22495,22533,22538,22534,22496,22500,22522,22530,22581,22519,22521,22816,22882,23094,23105,23113,23142,23146,23104,23100,23138,23130,23110,23114,23408,23495,23493,23492,23490,23487,23494,23561,23560,23559,23648,23644,23645,23815,23814,23822,23835,23830,23842,23825,23849,23828,63957,23844,23847,23831,24034,24120,24118,24115,24119,24247,24248,24246,24245,24254,24373,24375,24407,24428,24425,24427,24471,24473,24478,24472,24481,24480,24476,24703,24739,24713,24736,24744,24779,24756,24806,24765,24773,24763,24757,24796,24764,24792,24789,24774,24799,24760,24794,24775,25114,25115,25160,63861,25511,25458,25494,25506,25509,25463,25447,25496,25514,25457,25513,25481,25475,25499,25451,25512,25476,25480,25497,25505,25516,25490,25487,25472,63908,25449,25448,25466,25949,25942,25937,25945,25943,21855,25935,25944,25941,25940,26012,26011,26028,26063,26059,26060,26062,26205,26202,26212,26216,26214,26206,26361,21207,26395,63866,26799,26786,26771,26805,26751,26742,26801,26791,26775,26800,26755,26820,26797,26758,26757,26772,26781,63970,26783,26785,26754,27442,63856,27627,27628,27691,28046,28092,28147,28121,28082,28129,28108,28132,28155,28154,28165,28103,63989,28079,28113,28078,28126,28153,28088,28151,28149,28101,28114,28186,28085,63821,28139,28120,63958,28145,28142,28136,28102,28100,28074,28140,28095,28134,28921,28937,28938,28925,28911,29245,29309,29313,29468,29467,29462,29459,29465,63963,29701,29706,29699,63972,29694,29709,29920,29942,29943,29980,29986,63862,30054,30050,63842,30095,30164,30165,30133,30154,30157,30350,30420,30418,30427,30519,30526,30524,30518,30520,30522,63950,30787,30798,64026,31080,31085,31227,31378,31381,63992,31528,31515,31532,31526,31513,31518,31534,63993,31895,31893,32070,32067,32113,32046,32057,32060,32064,32048,32051,32068,63823,32066,32050,32049,32573,32670,63919,32716,32718,32722,32796,32842,63920,33071,33046,33059,33067,33065,33072,33060,33282,33333,33335,33334,33337,33678,33694,33688,33656,33698,33686,33725,33707,33682,33674,33683,33673,33696,33655,33659,33660,33670,33703,34389,24426,34503,34496,34486,34500,34485,34502,34507,34481,34479,34505,34899,34974,34952,34987,34962,34966,34957,34955,35219,35215,35370,35357,35363,35365,35377,35373,35359,35355,35362,35913,35930,36009,36012,36011,36008,36010,36007,36199,36198,36286,36282,36571,36575,36889,36877,36890,36887,63898,36895,36893,36880,36885,36894,36896,36879,36898,36886,36891,36884,37096,37101,64038,37207,37326,37365,37350,37347,37351,37357,37353,38281,38506,63833,38515,63955,38512,38516,38518,38519,38508,38592,38634,38633,31456,31455,38914,38915,39770,40165,40565,63808,40613,40635,20642,20621,20613,20633,20625,20608,20630,20632,20634,26368,20977,21106,21108,21109,21097,63791,21213,21211,21338,21413,21883,21888,21927,21884,21898,21917,21912,21890,21916,21930,21908,63755,21899,21891,21939,21934,21919,21822,21938,21914,21947,21932,21937,21886,21897,21931,21913,22285,22575,22570,22580,22564,22576,22577,22561,22557,22560,22777,22778,22880,23159,23194,23167,23186,23195,23207,23411,23409,23506,23500,23507,23504,23562,23563,23601,23884,63777,23860,23879,24061,24133,24125,24128,24131,24190,63784,24257,24258,24260,24380,24429,63846,24490,24488,24785,63929,24754,24758,24800,24860,24867,24826,24853,24816,24827,24820,24936,24817,24846,24822,24841,24832,24850,25119,25161,25507,25484,25551,25536,25577,25545,25542,25549,25554,25571,25552,25569,25558,25581,25582,25462,25588,25578,25563,25682,25562,25593,25950,25958,25954,25955,26001,26000,26031,26222,26224,64018,26230,26223,26257,26234,26238,26231,26366,26367,26399,26397,26874,26837,26848,26840,26839,26885,26847,26869,26862,26855,26873,26834,26866,26851,26827,26829,26893,26898,26894,26825,26842,26990,26875,27454,27450,27453,27544,27542,27580,27631,27694,27695,27692,28207,28216,28244,28193,28210,28263,28234,28192,28197,28195,28187,28251,28248,28196,28246,28270,28205,28198,28271,28212,28237,28218,28204,28227,28189,28222,28363,28297,28185,28238,28259,28228,28274,28265,28255,28953,28954,28966,28976,28961,28982,29038,28956,29260,29316,29312,29494,29477,29492,29481,29754,29738,29747,29730,29733,29749,29750,29748,29743,29723,29734,29736,29989,29990,30059,30058,63973,30171,30179,30169,30168,30174,30176,30331,30332,30358,30355,30388,30428,30543,30701,30813,30828,30831,31245,31240,31243,31237,31232,31384,31383,31382,31461,31459,31561,31574,31558,31568,31570,31572,31565,31563,31567,31569,31903,31909,32094,32080,32104,32085,32043,32110,32114,32097,32102,32098,32112,32115,21892,32724,32725,32779,32850,32901,33109,33108,33099,33105,33102,33081,33094,33086,33100,33107,33140,33298,33308,33769,33795,33784,33805,33760,33733,33803,33729,33775,63832,33780,33879,33802,33776,33804,33740,33789,33778,33738,33848,33806,33796,33756,33799,33748,33759,34395,34527,34521,34541,34516,34523,34532,34512,34526,34903,35009,63904,34993,35203,35222,35387,35424,35413,35422,35388,35393,35412,35419,35408,35398,35380,35386,35382,35414,35937,35970,36015,36028,36019,36029,36033,36027,36032,36020,36023,36022,36031,36024,36234,36229,36225,36302,36317,36299,36314,36305,36300,36315,36294,36603,36600,36604,36764,36910,36917,36913,64037,36914,36918,37122,37109,37129,37118,37219,37221,63870,37396,37397,37411,37385,37406,37389,37392,37383,37393,38292,38287,38283,38289,38291,38290,38286,38538,38542,38539,38525,38533,63964,38541,38514,38532,38593,38597,38596,38598,38599,38639,38642,38860,38917,38918,38920,39143,39146,64042,39145,39154,39149,39342,39341,40643,40653,40657,63771,20653,20661,20658,20659,20677,20670,20652,20663,20667,20655,20679,21119,21111,21117,21215,21222,21220,21218,21219,21295,21983,21992,21971,21990,21966,21980,21959,21969,21987,21988,21999,21978,21985,21957,21958,21989,21961,22290,22291,63852,22609,22616,22615,64016,22612,22635,22604,22637,22602,22626,22610,22603,22887,23233,23241,23244,23230,23229,23228,23219,23234,23218,23913,23919,24140,24185,63906,24264,24338,24409,24492,24494,24858,24847,24904,24863,24819,24859,24825,24833,24840,24910,24908,63961,24909,24894,24884,24871,24845,24838,24887,25121,25122,25619,25662,25630,25642,25645,25661,25644,25615,25628,25620,25613,25654,25622,25623,25606,25964,26015,26032,26263,26249,26247,63941,26262,26244,26264,26253,26371,27028,26989,26970,26999,26976,26964,26997,26928,27010,26954,26984,26987,26974,26963,27001,27014,26973,26979,26971,27463,27506,27584,27583,27603,27645,28322,28335,28371,28342,28354,28304,28317,28359,28357,28325,28312,28348,63980,28331,63748,28310,63947,28356,28372,28330,28327,28340,29006,29017,29033,29028,63891,29031,29020,29036,29030,29004,29029,29022,28998,29032,29014,29242,29266,29495,29509,29503,29502,29807,29786,29781,29791,29790,29761,29759,29785,29787,29788,30070,30072,30208,30192,30209,30194,30193,30202,30207,30196,30195,30430,30431,30555,30571,30566,30558,30563,30585,30570,30572,30556,30565,30568,30562,30702,30862,30896,30871,30872,63803,30857,30844,30865,30867,30847,31098,63804,31105,33836,31165,63830,31258,31264,31252,31263,31262,31391,31392,31607,31680,31584,31598,31591,31921,31923,31925,32147,32121,32145,32129,32143,32091,32622,32617,32618,32626,32681,32680,32676,32854,32856,32902,32900,33137,33136,33144,33125,33134,33139,33131,33145,33146,33126,33285,33351,33922,33911,63768,33841,33909,33894,33899,63854,33900,33883,33852,33845,33889,33891,33897,33901,33862,34398,63798,34399,34553,34579,34568,34567,34560,34558,34555,34562,34563,34566,34570,34905,35039,35028,35033,35036,35032,35037,63976,35018,35029,35026,35228,35299,35435,35442,35443,35430,35433,35440,35463,35452,35427,35488,35441,35461,35437,35426,35438,35436,35449,35451,35390,35432,35938,35978,35977,36042,36039,63747,36036,36018,36035,63816,36037,36321,36319,36328,63799,36339,36346,36330,36324,36326,36530,36611,36617,36606,36618,36767,36786,36939,36938,36947,36930,36948,36924,36949,36944,36935,36943,36942,36941,36945,36926,36929,37138,37143,37228,63769,37225,37321,37431,37463,37432,37437,37440,37438,37467,37451,37476,37457,63921,37449,37453,37445,37433,37439,37466,38296,38552,38548,38549,38605,38603,38601,38602,63817,38651,38649,63922,64028,38772,38774,38928,38929,38931,38922,38930,38924,64043,39156,39165,39166,39347,39345,39348,39649,40169,40578,40718,40723,40736,20711,20718,20709,20694,20717,63931,20693,20687,20689,20721,20686,20713,20834,20979,21123,21122,21297,21421,22014,22016,22043,22039,22013,22036,22022,22025,22029,22030,22007,22038,22047,22024,22032,22006,22296,22294,22645,22654,22659,22675,22666,22649,22661,22653,22781,22821,22818,22820,22890,22889,23265,23270,23273,23255,23254,23256,23267,23413,23518,63914,23521,23525,23526,23528,23522,23524,23519,23565,63819,23940,23943,24155,24163,24149,24151,24148,64011,24278,24330,24390,24432,24505,24903,24895,24907,24951,24930,24931,24927,24922,24920,24949,25130,25735,25688,25684,25764,25720,25695,25722,25681,25703,25652,25709,25723,25970,26017,26071,26070,26274,26280,26269,27036,27048,27029,27073,27054,27091,27083,27035,27063,27067,27051,27060,27088,27085,27053,27084,27046,27075,27043,27465,27468,27699,28467,28436,28414,28435,28404,28457,28478,28448,28460,63822,28418,28450,28415,28399,28422,28465,28472,28466,63890,28437,28459,28463,28552,28458,28396,28417,28402,28364,28407,29076,29081,29053,29066,29060,29074,29246,29330,29334,29508,29520,29796,29795,29802,29808,29805,29956,30097,30247,30221,30219,30217,30227,30433,30435,30596,30589,30591,30561,30913,30879,30887,30899,30889,30883,31118,64027,31117,31278,31281,31402,31401,31469,31471,31649,31637,31627,31605,31639,31645,31636,31631,31672,31623,31620,31929,31933,64029,32187,32176,32156,32189,63831,63805,32202,32180,32178,32177,32186,32162,32191,32181,32184,32173,32210,32199,32172,32624,32736,32737,32735,32862,32858,32903,33104,33152,33167,33160,33162,33151,33154,33255,33274,33287,33300,33310,33355,33993,33983,33990,33988,33945,33950,33970,33948,33995,33976,33984,34003,33936,33980,34001,33994,34623,34588,34619,34594,34597,34612,34584,34645,34615,34601,35059,35074,35060,35065,63762,35069,35048,35098,35055,35494,35468,35486,35491,35469,35489,35475,35492,63905,35493,35496,35480,35473,35482,35495,35946,35981,35980,36051,36049,36050,36203,36249,36245,36348,36628,36626,36629,36627,36771,36960,36952,36956,36963,36953,36958,36962,36957,36955,37145,37144,37150,37237,37240,37239,37236,37496,37504,37509,37528,37526,37499,37523,37532,37544,37500,37521,38305,38312,38313,38307,38309,38308,38553,38556,38555,38604,38610,38656,38780,38789,38902,38935,63924,39087,39089,39171,39173,39180,39177,39361,39599,39600,39654,39745,39746,40180,40182,40179,40636,40763,40778,20740,20736,20731,20725,20729,20738,20744,20745,20741,63828,21127,21128,63943,21133,21130,21232,21426,22062,22075,22073,22066,22079,22068,22057,22099,22094,22103,22132,22070,22063,22064,22656,22687,22686,22707,22684,22702,22697,22694,22893,23305,23291,23307,23285,23308,23304,63932,23532,23529,23531,23652,63967,23965,23956,24162,24159,24161,24290,24282,24287,24285,24291,24288,24392,24433,24503,24501,24950,24935,24942,24925,24917,24962,24956,24944,24939,24958,24999,63887,25003,24974,25004,24986,24996,24980,25006,63954,25705,25711,25721,25758,25778,25736,25744,25776,25765,25747,25749,25769,25746,25774,25773,25771,63889,25772,25753,25762,25779,25973,25975,63849,26286,26283,64006,26289,27171,27167,27112,27137,27166,27161,27133,27169,63820,27146,27123,63935,27141,27117,27153,27472,27470,27556,27589,27590,28479,28540,28548,28497,28518,28500,28550,28525,28507,28536,28526,28558,28538,28528,28516,28567,28504,28373,28527,28512,28511,29087,29100,29105,29096,29270,29339,29518,29527,63918,29835,29827,29822,29824,30079,30240,30249,30239,30244,30246,30241,30242,30362,30394,30436,30606,30599,30604,30609,30603,30923,30917,30906,63815,30910,30933,30908,30928,31295,31292,31296,31293,31287,31291,31407,31406,31661,31665,31684,31668,31686,31687,31681,31648,31692,31946,32224,63894,32239,32251,32216,32236,32221,32232,32227,32218,32222,32233,32158,32217,32242,32249,32629,32631,32687,32745,32806,33179,33180,33181,33184,33178,33176,34071,34109,34074,63897,34092,34093,34067,34065,34083,34081,34068,34028,34085,34047,34054,34690,34676,34678,34656,34662,34680,34664,34649,34647,34636,34643,34907,34909,35088,35079,35090,35091,35093,35082,35516,63869,35527,35524,35477,35531,64034,35506,35529,35522,35519,35504,63809,35533,35510,35513,35547,35916,35918,64022,36064,36062,36070,36068,36076,36077,36066,36067,36060,36074,36065,36205,36255,36259,36395,36368,36381,36386,36367,36393,36383,36385,36382,36538,36637,36635,36639,36649,63896,63959,36636,36638,36645,36969,36974,36968,36973,36983,37168,37165,37159,37169,37255,37257,37259,37251,37573,37563,37559,37610,37548,37604,37569,37555,37564,37586,37575,37616,37554,63878,38321,38660,38662,38663,38665,38752,38797,38795,38799,38945,38955,38940,39091,39178,39187,39186,39192,39389,39376,39391,39387,39377,39381,39378,39385,39607,39662,39663,39719,39749,39748,39799,63801,40198,40201,40195,40617,40638,63881,22696,40786,20754,20760,20756,20752,20757,20864,20906,20957,21137,21139,21235,22105,22123,22137,22121,22116,22136,22122,22120,22117,22129,22127,22124,22114,22134,22721,22718,22727,22725,22894,23325,23348,23416,23536,23566,24394,25010,24977,25001,24970,25037,25014,25022,25034,25032,25136,25797,25793,25803,25787,25788,25818,63792,25799,25794,25805,25791,25810,25812,25790,25972,63883,26313,26297,26308,26311,26296,27197,27192,27194,27225,27243,27224,27193,27204,27234,27233,27211,27207,27189,27231,27208,27481,63884,27653,28610,28593,28577,28611,28580,28609,28583,28595,28608,28601,28598,28582,28576,28596,29118,29129,63982,29138,29128,29141,29113,63936,29145,29148,29123,29124,29544,29852,29859,63983,29855,29854,29922,29964,29965,30260,30264,30266,63795,30437,30624,30622,30623,30629,30952,30938,30956,30951,31142,31309,31310,31302,31308,31307,31418,31705,31761,31689,31716,31707,31713,31721,31718,31957,64003,32266,32273,32264,32283,32291,32286,32285,32265,32272,63974,32690,32752,32753,32750,32808,33203,33193,33192,33275,33288,33368,33369,34122,34137,34120,34152,34153,34115,34121,34157,34154,34142,34691,34719,34718,34722,34701,34913,35114,35122,35109,35115,35105,35242,35238,35558,35578,35563,35569,35584,35548,35559,35566,63837,35585,35586,35575,35565,35571,35574,35580,35947,35949,35987,36084,36420,36401,36404,36418,36409,36405,64007,36655,36664,36659,36776,36774,36981,36980,36984,36978,63939,36986,37172,37266,37664,37686,37624,37683,37679,37666,37628,37675,63807,37658,37648,37670,37665,37653,37678,37657,38331,38567,38568,38570,38613,38670,38673,38678,38669,38675,38671,38747,38748,38758,38808,38960,38968,38971,38967,38957,38969,38948,39184,64044,39198,39195,39201,39194,39405,39394,63770,39608,39612,39675,39661,39720,39825,40213,40227,40230,40232,40210,40219,40664,40660,63940,63752,20778,20767,20769,20786,63871,22158,22144,22160,22149,22151,22159,22741,22739,22737,22734,23344,23338,23332,23418,23607,23656,23996,63915,23997,23992,24171,24396,24509,25033,25026,25031,25062,25035,25138,25140,25806,25802,25816,25824,25840,25830,25836,25841,25826,25837,25986,25987,26329,26326,27264,27284,27268,27298,27292,27355,27299,27262,27287,27280,27296,27484,63909,27610,27656,28632,28657,28639,28640,28635,28644,63778,28655,28544,28652,28641,28649,28629,28654,28656,29159,29151,29166,29158,29157,29165,29164,29172,29152,29237,29254,29552,29554,29865,29872,29862,29864,30278,63937,30284,30442,30643,30634,30640,30636,30631,30637,30703,30967,30970,30964,30959,30977,31143,31146,31319,31423,31751,31757,31742,31735,31756,31712,31968,31964,31966,31970,31967,31961,31965,32302,32318,32326,63824,32306,32323,32299,32317,32305,32325,32321,32308,32313,32328,32309,32319,32303,32580,32755,32764,32881,32882,32880,63895,32883,33222,33219,33210,33218,33216,33215,33213,33225,33214,63990,33289,33393,34218,34180,34174,34204,34193,34196,34223,34203,34183,34216,34186,34407,34752,34769,34739,34770,34758,34731,34747,63761,34760,34763,35131,35126,35140,35128,35133,35244,35598,35607,35609,35611,35594,35616,35613,35588,35600,35905,35903,35955,36090,36093,36092,36088,36091,36264,36425,36427,36424,36426,36676,36670,36674,36677,36671,36991,36989,36996,36993,36994,36992,37177,37283,37278,37276,37709,37762,37672,37749,63899,37733,37707,37656,37758,37740,37723,37744,37722,37716,38346,38347,38348,38344,38342,38577,63928,38614,38684,38686,38816,38867,38982,39094,39221,39425,39423,39854,39851,39850,39853,40251,40255,40587,40655,40670,40668,40669,40667,40766,40779,21474,22165,22190,22745,63818,23352,24413,25059,25139,25844,25842,25854,25862,25850,25851,25847,26039,26332,26406,27315,27308,27331,27323,27320,27330,27310,27311,27487,27512,27567,28681,28683,63876,28678,28666,28689,28687,29179,29180,29182,29176,29559,63911,29863,29887,29973,30294,30296,30290,30653,30655,30651,30652,30990,63926,31329,31330,31328,31428,31429,31787,31783,31786,31774,31779,31777,63867,32340,32341,32350,32346,32353,32338,32345,32584,32761,32763,32887,32886,33229,33231,33290,34255,34217,63779,34256,34249,34224,34234,34233,34214,34799,34796,34802,34784,35206,35250,35316,35624,35641,35628,35627,35920,36101,36441,36451,36454,36452,36447,36437,36544,36681,36685,36999,36995,37000,37291,37292,37328,37780,37770,37782,37794,37811,37806,37804,37808,37784,37786,37783,38356,38358,38352,38357,63978,38620,38617,38619,38622,38692,38819,38822,38829,38905,38989,38991,38988,38990,38995,39098,39230,39231,39229,39214,39333,39438,39617,39683,39686,39759,39758,39757,39882,39881,39933,39880,39872,40273,40285,40288,40672,40725,40748,20787,22181,22750,63810,22754,23541,40848,63874,25074,25079,63757,25077,25856,25871,26336,26333,27365,27357,27354,63793,28699,28703,28712,28698,28701,28693,28696,29190,29197,29272,29346,29560,29562,29885,29898,29923,30087,30086,30303,30305,30663,31001,31153,31339,31337,63910,31807,31800,31805,31799,31808,32363,32365,32377,32361,32362,63759,32371,32694,32697,32696,63782,34281,34269,34282,34261,34276,34277,34295,34811,34821,34829,34809,34814,35168,35167,35158,35166,35649,35676,63996,35657,35674,35662,35663,35654,35673,36104,36106,36476,36466,36487,36470,36460,36474,36468,36692,36686,36781,37002,37003,37297,37294,37857,37841,37855,37827,37832,37852,37853,37846,37858,37837,37848,37860,37847,37864,38364,38580,38627,38698,38695,38753,38876,38907,63952,39000,39003,39100,39237,39241,39446,39449,39693,39912,39911,39894,39899,40329,40289,40306,40298,40300,40594,63880,40595,40628,21240,22184,22199,22198,22196,22204,22756,23360,23363,23421,23542,24009,25080,25082,25880,25876,25881,26342,26407,27372,28734,28720,28722,63794,29563,29903,30306,30309,31014,63877,31020,31019,31431,31478,31820,31811,31821,31983,31984,36782,32381,32380,32386,32588,32768,33242,33382,34299,34297,34321,63984,63797,34315,34311,34314,34836,34837,63780,35258,35320,35696,35692,35686,35695,35679,35691,36111,36109,36489,36481,36485,36482,63927,37323,37912,37891,37885,38369,38704,39108,39250,39249,39336,39467,39472,39479,39477,39955,39949,40569,40629,40680,40751,40799,40803,40801,20791,20792,22209,22208,22210,22804,23660,24013,25084,25086,25885,25884,26005,26345,27387,63773,27386,27570,28748,63774,29351,29910,29908,63758,30675,31824,32399,32396,32700,34327,63775,34330,34851,34850,34849,63783,35178,35180,35261,35700,35703,35709,36115,36490,36493,36491,36703,36783,37306,37934,37939,37941,37946,37944,37938,37931,38370,38712,38713,63800,38911,39015,39013,39255,39493,39491,39488,39486,39631,39764,39761,39981,39973,40367,64045,40386,40376,40605,40687,40729,40796,40806,40807,20796,20795,22216,22218,22217,23423,24020,24018,24398,25087,25892,27402,27489,28753,28760,29568,29924,30090,30318,30316,31155,63812,31839,63813,32893,33247,35186,35183,35324,63834,36118,36119,36497,36499,36705,37192,37956,37969,37970,38717,38718,38851,38849,39019,39253,39509,39501,39634,39706,40009,39985,39998,39995,40403,40407,40756,40812,40810,40852,22220,24022,63888,25891,25899,25898,26348,27408,29914,31434,31844,31843,31845,32403,32406,32404,33250,34360,63760,34865,35722,37008,63763,37987,37984,37988,38760,39023,39260,39514,39515,39511,39635,39636,39633,40020,63986,40022,40421,63987,40692,22225,22761,25900,28766,30321,30322,30679,32592,32648,34870,34873,34914,35731,35730,35734,33399,36123,37312,37994,38722,63923,38724,38854,39024,39519,39714,39768,40031,40441,63802,40572,40573,40711,40823,40818,24307,27414,28771,31852,31854,34875,35264,36513,37313,38002,38000,39025,39262,39638,39715,40652,28772,30682,35738,38007,38857,39522,39525,32412,35740,36522,37317,38013,38014,38012,40055,40056,40695,35924,38015,40474,29224,63879,39729,40475,63776,31858,63153,63154,63155,63156,63157,63158,63159,63160,63161,63162,63163,63164,63165,63166,63167,63168,63169,63170,63171,63172,63173,63174,63175,63176,63177,63178,63179,63180,63181,63182,63183,63184,63185,63186,63187,63188,63189,63190,63191,63192,63193,63194,63195,63196,63197,63198,63199,63200,63201,63202,63203,63204,63205,63206,63207,63208,63209,63210,63211,63212,63213,63214,63215,63216,63217,63218,63219,63220,63221,63222,63223,63224,63225,63226,63227,63228,63229,63230,63231,63232,63233,63234,63235,63236,63237,63238,63239,63240,63241,63242,63243,63244,63245,63246,63247,63248,63249,63250,63251,63252,63253,63254,63255,63256,63257,63258,63259,63260,63261,63262,63263,63264,63265,63266,63267,63268,63269,63270,63271,63272,63273,63274,63275,63276,63277,63278,63279,63280,63281,63282,63283,63284,63285,63286,63287,63288,63289,63290,63291,63292,63293,63294,63295,63296,63297,63298,63299,63300,63301,63302,63303,63304,63305,63306,63307,63308,63309,63310,63311,63312,63313,63314,63315,63316,63317,63318,63319,63320,63321,63322,63323,63324,63325,63326,63327,63328,63329,63330,63331,63332,63333,63334,63335,63336,63337,63338,63339,63340,63341,63342,63343,63344,63345,63346,63347,63348,63349,63350,63351,63352,63353,63354,63355,63356,63357,63358,63359,63360,63361,63362,63363,63364,63365,63366,63367,63368,63369,63370,63371,63372,63373,63374,63375,63376,63377,63378,63379,63380,63381,63382,63383,63384,63385,63386,63387,63388,63389,63390,63391,63392,63393,63394,63395,63396,63397,63398,63399,63400,63401,63402,63403,63404,63405,63406,63407,63408,63409,63410,63411,63412,63413,63414,63415,63416,63417,63418,63419,63420,63421,63422,63423,63424,63425,63426,63427,63428,63429,63430,63431,63432,63433,63434,63435,63436,63437,63438,63439,63440,63441,63442,63443,63444,63445,63446,63447,63448,63449,63450,63451,63452,63453,63454,63455,63456,63457,63458,63459,63460,63461,63462,63463,63464,63465,63466,63467,63468,63469,63470,63471,63472,63473,63474,63475,63476,63477,63478,63479,63480,63481,63482,63483,63484,63485,63486,63487,63488,63489,63490,63491,63492,63493,63494,63495,63496,63497,63498,63499,63500,63501,63502,63503,63504,63505,63506,63507,63508,63509,63510,63511,63512,63513,63514,63515,63516,63517,63518,63519,63520,63521,63522,63523,63524,63525,63526,63527,63528,63529,63530,63531,63532,63533,63534,63535,63536,63537,63538,63539,63540,63541,63542,63543,63544,63545,63546,63547,63548,63549,63550,63551,63552,63553,63554,63555,63556,63557,63558,63559,63560,20034,20060,20981,21274,21378,19975,19980,20039,20109,22231,64012,23662,24435,19983,20871,19982,20014,20115,20162,20169,20168,20888,21244,21356,21433,22304,22787,22828,23568,24063,26081,27571,27596,27668,29247,20017,20028,20200,20188,20201,20193,20189,20186,21004,21276,21324,22306,22307,22807,22831,23425,23428,23570,23611,23668,23667,24068,24192,24194,24521,25097,25168,27669,27702,27715,27711,27707,29358,29360,29578,31160,32906,38430,20238,20248,20268,20213,20244,20209,20224,20215,20232,20253,20226,20229,20258,20243,20228,20212,20242,20913,21011,21001,21008,21158,21282,21279,21325,21386,21511,22241,22239,22318,22314,22324,22844,22912,22908,22917,22907,22910,22903,22911,23382,23573,23589,23676,23674,23675,23678,24031,24181,24196,24322,24346,24436,24533,24532,24527,25180,25182,25188,25185,25190,25186,25177,25184,25178,25189,26095,26094,26430,26425,26424,26427,26426,26431,26428,26419,27672,27718,27730,27740,27727,27722,27732,27723,27724,28785,29278,29364,29365,29582,29994,30335,31349,32593,33400,33404,33408,33405,33407,34381,35198,37017,37015,37016,37019,37012,38434,38436,38432,38435,20310,20283,20322,20297,20307,20324,20286,20327,20306,20319,20289,20312,20269,20275,20287,20321,20879,20921,21020,21022,21025,21165,21166,21257,21347,21362,21390,21391,21552,21559,21546,21588,21573,21529,21532,21541,21528,21565,21583,21569,21544,21540,21575,22254,22247,22245,22337,22341,22348,22345,22347,22354,22790,22848,22950,22936,22944,22935,22926,22946,22928,22927,22951,22945,23438,23442,23592,23594,23693,23695,23688,23691,23689,23698,23690,23686,23699,23701,24032,24074,24078,24203,24201,24204,24200,24205,24325,24349,24440,24438,24530,24529,24528,24557,24552,24558,24563,24545,24548,24547,24570,24559,24567,24571,24576,24564,25146,25219,25228,25230,25231,25236,25223,25201,25211,25210,25200,25217,25224,25207,25213,25202,25204,25911,26096,26100,26099,26098,26101,26437,26439,26457,26453,26444,26440,26461,26445,26458,26443,27600,27673,27674,27768,27751,27755,27780,27787,27791,27761,27759,27753,27802,27757,27783,27797,27804,27750,27763,27749,27771,27790,28788,28794,29283,29375,29373,29379,29382,29377,29370,29381,29589,29591,29587,29588,29586,30010,30009,30100,30101,30337,31037,32820,32917,32921,32912,32914,32924,33424,33423,33413,33422,33425,33427,33418,33411,33412,35960,36809,36799,37023,37025,37029,37022,37031,37024,38448,38440,38447,38445,20019,20376,20348,20357,20349,20352,20359,20342,20340,20361,20356,20343,20300,20375,20330,20378,20345,20353,20344,20368,20380,20372,20382,20370,20354,20373,20331,20334,20894,20924,20926,21045,21042,21043,21062,21041,21180,21258,21259,21308,21394,21396,21639,21631,21633,21649,21634,21640,21611,21626,21630,21605,21612,21620,21606,21645,21615,21601,21600,21656,21603,21607,21604,22263,63913,22383,22386,22381,22379,22385,22384,22390,22400,22389,22395,22387,22388,22370,22376,22397,22796,22853,22965,22970,22991,22990,22962,22988,22977,22966,22972,22979,22998,22961,22973,22976,22984,22964,22983,23394,23397,23443,23445,23620,23623,23726,23716,23712,23733,23727,23720,23724,23711,23715,23725,23714,23722,23719,23709,23717,23734,23728,23718,24087,24084,24089,24360,24354,24355,24356,24404,24450,24446,24445,24542,24549,24621,24614,24601,24626,24587,24628,24586,24599,24627,24602,24606,24620,24610,24589,24592,24622,24595,24593,24588,24585,63916,25108,25149,25261,25268,25297,25278,25258,25270,25290,25262,25267,25263,25275,25257,25264,25272,25917,26024,26043,26121,26108,26116,26130,26120,26107,26115,26123,26125,26117,26109,26129,26128,26358,26378,26501,26476,26510,26514,26486,63944,26520,26502,26500,26484,26509,26508,26490,26527,26513,26521,26499,26493,26497,26488,26489,26516,27429,27520,27518,27614,27677,27795,27884,27883,27886,27865,27830,27860,27821,27879,27831,27856,27842,27834,27843,27846,27885,27890,27858,27869,27828,27786,27805,27776,27870,27840,27952,27853,27847,27824,27897,27855,27881,27857,28820,28824,28805,28819,28806,28804,28817,28822,28802,28826,28803,29290,29398,29387,29400,29385,29404,29394,29396,29402,29388,29393,29604,29601,29613,29606,29602,29600,29612,29597,29917,29928,30015,30016,30014,30092,30104,30383,30451,30449,30448,30453,30712,30716,30713,30715,30714,30711,31042,31039,31173,31352,31355,31483,31861,31997,32821,32911,32942,32931,32952,32949,32941,33312,33440,33472,33451,33434,33432,33435,33461,33447,33454,33468,33438,33466,33460,33448,33441,33449,33474,33444,33475,33462,33442,34416,34415,34413,34414,35926,36818,36811,36819,36813,36822,36821,36823,37042,37044,37039,37043,37040,38457,38461,38460,38458,38467,20429,20421,20435,20402,20425,20427,20417,20436,20444,20441,20411,20403,20443,20423,20438,20410,20416,20409,20460,21060,21065,21184,21186,21309,21372,21399,21398,21401,21400,21690,21665,21677,21669,21711,21699,33549,21687,21678,21718,21686,21701,21702,21664,21616,21692,21666,21694,21618,21726,21680,22453,22430,22431,22436,22412,22423,22429,22427,22420,22424,22415,22425,22437,22426,22421,22772,22797,22867,23009,23006,23022,23040,23025,23005,23034,23037,23036,23030,23012,23026,23031,23003,23017,23027,23029,23008,23038,23028,23021,23464,23628,23760,23768,23756,23767,23755,23771,23774,23770,23753,23751,23754,23766,23763,23764,23759,23752,23750,23758,23775,23800,24057,24097,24098,24099,24096,24100,24240,24228,24226,24219,24227,24229,24327,24366,24406,24454,24631,24633,24660,24690,24670,24645,24659,24647,24649,24667,24652,24640,24642,24671,24612,24644,24664,24678,24686,25154,25155,63835,25357,25355,25333,25358,25347,25323,25337,25359,25356,25336,25334,25344,25363,25364,25338,25365,25339,25328,25921,25923,26026,26047,26166,26145,26162,26165,26140,26150,26146,26163,26155,26170,26141,26164,26169,26158,26383,26384,26561,26610,26568,26554,26588,26555,26616,26584,26560,26551,26565,26603,26596,26591,26549,26573,26547,26615,26614,26606,26595,26562,26553,26574,26599,26608,26546,26620,26566,26605,26572,26542,26598,26587,26618,26569,26570,26563,26602,26571,27432,27522,27524,27574,27606,27608,27616,27680,27681,27944,27956,27949,27935,27964,27967,27922,27914,27866,27955,27908,27929,27962,27930,27921,27904,27933,27970,27905,27928,27959,27907,27919,27968,27911,27936,27948,27912,27938,27913,27920,28855,28831,28862,28849,28848,28833,28852,28853,28841,29249,29257,29258,29292,29296,29299,29294,29386,29412,29416,29419,29407,29418,29414,29411,29573,29644,29634,29640,29637,29625,29622,29621,29620,29675,29631,29639,29630,29635,29638,29624,29643,29932,29934,29998,30023,30024,30119,30122,30329,30404,30472,30467,30468,30469,30474,30455,30459,30458,30695,30696,30726,30737,30738,30725,30736,30735,30734,30729,30723,30739,31050,31052,31051,31045,31044,31189,31181,31183,31190,31182,31360,31358,31441,31488,31489,31866,31864,31865,31871,31872,31873,32003,32008,32001,32600,32657,32653,32702,32775,32782,32783,32788,32823,32984,32967,32992,32977,32968,32962,32976,32965,32995,32985,32988,32970,32981,32969,32975,32983,32998,32973,33279,33313,33428,33497,33534,33529,33543,33512,33536,33493,33594,33515,33494,33524,33516,33505,33522,33525,33548,33531,33526,33520,33514,33508,33504,33530,33523,33517,34423,34420,34428,34419,34881,34894,34919,34922,34921,35283,35332,35335,36210,36835,36833,36846,36832,37105,37053,37055,37077,37061,37054,37063,37067,37064,37332,37331,38484,38479,38481,38483,38474,38478,20510,20485,20487,20499,20514,20528,20507,20469,20468,20531,20535,20524,20470,20471,20503,20508,20512,20519,20533,20527,20529,20494,20826,20884,20883,20938,20932,20933,20936,20942,21089,21082,21074,21086,21087,21077,21090,21197,21262,21406,21798,21730,21783,21778,21735,21747,21732,21786,21759,21764,21768,21739,21777,21765,21745,21770,21755,21751,21752,21728,21774,21763,21771,22273,22274,22476,22578,22485,22482,22458,22470,22461,22460,22456,22454,22463,22471,22480,22457,22465,22798,22858,23065,23062,23085,23086,23061,23055,23063,23050,23070,23091,23404,23463,23469,23468,23555,23638,23636,23788,23807,23790,23793,23799,23808,23801,24105,24104,24232,24238,24234,24236,24371,24368,24423,24669,24666,24679,24641,24738,24712,24704,24722,24705,24733,24707,24725,24731,24727,24711,24732,24718,25113,25158,25330,25360,25430,25388,25412,25413,25398,25411,25572,25401,25419,25418,25404,25385,25409,25396,25432,25428,25433,25389,25415,25395,25434,25425,25400,25431,25408,25416,25930,25926,26054,26051,26052,26050,26186,26207,26183,26193,26386,26387,26655,26650,26697,26674,26675,26683,26699,26703,26646,26673,26652,26677,26667,26669,26671,26702,26692,26676,26653,26642,26644,26662,26664,26670,26701,26682,26661,26656,27436,27439,27437,27441,27444,27501,32898,27528,27622,27620,27624,27619,27618,27623,27685,28026,28003,28004,28022,27917,28001,28050,27992,28002,28013,28015,28049,28045,28143,28031,28038,27998,28007,28000,28055,28016,28028,27999,28034,28056,27951,28008,28043,28030,28032,28036,27926,28035,28027,28029,28021,28048,28892,28883,28881,28893,28875,32569,28898,28887,28882,28894,28896,28884,28877,28869,28870,28871,28890,28878,28897,29250,29304,29303,29302,29440,29434,29428,29438,29430,29427,29435,29441,29651,29657,29669,29654,29628,29671,29667,29673,29660,29650,29659,29652,29661,29658,29655,29656,29672,29918,29919,29940,29941,29985,30043,30047,30128,30145,30139,30148,30144,30143,30134,30138,30346,30409,30493,30491,30480,30483,30482,30499,30481,30485,30489,30490,30498,30503,30755,30764,30754,30773,30767,30760,30766,30763,30753,30761,30771,30762,30769,31060,31067,31055,31068,31059,31058,31057,31211,31212,31200,31214,31213,31210,31196,31198,31197,31366,31369,31365,31371,31372,31370,31367,31448,31504,31492,31507,31493,31503,31496,31498,31502,31497,31506,31876,31889,31882,31884,31880,31885,31877,32030,32029,32017,32014,32024,32022,32019,32031,32018,32015,32012,32604,32609,32606,32608,32605,32603,32662,32658,32707,32706,32704,32790,32830,32825,33018,33010,33017,33013,33025,33019,33024,33281,33327,33317,33587,33581,33604,33561,33617,33573,33622,33599,33601,33574,33564,33570,33602,33614,33563,33578,33544,33596,33613,33558,33572,33568,33591,33583,33577,33607,33605,33612,33619,33566,33580,33611,33575,33608,34387,34386,34466,34472,34454,34445,34449,34462,34439,34455,34438,34443,34458,34437,34469,34457,34465,34471,34453,34456,34446,34461,34448,34452,34883,34884,34925,34933,34934,34930,34944,34929,34943,34927,34947,34942,34932,34940,35346,35911,35927,35963,36004,36003,36214,36216,36277,36279,36278,36561,36563,36862,36853,36866,36863,36859,36868,36860,36854,37078,37088,37081,37082,37091,37087,37093,37080,37083,37079,37084,37092,37200,37198,37199,37333,37346,37338,38492,38495,38588,39139,39647,39727,20095,20592,20586,20577,20574,20576,20563,20555,20573,20594,20552,20557,20545,20571,20554,20578,20501,20549,20575,20585,20587,20579,20580,20550,20544,20590,20595,20567,20561,20944,21099,21101,21100,21102,21206,21203,21293,21404,21877,21878,21820,21837,21840,21812,21802,21841,21858,21814,21813,21808,21842,21829,21772,21810,21861,21838,21817,21832,21805,21819,21824,21835,22282,22279,22523,22548,22498,22518,22492,22516,22528,22509,22525,22536,22520,22539,22515,22479,22535,22510,22499,22514,22501,22508,22497,22542,22524,22544,22503,22529,22540,22513,22505,22512,22541,22532,22876,23136,23128,23125,23143,23134,23096,23093,23149,23120,23135,23141,23148,23123,23140,23127,23107,23133,23122,23108,23131,23112,23182,23102,23117,23097,23116,23152,23145,23111,23121,23126,23106,23132,23410,23406,23489,23488,23641,23838,23819,23837,23834,23840,23820,23848,23821,23846,23845,23823,23856,23826,23843,23839,23854,24126,24116,24241,24244,24249,24242,24243,24374,24376,24475,24470,24479,24714,24720,24710,24766,24752,24762,24787,24788,24783,24804,24793,24797,24776,24753,24795,24759,24778,24767,24771,24781,24768,25394,25445,25482,25474,25469,25533,25502,25517,25501,25495,25515,25486,25455,25479,25488,25454,25519,25461,25500,25453,25518,25468,25508,25403,25503,25464,25477,25473,25489,25485,25456,25939,26061,26213,26209,26203,26201,26204,26210,26392,26745,26759,26768,26780,26733,26734,26798,26795,26966,26735,26787,26796,26793,26741,26740,26802,26767,26743,26770,26748,26731,26738,26794,26752,26737,26750,26779,26774,26763,26784,26761,26788,26744,26747,26769,26764,26762,26749,27446,27443,27447,27448,27537,27535,27533,27534,27532,27690,28096,28075,28084,28083,28276,28076,28137,28130,28087,28150,28116,28160,28104,28128,28127,28118,28094,28133,28124,28125,28123,28148,28106,28093,28141,28144,28090,28117,28098,28111,28105,28112,28146,28115,28157,28119,28109,28131,28091,28922,28941,28919,28951,28916,28940,28912,28932,28915,28944,28924,28927,28934,28947,28928,28920,28918,28939,28930,28942,29310,29307,29308,29311,29469,29463,29447,29457,29464,29450,29448,29439,29455,29470,29576,29686,29688,29685,29700,29697,29693,29703,29696,29690,29692,29695,29708,29707,29684,29704,30052,30051,30158,30162,30159,30155,30156,30161,30160,30351,30345,30419,30521,30511,30509,30513,30514,30516,30515,30525,30501,30523,30517,30792,30802,30793,30797,30794,30796,30758,30789,30800,31076,31079,31081,31082,31075,31083,31073,31163,31226,31224,31222,31223,31375,31380,31376,31541,31559,31540,31525,31536,31522,31524,31539,31512,31530,31517,31537,31531,31533,31535,31538,31544,31514,31523,31892,31896,31894,31907,32053,32061,32056,32054,32058,32069,32044,32041,32065,32071,32062,32063,32074,32059,32040,32611,32661,32668,32669,32667,32714,32715,32717,32720,32721,32711,32719,32713,32799,32798,32795,32839,32835,32840,33048,33061,33049,33051,33069,33055,33068,33054,33057,33045,33063,33053,33058,33297,33336,33331,33338,33332,33330,33396,33680,33699,33704,33677,33658,33651,33700,33652,33679,33665,33685,33689,33653,33684,33705,33661,33667,33676,33693,33691,33706,33675,33662,33701,33711,33672,33687,33712,33663,33702,33671,33710,33654,33690,34393,34390,34495,34487,34498,34497,34501,34490,34480,34504,34489,34483,34488,34508,34484,34491,34492,34499,34493,34494,34898,34953,34965,34984,34978,34986,34970,34961,34977,34975,34968,34983,34969,34971,34967,34980,34988,34956,34963,34958,35202,35286,35289,35285,35376,35367,35372,35358,35897,35899,35932,35933,35965,36005,36221,36219,36217,36284,36290,36281,36287,36289,36568,36574,36573,36572,36567,36576,36577,36900,36875,36881,36892,36876,36897,37103,37098,37104,37108,37106,37107,37076,37099,37100,37097,37206,37208,37210,37203,37205,37356,37364,37361,37363,37368,37348,37369,37354,37355,37367,37352,37358,38266,38278,38280,38524,38509,38507,38513,38511,38591,38762,38916,39141,39319,20635,20629,20628,20638,20619,20643,20611,20620,20622,20637,20584,20636,20626,20610,20615,20831,20948,21266,21265,21412,21415,21905,21928,21925,21933,21879,22085,21922,21907,21896,21903,21941,21889,21923,21906,21924,21885,21900,21926,21887,21909,21921,21902,22284,22569,22583,22553,22558,22567,22563,22568,22517,22600,22565,22556,22555,22579,22591,22582,22574,22585,22584,22573,22572,22587,22881,23215,23188,23199,23162,23202,23198,23160,23206,23164,23205,23212,23189,23214,23095,23172,23178,23191,23171,23179,23209,23163,23165,23180,23196,23183,23187,23197,23530,23501,23499,23508,23505,23498,23502,23564,23600,23863,23875,23915,23873,23883,23871,23861,23889,23886,23893,23859,23866,23890,23869,23857,23897,23874,23865,23881,23864,23868,23858,23862,23872,23877,24132,24129,24408,24486,24485,24491,24777,24761,24780,24802,24782,24772,24852,24818,24842,24854,24837,24821,24851,24824,24828,24830,24769,24835,24856,24861,24848,24831,24836,24843,25162,25492,25521,25520,25550,25573,25576,25583,25539,25757,25587,25546,25568,25590,25557,25586,25589,25697,25567,25534,25565,25564,25540,25560,25555,25538,25543,25548,25547,25544,25584,25559,25561,25906,25959,25962,25956,25948,25960,25957,25996,26013,26014,26030,26064,26066,26236,26220,26235,26240,26225,26233,26218,26226,26369,26892,26835,26884,26844,26922,26860,26858,26865,26895,26838,26871,26859,26852,26870,26899,26896,26867,26849,26887,26828,26888,26992,26804,26897,26863,26822,26900,26872,26832,26877,26876,26856,26891,26890,26903,26830,26824,26845,26846,26854,26868,26833,26886,26836,26857,26901,26917,26823,27449,27451,27455,27452,27540,27543,27545,27541,27581,27632,27634,27635,27696,28156,28230,28231,28191,28233,28296,28220,28221,28229,28258,28203,28223,28225,28253,28275,28188,28211,28235,28224,28241,28219,28163,28206,28254,28264,28252,28257,28209,28200,28256,28273,28267,28217,28194,28208,28243,28261,28199,28280,28260,28279,28245,28281,28242,28262,28213,28214,28250,28960,28958,28975,28923,28974,28977,28963,28965,28962,28978,28959,28968,28986,28955,29259,29274,29320,29321,29318,29317,29323,29458,29451,29488,29474,29489,29491,29479,29490,29485,29478,29475,29493,29452,29742,29740,29744,29739,29718,29722,29729,29741,29745,29732,29731,29725,29737,29728,29746,29947,29999,30063,30060,30183,30170,30177,30182,30173,30175,30180,30167,30357,30354,30426,30534,30535,30532,30541,30533,30538,30542,30539,30540,30686,30700,30816,30820,30821,30812,30829,30833,30826,30830,30832,30825,30824,30814,30818,31092,31091,31090,31088,31234,31242,31235,31244,31236,31385,31462,31460,31562,31547,31556,31560,31564,31566,31552,31576,31557,31906,31902,31912,31905,32088,32111,32099,32083,32086,32103,32106,32079,32109,32092,32107,32082,32084,32105,32081,32095,32078,32574,32575,32613,32614,32674,32672,32673,32727,32849,32847,32848,33022,32980,33091,33098,33106,33103,33095,33085,33101,33082,33254,33262,33271,33272,33273,33284,33340,33341,33343,33397,33595,33743,33785,33827,33728,33768,33810,33767,33764,33788,33782,33808,33734,33736,33771,33763,33727,33793,33757,33765,33752,33791,33761,33739,33742,33750,33781,63806,33801,33807,33758,33809,33798,33730,33779,33749,33786,33735,33745,33770,33811,33731,33772,33774,33732,33787,33751,33762,33819,33755,33790,34520,34530,34534,34515,34531,34522,34538,34525,34539,34524,34540,34537,34519,34536,34513,34888,34902,34901,35002,35031,35001,35000,35008,35006,34998,35004,34999,35005,34994,35073,35017,35221,35224,35223,35293,35290,35291,35406,35405,35385,35417,35392,35415,35416,35396,35397,35410,35400,35409,35402,35404,35407,35935,35969,35968,36026,36030,36016,36025,36021,36228,36224,36233,36312,36307,36301,36295,36310,36316,36303,36309,36313,36296,36311,36293,36591,36599,36602,36601,36582,36590,36581,36597,36583,36584,36598,36587,36593,36588,36596,36585,36909,36916,36911,37126,37164,37124,37119,37116,37128,37113,37115,37121,37120,37127,37125,37123,37217,37220,37215,37218,37216,37377,37386,37413,37379,37402,37414,37391,37388,37376,37394,37375,37373,37382,37380,37415,37378,37404,37412,37401,37399,37381,37398,38267,38285,38284,38288,38535,38526,38536,38537,38531,38528,38594,38600,38595,38641,38640,38764,38768,38766,38919,39081,39147,40166,40697,20099,20100,20150,20669,20671,20678,20654,20676,20682,20660,20680,20674,20656,20673,20666,20657,20683,20681,20662,20664,20951,21114,21112,21115,21116,21955,21979,21964,21968,21963,21962,21981,21952,21972,21956,21993,21951,21970,21901,21967,21973,21986,21974,21960,22002,21965,21977,21954,22292,22611,22632,22628,22607,22605,22601,22639,22613,22606,22621,22617,22629,22619,22589,22627,22641,22780,23239,23236,23243,23226,23224,23217,23221,23216,23231,23240,23227,23238,23223,23232,23242,23220,23222,23245,23225,23184,23510,23512,23513,23583,23603,23921,23907,23882,23909,23922,23916,23902,23912,23911,23906,24048,24143,24142,24138,24141,24139,24261,24268,24262,24267,24263,24384,24495,24493,24823,24905,24906,24875,24901,24886,24882,24878,24902,24879,24911,24873,24896,25120,37224,25123,25125,25124,25541,25585,25579,25616,25618,25609,25632,25636,25651,25667,25631,25621,25624,25657,25655,25634,25635,25612,25638,25648,25640,25665,25653,25647,25610,25626,25664,25637,25639,25611,25575,25627,25646,25633,25614,25967,26002,26067,26246,26252,26261,26256,26251,26250,26265,26260,26232,26400,26982,26975,26936,26958,26978,26993,26943,26949,26986,26937,26946,26967,26969,27002,26952,26953,26933,26988,26931,26941,26981,26864,27000,26932,26985,26944,26991,26948,26998,26968,26945,26996,26956,26939,26955,26935,26972,26959,26961,26930,26962,26927,27003,26940,27462,27461,27459,27458,27464,27457,27547,64013,27643,27644,27641,27639,27640,28315,28374,28360,28303,28352,28319,28307,28308,28320,28337,28345,28358,28370,28349,28353,28318,28361,28343,28336,28365,28326,28367,28338,28350,28355,28380,28376,28313,28306,28302,28301,28324,28321,28351,28339,28368,28362,28311,28334,28323,28999,29012,29010,29027,29024,28993,29021,29026,29042,29048,29034,29025,28994,29016,28995,29003,29040,29023,29008,29011,28996,29005,29018,29263,29325,29324,29329,29328,29326,29500,29506,29499,29498,29504,29514,29513,29764,29770,29771,29778,29777,29783,29760,29775,29776,29774,29762,29766,29773,29780,29921,29951,29950,29949,29981,30073,30071,27011,30191,30223,30211,30199,30206,30204,30201,30200,30224,30203,30198,30189,30197,30205,30361,30389,30429,30549,30559,30560,30546,30550,30554,30569,30567,30548,30553,30573,30688,30855,30874,30868,30863,30852,30869,30853,30854,30881,30851,30841,30873,30848,30870,30843,31100,31106,31101,31097,31249,31256,31257,31250,31255,31253,31266,31251,31259,31248,31395,31394,31390,31467,31590,31588,31597,31604,31593,31602,31589,31603,31601,31600,31585,31608,31606,31587,31922,31924,31919,32136,32134,32128,32141,32127,32133,32122,32142,32123,32131,32124,32140,32148,32132,32125,32146,32621,32619,32615,32616,32620,32678,32677,32679,32731,32732,32801,33124,33120,33143,33116,33129,33115,33122,33138,26401,33118,33142,33127,33135,33092,33121,33309,33353,33348,33344,33346,33349,34033,33855,33878,33910,33913,33935,33933,33893,33873,33856,33926,33895,33840,33869,33917,33882,33881,33908,33907,33885,34055,33886,33847,33850,33844,33914,33859,33912,33842,33861,33833,33753,33867,33839,33858,33837,33887,33904,33849,33870,33868,33874,33903,33989,33934,33851,33863,33846,33843,33896,33918,33860,33835,33888,33876,33902,33872,34571,34564,34551,34572,34554,34518,34549,34637,34552,34574,34569,34561,34550,34573,34565,35030,35019,35021,35022,35038,35035,35034,35020,35024,35205,35227,35295,35301,35300,35297,35296,35298,35292,35302,35446,35462,35455,35425,35391,35447,35458,35460,35445,35459,35457,35444,35450,35900,35915,35914,35941,35940,35942,35974,35972,35973,36044,36200,36201,36241,36236,36238,36239,36237,36243,36244,36240,36242,36336,36320,36332,36337,36334,36304,36329,36323,36322,36327,36338,36331,36340,36614,36607,36609,36608,36613,36615,36616,36610,36619,36946,36927,36932,36937,36925,37136,37133,37135,37137,37142,37140,37131,37134,37230,37231,37448,37458,37424,37434,37478,37427,37477,37470,37507,37422,37450,37446,37485,37484,37455,37472,37479,37487,37430,37473,37488,37425,37460,37475,37456,37490,37454,37459,37452,37462,37426,38303,38300,38302,38299,38546,38547,38545,38551,38606,38650,38653,38648,38645,38771,38775,38776,38770,38927,38925,38926,39084,39158,39161,39343,39346,39344,39349,39597,39595,39771,40170,40173,40167,40576,40701,20710,20692,20695,20712,20723,20699,20714,20701,20708,20691,20716,20720,20719,20707,20704,20952,21120,21121,21225,21227,21296,21420,22055,22037,22028,22034,22012,22031,22044,22017,22035,22018,22010,22045,22020,22015,22009,22665,22652,22672,22680,22662,22657,22655,22644,22667,22650,22663,22673,22670,22646,22658,22664,22651,22676,22671,22782,22891,23260,23278,23269,23253,23274,23258,23277,23275,23283,23266,23264,23259,23276,23262,23261,23257,23272,23263,23415,23520,23523,23651,23938,23936,23933,23942,23930,23937,23927,23946,23945,23944,23934,23932,23949,23929,23935,24152,24153,24147,24280,24273,24279,24270,24284,24277,24281,24274,24276,24388,24387,24431,24502,24876,24872,24897,24926,24945,24947,24914,24915,24946,24940,24960,24948,24916,24954,24923,24933,24891,24938,24929,24918,25129,25127,25131,25643,25677,25691,25693,25716,25718,25714,25715,25725,25717,25702,25766,25678,25730,25694,25692,25675,25683,25696,25680,25727,25663,25708,25707,25689,25701,25719,25971,26016,26273,26272,26271,26373,26372,26402,27057,27062,27081,27040,27086,27030,27056,27052,27068,27025,27033,27022,27047,27021,27049,27070,27055,27071,27076,27069,27044,27092,27065,27082,27034,27087,27059,27027,27050,27041,27038,27097,27031,27024,27074,27061,27045,27078,27466,27469,27467,27550,27551,27552,27587,27588,27646,28366,28405,28401,28419,28453,28408,28471,28411,28462,28425,28494,28441,28442,28455,28440,28475,28434,28397,28426,28470,28531,28409,28398,28461,28480,28464,28476,28469,28395,28423,28430,28483,28421,28413,28406,28473,28444,28412,28474,28447,28429,28446,28424,28449,29063,29072,29065,29056,29061,29058,29071,29051,29062,29057,29079,29252,29267,29335,29333,29331,29507,29517,29521,29516,29794,29811,29809,29813,29810,29799,29806,29952,29954,29955,30077,30096,30230,30216,30220,30229,30225,30218,30228,30392,30593,30588,30597,30594,30574,30592,30575,30590,30595,30898,30890,30900,30893,30888,30846,30891,30878,30885,30880,30892,30882,30884,31128,31114,31115,31126,31125,31124,31123,31127,31112,31122,31120,31275,31306,31280,31279,31272,31270,31400,31403,31404,31470,31624,31644,31626,31633,31632,31638,31629,31628,31643,31630,31621,31640,21124,31641,31652,31618,31931,31935,31932,31930,32167,32183,32194,32163,32170,32193,32192,32197,32157,32206,32196,32198,32203,32204,32175,32185,32150,32188,32159,32166,32174,32169,32161,32201,32627,32738,32739,32741,32734,32804,32861,32860,33161,33158,33155,33159,33165,33164,33163,33301,33943,33956,33953,33951,33978,33998,33986,33964,33966,33963,33977,33972,33985,33997,33962,33946,33969,34000,33949,33959,33979,33954,33940,33991,33996,33947,33961,33967,33960,34006,33944,33974,33999,33952,34007,34004,34002,34011,33968,33937,34401,34611,34595,34600,34667,34624,34606,34590,34593,34585,34587,34627,34604,34625,34622,34630,34592,34610,34602,34605,34620,34578,34618,34609,34613,34626,34598,34599,34616,34596,34586,34608,34577,35063,35047,35057,35058,35066,35070,35054,35068,35062,35067,35056,35052,35051,35229,35233,35231,35230,35305,35307,35304,35499,35481,35467,35474,35471,35478,35901,35944,35945,36053,36047,36055,36246,36361,36354,36351,36365,36349,36362,36355,36359,36358,36357,36350,36352,36356,36624,36625,36622,36621,37155,37148,37152,37154,37151,37149,37146,37156,37153,37147,37242,37234,37241,37235,37541,37540,37494,37531,37498,37536,37524,37546,37517,37542,37530,37547,37497,37527,37503,37539,37614,37518,37506,37525,37538,37501,37512,37537,37514,37510,37516,37529,37543,37502,37511,37545,37533,37515,37421,38558,38561,38655,38744,38781,38778,38782,38787,38784,38786,38779,38788,38785,38783,38862,38861,38934,39085,39086,39170,39168,39175,39325,39324,39363,39353,39355,39354,39362,39357,39367,39601,39651,39655,39742,39743,39776,39777,39775,40177,40178,40181,40615,20735,20739,20784,20728,20742,20743,20726,20734,20747,20748,20733,20746,21131,21132,21233,21231,22088,22082,22092,22069,22081,22090,22089,22086,22104,22106,22080,22067,22077,22060,22078,22072,22058,22074,22298,22699,22685,22705,22688,22691,22703,22700,22693,22689,22783,23295,23284,23293,23287,23286,23299,23288,23298,23289,23297,23303,23301,23311,23655,23961,23959,23967,23954,23970,23955,23957,23968,23964,23969,23962,23966,24169,24157,24160,24156,32243,24283,24286,24289,24393,24498,24971,24963,24953,25009,25008,24994,24969,24987,24979,25007,25005,24991,24978,25002,24993,24973,24934,25011,25133,25710,25712,25750,25760,25733,25751,25756,25743,25739,25738,25740,25763,25759,25704,25777,25752,25974,25978,25977,25979,26034,26035,26293,26288,26281,26290,26295,26282,26287,27136,27142,27159,27109,27128,27157,27121,27108,27168,27135,27116,27106,27163,27165,27134,27175,27122,27118,27156,27127,27111,27200,27144,27110,27131,27149,27132,27115,27145,27140,27160,27173,27151,27126,27174,27143,27124,27158,27473,27557,27555,27554,27558,27649,27648,27647,27650,28481,28454,28542,28551,28614,28562,28557,28553,28556,28514,28495,28549,28506,28566,28534,28524,28546,28501,28530,28498,28496,28503,28564,28563,28509,28416,28513,28523,28541,28519,28560,28499,28555,28521,28543,28565,28515,28535,28522,28539,29106,29103,29083,29104,29088,29082,29097,29109,29085,29093,29086,29092,29089,29098,29084,29095,29107,29336,29338,29528,29522,29534,29535,29536,29533,29531,29537,29530,29529,29538,29831,63892,29834,29830,29825,29821,29829,29832,29820,29817,29960,29959,30078,30245,30238,30233,30237,30236,30243,30234,30248,30235,30364,30365,30366,30363,30605,30607,30601,30600,30925,30907,30927,30924,30929,30926,30932,30920,30915,30916,30921,31130,31137,31136,31132,31138,31131,27510,31289,31410,31412,31411,31671,31691,31678,31660,31694,31663,31673,31690,31669,31941,31944,31948,31947,32247,32219,32234,32231,32215,32225,32259,32250,32230,32246,32241,32240,32238,32223,32630,32684,32688,32685,32749,32747,32746,32748,32742,32744,32868,32871,33187,33183,33182,33173,33186,33177,33175,33302,33359,33363,33362,33360,33358,33361,34084,34107,34063,34048,34089,34062,34057,34061,34079,34058,34087,34076,34043,34091,34042,34056,34060,34036,34090,34034,34069,34039,34027,34035,63938,34066,34026,34025,34070,34046,34088,34077,34094,34050,34045,34078,34038,34097,34086,34023,34024,34032,34031,34041,34072,34080,34096,34059,34073,34095,34402,34646,34659,34660,34679,34785,34675,34648,34644,34651,34642,34657,34650,34641,34654,34669,34666,34640,34638,34655,34653,34671,34668,34682,34670,34652,34661,34639,34683,34677,34658,34663,34665,34906,35077,35084,35092,35083,35095,35096,35097,35078,35094,35089,35086,35081,35234,35236,35235,35309,35312,35308,35535,35526,35512,35539,35537,35540,35541,35515,35543,35518,35520,35525,35544,35523,35514,35517,35545,35902,35917,35983,36069,36063,36057,36072,36058,36061,36071,36256,36252,36257,36251,36384,36387,36389,36388,36398,36373,36379,36374,36369,36377,36390,36391,36372,36370,36376,36371,36380,36375,36378,36652,36644,36632,36634,36640,36643,36630,36631,36979,36976,36975,36967,36971,37167,37163,37161,37162,37170,37158,37166,37253,37254,37258,37249,37250,37252,37248,37584,37571,37572,37568,37593,37558,37583,37617,37599,37592,37609,37591,37597,37580,37615,37570,37608,37578,37576,37582,37606,37581,37589,37577,37600,37598,37607,37585,37587,37557,37601,37574,37556,38268,38316,38315,38318,38320,38564,38562,38611,38661,38664,38658,38746,38794,38798,38792,38864,38863,38942,38941,38950,38953,38952,38944,38939,38951,39090,39176,39162,39185,39188,39190,39191,39189,39388,39373,39375,39379,39380,39374,39369,39382,39384,39371,39383,39372,39603,39660,39659,39667,39666,39665,39750,39747,39783,39796,39793,39782,39798,39797,39792,39784,39780,39788,40188,40186,40189,40191,40183,40199,40192,40185,40187,40200,40197,40196,40579,40659,40719,40720,20764,20755,20759,20762,20753,64021,21300,21473,22128,22112,22126,22131,22118,22115,22125,22130,22110,22135,22300,22299,22728,22717,22729,22719,22714,22722,22716,22726,23319,23321,23323,23329,23316,23315,23312,23318,23336,23322,23328,23326,23535,23980,23985,23977,23975,23989,23984,23982,23978,23976,23986,23981,23983,23988,24167,24168,24166,24175,24297,24295,24294,24296,24293,24395,24508,24989,25000,24982,25029,25012,25030,25025,25036,25018,25023,25016,24972,25815,25814,25808,25807,25801,25789,25737,25795,25819,25843,25817,25907,25983,25980,26018,26312,26302,26304,26314,26315,26319,26301,26299,26298,26316,26403,27188,27238,27209,27239,27186,27240,27198,27229,27245,27254,27227,27217,27176,27226,27195,27199,27201,27242,27236,27216,27215,27220,27247,27241,27232,27196,27230,27222,27221,27213,27214,27206,27477,27476,27478,27559,27562,27563,27592,27591,27652,27651,27654,28589,28619,28579,28615,28604,28622,28616,28510,28612,28605,28574,28618,28584,28676,28581,28590,28602,28588,28586,28623,28607,28600,28578,28617,28587,28621,28591,28594,28592,29125,29122,29119,29112,29142,29120,29121,29131,29140,29130,29127,29135,29117,29144,29116,29126,29146,29147,29341,29342,29545,29542,29543,29548,29541,29547,29546,29823,29850,29856,29844,29842,29845,29857,29963,30080,30255,30253,30257,30269,30259,30268,30261,30258,30256,30395,30438,30618,30621,30625,30620,30619,30626,30627,30613,30617,30615,30941,30953,30949,30954,30942,30947,30939,30945,30946,30957,30943,30944,31140,31300,31304,31303,31414,31416,31413,31409,31415,31710,31715,31719,31709,31701,31717,31706,31720,31737,31700,31722,31714,31708,31723,31704,31711,31954,31956,31959,31952,31953,32274,32289,32279,32268,32287,32288,32275,32270,32284,32277,32282,32290,32267,32271,32278,32269,32276,32293,32292,32579,32635,32636,32634,32689,32751,32810,32809,32876,33201,33190,33198,33209,33205,33195,33200,33196,33204,33202,33207,33191,33266,33365,33366,33367,34134,34117,34155,34125,34131,34145,34136,34112,34118,34148,34113,34146,34116,34129,34119,34147,34110,34139,34161,34126,34158,34165,34133,34151,34144,34188,34150,34141,34132,34149,34156,34403,34405,34404,34715,34703,34711,34707,34706,34696,34689,34710,34712,34681,34695,34723,34693,34704,34705,34717,34692,34708,34716,34714,34697,35102,35110,35120,35117,35118,35111,35121,35106,35113,35107,35119,35116,35103,35313,35552,35554,35570,35572,35573,35549,35604,35556,35551,35568,35528,35550,35553,35560,35583,35567,35579,35985,35986,35984,36085,36078,36081,36080,36083,36204,36206,36261,36263,36403,36414,36408,36416,36421,36406,36412,36413,36417,36400,36415,36541,36662,36654,36661,36658,36665,36663,36660,36982,36985,36987,36998,37114,37171,37173,37174,37267,37264,37265,37261,37263,37671,37662,37640,37663,37638,37647,37754,37688,37692,37659,37667,37650,37633,37702,37677,37646,37645,37579,37661,37626,37669,37651,37625,37623,37684,37634,37668,37631,37673,37689,37685,37674,37652,37644,37643,37630,37641,37632,37627,37654,38332,38349,38334,38329,38330,38326,38335,38325,38333,38569,38612,38667,38674,38672,38809,38807,38804,38896,38904,38965,38959,38962,39204,39199,39207,39209,39326,39406,39404,39397,39396,39408,39395,39402,39401,39399,39609,39615,39604,39611,39670,39674,39673,39671,39731,39808,39813,39815,39804,39806,39803,39810,39827,39826,39824,39802,39829,39805,39816,40229,40215,40224,40222,40212,40233,40221,40216,40226,40208,40217,40223,40584,40582,40583,40622,40621,40661,40662,40698,40722,40765,20774,20773,20770,20772,20768,20777,21236,22163,22156,22157,22150,22148,22147,22142,22146,22143,22145,22742,22740,22735,22738,23341,23333,23346,23331,23340,23335,23334,23343,23342,23419,23537,23538,23991,24172,24170,24510,24507,25027,25013,25020,25063,25056,25061,25060,25064,25054,25839,25833,25827,25835,25828,25832,25985,25984,26038,26074,26322,27277,27286,27265,27301,27273,27295,27291,27297,27294,27271,27283,27278,27285,27267,27304,27300,27281,27263,27302,27290,27269,27276,27282,27483,27565,27657,28620,28585,28660,28628,28643,28636,28653,28647,28646,28638,28658,28637,28642,28648,29153,29169,29160,29170,29156,29168,29154,29555,29550,29551,29847,29874,29867,29840,29866,29869,29873,29861,29871,29968,29969,29970,29967,30084,30275,30280,30281,30279,30372,30441,30645,30635,30642,30647,30646,30644,30641,30632,30704,30963,30973,30978,63844,30972,30962,30981,30969,30974,30980,31147,31144,31324,31323,31318,31320,31316,31322,31422,31424,31425,31749,31759,31730,31744,31743,31739,31758,31732,31755,31731,31746,31753,31747,31745,31736,31741,31750,31728,31729,31760,31754,31976,32301,32316,32322,32307,38984,32312,32298,32329,32320,32327,32297,32332,32304,32315,32310,32324,32314,32581,32639,32638,32637,32756,32754,32812,33211,33220,33228,33226,33221,33223,33212,33257,33371,33370,33372,34179,34176,34191,34215,34197,34208,34187,34211,34171,34212,34202,34206,34167,34172,34185,34209,34170,34168,34135,34190,34198,34182,34189,34201,34205,34177,34210,34178,34184,34181,34169,34166,34200,34192,34207,34408,34750,34730,34733,34757,34736,34732,34745,34741,34748,34734,34761,34755,34754,34764,34743,34735,34756,34762,34740,34742,34751,34744,34749,34782,34738,35125,35123,35132,35134,35137,35154,35127,35138,35245,35247,35246,35314,35315,35614,35608,35606,35601,35589,35595,35618,35599,35602,35605,35591,35597,35592,35590,35612,35603,35610,35919,35952,35954,35953,35951,35989,35988,36089,36207,36430,36429,36435,36432,36428,36423,36675,36672,36997,36990,37176,37274,37282,37275,37273,37279,37281,37277,37280,37793,37763,37807,37732,37718,37703,37756,37720,37724,37750,37705,37712,37713,37728,37741,37775,37708,37738,37753,37719,37717,37714,37711,37745,37751,37755,37729,37726,37731,37735,37760,37710,37721,38343,38336,38345,38339,38341,38327,38574,38576,38572,38688,38687,38680,38685,38681,38810,38817,38812,38814,38813,38869,38868,38897,38977,38980,38986,38985,38981,38979,39205,39211,39212,39210,39219,39218,39215,39213,39217,39216,39320,39331,39329,39426,39418,39412,39415,39417,39416,39414,39419,39421,39422,39420,39427,39614,39678,39677,39681,39676,39752,39834,39848,39838,39835,39846,39841,39845,39844,39814,39842,39840,39855,40243,40257,40295,40246,40238,40239,40241,40248,40240,40261,40258,40259,40254,40247,40256,40253,32757,40237,40586,40585,40589,40624,40648,40666,40699,40703,40740,40739,40738,40788,40864,20785,20781,20782,22168,22172,22167,22170,22173,22169,22896,23356,23657,23658,24000,24173,24174,25048,25055,25069,25070,25073,25066,25072,25067,25046,25065,25855,25860,25853,25848,25857,25859,25852,26004,26075,26330,26331,26328,27333,27321,27325,27361,27334,27322,27318,27319,27335,27316,27309,27486,27593,27659,28679,28684,28685,28673,28677,28692,28686,28671,28672,28667,28710,28668,28663,28682,29185,29183,29177,29187,29181,29558,29880,29888,29877,29889,29886,29878,29883,29890,29972,29971,30300,30308,30297,30288,30291,30295,30298,30374,30397,30444,30658,30650,30975,30988,30995,30996,30985,30992,30994,30993,31149,31148,31327,31772,31785,31769,31776,31775,31789,31773,31782,31784,31778,31781,31792,32348,32336,32342,32355,32344,32354,32351,32337,32352,32343,32339,32693,32691,32759,32760,32885,33233,33234,33232,33375,33374,34228,34246,34240,34243,34242,34227,34229,34237,34247,34244,34239,34251,34254,34248,34245,34225,34230,34258,34340,34232,34231,34238,34409,34791,34790,34786,34779,34795,34794,34789,34783,34803,34788,34772,34780,34771,34797,34776,34787,34724,34775,34777,34817,34804,34792,34781,35155,35147,35151,35148,35142,35152,35153,35145,35626,35623,35619,35635,35632,35637,35655,35631,35644,35646,35633,35621,35639,35622,35638,35630,35620,35643,35645,35642,35906,35957,35993,35992,35991,36094,36100,36098,36096,36444,36450,36448,36439,36438,36446,36453,36455,36443,36442,36449,36445,36457,36436,36678,36679,36680,36683,37160,37178,37179,37182,37288,37285,37287,37295,37290,37813,37772,37778,37815,37787,37789,37769,37799,37774,37802,37790,37798,37781,37768,37785,37791,37773,37809,37777,37810,37796,37800,37812,37795,37797,38354,38355,38353,38579,38615,38618,24002,38623,38616,38621,38691,38690,38693,38828,38830,38824,38827,38820,38826,38818,38821,38871,38873,38870,38872,38906,38992,38993,38994,39096,39233,39228,39226,39439,39435,39433,39437,39428,39441,39434,39429,39431,39430,39616,39644,39688,39684,39685,39721,39733,39754,39756,39755,39879,39878,39875,39871,39873,39861,39864,39891,39862,39876,39865,39869,40284,40275,40271,40266,40283,40267,40281,40278,40268,40279,40274,40276,40287,40280,40282,40590,40588,40671,40705,40704,40726,40741,40747,40746,40745,40744,40780,40789,20788,20789,21142,21239,21428,22187,22189,22182,22183,22186,22188,22746,22749,22747,22802,23357,23358,23359,24003,24176,24511,25083,25863,25872,25869,25865,25868,25870,25988,26078,26077,26334,27367,27360,27340,27345,27353,27339,27359,27356,27344,27371,27343,27341,27358,27488,27568,27660,28697,28711,28704,28694,28715,28705,28706,28707,28713,28695,28708,28700,28714,29196,29194,29191,29186,29189,29349,29350,29348,29347,29345,29899,29893,29879,29891,29974,30304,30665,30666,30660,30705,31005,31003,31009,31004,30999,31006,31152,31335,31336,31795,31804,31801,31788,31803,31980,31978,32374,32373,32376,32368,32375,32367,32378,32370,32372,32360,32587,32586,32643,32646,32695,32765,32766,32888,33239,33237,33380,33377,33379,34283,34289,34285,34265,34273,34280,34266,34263,34284,34290,34296,34264,34271,34275,34268,34257,34288,34278,34287,34270,34274,34816,34810,34819,34806,34807,34825,34828,34827,34822,34812,34824,34815,34826,34818,35170,35162,35163,35159,35169,35164,35160,35165,35161,35208,35255,35254,35318,35664,35656,35658,35648,35667,35670,35668,35659,35669,35665,35650,35666,35671,35907,35959,35958,35994,36102,36103,36105,36268,36266,36269,36267,36461,36472,36467,36458,36463,36475,36546,36690,36689,36687,36688,36691,36788,37184,37183,37296,37293,37854,37831,37839,37826,37850,37840,37881,37868,37836,37849,37801,37862,37834,37844,37870,37859,37845,37828,37838,37824,37842,37863,38269,38362,38363,38625,38697,38699,38700,38696,38694,38835,38839,38838,38877,38878,38879,39004,39001,39005,38999,39103,39101,39099,39102,39240,39239,39235,39334,39335,39450,39445,39461,39453,39460,39451,39458,39456,39463,39459,39454,39452,39444,39618,39691,39690,39694,39692,39735,39914,39915,39904,39902,39908,39910,39906,39920,39892,39895,39916,39900,39897,39909,39893,39905,39898,40311,40321,40330,40324,40328,40305,40320,40312,40326,40331,40332,40317,40299,40308,40309,40304,40297,40325,40307,40315,40322,40303,40313,40319,40327,40296,40596,40593,40640,40700,40749,40768,40769,40781,40790,40791,40792,21303,22194,22197,22195,22755,23365,24006,24007,24302,24303,24512,24513,25081,25879,25878,25877,25875,26079,26344,26339,26340,27379,27376,27370,27368,27385,27377,27374,27375,28732,28725,28719,28727,28724,28721,28738,28728,28735,28730,28729,28736,28731,28723,28737,29203,29204,29352,29565,29564,29882,30379,30378,30398,30445,30668,30670,30671,30669,30706,31013,31011,31015,31016,31012,31017,31154,31342,31340,31341,31479,31817,31816,31818,31815,31813,31982,32379,32382,32385,32384,32698,32767,32889,33243,33241,33291,33384,33385,34338,34303,34305,34302,34331,34304,34294,34308,34313,34309,34316,34301,34841,34832,34833,34839,34835,34838,35171,35174,35257,35319,35680,35690,35677,35688,35683,35685,35687,35693,36270,36486,36488,36484,36697,36694,36695,36693,36696,36698,37005,37187,37185,37303,37301,37298,37299,37899,37907,37883,37920,37903,37908,37886,37909,37904,37928,37913,37901,37877,37888,37879,37895,37902,37910,37906,37882,37897,37880,37898,37887,37884,37900,37878,37905,37894,38366,38368,38367,38702,38703,38841,38843,38909,38910,39008,39010,39011,39007,39105,39106,39248,39246,39257,39244,39243,39251,39474,39476,39473,39468,39466,39478,39465,39470,39480,39469,39623,39626,39622,39696,39698,39697,39947,39944,39927,39941,39954,39928,40000,39943,39950,39942,39959,39956,39945,40351,40345,40356,40349,40338,40344,40336,40347,40352,40340,40348,40362,40343,40353,40346,40354,40360,40350,40355,40383,40361,40342,40358,40359,40601,40603,40602,40677,40676,40679,40678,40752,40750,40795,40800,40798,40797,40793,40849,20794,20793,21144,21143,22211,22205,22206,23368,23367,24011,24015,24305,25085,25883,27394,27388,27395,27384,27392,28739,28740,28746,28744,28745,28741,28742,29213,29210,29209,29566,29975,30314,30672,31021,31025,31023,31828,31827,31986,32394,32391,32392,32395,32390,32397,32589,32699,32816,33245,34328,34346,34342,34335,34339,34332,34329,34343,34350,34337,34336,34345,34334,34341,34857,34845,34843,34848,34852,34844,34859,34890,35181,35177,35182,35179,35322,35705,35704,35653,35706,35707,36112,36116,36271,36494,36492,36702,36699,36701,37190,37188,37189,37305,37951,37947,37942,37929,37949,37948,37936,37945,37930,37943,37932,37952,37937,38373,38372,38371,38709,38714,38847,38881,39012,39113,39110,39104,39256,39254,39481,39485,39494,39492,39490,39489,39482,39487,39629,39701,39703,39704,39702,39738,39762,39979,39965,39964,39980,39971,39976,39977,39972,39969,40375,40374,40380,40385,40391,40394,40399,40382,40389,40387,40379,40373,40398,40377,40378,40364,40392,40369,40365,40396,40371,40397,40370,40570,40604,40683,40686,40685,40731,40728,40730,40753,40782,40805,40804,40850,20153,22214,22213,22219,22897,23371,23372,24021,24017,24306,25889,25888,25894,25890,27403,27400,27401,27661,28757,28758,28759,28754,29214,29215,29353,29567,29912,29909,29913,29911,30317,30381,31029,31156,31344,31345,31831,31836,31833,31835,31834,31988,31985,32401,32591,32647,33246,33387,34356,34357,34355,34348,34354,34358,34860,34856,34854,34858,34853,35185,35263,35262,35323,35710,35716,35714,35718,35717,35711,36117,36501,36500,36506,36498,36496,36502,36503,36704,63885,37191,37964,37968,37962,37963,37967,37959,37957,37960,37961,37958,38719,38883,39018,39017,39115,39252,39259,39502,39507,39508,39500,39503,39496,39498,39497,39506,39504,39632,39705,39723,39739,39766,39765,40006,40008,39999,40004,39993,39987,40001,39996,39991,39988,39986,39997,39990,40411,40402,40414,40410,40395,40400,40412,40401,40415,40425,40409,40408,40406,40437,40405,40413,40630,40688,40757,40755,40754,40770,40811,40853,40866,20797,21145,22760,22759,22898,23373,24024,34863,24399,25089,25091,25092,25897,25893,26006,26347,27409,27410,27407,27594,28763,28762,29218,29570,29569,29571,30320,30676,31847,31846,32405,33388,34362,34368,34361,34364,34353,34363,34366,34864,34866,34862,34867,35190,35188,35187,35326,35724,35726,35723,35720,35909,36121,36504,36708,36707,37308,37986,37973,37981,37975,37982,38852,38853,38912,39510,39513,39710,39711,39712,40018,40024,40016,40010,40013,40011,40021,40025,40012,40014,40443,40439,40431,40419,40427,40440,40420,40438,40417,40430,40422,40434,40432,40418,40428,40436,40435,40424,40429,40642,40656,40690,40691,40710,40732,40760,40759,40758,40771,40783,40817,40816,40814,40815,22227,22221,23374,23661,25901,26349,26350,27411,28767,28769,28765,28768,29219,29915,29925,30677,31032,31159,31158,31850,32407,32649,33389,34371,34872,34871,34869,34891,35732,35733,36510,36511,36512,36509,37310,37309,37314,37995,37992,37993,38629,38726,38723,38727,38855,38885,39518,39637,39769,40035,40039,40038,40034,40030,40032,40450,40446,40455,40451,40454,40453,40448,40449,40457,40447,40445,40452,40608,40734,40774,40820,40821,40822,22228,25902,26040,27416,27417,27415,27418,28770,29222,29354,30680,30681,31033,31849,31851,31990,32410,32408,32411,32409,33248,33249,34374,34375,34376,35193,35194,35196,35195,35327,35736,35737,36517,36516,36515,37998,37997,37999,38001,38003,38729,39026,39263,40040,40046,40045,40459,40461,40464,40463,40466,40465,40609,40693,40713,40775,40824,40827,40826,40825,22302,28774,31855,34876,36274,36518,37315,38004,38008,38006,38005,39520,40052,40051,40049,40053,40468,40467,40694,40714,40868,28776,28773,31991,34410,34878,34877,34879,35742,35996,36521,36553,38731,39027,39028,39116,39265,39339,39524,39526,39527,39716,40469,40471,40776,25095,27422,29223,34380,36520,38018,38016,38017,39529,39528,39726,40473,29225,34379,35743,38019,40057,40631,30325,39531,40058,40477,28777,28778,40612,40830,40777,40856,30849,37561,35023,22715,24658,31911,23290, 9556, 9574, 9559, 9568, 9580, 9571, 9562, 9577, 9565, 9554, 9572, 9557, 9566, 9578, 9569, 9560, 9575, 9563, 9555, 9573, 9558, 9567, 9579, 9570, 9561, 9576, 9564, 9553, 9552, 9581, 9582, 9584, 9583, 9619,57344,57345,57346,57347,57348,57349,57350,57351,57352,57353,57354,57355,57356,57357,57358,57359,57360,57361,57362,57363,57364,57365,57366,57367,57368,57369,57370,57371,57372,57373,57374,57375,57376,57377,57378,57379,57380,57381,57382,57383,57384,57385,57386,57387,57388,57389,57390,57391,57392,57393,57394,57395,57396,57397,57398,57399,57400,57401,57402,57403,57404,57405,57406,57407,57408,57409,57410,57411,57412,57413,57414,57415,57416,57417,57418,57419,57420,57421,57422,57423,57424,57425,57426,57427,57428,57429,57430,57431,57432,57433,57434,57435,57436,57437,57438,57439,57440,57441,57442,57443,57444,57445,57446,57447,57448,57449,57450,57451,57452,57453,57454,57455,57456,57457,57458,57459,57460,57461,57462,57463,57464,57465,57466,57467,57468,57469,57470,57471,57472,57473,57474,57475,57476,57477,57478,57479,57480,57481,57482,57483,57484,57485,57486,57487,57488,57489,57490,57491,57492,57493,57494,57495,57496,57497,57498,57499,57500,57501,57502,57503,57504,57505,57506,57507,57508,57509,57510,57511,57512,57513,57514,57515,57516,57517,57518,57519,57520,57521,57522,57523,57524,57525,57526,57527,57528,57529,57530,57531,57532,57533,57534,57535,57536,57537,57538,57539,57540,57541,57542,57543,57544,57545,57546,57547,57548,57549,57550,57551,57552,57553,57554,57555,57556,57557,57558,57559,57560,57561,57562,57563,57564,57565,57566,57567,57568,57569,57570,57571,57572,57573,57574,57575,57576,57577,57578,57579,57580,57581,57582,57583,57584,57585,57586,57587,57588,57589,57590,57591,57592,57593,57594,57595,57596,57597,57598,57599,57600,57601,57602,57603,57604,57605,57606,57607,57608,57609,57610,57611,57612,57613,57614,57615,57616,57617,57618,57619,57620,57621,57622,57623,57624,57625,57626,57627,57628,57629,57630,57631,57632,57633,57634,57635,57636,57637,57638,57639,57640,57641,57642,57643,57644,57645,57646,57647,57648,57649,57650,57651,57652,57653,57654,57655,57656,57657,57658,57659,57660,57661,57662,57663,57664,57665,57666,57667,57668,57669,57670,57671,57672,57673,57674,57675,57676,57677,57678,57679,57680,57681,57682,57683,57684,57685,57686,57687,57688,57689,57690,57691,57692,57693,57694,57695,57696,57697,57698,57699,57700,57701,57702,57703,57704,57705,57706,57707,57708,57709,57710,57711,57712,57713,57714,57715,57716,57717,57718,57719,57720,57721,57722,57723,57724,57725,57726,57727,57728,57729,57730,57731,57732,57733,57734,57735,57736,57737,57738,57739,57740,57741,57742,57743,57744,57745,57746,57747,57748,57749,57750,57751,57752,57753,57754,57755,57756,57757,57758,57759,57760,57761,57762,57763,57764,57765,57766,57767,57768,57769,57770,57771,57772,57773,57774,57775,57776,57777,57778,57779,57780,57781,57782,57783,57784,57785,57786,57787,57788,57789,57790,57791,57792,57793,57794,57795,57796,57797,57798,57799,57800,57801,57802,57803,57804,57805,57806,57807,57808,57809,57810,57811,57812,57813,57814,57815,57816,57817,57818,57819,57820,57821,57822,57823,57824,57825,57826,57827,57828,57829,57830,57831,57832,57833,57834,57835,57836,57837,57838,57839,57840,57841,57842,57843,57844,57845,57846,57847,57848,57849,57850,57851,57852,57853,57854,57855,57856,57857,57858,57859,57860,57861,57862,57863,57864,57865,57866,57867,57868,57869,57870,57871,57872,57873,57874,57875,57876,57877,57878,57879,57880,57881,57882,57883,57884,57885,57886,57887,57888,57889,57890,57891,57892,57893,57894,57895,57896,57897,57898,57899,57900,57901,57902,57903,57904,57905,57906,57907,57908,57909,57910,57911,57912,57913,57914,57915,57916,57917,57918,57919,57920,57921,57922,57923,57924,57925,57926,57927,57928,57929,57930,57931,57932,57933,57934,57935,57936,57937,57938,57939,57940,57941,57942,57943,57944,57945,57946,57947,57948,57949,57950,57951,57952,57953,57954,57955,57956,57957,57958,57959,57960,57961,57962,57963,57964,57965,57966,57967,57968,57969,57970,57971,57972,57973,57974,57975,57976,57977,57978,57979,57980,57981,57982,57983,57984,57985,57986,57987,57988,57989,57990,57991,57992,57993,57994,57995,57996,57997,57998,57999,58000,58001,58002,58003,58004,58005,58006,58007,58008,58009,58010,58011,58012,58013,58014,58015,58016,58017,58018,58019,58020,58021,58022,58023,58024,58025,58026,58027,58028,58029,58030,58031,58032,58033,58034,58035,58036,58037,58038,58039,58040,58041,58042,58043,58044,58045,58046,58047,58048,58049,58050,58051,58052,58053,58054,58055,58056,58057,58058,58059,58060,58061,58062,58063,58064,58065,58066,58067,58068,58069,58070,58071,58072,58073,58074,58075,58076,58077,58078,58079,58080,58081,58082,58083,58084,58085,58086,58087,58088,58089,58090,58091,58092,58093,58094,58095,58096,58097,58098,58099,58100,58101,58102,58103,58104,58105,58106,58107,58108,58109,58110,58111,58112,58113,58114,58115,58116,58117,58118,58119,58120,58121,58122,58123,58124,58125,58126,58127,58128 };

void big5toutf8_info(struct session *ses)
{
	int index, max = sizeof(big5_to_unicode_keys) / sizeof(unsigned short);
	char buf[7];

	for (index = 0 ; index < max; index++)
	{
		unicode_to_utf8(big5_to_unicode_vals[index], buf);

		tintin_printf2(ses, "%5d %7d  %02x-%02x  %7d %s", index, big5_to_unicode_keys[index], big5_to_unicode_keys[index] / 256, big5_to_unicode_keys[index] % 256, big5_to_unicode_vals[index], buf);
	}
	tintin_printf2(ses, "-------------------------------");
	tintin_printf2(ses, "%5s %7s  %5s  %5s", "index", "decimal", "hex", "unicode");
}

void utf8tobig5_info(struct session *ses)
{
	int index, max = sizeof(unicode_to_big5_keys) / sizeof(unsigned short);
	char buf[7];

	for (index = 0 ; index < max; index++)
	{
		unicode_to_utf8(unicode_to_big5_keys[index], buf);

		tintin_printf2(ses, "%5d %7d  %02x-%02x  %7d %s", index, unicode_to_big5_keys[index], unicode_to_big5_vals[index] / 256, unicode_to_big5_vals[index] % 256, unicode_to_big5_vals[index], buf);
	}
	tintin_printf2(ses, "-------------------------------");
	tintin_printf2(ses, "%5s %7s  %5s  %5s", "index", "unicode", "hex", "decimal");
}

int is_big5(char *str)
{
	unsigned char *ptu = (unsigned char *) str;

	return ptu[0] > 128 && ptu[0] < 255 && ptu[1] >= 64 && ptu[1] < 255;
}


int utf8_to_big5(char *input, char *output)
{
	char *pti, *pto;
	int size, index, result;

	pti = input;
	pto = output;

	while (*pti)
	{
		size = get_utf8_index(pti, &index);

		if (size > 1)
		{
			result = boundless_binary_index_search16(unicode_to_big5_keys, sizeof(unicode_to_big5_keys) / sizeof(unsigned short), index);

			if (result >= 0)
			{
				index = unicode_to_big5_vals[result];

				*pto++ = index / 256;
				*pto++ = index % 256;
			}
			else
			{
				tintin_printf(gtd->ses, "<118>utf8_to_big5: did not find unicode index '0x%04x'", index);

				*pto++ = '?';
				*pto++ = '?';
			}
			pti += size;
		}
		else
		{
			*pto++ = *pti++;
		}
	}
	*pto = 0;

	return pto - output;
}

int big5_to_utf8(char *input, char *output)
{
	unsigned char *pti, *pto;
	int index, val;

	pti = (unsigned char *) input;
	pto = (unsigned char *) output;

	while (*pti)
	{
		if (is_big5((char *) pti))
		{
			index = boundless_binary_index_search16(big5_to_unicode_keys, sizeof(big5_to_unicode_keys) / sizeof(unsigned short), (unsigned short) pti[0] * 256 + pti[1]);

			if (index >= 0)
			{
				val = big5_to_unicode_vals[index];

				if (index < 128)
				{
					*pto++ = val;
				}
				else if (val < 4096)
				{
					*pto++ = 192 + val / 64;
					*pto++ = 128 + val % 64;
				}
				else
				{
					*pto++ = 224 + val / 4096;
					*pto++ = 128 + val / 64 % 64;
					*pto++ = 128 + val % 64;
				}
				pti += 2;
			}
			else
			{
				tintin_printf(gtd->ses, "<118>big5_to_utf8: did not find big5 index '0x%02x%02x'", pti[0], pti[1]);
				*pto++ = '?';
				*pto++ = '?';
				pti += 2;
			}
		}
		else
		{
			*pto++ = *pti++;
		}
	}
	*pto = 0;

	return pto - (unsigned char *) output;
}

// GBK-1 

unsigned short unicode_to_gbk1_keys[] = {   164,  167,  168,  176,  177,  183,  215,  224,  225,  232,  233,  234,  236,  237,  242,  243,  247,  249,  250,  252,  257,  275,  283,  299,  324,  328,  333,  363,  462,  464,  466,  468,  470,  472,  474,  476,  593,  609,  711,  713,  714,  715,  729,  913,  914,  915,  916,  917,  918,  919,  920,  921,  922,  923,  924,  925,  926,  927,  928,  929,  931,  932,  933,  934,  935,  936,  937,  945,  946,  947,  948,  949,  950,  951,  952,  953,  954,  955,  956,  957,  958,  959,  960,  961,  963,  964,  965,  966,  967,  968,  969, 1025, 1040, 1041, 1042, 1043, 1044, 1045, 1046, 1047, 1048, 1049, 1050, 1051, 1052, 1053, 1054, 1055, 1056, 1057, 1058, 1059, 1060, 1061, 1062, 1063, 1064, 1065, 1066, 1067, 1068, 1069, 1070, 1071, 1072, 1073, 1074, 1075, 1076, 1077, 1078, 1079, 1080, 1081, 1082, 1083, 1084, 1085, 1086, 1087, 1088, 1089, 1090, 1091, 1092, 1093, 1094, 1095, 1096, 1097, 1098, 1099, 1100, 1101, 1102, 1103, 1105, 8208, 8211, 8212, 8213, 8214, 8216, 8217, 8220, 8221, 8229, 8230, 8240, 8242, 8243, 8245, 8251, 8364, 8451, 8453, 8457, 8470, 8481, 8544, 8545, 8546, 8547, 8548, 8549, 8550, 8551, 8552, 8553, 8554, 8555, 8560, 8561, 8562, 8563, 8564, 8565, 8566, 8567, 8568, 8569, 8592, 8593, 8594, 8595, 8598, 8599, 8600, 8601, 8712, 8719, 8721, 8725, 8730, 8733, 8734, 8735, 8736, 8739, 8741, 8743, 8744, 8745, 8746, 8747, 8750, 8756, 8757, 8758, 8759, 8765, 8776, 8780, 8786, 8800, 8801, 8804, 8805, 8806, 8807, 8814, 8815, 8853, 8857, 8869, 8895, 8978, 9312, 9313, 9314, 9315, 9316, 9317, 9318, 9319, 9320, 9321, 9332, 9333, 9334, 9335, 9336, 9337, 9338, 9339, 9340, 9341, 9342, 9343, 9344, 9345, 9346, 9347, 9348, 9349, 9350, 9351, 9352, 9353, 9354, 9355, 9356, 9357, 9358, 9359, 9360, 9361, 9362, 9363, 9364, 9365, 9366, 9367, 9368, 9369, 9370, 9371, 9472, 9473, 9474, 9475, 9476, 9477, 9478, 9479, 9480, 9481, 9482, 9483, 9484, 9485, 9486, 9487, 9488, 9489, 9490, 9491, 9492, 9493, 9494, 9495, 9496, 9497, 9498, 9499, 9500, 9501, 9502, 9503, 9504, 9505, 9506, 9507, 9508, 9509, 9510, 9511, 9512, 9513, 9514, 9515, 9516, 9517, 9518, 9519, 9520, 9521, 9522, 9523, 9524, 9525, 9526, 9527, 9528, 9529, 9530, 9531, 9532, 9533, 9534, 9535, 9536, 9537, 9538, 9539, 9540, 9541, 9542, 9543, 9544, 9545, 9546, 9547, 9552, 9553, 9554, 9555, 9556, 9557, 9558, 9559, 9560, 9561, 9562, 9563, 9564, 9565, 9566, 9567, 9568, 9569, 9570, 9571, 9572, 9573, 9574, 9575, 9576, 9577, 9578, 9579, 9580, 9581, 9582, 9583, 9584, 9585, 9586, 9587, 9601, 9602, 9603, 9604, 9605, 9606, 9607, 9608, 9609, 9610, 9611, 9612, 9613, 9614, 9615, 9619, 9620, 9621, 9632, 9633, 9650, 9651, 9660, 9661, 9670, 9671, 9675, 9678, 9679, 9698, 9699, 9700, 9701, 9733, 9737, 9792, 9794,11088,12288,12289,12290,12291,12293,12294,12295,12296,12297,12298,12299,12300,12301,12302,12303,12304,12305,12306,12307,12308,12309,12310,12311,12317,12318,12321,12322,12323,12324,12325,12326,12327,12328,12329,12353,12354,12355,12356,12357,12358,12359,12360,12361,12362,12363,12364,12365,12366,12367,12368,12369,12370,12371,12372,12373,12374,12375,12376,12377,12378,12379,12380,12381,12382,12383,12384,12385,12386,12387,12388,12389,12390,12391,12392,12393,12394,12395,12396,12397,12398,12399,12400,12401,12402,12403,12404,12405,12406,12407,12408,12409,12410,12411,12412,12413,12414,12415,12416,12417,12418,12419,12420,12421,12422,12423,12424,12425,12426,12427,12428,12429,12430,12431,12432,12433,12434,12435,12443,12444,12445,12446,12449,12450,12451,12452,12453,12454,12455,12456,12457,12458,12459,12460,12461,12462,12463,12464,12465,12466,12467,12468,12469,12470,12471,12472,12473,12474,12475,12476,12477,12478,12479,12480,12481,12482,12483,12484,12485,12486,12487,12488,12489,12490,12491,12492,12493,12494,12495,12496,12497,12498,12499,12500,12501,12502,12503,12504,12505,12506,12507,12508,12509,12510,12511,12512,12513,12514,12515,12516,12517,12518,12519,12520,12521,12522,12523,12524,12525,12526,12527,12528,12529,12530,12531,12532,12533,12534,12540,12541,12542,12549,12550,12551,12552,12553,12554,12555,12556,12557,12558,12559,12560,12561,12562,12563,12564,12565,12566,12567,12568,12569,12570,12571,12572,12573,12574,12575,12576,12577,12578,12579,12580,12581,12582,12583,12584,12585,12832,12833,12834,12835,12836,12837,12838,12839,12840,12841,12849,12963,13198,13199,13212,13213,13214,13217,13252,13262,13265,13266,13269,19968,19969,19970,19971,19972,19973,19974,19975,19976,19977,19978,19979,19980,19981,19982,19983,19984,19985,19986,19987,19988,19989,19990,19991,19992,19993,19994,19995,19996,19997,19998,19999,20000,20001,20002,20003,20004,20005,20006,20007,20008,20009,20010,20011,20012,20013,20014,20015,20016,20017,20018,20019,20020,20021,20022,20023,20024,20025,20026,20027,20028,20029,20030,20031,20032,20033,20034,20035,20036,20037,20038,20039,20040,20041,20042,20043,20044,20045,20046,20047,20048,20049,20050,20051,20052,20053,20054,20055,20056,20057,20058,20059,20060,20061,20062,20063,20064,20065,20066,20067,20068,20069,20070,20071,20072,20073,20074,20075,20076,20077,20078,20079,20080,20081,20082,20083,20084,20085,20086,20087,20088,20089,20090,20091,20092,20093,20094,20095,20096,20097,20098,20099,20100,20101,20102,20103,20104,20105,20106,20107,20108,20109,20110,20111,20112,20113,20114,20115,20116,20117,20118,20119,20120,20121,20122,20123,20124,20125,20126,20127,20128,20129,20130,20131,20132,20133,20134,20135,20136,20137,20138,20139,20140,20141,20142,20143,20144,20145,20146,20147,20148,20149,20150,20151,20152,20153,20154,20155,20156,20157,20158,20159,20160,20161,20162,20163,20164,20165,20166,20167,20168,20169,20170,20171,20172,20173,20174,20175,20176,20177,20178,20179,20180,20181,20182,20183,20184,20185,20186,20187,20188,20189,20190,20191,20192,20193,20194,20195,20196,20197,20198,20199,20200,20201,20202,20203,20204,20205,20206,20207,20208,20209,20210,20211,20212,20213,20214,20215,20216,20217,20218,20219,20220,20221,20222,20223,20224,20225,20226,20227,20228,20229,20230,20231,20232,20233,20234,20235,20236,20237,20238,20239,20240,20241,20242,20243,20244,20245,20246,20247,20248,20249,20250,20251,20252,20253,20254,20255,20256,20257,20258,20259,20260,20261,20262,20263,20264,20265,20266,20267,20268,20269,20270,20271,20272,20273,20274,20275,20276,20277,20278,20279,20280,20281,20282,20283,20284,20285,20286,20287,20288,20289,20290,20291,20292,20293,20294,20295,20296,20297,20298,20299,20300,20301,20302,20303,20304,20305,20306,20307,20308,20309,20310,20311,20312,20313,20314,20315,20316,20317,20318,20319,20320,20321,20322,20323,20324,20325,20326,20327,20328,20329,20330,20331,20332,20333,20334,20335,20336,20337,20338,20339,20340,20341,20342,20343,20344,20345,20346,20347,20348,20349,20350,20351,20352,20353,20354,20355,20356,20357,20358,20359,20360,20361,20362,20363,20364,20365,20366,20367,20368,20369,20370,20371,20372,20373,20374,20375,20376,20377,20378,20379,20380,20381,20382,20383,20384,20385,20386,20387,20388,20389,20390,20391,20392,20393,20394,20395,20396,20397,20398,20399,20400,20401,20402,20403,20404,20405,20406,20407,20408,20409,20410,20411,20412,20413,20414,20415,20416,20417,20418,20419,20420,20421,20422,20423,20424,20425,20426,20427,20428,20429,20430,20431,20432,20433,20434,20435,20436,20437,20438,20439,20440,20441,20442,20443,20444,20445,20446,20447,20448,20449,20450,20451,20452,20453,20454,20455,20456,20457,20458,20459,20460,20461,20462,20463,20464,20465,20466,20467,20468,20469,20470,20471,20472,20473,20474,20475,20476,20477,20478,20479,20480,20481,20482,20483,20484,20485,20486,20487,20488,20489,20490,20491,20492,20493,20494,20495,20496,20497,20498,20499,20500,20501,20502,20503,20504,20505,20506,20507,20508,20509,20510,20511,20512,20513,20514,20515,20516,20517,20518,20519,20520,20521,20522,20523,20524,20525,20526,20527,20528,20529,20530,20531,20532,20533,20534,20535,20536,20537,20538,20539,20540,20541,20542,20543,20544,20545,20546,20547,20548,20549,20550,20551,20552,20553,20554,20555,20556,20557,20558,20559,20560,20561,20562,20563,20564,20565,20566,20567,20568,20569,20570,20571,20572,20573,20574,20575,20576,20577,20578,20579,20580,20581,20582,20583,20584,20585,20586,20587,20588,20589,20590,20591,20592,20593,20594,20595,20596,20597,20598,20599,20600,20601,20602,20603,20604,20605,20606,20607,20608,20609,20610,20611,20612,20613,20614,20615,20616,20617,20618,20619,20620,20621,20622,20623,20624,20625,20626,20627,20628,20629,20630,20631,20632,20633,20634,20635,20636,20637,20638,20639,20640,20641,20642,20643,20644,20645,20646,20647,20648,20649,20650,20651,20652,20653,20654,20655,20656,20657,20658,20659,20660,20661,20662,20663,20664,20665,20666,20667,20668,20669,20670,20671,20672,20673,20674,20675,20676,20677,20678,20679,20680,20681,20682,20683,20684,20685,20686,20687,20688,20689,20690,20691,20692,20693,20694,20695,20696,20697,20698,20699,20700,20701,20702,20703,20704,20705,20706,20707,20708,20709,20710,20711,20712,20713,20714,20715,20716,20717,20718,20719,20720,20721,20722,20723,20724,20725,20726,20727,20728,20729,20730,20731,20732,20733,20734,20735,20736,20737,20738,20739,20740,20741,20742,20743,20744,20745,20746,20747,20748,20749,20750,20751,20752,20753,20754,20755,20756,20757,20758,20759,20760,20761,20762,20763,20764,20765,20766,20767,20768,20769,20770,20771,20772,20773,20774,20775,20776,20777,20778,20779,20780,20781,20782,20783,20784,20785,20786,20787,20788,20789,20790,20791,20792,20793,20794,20795,20796,20797,20798,20799,20800,20801,20802,20803,20804,20805,20806,20807,20808,20809,20810,20811,20812,20813,20814,20815,20816,20817,20818,20819,20820,20821,20822,20823,20824,20825,20826,20827,20828,20829,20830,20831,20832,20833,20834,20835,20836,20837,20838,20839,20840,20841,20842,20843,20844,20845,20846,20847,20848,20849,20850,20851,20852,20853,20854,20855,20856,20857,20858,20859,20860,20861,20862,20863,20864,20865,20866,20867,20868,20869,20870,20871,20872,20873,20874,20875,20876,20877,20878,20879,20880,20881,20882,20883,20884,20885,20886,20887,20888,20889,20890,20891,20892,20893,20894,20895,20896,20897,20898,20899,20900,20901,20902,20903,20904,20905,20906,20907,20908,20909,20910,20911,20912,20913,20914,20915,20916,20917,20918,20919,20920,20921,20922,20923,20924,20925,20926,20927,20928,20929,20930,20931,20932,20933,20934,20935,20936,20937,20938,20939,20940,20941,20942,20943,20944,20945,20946,20947,20948,20949,20950,20951,20952,20953,20954,20955,20956,20957,20958,20959,20960,20961,20962,20963,20964,20965,20966,20967,20968,20969,20970,20971,20972,20973,20974,20975,20976,20977,20978,20979,20980,20981,20982,20983,20984,20985,20986,20987,20988,20989,20990,20991,20992,20993,20994,20995,20996,20997,20998,20999,21000,21001,21002,21003,21004,21005,21006,21007,21008,21009,21010,21011,21012,21013,21014,21015,21016,21017,21018,21019,21020,21021,21022,21023,21024,21025,21026,21027,21028,21029,21030,21031,21032,21033,21034,21035,21036,21037,21038,21039,21040,21041,21042,21043,21044,21045,21046,21047,21048,21049,21050,21051,21052,21053,21054,21055,21056,21057,21058,21059,21060,21061,21062,21063,21064,21065,21066,21067,21068,21069,21070,21071,21072,21073,21074,21075,21076,21077,21078,21079,21080,21081,21082,21083,21084,21085,21086,21087,21088,21089,21090,21091,21092,21093,21094,21095,21096,21097,21098,21099,21100,21101,21102,21103,21104,21105,21106,21107,21108,21109,21110,21111,21112,21113,21114,21115,21116,21117,21118,21119,21120,21121,21122,21123,21124,21125,21126,21127,21128,21129,21130,21131,21132,21133,21134,21135,21136,21137,21138,21139,21140,21141,21142,21143,21144,21145,21146,21147,21148,21149,21150,21151,21152,21153,21154,21155,21156,21157,21158,21159,21160,21161,21162,21163,21164,21165,21166,21167,21168,21169,21170,21171,21172,21173,21174,21175,21176,21177,21178,21179,21180,21181,21182,21183,21184,21185,21186,21187,21188,21189,21190,21191,21192,21193,21194,21195,21196,21197,21198,21199,21200,21201,21202,21203,21204,21205,21206,21207,21208,21209,21210,21211,21212,21213,21214,21215,21216,21217,21218,21219,21220,21221,21222,21223,21224,21225,21226,21227,21228,21229,21230,21231,21232,21233,21234,21235,21236,21237,21238,21239,21240,21241,21242,21243,21244,21245,21246,21247,21248,21249,21250,21251,21252,21253,21254,21255,21256,21257,21258,21259,21260,21261,21262,21263,21264,21265,21266,21267,21268,21269,21270,21271,21272,21273,21274,21275,21276,21277,21278,21279,21280,21281,21282,21283,21284,21285,21286,21287,21288,21289,21290,21291,21292,21293,21294,21295,21296,21297,21298,21299,21300,21301,21302,21303,21304,21305,21306,21307,21308,21309,21310,21311,21312,21313,21314,21315,21316,21317,21318,21319,21320,21321,21322,21323,21324,21325,21326,21327,21328,21329,21330,21331,21332,21333,21334,21335,21336,21337,21338,21339,21340,21341,21342,21343,21344,21345,21346,21347,21348,21349,21350,21351,21352,21353,21354,21355,21356,21357,21358,21359,21360,21361,21362,21363,21364,21365,21366,21367,21368,21369,21370,21371,21372,21373,21374,21375,21376,21377,21378,21379,21380,21381,21382,21383,21384,21385,21386,21387,21388,21389,21390,21391,21392,21393,21394,21395,21396,21397,21398,21399,21400,21401,21402,21403,21404,21405,21406,21407,21408,21409,21410,21411,21412,21413,21414,21415,21416,21417,21418,21419,21420,21421,21422,21423,21424,21425,21426,21427,21428,21429,21430,21431,21432,21433,21434,21435,21436,21437,21438,21439,21440,21441,21442,21443,21444,21445,21446,21447,21448,21449,21450,21451,21452,21453,21454,21455,21456,21457,21458,21459,21460,21461,21462,21463,21464,21465,21466,21467,21468,21469,21470,21471,21472,21473,21474,21475,21476,21477,21478,21479,21480,21481,21482,21483,21484,21485,21486,21487,21488,21489,21490,21491,21492,21493,21494,21495,21496,21497,21498,21499,21500,21501,21502,21503,21504,21505,21506,21507,21508,21509,21510,21511,21512,21513,21514,21515,21516,21517,21518,21519,21520,21521,21522,21523,21524,21525,21526,21527,21528,21529,21530,21531,21532,21533,21534,21535,21536,21537,21538,21539,21540,21541,21542,21543,21544,21545,21546,21547,21548,21549,21550,21551,21552,21553,21554,21555,21556,21557,21558,21559,21560,21561,21562,21563,21564,21565,21566,21567,21568,21569,21570,21571,21572,21573,21574,21575,21576,21577,21578,21579,21580,21581,21582,21583,21584,21585,21586,21587,21588,21589,21590,21591,21592,21593,21594,21595,21596,21597,21598,21599,21600,21601,21602,21603,21604,21605,21606,21607,21608,21609,21610,21611,21612,21613,21614,21615,21616,21617,21618,21619,21620,21621,21622,21623,21624,21625,21626,21627,21628,21629,21630,21631,21632,21633,21634,21635,21636,21637,21638,21639,21640,21641,21642,21643,21644,21645,21646,21647,21648,21649,21650,21651,21652,21653,21654,21655,21656,21657,21658,21659,21660,21661,21662,21663,21664,21665,21666,21667,21668,21669,21670,21671,21672,21673,21674,21675,21676,21677,21678,21679,21680,21681,21682,21683,21684,21685,21686,21687,21688,21689,21690,21691,21692,21693,21694,21695,21696,21697,21698,21699,21700,21701,21702,21703,21704,21705,21706,21707,21708,21709,21710,21711,21712,21713,21714,21715,21716,21717,21718,21719,21720,21721,21722,21723,21724,21725,21726,21727,21728,21729,21730,21731,21732,21733,21734,21735,21736,21737,21738,21739,21740,21741,21742,21743,21744,21745,21746,21747,21748,21749,21750,21751,21752,21753,21754,21755,21756,21757,21758,21759,21760,21761,21762,21763,21764,21765,21766,21767,21768,21769,21770,21771,21772,21773,21774,21775,21776,21777,21778,21779,21780,21781,21782,21783,21784,21785,21786,21787,21788,21789,21790,21791,21792,21793,21794,21795,21796,21797,21798,21799,21800,21801,21802,21803,21804,21805,21806,21807,21808,21809,21810,21811,21812,21813,21814,21815,21816,21817,21818,21819,21820,21821,21822,21823,21824,21825,21826,21827,21828,21829,21830,21831,21832,21833,21834,21835,21836,21837,21838,21839,21840,21841,21842,21843,21844,21845,21846,21847,21848,21849,21850,21851,21852,21853,21854,21855,21856,21857,21858,21859,21860,21861,21862,21863,21864,21865,21866,21867,21868,21869,21870,21871,21872,21873,21874,21875,21876,21877,21878,21879,21880,21881,21882,21883,21884,21885,21886,21887,21888,21889,21890,21891,21892,21893,21894,21895,21896,21897,21898,21899,21900,21901,21902,21903,21904,21905,21906,21907,21908,21909,21910,21911,21912,21913,21914,21915,21916,21917,21918,21919,21920,21921,21922,21923,21924,21925,21926,21927,21928,21929,21930,21931,21932,21933,21934,21935,21936,21937,21938,21939,21940,21941,21942,21943,21944,21945,21946,21947,21948,21949,21950,21951,21952,21953,21954,21955,21956,21957,21958,21959,21960,21961,21962,21963,21964,21965,21966,21967,21968,21969,21970,21971,21972,21973,21974,21975,21976,21977,21978,21979,21980,21981,21982,21983,21984,21985,21986,21987,21988,21989,21990,21991,21992,21993,21994,21995,21996,21997,21998,21999,22000,22001,22002,22003,22004,22005,22006,22007,22008,22009,22010,22011,22012,22013,22014,22015,22016,22017,22018,22019,22020,22021,22022,22023,22024,22025,22026,22027,22028,22029,22030,22031,22032,22033,22034,22035,22036,22037,22038,22039,22040,22041,22042,22043,22044,22045,22046,22047,22048,22049,22050,22051,22052,22053,22054,22055,22056,22057,22058,22059,22060,22061,22062,22063,22064,22065,22066,22067,22068,22069,22070,22071,22072,22073,22074,22075,22076,22077,22078,22079,22080,22081,22082,22083,22084,22085,22086,22087,22088,22089,22090,22091,22092,22093,22094,22095,22096,22097,22098,22099,22100,22101,22102,22103,22104,22105,22106,22107,22108,22109,22110,22111,22112,22113,22114,22115,22116,22117,22118,22119,22120,22121,22122,22123,22124,22125,22126,22127,22128,22129,22130,22131,22132,22133,22134,22135,22136,22137,22138,22139,22140,22141,22142,22143,22144,22145,22146,22147,22148,22149,22150,22151,22152,22153,22154,22155,22156,22157,22158,22159,22160,22161,22162,22163,22164,22165,22166,22167,22168,22169,22170,22171,22172,22173,22174,22175,22176,22177,22178,22179,22180,22181,22182,22183,22184,22185,22186,22187,22188,22189,22190,22191,22192,22193,22194,22195,22196,22197,22198,22199,22200,22201,22202,22203,22204,22205,22206,22207,22208,22209,22210,22211,22212,22213,22214,22215,22216,22217,22218,22219,22220,22221,22222,22223,22224,22225,22226,22227,22228,22229,22230,22231,22232,22233,22234,22235,22236,22237,22238,22239,22240,22241,22242,22243,22244,22245,22246,22247,22248,22249,22250,22251,22252,22253,22254,22255,22256,22257,22258,22259,22260,22261,22262,22263,22264,22265,22266,22267,22268,22269,22270,22271,22272,22273,22274,22275,22276,22277,22278,22279,22280,22281,22282,22283,22284,22285,22286,22287,22288,22289,22290,22291,22292,22293,22294,22295,22296,22297,22298,22299,22300,22301,22302,22303,22304,22305,22306,22307,22308,22309,22310,22311,22312,22313,22314,22315,22316,22317,22318,22319,22320,22321,22322,22323,22324,22325,22326,22327,22328,22329,22330,22331,22332,22333,22334,22335,22336,22337,22338,22339,22340,22341,22342,22343,22344,22345,22346,22347,22348,22349,22350,22351,22352,22353,22354,22355,22356,22357,22358,22359,22360,22361,22362,22363,22364,22365,22366,22367,22368,22369,22370,22371,22372,22373,22374,22375,22376,22377,22378,22379,22380,22381,22382,22383,22384,22385,22386,22387,22388,22389,22390,22391,22392,22393,22394,22395,22396,22397,22398,22399,22400,22401,22402,22403,22404,22405,22406,22407,22408,22409,22410,22411,22412,22413,22414,22415,22416,22417,22418,22419,22420,22421,22422,22423,22424,22425,22426,22427,22428,22429,22430,22431,22432,22433,22434,22435,22436,22437,22438,22439,22440,22441,22442,22443,22444,22445,22446,22447,22448,22449,22450,22451,22452,22453,22454,22455,22456,22457,22458,22459,22460,22461,22462,22463,22464,22465,22466,22467,22468,22469,22470,22471,22472,22473,22474,22475,22476,22477,22478,22479,22480,22481,22482,22483,22484,22485,22486,22487,22488,22489,22490,22491,22492,22493,22494,22495,22496,22497,22498,22499,22500,22501,22502,22503,22504,22505,22506,22507,22508,22509,22510,22511,22512,22513,22514,22515,22516,22517,22518,22519,22520,22521,22522,22523,22524,22525,22526,22527,22528,22529,22530,22531,22532,22533,22534,22535,22536,22537,22538,22539,22540,22541,22542,22543,22544,22545,22546,22547,22548,22549,22550,22551,22552,22553,22554,22555,22556,22557,22558,22559,22560,22561,22562,22563,22564,22565,22566,22567,22568,22569,22570,22571,22572,22573,22574,22575,22576,22577,22578,22579,22580,22581,22582,22583,22584,22585,22586,22587,22588,22589,22590,22591,22592,22593,22594,22595,22596,22597,22598,22599,22600,22601,22602,22603,22604,22605,22606,22607,22608,22609,22610,22611,22612,22613,22614,22615,22616,22617,22618,22619,22620,22621,22622,22623,22624,22625,22626,22627,22628,22629,22630,22631,22632,22633,22634,22635,22636,22637,22638,22639,22640,22641,22642,22643,22644,22645,22646,22647,22648,22649,22650,22651,22652,22653,22654,22655,22656,22657,22658,22659,22660,22661,22662,22663,22664,22665,22666,22667,22668,22669,22670,22671,22672,22673,22674,22675,22676,22677,22678,22679,22680,22681,22682,22683,22684,22685,22686,22687,22688,22689,22690,22691,22692,22693,22694,22695,22696,22697,22698,22699,22700,22701,22702,22703,22704,22705,22706,22707,22708,22709,22710,22711,22712,22713,22714,22715,22716,22717,22718,22719,22720,22721,22722,22723,22724,22725,22726,22727,22728,22729,22730,22731,22732,22733,22734,22735,22736,22737,22738,22739,22740,22741,22742,22743,22744,22745,22746,22747,22748,22749,22750,22751,22752,22753,22754,22755,22756,22757,22758,22759,22760,22761,22762,22763,22764,22765,22766,22767,22768,22769,22770,22771,22772,22773,22774,22775,22776,22777,22778,22779,22780,22781,22782,22783,22784,22785,22786,22787,22788,22789,22790,22791,22792,22793,22794,22795,22796,22797,22798,22799,22800,22801,22802,22803,22804,22805,22806,22807,22808,22809,22810,22811,22812,22813,22814,22815,22816,22817,22818,22819,22820,22821,22822,22823,22824,22825,22826,22827,22828,22829,22830,22831,22832,22833,22834,22835,22836,22837,22838,22839,22840,22841,22842,22843,22844,22845,22846,22847,22848,22849,22850,22851,22852,22853,22854,22855,22856,22857,22858,22859,22860,22861,22862,22863,22864,22865,22866,22867,22868,22869,22870,22871,22872,22873,22874,22875,22876,22877,22878,22879,22880,22881,22882,22883,22884,22885,22886,22887,22888,22889,22890,22891,22892,22893,22894,22895,22896,22897,22898,22899,22900,22901,22902,22903,22904,22905,22906,22907,22908,22909,22910,22911,22912,22913,22914,22915,22916,22917,22918,22919,22920,22921,22922,22923,22924,22925,22926,22927,22928,22929,22930,22931,22932,22933,22934,22935,22936,22937,22938,22939,22940,22941,22942,22943,22944,22945,22946,22947,22948,22949,22950,22951,22952,22953,22954,22955,22956,22957,22958,22959,22960,22961,22962,22963,22964,22965,22966,22967,22968,22969,22970,22971,22972,22973,22974,22975,22976,22977,22978,22979,22980,22981,22982,22983,22984,22985,22986,22987,22988,22989,22990,22991,22992,22993,22994,22995,22996,22997,22998,22999,23000,23001,23002,23003,23004,23005,23006,23007,23008,23009,23010,23011,23012,23013,23014,23015,23016,23017,23018,23019,23020,23021,23022,23023,23024,23025,23026,23027,23028,23029,23030,23031,23032,23033,23034,23035,23036,23037,23038,23039,23040,23041,23042,23043,23044,23045,23046,23047,23048,23049,23050,23051,23052,23053,23054,23055,23056,23057,23058,23059,23060,23061,23062,23063,23064,23065,23066,23067,23068,23069,23070,23071,23072,23073,23074,23075,23076,23077,23078,23079,23080,23081,23082,23083,23084,23085,23086,23087,23088,23089,23090,23091,23092,23093,23094,23095,23096,23097,23098,23099,23100,23101,23102,23103,23104,23105,23106,23107,23108,23109,23110,23111,23112,23113,23114,23115,23116,23117,23118,23119,23120,23121,23122,23123,23124,23125,23126,23127,23128,23129,23130,23131,23132,23133,23134,23135,23136,23137,23138,23139,23140,23141,23142,23143,23144,23145,23146,23147,23148,23149,23150,23151,23152,23153,23154,23155,23156,23157,23158,23159,23160,23161,23162,23163,23164,23165,23166,23167,23168,23169,23170,23171,23172,23173,23174,23175,23176,23177,23178,23179,23180,23181,23182,23183,23184,23185,23186,23187,23188,23189,23190,23191,23192,23193,23194,23195,23196,23197,23198,23199,23200,23201,23202,23203,23204,23205,23206,23207,23208,23209,23210,23211,23212,23213,23214,23215,23216,23217,23218,23219,23220,23221,23222,23223,23224,23225,23226,23227,23228,23229,23230,23231,23232,23233,23234,23235,23236,23237,23238,23239,23240,23241,23242,23243,23244,23245,23246,23247,23248,23249,23250,23251,23252,23253,23254,23255,23256,23257,23258,23259,23260,23261,23262,23263,23264,23265,23266,23267,23268,23269,23270,23271,23272,23273,23274,23275,23276,23277,23278,23279,23280,23281,23282,23283,23284,23285,23286,23287,23288,23289,23290,23291,23292,23293,23294,23295,23296,23297,23298,23299,23300,23301,23302,23303,23304,23305,23306,23307,23308,23309,23310,23311,23312,23313,23314,23315,23316,23317,23318,23319,23320,23321,23322,23323,23324,23325,23326,23327,23328,23329,23330,23331,23332,23333,23334,23335,23336,23337,23338,23339,23340,23341,23342,23343,23344,23345,23346,23347,23348,23349,23350,23351,23352,23353,23354,23355,23356,23357,23358,23359,23360,23361,23362,23363,23364,23365,23366,23367,23368,23369,23370,23371,23372,23373,23374,23375,23376,23377,23378,23379,23380,23381,23382,23383,23384,23385,23386,23387,23388,23389,23390,23391,23392,23393,23394,23395,23396,23397,23398,23399,23400,23401,23402,23403,23404,23405,23406,23407,23408,23409,23410,23411,23412,23413,23414,23415,23416,23417,23418,23419,23420,23421,23422,23423,23424,23425,23426,23427,23428,23429,23430,23431,23432,23433,23434,23435,23436,23437,23438,23439,23440,23441,23442,23443,23444,23445,23446,23447,23448,23449,23450,23451,23452,23453,23454,23455,23456,23457,23458,23459,23460,23461,23462,23463,23464,23465,23466,23467,23468,23469,23470,23471,23472,23473,23474,23475,23476,23477,23478,23479,23480,23481,23482,23483,23484,23485,23486,23487,23488,23489,23490,23491,23492,23493,23494,23495,23496,23497,23498,23499,23500,23501,23502,23503,23504,23505,23506,23507,23508,23509,23510,23511,23512,23513,23514,23515,23516,23517,23518,23519,23520,23521,23522,23523,23524,23525,23526,23527,23528,23529,23530,23531,23532,23533,23534,23535,23536,23537,23538,23539,23540,23541,23542,23543,23544,23545,23546,23547,23548,23549,23550,23551,23552,23553,23554,23555,23556,23557,23558,23559,23560,23561,23562,23563,23564,23565,23566,23567,23568,23569,23570,23571,23572,23573,23574,23575,23576,23577,23578,23579,23580,23581,23582,23583,23584,23585,23586,23587,23588,23589,23590,23591,23592,23593,23594,23595,23596,23597,23598,23599,23600,23601,23602,23603,23604,23605,23606,23607,23608,23609,23610,23611,23612,23613,23614,23615,23616,23617,23618,23619,23620,23621,23622,23623,23624,23625,23626,23627,23628,23629,23630,23631,23632,23633,23634,23635,23636,23637,23638,23639,23640,23641,23642,23643,23644,23645,23646,23647,23648,23649,23650,23651,23652,23653,23654,23655,23656,23657,23658,23659,23660,23661,23662,23663,23664,23665,23666,23667,23668,23669,23670,23671,23672,23673,23674,23675,23676,23677,23678,23679,23680,23681,23682,23683,23684,23685,23686,23687,23688,23689,23690,23691,23692,23693,23694,23695,23696,23697,23698,23699,23700,23701,23702,23703,23704,23705,23706,23707,23708,23709,23710,23711,23712,23713,23714,23715,23716,23717,23718,23719,23720,23721,23722,23723,23724,23725,23726,23727,23728,23729,23730,23731,23732,23733,23734,23735,23736,23737,23738,23739,23740,23741,23742,23743,23744,23745,23746,23747,23748,23749,23750,23751,23752,23753,23754,23755,23756,23757,23758,23759,23760,23761,23762,23763,23764,23765,23766,23767,23768,23769,23770,23771,23772,23773,23774,23775,23776,23777,23778,23779,23780,23781,23782,23783,23784,23785,23786,23787,23788,23789,23790,23791,23792,23793,23794,23795,23796,23797,23798,23799,23800,23801,23802,23803,23804,23805,23806,23807,23808,23809,23810,23811,23812,23813,23814,23815,23816,23817,23818,23819,23820,23821,23822,23823,23824,23825,23826,23827,23828,23829,23830,23831,23832,23833,23834,23835,23836,23837,23838,23839,23840,23841,23842,23843,23844,23845,23846,23847,23848,23849,23850,23851,23852,23853,23854,23855,23856,23857,23858,23859,23860,23861,23862,23863,23864,23865,23866,23867,23868,23869,23870,23871,23872,23873,23874,23875,23876,23877,23878,23879,23880,23881,23882,23883,23884,23885,23886,23887,23888,23889,23890,23891,23892,23893,23894,23895,23896,23897,23898,23899,23900,23901,23902,23903,23904,23905,23906,23907,23908,23909,23910,23911,23912,23913,23914,23915,23916,23917,23918,23919,23920,23921,23922,23923,23924,23925,23926,23927,23928,23929,23930,23931,23932,23933,23934,23935,23936,23937,23938,23939,23940,23941,23942,23943,23944,23945,23946,23947,23948,23949,23950,23951,23952,23953,23954,23955,23956,23957,23958,23959,23960,23961,23962,23963,23964,23965,23966,23967,23968,23969,23970,23971,23972,23973,23974,23975,23976,23977,23978,23979,23980,23981,23982,23983,23984,23985,23986,23987,23988,23989,23990,23991,23992,23993,23994,23995,23996,23997,23998,23999,24000,24001,24002,24003,24004,24005,24006,24007,24008,24009,24010,24011,24012,24013,24014,24015,24016,24017,24018,24019,24020,24021,24022,24023,24024,24025,24026,24027,24028,24029,24030,24031,24032,24033,24034,24035,24036,24037,24038,24039,24040,24041,24042,24043,24044,24045,24046,24047,24048,24049,24050,24051,24052,24053,24054,24055,24056,24057,24058,24059,24060,24061,24062,24063,24064,24065,24066,24067,24068,24069,24070,24071,24072,24073,24074,24075,24076,24077,24078,24079,24080,24081,24082,24083,24084,24085,24086,24087,24088,24089,24090,24091,24092,24093,24094,24095,24096,24097,24098,24099,24100,24101,24102,24103,24104,24105,24106,24107,24108,24109,24110,24111,24112,24113,24114,24115,24116,24117,24118,24119,24120,24121,24122,24123,24124,24125,24126,24127,24128,24129,24130,24131,24132,24133,24134,24135,24136,24137,24138,24139,24140,24141,24142,24143,24144,24145,24146,24147,24148,24149,24150,24151,24152,24153,24154,24155,24156,24157,24158,24159,24160,24161,24162,24163,24164,24165,24166,24167,24168,24169,24170,24171,24172,24173,24174,24175,24176,24177,24178,24179,24180,24181,24182,24183,24184,24185,24186,24187,24188,24189,24190,24191,24192,24193,24194,24195,24196,24197,24198,24199,24200,24201,24202,24203,24204,24205,24206,24207,24208,24209,24210,24211,24212,24213,24214,24215,24216,24217,24218,24219,24220,24221,24222,24223,24224,24225,24226,24227,24228,24229,24230,24231,24232,24233,24234,24235,24236,24237,24238,24239,24240,24241,24242,24243,24244,24245,24246,24247,24248,24249,24250,24251,24252,24253,24254,24255,24256,24257,24258,24259,24260,24261,24262,24263,24264,24265,24266,24267,24268,24269,24270,24271,24272,24273,24274,24275,24276,24277,24278,24279,24280,24281,24282,24283,24284,24285,24286,24287,24288,24289,24290,24291,24292,24293,24294,24295,24296,24297,24298,24299,24300,24301,24302,24303,24304,24305,24306,24307,24308,24309,24310,24311,24312,24313,24314,24315,24316,24317,24318,24319,24320,24321,24322,24323,24324,24325,24326,24327,24328,24329,24330,24331,24332,24333,24334,24335,24336,24337,24338,24339,24340,24341,24342,24343,24344,24345,24346,24347,24348,24349,24350,24351,24352,24353,24354,24355,24356,24357,24358,24359,24360,24361,24362,24363,24364,24365,24366,24367,24368,24369,24370,24371,24372,24373,24374,24375,24376,24377,24378,24379,24380,24381,24382,24383,24384,24385,24386,24387,24388,24389,24390,24391,24392,24393,24394,24395,24396,24397,24398,24399,24400,24401,24402,24403,24404,24405,24406,24407,24408,24409,24410,24411,24412,24413,24414,24415,24416,24417,24418,24419,24420,24421,24422,24423,24424,24425,24426,24427,24428,24429,24430,24431,24432,24433,24434,24435,24436,24437,24438,24439,24440,24441,24442,24443,24444,24445,24446,24447,24448,24449,24450,24451,24452,24453,24454,24455,24456,24457,24458,24459,24460,24461,24462,24463,24464,24465,24466,24467,24468,24469,24470,24471,24472,24473,24474,24475,24476,24477,24478,24479,24480,24481,24482,24483,24484,24485,24486,24487,24488,24489,24490,24491,24492,24493,24494,24495,24496,24497,24498,24499,24500,24501,24502,24503,24504,24505,24506,24507,24508,24509,24510,24511,24512,24513,24514,24515,24516,24517,24518,24519,24520,24521,24522,24523,24524,24525,24526,24527,24528,24529,24530,24531,24532,24533,24534,24535,24536,24537,24538,24539,24540,24541,24542,24543,24544,24545,24546,24547,24548,24549,24550,24551,24552,24553,24554,24555,24556,24557,24558,24559,24560,24561,24562,24563,24564,24565,24566,24567,24568,24569,24570,24571,24572,24573,24574,24575,24576,24577,24578,24579,24580,24581,24582,24583,24584,24585,24586,24587,24588,24589,24590,24591,24592,24593,24594,24595,24596,24597,24598,24599,24600,24601,24602,24603,24604,24605,24606,24607,24608,24609,24610,24611,24612,24613,24614,24615,24616,24617,24618,24619,24620,24621,24622,24623,24624,24625,24626,24627,24628,24629,24630,24631,24632,24633,24634,24635,24636,24637,24638,24639,24640,24641,24642,24643,24644,24645,24646,24647,24648,24649,24650,24651,24652,24653,24654,24655,24656,24657,24658,24659,24660,24661,24662,24663,24664,24665,24666,24667,24668,24669,24670,24671,24672,24673,24674,24675,24676,24677,24678,24679,24680,24681,24682,24683,24684,24685,24686,24687,24688,24689,24690,24691,24692,24693,24694,24695,24696,24697,24698,24699,24700,24701,24702,24703,24704,24705,24706,24707,24708,24709,24710,24711,24712,24713,24714,24715,24716,24717,24718,24719,24720,24721,24722,24723,24724,24725,24726,24727,24728,24729,24730,24731,24732,24733,24734,24735,24736,24737,24738,24739,24740,24741,24742,24743,24744,24745,24746,24747,24748,24749,24750,24751,24752,24753,24754,24755,24756,24757,24758,24759,24760,24761,24762,24763,24764,24765,24766,24767,24768,24769,24770,24771,24772,24773,24774,24775,24776,24777,24778,24779,24780,24781,24782,24783,24784,24785,24786,24787,24788,24789,24790,24791,24792,24793,24794,24795,24796,24797,24798,24799,24800,24801,24802,24803,24804,24805,24806,24807,24808,24809,24810,24811,24812,24813,24814,24815,24816,24817,24818,24819,24820,24821,24822,24823,24824,24825,24826,24827,24828,24829,24830,24831,24832,24833,24834,24835,24836,24837,24838,24839,24840,24841,24842,24843,24844,24845,24846,24847,24848,24849,24850,24851,24852,24853,24854,24855,24856,24857,24858,24859,24860,24861,24862,24863,24864,24865,24866,24867,24868,24869,24870,24871,24872,24873,24874,24875,24876,24877,24878,24879,24880,24881,24882,24883,24884,24885,24886,24887,24888,24889,24890,24891,24892,24893,24894,24895,24896,24897,24898,24899,24900,24901,24902,24903,24904,24905,24906,24907,24908,24909,24910,24911,24912,24913,24914,24915,24916,24917,24918,24919,24920,24921,24922,24923,24924,24925,24926,24927,24928,24929,24930,24931,24932,24933,24934,24935,24936,24937,24938,24939,24940,24941,24942,24943,24944,24945,24946,24947,24948,24949,24950,24951,24952,24953,24954,24955,24956,24957,24958,24959,24960,24961,24962,24963,24964,24965,24966,24967,24968,24969,24970,24971,24972,24973,24974,24975,24976,24977,24978,24979,24980,24981,24982,24983,24984,24985,24986,24987,24988,24989,24990,24991,24992,24993,24994,24995,24996,24997,24998,24999,25000,25001,25002,25003,25004,25005,25006,25007,25008,25009,25010,25011,25012,25013,25014,25015,25016,25017,25018,25019,25020,25021,25022,25023,25024,25025,25026,25027,25028,25029,25030,25031,25032,25033,25034,25035,25036,25037,25038,25039,25040,25041,25042,25043,25044,25045,25046,25047,25048,25049,25050,25051,25052,25053,25054,25055,25056,25057,25058,25059,25060,25061,25062,25063,25064,25065,25066,25067,25068,25069,25070,25071,25072,25073,25074,25075,25076,25077,25078,25079,25080,25081,25082,25083,25084,25085,25086,25087,25088,25089,25090,25091,25092,25093,25094,25095,25096,25097,25098,25099,25100,25101,25102,25103,25104,25105,25106,25107,25108,25109,25110,25111,25112,25113,25114,25115,25116,25117,25118,25119,25120,25121,25122,25123,25124,25125,25126,25127,25128,25129,25130,25131,25132,25133,25134,25135,25136,25137,25138,25139,25140,25141,25142,25143,25144,25145,25146,25147,25148,25149,25150,25151,25152,25153,25154,25155,25156,25157,25158,25159,25160,25161,25162,25163,25164,25165,25166,25167,25168,25169,25170,25171,25172,25173,25174,25175,25176,25177,25178,25179,25180,25181,25182,25183,25184,25185,25186,25187,25188,25189,25190,25191,25192,25193,25194,25195,25196,25197,25198,25199,25200,25201,25202,25203,25204,25205,25206,25207,25208,25209,25210,25211,25212,25213,25214,25215,25216,25217,25218,25219,25220,25221,25222,25223,25224,25225,25226,25227,25228,25229,25230,25231,25232,25233,25234,25235,25236,25237,25238,25239,25240,25241,25242,25243,25244,25245,25246,25247,25248,25249,25250,25251,25252,25253,25254,25255,25256,25257,25258,25259,25260,25261,25262,25263,25264,25265,25266,25267,25268,25269,25270,25271,25272,25273,25274,25275,25276,25277,25278,25279,25280,25281,25282,25283,25284,25285,25286,25287,25288,25289,25290,25291,25292,25293,25294,25295,25296,25297,25298,25299,25300,25301,25302,25303,25304,25305,25306,25307,25308,25309,25310,25311,25312,25313,25314,25315,25316,25317,25318,25319,25320,25321,25322,25323,25324,25325,25326,25327,25328,25329,25330,25331,25332,25333,25334,25335,25336,25337,25338,25339,25340,25341,25342,25343,25344,25345,25346,25347,25348,25349,25350,25351,25352,25353,25354,25355,25356,25357,25358,25359,25360,25361,25362,25363,25364,25365,25366,25367,25368,25369,25370,25371,25372,25373,25374,25375,25376,25377,25378,25379,25380,25381,25382,25383,25384,25385,25386,25387,25388,25389,25390,25391,25392,25393,25394,25395,25396,25397,25398,25399,25400,25401,25402,25403,25404,25405,25406,25407,25408,25409,25410,25411,25412,25413,25414,25415,25416,25417,25418,25419,25420,25421,25422,25423,25424,25425,25426,25427,25428,25429,25430,25431,25432,25433,25434,25435,25436,25437,25438,25439,25440,25441,25442,25443,25444,25445,25446,25447,25448,25449,25450,25451,25452,25453,25454,25455,25456,25457,25458,25459,25460,25461,25462,25463,25464,25465,25466,25467,25468,25469,25470,25471,25472,25473,25474,25475,25476,25477,25478,25479,25480,25481,25482,25483,25484,25485,25486,25487,25488,25489,25490,25491,25492,25493,25494,25495,25496,25497,25498,25499,25500,25501,25502,25503,25504,25505,25506,25507,25508,25509,25510,25511,25512,25513,25514,25515,25516,25517,25518,25519,25520,25521,25522,25523,25524,25525,25526,25527,25528,25529,25530,25531,25532,25533,25534,25535,25536,25537,25538,25539,25540,25541,25542,25543,25544,25545,25546,25547,25548,25549,25550,25551,25552,25553,25554,25555,25556,25557,25558,25559,25560,25561,25562,25563,25564,25565,25566,25567,25568,25569,25570,25571,25572,25573,25574,25575,25576,25577,25578,25579,25580,25581,25582,25583,25584,25585,25586,25587,25588,25589,25590,25591,25592,25593,25594,25595,25596,25597,25598,25599,25600,25601,25602,25603,25604,25605,25606,25607,25608,25609,25610,25611,25612,25613,25614,25615,25616,25617,25618,25619,25620,25621,25622,25623,25624,25625,25626,25627,25628,25629,25630,25631,25632,25633,25634,25635,25636,25637,25638,25639,25640,25641,25642,25643,25644,25645,25646,25647,25648,25649,25650,25651,25652,25653,25654,25655,25656,25657,25658,25659,25660,25661,25662,25663,25664,25665,25666,25667,25668,25669,25670,25671,25672,25673,25674,25675,25676,25677,25678,25679,25680,25681,25682,25683,25684,25685,25686,25687,25688,25689,25690,25691,25692,25693,25694,25695,25696,25697,25698,25699,25700,25701,25702,25703,25704,25705,25706,25707,25708,25709,25710,25711,25712,25713,25714,25715,25716,25717,25718,25719,25720,25721,25722,25723,25724,25725,25726,25727,25728,25729,25730,25731,25732,25733,25734,25735,25736,25737,25738,25739,25740,25741,25742,25743,25744,25745,25746,25747,25748,25749,25750,25751,25752,25753,25754,25755,25756,25757,25758,25759,25760,25761,25762,25763,25764,25765,25766,25767,25768,25769,25770,25771,25772,25773,25774,25775,25776,25777,25778,25779,25780,25781,25782,25783,25784,25785,25786,25787,25788,25789,25790,25791,25792,25793,25794,25795,25796,25797,25798,25799,25800,25801,25802,25803,25804,25805,25806,25807,25808,25809,25810,25811,25812,25813,25814,25815,25816,25817,25818,25819,25820,25821,25822,25823,25824,25825,25826,25827,25828,25829,25830,25831,25832,25833,25834,25835,25836,25837,25838,25839,25840,25841,25842,25843,25844,25845,25846,25847,25848,25849,25850,25851,25852,25853,25854,25855,25856,25857,25858,25859,25860,25861,25862,25863,25864,25865,25866,25867,25868,25869,25870,25871,25872,25873,25874,25875,25876,25877,25878,25879,25880,25881,25882,25883,25884,25885,25886,25887,25888,25889,25890,25891,25892,25893,25894,25895,25896,25897,25898,25899,25900,25901,25902,25903,25904,25905,25906,25907,25908,25909,25910,25911,25912,25913,25914,25915,25916,25917,25918,25919,25920,25921,25922,25923,25924,25925,25926,25927,25928,25929,25930,25931,25932,25933,25934,25935,25936,25937,25938,25939,25940,25941,25942,25943,25944,25945,25946,25947,25948,25949,25950,25951,25952,25953,25954,25955,25956,25957,25958,25959,25960,25961,25962,25963,25964,25965,25966,25967,25968,25969,25970,25971,25972,25973,25974,25975,25976,25977,25978,25979,25980,25981,25982,25983,25984,25985,25986,25987,25988,25989,25990,25991,25992,25993,25994,25995,25996,25997,25998,25999,26000,26001,26002,26003,26004,26005,26006,26007,26008,26009,26010,26011,26012,26013,26014,26015,26016,26017,26018,26019,26020,26021,26022,26023,26024,26025,26026,26027,26028,26029,26030,26031,26032,26033,26034,26035,26036,26037,26038,26039,26040,26041,26042,26043,26044,26045,26046,26047,26048,26049,26050,26051,26052,26053,26054,26055,26056,26057,26058,26059,26060,26061,26062,26063,26064,26065,26066,26067,26068,26069,26070,26071,26072,26073,26074,26075,26076,26077,26078,26079,26080,26081,26082,26083,26084,26085,26086,26087,26088,26089,26090,26091,26092,26093,26094,26095,26096,26097,26098,26099,26100,26101,26102,26103,26104,26105,26106,26107,26108,26109,26110,26111,26112,26113,26114,26115,26116,26117,26118,26119,26120,26121,26122,26123,26124,26125,26126,26127,26128,26129,26130,26131,26132,26133,26134,26135,26136,26137,26138,26139,26140,26141,26142,26143,26144,26145,26146,26147,26148,26149,26150,26151,26152,26153,26154,26155,26156,26157,26158,26159,26160,26161,26162,26163,26164,26165,26166,26167,26168,26169,26170,26171,26172,26173,26174,26175,26176,26177,26178,26179,26180,26181,26182,26183,26184,26185,26186,26187,26188,26189,26190,26191,26192,26193,26194,26195,26196,26197,26198,26199,26200,26201,26202,26203,26204,26205,26206,26207,26208,26209,26210,26211,26212,26213,26214,26215,26216,26217,26218,26219,26220,26221,26222,26223,26224,26225,26226,26227,26228,26229,26230,26231,26232,26233,26234,26235,26236,26237,26238,26239,26240,26241,26242,26243,26244,26245,26246,26247,26248,26249,26250,26251,26252,26253,26254,26255,26256,26257,26258,26259,26260,26261,26262,26263,26264,26265,26266,26267,26268,26269,26270,26271,26272,26273,26274,26275,26276,26277,26278,26279,26280,26281,26282,26283,26284,26285,26286,26287,26288,26289,26290,26291,26292,26293,26294,26295,26296,26297,26298,26299,26300,26301,26302,26303,26304,26305,26306,26307,26308,26309,26310,26311,26312,26313,26314,26315,26316,26317,26318,26319,26320,26321,26322,26323,26324,26325,26326,26327,26328,26329,26330,26331,26332,26333,26334,26335,26336,26337,26338,26339,26340,26341,26342,26343,26344,26345,26346,26347,26348,26349,26350,26351,26352,26353,26354,26355,26356,26357,26358,26359,26360,26361,26362,26363,26364,26365,26366,26367,26368,26369,26370,26371,26372,26373,26374,26375,26376,26377,26378,26379,26380,26381,26382,26383,26384,26385,26386,26387,26388,26389,26390,26391,26392,26393,26394,26395,26396,26397,26398,26399,26400,26401,26402,26403,26404,26405,26406,26407,26408,26409,26410,26411,26412,26413,26414,26415,26416,26417,26418,26419,26420,26421,26422,26423,26424,26425,26426,26427,26428,26429,26430,26431,26432,26433,26434,26435,26436,26437,26438,26439,26440,26441,26442,26443,26444,26445,26446,26447,26448,26449,26450,26451,26452,26453,26454,26455,26456,26457,26458,26459,26460,26461,26462,26463,26464,26465,26466,26467,26468,26469,26470,26471,26472,26473,26474,26475,26476,26477,26478,26479,26480,26481,26482,26483,26484,26485,26486,26487,26488,26489,26490,26491,26492,26493,26494,26495,26496,26497,26498,26499,26500,26501,26502,26503,26504,26505,26506,26507,26508,26509,26510,26511,26512,26513,26514,26515,26516,26517,26518,26519,26520,26521,26522,26523,26524,26525,26526,26527,26528,26529,26530,26531,26532,26533,26534,26535,26536,26537,26538,26539,26540,26541,26542,26543,26544,26545,26546,26547,26548,26549,26550,26551,26552,26553,26554,26555,26556,26557,26558,26559,26560,26561,26562,26563,26564,26565,26566,26567,26568,26569,26570,26571,26572,26573,26574,26575,26576,26577,26578,26579,26580,26581,26582,26583,26584,26585,26586,26587,26588,26589,26590,26591,26592,26593,26594,26595,26596,26597,26598,26599,26600,26601,26602,26603,26604,26605,26606,26607,26608,26609,26610,26611,26612,26613,26614,26615,26616,26617,26618,26619,26620,26621,26622,26623,26624,26625,26626,26627,26628,26629,26630,26631,26632,26633,26634,26635,26636,26637,26638,26639,26640,26641,26642,26643,26644,26645,26646,26647,26648,26649,26650,26651,26652,26653,26654,26655,26656,26657,26658,26659,26660,26661,26662,26663,26664,26665,26666,26667,26668,26669,26670,26671,26672,26673,26674,26675,26676,26677,26678,26679,26680,26681,26682,26683,26684,26685,26686,26687,26688,26689,26690,26691,26692,26693,26694,26695,26696,26697,26698,26699,26700,26701,26702,26703,26704,26705,26706,26707,26708,26709,26710,26711,26712,26713,26714,26715,26716,26717,26718,26719,26720,26721,26722,26723,26724,26725,26726,26727,26728,26729,26730,26731,26732,26733,26734,26735,26736,26737,26738,26739,26740,26741,26742,26743,26744,26745,26746,26747,26748,26749,26750,26751,26752,26753,26754,26755,26756,26757,26758,26759,26760,26761,26762,26763,26764,26765,26766,26767,26768,26769,26770,26771,26772,26773,26774,26775,26776,26777,26778,26779,26780,26781,26782,26783,26784,26785,26786,26787,26788,26789,26790,26791,26792,26793,26794,26795,26796,26797,26798,26799,26800,26801,26802,26803,26804,26805,26806,26807,26808,26809,26810,26811,26812,26813,26814,26815,26816,26817,26818,26819,26820,26821,26822,26823,26824,26825,26826,26827,26828,26829,26830,26831,26832,26833,26834,26835,26836,26837,26838,26839,26840,26841,26842,26843,26844,26845,26846,26847,26848,26849,26850,26851,26852,26853,26854,26855,26856,26857,26858,26859,26860,26861,26862,26863,26864,26865,26866,26867,26868,26869,26870,26871,26872,26873,26874,26875,26876,26877,26878,26879,26880,26881,26882,26883,26884,26885,26886,26887,26888,26889,26890,26891,26892,26893,26894,26895,26896,26897,26898,26899,26900,26901,26902,26903,26904,26905,26906,26907,26908,26909,26910,26911,26912,26913,26914,26915,26916,26917,26918,26919,26920,26921,26922,26923,26924,26925,26926,26927,26928,26929,26930,26931,26932,26933,26934,26935,26936,26937,26938,26939,26940,26941,26942,26943,26944,26945,26946,26947,26948,26949,26950,26951,26952,26953,26954,26955,26956,26957,26958,26959,26960,26961,26962,26963,26964,26965,26966,26967,26968,26969,26970,26971,26972,26973,26974,26975,26976,26977,26978,26979,26980,26981,26982,26983,26984,26985,26986,26987,26988,26989,26990,26991,26992,26993,26994,26995,26996,26997,26998,26999,27000,27001,27002,27003,27004,27005,27006,27007,27008,27009,27010,27011,27012,27013,27014,27015,27016,27017,27018,27019,27020,27021,27022,27023,27024,27025,27026,27027,27028,27029,27030,27031,27032,27033,27034,27035,27036,27037,27038,27039,27040,27041,27042,27043,27044,27045,27046,27047,27048,27049,27050,27051,27052,27053,27054,27055,27056,27057,27058,27059,27060,27061,27062,27063,27064,27065,27066,27067,27068,27069,27070,27071,27072,27073,27074,27075,27076,27077,27078,27079,27080,27081,27082,27083,27084,27085,27086,27087,27088,27089,27090,27091,27092,27093,27094,27095,27096,27097,27098,27099,27100,27101,27102,27103,27104,27105,27106,27107,27108,27109,27110,27111,27112,27113,27114,27115,27116,27117,27118,27119,27120,27121,27122,27123,27124,27125,27126,27127,27128,27129,27130,27131,27132,27133,27134,27135,27136,27137,27138,27139,27140,27141,27142,27143,27144,27145,27146,27147,27148,27149,27150,27151,27152,27153,27154,27155,27156,27157,27158,27159,27160,27161,27162,27163,27164,27165,27166,27167,27168,27169,27170,27171,27172,27173,27174,27175,27176,27177,27178,27179,27180,27181,27182,27183,27184,27185,27186,27187,27188,27189,27190,27191,27192,27193,27194,27195,27196,27197,27198,27199,27200,27201,27202,27203,27204,27205,27206,27207,27208,27209,27210,27211,27212,27213,27214,27215,27216,27217,27218,27219,27220,27221,27222,27223,27224,27225,27226,27227,27228,27229,27230,27231,27232,27233,27234,27235,27236,27237,27238,27239,27240,27241,27242,27243,27244,27245,27246,27247,27248,27249,27250,27251,27252,27253,27254,27255,27256,27257,27258,27259,27260,27261,27262,27263,27264,27265,27266,27267,27268,27269,27270,27271,27272,27273,27274,27275,27276,27277,27278,27279,27280,27281,27282,27283,27284,27285,27286,27287,27288,27289,27290,27291,27292,27293,27294,27295,27296,27297,27298,27299,27300,27301,27302,27303,27304,27305,27306,27307,27308,27309,27310,27311,27312,27313,27314,27315,27316,27317,27318,27319,27320,27321,27322,27323,27324,27325,27326,27327,27328,27329,27330,27331,27332,27333,27334,27335,27336,27337,27338,27339,27340,27341,27342,27343,27344,27345,27346,27347,27348,27349,27350,27351,27352,27353,27354,27355,27356,27357,27358,27359,27360,27361,27362,27363,27364,27365,27366,27367,27368,27369,27370,27371,27372,27373,27374,27375,27376,27377,27378,27379,27380,27381,27382,27383,27384,27385,27386,27387,27388,27389,27390,27391,27392,27393,27394,27395,27396,27397,27398,27399,27400,27401,27402,27403,27404,27405,27406,27407,27408,27409,27410,27411,27412,27413,27414,27415,27416,27417,27418,27419,27420,27421,27422,27423,27424,27425,27426,27427,27428,27429,27430,27431,27432,27433,27434,27435,27436,27437,27438,27439,27440,27441,27442,27443,27444,27445,27446,27447,27448,27449,27450,27451,27452,27453,27454,27455,27456,27457,27458,27459,27460,27461,27462,27463,27464,27465,27466,27467,27468,27469,27470,27471,27472,27473,27474,27475,27476,27477,27478,27479,27480,27481,27482,27483,27484,27485,27486,27487,27488,27489,27490,27491,27492,27493,27494,27495,27496,27497,27498,27499,27500,27501,27502,27503,27504,27505,27506,27507,27508,27509,27510,27511,27512,27513,27514,27515,27516,27517,27518,27519,27520,27521,27522,27523,27524,27525,27526,27527,27528,27529,27530,27531,27532,27533,27534,27535,27536,27537,27538,27539,27540,27541,27542,27543,27544,27545,27546,27547,27548,27549,27550,27551,27552,27553,27554,27555,27556,27557,27558,27559,27560,27561,27562,27563,27564,27565,27566,27567,27568,27569,27570,27571,27572,27573,27574,27575,27576,27577,27578,27579,27580,27581,27582,27583,27584,27585,27586,27587,27588,27589,27590,27591,27592,27593,27594,27595,27596,27597,27598,27599,27600,27601,27602,27603,27604,27605,27606,27607,27608,27609,27610,27611,27612,27613,27614,27615,27616,27617,27618,27619,27620,27621,27622,27623,27624,27625,27626,27627,27628,27629,27630,27631,27632,27633,27634,27635,27636,27637,27638,27639,27640,27641,27642,27643,27644,27645,27646,27647,27648,27649,27650,27651,27652,27653,27654,27655,27656,27657,27658,27659,27660,27661,27662,27663,27664,27665,27666,27667,27668,27669,27670,27671,27672,27673,27674,27675,27676,27677,27678,27679,27680,27681,27682,27683,27684,27685,27686,27687,27688,27689,27690,27691,27692,27693,27694,27695,27696,27697,27698,27699,27700,27701,27702,27703,27704,27705,27706,27707,27708,27709,27710,27711,27712,27713,27714,27715,27716,27717,27718,27719,27720,27721,27722,27723,27724,27725,27726,27727,27728,27729,27730,27731,27732,27733,27734,27735,27736,27737,27738,27739,27740,27741,27742,27743,27744,27745,27746,27747,27748,27749,27750,27751,27752,27753,27754,27755,27756,27757,27758,27759,27760,27761,27762,27763,27764,27765,27766,27767,27768,27769,27770,27771,27772,27773,27774,27775,27776,27777,27778,27779,27780,27781,27782,27783,27784,27785,27786,27787,27788,27789,27790,27791,27792,27793,27794,27795,27796,27797,27798,27799,27800,27801,27802,27803,27804,27805,27806,27807,27808,27809,27810,27811,27812,27813,27814,27815,27816,27817,27818,27819,27820,27821,27822,27823,27824,27825,27826,27827,27828,27829,27830,27831,27832,27833,27834,27835,27836,27837,27838,27839,27840,27841,27842,27843,27844,27845,27846,27847,27848,27849,27850,27851,27852,27853,27854,27855,27856,27857,27858,27859,27860,27861,27862,27863,27864,27865,27866,27867,27868,27869,27870,27871,27872,27873,27874,27875,27876,27877,27878,27879,27880,27881,27882,27883,27884,27885,27886,27887,27888,27889,27890,27891,27892,27893,27894,27895,27896,27897,27898,27899,27900,27901,27902,27903,27904,27905,27906,27907,27908,27909,27910,27911,27912,27913,27914,27915,27916,27917,27918,27919,27920,27921,27922,27923,27924,27925,27926,27927,27928,27929,27930,27931,27932,27933,27934,27935,27936,27937,27938,27939,27940,27941,27942,27943,27944,27945,27946,27947,27948,27949,27950,27951,27952,27953,27954,27955,27956,27957,27958,27959,27960,27961,27962,27963,27964,27965,27966,27967,27968,27969,27970,27971,27972,27973,27974,27975,27976,27977,27978,27979,27980,27981,27982,27983,27984,27985,27986,27987,27988,27989,27990,27991,27992,27993,27994,27995,27996,27997,27998,27999,28000,28001,28002,28003,28004,28005,28006,28007,28008,28009,28010,28011,28012,28013,28014,28015,28016,28017,28018,28019,28020,28021,28022,28023,28024,28025,28026,28027,28028,28029,28030,28031,28032,28033,28034,28035,28036,28037,28038,28039,28040,28041,28042,28043,28044,28045,28046,28047,28048,28049,28050,28051,28052,28053,28054,28055,28056,28057,28058,28059,28060,28061,28062,28063,28064,28065,28066,28067,28068,28069,28070,28071,28072,28073,28074,28075,28076,28077,28078,28079,28080,28081,28082,28083,28084,28085,28086,28087,28088,28089,28090,28091,28092,28093,28094,28095,28096,28097,28098,28099,28100,28101,28102,28103,28104,28105,28106,28107,28108,28109,28110,28111,28112,28113,28114,28115,28116,28117,28118,28119,28120,28121,28122,28123,28124,28125,28126,28127,28128,28129,28130,28131,28132,28133,28134,28135,28136,28137,28138,28139,28140,28141,28142,28143,28144,28145,28146,28147,28148,28149,28150,28151,28152,28153,28154,28155,28156,28157,28158,28159,28160,28161,28162,28163,28164,28165,28166,28167,28168,28169,28170,28171,28172,28173,28174,28175,28176,28177,28178,28179,28180,28181,28182,28183,28184,28185,28186,28187,28188,28189,28190,28191,28192,28193,28194,28195,28196,28197,28198,28199,28200,28201,28202,28203,28204,28205,28206,28207,28208,28209,28210,28211,28212,28213,28214,28215,28216,28217,28218,28219,28220,28221,28222,28223,28224,28225,28226,28227,28228,28229,28230,28231,28232,28233,28234,28235,28236,28237,28238,28239,28240,28241,28242,28243,28244,28245,28246,28247,28248,28249,28250,28251,28252,28253,28254,28255,28256,28257,28258,28259,28260,28261,28262,28263,28264,28265,28266,28267,28268,28269,28270,28271,28272,28273,28274,28275,28276,28277,28278,28279,28280,28281,28282,28283,28284,28285,28286,28287,28288,28289,28290,28291,28292,28293,28294,28295,28296,28297,28298,28299,28300,28301,28302,28303,28304,28305,28306,28307,28308,28309,28310,28311,28312,28313,28314,28315,28316,28317,28318,28319,28320,28321,28322,28323,28324,28325,28326,28327,28328,28329,28330,28331,28332,28333,28334,28335,28336,28337,28338,28339,28340,28341,28342,28343,28344,28345,28346,28347,28348,28349,28350,28351,28352,28353,28354,28355,28356,28357,28358,28359,28360,28361,28362,28363,28364,28365,28366,28367,28368,28369,28370,28371,28372,28373,28374,28375,28376,28377,28378,28379,28380,28381,28382,28383,28384,28385,28386,28387,28388,28389,28390,28391,28392,28393,28394,28395,28396,28397,28398,28399,28400,28401,28402,28403,28404,28405,28406,28407,28408,28409,28410,28411,28412,28413,28414,28415,28416,28417,28418,28419,28420,28421,28422,28423,28424,28425,28426,28427,28428,28429,28430,28431,28432,28433,28434,28435,28436,28437,28438,28439,28440,28441,28442,28443,28444,28445,28446,28447,28448,28449,28450,28451,28452,28453,28454,28455,28456,28457,28458,28459,28460,28461,28462,28463,28464,28465,28466,28467,28468,28469,28470,28471,28472,28473,28474,28475,28476,28477,28478,28479,28480,28481,28482,28483,28484,28485,28486,28487,28488,28489,28490,28491,28492,28493,28494,28495,28496,28497,28498,28499,28500,28501,28502,28503,28504,28505,28506,28507,28508,28509,28510,28511,28512,28513,28514,28515,28516,28517,28518,28519,28520,28521,28522,28523,28524,28525,28526,28527,28528,28529,28530,28531,28532,28533,28534,28535,28536,28537,28538,28539,28540,28541,28542,28543,28544,28545,28546,28547,28548,28549,28550,28551,28552,28553,28554,28555,28556,28557,28558,28559,28560,28561,28562,28563,28564,28565,28566,28567,28568,28569,28570,28571,28572,28573,28574,28575,28576,28577,28578,28579,28580,28581,28582,28583,28584,28585,28586,28587,28588,28589,28590,28591,28592,28593,28594,28595,28596,28597,28598,28599,28600,28601,28602,28603,28604,28605,28606,28607,28608,28609,28610,28611,28612,28613,28614,28615,28616,28617,28618,28619,28620,28621,28622,28623,28624,28625,28626,28627,28628,28629,28630,28631,28632,28633,28634,28635,28636,28637,28638,28639,28640,28641,28642,28643,28644,28645,28646,28647,28648,28649,28650,28651,28652,28653,28654,28655,28656,28657,28658,28659,28660,28661,28662,28663,28664,28665,28666,28667,28668,28669,28670,28671,28672,28673,28674,28675,28676,28677,28678,28679,28680,28681,28682,28683,28684,28685,28686,28687,28688,28689,28690,28691,28692,28693,28694,28695,28696,28697,28698,28699,28700,28701,28702,28703,28704,28705,28706,28707,28708,28709,28710,28711,28712,28713,28714,28715,28716,28717,28718,28719,28720,28721,28722,28723,28724,28725,28726,28727,28728,28729,28730,28731,28732,28733,28734,28735,28736,28737,28738,28739,28740,28741,28742,28743,28744,28745,28746,28747,28748,28749,28750,28751,28752,28753,28754,28755,28756,28757,28758,28759,28760,28761,28762,28763,28764,28765,28766,28767,28768,28769,28770,28771,28772,28773,28774,28775,28776,28777,28778,28779,28780,28781,28782,28783,28784,28785,28786,28787,28788,28789,28790,28791,28792,28793,28794,28795,28796,28797,28798,28799,28800,28801,28802,28803,28804,28805,28806,28807,28808,28809,28810,28811,28812,28813,28814,28815,28816,28817,28818,28819,28820,28821,28822,28823,28824,28825,28826,28827,28828,28829,28830,28831,28832,28833,28834,28835,28836,28837,28838,28839,28840,28841,28842,28843,28844,28845,28846,28847,28848,28849,28850,28851,28852,28853,28854,28855,28856,28857,28858,28859,28860,28861,28862,28863,28864,28865,28866,28867,28868,28869,28870,28871,28872,28873,28874,28875,28876,28877,28878,28879,28880,28881,28882,28883,28884,28885,28886,28887,28888,28889,28890,28891,28892,28893,28894,28895,28896,28897,28898,28899,28900,28901,28902,28903,28904,28905,28906,28907,28908,28909,28910,28911,28912,28913,28914,28915,28916,28917,28918,28919,28920,28921,28922,28923,28924,28925,28926,28927,28928,28929,28930,28931,28932,28933,28934,28935,28936,28937,28938,28939,28940,28941,28942,28943,28944,28945,28946,28947,28948,28949,28950,28951,28952,28953,28954,28955,28956,28957,28958,28959,28960,28961,28962,28963,28964,28965,28966,28967,28968,28969,28970,28971,28972,28973,28974,28975,28976,28977,28978,28979,28980,28981,28982,28983,28984,28985,28986,28987,28988,28989,28990,28991,28992,28993,28994,28995,28996,28997,28998,28999,29000,29001,29002,29003,29004,29005,29006,29007,29008,29009,29010,29011,29012,29013,29014,29015,29016,29017,29018,29019,29020,29021,29022,29023,29024,29025,29026,29027,29028,29029,29030,29031,29032,29033,29034,29035,29036,29037,29038,29039,29040,29041,29042,29043,29044,29045,29046,29047,29048,29049,29050,29051,29052,29053,29054,29055,29056,29057,29058,29059,29060,29061,29062,29063,29064,29065,29066,29067,29068,29069,29070,29071,29072,29073,29074,29075,29076,29077,29078,29079,29080,29081,29082,29083,29084,29085,29086,29087,29088,29089,29090,29091,29092,29093,29094,29095,29096,29097,29098,29099,29100,29101,29102,29103,29104,29105,29106,29107,29108,29109,29110,29111,29112,29113,29114,29115,29116,29117,29118,29119,29120,29121,29122,29123,29124,29125,29126,29127,29128,29129,29130,29131,29132,29133,29134,29135,29136,29137,29138,29139,29140,29141,29142,29143,29144,29145,29146,29147,29148,29149,29150,29151,29152,29153,29154,29155,29156,29157,29158,29159,29160,29161,29162,29163,29164,29165,29166,29167,29168,29169,29170,29171,29172,29173,29174,29175,29176,29177,29178,29179,29180,29181,29182,29183,29184,29185,29186,29187,29188,29189,29190,29191,29192,29193,29194,29195,29196,29197,29198,29199,29200,29201,29202,29203,29204,29205,29206,29207,29208,29209,29210,29211,29212,29213,29214,29215,29216,29217,29218,29219,29220,29221,29222,29223,29224,29225,29226,29227,29228,29229,29230,29231,29232,29233,29234,29235,29236,29237,29238,29239,29240,29241,29242,29243,29244,29245,29246,29247,29248,29249,29250,29251,29252,29253,29254,29255,29256,29257,29258,29259,29260,29261,29262,29263,29264,29265,29266,29267,29268,29269,29270,29271,29272,29273,29274,29275,29276,29277,29278,29279,29280,29281,29282,29283,29284,29285,29286,29287,29288,29289,29290,29291,29292,29293,29294,29295,29296,29297,29298,29299,29300,29301,29302,29303,29304,29305,29306,29307,29308,29309,29310,29311,29312,29313,29314,29315,29316,29317,29318,29319,29320,29321,29322,29323,29324,29325,29326,29327,29328,29329,29330,29331,29332,29333,29334,29335,29336,29337,29338,29339,29340,29341,29342,29343,29344,29345,29346,29347,29348,29349,29350,29351,29352,29353,29354,29355,29356,29357,29358,29359,29360,29361,29362,29363,29364,29365,29366,29367,29368,29369,29370,29371,29372,29373,29374,29375,29376,29377,29378,29379,29380,29381,29382,29383,29384,29385,29386,29387,29388,29389,29390,29391,29392,29393,29394,29395,29396,29397,29398,29399,29400,29401,29402,29403,29404,29405,29406,29407,29408,29409,29410,29411,29412,29413,29414,29415,29416,29417,29418,29419,29420,29421,29422,29423,29424,29425,29426,29427,29428,29429,29430,29431,29432,29433,29434,29435,29436,29437,29438,29439,29440,29441,29442,29443,29444,29445,29446,29447,29448,29449,29450,29451,29452,29453,29454,29455,29456,29457,29458,29459,29460,29461,29462,29463,29464,29465,29466,29467,29468,29469,29470,29471,29472,29473,29474,29475,29476,29477,29478,29479,29480,29481,29482,29483,29484,29485,29486,29487,29488,29489,29490,29491,29492,29493,29494,29495,29496,29497,29498,29499,29500,29501,29502,29503,29504,29505,29506,29507,29508,29509,29510,29511,29512,29513,29514,29515,29516,29517,29518,29519,29520,29521,29522,29523,29524,29525,29526,29527,29528,29529,29530,29531,29532,29533,29534,29535,29536,29537,29538,29539,29540,29541,29542,29543,29544,29545,29546,29547,29548,29549,29550,29551,29552,29553,29554,29555,29556,29557,29558,29559,29560,29561,29562,29563,29564,29565,29566,29567,29568,29569,29570,29571,29572,29573,29574,29575,29576,29577,29578,29579,29580,29581,29582,29583,29584,29585,29586,29587,29588,29589,29590,29591,29592,29593,29594,29595,29596,29597,29598,29599,29600,29601,29602,29603,29604,29605,29606,29607,29608,29609,29610,29611,29612,29613,29614,29615,29616,29617,29618,29619,29620,29621,29622,29623,29624,29625,29626,29627,29628,29629,29630,29631,29632,29633,29634,29635,29636,29637,29638,29639,29640,29641,29642,29643,29644,29645,29646,29647,29648,29649,29650,29651,29652,29653,29654,29655,29656,29657,29658,29659,29660,29661,29662,29663,29664,29665,29666,29667,29668,29669,29670,29671,29672,29673,29674,29675,29676,29677,29678,29679,29680,29681,29682,29683,29684,29685,29686,29687,29688,29689,29690,29691,29692,29693,29694,29695,29696,29697,29698,29699,29700,29701,29702,29703,29704,29705,29706,29707,29708,29709,29710,29711,29712,29713,29714,29715,29716,29717,29718,29719,29720,29721,29722,29723,29724,29725,29726,29727,29728,29729,29730,29731,29732,29733,29734,29735,29736,29737,29738,29739,29740,29741,29742,29743,29744,29745,29746,29747,29748,29749,29750,29751,29752,29753,29754,29755,29756,29757,29758,29759,29760,29761,29762,29763,29764,29765,29766,29767,29768,29769,29770,29771,29772,29773,29774,29775,29776,29777,29778,29779,29780,29781,29782,29783,29784,29785,29786,29787,29788,29789,29790,29791,29792,29793,29794,29795,29796,29797,29798,29799,29800,29801,29802,29803,29804,29805,29806,29807,29808,29809,29810,29811,29812,29813,29814,29815,29816,29817,29818,29819,29820,29821,29822,29823,29824,29825,29826,29827,29828,29829,29830,29831,29832,29833,29834,29835,29836,29837,29838,29839,29840,29841,29842,29843,29844,29845,29846,29847,29848,29849,29850,29851,29852,29853,29854,29855,29856,29857,29858,29859,29860,29861,29862,29863,29864,29865,29866,29867,29868,29869,29870,29871,29872,29873,29874,29875,29876,29877,29878,29879,29880,29881,29882,29883,29884,29885,29886,29887,29888,29889,29890,29891,29892,29893,29894,29895,29896,29897,29898,29899,29900,29901,29902,29903,29904,29905,29906,29907,29908,29909,29910,29911,29912,29913,29914,29915,29916,29917,29918,29919,29920,29921,29922,29923,29924,29925,29926,29927,29928,29929,29930,29931,29932,29933,29934,29935,29936,29937,29938,29939,29940,29941,29942,29943,29944,29945,29946,29947,29948,29949,29950,29951,29952,29953,29954,29955,29956,29957,29958,29959,29960,29961,29962,29963,29964,29965,29966,29967,29968,29969,29970,29971,29972,29973,29974,29975,29976,29977,29978,29979,29980,29981,29982,29983,29984,29985,29986,29987,29988,29989,29990,29991,29992,29993,29994,29995,29996,29997,29998,29999,30000,30001,30002,30003,30004,30005,30006,30007,30008,30009,30010,30011,30012,30013,30014,30015,30016,30017,30018,30019,30020,30021,30022,30023,30024,30025,30026,30027,30028,30029,30030,30031,30032,30033,30034,30035,30036,30037,30038,30039,30040,30041,30042,30043,30044,30045,30046,30047,30048,30049,30050,30051,30052,30053,30054,30055,30056,30057,30058,30059,30060,30061,30062,30063,30064,30065,30066,30067,30068,30069,30070,30071,30072,30073,30074,30075,30076,30077,30078,30079,30080,30081,30082,30083,30084,30085,30086,30087,30088,30089,30090,30091,30092,30093,30094,30095,30096,30097,30098,30099,30100,30101,30102,30103,30104,30105,30106,30107,30108,30109,30110,30111,30112,30113,30114,30115,30116,30117,30118,30119,30120,30121,30122,30123,30124,30125,30126,30127,30128,30129,30130,30131,30132,30133,30134,30135,30136,30137,30138,30139,30140,30141,30142,30143,30144,30145,30146,30147,30148,30149,30150,30151,30152,30153,30154,30155,30156,30157,30158,30159,30160,30161,30162,30163,30164,30165,30166,30167,30168,30169,30170,30171,30172,30173,30174,30175,30176,30177,30178,30179,30180,30181,30182,30183,30184,30185,30186,30187,30188,30189,30190,30191,30192,30193,30194,30195,30196,30197,30198,30199,30200,30201,30202,30203,30204,30205,30206,30207,30208,30209,30210,30211,30212,30213,30214,30215,30216,30217,30218,30219,30220,30221,30222,30223,30224,30225,30226,30227,30228,30229,30230,30231,30232,30233,30234,30235,30236,30237,30238,30239,30240,30241,30242,30243,30244,30245,30246,30247,30248,30249,30250,30251,30252,30253,30254,30255,30256,30257,30258,30259,30260,30261,30262,30263,30264,30265,30266,30267,30268,30269,30270,30271,30272,30273,30274,30275,30276,30277,30278,30279,30280,30281,30282,30283,30284,30285,30286,30287,30288,30289,30290,30291,30292,30293,30294,30295,30296,30297,30298,30299,30300,30301,30302,30303,30304,30305,30306,30307,30308,30309,30310,30311,30312,30313,30314,30315,30316,30317,30318,30319,30320,30321,30322,30323,30324,30325,30326,30327,30328,30329,30330,30331,30332,30333,30334,30335,30336,30337,30338,30339,30340,30341,30342,30343,30344,30345,30346,30347,30348,30349,30350,30351,30352,30353,30354,30355,30356,30357,30358,30359,30360,30361,30362,30363,30364,30365,30366,30367,30368,30369,30370,30371,30372,30373,30374,30375,30376,30377,30378,30379,30380,30381,30382,30383,30384,30385,30386,30387,30388,30389,30390,30391,30392,30393,30394,30395,30396,30397,30398,30399,30400,30401,30402,30403,30404,30405,30406,30407,30408,30409,30410,30411,30412,30413,30414,30415,30416,30417,30418,30419,30420,30421,30422,30423,30424,30425,30426,30427,30428,30429,30430,30431,30432,30433,30434,30435,30436,30437,30438,30439,30440,30441,30442,30443,30444,30445,30446,30447,30448,30449,30450,30451,30452,30453,30454,30455,30456,30457,30458,30459,30460,30461,30462,30463,30464,30465,30466,30467,30468,30469,30470,30471,30472,30473,30474,30475,30476,30477,30478,30479,30480,30481,30482,30483,30484,30485,30486,30487,30488,30489,30490,30491,30492,30493,30494,30495,30496,30497,30498,30499,30500,30501,30502,30503,30504,30505,30506,30507,30508,30509,30510,30511,30512,30513,30514,30515,30516,30517,30518,30519,30520,30521,30522,30523,30524,30525,30526,30527,30528,30529,30530,30531,30532,30533,30534,30535,30536,30537,30538,30539,30540,30541,30542,30543,30544,30545,30546,30547,30548,30549,30550,30551,30552,30553,30554,30555,30556,30557,30558,30559,30560,30561,30562,30563,30564,30565,30566,30567,30568,30569,30570,30571,30572,30573,30574,30575,30576,30577,30578,30579,30580,30581,30582,30583,30584,30585,30586,30587,30588,30589,30590,30591,30592,30593,30594,30595,30596,30597,30598,30599,30600,30601,30602,30603,30604,30605,30606,30607,30608,30609,30610,30611,30612,30613,30614,30615,30616,30617,30618,30619,30620,30621,30622,30623,30624,30625,30626,30627,30628,30629,30630,30631,30632,30633,30634,30635,30636,30637,30638,30639,30640,30641,30642,30643,30644,30645,30646,30647,30648,30649,30650,30651,30652,30653,30654,30655,30656,30657,30658,30659,30660,30661,30662,30663,30664,30665,30666,30667,30668,30669,30670,30671,30672,30673,30674,30675,30676,30677,30678,30679,30680,30681,30682,30683,30684,30685,30686,30687,30688,30689,30690,30691,30692,30693,30694,30695,30696,30697,30698,30699,30700,30701,30702,30703,30704,30705,30706,30707,30708,30709,30710,30711,30712,30713,30714,30715,30716,30717,30718,30719,30720,30721,30722,30723,30724,30725,30726,30727,30728,30729,30730,30731,30732,30733,30734,30735,30736,30737,30738,30739,30740,30741,30742,30743,30744,30745,30746,30747,30748,30749,30750,30751,30752,30753,30754,30755,30756,30757,30758,30759,30760,30761,30762,30763,30764,30765,30766,30767,30768,30769,30770,30771,30772,30773,30774,30775,30776,30777,30778,30779,30780,30781,30782,30783,30784,30785,30786,30787,30788,30789,30790,30791,30792,30793,30794,30795,30796,30797,30798,30799,30800,30801,30802,30803,30804,30805,30806,30807,30808,30809,30810,30811,30812,30813,30814,30815,30816,30817,30818,30819,30820,30821,30822,30823,30824,30825,30826,30827,30828,30829,30830,30831,30832,30833,30834,30835,30836,30837,30838,30839,30840,30841,30842,30843,30844,30845,30846,30847,30848,30849,30850,30851,30852,30853,30854,30855,30856,30857,30858,30859,30860,30861,30862,30863,30864,30865,30866,30867,30868,30869,30870,30871,30872,30873,30874,30875,30876,30877,30878,30879,30880,30881,30882,30883,30884,30885,30886,30887,30888,30889,30890,30891,30892,30893,30894,30895,30896,30897,30898,30899,30900,30901,30902,30903,30904,30905,30906,30907,30908,30909,30910,30911,30912,30913,30914,30915,30916,30917,30918,30919,30920,30921,30922,30923,30924,30925,30926,30927,30928,30929,30930,30931,30932,30933,30934,30935,30936,30937,30938,30939,30940,30941,30942,30943,30944,30945,30946,30947,30948,30949,30950,30951,30952,30953,30954,30955,30956,30957,30958,30959,30960,30961,30962,30963,30964,30965,30966,30967,30968,30969,30970,30971,30972,30973,30974,30975,30976,30977,30978,30979,30980,30981,30982,30983,30984,30985,30986,30987,30988,30989,30990,30991,30992,30993,30994,30995,30996,30997,30998,30999,31000,31001,31002,31003,31004,31005,31006,31007,31008,31009,31010,31011,31012,31013,31014,31015,31016,31017,31018,31019,31020,31021,31022,31023,31024,31025,31026,31027,31028,31029,31030,31031,31032,31033,31034,31035,31036,31037,31038,31039,31040,31041,31042,31043,31044,31045,31046,31047,31048,31049,31050,31051,31052,31053,31054,31055,31056,31057,31058,31059,31060,31061,31062,31063,31064,31065,31066,31067,31068,31069,31070,31071,31072,31073,31074,31075,31076,31077,31078,31079,31080,31081,31082,31083,31084,31085,31086,31087,31088,31089,31090,31091,31092,31093,31094,31095,31096,31097,31098,31099,31100,31101,31102,31103,31104,31105,31106,31107,31108,31109,31110,31111,31112,31113,31114,31115,31116,31117,31118,31119,31120,31121,31122,31123,31124,31125,31126,31127,31128,31129,31130,31131,31132,31133,31134,31135,31136,31137,31138,31139,31140,31141,31142,31143,31144,31145,31146,31147,31148,31149,31150,31151,31152,31153,31154,31155,31156,31157,31158,31159,31160,31161,31162,31163,31164,31165,31166,31167,31168,31169,31170,31171,31172,31173,31174,31175,31176,31177,31178,31179,31180,31181,31182,31183,31184,31185,31186,31187,31188,31189,31190,31191,31192,31193,31194,31195,31196,31197,31198,31199,31200,31201,31202,31203,31204,31205,31206,31207,31208,31209,31210,31211,31212,31213,31214,31215,31216,31217,31218,31219,31220,31221,31222,31223,31224,31225,31226,31227,31228,31229,31230,31231,31232,31233,31234,31235,31236,31237,31238,31239,31240,31241,31242,31243,31244,31245,31246,31247,31248,31249,31250,31251,31252,31253,31254,31255,31256,31257,31258,31259,31260,31261,31262,31263,31264,31265,31266,31267,31268,31269,31270,31271,31272,31273,31274,31275,31276,31277,31278,31279,31280,31281,31282,31283,31284,31285,31286,31287,31288,31289,31290,31291,31292,31293,31294,31295,31296,31297,31298,31299,31300,31301,31302,31303,31304,31305,31306,31307,31308,31309,31310,31311,31312,31313,31314,31315,31316,31317,31318,31319,31320,31321,31322,31323,31324,31325,31326,31327,31328,31329,31330,31331,31332,31333,31334,31335,31336,31337,31338,31339,31340,31341,31342,31343,31344,31345,31346,31347,31348,31349,31350,31351,31352,31353,31354,31355,31356,31357,31358,31359,31360,31361,31362,31363,31364,31365,31366,31367,31368,31369,31370,31371,31372,31373,31374,31375,31376,31377,31378,31379,31380,31381,31382,31383,31384,31385,31386,31387,31388,31389,31390,31391,31392,31393,31394,31395,31396,31397,31398,31399,31400,31401,31402,31403,31404,31405,31406,31407,31408,31409,31410,31411,31412,31413,31414,31415,31416,31417,31418,31419,31420,31421,31422,31423,31424,31425,31426,31427,31428,31429,31430,31431,31432,31433,31434,31435,31436,31437,31438,31439,31440,31441,31442,31443,31444,31445,31446,31447,31448,31449,31450,31451,31452,31453,31454,31455,31456,31457,31458,31459,31460,31461,31462,31463,31464,31465,31466,31467,31468,31469,31470,31471,31472,31473,31474,31475,31476,31477,31478,31479,31480,31481,31482,31483,31484,31485,31486,31487,31488,31489,31490,31491,31492,31493,31494,31495,31496,31497,31498,31499,31500,31501,31502,31503,31504,31505,31506,31507,31508,31509,31510,31511,31512,31513,31514,31515,31516,31517,31518,31519,31520,31521,31522,31523,31524,31525,31526,31527,31528,31529,31530,31531,31532,31533,31534,31535,31536,31537,31538,31539,31540,31541,31542,31543,31544,31545,31546,31547,31548,31549,31550,31551,31552,31553,31554,31555,31556,31557,31558,31559,31560,31561,31562,31563,31564,31565,31566,31567,31568,31569,31570,31571,31572,31573,31574,31575,31576,31577,31578,31579,31580,31581,31582,31583,31584,31585,31586,31587,31588,31589,31590,31591,31592,31593,31594,31595,31596,31597,31598,31599,31600,31601,31602,31603,31604,31605,31606,31607,31608,31609,31610,31611,31612,31613,31614,31615,31616,31617,31618,31619,31620,31621,31622,31623,31624,31625,31626,31627,31628,31629,31630,31631,31632,31633,31634,31635,31636,31637,31638,31639,31640,31641,31642,31643,31644,31645,31646,31647,31648,31649,31650,31651,31652,31653,31654,31655,31656,31657,31658,31659,31660,31661,31662,31663,31664,31665,31666,31667,31668,31669,31670,31671,31672,31673,31674,31675,31676,31677,31678,31679,31680,31681,31682,31683,31684,31685,31686,31687,31688,31689,31690,31691,31692,31693,31694,31695,31696,31697,31698,31699,31700,31701,31702,31703,31704,31705,31706,31707,31708,31709,31710,31711,31712,31713,31714,31715,31716,31717,31718,31719,31720,31721,31722,31723,31724,31725,31726,31727,31728,31729,31730,31731,31732,31733,31734,31735,31736,31737,31738,31739,31740,31741,31742,31743,31744,31745,31746,31747,31748,31749,31750,31751,31752,31753,31754,31755,31756,31757,31758,31759,31760,31761,31762,31763,31764,31765,31766,31767,31768,31769,31770,31771,31772,31773,31774,31775,31776,31777,31778,31779,31780,31781,31782,31783,31784,31785,31786,31787,31788,31789,31790,31791,31792,31793,31794,31795,31796,31797,31798,31799,31800,31801,31802,31803,31804,31805,31806,31807,31808,31809,31810,31811,31812,31813,31814,31815,31816,31817,31818,31819,31820,31821,31822,31823,31824,31825,31826,31827,31828,31829,31830,31831,31832,31833,31834,31835,31836,31837,31838,31839,31840,31841,31842,31843,31844,31845,31846,31847,31848,31849,31850,31851,31852,31853,31854,31855,31856,31857,31858,31859,31860,31861,31862,31863,31864,31865,31866,31867,31868,31869,31870,31871,31872,31873,31874,31875,31876,31877,31878,31879,31880,31881,31882,31883,31884,31885,31886,31887,31888,31889,31890,31891,31892,31893,31894,31895,31896,31897,31898,31899,31900,31901,31902,31903,31904,31905,31906,31907,31908,31909,31910,31911,31912,31913,31914,31915,31916,31917,31918,31919,31920,31921,31922,31923,31924,31925,31926,31927,31928,31929,31930,31931,31932,31933,31934,31935,31936,31937,31938,31939,31940,31941,31942,31943,31944,31945,31946,31947,31948,31949,31950,31951,31952,31953,31954,31955,31956,31957,31958,31959,31960,31961,31962,31963,31964,31965,31966,31967,31968,31969,31970,31971,31972,31973,31974,31975,31976,31977,31978,31979,31980,31981,31982,31983,31984,31985,31986,31987,31988,31989,31990,31991,31992,31993,31994,31995,31996,31997,31998,31999,32000,32001,32002,32003,32004,32005,32006,32007,32008,32009,32010,32011,32012,32013,32014,32015,32016,32017,32018,32019,32020,32021,32022,32023,32024,32025,32026,32027,32028,32029,32030,32031,32032,32033,32034,32035,32036,32037,32038,32039,32040,32041,32042,32043,32044,32045,32046,32047,32048,32049,32050,32051,32052,32053,32054,32055,32056,32057,32058,32059,32060,32061,32062,32063,32064,32065,32066,32067,32068,32069,32070,32071,32072,32073,32074,32075,32076,32077,32078,32079,32080,32081,32082,32083,32084,32085,32086,32087,32088,32089,32090,32091,32092,32093,32094,32095,32096,32097,32098,32099,32100,32101,32102,32103,32104,32105,32106,32107,32108,32109,32110,32111,32112,32113,32114,32115,32116,32117,32118,32119,32120,32121,32122,32123,32124,32125,32126,32127,32128,32129,32130,32131,32132,32133,32134,32135,32136,32137,32138,32139,32140,32141,32142,32143,32144,32145,32146,32147,32148,32149,32150,32151,32152,32153,32154,32155,32156,32157,32158,32159,32160,32161,32162,32163,32164,32165,32166,32167,32168,32169,32170,32171,32172,32173,32174,32175,32176,32177,32178,32179,32180,32181,32182,32183,32184,32185,32186,32187,32188,32189,32190,32191,32192,32193,32194,32195,32196,32197,32198,32199,32200,32201,32202,32203,32204,32205,32206,32207,32208,32209,32210,32211,32212,32213,32214,32215,32216,32217,32218,32219,32220,32221,32222,32223,32224,32225,32226,32227,32228,32229,32230,32231,32232,32233,32234,32235,32236,32237,32238,32239,32240,32241,32242,32243,32244,32245,32246,32247,32248,32249,32250,32251,32252,32253,32254,32255,32256,32257,32258,32259,32260,32261,32262,32263,32264,32265,32266,32267,32268,32269,32270,32271,32272,32273,32274,32275,32276,32277,32278,32279,32280,32281,32282,32283,32284,32285,32286,32287,32288,32289,32290,32291,32292,32293,32294,32295,32296,32297,32298,32299,32300,32301,32302,32303,32304,32305,32306,32307,32308,32309,32310,32311,32312,32313,32314,32315,32316,32317,32318,32319,32320,32321,32322,32323,32324,32325,32326,32327,32328,32329,32330,32331,32332,32333,32334,32335,32336,32337,32338,32339,32340,32341,32342,32343,32344,32345,32346,32347,32348,32349,32350,32351,32352,32353,32354,32355,32356,32357,32358,32359,32360,32361,32362,32363,32364,32365,32366,32367,32368,32369,32370,32371,32372,32373,32374,32375,32376,32377,32378,32379,32380,32381,32382,32383,32384,32385,32386,32387,32388,32389,32390,32391,32392,32393,32394,32395,32396,32397,32398,32399,32400,32401,32402,32403,32404,32405,32406,32407,32408,32409,32410,32411,32412,32413,32414,32415,32416,32417,32418,32419,32420,32421,32422,32423,32424,32425,32426,32427,32428,32429,32430,32431,32432,32433,32434,32435,32436,32437,32438,32439,32440,32441,32442,32443,32444,32445,32446,32447,32448,32449,32450,32451,32452,32453,32454,32455,32456,32457,32458,32459,32460,32461,32462,32463,32464,32465,32466,32467,32468,32469,32470,32471,32472,32473,32474,32475,32476,32477,32478,32479,32480,32481,32482,32483,32484,32485,32486,32487,32488,32489,32490,32491,32492,32493,32494,32495,32496,32497,32498,32499,32500,32501,32502,32503,32504,32505,32506,32507,32508,32509,32510,32511,32512,32513,32514,32515,32516,32517,32518,32519,32520,32521,32522,32523,32524,32525,32526,32527,32528,32529,32530,32531,32532,32533,32534,32535,32536,32537,32538,32539,32540,32541,32542,32543,32544,32545,32546,32547,32548,32549,32550,32551,32552,32553,32554,32555,32556,32557,32558,32559,32560,32561,32562,32563,32564,32565,32566,32567,32568,32569,32570,32571,32572,32573,32574,32575,32576,32577,32578,32579,32580,32581,32582,32583,32584,32585,32586,32587,32588,32589,32590,32591,32592,32593,32594,32595,32596,32597,32598,32599,32600,32601,32602,32603,32604,32605,32606,32607,32608,32609,32610,32611,32612,32613,32614,32615,32616,32617,32618,32619,32620,32621,32622,32623,32624,32625,32626,32627,32628,32629,32630,32631,32632,32633,32634,32635,32636,32637,32638,32639,32640,32641,32642,32643,32644,32645,32646,32647,32648,32649,32650,32651,32652,32653,32654,32655,32656,32657,32658,32659,32660,32661,32662,32663,32664,32665,32666,32667,32668,32669,32670,32671,32672,32673,32674,32675,32676,32677,32678,32679,32680,32681,32682,32683,32684,32685,32686,32687,32688,32689,32690,32691,32692,32693,32694,32695,32696,32697,32698,32699,32700,32701,32702,32703,32704,32705,32706,32707,32708,32709,32710,32711,32712,32713,32714,32715,32716,32717,32718,32719,32720,32721,32722,32723,32724,32725,32726,32727,32728,32729,32730,32731,32732,32733,32734,32735,32736,32737,32738,32739,32740,32741,32742,32743,32744,32745,32746,32747,32748,32749,32750,32751,32752,32753,32754,32755,32756,32757,32758,32759,32760,32761,32762,32763,32764,32765,32766,32767,32768,32769,32770,32771,32772,32773,32774,32775,32776,32777,32778,32779,32780,32781,32782,32783,32784,32785,32786,32787,32788,32789,32790,32791,32792,32793,32794,32795,32796,32797,32798,32799,32800,32801,32802,32803,32804,32805,32806,32807,32808,32809,32810,32811,32812,32813,32814,32815,32816,32817,32818,32819,32820,32821,32822,32823,32824,32825,32826,32827,32828,32829,32830,32831,32832,32833,32834,32835,32836,32837,32838,32839,32840,32841,32842,32843,32844,32845,32846,32847,32848,32849,32850,32851,32852,32853,32854,32855,32856,32857,32858,32859,32860,32861,32862,32863,32864,32865,32866,32867,32868,32869,32870,32871,32872,32873,32874,32875,32876,32877,32878,32879,32880,32881,32882,32883,32884,32885,32886,32887,32888,32889,32890,32891,32892,32893,32894,32895,32896,32897,32898,32899,32900,32901,32902,32903,32904,32905,32906,32907,32908,32909,32910,32911,32912,32913,32914,32915,32916,32917,32918,32919,32920,32921,32922,32923,32924,32925,32926,32927,32928,32929,32930,32931,32932,32933,32934,32935,32936,32937,32938,32939,32940,32941,32942,32943,32944,32945,32946,32947,32948,32949,32950,32951,32952,32953,32954,32955,32956,32957,32958,32959,32960,32961,32962,32963,32964,32965,32966,32967,32968,32969,32970,32971,32972,32973,32974,32975,32976,32977,32978,32979,32980,32981,32982,32983,32984,32985,32986,32987,32988,32989,32990,32991,32992,32993,32994,32995,32996,32997,32998,32999,33000,33001,33002,33003,33004,33005,33006,33007,33008,33009,33010,33011,33012,33013,33014,33015,33016,33017,33018,33019,33020,33021,33022,33023,33024,33025,33026,33027,33028,33029,33030,33031,33032,33033,33034,33035,33036,33037,33038,33039,33040,33041,33042,33043,33044,33045,33046,33047,33048,33049,33050,33051,33052,33053,33054,33055,33056,33057,33058,33059,33060,33061,33062,33063,33064,33065,33066,33067,33068,33069,33070,33071,33072,33073,33074,33075,33076,33077,33078,33079,33080,33081,33082,33083,33084,33085,33086,33087,33088,33089,33090,33091,33092,33093,33094,33095,33096,33097,33098,33099,33100,33101,33102,33103,33104,33105,33106,33107,33108,33109,33110,33111,33112,33113,33114,33115,33116,33117,33118,33119,33120,33121,33122,33123,33124,33125,33126,33127,33128,33129,33130,33131,33132,33133,33134,33135,33136,33137,33138,33139,33140,33141,33142,33143,33144,33145,33146,33147,33148,33149,33150,33151,33152,33153,33154,33155,33156,33157,33158,33159,33160,33161,33162,33163,33164,33165,33166,33167,33168,33169,33170,33171,33172,33173,33174,33175,33176,33177,33178,33179,33180,33181,33182,33183,33184,33185,33186,33187,33188,33189,33190,33191,33192,33193,33194,33195,33196,33197,33198,33199,33200,33201,33202,33203,33204,33205,33206,33207,33208,33209,33210,33211,33212,33213,33214,33215,33216,33217,33218,33219,33220,33221,33222,33223,33224,33225,33226,33227,33228,33229,33230,33231,33232,33233,33234,33235,33236,33237,33238,33239,33240,33241,33242,33243,33244,33245,33246,33247,33248,33249,33250,33251,33252,33253,33254,33255,33256,33257,33258,33259,33260,33261,33262,33263,33264,33265,33266,33267,33268,33269,33270,33271,33272,33273,33274,33275,33276,33277,33278,33279,33280,33281,33282,33283,33284,33285,33286,33287,33288,33289,33290,33291,33292,33293,33294,33295,33296,33297,33298,33299,33300,33301,33302,33303,33304,33305,33306,33307,33308,33309,33310,33311,33312,33313,33314,33315,33316,33317,33318,33319,33320,33321,33322,33323,33324,33325,33326,33327,33328,33329,33330,33331,33332,33333,33334,33335,33336,33337,33338,33339,33340,33341,33342,33343,33344,33345,33346,33347,33348,33349,33350,33351,33352,33353,33354,33355,33356,33357,33358,33359,33360,33361,33362,33363,33364,33365,33366,33367,33368,33369,33370,33371,33372,33373,33374,33375,33376,33377,33378,33379,33380,33381,33382,33383,33384,33385,33386,33387,33388,33389,33390,33391,33392,33393,33394,33395,33396,33397,33398,33399,33400,33401,33402,33403,33404,33405,33406,33407,33408,33409,33410,33411,33412,33413,33414,33415,33416,33417,33418,33419,33420,33421,33422,33423,33424,33425,33426,33427,33428,33429,33430,33431,33432,33433,33434,33435,33436,33437,33438,33439,33440,33441,33442,33443,33444,33445,33446,33447,33448,33449,33450,33451,33452,33453,33454,33455,33456,33457,33458,33459,33460,33461,33462,33463,33464,33465,33466,33467,33468,33469,33470,33471,33472,33473,33474,33475,33476,33477,33478,33479,33480,33481,33482,33483,33484,33485,33486,33487,33488,33489,33490,33491,33492,33493,33494,33495,33496,33497,33498,33499,33500,33501,33502,33503,33504,33505,33506,33507,33508,33509,33510,33511,33512,33513,33514,33515,33516,33517,33518,33519,33520,33521,33522,33523,33524,33525,33526,33527,33528,33529,33530,33531,33532,33533,33534,33535,33536,33537,33538,33539,33540,33541,33542,33543,33544,33545,33546,33547,33548,33549,33550,33551,33552,33553,33554,33555,33556,33557,33558,33559,33560,33561,33562,33563,33564,33565,33566,33567,33568,33569,33570,33571,33572,33573,33574,33575,33576,33577,33578,33579,33580,33581,33582,33583,33584,33585,33586,33587,33588,33589,33590,33591,33592,33593,33594,33595,33596,33597,33598,33599,33600,33601,33602,33603,33604,33605,33606,33607,33608,33609,33610,33611,33612,33613,33614,33615,33616,33617,33618,33619,33620,33621,33622,33623,33624,33625,33626,33627,33628,33629,33630,33631,33632,33633,33634,33635,33636,33637,33638,33639,33640,33641,33642,33643,33644,33645,33646,33647,33648,33649,33650,33651,33652,33653,33654,33655,33656,33657,33658,33659,33660,33661,33662,33663,33664,33665,33666,33667,33668,33669,33670,33671,33672,33673,33674,33675,33676,33677,33678,33679,33680,33681,33682,33683,33684,33685,33686,33687,33688,33689,33690,33691,33692,33693,33694,33695,33696,33697,33698,33699,33700,33701,33702,33703,33704,33705,33706,33707,33708,33709,33710,33711,33712,33713,33714,33715,33716,33717,33718,33719,33720,33721,33722,33723,33724,33725,33726,33727,33728,33729,33730,33731,33732,33733,33734,33735,33736,33737,33738,33739,33740,33741,33742,33743,33744,33745,33746,33747,33748,33749,33750,33751,33752,33753,33754,33755,33756,33757,33758,33759,33760,33761,33762,33763,33764,33765,33766,33767,33768,33769,33770,33771,33772,33773,33774,33775,33776,33777,33778,33779,33780,33781,33782,33783,33784,33785,33786,33787,33788,33789,33790,33791,33792,33793,33794,33795,33796,33797,33798,33799,33800,33801,33802,33803,33804,33805,33806,33807,33808,33809,33810,33811,33812,33813,33814,33815,33816,33817,33818,33819,33820,33821,33822,33823,33824,33825,33826,33827,33828,33829,33830,33831,33832,33833,33834,33835,33836,33837,33838,33839,33840,33841,33842,33843,33844,33845,33846,33847,33848,33849,33850,33851,33852,33853,33854,33855,33856,33857,33858,33859,33860,33861,33862,33863,33864,33865,33866,33867,33868,33869,33870,33871,33872,33873,33874,33875,33876,33877,33878,33879,33880,33881,33882,33883,33884,33885,33886,33887,33888,33889,33890,33891,33892,33893,33894,33895,33896,33897,33898,33899,33900,33901,33902,33903,33904,33905,33906,33907,33908,33909,33910,33911,33912,33913,33914,33915,33916,33917,33918,33919,33920,33921,33922,33923,33924,33925,33926,33927,33928,33929,33930,33931,33932,33933,33934,33935,33936,33937,33938,33939,33940,33941,33942,33943,33944,33945,33946,33947,33948,33949,33950,33951,33952,33953,33954,33955,33956,33957,33958,33959,33960,33961,33962,33963,33964,33965,33966,33967,33968,33969,33970,33971,33972,33973,33974,33975,33976,33977,33978,33979,33980,33981,33982,33983,33984,33985,33986,33987,33988,33989,33990,33991,33992,33993,33994,33995,33996,33997,33998,33999,34000,34001,34002,34003,34004,34005,34006,34007,34008,34009,34010,34011,34012,34013,34014,34015,34016,34017,34018,34019,34020,34021,34022,34023,34024,34025,34026,34027,34028,34029,34030,34031,34032,34033,34034,34035,34036,34037,34038,34039,34040,34041,34042,34043,34044,34045,34046,34047,34048,34049,34050,34051,34052,34053,34054,34055,34056,34057,34058,34059,34060,34061,34062,34063,34064,34065,34066,34067,34068,34069,34070,34071,34072,34073,34074,34075,34076,34077,34078,34079,34080,34081,34082,34083,34084,34085,34086,34087,34088,34089,34090,34091,34092,34093,34094,34095,34096,34097,34098,34099,34100,34101,34102,34103,34104,34105,34106,34107,34108,34109,34110,34111,34112,34113,34114,34115,34116,34117,34118,34119,34120,34121,34122,34123,34124,34125,34126,34127,34128,34129,34130,34131,34132,34133,34134,34135,34136,34137,34138,34139,34140,34141,34142,34143,34144,34145,34146,34147,34148,34149,34150,34151,34152,34153,34154,34155,34156,34157,34158,34159,34160,34161,34162,34163,34164,34165,34166,34167,34168,34169,34170,34171,34172,34173,34174,34175,34176,34177,34178,34179,34180,34181,34182,34183,34184,34185,34186,34187,34188,34189,34190,34191,34192,34193,34194,34195,34196,34197,34198,34199,34200,34201,34202,34203,34204,34205,34206,34207,34208,34209,34210,34211,34212,34213,34214,34215,34216,34217,34218,34219,34220,34221,34222,34223,34224,34225,34226,34227,34228,34229,34230,34231,34232,34233,34234,34235,34236,34237,34238,34239,34240,34241,34242,34243,34244,34245,34246,34247,34248,34249,34250,34251,34252,34253,34254,34255,34256,34257,34258,34259,34260,34261,34262,34263,34264,34265,34266,34267,34268,34269,34270,34271,34272,34273,34274,34275,34276,34277,34278,34279,34280,34281,34282,34283,34284,34285,34286,34287,34288,34289,34290,34291,34292,34293,34294,34295,34296,34297,34298,34299,34300,34301,34302,34303,34304,34305,34306,34307,34308,34309,34310,34311,34312,34313,34314,34315,34316,34317,34318,34319,34320,34321,34322,34323,34324,34325,34326,34327,34328,34329,34330,34331,34332,34333,34334,34335,34336,34337,34338,34339,34340,34341,34342,34343,34344,34345,34346,34347,34348,34349,34350,34351,34352,34353,34354,34355,34356,34357,34358,34359,34360,34361,34362,34363,34364,34365,34366,34367,34368,34369,34370,34371,34372,34373,34374,34375,34376,34377,34378,34379,34380,34381,34382,34383,34384,34385,34386,34387,34388,34389,34390,34391,34392,34393,34394,34395,34396,34397,34398,34399,34400,34401,34402,34403,34404,34405,34406,34407,34408,34409,34410,34411,34412,34413,34414,34415,34416,34417,34418,34419,34420,34421,34422,34423,34424,34425,34426,34427,34428,34429,34430,34431,34432,34433,34434,34435,34436,34437,34438,34439,34440,34441,34442,34443,34444,34445,34446,34447,34448,34449,34450,34451,34452,34453,34454,34455,34456,34457,34458,34459,34460,34461,34462,34463,34464,34465,34466,34467,34468,34469,34470,34471,34472,34473,34474,34475,34476,34477,34478,34479,34480,34481,34482,34483,34484,34485,34486,34487,34488,34489,34490,34491,34492,34493,34494,34495,34496,34497,34498,34499,34500,34501,34502,34503,34504,34505,34506,34507,34508,34509,34510,34511,34512,34513,34514,34515,34516,34517,34518,34519,34520,34521,34522,34523,34524,34525,34526,34527,34528,34529,34530,34531,34532,34533,34534,34535,34536,34537,34538,34539,34540,34541,34542,34543,34544,34545,34546,34547,34548,34549,34550,34551,34552,34553,34554,34555,34556,34557,34558,34559,34560,34561,34562,34563,34564,34565,34566,34567,34568,34569,34570,34571,34572,34573,34574,34575,34576,34577,34578,34579,34580,34581,34582,34583,34584,34585,34586,34587,34588,34589,34590,34591,34592,34593,34594,34595,34596,34597,34598,34599,34600,34601,34602,34603,34604,34605,34606,34607,34608,34609,34610,34611,34612,34613,34614,34615,34616,34617,34618,34619,34620,34621,34622,34623,34624,34625,34626,34627,34628,34629,34630,34631,34632,34633,34634,34635,34636,34637,34638,34639,34640,34641,34642,34643,34644,34645,34646,34647,34648,34649,34650,34651,34652,34653,34654,34655,34656,34657,34658,34659,34660,34661,34662,34663,34664,34665,34666,34667,34668,34669,34670,34671,34672,34673,34674,34675,34676,34677,34678,34679,34680,34681,34682,34683,34684,34685,34686,34687,34688,34689,34690,34691,34692,34693,34694,34695,34696,34697,34698,34699,34700,34701,34702,34703,34704,34705,34706,34707,34708,34709,34710,34711,34712,34713,34714,34715,34716,34717,34718,34719,34720,34721,34722,34723,34724,34725,34726,34727,34728,34729,34730,34731,34732,34733,34734,34735,34736,34737,34738,34739,34740,34741,34742,34743,34744,34745,34746,34747,34748,34749,34750,34751,34752,34753,34754,34755,34756,34757,34758,34759,34760,34761,34762,34763,34764,34765,34766,34767,34768,34769,34770,34771,34772,34773,34774,34775,34776,34777,34778,34779,34780,34781,34782,34783,34784,34785,34786,34787,34788,34789,34790,34791,34792,34793,34794,34795,34796,34797,34798,34799,34800,34801,34802,34803,34804,34805,34806,34807,34808,34809,34810,34811,34812,34813,34814,34815,34816,34817,34818,34819,34820,34821,34822,34823,34824,34825,34826,34827,34828,34829,34830,34831,34832,34833,34834,34835,34836,34837,34838,34839,34840,34841,34842,34843,34844,34845,34846,34847,34848,34849,34850,34851,34852,34853,34854,34855,34856,34857,34858,34859,34860,34861,34862,34863,34864,34865,34866,34867,34868,34869,34870,34871,34872,34873,34874,34875,34876,34877,34878,34879,34880,34881,34882,34883,34884,34885,34886,34887,34888,34889,34890,34891,34892,34893,34894,34895,34896,34897,34898,34899,34900,34901,34902,34903,34904,34905,34906,34907,34908,34909,34910,34911,34912,34913,34914,34915,34916,34917,34918,34919,34920,34921,34922,34923,34924,34925,34926,34927,34928,34929,34930,34931,34932,34933,34934,34935,34936,34937,34938,34939,34940,34941,34942,34943,34944,34945,34946,34947,34948,34949,34950,34951,34952,34953,34954,34955,34956,34957,34958,34959,34960,34961,34962,34963,34964,34965,34966,34967,34968,34969,34970,34971,34972,34973,34974,34975,34976,34977,34978,34979,34980,34981,34982,34983,34984,34985,34986,34987,34988,34989,34990,34991,34992,34993,34994,34995,34996,34997,34998,34999,35000,35001,35002,35003,35004,35005,35006,35007,35008,35009,35010,35011,35012,35013,35014,35015,35016,35017,35018,35019,35020,35021,35022,35023,35024,35025,35026,35027,35028,35029,35030,35031,35032,35033,35034,35035,35036,35037,35038,35039,35040,35041,35042,35043,35044,35045,35046,35047,35048,35049,35050,35051,35052,35053,35054,35055,35056,35057,35058,35059,35060,35061,35062,35063,35064,35065,35066,35067,35068,35069,35070,35071,35072,35073,35074,35075,35076,35077,35078,35079,35080,35081,35082,35083,35084,35085,35086,35087,35088,35089,35090,35091,35092,35093,35094,35095,35096,35097,35098,35099,35100,35101,35102,35103,35104,35105,35106,35107,35108,35109,35110,35111,35112,35113,35114,35115,35116,35117,35118,35119,35120,35121,35122,35123,35124,35125,35126,35127,35128,35129,35130,35131,35132,35133,35134,35135,35136,35137,35138,35139,35140,35141,35142,35143,35144,35145,35146,35147,35148,35149,35150,35151,35152,35153,35154,35155,35156,35157,35158,35159,35160,35161,35162,35163,35164,35165,35166,35167,35168,35169,35170,35171,35172,35173,35174,35175,35176,35177,35178,35179,35180,35181,35182,35183,35184,35185,35186,35187,35188,35189,35190,35191,35192,35193,35194,35195,35196,35197,35198,35199,35200,35201,35202,35203,35204,35205,35206,35207,35208,35209,35210,35211,35212,35213,35214,35215,35216,35217,35218,35219,35220,35221,35222,35223,35224,35225,35226,35227,35228,35229,35230,35231,35232,35233,35234,35235,35236,35237,35238,35239,35240,35241,35242,35243,35244,35245,35246,35247,35248,35249,35250,35251,35252,35253,35254,35255,35256,35257,35258,35259,35260,35261,35262,35263,35264,35265,35266,35267,35268,35269,35270,35271,35272,35273,35274,35275,35276,35277,35278,35279,35280,35281,35282,35283,35284,35285,35286,35287,35288,35289,35290,35291,35292,35293,35294,35295,35296,35297,35298,35299,35300,35301,35302,35303,35304,35305,35306,35307,35308,35309,35310,35311,35312,35313,35314,35315,35316,35317,35318,35319,35320,35321,35322,35323,35324,35325,35326,35327,35328,35329,35330,35331,35332,35333,35334,35335,35336,35337,35338,35339,35340,35341,35342,35343,35344,35345,35346,35347,35348,35349,35350,35351,35352,35353,35354,35355,35356,35357,35358,35359,35360,35361,35362,35363,35364,35365,35366,35367,35368,35369,35370,35371,35372,35373,35374,35375,35376,35377,35378,35379,35380,35381,35382,35383,35384,35385,35386,35387,35388,35389,35390,35391,35392,35393,35394,35395,35396,35397,35398,35399,35400,35401,35402,35403,35404,35405,35406,35407,35408,35409,35410,35411,35412,35413,35414,35415,35416,35417,35418,35419,35420,35421,35422,35423,35424,35425,35426,35427,35428,35429,35430,35431,35432,35433,35434,35435,35436,35437,35438,35439,35440,35441,35442,35443,35444,35445,35446,35447,35448,35449,35450,35451,35452,35453,35454,35455,35456,35457,35458,35459,35460,35461,35462,35463,35464,35465,35466,35467,35468,35469,35470,35471,35472,35473,35474,35475,35476,35477,35478,35479,35480,35481,35482,35483,35484,35485,35486,35487,35488,35489,35490,35491,35492,35493,35494,35495,35496,35497,35498,35499,35500,35501,35502,35503,35504,35505,35506,35507,35508,35509,35510,35511,35512,35513,35514,35515,35516,35517,35518,35519,35520,35521,35522,35523,35524,35525,35526,35527,35528,35529,35530,35531,35532,35533,35534,35535,35536,35537,35538,35539,35540,35541,35542,35543,35544,35545,35546,35547,35548,35549,35550,35551,35552,35553,35554,35555,35556,35557,35558,35559,35560,35561,35562,35563,35564,35565,35566,35567,35568,35569,35570,35571,35572,35573,35574,35575,35576,35577,35578,35579,35580,35581,35582,35583,35584,35585,35586,35587,35588,35589,35590,35591,35592,35593,35594,35595,35596,35597,35598,35599,35600,35601,35602,35603,35604,35605,35606,35607,35608,35609,35610,35611,35612,35613,35614,35615,35616,35617,35618,35619,35620,35621,35622,35623,35624,35625,35626,35627,35628,35629,35630,35631,35632,35633,35634,35635,35636,35637,35638,35639,35640,35641,35642,35643,35644,35645,35646,35647,35648,35649,35650,35651,35652,35653,35654,35655,35656,35657,35658,35659,35660,35661,35662,35663,35664,35665,35666,35667,35668,35669,35670,35671,35672,35673,35674,35675,35676,35677,35678,35679,35680,35681,35682,35683,35684,35685,35686,35687,35688,35689,35690,35691,35692,35693,35694,35695,35696,35697,35698,35699,35700,35701,35702,35703,35704,35705,35706,35707,35708,35709,35710,35711,35712,35713,35714,35715,35716,35717,35718,35719,35720,35721,35722,35723,35724,35725,35726,35727,35728,35729,35730,35731,35732,35733,35734,35735,35736,35737,35738,35739,35740,35741,35742,35743,35744,35745,35746,35747,35748,35749,35750,35751,35752,35753,35754,35755,35756,35757,35758,35759,35760,35761,35762,35763,35764,35765,35766,35767,35768,35769,35770,35771,35772,35773,35774,35775,35776,35777,35778,35779,35780,35781,35782,35783,35784,35785,35786,35787,35788,35789,35790,35791,35792,35793,35794,35795,35796,35797,35798,35799,35800,35801,35802,35803,35804,35805,35806,35807,35808,35809,35810,35811,35812,35813,35814,35815,35816,35817,35818,35819,35820,35821,35822,35823,35824,35825,35826,35827,35828,35829,35830,35831,35832,35833,35834,35835,35836,35837,35838,35839,35840,35841,35842,35843,35844,35845,35846,35847,35848,35849,35850,35851,35852,35853,35854,35855,35856,35857,35858,35859,35860,35861,35862,35863,35864,35865,35866,35867,35868,35869,35870,35871,35872,35873,35874,35875,35876,35877,35878,35879,35880,35881,35882,35883,35884,35885,35886,35887,35888,35889,35890,35891,35892,35893,35894,35895,35896,35897,35898,35899,35900,35901,35902,35903,35904,35905,35906,35907,35908,35909,35910,35911,35912,35913,35914,35915,35916,35917,35918,35919,35920,35921,35922,35923,35924,35925,35926,35927,35928,35929,35930,35931,35932,35933,35934,35935,35936,35937,35938,35939,35940,35941,35942,35943,35944,35945,35946,35947,35948,35949,35950,35951,35952,35953,35954,35955,35956,35957,35958,35959,35960,35961,35962,35963,35964,35965,35966,35967,35968,35969,35970,35971,35972,35973,35974,35975,35976,35977,35978,35979,35980,35981,35982,35983,35984,35985,35986,35987,35988,35989,35990,35991,35992,35993,35994,35995,35996,35997,35998,35999,36000,36001,36002,36003,36004,36005,36006,36007,36008,36009,36010,36011,36012,36013,36014,36015,36016,36017,36018,36019,36020,36021,36022,36023,36024,36025,36026,36027,36028,36029,36030,36031,36032,36033,36034,36035,36036,36037,36038,36039,36040,36041,36042,36043,36044,36045,36046,36047,36048,36049,36050,36051,36052,36053,36054,36055,36056,36057,36058,36059,36060,36061,36062,36063,36064,36065,36066,36067,36068,36069,36070,36071,36072,36073,36074,36075,36076,36077,36078,36079,36080,36081,36082,36083,36084,36085,36086,36087,36088,36089,36090,36091,36092,36093,36094,36095,36096,36097,36098,36099,36100,36101,36102,36103,36104,36105,36106,36107,36108,36109,36110,36111,36112,36113,36114,36115,36116,36117,36118,36119,36120,36121,36122,36123,36124,36125,36126,36127,36128,36129,36130,36131,36132,36133,36134,36135,36136,36137,36138,36139,36140,36141,36142,36143,36144,36145,36146,36147,36148,36149,36150,36151,36152,36153,36154,36155,36156,36157,36158,36159,36160,36161,36162,36163,36164,36165,36166,36167,36168,36169,36170,36171,36172,36173,36174,36175,36176,36177,36178,36179,36180,36181,36182,36183,36184,36185,36186,36187,36188,36189,36190,36191,36192,36193,36194,36195,36196,36197,36198,36199,36200,36201,36202,36203,36204,36205,36206,36207,36208,36209,36210,36211,36212,36213,36214,36215,36216,36217,36218,36219,36220,36221,36222,36223,36224,36225,36226,36227,36228,36229,36230,36231,36232,36233,36234,36235,36236,36237,36238,36239,36240,36241,36242,36243,36244,36245,36246,36247,36248,36249,36250,36251,36252,36253,36254,36255,36256,36257,36258,36259,36260,36261,36262,36263,36264,36265,36266,36267,36268,36269,36270,36271,36272,36273,36274,36275,36276,36277,36278,36279,36280,36281,36282,36283,36284,36285,36286,36287,36288,36289,36290,36291,36292,36293,36294,36295,36296,36297,36298,36299,36300,36301,36302,36303,36304,36305,36306,36307,36308,36309,36310,36311,36312,36313,36314,36315,36316,36317,36318,36319,36320,36321,36322,36323,36324,36325,36326,36327,36328,36329,36330,36331,36332,36333,36334,36335,36336,36337,36338,36339,36340,36341,36342,36343,36344,36345,36346,36347,36348,36349,36350,36351,36352,36353,36354,36355,36356,36357,36358,36359,36360,36361,36362,36363,36364,36365,36366,36367,36368,36369,36370,36371,36372,36373,36374,36375,36376,36377,36378,36379,36380,36381,36382,36383,36384,36385,36386,36387,36388,36389,36390,36391,36392,36393,36394,36395,36396,36397,36398,36399,36400,36401,36402,36403,36404,36405,36406,36407,36408,36409,36410,36411,36412,36413,36414,36415,36416,36417,36418,36419,36420,36421,36422,36423,36424,36425,36426,36427,36428,36429,36430,36431,36432,36433,36434,36435,36436,36437,36438,36439,36440,36441,36442,36443,36444,36445,36446,36447,36448,36449,36450,36451,36452,36453,36454,36455,36456,36457,36458,36459,36460,36461,36462,36463,36464,36465,36466,36467,36468,36469,36470,36471,36472,36473,36474,36475,36476,36477,36478,36479,36480,36481,36482,36483,36484,36485,36486,36487,36488,36489,36490,36491,36492,36493,36494,36495,36496,36497,36498,36499,36500,36501,36502,36503,36504,36505,36506,36507,36508,36509,36510,36511,36512,36513,36514,36515,36516,36517,36518,36519,36520,36521,36522,36523,36524,36525,36526,36527,36528,36529,36530,36531,36532,36533,36534,36535,36536,36537,36538,36539,36540,36541,36542,36543,36544,36545,36546,36547,36548,36549,36550,36551,36552,36553,36554,36555,36556,36557,36558,36559,36560,36561,36562,36563,36564,36565,36566,36567,36568,36569,36570,36571,36572,36573,36574,36575,36576,36577,36578,36579,36580,36581,36582,36583,36584,36585,36586,36587,36588,36589,36590,36591,36592,36593,36594,36595,36596,36597,36598,36599,36600,36601,36602,36603,36604,36605,36606,36607,36608,36609,36610,36611,36612,36613,36614,36615,36616,36617,36618,36619,36620,36621,36622,36623,36624,36625,36626,36627,36628,36629,36630,36631,36632,36633,36634,36635,36636,36637,36638,36639,36640,36641,36642,36643,36644,36645,36646,36647,36648,36649,36650,36651,36652,36653,36654,36655,36656,36657,36658,36659,36660,36661,36662,36663,36664,36665,36666,36667,36668,36669,36670,36671,36672,36673,36674,36675,36676,36677,36678,36679,36680,36681,36682,36683,36684,36685,36686,36687,36688,36689,36690,36691,36692,36693,36694,36695,36696,36697,36698,36699,36700,36701,36702,36703,36704,36705,36706,36707,36708,36709,36710,36711,36712,36713,36714,36715,36716,36717,36718,36719,36720,36721,36722,36723,36724,36725,36726,36727,36728,36729,36730,36731,36732,36733,36734,36735,36736,36737,36738,36739,36740,36741,36742,36743,36744,36745,36746,36747,36748,36749,36750,36751,36752,36753,36754,36755,36756,36757,36758,36759,36760,36761,36762,36763,36764,36765,36766,36767,36768,36769,36770,36771,36772,36773,36774,36775,36776,36777,36778,36779,36780,36781,36782,36783,36784,36785,36786,36787,36788,36789,36790,36791,36792,36793,36794,36795,36796,36797,36798,36799,36800,36801,36802,36803,36804,36805,36806,36807,36808,36809,36810,36811,36812,36813,36814,36815,36816,36817,36818,36819,36820,36821,36822,36823,36824,36825,36826,36827,36828,36829,36830,36831,36832,36833,36834,36835,36836,36837,36838,36839,36840,36841,36842,36843,36844,36845,36846,36847,36848,36849,36850,36851,36852,36853,36854,36855,36856,36857,36858,36859,36860,36861,36862,36863,36864,36865,36866,36867,36868,36869,36870,36871,36872,36873,36874,36875,36876,36877,36878,36879,36880,36881,36882,36883,36884,36885,36886,36887,36888,36889,36890,36891,36892,36893,36894,36895,36896,36897,36898,36899,36900,36901,36902,36903,36904,36905,36906,36907,36908,36909,36910,36911,36912,36913,36914,36915,36916,36917,36918,36919,36920,36921,36922,36923,36924,36925,36926,36927,36928,36929,36930,36931,36932,36933,36934,36935,36936,36937,36938,36939,36940,36941,36942,36943,36944,36945,36946,36947,36948,36949,36950,36951,36952,36953,36954,36955,36956,36957,36958,36959,36960,36961,36962,36963,36964,36965,36966,36967,36968,36969,36970,36971,36972,36973,36974,36975,36976,36977,36978,36979,36980,36981,36982,36983,36984,36985,36986,36987,36988,36989,36990,36991,36992,36993,36994,36995,36996,36997,36998,36999,37000,37001,37002,37003,37004,37005,37006,37007,37008,37009,37010,37011,37012,37013,37014,37015,37016,37017,37018,37019,37020,37021,37022,37023,37024,37025,37026,37027,37028,37029,37030,37031,37032,37033,37034,37035,37036,37037,37038,37039,37040,37041,37042,37043,37044,37045,37046,37047,37048,37049,37050,37051,37052,37053,37054,37055,37056,37057,37058,37059,37060,37061,37062,37063,37064,37065,37066,37067,37068,37069,37070,37071,37072,37073,37074,37075,37076,37077,37078,37079,37080,37081,37082,37083,37084,37085,37086,37087,37088,37089,37090,37091,37092,37093,37094,37095,37096,37097,37098,37099,37100,37101,37102,37103,37104,37105,37106,37107,37108,37109,37110,37111,37112,37113,37114,37115,37116,37117,37118,37119,37120,37121,37122,37123,37124,37125,37126,37127,37128,37129,37130,37131,37132,37133,37134,37135,37136,37137,37138,37139,37140,37141,37142,37143,37144,37145,37146,37147,37148,37149,37150,37151,37152,37153,37154,37155,37156,37157,37158,37159,37160,37161,37162,37163,37164,37165,37166,37167,37168,37169,37170,37171,37172,37173,37174,37175,37176,37177,37178,37179,37180,37181,37182,37183,37184,37185,37186,37187,37188,37189,37190,37191,37192,37193,37194,37195,37196,37197,37198,37199,37200,37201,37202,37203,37204,37205,37206,37207,37208,37209,37210,37211,37212,37213,37214,37215,37216,37217,37218,37219,37220,37221,37222,37223,37224,37225,37226,37227,37228,37229,37230,37231,37232,37233,37234,37235,37236,37237,37238,37239,37240,37241,37242,37243,37244,37245,37246,37247,37248,37249,37250,37251,37252,37253,37254,37255,37256,37257,37258,37259,37260,37261,37262,37263,37264,37265,37266,37267,37268,37269,37270,37271,37272,37273,37274,37275,37276,37277,37278,37279,37280,37281,37282,37283,37284,37285,37286,37287,37288,37289,37290,37291,37292,37293,37294,37295,37296,37297,37298,37299,37300,37301,37302,37303,37304,37305,37306,37307,37308,37309,37310,37311,37312,37313,37314,37315,37316,37317,37318,37319,37320,37321,37322,37323,37324,37325,37326,37327,37328,37329,37330,37331,37332,37333,37334,37335,37336,37337,37338,37339,37340,37341,37342,37343,37344,37345,37346,37347,37348,37349,37350,37351,37352,37353,37354,37355,37356,37357,37358,37359,37360,37361,37362,37363,37364,37365,37366,37367,37368,37369,37370,37371,37372,37373,37374,37375,37376,37377,37378,37379,37380,37381,37382,37383,37384,37385,37386,37387,37388,37389,37390,37391,37392,37393,37394,37395,37396,37397,37398,37399,37400,37401,37402,37403,37404,37405,37406,37407,37408,37409,37410,37411,37412,37413,37414,37415,37416,37417,37418,37419,37420,37421,37422,37423,37424,37425,37426,37427,37428,37429,37430,37431,37432,37433,37434,37435,37436,37437,37438,37439,37440,37441,37442,37443,37444,37445,37446,37447,37448,37449,37450,37451,37452,37453,37454,37455,37456,37457,37458,37459,37460,37461,37462,37463,37464,37465,37466,37467,37468,37469,37470,37471,37472,37473,37474,37475,37476,37477,37478,37479,37480,37481,37482,37483,37484,37485,37486,37487,37488,37489,37490,37491,37492,37493,37494,37495,37496,37497,37498,37499,37500,37501,37502,37503,37504,37505,37506,37507,37508,37509,37510,37511,37512,37513,37514,37515,37516,37517,37518,37519,37520,37521,37522,37523,37524,37525,37526,37527,37528,37529,37530,37531,37532,37533,37534,37535,37536,37537,37538,37539,37540,37541,37542,37543,37544,37545,37546,37547,37548,37549,37550,37551,37552,37553,37554,37555,37556,37557,37558,37559,37560,37561,37562,37563,37564,37565,37566,37567,37568,37569,37570,37571,37572,37573,37574,37575,37576,37577,37578,37579,37580,37581,37582,37583,37584,37585,37586,37587,37588,37589,37590,37591,37592,37593,37594,37595,37596,37597,37598,37599,37600,37601,37602,37603,37604,37605,37606,37607,37608,37609,37610,37611,37612,37613,37614,37615,37616,37617,37618,37619,37620,37621,37622,37623,37624,37625,37626,37627,37628,37629,37630,37631,37632,37633,37634,37635,37636,37637,37638,37639,37640,37641,37642,37643,37644,37645,37646,37647,37648,37649,37650,37651,37652,37653,37654,37655,37656,37657,37658,37659,37660,37661,37662,37663,37664,37665,37666,37667,37668,37669,37670,37671,37672,37673,37674,37675,37676,37677,37678,37679,37680,37681,37682,37683,37684,37685,37686,37687,37688,37689,37690,37691,37692,37693,37694,37695,37696,37697,37698,37699,37700,37701,37702,37703,37704,37705,37706,37707,37708,37709,37710,37711,37712,37713,37714,37715,37716,37717,37718,37719,37720,37721,37722,37723,37724,37725,37726,37727,37728,37729,37730,37731,37732,37733,37734,37735,37736,37737,37738,37739,37740,37741,37742,37743,37744,37745,37746,37747,37748,37749,37750,37751,37752,37753,37754,37755,37756,37757,37758,37759,37760,37761,37762,37763,37764,37765,37766,37767,37768,37769,37770,37771,37772,37773,37774,37775,37776,37777,37778,37779,37780,37781,37782,37783,37784,37785,37786,37787,37788,37789,37790,37791,37792,37793,37794,37795,37796,37797,37798,37799,37800,37801,37802,37803,37804,37805,37806,37807,37808,37809,37810,37811,37812,37813,37814,37815,37816,37817,37818,37819,37820,37821,37822,37823,37824,37825,37826,37827,37828,37829,37830,37831,37832,37833,37834,37835,37836,37837,37838,37839,37840,37841,37842,37843,37844,37845,37846,37847,37848,37849,37850,37851,37852,37853,37854,37855,37856,37857,37858,37859,37860,37861,37862,37863,37864,37865,37866,37867,37868,37869,37870,37871,37872,37873,37874,37875,37876,37877,37878,37879,37880,37881,37882,37883,37884,37885,37886,37887,37888,37889,37890,37891,37892,37893,37894,37895,37896,37897,37898,37899,37900,37901,37902,37903,37904,37905,37906,37907,37908,37909,37910,37911,37912,37913,37914,37915,37916,37917,37918,37919,37920,37921,37922,37923,37924,37925,37926,37927,37928,37929,37930,37931,37932,37933,37934,37935,37936,37937,37938,37939,37940,37941,37942,37943,37944,37945,37946,37947,37948,37949,37950,37951,37952,37953,37954,37955,37956,37957,37958,37959,37960,37961,37962,37963,37964,37965,37966,37967,37968,37969,37970,37971,37972,37973,37974,37975,37976,37977,37978,37979,37980,37981,37982,37983,37984,37985,37986,37987,37988,37989,37990,37991,37992,37993,37994,37995,37996,37997,37998,37999,38000,38001,38002,38003,38004,38005,38006,38007,38008,38009,38010,38011,38012,38013,38014,38015,38016,38017,38018,38019,38020,38021,38022,38023,38024,38025,38026,38027,38028,38029,38030,38031,38032,38033,38034,38035,38036,38037,38038,38039,38040,38041,38042,38043,38044,38045,38046,38047,38048,38049,38050,38051,38052,38053,38054,38055,38056,38057,38058,38059,38060,38061,38062,38063,38064,38065,38066,38067,38068,38069,38070,38071,38072,38073,38074,38075,38076,38077,38078,38079,38080,38081,38082,38083,38084,38085,38086,38087,38088,38089,38090,38091,38092,38093,38094,38095,38096,38097,38098,38099,38100,38101,38102,38103,38104,38105,38106,38107,38108,38109,38110,38111,38112,38113,38114,38115,38116,38117,38118,38119,38120,38121,38122,38123,38124,38125,38126,38127,38128,38129,38130,38131,38132,38133,38134,38135,38136,38137,38138,38139,38140,38141,38142,38143,38144,38145,38146,38147,38148,38149,38150,38151,38152,38153,38154,38155,38156,38157,38158,38159,38160,38161,38162,38163,38164,38165,38166,38167,38168,38169,38170,38171,38172,38173,38174,38175,38176,38177,38178,38179,38180,38181,38182,38183,38184,38185,38186,38187,38188,38189,38190,38191,38192,38193,38194,38195,38196,38197,38198,38199,38200,38201,38202,38203,38204,38205,38206,38207,38208,38209,38210,38211,38212,38213,38214,38215,38216,38217,38218,38219,38220,38221,38222,38223,38224,38225,38226,38227,38228,38229,38230,38231,38232,38233,38234,38235,38236,38237,38238,38239,38240,38241,38242,38243,38244,38245,38246,38247,38248,38249,38250,38251,38252,38253,38254,38255,38256,38257,38258,38259,38260,38261,38262,38263,38264,38265,38266,38267,38268,38269,38270,38271,38272,38273,38274,38275,38276,38277,38278,38279,38280,38281,38282,38283,38284,38285,38286,38287,38288,38289,38290,38291,38292,38293,38294,38295,38296,38297,38298,38299,38300,38301,38302,38303,38304,38305,38306,38307,38308,38309,38310,38311,38312,38313,38314,38315,38316,38317,38318,38319,38320,38321,38322,38323,38324,38325,38326,38327,38328,38329,38330,38331,38332,38333,38334,38335,38336,38337,38338,38339,38340,38341,38342,38343,38344,38345,38346,38347,38348,38349,38350,38351,38352,38353,38354,38355,38356,38357,38358,38359,38360,38361,38362,38363,38364,38365,38366,38367,38368,38369,38370,38371,38372,38373,38374,38375,38376,38377,38378,38379,38380,38381,38382,38383,38384,38385,38386,38387,38388,38389,38390,38391,38392,38393,38394,38395,38396,38397,38398,38399,38400,38401,38402,38403,38404,38405,38406,38407,38408,38409,38410,38411,38412,38413,38414,38415,38416,38417,38418,38419,38420,38421,38422,38423,38424,38425,38426,38427,38428,38429,38430,38431,38432,38433,38434,38435,38436,38437,38438,38439,38440,38441,38442,38443,38444,38445,38446,38447,38448,38449,38450,38451,38452,38453,38454,38455,38456,38457,38458,38459,38460,38461,38462,38463,38464,38465,38466,38467,38468,38469,38470,38471,38472,38473,38474,38475,38476,38477,38478,38479,38480,38481,38482,38483,38484,38485,38486,38487,38488,38489,38490,38491,38492,38493,38494,38495,38496,38497,38498,38499,38500,38501,38502,38503,38504,38505,38506,38507,38508,38509,38510,38511,38512,38513,38514,38515,38516,38517,38518,38519,38520,38521,38522,38523,38524,38525,38526,38527,38528,38529,38530,38531,38532,38533,38534,38535,38536,38537,38538,38539,38540,38541,38542,38543,38544,38545,38546,38547,38548,38549,38550,38551,38552,38553,38554,38555,38556,38557,38558,38559,38560,38561,38562,38563,38564,38565,38566,38567,38568,38569,38570,38571,38572,38573,38574,38575,38576,38577,38578,38579,38580,38581,38582,38583,38584,38585,38586,38587,38588,38589,38590,38591,38592,38593,38594,38595,38596,38597,38598,38599,38600,38601,38602,38603,38604,38605,38606,38607,38608,38609,38610,38611,38612,38613,38614,38615,38616,38617,38618,38619,38620,38621,38622,38623,38624,38625,38626,38627,38628,38629,38630,38631,38632,38633,38634,38635,38636,38637,38638,38639,38640,38641,38642,38643,38644,38645,38646,38647,38648,38649,38650,38651,38652,38653,38654,38655,38656,38657,38658,38659,38660,38661,38662,38663,38664,38665,38666,38667,38668,38669,38670,38671,38672,38673,38674,38675,38676,38677,38678,38679,38680,38681,38682,38683,38684,38685,38686,38687,38688,38689,38690,38691,38692,38693,38694,38695,38696,38697,38698,38699,38700,38701,38702,38703,38704,38705,38706,38707,38708,38709,38710,38711,38712,38713,38714,38715,38716,38717,38718,38719,38720,38721,38722,38723,38724,38725,38726,38727,38728,38729,38730,38731,38732,38733,38734,38735,38736,38737,38738,38739,38740,38741,38742,38743,38744,38745,38746,38747,38748,38749,38750,38751,38752,38753,38754,38755,38756,38757,38758,38759,38760,38761,38762,38763,38764,38765,38766,38767,38768,38769,38770,38771,38772,38773,38774,38775,38776,38777,38778,38779,38780,38781,38782,38783,38784,38785,38786,38787,38788,38789,38790,38791,38792,38793,38794,38795,38796,38797,38798,38799,38800,38801,38802,38803,38804,38805,38806,38807,38808,38809,38810,38811,38812,38813,38814,38815,38816,38817,38818,38819,38820,38821,38822,38823,38824,38825,38826,38827,38828,38829,38830,38831,38832,38833,38834,38835,38836,38837,38838,38839,38840,38841,38842,38843,38844,38845,38846,38847,38848,38849,38850,38851,38852,38853,38854,38855,38856,38857,38858,38859,38860,38861,38862,38863,38864,38865,38866,38867,38868,38869,38870,38871,38872,38873,38874,38875,38876,38877,38878,38879,38880,38881,38882,38883,38884,38885,38886,38887,38888,38889,38890,38891,38892,38893,38894,38895,38896,38897,38898,38899,38900,38901,38902,38903,38904,38905,38906,38907,38908,38909,38910,38911,38912,38913,38914,38915,38916,38917,38918,38919,38920,38921,38922,38923,38924,38925,38926,38927,38928,38929,38930,38931,38932,38933,38934,38935,38936,38937,38938,38939,38940,38941,38942,38943,38944,38945,38946,38947,38948,38949,38950,38951,38952,38953,38954,38955,38956,38957,38958,38959,38960,38961,38962,38963,38964,38965,38966,38967,38968,38969,38970,38971,38972,38973,38974,38975,38976,38977,38978,38979,38980,38981,38982,38983,38984,38985,38986,38987,38988,38989,38990,38991,38992,38993,38994,38995,38996,38997,38998,38999,39000,39001,39002,39003,39004,39005,39006,39007,39008,39009,39010,39011,39012,39013,39014,39015,39016,39017,39018,39019,39020,39021,39022,39023,39024,39025,39026,39027,39028,39029,39030,39031,39032,39033,39034,39035,39036,39037,39038,39039,39040,39041,39042,39043,39044,39045,39046,39047,39048,39049,39050,39051,39052,39053,39054,39055,39056,39057,39058,39059,39060,39061,39062,39063,39064,39065,39066,39067,39068,39069,39070,39071,39072,39073,39074,39075,39076,39077,39078,39079,39080,39081,39082,39083,39084,39085,39086,39087,39088,39089,39090,39091,39092,39093,39094,39095,39096,39097,39098,39099,39100,39101,39102,39103,39104,39105,39106,39107,39108,39109,39110,39111,39112,39113,39114,39115,39116,39117,39118,39119,39120,39121,39122,39123,39124,39125,39126,39127,39128,39129,39130,39131,39132,39133,39134,39135,39136,39137,39138,39139,39140,39141,39142,39143,39144,39145,39146,39147,39148,39149,39150,39151,39152,39153,39154,39155,39156,39157,39158,39159,39160,39161,39162,39163,39164,39165,39166,39167,39168,39169,39170,39171,39172,39173,39174,39175,39176,39177,39178,39179,39180,39181,39182,39183,39184,39185,39186,39187,39188,39189,39190,39191,39192,39193,39194,39195,39196,39197,39198,39199,39200,39201,39202,39203,39204,39205,39206,39207,39208,39209,39210,39211,39212,39213,39214,39215,39216,39217,39218,39219,39220,39221,39222,39223,39224,39225,39226,39227,39228,39229,39230,39231,39232,39233,39234,39235,39236,39237,39238,39239,39240,39241,39242,39243,39244,39245,39246,39247,39248,39249,39250,39251,39252,39253,39254,39255,39256,39257,39258,39259,39260,39261,39262,39263,39264,39265,39266,39267,39268,39269,39270,39271,39272,39273,39274,39275,39276,39277,39278,39279,39280,39281,39282,39283,39284,39285,39286,39287,39288,39289,39290,39291,39292,39293,39294,39295,39296,39297,39298,39299,39300,39301,39302,39303,39304,39305,39306,39307,39308,39309,39310,39311,39312,39313,39314,39315,39316,39317,39318,39319,39320,39321,39322,39323,39324,39325,39326,39327,39328,39329,39330,39331,39332,39333,39334,39335,39336,39337,39338,39339,39340,39341,39342,39343,39344,39345,39346,39347,39348,39349,39350,39351,39352,39353,39354,39355,39356,39357,39358,39359,39360,39361,39362,39363,39364,39365,39366,39367,39368,39369,39370,39371,39372,39373,39374,39375,39376,39377,39378,39379,39380,39381,39382,39383,39384,39385,39386,39387,39388,39389,39390,39391,39392,39393,39394,39395,39396,39397,39398,39399,39400,39401,39402,39403,39404,39405,39406,39407,39408,39409,39410,39411,39412,39413,39414,39415,39416,39417,39418,39419,39420,39421,39422,39423,39424,39425,39426,39427,39428,39429,39430,39431,39432,39433,39434,39435,39436,39437,39438,39439,39440,39441,39442,39443,39444,39445,39446,39447,39448,39449,39450,39451,39452,39453,39454,39455,39456,39457,39458,39459,39460,39461,39462,39463,39464,39465,39466,39467,39468,39469,39470,39471,39472,39473,39474,39475,39476,39477,39478,39479,39480,39481,39482,39483,39484,39485,39486,39487,39488,39489,39490,39491,39492,39493,39494,39495,39496,39497,39498,39499,39500,39501,39502,39503,39504,39505,39506,39507,39508,39509,39510,39511,39512,39513,39514,39515,39516,39517,39518,39519,39520,39521,39522,39523,39524,39525,39526,39527,39528,39529,39530,39531,39532,39533,39534,39535,39536,39537,39538,39539,39540,39541,39542,39543,39544,39545,39546,39547,39548,39549,39550,39551,39552,39553,39554,39555,39556,39557,39558,39559,39560,39561,39562,39563,39564,39565,39566,39567,39568,39569,39570,39571,39572,39573,39574,39575,39576,39577,39578,39579,39580,39581,39582,39583,39584,39585,39586,39587,39588,39589,39590,39591,39592,39593,39594,39595,39596,39597,39598,39599,39600,39601,39602,39603,39604,39605,39606,39607,39608,39609,39610,39611,39612,39613,39614,39615,39616,39617,39618,39619,39620,39621,39622,39623,39624,39625,39626,39627,39628,39629,39630,39631,39632,39633,39634,39635,39636,39637,39638,39639,39640,39641,39642,39643,39644,39645,39646,39647,39648,39649,39650,39651,39652,39653,39654,39655,39656,39657,39658,39659,39660,39661,39662,39663,39664,39665,39666,39667,39668,39669,39670,39671,39672,39673,39674,39675,39676,39677,39678,39679,39680,39681,39682,39683,39684,39685,39686,39687,39688,39689,39690,39691,39692,39693,39694,39695,39696,39697,39698,39699,39700,39701,39702,39703,39704,39705,39706,39707,39708,39709,39710,39711,39712,39713,39714,39715,39716,39717,39718,39719,39720,39721,39722,39723,39724,39725,39726,39727,39728,39729,39730,39731,39732,39733,39734,39735,39736,39737,39738,39739,39740,39741,39742,39743,39744,39745,39746,39747,39748,39749,39750,39751,39752,39753,39754,39755,39756,39757,39758,39759,39760,39761,39762,39763,39764,39765,39766,39767,39768,39769,39770,39771,39772,39773,39774,39775,39776,39777,39778,39779,39780,39781,39782,39783,39784,39785,39786,39787,39788,39789,39790,39791,39792,39793,39794,39795,39796,39797,39798,39799,39800,39801,39802,39803,39804,39805,39806,39807,39808,39809,39810,39811,39812,39813,39814,39815,39816,39817,39818,39819,39820,39821,39822,39823,39824,39825,39826,39827,39828,39829,39830,39831,39832,39833,39834,39835,39836,39837,39838,39839,39840,39841,39842,39843,39844,39845,39846,39847,39848,39849,39850,39851,39852,39853,39854,39855,39856,39857,39858,39859,39860,39861,39862,39863,39864,39865,39866,39867,39868,39869,39870,39871,39872,39873,39874,39875,39876,39877,39878,39879,39880,39881,39882,39883,39884,39885,39886,39887,39888,39889,39890,39891,39892,39893,39894,39895,39896,39897,39898,39899,39900,39901,39902,39903,39904,39905,39906,39907,39908,39909,39910,39911,39912,39913,39914,39915,39916,39917,39918,39919,39920,39921,39922,39923,39924,39925,39926,39927,39928,39929,39930,39931,39932,39933,39934,39935,39936,39937,39938,39939,39940,39941,39942,39943,39944,39945,39946,39947,39948,39949,39950,39951,39952,39953,39954,39955,39956,39957,39958,39959,39960,39961,39962,39963,39964,39965,39966,39967,39968,39969,39970,39971,39972,39973,39974,39975,39976,39977,39978,39979,39980,39981,39982,39983,39984,39985,39986,39987,39988,39989,39990,39991,39992,39993,39994,39995,39996,39997,39998,39999,40000,40001,40002,40003,40004,40005,40006,40007,40008,40009,40010,40011,40012,40013,40014,40015,40016,40017,40018,40019,40020,40021,40022,40023,40024,40025,40026,40027,40028,40029,40030,40031,40032,40033,40034,40035,40036,40037,40038,40039,40040,40041,40042,40043,40044,40045,40046,40047,40048,40049,40050,40051,40052,40053,40054,40055,40056,40057,40058,40059,40060,40061,40062,40063,40064,40065,40066,40067,40068,40069,40070,40071,40072,40073,40074,40075,40076,40077,40078,40079,40080,40081,40082,40083,40084,40085,40086,40087,40088,40089,40090,40091,40092,40093,40094,40095,40096,40097,40098,40099,40100,40101,40102,40103,40104,40105,40106,40107,40108,40109,40110,40111,40112,40113,40114,40115,40116,40117,40118,40119,40120,40121,40122,40123,40124,40125,40126,40127,40128,40129,40130,40131,40132,40133,40134,40135,40136,40137,40138,40139,40140,40141,40142,40143,40144,40145,40146,40147,40148,40149,40150,40151,40152,40153,40154,40155,40156,40157,40158,40159,40160,40161,40162,40163,40164,40165,40166,40167,40168,40169,40170,40171,40172,40173,40174,40175,40176,40177,40178,40179,40180,40181,40182,40183,40184,40185,40186,40187,40188,40189,40190,40191,40192,40193,40194,40195,40196,40197,40198,40199,40200,40201,40202,40203,40204,40205,40206,40207,40208,40209,40210,40211,40212,40213,40214,40215,40216,40217,40218,40219,40220,40221,40222,40223,40224,40225,40226,40227,40228,40229,40230,40231,40232,40233,40234,40235,40236,40237,40238,40239,40240,40241,40242,40243,40244,40245,40246,40247,40248,40249,40250,40251,40252,40253,40254,40255,40256,40257,40258,40259,40260,40261,40262,40263,40264,40265,40266,40267,40268,40269,40270,40271,40272,40273,40274,40275,40276,40277,40278,40279,40280,40281,40282,40283,40284,40285,40286,40287,40288,40289,40290,40291,40292,40293,40294,40295,40296,40297,40298,40299,40300,40301,40302,40303,40304,40305,40306,40307,40308,40309,40310,40311,40312,40313,40314,40315,40316,40317,40318,40319,40320,40321,40322,40323,40324,40325,40326,40327,40328,40329,40330,40331,40332,40333,40334,40335,40336,40337,40338,40339,40340,40341,40342,40343,40344,40345,40346,40347,40348,40349,40350,40351,40352,40353,40354,40355,40356,40357,40358,40359,40360,40361,40362,40363,40364,40365,40366,40367,40368,40369,40370,40371,40372,40373,40374,40375,40376,40377,40378,40379,40380,40381,40382,40383,40384,40385,40386,40387,40388,40389,40390,40391,40392,40393,40394,40395,40396,40397,40398,40399,40400,40401,40402,40403,40404,40405,40406,40407,40408,40409,40410,40411,40412,40413,40414,40415,40416,40417,40418,40419,40420,40421,40422,40423,40424,40425,40426,40427,40428,40429,40430,40431,40432,40433,40434,40435,40436,40437,40438,40439,40440,40441,40442,40443,40444,40445,40446,40447,40448,40449,40450,40451,40452,40453,40454,40455,40456,40457,40458,40459,40460,40461,40462,40463,40464,40465,40466,40467,40468,40469,40470,40471,40472,40473,40474,40475,40476,40477,40478,40479,40480,40481,40482,40483,40484,40485,40486,40487,40488,40489,40490,40491,40492,40493,40494,40495,40496,40497,40498,40499,40500,40501,40502,40503,40504,40505,40506,40507,40508,40509,40510,40511,40512,40513,40514,40515,40516,40517,40518,40519,40520,40521,40522,40523,40524,40525,40526,40527,40528,40529,40530,40531,40532,40533,40534,40535,40536,40537,40538,40539,40540,40541,40542,40543,40544,40545,40546,40547,40548,40549,40550,40551,40552,40553,40554,40555,40556,40557,40558,40559,40560,40561,40562,40563,40564,40565,40566,40567,40568,40569,40570,40571,40572,40573,40574,40575,40576,40577,40578,40579,40580,40581,40582,40583,40584,40585,40586,40587,40588,40589,40590,40591,40592,40593,40594,40595,40596,40597,40598,40599,40600,40601,40602,40603,40604,40605,40606,40607,40608,40609,40610,40611,40612,40613,40614,40615,40616,40617,40618,40619,40620,40621,40622,40623,40624,40625,40626,40627,40628,40629,40630,40631,40632,40633,40634,40635,40636,40637,40638,40639,40640,40641,40642,40643,40644,40645,40646,40647,40648,40649,40650,40651,40652,40653,40654,40655,40656,40657,40658,40659,40660,40661,40662,40663,40664,40665,40666,40667,40668,40669,40670,40671,40672,40673,40674,40675,40676,40677,40678,40679,40680,40681,40682,40683,40684,40685,40686,40687,40688,40689,40690,40691,40692,40693,40694,40695,40696,40697,40698,40699,40700,40701,40702,40703,40704,40705,40706,40707,40708,40709,40710,40711,40712,40713,40714,40715,40716,40717,40718,40719,40720,40721,40722,40723,40724,40725,40726,40727,40728,40729,40730,40731,40732,40733,40734,40735,40736,40737,40738,40739,40740,40741,40742,40743,40744,40745,40746,40747,40748,40749,40750,40751,40752,40753,40754,40755,40756,40757,40758,40759,40760,40761,40762,40763,40764,40765,40766,40767,40768,40769,40770,40771,40772,40773,40774,40775,40776,40777,40778,40779,40780,40781,40782,40783,40784,40785,40786,40787,40788,40789,40790,40791,40792,40793,40794,40795,40796,40797,40798,40799,40800,40801,40802,40803,40804,40805,40806,40807,40808,40809,40810,40811,40812,40813,40814,40815,40816,40817,40818,40819,40820,40821,40822,40823,40824,40825,40826,40827,40828,40829,40830,40831,40832,40833,40834,40835,40836,40837,40838,40839,40840,40841,40842,40843,40844,40845,40846,40847,40848,40849,40850,40851,40852,40853,40854,40855,40856,40857,40858,40859,40860,40861,40862,40863,40864,40865,40866,40867,40868,40869,57344,57345,57346,57347,57348,57349,57350,57351,57352,57353,57354,57355,57356,57357,57358,57359,57360,57361,57362,57363,57364,57365,57366,57367,57368,57369,57370,57371,57372,57373,57374,57375,57376,57377,57378,57379,57380,57381,57382,57383,57384,57385,57386,57387,57388,57389,57390,57391,57392,57393,57394,57395,57396,57397,57398,57399,57400,57401,57402,57403,57404,57405,57406,57407,57408,57409,57410,57411,57412,57413,57414,57415,57416,57417,57418,57419,57420,57421,57422,57423,57424,57425,57426,57427,57428,57429,57430,57431,57432,57433,57434,57435,57436,57437,57438,57439,57440,57441,57442,57443,57444,57445,57446,57447,57448,57449,57450,57451,57452,57453,57454,57455,57456,57457,57458,57459,57460,57461,57462,57463,57464,57465,57466,57467,57468,57469,57470,57471,57472,57473,57474,57475,57476,57477,57478,57479,57480,57481,57482,57483,57484,57485,57486,57487,57488,57489,57490,57491,57492,57493,57494,57495,57496,57497,57498,57499,57500,57501,57502,57503,57504,57505,57506,57507,57508,57509,57510,57511,57512,57513,57514,57515,57516,57517,57518,57519,57520,57521,57522,57523,57524,57525,57526,57527,57528,57529,57530,57531,57532,57533,57534,57535,57536,57537,57538,57539,57540,57541,57542,57543,57544,57545,57546,57547,57548,57549,57550,57551,57552,57553,57554,57555,57556,57557,57558,57559,57560,57561,57562,57563,57564,57565,57566,57567,57568,57569,57570,57571,57572,57573,57574,57575,57576,57577,57578,57579,57580,57581,57582,57583,57584,57585,57586,57587,57588,57589,57590,57591,57592,57593,57594,57595,57596,57597,57598,57599,57600,57601,57602,57603,57604,57605,57606,57607,57608,57609,57610,57611,57612,57613,57614,57615,57616,57617,57618,57619,57620,57621,57622,57623,57624,57625,57626,57627,57628,57629,57630,57631,57632,57633,57634,57635,57636,57637,57638,57639,57640,57641,57642,57643,57644,57645,57646,57647,57648,57649,57650,57651,57652,57653,57654,57655,57656,57657,57658,57659,57660,57661,57662,57663,57664,57665,57666,57667,57668,57669,57670,57671,57672,57673,57674,57675,57676,57677,57678,57679,57680,57681,57682,57683,57684,57685,57686,57687,57688,57689,57690,57691,57692,57693,57694,57695,57696,57697,57698,57699,57700,57701,57702,57703,57704,57705,57706,57707,57708,57709,57710,57711,57712,57713,57714,57715,57716,57717,57718,57719,57720,57721,57722,57723,57724,57725,57726,57727,57728,57729,57730,57731,57732,57733,57734,57735,57736,57737,57738,57739,57740,57741,57742,57743,57744,57745,57746,57747,57748,57749,57750,57751,57752,57753,57754,57755,57756,57757,57758,57759,57760,57761,57762,57763,57764,57765,57766,57767,57768,57769,57770,57771,57772,57773,57774,57775,57776,57777,57778,57779,57780,57781,57782,57783,57784,57785,57786,57787,57788,57789,57790,57791,57792,57793,57794,57795,57796,57797,57798,57799,57800,57801,57802,57803,57804,57805,57806,57807,57808,57809,57810,57811,57812,57813,57814,57815,57816,57817,57818,57819,57820,57821,57822,57823,57824,57825,57826,57827,57828,57829,57830,57831,57832,57833,57834,57835,57836,57837,57838,57839,57840,57841,57842,57843,57844,57845,57846,57847,57848,57849,57850,57851,57852,57853,57854,57855,57856,57857,57858,57859,57860,57861,57862,57863,57864,57865,57866,57867,57868,57869,57870,57871,57872,57873,57874,57875,57876,57877,57878,57879,57880,57881,57882,57883,57884,57885,57886,57887,57888,57889,57890,57891,57892,57893,57894,57895,57896,57897,57898,57899,57900,57901,57902,57903,57904,57905,57906,57907,57908,57909,57910,57911,57912,57913,57914,57915,57916,57917,57918,57919,57920,57921,57922,57923,57924,57925,57926,57927,57928,57929,57930,57931,57932,57933,57934,57935,57936,57937,57938,57939,57940,57941,57942,57943,57944,57945,57946,57947,57948,57949,57950,57951,57952,57953,57954,57955,57956,57957,57958,57959,57960,57961,57962,57963,57964,57965,57966,57967,57968,57969,57970,57971,57972,57973,57974,57975,57976,57977,57978,57979,57980,57981,57982,57983,57984,57985,57986,57987,57988,57989,57990,57991,57992,57993,57994,57995,57996,57997,57998,57999,58000,58001,58002,58003,58004,58005,58006,58007,58008,58009,58010,58011,58012,58013,58014,58015,58016,58017,58018,58019,58020,58021,58022,58023,58024,58025,58026,58027,58028,58029,58030,58031,58032,58033,58034,58035,58036,58037,58038,58039,58040,58041,58042,58043,58044,58045,58046,58047,58048,58049,58050,58051,58052,58053,58054,58055,58056,58057,58058,58059,58060,58061,58062,58063,58064,58065,58066,58067,58068,58069,58070,58071,58072,58073,58074,58075,58076,58077,58078,58079,58080,58081,58082,58083,58084,58085,58086,58087,58088,58089,58090,58091,58092,58093,58094,58095,58096,58097,58098,58099,58100,58101,58102,58103,58104,58105,58106,58107,58108,58109,58110,58111,58112,58113,58114,58115,58116,58117,58118,58119,58120,58121,58122,58123,58124,58125,58126,58127,58128,58129,58130,58131,58132,58133,58134,58135,58136,58137,58138,58139,58140,58141,58142,58143,58144,58145,58146,58147,58148,58149,58150,58151,58152,58153,58154,58155,58156,58157,58158,58159,58160,58161,58162,58163,58164,58165,58166,58167,58168,58169,58170,58171,58172,58173,58174,58175,58176,58177,58178,58179,58180,58181,58182,58183,58184,58185,58186,58187,58188,58189,58190,58191,58192,58193,58194,58195,58196,58197,58198,58199,58200,58201,58202,58203,58204,58205,58206,58207,58208,58209,58210,58211,58212,58213,58214,58215,58216,58217,58218,58219,58220,58221,58222,58223,58224,58225,58226,58227,58228,58229,58230,58231,58232,58233,58234,58235,58236,58237,58238,58239,58240,58241,58242,58243,58244,58245,58246,58247,58248,58249,58250,58251,58252,58253,58254,58255,58256,58257,58258,58259,58260,58261,58262,58263,58264,58265,58266,58267,58268,58269,58270,58271,58272,58273,58274,58275,58276,58277,58278,58279,58280,58281,58282,58283,58284,58285,58286,58287,58288,58289,58290,58291,58292,58293,58294,58295,58296,58297,58298,58299,58300,58301,58302,58303,58304,58305,58306,58307,58308,58309,58310,58311,58312,58313,58314,58315,58316,58317,58318,58319,58320,58321,58322,58323,58324,58325,58326,58327,58328,58329,58330,58331,58332,58333,58334,58335,58336,58337,58338,58339,58340,58341,58342,58343,58344,58345,58346,58347,58348,58349,58350,58351,58352,58353,58354,58355,58356,58357,58358,58359,58360,58361,58362,58363,58364,58365,58366,58367,58368,58369,58370,58371,58372,58373,58374,58375,58376,58377,58378,58379,58380,58381,58382,58383,58384,58385,58386,58387,58388,58389,58390,58391,58392,58393,58394,58395,58396,58397,58398,58399,58400,58401,58402,58403,58404,58405,58406,58407,58408,58409,58410,58411,58412,58413,58414,58415,58416,58417,58418,58419,58420,58421,58422,58423,58424,58425,58426,58427,58428,58429,58430,58431,58432,58433,58434,58435,58436,58437,58438,58439,58440,58441,58442,58443,58444,58445,58446,58447,58448,58449,58450,58451,58452,58453,58454,58455,58456,58457,58458,58459,58460,58461,58462,58463,58464,58465,58466,58467,58468,58469,58470,58471,58472,58473,58474,58475,58476,58477,58478,58479,58480,58481,58482,58483,58484,58485,58486,58487,58488,58489,58490,58491,58492,58493,58494,58495,58496,58497,58498,58499,58500,58501,58502,58503,58504,58505,58506,58507,58508,58509,58510,58511,58512,58513,58514,58515,58516,58517,58518,58519,58520,58521,58522,58523,58524,58525,58526,58527,58528,58529,58530,58531,58532,58533,58534,58535,58536,58537,58538,58539,58540,58541,58542,58543,58544,58545,58546,58547,58548,58549,58550,58551,58552,58553,58554,58555,58556,58557,58558,58559,58560,58561,58562,58563,58564,58565,58566,58567,58568,58569,58570,58571,58572,58573,58574,58575,58576,58577,58578,58579,58580,58581,58582,58583,58584,58585,58586,58587,58588,58589,58590,58591,58592,58593,58594,58595,58596,58597,58598,58599,58600,58601,58602,58603,58604,58605,58606,58607,58608,58609,58610,58611,58612,58613,58614,58615,58616,58617,58618,58619,58620,58621,58622,58623,58624,58625,58626,58627,58628,58629,58630,58631,58632,58633,58634,58635,58636,58637,58638,58639,58640,58641,58642,58643,58644,58645,58646,58647,58648,58649,58650,58651,58652,58653,58654,58655,58656,58657,58658,58659,58660,58661,58662,58663,58664,58665,58666,58667,58668,58669,58670,58671,58672,58673,58674,58675,58676,58677,58678,58679,58680,58681,58682,58683,58684,58685,58686,58687,58688,58689,58690,58691,58692,58693,58694,58695,58696,58697,58698,58699,58700,58701,58702,58703,58704,58705,58706,58707,58708,58709,58710,58711,58712,58713,58714,58715,58716,58717,58718,58719,58720,58721,58722,58723,58724,58725,58726,58727,58728,58729,58730,58731,58732,58733,58734,58735,58736,58737,58738,58739,58740,58741,58742,58743,58744,58745,58746,58747,58748,58749,58750,58751,58752,58753,58754,58755,58756,58757,58758,58759,58760,58761,58762,58763,58764,58765,58766,58767,58768,58769,58770,58771,58772,58773,58774,58775,58776,58777,58778,58779,58780,58781,58782,58783,58784,58785,58786,58787,58788,58789,58790,58791,58792,58793,58794,58795,58796,58797,58798,58799,58800,58801,58802,58803,58804,58805,58806,58807,58808,58809,58810,58811,58812,58813,58814,58815,58816,58817,58818,58819,58820,58821,58822,58823,58824,58825,58826,58827,58828,58829,58830,58831,58832,58833,58834,58835,58836,58837,58838,58839,58840,58841,58842,58843,58844,58845,58846,58847,58848,58849,58850,58851,58852,58853,58854,58855,58856,58857,58858,58859,58860,58861,58862,58863,58864,58865,58866,58867,58868,58869,58870,58871,58872,58873,58874,58875,58876,58877,58878,58879,58880,58881,58882,58883,58884,58885,58886,58887,58888,58889,58890,58891,58892,58893,58894,58895,58896,58897,58898,58899,58900,58901,58902,58903,58904,58905,58906,58907,58908,58909,58910,58911,58912,58913,58914,58915,58916,58917,58918,58919,58920,58921,58922,58923,58924,58925,58926,58927,58928,58929,58930,58931,58932,58933,58934,58935,58936,58937,58938,58939,58940,58941,58942,58943,58944,58945,58946,58947,58948,58949,58950,58951,58952,58953,58954,58955,58956,58957,58958,58959,58960,58961,58962,58963,58964,58965,58966,58967,58968,58969,58970,58971,58972,58973,58974,58975,58976,58977,58978,58979,58980,58981,58982,58983,58984,58985,58986,58987,58988,58989,58990,58991,58992,58993,58994,58995,58996,58997,58998,58999,59000,59001,59002,59003,59004,59005,59006,59007,59008,59009,59010,59011,59012,59013,59014,59015,59016,59017,59018,59019,59020,59021,59022,59023,59024,59025,59026,59027,59028,59029,59030,59031,59032,59033,59034,59035,59036,59037,59038,59039,59040,59041,59042,59043,59044,59045,59046,59047,59048,59049,59050,59051,59052,59053,59054,59055,59056,59057,59058,59059,59060,59061,59062,59063,59064,59065,59066,59067,59068,59069,59070,59071,59072,59073,59074,59075,59076,59077,59078,59079,59080,59081,59082,59083,59084,59085,59086,59087,59088,59089,59090,59091,59092,59093,59094,59095,59096,59097,59098,59099,59100,59101,59102,59103,59104,59105,59106,59107,59108,59109,59110,59111,59112,59113,59114,59115,59116,59117,59118,59119,59120,59121,59122,59123,59124,59125,59126,59127,59128,59129,59130,59131,59132,59133,59134,59135,59136,59137,59138,59139,59140,59141,59142,59143,59144,59145,59146,59147,59148,59149,59150,59151,59152,59153,59154,59155,59156,59157,59158,59159,59160,59161,59162,59163,59164,59165,59166,59167,59168,59169,59170,59171,59172,59173,59174,59175,59176,59177,59178,59179,59180,59181,59182,59183,59184,59185,59186,59187,59188,59189,59190,59191,59192,59193,59194,59195,59196,59197,59198,59199,59200,59201,59202,59203,59204,59205,59206,59207,59208,59209,59210,59211,59212,59213,59214,59215,59216,59217,59218,59219,59220,59221,59222,59223,59224,59225,59226,59227,59228,59229,59230,59231,59232,59233,59234,59235,59236,59237,59238,59239,59240,59241,59242,59243,59244,59245,59246,59247,59248,59249,59250,59251,59252,59253,59254,59255,59256,59257,59258,59259,59260,59261,59262,59263,59264,59265,59266,59267,59268,59269,59270,59271,59272,59273,59274,59275,59276,59277,59278,59279,59280,59281,59282,59283,59284,59285,59286,59287,59288,59289,59290,59291,59292,59293,59294,59295,59296,59297,59298,59299,59300,59301,59302,59303,59304,59305,59306,59307,59308,59309,59310,59311,59312,59313,59314,59315,59316,59317,59318,59319,59320,59321,59322,59323,59324,59325,59326,59327,59328,59329,59330,59331,59332,59333,59334,59335,59336,59337,59338,59339,59340,59341,59342,59343,59344,59345,59346,59347,59348,59349,59350,59351,59352,59353,59354,59355,59356,59357,59358,59359,59360,59361,59362,59363,59364,59365,59366,59367,59368,59369,59370,59371,59372,59373,59374,59375,59376,59377,59378,59379,59380,59381,59382,59383,59384,59385,59386,59387,59388,59389,59390,59391,59392,59393,59394,59395,59396,59397,59398,59399,59400,59401,59402,59403,59404,59405,59406,59407,59408,59409,59410,59411,59412,59413,59414,59415,59416,59417,59418,59419,59420,59421,59422,59423,59424,59425,59426,59427,59428,59429,59430,59431,59432,59433,59434,59435,59436,59437,59438,59439,59440,59441,59442,59443,59444,59445,59446,59447,59448,59449,59450,59451,59452,59453,59454,59455,59456,59457,59458,59459,59460,59461,59462,59463,59464,59465,59466,59467,59468,59469,59470,59471,59472,59473,59474,59475,59476,59477,59478,59479,59480,59481,59482,59483,59484,59485,59486,59487,59488,59489,59490,59491,59492,63788,63865,63893,63975,63985,64012,64013,64014,64015,64017,64019,64020,64024,64031,64032,64033,64035,64036,64039,64040,64041,65072,65073,65075,65076,65077,65078,65079,65080,65081,65082,65083,65084,65085,65086,65087,65088,65089,65090,65091,65092,65097,65098,65099,65100,65101,65102,65103,65104,65105,65106,65108,65109,65110,65111,65113,65114,65115,65116,65117,65118,65119,65120,65121,65122,65123,65124,65125,65126,65128,65129,65130,65131,65281,65282,65283,65284,65285,65286,65287,65288,65289,65290,65291,65292,65293,65294,65295,65296,65297,65298,65299,65300,65301,65302,65303,65304,65305,65306,65307,65308,65309,65310,65311,65312,65313,65314,65315,65316,65317,65318,65319,65320,65321,65322,65323,65324,65325,65326,65327,65328,65329,65330,65331,65332,65333,65334,65335,65336,65337,65338,65339,65340,65341,65342,65343,65344,65345,65346,65347,65348,65349,65350,65351,65352,65353,65354,65355,65356,65357,65358,65359,65360,65361,65362,65363,65364,65365,65366,65367,65368,65369,65370,65371,65372,65373,65374,65504,65505,65506,65507,65508,65509 };
unsigned short unicode_to_gbk1_vals[] = { 41448,41452,41383,41443,41408,41380,41409,43172,43170,43176,43174,43194,43180,43178,43184,43182,41410,43188,43186,43193,43169,43173,43175,43177,43197,43198,43181,43185,43171,43179,43183,43187,43189,43190,43191,43192,43195,43200,41382,41381,43072,43073,43074,42657,42658,42659,42660,42661,42662,42663,42664,42665,42666,42667,42668,42669,42670,42671,42672,42673,42674,42675,42676,42677,42678,42679,42680,42689,42690,42691,42692,42693,42694,42695,42696,42697,42698,42699,42700,42701,42702,42703,42704,42705,42706,42707,42708,42709,42710,42711,42712,42919,42913,42914,42915,42916,42917,42918,42920,42921,42922,42923,42924,42925,42926,42927,42928,42929,42930,42931,42932,42933,42934,42935,42936,42937,42938,42939,42940,42941,42942,42943,42944,42945,42961,42962,42963,42964,42965,42966,42968,42969,42970,42971,42972,42973,42974,42975,42976,42977,42978,42979,42980,42981,42982,42983,42984,42985,42986,42987,42988,42989,42990,42991,42992,42993,42967,43356,43075,41386,43076,41388,41390,41391,41392,41393,43077,41389,41451,41444,41445,43078,41465,  128,41446,43079,43080,41453,43353,41713,41714,41715,41716,41717,41718,41719,41720,41721,41722,41723,41724,41633,41634,41635,41636,41637,41638,41639,41640,41641,41642,41467,41468,41466,41469,43081,43082,43083,43084,41418,41415,41414,43085,41420,41432,41438,43086,41423,43087,41422,41412,41413,41417,41416,41426,41427,41440,41439,41411,41419,41431,41430,41429,43088,41433,41428,41436,41437,43089,43090,41434,41435,43154,41425,41421,43091,41424,41689,41690,41691,41692,41693,41694,41695,41696,41697,41698,41669,41670,41671,41672,41673,41674,41675,41676,41677,41678,41679,41680,41681,41682,41683,41684,41685,41686,41687,41688,41649,41650,41651,41652,41653,41654,41655,41656,41657,41658,41659,41660,41661,41662,41663,41664,41665,41666,41667,41668,43428,43429,43430,43431,43432,43433,43434,43435,43436,43437,43438,43439,43440,43441,43442,43443,43444,43445,43446,43447,43448,43449,43450,43451,43452,43453,43454,43455,43456,43457,43458,43459,43460,43461,43462,43463,43464,43465,43466,43467,43468,43469,43470,43471,43472,43473,43474,43475,43476,43477,43478,43479,43480,43481,43482,43483,43484,43485,43486,43487,43488,43489,43490,43491,43492,43493,43494,43495,43496,43497,43498,43499,43500,43501,43502,43503,43092,43093,43094,43095,43096,43097,43098,43099,43100,43101,43102,43103,43104,43105,43106,43107,43108,43109,43110,43111,43112,43113,43114,43115,43116,43117,43118,43119,43120,43121,43122,43123,43124,43125,43126,43127,43128,43129,43130,43131,43132,43133,43134,43136,43137,43138,43139,43140,43141,43142,43143,43144,43145,43146,41462,41461,41464,41463,43147,43148,41460,41459,41456,41458,41457,43149,43150,43151,43152,41455,43153,41442,41441,41454,41377,41378,41379,41384,41385,43365,43414,41396,41397,41398,41399,41400,41401,41402,41403,41406,41407,43155,41470,41394,41395,41404,41405,43156,43157,43328,43329,43330,43331,43332,43333,43334,43335,43336,42145,42146,42147,42148,42149,42150,42151,42152,42153,42154,42155,42156,42157,42158,42159,42160,42161,42162,42163,42164,42165,42166,42167,42168,42169,42170,42171,42172,42173,42174,42175,42176,42177,42178,42179,42180,42181,42182,42183,42184,42185,42186,42187,42188,42189,42190,42191,42192,42193,42194,42195,42196,42197,42198,42199,42200,42201,42202,42203,42204,42205,42206,42207,42208,42209,42210,42211,42212,42213,42214,42215,42216,42217,42218,42219,42220,42221,42222,42223,42224,42225,42226,42227,43361,43362,43366,43367,42401,42402,42403,42404,42405,42406,42407,42408,42409,42410,42411,42412,42413,42414,42415,42416,42417,42418,42419,42420,42421,42422,42423,42424,42425,42426,42427,42428,42429,42430,42431,42432,42433,42434,42435,42436,42437,42438,42439,42440,42441,42442,42443,42444,42445,42446,42447,42448,42449,42450,42451,42452,42453,42454,42455,42456,42457,42458,42459,42460,42461,42462,42463,42464,42465,42466,42467,42468,42469,42470,42471,42472,42473,42474,42475,42476,42477,42478,42479,42480,42481,42482,42483,42484,42485,42486,43360,43363,43364,43205,43206,43207,43208,43209,43210,43211,43212,43213,43214,43215,43216,43217,43218,43219,43220,43221,43222,43223,43224,43225,43226,43227,43228,43229,43230,43231,43232,43233,43234,43235,43236,43237,43238,43239,43240,43241,41701,41702,41703,41704,41705,41706,41707,41708,41709,41710,43354,43337,43338,43339,43340,43341,43342,43343,43344,43345,43346,43347,43348,53947,46753,33088,50911,33089,33090,33091,52722,54729,51453,51663,53186,55458,45755,54251,33092,55460,46067,33093,55208,51154,55463,51904,33094,51184,45563,53941,46292,46763,52159,55465,33095,33096,33097,46762,33098,49597,53711,33099,51621,55469,33100,47350,53694,58332,54992,33101,33102,47073,33103,46254,33104,49625,33105,55484,33106,52712,46500,52906,55031,33107,49398,48857,55471,33108,33109,33110,50379,33111,48835,33112,55473,50100,53989,33113,54958,52954,54695,47861,47014,49366,33114,50873,50642,51143,33115,47572,33116,46027,53970,33117,33118,55487,48837,50930,53938,53168,53223,33119,33120,33121,33122,51945,33123,33124,55488,33125,33126,33127,33128,33129,33130,49906,49874,33131,51433,33132,33133,33134,33135,33136,33137,33138,33139,33140,33141,51116,33142,33143,33144,33145,33146,33147,33148,49611,33149,54248,54777,33150,51906,46846,55457,54234,49143,33152,54470,48037,55489,52965,48814,33153,33154,55464,33155,53703,53417,33156,33157,33158,55485,55791,52726,49082,33159,48571,47781,53984,45818,47840,50358,33160,53229,48809,52644,49601,33161,33162,33163,51159,55793,33164,55796,33165,33166,33167,33168,51403,55529,33169,33170,33171,53978,51890,51402,55532,55530,55494,48630,50893,46064,33172,55531,48625,48617,33173,51412,46291,33174,33175,49880,33176,45782,55248,51915,52219,54732,47286,53193,33177,33178,33179,55770,55536,51114,33180,55534,33181,46330,49646,53972,33182,33183,55533,33184,53959,55535,50119,33185,33186,33187,53750,33188,55001,55538,33189,55541,48382,48347,33190,33191,33192,51406,33193,47069,33194,47042,33195,50931,33196,33197,33198,33199,33200,33201,33202,55544,53953,33203,33204,52969,48319,47100,47013,53469,33205,33206,33207,33208,33209,55002,54213,48111,48097,55537,33210,33211,51617,52912,46251,33212,55539,33213,51659,55542,49879,55543,33214,33215,52913,55545,33216,33217,33218,45742,47552,33219,55715,33220,45289,33221,49638,33222,51692,33223,52165,33224,52166,55716,33225,33226,33227,33228,33229,46568,33230,33231,46507,33232,33233,33234,33235,33236,33237,52923,46541,55201,55284,54227,33238,52453,33239,47822,33240,55714,55772,54240,55549,47088,55287,55550,55546,55713,50403,33241,33242,54198,55540,55773,33243,55547,33244,50661,33245,33246,49360,33247,33248,53744,45275,33249,33250,48337,55718,33251,55717,33252,33253,33254,33255,55724,55726,33256,55723,51897,33257,33258,33259,55721,54966,33260,33261,33262,46046,55720,33263,49405,33264,51916,33265,55722,33266,55719,33267,33268,55728,33269,33270,46769,33271,33272,33273,47529,33274,53952,33275,33276,53184,33277,33278,49858,33344,48580,54764,45792,51144,49131,55725,33345,55727,33346,52970,47854,33347,33348,33349,33350,33351,51158,33352,33353,33354,33355,33356,33357,33358,33359,33360,45539,33361,33362,33363,46297,46829,55732,33364,33365,33366,33367,49057,33368,33369,33370,55774,51150,49406,55736,33371,33372,33373,33374,33375,52183,47101,33376,55733,33377,55735,45475,54241,55737,33378,53445,33379,55734,33380,33381,55729,33382,55730,49577,55731,33383,33384,48371,53470,47273,33385,48867,33386,55741,33387,33388,33389,33390,55738,33391,45235,33392,33393,33394,55746,33395,33396,33397,33398,33399,33400,33401,33402,33403,33404,33405,33406,33408,55748,45494,33409,55743,33410,33411,46521,33412,48883,33413,33414,33415,52424,47858,53968,33416,55747,33417,33418,48616,33419,45995,33420,33421,33422,55749,48875,33423,55750,55739,50399,33424,55742,55745,55744,33425,33426,33427,33428,33429,33430,33431,33432,33433,33434,33435,54702,33436,54965,33437,51171,33438,33439,33440,33441,55752,33442,33443,33444,48345,55754,33445,33446,33447,55740,33448,55755,50859,33449,33450,33451,33452,33453,55753,33454,33455,33456,33457,55286,33458,52643,33459,33460,33461,33462,33463,33464,33465,33466,48545,33467,33468,33469,33470,33471,33472,55756,33473,33474,33475,33476,33477,33478,33479,33480,33481,50620,52661,33482,33483,33484,55757,33485,33486,55751,45989,49150,33487,33488,33489,33490,47285,33491,33492,49404,33493,33494,33495,33496,45304,33497,33498,33499,33500,33501,33502,33503,33504,33505,33506,33507,33508,33509,33510,33511,33512,33513,33514,33515,33516,33517,46326,33518,55758,33519,55759,46242,55760,33520,33521,46303,33522,33523,33524,33525,33526,45249,33527,33528,33529,33530,33531,33532,33533,55761,51637,33534,33600,33601,33602,33603,33604,33605,33606,33607,33608,33609,33610,33611,33612,33613,33614,33615,33616,33617,53233,33618,33619,33620,33621,33622,33623,55762,33624,33625,33626,49605,33627,33628,33629,33630,33631,33632,33633,33634,33635,33636,33637,55766,51630,33638,33639,33640,33641,55765,55764,55767,33642,33643,33644,33645,52187,33646,48553,33647,33648,33649,33650,33651,50855,33652,33653,33654,33655,33656,33657,33658,33659,33660,33661,55763,55768,33662,33664,33665,55769,33666,33667,33668,33669,33670,33671,51429,33672,33673,33674,33675,33676,33677,33678,33679,33680,33681,33682,33683,33684,33685,49372,33686,33687,33688,33689,33690,33691,33692,33693,33694,33695,33696,33697,33698,33699,33700,33701,33702,33703,33704,33705,33706,33707,33708,33709,33710,33711,33712,33713,33714,46841,55459,54474,33715,54442,53462,46052,54743,33716,53192,47586,33717,49099,33718,50146,33719,33720,33721,46802,33722,33723,52675,55790,55792,33724,33725,33726,46515,33727,46773,33728,33729,33730,33731,33732,48804,33733,33734,51435,33735,33736,51371,33737,33738,45259,47531,49657,55778,33739,49340,47538,33740,47576,53451,45560,50916,48863,46564,55240,33741,53752,48358,51934,33742,33743,48317,55782,55527,33744,33745,50394,33746,33747,47316,51389,33748,33749,45793,54489,33750,33751,33752,33753,50096,33754,33755,50145,55970,51423,33756,53428,33757,48892,50601,33758,33759,33760,47578,33761,55971,33762,54441,55972,33763,33764,33765,33766,33767,55803,46764,33768,33769,47083,45561,55804,46053,48886,33770,49142,53937,49380,33771,33772,33773,46771,55806,55805,33774,33775,48827,33776,33777,33778,50912,33779,55228,55969,33780,49593,33781,46578,49640,33782,33783,48373,33784,46293,33785,33786,33787,33788,33789,33790,33856,33857,33858,49629,33859,50429,33860,33861,48312,47026,33862,33863,47087,33864,33865,33866,33867,33868,33869,55788,33870,50878,33871,49069,48075,33872,33873,46538,33874,56265,53463,33875,52665,45244,46070,48119,56266,47791,33876,54500,46518,46579,55510,51408,33877,33878,47062,51152,55511,33879,49071,33880,33881,56251,55512,33882,33883,53452,48046,33884,33885,33886,60350,49616,49653,54514,47317,46260,33887,46069,33888,33889,51646,33890,33891,33892,50640,33893,33894,33895,50649,49403,33896,45552,33897,55513,47566,33898,46525,33899,33900,55514,33901,33902,54982,52130,51375,51634,46284,49100,33903,47604,33904,55515,55516,46823,48321,52458,33905,33906,33907,33908,33909,33910,53239,33911,55517,51120,33912,33913,47568,48547,33914,33915,52446,33916,50890,33917,33918,33920,33921,33922,55520,33923,55518,33924,33925,55519,33926,33927,33928,45310,33929,48871,33930,51875,48372,33931,33932,33933,33934,47281,33935,33936,47342,33937,33938,33939,33940,33941,33942,33943,33944,33945,33946,55522,33947,48587,33948,55524,55523,33949,33950,33951,33952,33953,50684,33954,33955,33956,33957,33958,33959,33960,55525,33961,33962,55526,33963,33964,33965,33966,33967,33968,33969,49574,33970,51376,45292,47526,48339,52977,56253,49619,33971,33972,33973,33974,46767,55034,50604,48601,56254,56255,33975,33976,33977,49400,48802,49357,33978,33979,33980,33981,33982,33983,33984,33985,33986,33987,56256,51910,33988,33989,33990,45738,33991,33992,33993,54210,33994,50147,33995,53675,33996,33997,33998,33999,56258,34000,49365,34001,34002,34003,56259,34004,49073,34005,34006,34007,34008,34009,34010,50364,34011,34012,34013,34014,51162,34015,34016,34017,34018,34019,34020,34021,34022,34023,34024,34025,56260,34026,34027,34028,34029,34030,34031,34032,34033,55784,51671,34034,34035,34036,47540,52976,54472,34037,34038,34039,34040,45308,46290,34041,53465,34042,34043,34044,34045,55785,34046,57035,55787,34112,34113,34114,34115,55472,48047,45489,34116,46039,55502,34117,34118,54481,34119,34120,48563,49135,34121,53179,34122,34123,55504,34124,34125,34126,47051,34127,34128,34129,55505,34130,34131,34132,34133,34134,34135,34136,34137,34138,34139,50853,51192,53949,34140,34141,55506,50404,34142,51886,34143,51111,34144,55462,34145,51709,52967,48092,45291,34146,34147,34148,48042,53421,34149,45488,55268,55231,34150,46501,49908,50383,34151,34152,45737,34153,45751,34154,45541,57266,54716,49064,49836,55509,49841,34155,55508,52948,34156,56032,34157,52928,34158,34159,55476,50094,54177,52899,34160,48308,51380,49873,34161,48877,53430,34162,56033,34163,34164,34165,34166,51172,34167,34168,45991,34169,46834,52476,49402,34170,34171,49399,34172,53689,53729,55495,34173,34174,34176,34177,34178,34179,34180,45790,34181,34182,49381,34183,47857,34184,34185,55496,34186,54445,34187,34188,53217,55497,34189,55498,53187,34190,46072,48839,34191,34192,34193,34194,55499,34195,34196,34197,34198,34199,34200,34201,56268,34202,34203,34204,34205,51365,34206,34207,34208,53208,34209,51454,45774,34210,34211,34212,34213,34214,54230,45798,48304,54225,52139,47028,34215,34216,34217,47010,34218,34219,51941,34220,51361,51932,45540,53488,34221,50641,34222,34223,34224,56261,46590,34225,34226,49114,47557,48868,49645,34227,57270,57269,54971,48592,54745,45256,46755,49097,52392,57267,51895,54226,34228,55503,53942,47813,52158,52414,34229,57271,46576,57268,34230,34231,34232,54261,34233,46036,47351,34234,57274,34235,47823,48298,46581,34236,52652,50171,47859,49396,52674,53234,57272,53189,34237,49856,57273,49904,34238,34239,34240,48893,34241,49631,52684,54007,47053,57281,34242,57284,34243,34244,47089,45257,46806,47060,34245,47788,52477,49108,52145,50932,34246,54952,57285,34247,52962,46003,34248,34249,52988,46261,34250,52935,47856,34251,52961,34252,53693,34253,34254,57280,34255,34256,46324,34257,46026,34258,47334,57275,34259,34260,34261,34262,50373,34263,57276,57277,57278,50619,57279,57282,54449,57283,34264,51130,52952,34265,34266,34267,34268,34269,50392,34270,57290,34271,57295,34272,55004,34273,34274,34275,34276,34277,34278,34279,34280,57289,57306,52918,34281,47815,57294,57288,50654,34282,34283,51691,47860,50172,34284,34285,48855,34286,57286,34287,57293,34288,50648,34289,34290,34291,34292,54694,47821,34293,48844,54205,47296,34294,55012,34295,57287,47550,49063,34296,34297,49660,57291,57292,34298,57296,34299,34300,34301,34302,34368,57307,57317,34369,57303,57302,55241,57315,57316,58859,53927,57298,34370,49065,34371,54491,34372,49096,57300,34373,34374,34375,53196,34376,34377,57309,34378,53706,34379,57310,45223,50871,57299,34380,47845,34381,46815,52699,47614,54485,34382,34383,57311,53228,45221,57319,57297,53702,57301,57304,57305,57308,34384,48041,34385,57312,57313,34386,57314,57318,57320,54196,34387,34388,34389,34390,34391,47335,50614,57322,51674,49576,50372,34392,34393,49118,53240,34394,34395,34396,54748,57326,34397,34398,34399,34400,34401,34402,45752,34403,47839,57324,34404,56257,34405,53732,34406,34407,34408,34409,52212,46269,34410,45222,34411,34412,34413,34414,34415,57329,52422,57330,34416,34417,57325,34418,34419,34420,34421,34422,34423,57321,34424,34425,34426,34427,57323,34428,57327,57328,48061,34429,34430,57331,34432,34433,57332,34434,48035,34435,51931,52904,57511,45994,34436,57510,34437,34438,34439,57505,34440,34441,34442,34443,57342,34444,52697,57340,34445,57338,34446,49104,55236,34447,51660,34448,34449,57336,45217,34450,34451,34452,34453,34454,57341,34455,34456,34457,34458,57339,57506,34459,34460,34461,34462,34463,57512,34464,34465,34466,34467,47048,34468,34469,50849,51638,49330,57333,34470,34471,50622,34472,55492,57337,50422,34473,34474,34475,34476,34477,34478,57507,57508,57509,53413,34479,34480,57524,52452,34481,57521,34482,49062,57519,52921,57515,51654,34483,34484,49326,57518,47853,47792,57513,34485,34486,34487,57334,34488,57523,34489,34490,57528,34491,34492,34493,46253,57529,34494,34495,53170,47816,34496,57520,34497,34498,34499,34500,34501,34502,34503,53498,34504,34505,34506,34507,34508,34509,34510,34511,34512,57516,34513,54523,34514,57335,34515,50663,34516,57517,34517,54263,34518,57526,57527,34519,34520,34521,34522,34523,57540,53473,34524,34525,34526,57532,34527,34528,57545,57546,34529,34530,34531,57534,57514,51620,57537,34532,57522,34533,34534,34535,34536,34537,51912,57539,34538,57525,34539,52939,34540,52163,57549,57542,57538,34541,57547,34542,57530,57535,57536,34543,34544,57541,34545,34546,57543,57544,34547,57548,34548,57531,34549,34550,34551,34552,34553,52180,57557,34554,57558,57554,34555,34556,34557,34558,34624,34625,57552,48334,34626,34627,57553,34628,47298,55493,34629,34630,34631,34632,34633,34634,34635,34636,53482,34637,34638,49903,34639,34640,57551,57533,34641,34642,34643,57556,57555,34644,34645,57559,34646,34647,34648,34649,57564,57560,34650,34651,34652,55030,46000,34653,55276,34654,52155,34655,34656,57562,34657,52987,34658,34659,34660,47833,34661,34662,34663,34664,34665,34666,34667,34668,34669,34670,34671,34672,57569,57565,53933,34673,34674,34675,34676,34677,57570,34678,34679,57563,57561,57567,34680,34681,57568,34682,34683,34684,34685,34686,57566,34688,57572,34689,34690,34691,50935,55468,54507,57574,51913,34692,34693,34694,34695,57573,34696,34697,34698,34699,47297,34700,34701,34702,34703,57575,57576,34704,34705,34706,34707,34708,34709,34710,34711,57577,57571,34712,34713,34714,34715,34716,34717,34718,47807,52455,34719,34720,34721,57578,34722,34723,34724,34725,34726,34727,34728,34729,34730,34731,34732,34733,34734,34735,34736,53241,34737,34738,34739,34740,34741,34742,34743,34744,34745,34746,34747,57579,34748,34749,34750,34751,34752,34753,34754,51394,34755,34756,34757,34758,48576,34759,34760,34761,34762,34763,34764,34765,34766,34767,34768,34769,34770,34771,50386,34772,34773,34774,34775,34776,34777,34778,34779,34780,57580,34781,34782,57581,34783,34784,51188,52164,34785,57582,48088,55478,54002,57583,52677,34786,46810,34787,34788,34789,34790,34791,34792,57585,34793,54448,34794,34795,49319,46289,34796,34797,52903,57584,34798,34799,34800,57586,47564,34801,34802,47610,52668,57587,34803,34804,34805,50900,57588,34806,54450,34807,51366,57590,57589,34808,34809,34810,34811,34812,34813,34814,34880,34881,34882,34883,34884,34885,34886,34887,34888,34889,57591,34890,34891,52673,34892,34893,34894,51877,34895,34896,34897,34898,54490,56279,56281,34899,56280,47591,56284,56285,46552,34900,34901,56282,34902,34903,34904,34905,34906,56283,45985,56287,34907,34908,48120,34909,54967,34910,56288,34911,34912,34913,34914,48889,34915,34916,47035,34917,56272,52398,49074,48053,55288,49107,34918,34919,34920,34921,34922,49129,34923,34924,48353,52403,56286,45267,52971,47064,55225,50882,34925,34926,49316,34927,52409,34928,56295,56289,50874,56291,34929,56296,34930,50679,34931,34932,34933,56298,34934,34935,56297,49088,34936,34937,34938,56294,56293,34939,34940,34941,34942,34944,46265,49324,49826,56290,56292,34945,34946,34947,34948,53453,56301,34949,34950,34951,34952,34953,49373,56306,34954,34955,34956,34957,34958,34959,34960,46818,34961,34962,34963,34964,56307,56274,47544,54443,56300,34965,49105,56304,34966,56273,34967,46566,34968,56299,49125,34969,34970,34971,56302,34972,56305,34973,34974,34975,56313,34976,34977,34978,34979,34980,34981,34982,34983,34984,47521,45219,34985,34986,34987,34988,34989,34990,34991,49905,34992,34993,46023,56303,34994,34995,56312,34996,50898,56308,34997,34998,56309,56311,56310,34999,35000,56318,35001,54258,45754,35002,35003,35004,56317,35005,35006,35007,35008,35009,35010,35011,35012,56484,35013,56315,35014,35015,35016,35017,56314,35018,35019,35020,56316,50656,48121,35021,35022,56483,35023,35024,56485,35025,52419,35026,35027,35028,46801,56768,35029,35030,35031,56481,35032,56482,35033,35034,35035,51125,35036,35037,35038,46825,35039,35040,35041,56487,35042,35043,35044,35045,56486,35046,56489,45476,35047,35048,46540,35049,35050,35051,35052,35053,49072,35054,35055,35056,35057,35058,53727,35059,35060,35061,35062,46786,35063,35064,35065,35066,35067,35068,35069,35070,35136,35137,35138,35139,35140,35141,56488,35142,35143,35144,35145,35146,35147,35148,52218,60403,35149,35150,35151,52188,35152,35153,52222,35154,35155,35156,52417,35157,35158,35159,35160,35161,51451,35162,35163,35164,35165,35166,35167,56490,35168,35169,35170,35171,35172,52462,56491,35173,35174,35175,35176,35177,35178,35179,35180,35181,35182,35183,35184,35185,35186,35187,35188,35189,56275,35190,56495,56492,35191,48819,35192,51963,35193,35194,35195,56493,35196,35197,35198,35200,35201,35202,35203,35204,51658,50361,35205,35206,35207,35208,35209,51133,56494,35210,35211,35212,54518,53478,35213,35214,35215,35216,35217,35218,35219,35220,50347,46805,35221,35222,35223,35224,35225,35226,35227,35228,35229,35230,35231,35232,35233,35234,35235,35236,35237,35238,56276,35239,35240,35241,35242,45530,35243,35244,35245,56277,35246,35247,35248,35249,35250,35251,35252,35253,35254,35255,35256,56278,35257,35258,35259,47806,35260,35261,35262,35263,35264,35265,35266,35267,35268,35269,35270,35271,35272,35273,51392,35274,35275,35276,35277,35278,35279,51903,51401,35280,55219,35281,51705,35282,35283,49095,35284,35285,47864,35286,35287,53948,35288,35289,35290,35291,35292,35293,35294,35295,58042,35296,46246,35297,35298,45496,35299,35300,35301,35302,35303,47284,35304,53188,35305,35306,35307,35308,55783,53158,52706,35309,35310,55789,46816,35311,53945,35312,35313,47547,35314,35315,35316,35317,58041,58039,35318,46323,35319,52460,52395,47090,35320,55474,53739,47803,35321,51879,35322,35323,52663,35324,35325,53956,49124,48336,46817,35326,57029,35392,35393,35394,35395,57030,56252,35396,53721,35397,35398,50918,50382,47086,35399,47068,35400,35401,49148,55264,35402,50933,35403,35404,45500,57032,48561,52439,57034,35405,57033,35406,35407,35408,35409,35410,46572,35411,51677,35412,35413,45250,35414,35415,35416,35417,35418,35419,35420,35421,35422,35423,35424,35425,35426,50606,50603,35427,50380,35428,48361,52221,35429,35430,35431,47811,35432,35433,35434,58873,51431,58874,52733,35435,55217,47294,49896,35436,51409,35437,35438,58875,35439,35440,35441,35442,46794,48331,35443,35444,53757,59041,35445,50158,35446,35447,35448,35449,59044,35450,35451,35452,35453,58878,59045,52695,35454,35456,47041,58876,58877,59043,35457,35458,50397,59048,35459,35460,59047,35461,35462,35463,35464,35465,35466,50115,35467,50910,35468,35469,59050,35470,35471,35472,35473,35474,35475,35476,50359,35477,35478,35479,59042,51900,35480,35481,35482,35483,48611,47555,59046,53461,52911,35484,35485,59049,59056,35486,53926,35487,48554,59053,35488,35489,35490,35491,35492,59055,35493,49361,35494,35495,53964,35496,35497,35498,48295,35499,35500,35501,35502,35503,35504,35505,35506,35507,35508,35509,35510,59057,35511,54006,35512,35513,35514,55243,35515,52734,35516,52702,49830,59051,59052,48575,59054,59059,35517,35518,59058,35519,35520,35521,35522,59062,35523,59064,35524,35525,35526,35527,50415,35528,35529,35530,50376,35531,35532,48874,51695,35533,35534,59063,35535,46832,35536,35537,35538,50148,35539,35540,35541,35542,35543,35544,35545,54249,59060,35546,59061,35547,51362,35548,35549,35550,35551,35552,59069,35553,35554,35555,59065,35556,35557,35558,35559,35560,50885,35561,35562,52721,59067,35563,35564,35565,35566,35567,35568,35569,35570,35571,35572,59068,35573,35574,35575,35576,48105,35577,35578,35579,35580,35581,35582,35648,59070,35649,35650,35651,35652,59066,35653,35654,49335,35655,35656,35657,35658,35659,35660,35661,35662,35663,54180,59071,51700,59075,35664,35665,59076,35666,35667,35668,35669,53494,35670,35671,35672,35673,35674,35675,35676,35677,35678,35679,35680,35681,35682,35683,35684,35685,35686,35687,50109,35688,35689,35690,35691,35692,35693,35694,50116,59074,35695,35696,35697,35698,35699,35700,35701,35702,35703,35704,35705,35706,35707,35708,59073,35709,35710,35712,35713,35714,35715,35716,59079,53169,35717,60404,35718,35719,59082,35720,35721,35722,35723,35724,59077,35725,35726,48350,51625,35727,35728,35729,35730,35731,35732,48309,35733,35734,53203,35735,35736,35737,35738,35739,59080,35740,59081,35741,59086,35742,59088,35743,35744,35745,59089,35746,35747,35748,59083,46549,35749,59084,35750,35751,59087,35752,35753,50395,35754,59078,35755,35756,35757,35758,35759,59085,35760,35761,35762,35763,35764,35765,35766,35767,35768,35769,35770,35771,35772,35773,35774,35775,35776,35777,35778,35779,35780,35781,35782,59090,35783,35784,35785,35786,35787,35788,35789,35790,35791,35792,35793,35794,59092,59091,35795,35796,35797,35798,35799,35800,35801,35802,35803,35804,35805,35806,35807,35808,35809,35810,35811,35812,35813,35814,35815,35816,35817,35818,35819,35820,59093,35821,55800,35822,35823,59094,35824,35825,35826,35827,35828,35829,35830,35831,59095,35832,35833,35834,35835,35836,35837,35838,35904,35905,35906,35907,35908,35909,35910,35911,55251,59101,35912,59102,49111,54480,35913,55254,46310,52207,59098,55491,55246,53410,35914,50127,35915,35916,59103,48318,47554,59099,53671,35917,35918,47778,49871,35919,55467,35920,35921,35922,51947,58862,35923,59100,35924,47093,35925,35926,35927,35928,51430,35929,35930,50421,35931,35932,58802,50430,35933,52220,58803,54700,35934,54254,51928,45234,35935,52174,52714,35936,35937,47850,35938,35939,35940,58805,35941,58804,35942,55258,47577,55014,46760,52720,53963,45478,51893,35943,46056,51699,49101,53499,51922,58806,48066,35944,35945,35946,53212,47532,35947,35948,35949,35950,54487,35951,35952,47782,53735,53244,48338,35953,58807,51421,35954,35955,35956,49133,45558,52190,35957,35958,48325,35959,48324,54010,50140,49116,35960,35961,35962,35963,47291,35964,35965,35966,50114,35968,47790,54434,35969,35970,35971,35972,35973,35974,35975,35976,35977,51166,50351,45804,35978,47569,35979,35980,58811,49608,35981,35982,54703,35983,35984,35985,35986,35987,58812,35988,58814,35989,35990,35991,35992,35993,35994,35995,46311,46804,52162,53680,46524,35996,35997,51929,35998,47074,35999,36000,51684,36001,48555,36002,36003,52926,55280,36004,36005,36006,36007,53409,36008,51673,36009,36010,46843,59096,48354,36011,46014,36012,51664,36013,59097,45986,36014,36015,36016,36017,57036,36018,54216,57037,36019,53922,36020,36021,36022,36023,57038,36024,36025,36026,36027,48845,36028,36029,57039,36030,36031,36032,51884,54012,46047,58858,50401,48801,52914,50418,48854,50856,45795,36033,36034,48851,36035,36036,51196,52459,48620,52957,36037,36038,51898,50881,58860,53436,36039,36040,36041,54713,36042,36043,36044,58861,36045,36046,36047,36048,51956,36049,52672,49861,36050,58863,36051,49860,58864,36052,36053,36054,36055,36056,36057,36058,58872,52685,36059,51645,36060,36061,36062,36063,36064,36065,36066,53977,57768,36067,36068,36069,36070,54252,36071,52202,50929,36072,36073,36074,36075,36076,57772,36077,36078,36079,57767,57769,36080,36081,57770,57775,36082,36083,45805,36084,57771,47322,57773,57774,57776,46522,57777,36085,36086,36087,36088,36089,57779,57784,36090,36091,36092,36093,36094,53714,36160,57782,57781,49643,36161,36162,36163,57783,36164,54464,36165,57778,36166,57786,45238,36167,36168,36169,36170,57780,36171,49145,36172,57785,36173,36174,57787,36175,36176,36177,36178,36179,36180,57790,36181,36182,36183,36184,36185,36186,57788,36187,36188,36189,36190,36191,36192,54981,36193,36194,36195,36196,36197,36198,36199,53183,36200,36201,57789,57791,49869,36202,46827,36203,54264,36204,36205,51149,36206,36207,47077,36208,36209,36210,36211,36212,36213,36214,36215,36216,36217,48894,36218,36219,36220,36221,36222,36224,57792,57793,36225,36226,57799,46055,36227,36228,36229,36230,36231,36232,50921,36233,36234,36235,36236,36237,46302,36238,53698,36239,36240,36241,36242,57800,36243,36244,57798,36245,36246,36247,36248,36249,57797,36250,57795,57794,36251,45504,36252,36253,36254,54712,57796,36255,36256,36257,36258,36259,57803,36260,36261,36262,36263,36264,36265,36266,36267,57804,57802,36268,36269,36270,36271,36272,36273,36274,36275,61434,36276,36277,57811,57810,51126,36278,36279,36280,36281,36282,36283,36284,36285,36286,36287,36288,57801,36289,36290,57806,36291,57808,36292,36293,36294,36295,36296,36297,36298,36299,36300,36301,36302,57812,36303,57809,57805,36304,36305,57807,36306,36307,36308,36309,57813,36310,36311,36312,36313,36314,36315,36316,36317,36318,36319,36320,36321,36322,57814,36323,36324,36325,36326,36327,36328,36329,36330,36331,36332,36333,36334,36335,36336,36337,36338,36339,36340,36341,36342,36343,36344,57815,36345,36346,36347,57816,36348,36349,36350,36416,36417,36418,36419,36420,36421,36422,36423,36424,36425,36426,36427,36428,36429,36430,36431,36432,36433,36434,36435,36436,36437,57818,36438,36439,36440,36441,36442,36443,36444,36445,36446,36447,36448,36449,36450,57819,36451,36452,36453,36454,36455,36456,36457,52897,36458,36459,36460,36461,36462,36463,36464,36465,36466,36467,36468,36469,36470,59357,36471,46248,55005,36472,36473,53682,46002,36474,36475,47524,55283,51145,48862,47534,36476,52951,36477,36478,45806,56271,36480,48314,53969,52168,45261,36481,36482,53231,36483,36484,36485,36486,36487,55779,48621,36488,36489,45522,51920,45756,36490,52135,47019,36491,51878,36492,36493,36494,53155,36495,36496,57592,54730,57595,36497,36498,57594,50625,52475,36499,49585,57593,55011,45743,54980,46555,36500,36501,36502,36503,36504,36505,36506,36507,46328,54945,36508,36509,36510,36511,36512,53167,45295,36513,36514,57596,36515,36516,36517,36518,36519,57761,45987,36520,36521,57597,57598,50097,36522,36523,36524,36525,50141,36526,57762,47097,36527,36528,36529,36530,36531,36532,48079,36533,36534,36535,36536,36537,36538,36539,57763,50363,36540,36541,36542,36543,36544,57764,36545,36546,57765,36547,36548,57766,46257,36549,36550,36551,36552,36553,36554,36555,36556,36557,36558,36559,36560,36561,36562,36563,47305,50877,50410,36564,45730,36565,53458,36566,59355,48067,54231,54212,36567,47587,58063,36568,36569,36570,55215,36571,51180,45523,36572,36573,46258,58065,36574,36575,36576,53490,49838,58064,36577,49122,54182,46551,58066,46570,36578,50157,47357,36579,47278,36580,50643,47055,58068,36581,36582,36583,36584,58067,46792,55289,36585,36586,36587,36588,36589,52645,36590,36591,36592,36593,36594,58072,36595,58070,51964,49077,54201,58069,36596,36597,36598,36599,58071,36600,36601,36602,36603,36604,36605,36606,36672,36673,36674,49582,49352,36675,36676,36677,36678,36679,36680,58075,58074,49322,36681,36682,49614,36683,36684,36685,36686,58076,36687,36688,36689,36690,36691,36692,36693,36694,36695,36696,36697,36698,58077,36699,58078,36700,36701,36702,36703,36704,36705,36706,36707,36708,56264,36709,53715,52642,36710,36711,48552,36712,36713,36714,57027,55461,49066,56269,53996,50938,50602,36715,36716,36717,57028,36718,45527,57262,36719,36720,36721,51901,36722,57265,36723,47533,36724,54013,36725,47269,47851,36726,36727,46042,36728,36729,36730,46556,54725,36731,36732,36733,36734,50134,53202,48033,36736,58867,58866,36737,36738,58868,36739,52708,36740,51445,36741,36742,36743,36744,36745,36746,36747,46511,51135,36748,58870,36749,36750,36751,60592,36752,36753,36754,36755,36756,36757,36758,36759,36760,36761,36762,36763,36764,36765,36766,58854,36767,47593,46513,36768,49852,58856,58855,58857,36769,36770,36771,36772,53965,36773,36774,36775,57834,53454,36776,52654,36777,53733,36778,36779,45770,45547,36780,45554,50669,36781,36782,54723,54192,36783,57820,36784,36785,36786,57821,36787,53979,36788,46009,45515,36789,36790,36791,52729,54775,57822,36792,48822,46333,36793,57823,47836,57824,48050,49865,57825,36794,36795,36796,53484,36797,52669,36798,36799,57826,36800,46531,50631,57827,36801,36802,57828,36803,36804,36805,36806,54265,36807,36808,36809,36810,36811,36812,57829,36813,53677,36814,36815,57830,52898,36816,36817,36818,36819,36820,36821,57831,36822,46530,36823,36824,36825,36826,57832,48085,36827,36828,36829,36830,36831,53444,58080,45528,53988,36832,36833,58081,36834,36835,48329,51404,36836,58083,60670,60669,57263,36837,36838,36839,58082,54974,52732,50086,36840,36841,36842,58307,36843,36844,54994,58087,36845,36846,58088,36847,36848,54215,36849,36850,58092,49132,36851,58093,58085,36852,36853,46016,36854,36855,36856,50414,36857,36858,58094,36859,36860,53443,36861,47862,58089,47070,48051,52396,52171,58084,58086,58090,58091,36862,36928,36929,58103,36930,36931,58100,54517,58099,36932,36933,50605,36934,54778,50626,45760,36935,36936,58095,36937,58098,49583,52156,36938,36939,46497,58105,36940,36941,36942,48305,58097,53460,54457,58101,47574,58102,36943,36944,36945,51155,36946,36947,36948,36949,36950,58096,36951,36952,36953,36954,36955,55260,60833,36956,36957,58104,36958,60837,58110,51921,36959,36960,36961,36962,36963,36964,36965,49589,36966,48080,36967,36968,49110,36969,47843,36970,36971,52129,36972,36973,36974,60838,60835,36975,36976,60834,36977,36978,36979,36980,48086,60839,53492,36981,36982,60836,47838,46839,58273,46770,52465,47527,36983,53154,51105,36984,36985,49106,36986,36987,46833,36988,58106,58107,58109,58108,50389,58274,36989,54209,36990,36992,36993,58279,51140,36994,36995,36996,36997,53156,36998,36999,58281,47799,37000,37001,37002,37003,58280,37004,48090,37005,58275,37006,37007,37008,58276,58282,37009,58278,37010,52978,54214,37011,37012,48060,37013,37014,54467,37015,50426,37016,37017,60840,53500,58277,37018,50165,37019,58285,45487,37020,58290,37021,37022,37023,48322,37024,37025,58284,46527,37026,37027,37028,37029,37030,37031,37032,37033,51177,58288,37034,37035,37036,48810,52719,37037,37038,37039,37040,37041,48115,37042,37043,37044,52456,37045,37046,58287,37047,58289,37048,53159,58286,37049,52905,48093,37050,37051,37052,37053,37054,46571,48869,45778,46029,37055,45497,58283,45777,46508,47583,46824,37056,37057,53227,58295,37058,48076,37059,37060,51399,53450,37061,37062,37063,37064,37065,58296,46062,37066,37067,37068,37069,60841,37070,54266,54244,37071,37072,37073,60842,58297,53986,37074,37075,37076,37077,37078,58293,37079,37080,37081,37082,54238,37083,37084,37085,37086,47312,58291,37087,37088,58294,47071,37089,58292,49314,37090,37091,37092,58298,37093,37094,37095,37096,37097,37098,37099,37100,37101,37102,37103,37104,37105,37106,37107,37108,37109,37110,37111,54456,37112,37113,37114,37115,37116,37117,37118,37184,46280,37185,58299,37186,48069,37187,51703,37188,37189,51685,37190,37191,37192,50365,37193,37194,37195,37196,37197,37198,37199,60843,37200,37201,37202,37203,49917,37204,37205,37206,37207,48091,49070,37208,37209,37210,37211,37212,37213,37214,52927,37215,37216,37217,37218,58300,37219,49078,37220,37221,37222,37223,37224,37225,37226,37227,37228,37229,37230,37231,37232,37233,37234,37235,37236,37237,37238,45551,37239,37240,54519,37241,37242,37243,37244,37245,58302,37246,37248,37249,37250,37251,37252,37253,37254,60845,37255,37256,37257,37258,37259,37260,37261,37262,37263,58303,47785,60844,37264,37265,58301,37266,37267,37268,37269,37270,37271,37272,37273,37274,37275,58304,37276,37277,37278,37279,37280,37281,47798,37282,37283,37284,46766,37285,37286,37287,37288,37289,53432,37290,45251,60846,37291,37292,37293,37294,37295,60847,49345,37296,58305,37297,37298,37299,37300,37301,37302,37303,37304,37305,37306,37307,37308,37309,37310,37311,37312,37313,50611,37314,37315,37316,37317,37318,37319,37320,37321,37322,37323,37324,37325,37326,37327,58306,37328,37329,37330,37331,37332,37333,37334,37335,37336,56498,37337,37338,37339,37340,37341,37342,60848,37343,47338,37344,52972,60071,53479,51961,51414,53175,46025,52946,48612,37345,37346,58334,48114,60072,54717,37347,50909,60073,37348,37349,37350,60074,37351,60076,60075,37352,60078,60077,37353,37354,37355,37356,48600,37357,60079,37358,49854,37359,37360,37361,37362,46273,46327,37363,37364,48039,37365,37366,37367,37368,37369,60646,60645,47039,52217,45538,37370,60647,37371,37372,37373,51656,60648,60649,37374,51926,57040,45765,54522,37440,37441,50891,45255,46322,51411,37442,37443,37444,52688,37445,37446,49080,37447,37448,37449,37450,37451,37452,37453,49115,37454,37455,51108,54964,37456,49321,57041,51624,53743,50596,45287,46006,51397,37457,37458,45282,37459,37460,47094,37461,37462,50682,37463,37464,46835,37465,54738,46032,48316,37466,37467,37468,45997,37469,37470,37471,37472,48881,45265,37473,37474,37475,37476,37477,37478,53974,51939,55205,37479,52662,46774,49081,54747,37480,47271,50647,37481,37482,37483,57042,49113,49877,51136,37484,48036,45480,37485,37486,50666,37487,37488,50683,52391,37489,37490,37491,37492,45479,37493,37494,37495,46550,37496,37497,37498,50344,37499,57043,53690,46057,37500,50162,37501,37502,47095,37504,55028,46499,45808,50356,50409,49325,57044,37505,45288,50628,49632,37506,47573,37507,48860,52696,45262,37508,52687,57046,48848,55230,57045,54736,45277,37509,37510,50402,37511,37512,49827,48368,37513,54197,49337,50593,45734,54513,37514,37515,49320,51907,57047,54780,37516,47536,37517,51373,52137,37518,57049,49085,37519,37520,37521,37522,50868,55207,51888,50371,37523,46038,47570,37524,37525,37526,37527,54968,60156,45236,37528,37529,37530,37531,49126,37532,37533,52468,37534,37535,37536,37537,52698,37538,37539,37540,54975,49870,37541,52942,52386,53422,50387,46514,57048,54773,48311,48083,37542,37543,45220,37544,50610,46316,37545,37546,37547,54769,37548,37549,60157,37550,37551,37552,37553,37554,37555,57050,52646,37556,37557,52716,37558,37559,37560,37561,52966,57052,37562,52657,49318,37563,37564,55229,37565,57051,45254,47796,51667,50419,48872,37566,37567,37568,37569,45750,37570,37571,37572,37573,37574,37575,37576,37577,49356,52208,37578,48369,48059,46519,37579,37580,37581,50677,37582,57062,37583,37584,37585,57059,48861,37586,37587,57055,37588,37589,37590,37591,46263,48605,37592,37593,57056,50413,37594,37595,37596,37597,53190,37598,46560,37599,37600,37601,37602,46814,51930,46580,57061,37603,54726,37604,57057,52429,50942,37605,50629,37606,37607,37608,53940,37609,48882,37610,37611,37612,37613,37614,37615,37616,49875,37617,52413,46008,37618,48595,37619,49112,52678,53722,46315,37620,57060,57053,57063,37621,60158,37622,37623,49840,57058,37624,37625,54976,46503,37626,45812,37627,57064,37628,57074,37629,37630,37696,37697,37698,57069,37699,57073,37700,37701,51424,37702,37703,37704,55265,57071,50152,52449,37705,45797,37706,37707,37708,53950,37709,37710,37711,37712,37713,37714,37715,57070,37716,57067,52949,37717,46247,37718,37719,37720,37721,37722,49067,48830,37723,37724,48594,37725,37726,37727,37728,57065,37729,54446,37730,57054,37731,57066,37732,37733,37734,37735,49343,37736,57068,45811,47337,49831,37737,37738,48577,37739,37740,37741,37742,37743,57077,57080,37744,37745,45739,46244,37746,37747,46314,51622,37748,37749,37750,37751,37752,37753,57078,52177,37754,47331,37755,57079,57082,37756,37757,37758,37760,57081,37761,37762,37763,52418,37764,45281,46318,37765,37766,37767,37768,37769,37770,58810,37771,37772,37773,37774,37775,53423,37776,37777,45803,37778,60321,37779,57076,37780,37781,51683,57075,45274,53921,45559,37782,52399,37783,37784,37785,37786,37787,37788,37789,57072,37790,52132,37791,37792,37793,54698,37794,37795,37796,37797,37798,57083,37799,37800,37801,37802,37803,37804,37805,37806,46301,37807,50342,37808,37809,37810,57085,37811,37812,37813,37814,37815,37816,37817,37818,37819,37820,50174,50337,57249,37821,37822,37823,37824,37825,37826,37827,49612,37828,57084,48879,37829,50866,37830,37831,37832,37833,37834,37835,37836,37837,37838,46021,51446,37839,37840,52154,57086,37841,37842,57252,37843,37844,37845,37846,55218,37847,37848,37849,37850,37851,46007,37852,37853,37854,37855,49603,37856,37857,51147,45733,46313,37858,55211,37859,37860,37861,37862,50412,37863,57250,57251,37864,57253,37865,47795,37866,37867,37868,57254,37869,49374,37870,37871,51651,37872,37873,37874,37875,37876,37877,37878,45785,51174,37879,57255,37880,51164,37881,37882,37883,37884,57256,60322,37885,37886,37952,37953,37954,52179,37955,37956,37957,57258,37958,57257,37959,45761,37960,37961,37962,37963,37964,37965,37966,37967,37968,37969,37970,37971,37972,37973,37974,37975,37976,37977,37978,37979,37980,37981,37982,37983,37984,50634,37985,37986,37987,37988,37989,37990,37991,37992,57259,37993,37994,37995,37996,37997,37998,37999,38000,54492,38001,38002,38003,38004,38005,51393,38006,38007,38008,38009,38010,38011,38012,38013,38014,38016,38017,38018,57260,38019,38020,38021,38022,38023,48880,38024,38025,57261,54951,38026,38027,38028,38029,60087,60342,51925,38030,55548,47300,38031,47525,38032,38033,47045,54782,38034,38035,38036,38037,38038,47562,38039,38040,53415,62669,38041,38042,46544,38043,38044,50164,38045,48840,38046,38047,38048,60343,45245,38049,38050,48588,38051,49586,38052,45526,45992,38053,38054,38055,47314,51618,38056,38057,46808,38058,38059,38060,38061,60344,48820,38062,38063,38064,51965,38065,51139,38066,54779,38067,38068,47091,38069,38070,38071,38072,38073,38074,38075,38076,38077,38078,38079,38080,38081,38082,38083,52932,38084,38085,38086,54699,45555,38087,38088,38089,60595,45279,38090,60597,38091,38092,38093,46775,38094,49615,38095,62970,53425,38096,38097,54757,38098,52947,38099,38100,48623,46050,38101,47275,38102,54710,38103,60861,38104,46799,38105,52153,53442,38106,38107,38108,38109,38110,38111,38112,38113,47037,38114,38115,60598,51881,38116,38117,38118,50644,38119,60601,60600,49859,60599,38120,38121,38122,38123,53501,60602,38124,60603,55269,38125,38126,60604,38127,38128,38129,60605,50924,38130,38131,38132,38133,38134,38135,38136,38137,52958,38138,48328,38139,38140,51413,46505,48841,54972,54503,38141,38142,53678,53489,60088,60089,60090,47797,38208,38209,38210,38211,51889,49141,38212,38213,52730,38214,38215,38216,38217,38218,60096,38219,45242,60094,38220,38221,49317,38222,38223,38224,60091,38225,45821,38226,50167,48104,38227,38228,38229,53975,52980,60095,38230,38231,38232,60092,38233,38234,38235,60099,38236,53447,54195,38237,38238,38239,38240,46266,38241,50113,55282,38242,38243,38244,38245,54737,38246,51911,38247,60101,38248,38249,60100,60103,60102,38250,38251,38252,38253,38254,55015,38255,53204,38256,38257,60107,38258,48078,38259,38260,38261,38262,38263,38264,38265,48634,51662,38266,38267,60108,38268,38269,51641,53246,60106,54478,60109,60111,38270,38272,52717,38273,38274,38275,38276,60105,38277,60110,38278,38279,52974,38280,48094,38281,46015,38282,38283,38284,38285,38286,50901,48816,52986,38287,38288,38289,51175,38290,48807,60112,38291,38292,54983,38293,38294,38295,49600,38296,38297,38298,54493,38299,60113,38300,38301,53182,38302,38303,38304,38305,60114,38306,38307,38308,38309,51950,38310,38311,38312,38313,50607,45237,38314,38315,38316,38317,38318,60116,38319,38320,38321,38322,38323,38324,38325,38326,38327,60115,62687,38328,38329,38330,38331,38332,50362,38333,38334,38335,38336,38337,45481,38338,38339,38340,38341,58847,38342,38343,38344,38345,60117,38346,38347,38348,38349,38350,38351,38352,38353,38354,38355,38356,38357,38358,38359,38360,38361,38362,38363,38364,38365,38366,38367,38368,38369,38370,38371,51951,38372,60118,60119,50904,38373,38374,38375,38376,38377,38378,38379,38380,60120,38381,38382,60121,38383,38384,38385,38386,38387,38388,54459,38389,51194,53943,47356,38390,38391,60098,38392,45788,38393,38394,49916,38395,54520,52454,55278,38396,38397,38398,38464,38465,38466,38467,54466,54224,60355,50675,38468,47102,38469,38470,60372,38471,38472,38473,52151,60382,38474,49354,38475,38476,38477,52731,38478,45999,38479,50906,38480,38481,38482,38483,38484,38485,60412,38486,50366,38487,52916,50345,45502,54525,38488,51957,38489,55020,38490,38491,50899,46820,38492,38493,38494,38495,48122,38496,38497,53472,38498,38499,51633,38500,54483,51368,38501,38502,47307,38503,59582,51644,38504,38505,59579,38506,49390,53459,45764,46309,38507,59580,38508,38509,54728,38510,38511,38512,38513,38514,46789,38515,59581,51960,47324,52469,38516,38517,38518,49332,38519,38520,53742,59583,59586,38521,38522,47804,38523,45485,48604,38524,60093,59587,38525,59590,38526,59595,38528,38529,38530,38531,59596,38532,52169,45285,38533,48299,38534,38535,47545,38536,38537,59585,38538,52727,38539,59594,38540,38541,38542,38543,52982,38544,38545,38546,38547,54765,38548,49622,59588,38549,50102,38550,47611,54950,59592,38551,38552,38553,51936,54502,38554,59584,38555,59589,59591,38556,51129,47075,38557,59593,38558,49117,59602,38559,38560,59607,38561,59605,48348,48335,59611,38562,38563,38564,38565,38566,38567,38568,38569,59614,38570,59610,45562,38571,38572,38573,38574,38575,38576,38577,38578,38579,38580,45272,50355,47308,50914,51390,51425,38581,38582,38583,59599,59604,59606,38584,47601,59608,55285,38585,50427,38586,59612,38587,38588,45801,38589,38590,38591,59601,38592,38593,48365,38594,38595,49090,59597,55033,38596,49656,45809,38597,38598,38599,38600,38601,38602,38603,38604,59615,38605,51905,59609,38606,38607,38608,38609,54692,38610,45546,54715,59598,59600,46768,59603,38611,59613,49336,38612,51959,38613,52136,38614,38615,50908,49397,38616,38617,38618,38619,38620,59625,38621,38622,38623,53411,38624,38625,38626,38627,38628,38629,38630,59634,55018,38631,38632,38633,38634,38635,38636,38637,59616,59617,38638,38639,38640,53753,47819,47353,38641,38642,47345,54484,59631,38643,59630,59628,47600,52434,59622,52902,49138,38644,45240,59633,59632,38645,55232,38646,59620,38647,52649,51619,38648,48056,48603,59626,38649,38650,38651,38652,38653,38654,38720,38721,38722,38723,59618,59619,59621,46517,59623,51141,59627,59629,48560,55214,38724,59640,38725,38726,38727,38728,38729,38730,38731,38732,59637,38733,52656,59638,38734,38735,38736,38737,38738,38739,38740,38741,38742,49594,38743,59624,38744,50103,45296,38745,38746,38747,38748,38749,38750,38751,38752,59636,38753,38754,38755,59639,38756,38757,38758,47523,38759,38760,38761,38762,38763,38764,38765,38766,38767,38768,51666,38769,38770,38771,50126,52960,49382,38772,38773,38774,38775,52211,38776,52445,53429,38777,38778,51937,38779,59635,38780,38781,38782,38784,38785,38786,38787,38788,38789,38790,48364,38791,59641,38792,38793,38794,38795,38796,38797,50142,38798,50917,38799,47607,38800,38801,38802,38803,45300,38804,38805,55256,38806,38807,48300,38808,50671,38809,38810,38811,38812,38813,52420,38814,38815,59814,38816,38817,38818,38819,38820,38821,38822,38823,38824,38825,51629,38826,59810,49378,38827,38828,38829,49091,38830,38831,38832,59646,47575,38833,59643,38834,38835,38836,38837,59812,38838,38839,38840,53966,38841,38842,38843,38844,38845,59811,38846,54962,55221,38847,59815,38848,48567,38849,38850,38851,38852,38853,38854,38855,38856,38857,38858,38859,38860,59644,59645,38861,38862,38863,59809,38864,38865,38866,38867,38868,38869,38870,38871,52694,38872,38873,53932,38874,38875,38876,59826,38877,38878,38879,38880,59817,38881,38882,38883,46250,38884,46267,38885,38886,59819,38887,38888,38889,38890,38891,38892,38893,38894,38895,38896,38897,38898,38899,38900,38901,38902,38903,53416,38904,38905,59813,38906,38907,46078,38908,38909,59820,49379,38910,59818,38976,38977,59833,38978,38979,59832,38980,38981,38982,38983,59822,38984,38985,59642,38986,38987,59816,38988,38989,38990,38991,38992,49068,59825,59834,38993,38994,49829,38995,38996,38997,59823,38998,47301,38999,59821,39000,54236,59828,59829,59831,39001,39002,39003,59847,39004,39005,39006,39007,39008,39009,49350,59845,39010,39011,59824,39012,39013,59835,45297,39014,39015,39016,39017,39018,39019,39020,39021,39022,39023,59836,54693,39024,39025,59838,39026,59839,39027,39028,39029,59841,39030,39031,49649,39032,39033,51382,39034,39035,39036,59837,39037,39038,39040,39041,39042,59842,39043,39044,39045,39046,39047,39048,39049,39050,59843,39051,59827,39052,59830,39053,48049,39054,39055,39056,59840,39057,39058,39059,39060,39061,39062,48375,39063,39064,39065,59844,59846,39066,39067,39068,39069,39070,39071,39072,39073,39074,39075,39076,39077,59850,39078,39079,39080,39081,59854,39082,39083,39084,39085,39086,39087,39088,39089,39090,39091,45787,39092,59848,39093,39094,39095,39096,39097,39098,39099,39100,39101,39102,47022,39103,39104,39105,39106,39107,39108,39109,39110,39111,39112,39113,39114,59851,59852,39115,39116,39117,39118,39119,39120,54721,39121,50339,39122,39123,39124,39125,39126,39127,59864,39128,47841,39129,39130,39131,39132,59849,39133,54179,39134,39135,39136,59860,39137,39138,39139,39140,39141,39142,39143,59863,59856,39144,39145,39146,39147,39148,59855,39149,39150,51137,39151,39152,39153,39154,39155,39156,39157,39158,59858,39159,39160,39161,39162,39163,39164,39165,59865,46024,39166,59859,39232,39233,39234,39235,39236,53232,39237,39238,39239,59853,39240,39241,39242,39243,39244,39245,39246,39247,39248,39249,39250,46071,39251,39252,39253,39254,39255,39256,39257,59862,39258,39259,59866,39260,39261,39262,52404,39263,39264,39265,53165,39266,39267,39268,39269,39270,39271,39272,39273,39274,59861,39275,59868,59867,39276,39277,39278,39279,39280,59870,39281,39282,39283,39284,39285,39286,39287,39288,59857,39289,39290,39291,39292,39293,39294,39296,39297,59869,39298,59871,50122,39299,39300,39301,39302,39303,39304,39305,39306,39307,39308,39309,39310,39311,39312,39313,39314,39315,39316,39317,39318,39319,39320,39321,39322,39323,39324,39325,39326,39327,39328,39329,39330,39331,39332,39333,39334,39335,39336,39337,39338,39339,39340,39341,39342,39343,39344,39345,39346,39347,39348,39349,39350,39351,39352,39353,39354,39355,39356,39357,39358,39359,39360,39361,39362,39363,39364,39365,39366,39367,39368,39369,39370,39371,39372,39373,39374,39375,39376,39377,39378,39379,39380,39381,39382,39383,39384,39385,39386,39387,39388,39389,39390,39391,39392,39393,39394,39395,39396,39397,39398,39399,39400,39401,39402,39403,39404,39405,39406,39407,39408,39409,39410,39411,39412,39413,51127,46286,48054,53440,60579,39414,39415,50615,39416,39417,39418,39419,39420,39421,39422,39488,39489,39490,54267,39491,39492,39493,39494,60580,39495,60581,50907,39496,39497,39498,49134,39499,39500,39501,39502,60582,39503,39504,60583,53418,39505,51128,39506,39507,47336,39508,39509,39510,39511,39512,39513,39514,39515,39516,39517,39518,39519,60584,39520,39521,39522,39523,39524,39525,39526,39527,54969,54781,46283,45757,52964,50919,39528,39529,52705,39530,39531,39532,39533,39534,39535,39536,39537,39538,39539,39540,39541,39542,39543,46325,39544,52160,48351,39545,39546,39547,39548,59874,59875,53738,59877,39549,46329,59876,39550,53683,51938,45776,39552,59880,39553,39554,39555,39556,59878,59879,39557,39558,54963,39559,39560,39561,59881,59882,39562,39563,39564,39565,39566,59883,39567,39568,39569,39570,39571,39572,39573,39574,59884,39575,39576,39577,39578,39579,39580,39581,39582,60591,50617,46798,39583,54003,39584,39585,39586,39587,39588,39589,39590,46574,39591,48089,60593,39592,39593,53987,39594,39595,39596,39597,39598,52963,39599,50360,39600,50111,39601,39602,46782,55481,45512,45519,45521,50686,39603,45520,39604,50091,39605,39606,39607,39608,39609,54705,39610,39611,39612,39613,39614,39615,39616,39617,60324,47809,39618,39619,39620,52410,39621,39622,39623,60325,39624,60327,39625,39626,39627,60328,39628,39629,39630,60326,39631,39632,39633,39634,39635,39636,39637,60329,60331,60330,39638,39639,39640,39641,39642,60332,39643,51919,55477,50161,39644,50085,50936,60333,50378,39645,60334,60335,60336,47061,39646,39647,39648,47098,39649,60337,51170,39650,60339,39651,47780,53749,45233,60338,60340,39652,39653,39654,46506,49864,51176,39655,60341,39656,52142,58335,39657,39658,54208,39659,39660,39661,39662,55771,39663,39664,52641,54957,51187,39665,39666,39667,55776,48099,39668,47802,58338,39669,39670,39671,39672,39673,53163,39674,39675,39676,58336,51655,39677,47801,39678,39744,39745,53684,58337,51434,47535,48557,46040,52955,39746,39747,52416,39748,39749,39750,58344,58345,52724,39751,39752,39753,39754,39755,52397,39756,48307,39757,58346,39758,58347,39759,39760,53466,39761,39762,39763,50939,47066,39764,39765,51167,53962,52950,39766,58340,58348,39767,51698,46017,39768,39769,58343,39770,39771,50915,58341,39772,39773,60851,58342,39774,39775,39776,39777,51635,39778,50662,39779,39780,39781,47541,39782,50107,39783,58339,50621,49572,49881,45783,39784,58349,48038,50349,39785,58352,48858,39786,39787,58363,58357,47827,39788,39789,39790,39791,47056,54221,39792,54990,54739,47553,54708,53720,39793,39794,39795,39796,53433,51190,39797,39798,39799,51370,45748,39800,50138,39801,39802,39803,58350,39804,39805,58364,58351,47016,58359,58356,39806,39808,39809,47034,39810,39811,50594,39812,58358,50653,45736,50940,39813,50400,39814,39815,55202,39816,49377,58361,39817,39818,58362,58365,52393,58355,39819,54206,39820,45507,60852,58353,58354,39821,58360,53434,50883,54515,58366,39822,39823,48608,39824,39825,58535,39826,39827,58534,39828,39829,39830,53747,58531,39831,58537,39832,39833,39834,51447,39835,39836,39837,39838,53172,39839,58536,58542,49893,39840,39841,46772,39842,39843,39844,39845,39846,39847,48626,39848,58530,39849,39850,47849,58538,39851,39852,58540,39853,39854,46845,55006,58546,39855,58541,39856,39857,39858,58529,39859,48110,52701,51106,50633,39860,39861,49655,39862,58532,39863,51123,48556,48573,58533,39864,55239,45794,39865,58539,48323,58543,39866,48107,58544,50600,58545,39867,39868,39869,39870,54755,49059,39871,58554,39872,58551,39873,58555,39874,39875,58557,39876,39877,50902,39878,39879,47814,49355,39880,39881,39882,47265,58548,39883,39884,39885,39886,54433,39887,39888,47779,48638,39889,39890,39891,58556,39892,39893,39894,39895,39896,52671,39897,39898,50425,39899,39900,53243,51686,39901,39902,54207,39903,53201,39904,39905,58547,39906,58552,58553,52457,39907,39908,39909,39910,39911,52430,39912,49364,58549,49584,58550,52944,39913,48065,46547,39914,51443,48551,54727,51628,47266,58570,39915,39916,58572,53700,39917,39918,53946,39919,39920,47789,39921,39922,47828,39923,39924,39925,39926,39927,39928,58563,46573,39929,39930,39931,55245,58560,53245,58559,39932,39933,39934,49628,52426,40000,40001,40002,40003,51943,40004,40005,40006,40007,50391,40008,52436,58568,40009,40010,40011,58567,58561,40012,58564,46509,40013,40014,54233,40015,58566,40016,40017,40018,40019,54009,46307,40020,48052,40021,40022,51694,40023,46270,40024,40025,40026,48108,40027,53709,40028,52461,60853,40029,40030,40031,40032,40033,40034,40035,40036,51173,40037,40038,40039,40040,54440,40041,58571,55253,58562,40042,48549,58565,40043,40044,54246,40045,58569,51704,40046,40047,58558,40048,40049,54245,40050,40051,51198,46793,40052,54524,45747,58583,40053,40054,40055,52930,40056,58573,40057,52924,40058,47323,40059,40060,58582,40061,49098,40062,40064,40065,54222,40066,50156,40067,40068,40069,40070,40071,40072,40073,40074,50632,58584,40075,40076,40077,40078,40079,40080,40081,40082,52676,58575,40083,40084,40085,40086,58580,58581,40087,47870,40088,53222,40089,40090,54719,40091,40092,40093,58578,40094,40095,40096,40097,40098,40099,40100,40101,40102,40103,40104,58576,40105,40106,58574,40107,40108,40109,40110,40111,40112,40113,40114,40115,40116,40117,40118,40119,40120,40121,52709,51882,40122,40123,40124,49315,40125,48550,58579,40126,40127,47304,40128,40129,40130,40131,40132,58599,54452,40133,40134,40135,40136,40137,40138,40139,58587,40140,40141,40142,49647,40143,40144,58601,40145,40146,53991,40147,40148,58591,40149,58592,40150,40151,53162,40152,40153,40154,40155,52189,40156,58586,58577,40157,58597,40158,51420,58595,40159,40160,50407,58594,40161,58593,40162,40163,40164,46076,58600,40165,40166,40167,40168,46561,40169,40170,40171,55244,40172,40173,40174,58598,40175,48044,40176,55250,52431,60408,40177,58596,40178,40179,47606,40180,40181,40182,54989,58585,58588,49914,58590,40183,49867,49348,49872,40184,45557,52402,40185,40186,40187,40188,40189,40190,40256,40257,40258,40259,46542,40260,40261,40262,40263,58607,40264,40265,40266,40267,40268,40269,40270,40271,50863,40272,40273,40274,50913,40275,40276,58613,40277,40278,40279,40280,40281,49833,40282,40283,40284,49388,53725,58606,40285,40286,40287,40288,40289,40290,40291,40292,40293,40294,50350,40295,40296,40297,58605,40298,40299,40300,40301,58614,58612,49918,40302,58589,40303,58608,40304,51966,40305,54724,40306,40307,58609,40308,40309,40310,40311,40312,40313,40314,53754,40315,40316,40317,40318,40320,40321,40322,58603,58604,40323,40324,40325,58610,40326,52907,40327,40328,40329,40330,40331,40332,40333,40334,40335,40336,50635,40337,40338,40339,51121,40340,49850,40341,40342,40343,58602,40344,40345,40346,49610,40347,40348,40349,40350,40351,40352,52406,46001,40353,40354,40355,58619,40356,58611,40357,40358,40359,58618,40360,58621,40361,58620,40362,40363,40364,40365,40366,40367,40368,46030,40369,40370,40371,46010,58615,40372,40373,58617,58616,50668,40374,40375,40376,40377,40378,40379,40380,40381,40382,40383,40384,40385,40386,49341,40387,40388,40389,40390,54504,40391,40392,40393,40394,40395,58786,40396,40397,40398,40399,40400,40401,40402,40403,40404,40405,40406,45252,40407,40408,58788,40409,40410,58787,40411,40412,40413,40414,40415,40416,48292,40417,58789,40418,40419,40420,40421,40422,40423,58785,40424,40425,40426,40427,40428,40429,40430,58622,45556,40431,40432,40433,40434,40435,40436,40437,40438,40439,40440,40441,58792,40442,58793,58790,40443,40444,40445,40446,40512,40513,40514,40515,40516,40517,40518,40519,58791,58794,40520,40521,40522,40523,40524,40525,40526,40527,40528,40529,40530,40531,40532,40533,40534,40535,40536,40537,40538,40539,40540,40541,40542,40543,40544,40545,40546,40547,40548,40549,40550,40551,40552,50905,40553,40554,40555,40556,40557,40558,40559,40560,58795,58797,40561,40562,40563,40564,40565,40566,40567,58796,40568,40569,40570,40571,40572,40573,40574,40576,40577,40578,40579,40580,40581,40582,40583,40584,40585,58799,40586,40587,40588,58798,40589,40590,40591,40592,40593,40594,40595,40596,40597,40598,40599,40600,40601,40602,40603,40604,40605,40606,47584,40607,40608,58800,40609,40610,40611,40612,40613,40614,40615,40616,40617,40618,40619,40620,40621,40622,58801,40623,40624,40625,40626,40627,40628,40629,40630,40631,40632,40633,40634,48112,60641,50160,40635,46534,48082,40636,40637,40638,40639,49641,54510,40640,48836,40641,40642,40643,55238,40644,54486,45779,60606,40645,40646,40647,40648,60097,40649,40650,40651,49839,46262,40652,40653,40654,53719,40655,40656,40657,46004,40658,51378,49083,60608,40659,40660,54987,40661,40662,60607,60609,40663,40664,40665,40666,40667,40668,40669,40670,40671,40672,40673,40674,40675,60613,48870,52415,50650,48828,40676,60614,40677,45566,40678,40679,40680,60612,54696,46563,40681,60610,49590,46051,40682,40683,60611,52152,49347,52478,40684,40685,40686,40687,49618,40688,60616,40689,40690,40691,40692,40693,40694,40695,40696,40697,40698,40699,40700,40701,47846,49363,40702,55026,40768,40769,40770,53708,40771,40772,40773,40774,49086,40775,47027,51669,60615,48098,40776,52428,48637,51400,40777,53161,40778,40779,40780,40781,40782,40783,40784,52713,40785,50667,40786,40787,40788,47081,40789,40790,40791,40792,40793,40794,40795,40796,40797,40798,40799,53705,47800,40800,40801,40802,40803,40804,60617,40805,40806,60618,40807,48064,60619,40808,60642,45498,47065,40809,40810,40811,40812,40813,40814,40815,40816,40817,40818,40819,48569,40820,40821,40822,40823,40824,40825,40826,40827,60620,53734,60621,40828,40829,40830,40832,51387,40833,40834,40835,40836,40837,40838,40839,40840,40841,40842,40843,40844,40845,40846,60625,40847,40848,40849,40850,60627,40851,48077,40852,48357,40853,40854,40855,40856,40857,40858,40859,40860,40861,40862,40863,40864,40865,60623,40866,51639,40867,40868,40869,40870,40871,50106,40872,60643,54741,60624,40873,40874,40875,40876,40877,55027,40878,40879,40880,60626,60622,40881,40882,40883,40884,60628,40885,60629,40886,40887,51647,40888,40889,40890,40891,40892,40893,53160,40894,40895,40896,40897,40898,53468,40899,40900,40901,40902,53676,40903,40904,40905,40906,51419,40907,40908,40909,60630,52981,40910,40911,40912,40913,40914,51948,60634,40915,40916,40917,40918,40919,40920,40921,60633,40922,40923,40924,45246,40925,40926,40927,40928,40929,40930,60631,40931,60632,40932,40933,40934,60644,40935,40936,40937,40938,40939,40940,40941,40942,40943,51388,40944,40945,40946,40947,40948,40949,40950,40951,40952,40953,49607,40954,40955,40956,40957,40958,60636,53728,41024,41025,41026,41027,41028,41029,41030,41031,41032,41033,60635,41034,41035,41036,41037,54511,41038,60637,41039,41040,41041,41042,41043,41044,56262,41045,41046,41047,41048,41049,41050,41051,41052,41053,41054,60638,41055,41056,41057,41058,41059,41060,41061,41062,41063,41064,41065,41066,45484,41067,41068,41069,41070,41071,41072,41073,41074,41075,41076,41077,41078,41079,41080,41081,41082,41083,41084,41085,41086,41088,41089,60639,41090,41091,41092,41093,41094,41095,41096,41097,41098,41099,60640,41100,55206,41101,50624,41102,41103,41104,60348,45230,41105,41106,41107,48884,47288,53935,45270,46585,41108,55475,41109,52140,41110,58333,41111,41112,41113,41114,41115,41116,41117,50860,45286,41118,41119,41120,50630,60345,41121,41122,41123,41124,60346,41125,41126,41127,60347,41128,41129,53696,41130,50595,41131,60146,41132,50354,41133,50357,49358,41134,41135,41136,60147,50369,41137,52975,41138,41139,41140,41141,60144,60148,41142,41143,51708,41144,41145,51107,41146,41147,41148,52440,52990,41149,41150,41151,60149,60150,53164,49383,41152,41153,60151,41154,41155,41156,41157,41158,46783,60152,41159,60153,41160,60154,41161,41162,60155,41163,41164,41165,41166,41167,41168,41169,41170,41171,41172,41173,41174,60145,41175,41176,41177,41178,41179,41180,41181,41182,41183,41184,41185,41186,51374,57835,41187,47032,57836,41188,41189,41190,57837,41191,55220,57838,57839,54220,41192,41193,41194,41195,41196,41197,41198,57841,49137,57840,46546,41199,41200,41201,45495,41202,41203,41204,41205,57843,57842,41206,47868,41207,57844,41208,41209,41210,41211,47543,41212,48849,41213,41214,43584,43585,50428,43586,47837,48582,43587,43588,43589,43590,43591,43592,57845,57847,43593,43594,46784,53185,51880,57846,54776,54268,57848,57852,57849,43595,43596,57850,49386,43597,57854,58017,49351,43598,43599,43600,43601,57851,43602,57853,43603,43604,43605,43606,43607,43608,58021,43609,43610,43611,49620,43612,43613,43614,43615,58019,43616,58024,45822,58018,43617,43618,43619,50125,45762,58023,58022,43620,43621,58020,58025,43622,43623,58027,43624,43625,43626,53449,55021,50088,58028,43627,53207,43628,43629,58030,43630,43631,47855,43632,43633,59872,58029,58026,43634,43635,43636,43637,48043,54451,43638,43639,43640,43641,43642,43643,43644,43645,43646,43648,43649,43650,43651,58032,43652,43653,58031,43654,59873,43655,43656,43657,43658,58033,43659,43660,43661,43662,43663,43664,43665,43666,58034,43667,43668,43669,43670,43671,43672,43673,43674,43675,43676,43677,58035,52385,43678,58036,43679,43680,43840,43841,43842,43843,43844,43845,43846,43847,43848,43849,43850,43851,58037,43852,43853,43854,43855,43856,53502,43857,43858,49866,43859,54257,43860,52725,43861,43862,59360,43863,43864,59361,43865,43866,43867,43868,48833,43869,43870,43871,43872,49898,43873,43874,43875,59364,43876,43877,59363,43878,43879,43880,43881,43882,43883,52710,43884,50101,43885,43886,59362,48055,53206,43887,49633,59369,43888,43889,43890,59368,43891,43892,59380,45731,43893,43894,43895,43896,59370,43897,59366,43898,43899,43900,43901,43902,59372,59371,51642,43904,43905,54756,43906,59365,47017,59367,43907,43908,43909,43910,43911,43912,43913,59374,43914,43915,43916,43917,59379,43918,55017,43919,43920,43921,43922,59373,43923,59378,43924,59377,43925,43926,43927,45280,43928,43929,43930,43931,59381,43932,43933,43934,43935,43936,44096,44097,44098,44099,44100,44101,44102,44103,44104,44105,44106,51186,44107,49349,49389,44108,44109,49648,59376,44110,44111,44112,44113,59382,52214,44114,44115,44116,44117,44118,44119,44120,44121,44122,59554,59553,44123,44124,44125,44126,44127,44128,55233,44129,44130,59386,59385,44131,59387,44132,59383,44133,59390,44134,59389,44135,59388,44136,44137,49621,51161,50685,50627,44138,44139,44140,44141,44142,51181,44143,44144,44145,44146,59555,44147,44148,44149,44150,44151,44152,44153,44154,44155,44156,44157,44158,44160,44161,44162,44163,44164,44165,44166,59558,44167,59557,44168,59559,47863,59384,59556,44169,51440,51626,44170,44171,44172,44173,44174,44175,44176,44177,44178,44179,44180,44181,44182,59561,44183,44184,47589,44185,44186,44187,44188,44189,53758,59560,44190,44191,44192,44352,44353,44354,59562,44355,59565,59566,44356,49575,44357,44358,44359,59567,44360,44361,44362,59568,44363,44364,59564,44365,59572,44366,44367,44368,44369,44370,44371,44372,44373,44374,44375,44376,59563,44377,59569,44378,44379,44380,44381,44382,44383,44384,44385,59573,59570,59571,44386,44387,44388,44389,44390,44391,44392,44393,44394,44395,44396,44397,44398,44399,44400,44401,59575,44402,44403,44404,44405,44406,44407,44408,44409,44410,44411,44412,44413,44414,44416,44417,44418,44419,44420,44421,44422,44423,44424,44425,59574,44426,44427,44428,44429,44430,44431,44432,44433,44434,47567,44435,61612,44436,61613,44437,50864,45290,51391,44438,52703,44439,44440,44441,44442,44443,44444,44445,52941,60081,44446,44447,44448,44608,60082,44609,50879,46281,44610,44611,44612,44613,44614,44615,44616,60083,44617,44618,44619,44620,54759,44621,44622,44623,44624,44625,44626,44627,44628,56825,44629,60084,44630,60085,44631,60086,44632,44633,44634,44635,47306,57264,51701,44636,52464,44637,44638,51706,44639,44640,44641,44642,44643,51707,44644,44645,54211,52134,44646,47270,61614,45506,44647,58808,52463,54217,48343,51690,44648,46567,44649,50384,46569,44650,61102,48045,44651,44652,59358,44653,61103,44654,44655,44656,44657,45993,44658,44659,61106,44660,44661,61105,48615,44662,61104,52919,44663,44664,44665,44666,50639,44667,44668,44669,44670,49652,56270,61107,53491,44672,44673,44674,44675,44676,44677,44678,44679,49876,50920,44680,44681,44682,47020,44683,44684,44685,44686,44687,44688,44689,61108,44690,46059,44691,44692,44693,48123,61109,44694,44695,44696,44697,44698,59356,44699,44700,44701,61110,44702,44703,48558,44704,44864,44865,44866,61922,44867,44868,44869,51944,44870,53961,61658,44871,61659,44872,61660,49606,44873,47341,48846,44874,44875,61662,44876,50609,61661,53745,44877,61664,45260,48618,44878,44879,44880,44881,44882,53983,61663,44883,46255,47080,61670,61669,50851,61665,61666,46275,44884,44885,61667,54766,44886,44887,52443,48850,48306,44888,44889,44890,61672,61671,61668,45729,44891,54946,54200,48823,51372,44892,44893,61674,44894,44895,44896,44897,53751,44898,54988,47835,61673,44899,46779,44900,44901,52660,44902,44903,50854,44904,44905,44906,49569,61675,61678,44907,61677,61680,61676,44908,48062,61679,44909,44910,44911,44912,52405,61682,44913,44914,46037,44915,44916,44917,44918,45524,44919,44920,61683,44921,44922,61684,61686,46305,44923,61681,44924,61687,44925,44926,44928,44929,61690,44930,61688,44931,44932,44933,61685,44934,44935,44936,44937,61693,44938,61689,61692,61694,44939,61857,44940,44941,44942,52929,61860,44943,61859,44944,49654,61691,51933,44945,44946,46321,45553,52401,44947,61862,44948,44949,61863,44950,44951,61868,54734,61865,44952,44953,51379,44954,44955,44956,61858,44957,61867,61864,61861,44958,44959,61866,44960,45120,45121,45122,45123,45124,45125,45126,45225,61869,45127,45128,45129,45130,45131,45132,61871,45133,61873,45134,45135,45136,45137,45138,61872,45139,61870,45140,45141,45142,45143,53666,45144,45145,45146,45147,45148,45149,45150,61874,45151,45152,45153,61875,45154,45155,45156,45157,45158,45159,45160,45161,47599,45162,45163,46535,45164,45271,45273,45165,45166,45167,54509,45168,46532,45169,48596,48074,61607,45170,45171,47326,45172,45173,61608,45174,45175,45224,45176,61609,45177,45178,52718,45179,45180,61610,45181,45182,45184,45185,45186,45187,45188,45189,45190,45191,61611,45192,45193,45194,45195,45196,45197,45198,45199,45200,50852,45201,45202,55013,61924,45203,61925,45204,45205,45206,45207,45208,45209,45210,45211,45212,45213,50163,45214,45215,54235,45216,45376,54993,50664,45377,54191,45378,53990,45379,45380,61121,45243,54709,53710,48352,47824,45381,49144,45382,47303,46529,50636,45383,45384,51874,45385,45386,45387,50123,45388,45389,45390,45391,45392,61122,45393,45394,45395,45396,45397,45398,45399,45400,50367,46754,45401,60908,50084,45402,54961,45403,45404,45405,53216,60911,45406,45407,50638,45408,46812,45409,45410,51873,45411,45412,60909,45413,45414,60912,60913,50108,45415,49076,45416,60910,45417,45418,45419,45420,45421,45422,45423,45424,45425,45426,45427,60916,60914,45428,45429,45430,45431,54758,50143,45432,60915,45433,45434,45435,60918,45436,54691,53667,45437,45438,45440,60917,45441,50128,45442,45443,45444,45445,45446,60919,49140,48876,60920,45447,52471,45448,53723,45449,45450,45451,55237,54774,45452,60924,45453,45454,45455,60923,45456,45457,45458,45459,45460,45461,45462,45463,60921,60922,45464,45465,45466,45467,45468,45469,45470,45471,60925,48806,45472,45632,45633,45634,45635,52143,61089,46781,45636,61090,50368,45637,60926,45638,45639,48606,45767,45640,45641,45642,45643,45644,45645,45646,45647,45648,45649,45650,45651,46787,45652,45653,45654,61093,55482,61091,61094,45655,45656,45657,50153,46066,45658,45659,45660,45661,45662,45663,61095,61092,53177,45664,45665,61096,49911,45666,45667,45668,45669,45670,45671,45672,45673,45674,45675,45676,45677,61097,61098,45678,57003,45679,45680,50867,45681,51142,45682,55029,46537,45683,52146,45684,45685,45686,61099,45687,45688,52651,45689,61100,45690,45691,45692,45693,45694,54704,45696,61101,45697,63172,45698,45699,45700,45701,45702,45703,45704,45705,45706,45707,45708,45709,45710,56263,45711,45712,45713,45714,45715,45716,45717,45718,45719,46243,45720,45721,45722,50092,61926,45723,45724,45725,45726,45727,51896,53971,45728,54954,45888,61426,45889,48856,45890,48579,61427,46796,45227,45891,45892,45893,45894,51887,45895,45896,60854,45897,60855,45898,45899,45900,45901,52985,47023,49139,60856,49899,51632,45902,45903,45904,45905,45906,45907,60857,45908,45909,50934,49075,45910,45911,45912,60860,50680,45913,53712,45914,55209,60858,60859,45915,53730,45916,60863,60864,45917,60868,45918,45919,45920,60872,45921,60870,60878,54760,45922,60873,45923,45924,60871,60862,45925,45926,50665,45927,45928,45929,50886,45930,45931,51689,54482,60865,60866,60867,60869,45932,49401,45933,46241,45934,45935,45936,45937,47592,45938,60880,45939,45940,45941,45942,60881,45943,60874,45944,60879,45945,52984,45946,45947,52150,60876,60877,45948,45949,45950,45952,45953,53237,45954,45955,45956,45957,45958,45959,45960,45961,45962,45963,45964,45965,60882,49650,54194,60875,51383,45966,45967,45968,45969,45970,45971,45972,45973,48367,45974,45975,45976,45977,50672,45978,45979,45980,45981,45982,45983,45984,46144,46145,46146,60886,46147,46575,46148,46149,49845,45229,52201,46150,46151,45486,46152,60884,46153,46154,46155,52715,46562,46156,60885,60883,60887,46157,46158,46586,46159,60888,46160,60889,46161,60892,46162,45516,46163,46164,46165,46166,46167,46168,46169,46170,50678,48366,60890,52412,45802,46171,46172,46173,46174,60891,46175,46176,46177,46178,50411,46179,46180,46277,46181,46182,46183,45301,46184,46185,46186,60895,49370,46312,46187,46188,46189,46190,50637,46191,46192,46193,60893,49092,46194,46195,46196,60894,46197,46198,46199,46200,46201,46202,46203,46204,46205,46206,46208,46209,46210,46211,50341,46212,46213,46214,60896,46215,46216,46217,46218,46219,60897,46220,60899,46221,46222,49623,46223,46224,48071,46225,46226,46227,46228,46229,46230,48568,46231,46232,46233,60898,46234,46235,46236,46237,46238,46239,46240,46400,46401,46402,46403,46404,46405,60900,46406,46407,46408,46409,46410,46411,46412,46413,46414,46415,60902,46416,46417,46418,46419,46420,60901,46421,46422,46423,46424,46425,46426,46427,46428,46429,46430,46431,46432,46433,46434,46435,60903,46436,46437,46438,46439,46440,51902,60650,49393,46441,51687,46442,60651,50926,46443,46444,46445,46446,60652,46447,50925,60653,46448,46449,46450,46451,46452,46453,46454,46455,46456,60656,46457,46458,55270,60659,46459,46460,60657,60654,60655,55203,51697,52206,60660,46461,60658,46462,46464,53225,46465,60662,50865,46466,46467,46468,46469,48320,46470,60661,46471,46472,46473,46474,46475,46476,46477,46523,48118,46478,60663,46479,46480,46481,46482,46483,55799,48635,46484,46485,49851,60664,46486,46487,46488,46489,60665,46490,46491,46492,46493,47267,46494,46495,46496,46656,46657,46658,46659,46660,46661,46662,60666,46663,46664,46665,46666,46667,46668,46669,46670,46671,46672,46673,46674,60667,46675,46676,46677,46678,46679,46680,46681,46682,46683,46684,46685,60668,46686,46687,46688,46689,46690,54253,55470,49387,46691,51165,47820,46692,53475,52157,46693,52666,46694,46695,47313,46696,46697,45564,46698,51183,46699,54998,46700,46701,46702,49094,50155,46703,46704,61429,46705,46706,50136,46707,46708,46709,46710,46711,46712,55266,46713,46714,46715,61431,46035,46716,51160,53741,46717,54984,46718,61432,46720,61430,46721,48125,46022,46722,46723,46724,46725,46726,46727,46728,48597,46729,46730,53958,46731,48096,46732,46733,53153,46734,61436,61435,46735,46736,61433,46737,46738,46739,46740,46028,46741,51668,52144,46742,46743,46744,46745,46746,61438,46747,46748,45278,46749,46750,54985,46751,46752,46912,61437,46913,46061,46914,46915,63189,46916,46917,46918,46919,46920,46921,46922,46923,46924,46925,46926,46927,46928,46929,46930,52936,46931,46932,46933,61602,46934,61601,46935,46526,48346,48124,46936,47333,46937,46938,46939,46940,46941,46942,50370,46943,46944,46945,46946,46947,46948,46949,46950,46951,46952,61603,46953,46954,46955,46956,46957,52203,46958,46959,46960,46961,46962,46963,46964,46965,46966,46967,46968,46969,46970,46971,46972,46973,46974,46976,46977,46978,46979,46980,46981,46982,61606,46983,46984,46985,53672,46986,48831,51182,61878,61879,49109,46987,46988,46989,46990,46249,61880,52667,46991,51156,54701,46992,61881,46993,61882,46994,46995,46996,46997,51151,46998,46999,47000,53924,54991,47001,47002,61883,48593,46256,48829,47003,47004,47005,46300,52945,47006,49119,61885,47007,47008,47168,47169,49146,61884,47170,61887,47171,47172,47173,61886,61888,47174,47175,47176,47177,47178,61889,47179,47180,47181,47182,47183,47184,47185,47186,47187,47188,47189,49662,47190,47191,47192,47193,47194,47195,47196,47197,47198,47199,47200,49570,47201,47202,47203,47204,47205,47206,47207,47208,47209,47210,51962,47211,47212,54718,47213,47214,47215,47216,48826,48825,54722,47217,47218,49058,47219,52655,61877,47220,47221,47222,47223,47224,47225,48607,47226,46795,47227,47228,47229,47230,47232,47233,47234,47235,47236,55025,62403,47237,47238,62404,47239,47309,47240,47241,47242,62406,62407,47243,45258,47244,62405,47245,62409,52209,47246,47247,47248,62411,47249,53414,47250,47251,45514,62408,47252,47253,47254,62415,47255,46545,47256,47257,62423,47258,62418,47259,47260,47261,62420,62419,47099,47262,45503,47263,62414,62410,46554,47264,62416,47424,47425,62417,47426,62421,47427,47428,47429,47430,62413,47431,48355,47432,49661,47433,62422,47434,47435,47436,47437,47438,47439,62426,47440,62412,47441,46536,47442,48622,62428,47443,47444,47012,49136,55038,52658,47445,46320,47446,45791,47447,62424,47448,62425,51640,47449,62429,47450,47451,62430,47452,62433,47453,47454,47455,47456,47457,47458,47459,47460,47461,47462,47463,62431,47464,47465,62435,62434,47466,47467,62427,47468,49130,47469,46063,47470,62432,47471,47472,51113,47473,48370,47474,47475,47476,47477,62443,47478,47479,47480,47481,47482,47483,47484,47551,47485,47486,62436,47488,47489,47490,45741,48126,47491,52195,47492,47493,47494,47495,62445,62441,47496,47497,47498,47580,62446,47499,47500,47501,62437,62438,62442,49889,62444,62447,62440,48381,47502,47503,47504,53220,47505,47506,62448,47507,47508,47509,62439,47510,47511,47512,47513,47514,47515,47516,47517,62450,47518,47519,47520,47680,55213,50858,47681,47682,47683,47684,62451,47685,47686,47687,47688,62449,47689,49832,47690,47691,47692,47693,47694,47325,62453,47695,47696,62452,47697,47698,47699,46299,47700,47701,47702,62454,62455,47703,47704,47705,62456,47706,47707,47708,49338,47709,47710,49385,47711,47712,47713,47714,47715,50673,47716,47717,47718,47719,62459,47720,62458,47721,47722,47723,47724,47725,47726,47727,47728,46296,47729,47730,47731,62462,62457,47732,47733,62460,47734,47735,47736,47737,47738,47739,62461,47740,47741,47742,47744,47745,47746,47747,47748,62625,47749,47750,47751,47752,47753,47754,62627,48073,47755,47756,62626,47757,47758,47759,47760,47761,47762,47763,47764,47765,47766,47767,47768,47769,62628,47770,47771,47772,47773,47774,47775,45758,62630,62629,47776,47936,47937,47938,47939,47940,47941,47942,47943,47944,47945,48302,47946,47947,47948,47949,47950,47951,47952,47953,47954,47955,47956,47957,47958,47959,47960,47961,47962,47963,47964,47965,47966,47967,47968,47969,47970,47971,47972,47973,47974,47975,47976,47977,47978,47979,47980,47981,47982,50135,55777,47983,47984,47985,47986,47987,47988,49376,62668,55249,47989,47990,47991,47992,47993,47994,47995,47996,47997,47998,48000,47067,48001,48002,48003,48004,48005,48006,48007,62670,49571,48008,48009,50889,48010,46294,54707,48011,48012,48013,62672,62671,62673,52186,48014,48015,62674,48016,54465,55008,48017,48018,48019,48020,47072,48021,48022,48023,49592,48024,48025,49595,62675,48812,48026,48027,48028,48029,48030,46306,48031,48032,62676,62677,48811,48192,48193,62678,48194,48195,48196,62683,48197,62679,62682,48198,47869,48199,62680,62681,48200,48201,48202,48203,48204,48205,48206,47330,52423,62684,48207,45786,48208,48209,50131,48210,48211,54499,49079,48212,48213,48214,48215,48216,48217,48218,62685,48219,48220,48221,48222,48223,48224,50612,48225,48226,48227,48228,48229,48230,48231,48232,62697,48233,48234,53173,48235,48236,48237,48238,48239,48240,48241,48242,48243,48244,48245,48246,48247,48248,52937,48249,48250,48251,48252,48253,48254,48256,48257,48258,48259,48260,48261,48262,48263,48264,48265,48266,48267,48268,48269,48270,52184,48271,52215,48272,48273,48274,48275,48628,48276,48277,48278,55247,48279,48280,48281,49371,48282,48283,48284,48285,48286,48287,48288,48448,48449,48450,48451,48452,48453,48454,48455,48456,48457,48458,48459,48460,48461,48462,48463,48464,48465,48466,48467,48468,48469,48470,48471,48472,48473,48474,48475,48476,48477,48478,48479,48480,48481,48482,48483,48484,48485,48486,48487,48488,48489,48490,48491,48492,48493,48494,48495,48496,48497,48498,48499,48500,48501,48502,53493,48503,48504,48505,48506,48507,48508,48509,48510,62698,48512,48513,48514,48515,48516,48517,48518,48519,48520,48521,48522,48523,48524,48525,48526,48527,48528,48529,48530,48531,48532,48533,48534,48535,48536,48537,48538,48539,48540,48541,48542,48543,48544,48704,48705,48706,48707,48708,48709,48710,48711,48712,48713,48714,48715,48716,62699,48717,48718,48719,48720,48721,48722,48723,62700,48724,48725,48726,48727,48728,48729,48730,48731,48732,48733,48734,48735,48736,48737,48738,48739,48740,48741,48742,48743,48744,48745,48746,48747,48748,48749,48750,48751,48752,48753,48754,48755,48756,48757,48758,48759,48760,48761,48762,48763,48764,48765,48766,48768,48769,48770,48771,48772,48773,48774,48775,48776,48777,48778,48779,48780,48781,48782,48783,48784,48785,48786,48787,48788,48789,48790,48791,48792,48793,48794,48795,48796,48797,48798,48799,48800,48960,48961,48962,48963,48964,48965,48966,48967,48968,48969,48970,48971,48972,48973,48974,48975,48976,48977,48978,48979,48980,48981,48982,48983,48984,48985,48986,48987,48988,48989,48990,48991,48992,48993,48994,48995,48996,48997,48998,48999,49000,49001,49002,49003,49004,49005,49006,49007,49008,49009,49010,49011,49012,49013,49014,49015,49016,49017,49018,49019,49020,49021,49022,49024,63459,49025,49026,49027,49028,49029,47025,49030,49031,49032,49033,49034,62701,49035,49036,49037,49038,49039,49040,49041,49042,49043,49044,49045,49046,49047,49048,49049,49050,49051,49052,49053,49054,49055,49056,49216,49217,49218,49219,49220,49221,49222,49223,49224,49225,49226,49227,49228,49229,49230,49231,49232,49233,49234,49235,49236,49237,49238,49239,49240,49241,49242,49243,49244,49245,49246,49247,49248,49249,49250,49251,55275,49252,49253,49254,49255,49256,49257,49258,49259,49260,49261,49262,49263,49264,49265,49266,49267,49268,49269,49270,49271,49272,49273,49274,49275,62702,49276,49277,49278,59129,48832,59130,47852,59131,53195,59132,54460,48310,59133,59134,48333,51410,52915,59297,49280,46271,59298,51636,47321,50377,49281,55261,49882,47063,54973,52934,47044,49282,49283,50598,59299,53215,59300,59301,59302,49591,55273,51696,53176,54959,54997,59303,45293,59304,59305,51676,53999,48813,59306,45299,51422,48609,59307,51398,49284,59308,48102,47352,53668,59309,49895,48888,48586,52659,59310,59311,48878,53477,49285,52199,52432,48332,59312,48296,53495,59313,49286,53496,59314,59315,46274,59316,59317,51710,52908,50144,59319,45505,46065,49287,59320,59321,55259,54720,59322,49868,55226,59323,59324,59325,48362,50149,49346,59326,59327,48297,49288,59328,59329,59318,46800,59330,49289,59331,59332,48058,46558,49862,45536,59333,54453,59334,47295,59336,59335,47084,49290,59337,45816,59338,59339,59340,59341,59342,59343,59344,54183,52213,59345,59346,59347,59348,51657,59349,59350,59351,59352,59353,48585,59354,62398,49291,47319,49292,51377,49293,49294,49295,49296,49297,49298,49299,62399,49300,62400,62401,49301,49302,49303,49304,49305,49306,49307,49308,49309,49310,47582,52728,49311,49312,55528,47793,49472,49886,61111,49473,47011,49474,49475,49476,49477,61113,49478,61112,45269,49479,49480,49481,49482,49483,61115,54742,55279,49484,49485,49486,54979,49487,49488,61117,51952,49489,61116,49490,49491,49492,49493,61118,49494,49495,49496,49497,61120,49498,49499,61119,49500,49501,49502,49503,49504,49505,49506,49507,53746,49508,51132,49509,50112,49510,49511,49512,49513,49514,47329,49515,49516,49517,49518,49519,49639,49520,49521,62662,53471,62663,49522,53211,49523,49524,51386,49525,49526,62664,49527,49528,49529,49530,49531,49532,49533,62665,62666,49534,62667,49536,49537,49538,49539,49540,55802,47358,49541,49542,58865,54256,49543,62688,49544,52940,49545,49546,49547,46049,49548,49549,49550,49551,61876,49552,53998,49553,62689,49554,49555,49556,49557,49558,53224,62690,49559,49560,51148,49561,49562,49563,49564,49565,49566,46548,46308,62692,49567,49568,49728,62691,62693,49729,49730,62694,49731,49732,49733,49734,62695,49735,47794,45247,49736,62696,49737,49738,49739,49740,49741,49742,49743,47021,53997,49744,49745,49746,53931,49359,49747,49084,60323,54751,60104,49748,49749,49750,49751,61939,46840,52131,49752,49753,50381,49754,61927,49755,61928,47355,61929,47812,54469,45266,49756,49757,61930,49758,49759,49760,61931,49761,61932,49762,49763,61933,61934,61935,61937,61936,50645,49764,49765,49766,49767,49768,49769,61938,49770,46842,49771,61940,53934,57031,52170,49772,49773,46044,49774,46498,49775,47522,49776,49777,50420,61941,49778,49779,61942,49780,49781,49782,49604,49659,54960,61943,49783,49784,49785,49786,61944,49787,49578,49788,49789,49790,50872,49792,48859,49793,49794,49795,49796,49797,49798,49799,49800,49801,49802,49803,49804,49805,49806,61945,46287,49807,49808,49809,49810,49811,49812,61946,49813,49814,49815,49816,49817,49818,49819,49820,49821,49822,49823,49824,49984,60850,60849,49985,49986,52192,53982,49987,52161,54744,49988,51426,49989,49375,48289,49990,49991,49992,49993,49994,49995,60353,49996,49997,53412,49998,55010,49999,46791,47320,60352,47310,50000,60351,45990,47561,54955,50001,47092,47050,50002,50003,50004,48359,47038,60358,50005,60359,45241,49103,50006,60357,54269,50007,60360,50008,50009,60361,50010,50011,47054,50012,60354,60356,51702,54999,54733,53426,60367,52920,60368,50013,46504,50014,50015,50016,50017,50018,45491,60370,52389,50019,50020,50021,50022,50023,50024,50025,50646,60371,50026,60369,50655,60366,51876,60373,45307,50027,50028,47866,50029,50030,55479,61923,50031,60362,60363,60364,60365,60374,59072,60377,50032,49128,53960,60375,60380,47340,60376,50033,48570,50034,53464,50035,45239,50036,60381,50396,50037,50038,50039,50040,54956,50041,50042,50043,46304,50044,50045,49910,48313,50046,50048,60378,60379,54496,50922,50388,60383,50599,55797,50049,45745,50050,60388,50051,48581,50052,50053,50054,60386,50055,50056,50057,50058,50059,50060,50061,50062,50063,50064,50065,50066,50067,60387,50068,50069,47276,50070,52689,60389,50071,50072,50073,60385,50074,49587,50075,50076,50077,50078,50079,50850,50080,50240,50241,50242,50243,50244,50245,52467,50246,60390,50247,49328,53944,60391,50248,50249,50250,47279,47277,50251,60392,51131,52723,50252,50253,50254,60394,60395,50255,50256,50257,50258,50259,60397,50260,50261,50262,50263,53448,50264,60402,50265,60398,50266,50267,50268,60401,51449,50269,53756,60396,50270,50271,60393,50272,50273,50274,50275,47289,53209,50405,60399,60400,52442,52680,45298,50276,60406,50277,50278,50279,50280,50281,60405,50282,45746,50283,50284,50285,50286,47328,50287,60407,50288,50289,50290,50291,50292,50293,45548,50294,50295,52421,50340,53157,50296,50297,50298,50299,50300,60409,50301,50302,60578,50304,50674,50305,60410,50306,50307,50308,50309,50310,50311,50312,50313,51653,50314,50315,50316,50317,50318,50319,58079,60414,50320,50321,50322,50323,52686,60577,45531,54199,50324,50325,53980,50326,50327,50328,60413,50329,60411,50330,50331,50332,50333,50334,50335,50336,50496,50497,50498,50499,50500,50501,50502,50503,50504,50505,50506,50507,50508,50509,50510,46012,50511,50512,50513,60080,50514,50515,55252,50516,62635,46068,50517,50518,50519,50520,50521,54977,54978,50522,50523,50524,50525,50526,50527,54761,48842,50528,62631,50529,53928,62632,62633,50530,62634,48843,54239,50531,50532,50533,50534,50535,51680,51681,50536,50537,62402,50538,51942,50539,52466,50540,50541,50542,50543,50544,50545,58038,52148,50546,52968,55003,50547,62637,62638,62639,50548,50549,50550,50551,62642,50552,47805,62643,45283,62640,50553,62641,48546,45781,50554,62646,62647,46822,45744,53199,62644,46252,50555,62645,50556,50557,62648,50558,50560,50561,50562,50563,62649,50564,50565,52647,50566,62650,50567,62651,50568,50569,50570,62652,50571,50572,50573,50574,50575,50576,50577,50578,52178,50579,62653,50580,50581,50582,50583,62654,50584,50585,50586,50587,50588,50589,50590,50591,62655,50592,50752,50753,50754,50755,62686,49596,48360,50756,51627,53726,58869,50757,50758,50759,50760,56499,53973,50761,50762,56500,45228,56501,50763,50764,48602,50765,56505,50766,50767,50768,55490,50769,56503,54259,50770,51670,56506,56502,50771,56507,50082,50772,50773,50774,50775,56508,56517,56509,50776,50777,52959,54949,50778,56527,50779,56525,50780,50781,56530,48614,49835,50782,56504,56523,56526,56510,47058,45253,56519,53438,56513,48040,50783,47036,56524,50784,50785,56518,56511,51163,50786,50787,50788,53695,56512,50789,50790,56522,50791,50792,56528,50793,50794,52909,56514,50795,56515,56520,56521,45780,56529,52181,50796,54455,56539,56543,52390,56550,50797,50151,56540,50798,50799,49089,56537,50800,45306,47542,56549,56531,50801,56516,56534,51444,49120,50802,50803,50804,50805,51643,50806,50807,50808,45501,50809,54178,50810,50811,56538,50812,50813,56533,50814,50875,50816,56542,50817,50818,50819,50820,50821,55234,50095,47030,51153,50089,56546,56536,56555,56532,50822,50823,56541,50824,48805,56535,50825,56544,50826,50827,56547,56548,50828,56568,50829,50830,56545,56738,56551,50831,50832,50833,50834,50835,50836,50837,50838,50839,50840,48363,46276,50841,50842,50083,45799,56570,50843,56562,50844,56559,50845,56572,56558,54000,45800,50846,51415,51427,56571,50847,56557,50848,51008,51009,56567,51010,51011,56565,51012,51013,48803,56564,51014,45789,51015,51016,51017,51018,51019,56563,48374,56552,48068,51020,49395,51021,51022,51023,51024,51025,48340,56553,56554,51026,56561,56566,56569,46516,51027,51417,48103,56574,56573,54187,56737,56739,56741,54001,56740,56742,56743,53929,51028,51029,51030,51031,51032,51033,51034,47817,56745,51035,51036,56758,56753,56756,51037,51038,51039,51040,51041,51042,51043,56752,50894,51044,51045,49394,51046,51047,51048,51049,51631,51050,51051,51052,56556,56750,51053,51054,51055,51056,56759,51057,51058,56560,56751,51059,56760,51060,56748,51061,51062,51063,51064,51065,51066,51067,56761,56755,56749,50346,51068,51069,51070,51072,56744,49331,49579,56746,56747,51073,56754,48113,56757,54184,56762,51074,56763,50087,51075,51076,56786,56764,51077,51078,51079,56785,51080,47549,51081,51082,48853,51083,48890,51084,51085,47818,51086,51087,51088,51089,56778,51090,56773,51091,56767,51092,51093,51094,45771,56771,51095,56779,45732,56789,51096,51097,51098,56766,51099,51100,51101,50896,56784,51102,51103,51104,51264,51265,56788,49634,47046,51266,51267,51268,51269,51270,56782,56783,51271,51272,51273,56772,51274,51275,51276,56765,51277,56781,52433,51278,56777,51279,51280,51281,51282,56770,50120,50876,52910,56780,51283,56776,51284,51285,51286,51287,51288,51289,56769,51290,51291,51292,56774,49884,51293,51294,51295,51296,51297,51298,54185,54186,56787,53236,51448,51299,51300,51301,51302,51303,51304,51305,51306,56806,51307,51308,51309,51310,51311,51312,56775,51313,51314,51315,56800,49892,51316,51317,51318,51319,51320,51321,51322,51323,56801,51324,51325,51326,51328,51329,51330,51331,51332,51333,51334,56791,51335,51336,51337,51338,51339,55032,51340,56793,56792,47344,56790,51341,51342,51343,51344,50895,51345,46765,51346,51347,51348,51349,51350,56802,51351,47865,54497,56807,51352,51353,51354,46288,51355,56794,51356,49147,56803,51357,56799,51358,56797,51359,51360,51520,51521,51522,51523,51524,46553,51525,51526,51527,51528,56795,56796,56798,51529,48559,56804,51530,56805,51531,51532,51533,51534,51535,51536,51537,51538,56821,51539,50121,51540,51541,52194,51542,51543,51544,51545,56818,51546,51547,51548,51549,51550,51551,51552,51553,51554,51555,51556,51557,51558,55521,51559,51560,50897,51561,56820,51562,51563,51564,54772,56819,56816,51565,51566,56812,51567,56815,51568,56808,51569,51570,53486,51571,51572,51573,51574,51416,56814,51575,51576,56809,51577,51578,56810,52210,51579,56813,51580,51581,45517,51582,51584,51585,51586,51587,51588,49334,51589,48315,56817,51590,51591,56823,51592,56822,56811,51593,51594,51595,51596,51597,50670,51598,51599,51600,56827,51601,51602,51603,51604,51605,51606,51607,51608,51609,51610,51611,56996,51612,51613,56995,51614,51615,51616,51776,51777,51778,51779,51780,51781,51782,51783,51784,56824,51785,51786,51787,51788,50159,51789,49915,51790,51791,51792,54753,51793,51794,52917,51795,51796,51797,51798,56829,51799,45772,51800,51801,51802,51803,51804,51805,51806,51807,51808,50408,51935,51809,51810,51811,51812,51813,51814,51815,51816,51817,51818,51134,56826,56828,56830,56994,45226,45518,51819,51820,51821,51822,51823,57004,51824,51825,51826,51827,56998,48566,51439,51828,51829,51830,51831,51832,51833,51834,51835,51836,51837,51838,56993,51840,51841,56997,51842,51843,51844,51845,57001,51846,51847,51848,51849,51850,57000,51851,51852,51853,56999,51854,51855,51856,51857,51858,51859,51860,51861,51862,57005,51863,54476,51864,51865,51866,51867,57011,57002,57006,51868,51869,49369,51870,51871,51872,52032,52033,45473,57014,52034,57009,52035,52036,52037,52038,52039,52040,52041,57010,52042,52043,52044,52045,52046,52047,52048,52049,52050,52051,52052,53670,57013,52053,52054,52055,52056,52057,52058,52059,57007,52060,52061,52062,57008,52063,53437,52064,52065,52066,57012,51949,57017,52067,52068,52069,52070,52071,52072,57016,52073,57015,52074,52075,52076,52077,52078,52079,52080,57019,52081,52082,52083,52084,52085,52086,52087,48613,52088,52089,52090,52091,52092,45784,50154,52093,52094,57018,52096,50618,52097,52098,52099,52100,52101,52102,57020,52103,52104,52105,52106,52107,52108,52109,52441,52110,52111,52112,52113,47018,52114,52115,52116,52117,52118,52119,52120,52121,52122,52123,52124,52125,52126,52127,52128,52288,52289,54501,52290,52291,52292,57021,52293,52294,52295,52296,52297,57023,52298,52299,52300,52301,52302,52303,52304,52305,52306,52307,52308,50338,52309,52310,52311,52312,57025,52313,52314,52315,52316,52317,52318,52319,52320,52321,52322,52323,52324,52325,52326,52327,52328,57022,52329,57024,52330,52331,52332,52333,52334,52335,52336,52337,52338,52339,52340,52341,52342,52343,54714,52344,52345,52346,57026,52347,52348,52349,52350,52352,52353,52354,52355,52356,52357,52358,52359,52360,52361,52362,52363,62126,48034,49842,50608,49863,52364,52365,62127,52366,52367,52368,52369,52370,53481,52371,52372,52373,54237,52374,52375,52376,60349,52377,52378,52379,52380,52381,52382,52383,52384,46054,62128,52544,62129,52545,52546,51885,52547,52548,52549,52550,52551,52552,52553,47847,62131,62133,62132,52196,53178,62130,51892,53967,49900,52554,52555,52556,52557,52558,52559,52560,52931,62136,45302,62135,52561,52562,52563,52564,52565,62142,52566,45775,52567,52568,52569,52570,52571,52572,53697,62138,52573,52574,52575,52576,52577,62140,54505,52578,52579,62139,62134,62143,62141,52580,62137,52581,52582,62151,62148,62150,52583,52584,62154,62146,62144,52585,52586,52587,62149,52588,52589,52590,52591,52592,55035,52593,52594,52595,62145,52596,51193,51679,52597,62152,47558,46512,52598,52599,62147,62153,62160,62166,52600,52601,48087,52602,52603,52604,62165,52700,52605,55019,52606,52608,62162,62164,52609,52610,52611,52612,47346,52613,52614,52615,52616,62155,52617,52618,52619,62158,49913,52620,54749,62156,62157,62159,62163,52621,52622,52623,62169,54204,52624,52625,52626,52627,46826,52628,51953,52629,47076,62167,52630,52631,52632,62168,62170,62173,62171,52633,52634,62172,52635,52636,52637,52638,53713,62161,52639,52681,52640,52943,54953,52800,62179,52801,50139,52802,62176,52803,52804,49327,62188,62174,52805,62177,52806,52807,52808,62184,52809,52810,52811,52812,62178,52813,52814,62183,52815,52816,62182,52817,52818,62185,52819,52820,52821,62175,52822,52823,62180,62186,52824,52825,52826,52827,52828,52829,52830,54188,62181,45813,52831,52832,62194,52833,53419,52834,52835,52836,52837,62197,52838,52839,52840,48072,52841,62201,52842,52843,52844,52845,52846,52847,62192,52848,52849,62198,62200,62202,52850,52851,52852,52853,52854,52855,52856,52857,62195,52858,62193,52859,52860,52861,47867,52862,46587,52864,52865,52866,52867,62191,62199,62189,62190,52868,52869,52870,62187,62374,52871,62371,52872,52873,62370,52874,52875,62196,52876,51418,52877,52878,52879,52880,52881,62203,52882,52883,52884,62373,52885,52886,52887,52888,52889,52890,52891,50168,52892,52893,52894,52895,52896,53056,53057,53058,62205,53059,53060,62375,62377,62372,53061,62204,53062,53063,53064,62379,53065,62378,53066,53067,53068,53069,49885,53070,53071,62382,53072,53073,62384,53074,53075,53076,53077,53078,62369,53079,53080,53081,62385,62380,53082,53083,53084,53085,53086,62383,62206,62381,53087,53088,53089,53090,53091,53092,53093,62386,53094,53095,53096,53097,62388,53098,53099,53100,53101,62376,53102,53103,53104,53105,62387,53106,53107,53108,62389,53109,53110,53111,53112,53113,53114,53115,53116,53117,53118,53431,53120,53121,53122,53123,62392,53124,53125,53126,53127,55801,53128,53129,53130,53131,53132,53133,62393,53134,53135,53136,53137,53138,53139,53140,53141,62391,53142,51428,62390,53143,53144,53145,53146,62394,53147,53148,53149,53150,53151,62395,46272,53152,53312,53313,53314,53315,53316,53317,53318,53319,53320,53321,53322,53323,53324,53325,61123,53326,53327,53328,53329,53330,53331,62396,53332,53333,62397,53334,53335,53336,53674,53337,53338,53339,62636,53446,53340,53341,53342,53343,53344,53345,53456,53724,53346,53347,53348,53349,53350,53351,53198,53352,53353,48598,53354,53699,53355,53356,53357,53358,53359,53360,53361,47842,57833,53954,61890,45753,53362,53363,45549,61891,53364,51648,46020,53365,55794,53366,52133,53367,61892,53368,53369,53370,53371,54996,53372,53373,53374,53376,53377,61893,62656,61894,53378,54444,61895,53379,45248,62657,53380,53381,62658,53382,53383,46332,53384,50651,53385,53386,53387,53388,52411,53389,53390,53391,53476,53392,53393,53394,53395,53396,52704,53397,53398,53399,53400,53401,61896,53402,55795,53403,53404,53405,53406,53407,53408,45499,53568,53166,53569,53570,53571,47268,53572,53573,53574,53575,53576,61898,53577,53578,53579,53580,61899,53581,53582,53583,53584,45763,49617,53585,53586,55216,61897,53587,53588,61900,53589,53590,53591,53592,61902,53593,53594,53595,55798,53596,53985,54435,53597,53598,62659,51385,53599,53600,53601,53602,53603,62660,53604,53605,61901,61903,49123,61904,53606,53607,61908,53608,53609,53610,53611,53612,53613,53614,61910,61905,53615,51665,50657,53616,53617,53618,49891,47612,53619,53620,61907,53621,61909,53622,53623,53624,47571,53625,53626,53627,53628,53629,53630,53632,61915,53633,53634,53635,53636,53637,47830,53638,45309,61913,53639,53640,53641,53642,53643,61912,61906,61914,53644,53645,53646,53647,53648,61911,53649,53650,53651,51436,53652,53653,53654,53655,52682,61917,53656,53657,53658,53659,58813,53660,53661,53662,61916,53663,61918,53664,53824,53825,53826,53827,53828,53829,53830,53831,53832,61919,53833,53834,53221,53835,53836,53837,53838,53839,53840,53841,53842,53843,53844,53845,53846,53847,53848,53849,53850,53851,53852,53853,53854,53855,53856,53857,53858,53859,62661,48627,53860,53861,53862,53863,53864,53865,61920,53866,53867,53868,53869,53870,53871,53872,53873,53874,53875,53876,53877,53878,53879,53880,53881,53882,53883,53884,53885,61921,53886,53888,53889,52983,53890,53930,53891,61947,53892,53893,47282,53894,53895,53896,53897,53898,53899,53900,53901,53902,53903,53904,53905,53906,53907,53908,53909,53910,53911,53912,53913,53914,53915,53916,53917,53918,53919,53920,54080,54081,54082,54083,54084,54085,54086,54087,54088,54089,54090,54091,54092,54093,54094,54095,54096,54097,54098,54099,54100,54101,54102,54103,54104,54105,54106,54107,54108,54109,54110,48379,47579,54111,47590,50137,51923,60136,49344,48885,60137,60138,60139,54112,60140,60141,60142,60143,48583,54113,54114,54115,62971,54116,54117,54118,62973,54119,62974,54120,62972,54121,54122,54123,54124,48610,54125,63137,46245,54126,54127,54128,54129,63138,54130,54131,54132,63139,54133,54134,54135,60594,54136,54137,54138,54139,54140,54141,54142,54144,54145,54146,54147,54148,53716,54149,54150,54151,54152,54153,54154,55786,54155,54156,54157,54158,54159,54160,54161,54162,54163,54164,54165,54166,54167,54168,54169,54170,54171,54172,54173,54174,54175,54176,54336,54337,54338,54339,54340,54341,54342,54343,54344,54345,54346,54347,54348,54349,54350,54351,54352,54353,54354,54355,54356,54357,54358,54359,54360,54361,54362,54363,54364,54365,54366,54367,63140,54368,54369,54370,54371,54372,54373,54374,54375,54376,61114,54377,54378,54379,54380,54381,54382,54383,54384,54385,54386,54387,54388,54389,54390,54391,54392,54393,54394,54395,54396,54397,54398,54400,54401,54402,54403,54404,54405,54406,54407,54408,54409,54410,54411,54412,54413,54414,54415,54416,54417,54418,54419,54420,54421,54422,54423,54424,54425,54706,54426,54427,54428,54429,54430,54431,54432,54592,54593,54594,54595,54596,54597,54598,54599,54270,52444,54600,54601,54602,54603,54604,54605,54606,54607,51908,54608,54609,54610,54611,54612,54613,54614,54615,54616,54617,54618,54619,54620,54621,54622,54623,54624,54625,54626,54627,54628,54629,54630,54631,54632,54633,54634,54635,54636,54637,54638,54639,54640,54641,54642,54643,54644,54645,54646,54647,54648,54649,54650,54651,54652,54653,54654,54656,54657,54658,54659,54660,54661,54662,54663,54664,54665,54666,54667,54668,54669,54670,54671,54672,54673,54674,54675,54676,54677,54678,54679,54680,54681,54682,54683,54684,54685,54686,54687,54688,54848,54849,54850,54851,54852,54853,54854,54855,54856,54857,54858,54859,54860,54861,54862,54863,54864,54865,54866,54867,54868,54869,54870,54871,54872,54873,54874,54875,54876,54877,54878,54879,54880,54881,54882,58816,54883,54884,54885,54886,54887,54888,54889,54890,54891,54892,54893,54894,54895,54896,54897,54898,54899,54900,54901,54902,54903,54904,54905,54906,54907,54908,54909,54910,54912,54913,63141,54914,54915,54916,54917,54918,54919,54920,54921,54922,54923,54924,54925,54926,54927,54928,54929,54930,54931,54932,54933,54934,54935,54936,54937,54938,54939,54940,54941,54942,54943,54944,55104,55105,55106,55107,55108,55109,55110,55111,55112,55113,55114,55115,55116,55117,55118,55119,55120,55121,55122,55123,55124,55125,55126,55127,55128,55129,55130,55131,55132,55133,55134,55135,48815,55136,55137,55138,55139,55140,50857,55141,55142,55143,55144,55145,55146,55147,55148,55149,55150,55151,55152,55153,55154,55155,55156,55157,55158,55159,55160,55161,55162,55163,55164,55165,55166,55168,55169,55170,55171,55172,55173,55174,55175,55176,55177,55178,55179,55180,55181,55182,55183,55184,55185,55186,55187,55188,55189,55190,55191,55192,55973,48326,46761,47292,51407,48293,55974,55975,52438,51395,55976,50941,55193,53685,53993,53686,48327,55194,48562,48100,55977,55978,53704,55979,53485,46831,49883,55195,52175,47085,51688,47043,48887,54948,55980,55981,50880,55271,51894,55196,54697,52191,54767,55982,55007,46282,55984,55983,55197,53995,55985,55986,55987,51924,55988,51883,55989,55990,46031,55023,55991,48048,46510,55992,55993,47598,53679,53992,55994,47299,53226,45807,55995,55996,55198,48619,52956,54255,55997,52979,55998,54229,48101,55999,52149,52176,56000,51179,55022,56001,50613,46785,56002,47052,49102,56003,56004,52141,56005,46583,56006,49602,55227,56007,52408,55199,53994,50353,56008,46589,48081,56009,53427,56010,56011,52925,56012,56013,56014,45815,56017,56015,53736,56016,50133,56018,55200,56019,56020,56021,53435,53925,45305,56022,51115,56023,48631,50081,56024,56025,50173,52407,56026,56027,49342,50903,56028,56029,51124,56030,56031,47560,55360,55361,55362,55363,55364,55365,55366,55367,55368,48109,55369,55370,55371,55372,46777,62712,55373,62713,55374,55375,52707,55376,55377,55378,55379,55380,55381,55382,55383,62905,55384,55385,55386,55387,60384,55388,55389,55390,55391,55392,55393,53235,48063,55394,55395,55396,55397,55398,55399,55400,47808,54437,55401,55402,55403,55404,55405,55406,55407,57817,55408,55409,55410,55411,62964,45482,45810,55412,55413,55414,55415,55416,55417,55418,62965,55419,55420,62967,55421,55422,55424,47825,62966,55425,50098,55426,55427,55428,55429,55430,55431,55432,62969,55433,55434,55435,62968,55436,55437,55438,55439,55440,55441,55442,55443,55444,55445,55446,55447,55448,55449,55450,55451,55452,55453,55454,55455,55456,55616,55617,55618,55619,55620,55621,55622,55623,55624,55625,55626,55627,55628,55629,55630,55631,55632,55633,55634,55635,55636,55637,55638,55639,55640,55641,55642,55643,55644,55645,55646,55647,55648,55649,55650,55651,55652,55653,55654,55655,55656,55657,55658,55659,55660,55661,55662,55663,55664,55665,55666,55667,55668,55669,55670,55671,55672,55673,55674,55675,55676,55677,55678,55680,55681,55682,55683,55684,55685,55686,55687,55688,55689,55690,55691,55692,55693,55694,55695,55696,55697,55698,55699,55700,55701,55702,55703,55704,55705,55706,55707,55708,55709,55710,55711,55712,55872,55873,55874,55875,55876,55877,55878,55879,55880,55881,55882,55883,55884,55885,55886,45492,54762,47290,55887,47537,45766,54512,53197,45276,54731,48117,54986,47031,52400,50870,45537,47546,55036,47585,47009,48378,60122,60123,52473,47603,60124,46331,50099,47057,47832,60125,54516,60126,48342,48095,60127,49630,49848,54495,55242,60128,60129,60132,60130,60131,51678,47283,46788,60133,51946,51661,46285,55888,55889,58073,50658,60134,49333,55890,55224,60135,55212,51452,55507,55501,54494,55891,54521,51652,54190,47315,46048,55892,51682,62710,55893,55894,55895,47829,55896,62711,55897,55898,55263,55899,55900,62705,47280,54740,47311,50928,55901,55902,55903,55904,55905,55906,55907,55908,55909,46019,55910,55911,62706,45996,55912,55913,55914,55915,54461,51191,55916,55917,55918,55919,55920,62708,55921,55922,62707,55923,55924,55925,55926,55927,55928,55929,55930,55931,55932,52427,55933,55934,55936,51364,55937,55938,55939,55940,55941,55942,55943,55944,55945,55946,55947,55948,55949,62709,55950,55267,50623,62912,55951,55952,62907,55953,62915,55954,62914,55955,54970,62913,55956,55957,55958,54462,62916,55959,62924,55960,55961,55962,55963,45263,46584,55964,62921,62922,55965,50652,55966,55967,55968,56128,62917,62918,56129,56130,62919,62923,56131,48864,62920,47354,56132,56133,56134,62928,62931,56135,56136,56137,49127,56138,47602,62908,62925,56139,56140,49847,56141,56142,56143,52472,56144,48377,56145,62926,62927,62929,46821,62930,56146,62933,56147,56148,56149,56150,56151,56152,56153,62909,56154,56155,56156,62932,54203,56157,46060,56158,56159,52388,56160,56161,56162,56163,62934,56164,56165,56166,56167,56168,56169,56170,56171,62935,48865,62936,56172,56173,52447,62939,56174,56175,56176,56177,56178,45768,55257,56179,62937,56180,62938,62940,56181,62946,56182,56183,56184,62944,56185,56186,56187,62943,62941,56188,56189,62945,56190,56192,62942,62948,62949,56193,52451,56194,56195,58815,46520,62947,62952,52387,56196,56197,56198,56199,56200,62950,62951,56201,56202,56203,56204,56205,56206,62910,56207,56208,56209,56210,56211,56212,56213,56214,56215,56216,56217,56218,45508,56219,56220,62911,56221,56222,46533,45796,56223,62956,62953,56224,46807,56384,62957,56385,62954,56386,56387,56388,56389,56390,62955,56391,56392,46298,56393,54506,56394,56395,56396,62958,56397,46073,56398,56399,56400,56401,56402,56403,56404,62959,62961,56405,56406,56407,62960,56408,56409,56410,56411,56412,56413,56414,62962,56415,62963,56416,56417,56418,56419,56420,56421,56422,56423,56424,56425,56426,56427,51693,47530,56428,56429,51195,56430,56431,46819,56432,56433,56434,56435,56436,56437,56438,52425,56439,56440,56441,56442,56443,56444,56445,56446,56448,56449,56450,56451,56452,56453,56454,56455,56456,56457,56458,60070,56459,56460,56461,56462,56463,56464,56465,56466,56467,56468,56469,56470,56471,56472,56473,56474,56475,56476,56477,56478,56479,56480,56640,56641,56642,56643,56644,56645,56646,56647,56648,56649,56650,56651,56652,56653,56654,56655,56656,56657,56658,56659,56660,56661,56662,56663,56664,56665,56666,56667,56668,56669,56670,56671,56672,56673,56674,56675,56676,56677,56678,56679,56680,56681,56682,56683,56684,56685,56686,56687,56688,56689,56690,56691,56692,56693,56694,56695,56696,56697,56698,56699,56700,56701,56702,56704,56705,56706,56707,56708,56709,56710,56711,56712,56713,56714,56715,56716,56717,56718,56719,56720,56721,56722,56723,56724,56725,56726,56727,56728,56729,56730,56731,56732,56733,56734,56735,56736,56896,56897,56898,56899,56900,56901,56902,56903,56904,56905,56906,56907,56908,56909,56910,56911,56912,56913,56914,56915,56916,56917,56918,56919,56920,56921,56922,56923,56924,56925,56926,56927,56928,46005,54526,47596,53497,56929,59885,55210,59886,49878,51437,47844,59887,59888,59889,55009,59890,59891,59893,59892,59894,59895,51169,59896,54488,59897,48590,56930,59898,59899,48591,59900,47272,49598,59901,45490,48084,47605,59902,56931,60065,60066,60067,47096,48301,56932,51940,57550,54447,53181,54711,60068,54750,60069,53441,47548,56933,46279,45529,56934,56935,56936,49329,56937,56938,56939,56940,45542,45543,56941,45544,56942,56943,56944,56945,46013,51432,56946,56947,56948,56949,58817,56950,56951,45535,56952,56953,56954,49609,46319,56955,56956,51112,54232,56957,50937,53688,56958,47613,49909,56960,56961,56962,56963,56964,54189,56965,54475,48636,56966,58818,47029,58819,56967,56968,48057,54754,56969,48632,54454,52901,49580,46041,56970,56971,52470,56972,58822,58820,58824,56973,58826,58823,46543,50888,56974,46588,58821,56975,51958,56976,56977,58825,56978,56979,56980,50132,45509,48291,56981,56982,56983,55223,56984,56985,52683,52173,51914,52435,58828,58827,50406,56986,56987,53665,53687,58829,56988,58832,56989,52664,55024,58831,46557,56990,52670,56991,58833,46778,56992,57152,52648,47588,57153,51909,46033,52185,54508,58834,47082,57154,57155,57156,58830,57157,57158,57159,57160,57161,57162,58837,46334,58838,57163,57164,57165,57166,57167,58835,58836,57168,53981,57169,57170,49887,45510,57171,54242,57172,57173,46813,52204,57174,58839,57175,57176,54262,57177,57178,57179,57180,57181,45545,57182,46836,58842,58840,58841,46528,57183,57184,57185,53957,58844,57186,57187,58846,57188,57189,57190,57191,57192,57193,58845,51122,57194,53923,57195,57196,58843,57197,57198,57199,57200,54498,54746,57201,57202,57203,57204,57205,58848,55281,57206,57207,57208,57209,57210,57211,57212,58849,57213,45532,53755,57214,58850,58852,57216,57217,57218,57219,58851,57220,57221,58853,57222,57223,57224,57225,57226,53976,57227,46539,57228,59359,57229,56053,57230,56056,57231,56054,57232,56055,57233,57234,57235,56058,53455,50375,57236,57237,45294,57238,57239,57240,53424,57241,56057,57242,54218,47786,56226,51185,57243,56060,56059,51675,56061,57244,56225,55262,56062,49626,57245,57246,56229,57247,57248,54260,57408,57409,56231,56228,57410,56232,57411,57412,48572,57413,57414,57415,49353,56227,56230,54947,57416,56233,57417,57418,57419,56237,57420,57421,57422,56238,56236,47810,57423,57424,57425,49060,56235,57426,57427,57428,56234,54471,45759,57429,57430,56239,57431,47609,57432,56240,57433,57434,57435,57436,46011,57437,57438,57439,46502,57440,57441,57442,57443,46780,56241,57444,57445,57446,46837,57447,56242,57448,57449,57450,57451,57452,57453,57454,57455,57456,57457,57458,57459,57460,57461,57462,57463,57464,57465,57466,57467,45513,57468,57469,57470,57472,56244,57473,57474,57475,56243,56245,57476,57477,57478,57479,57480,57481,57482,57483,57484,57485,57486,56247,57487,56246,57488,57489,57490,57491,57492,57493,57494,56248,57495,57496,57497,57498,57499,57500,57501,57502,57503,56249,57504,57664,56250,57665,57666,54223,62714,51189,55235,50660,62716,62717,62715,57667,48838,57668,57669,57670,57671,53487,57672,57673,47059,57674,57675,54477,52394,57676,57677,62882,62881,47784,62718,52182,57678,57679,57680,62884,49362,57681,46058,57682,52650,62885,62883,48564,62888,57683,62889,48589,50104,49121,52193,62890,57684,57685,57686,62886,62887,50416,57687,57688,57689,57690,57691,62892,57692,46268,57693,55277,57694,46295,62891,62894,57695,57696,62893,62895,53457,57697,57698,57699,57700,57701,57702,57703,50129,51369,57704,57705,57706,57707,57708,57709,62896,62897,57710,57711,57712,57713,57714,57715,62898,57716,57717,62899,62900,62901,57718,57719,57720,57721,62903,62902,57722,57723,57724,57725,62904,57726,57728,57729,57730,57731,57732,57733,57734,57735,57736,57737,57738,45769,57739,54228,51917,57740,49391,55000,53936,49599,57741,48624,57742,57743,57744,57745,57746,57747,57748,57749,57750,57751,47274,57752,57753,57754,57755,57756,57757,57758,57759,57760,57920,57921,57922,57923,57924,57925,57926,57927,57928,57929,57930,57931,57932,57933,57934,57935,57936,57937,57938,57939,57940,57941,57942,57943,57944,57945,57946,57947,57948,57949,57950,57951,57952,57953,57954,57955,57956,57957,57958,57959,57960,57961,57962,57963,57964,57965,57966,57967,57968,57969,57970,57971,57972,57973,57974,57975,57976,57977,57978,57979,57980,57981,57982,57984,57985,57986,57987,57988,57989,57990,57991,57992,57993,57994,57995,57996,57997,57998,57999,58000,58001,58002,58003,58004,58005,58006,58007,58008,58009,58010,58011,58012,58013,58014,58015,58016,58176,58177,58178,58179,58180,58181,58182,58183,58184,58185,58186,58187,58188,58189,58190,58191,58192,58193,58194,58195,58196,58197,58198,58199,58200,58201,58202,58203,58204,58205,58206,58207,58208,58209,58210,58211,58212,58213,58214,58215,58216,58217,58218,58219,58220,58221,48376,58222,58223,58224,58225,58226,58227,58228,58229,58230,58231,58232,58233,58234,58235,58236,58237,58238,58240,58241,58242,58243,58244,58245,58246,58247,63174,58248,58249,58250,58251,58252,58253,58254,58255,58256,58257,58258,58259,58260,58261,58262,58263,58264,58265,58266,58267,58268,58269,58270,58271,58272,58432,58433,58434,58435,58436,58437,63175,58438,58439,58440,58441,58442,58443,58444,58445,58446,58447,58448,58449,58450,58451,58452,58453,58454,58455,58456,58457,58458,58459,58460,58461,58462,63176,58463,58464,58465,58466,58467,58468,58469,58470,58471,58472,58473,58474,58475,58476,58477,58478,58479,58480,58481,58482,58483,58484,58485,58486,58487,58488,58489,58490,58491,58492,58493,58494,58496,58497,58498,58499,58500,58501,58502,58503,58504,58505,58506,58507,58508,58509,58510,58511,58512,58513,58514,58515,58516,58517,58518,58519,58520,58521,58522,58523,58524,58525,58526,58527,58528,58688,58689,58690,58691,58692,58693,58694,58695,58696,58697,58698,58699,58700,58701,58702,58703,58704,58705,58706,58707,58708,58709,58710,58711,58712,58713,58714,58715,58716,58717,58718,58719,58720,58721,58722,58723,58724,58725,58726,58727,58728,58729,58730,58731,58732,58733,58734,58735,58736,58737,58738,58739,63177,58740,58741,58742,58743,58744,58745,58746,58747,58748,58749,58750,58752,58753,58754,58755,58756,58757,58758,58759,58760,58761,58762,58763,58764,58765,58766,58767,58768,58769,58770,58771,58772,58773,58774,58775,58776,58777,58778,58779,58780,58781,58782,58783,63178,58784,58944,58945,58946,58947,58948,58949,58950,58951,58952,58953,58954,58955,58956,58957,58958,58959,58960,58961,58962,58963,58964,58965,58966,58967,58968,58969,58970,58971,58972,58973,58974,58975,58976,58977,58978,63180,58979,58980,58981,58982,58983,58984,58985,58986,58987,58988,58989,58990,58991,58992,58993,58994,58995,58996,58997,58998,58999,59000,59001,59002,59003,59004,59005,59006,59008,59009,59010,59011,59012,59013,59014,59015,59016,59017,59018,59019,59020,59021,59022,59023,59024,59025,59026,59027,59028,59029,59030,59031,59032,59033,59034,59035,59036,59037,63179,59038,59039,59040,59200,59201,59202,59203,59204,59205,59206,59207,63465,59208,59209,59210,59211,59212,59213,59214,59215,59216,59217,59218,59219,59220,59221,59222,59223,59224,59225,59226,59227,59228,59229,59230,59231,59232,59233,59234,59235,59236,59237,59238,59239,59240,59241,59242,59243,59244,59245,59246,59247,59248,59249,59250,59251,59252,59253,59254,59255,59256,59257,59258,59259,59260,59261,59262,59264,59265,59266,59267,59268,59269,59270,59271,59272,59273,59274,59275,59276,59277,59278,59279,59280,59281,59282,59283,59284,59285,59286,59287,59288,59289,59290,59291,59292,59293,59294,59295,59296,59456,59457,59458,59459,59460,59461,59462,59463,59464,59465,59466,59467,59468,59469,59470,63181,59471,59472,59473,59474,59475,59476,59477,59478,59479,59480,59481,59482,59483,59484,59485,59486,59487,59488,59489,59490,59491,59492,59493,59494,59495,59496,59497,59498,59499,59500,59501,59502,59503,59504,59505,59506,59507,59508,59509,59510,59511,59512,59513,59514,63182,59515,59516,59517,59518,59520,59521,59522,59523,59524,59525,59526,59527,59528,59529,59530,59531,59532,59533,59534,59535,59536,59537,59538,59539,59540,61124,61125,61126,54763,46756,61128,61127,61129,61130,51109,61131,61132,59541,47024,46582,61133,61135,59542,61134,59543,47302,61136,61137,61138,46811,45998,54995,50374,45493,47318,61139,61140,54463,51157,48891,52953,47539,61142,61141,61144,61143,50597,61145,61146,51118,61147,51119,61148,45735,61149,61150,61151,61152,61153,55274,61154,61155,48344,61156,54219,52474,45740,49637,61157,51110,50093,59544,61158,61159,61160,61161,61162,61163,61164,59545,61165,61166,61167,59546,59547,61168,61169,61170,61172,61171,59548,61173,52653,49857,61174,61175,61176,54689,61177,53171,61178,61179,59549,61180,61181,61345,61182,61346,47349,50170,61347,61348,48578,53951,45817,61349,61350,61351,54008,61352,55037,61353,50892,59550,61354,61355,49588,61356,53242,52216,61358,61357,46074,47608,61359,61360,53474,61361,61362,47078,53439,61363,61364,61365,51441,52448,61366,61367,61368,61369,61370,54752,61371,46317,50090,61372,59551,61373,61374,61375,59552,52989,61376,49888,46264,55222,48629,59712,53191,61379,61377,61378,61380,46759,48380,48866,50124,61381,61382,59713,61383,61391,61384,61385,61386,51138,61425,46797,61387,59714,61388,61389,46790,50110,61390,59715,61392,61393,61394,54770,59716,61395,50423,59717,61396,50424,61397,61398,47332,45303,61399,61400,61401,59718,61402,61403,61404,61405,59719,61406,48821,61409,61407,61408,59720,61410,61411,49613,61412,61413,61414,61415,61416,61417,61418,61419,61420,49368,59721,61421,49581,61422,61423,61424,59722,59723,53218,59724,59725,59726,59727,59728,59729,59730,59731,45988,59732,59733,59734,59735,59736,59737,59738,59739,59740,59741,59742,59743,59744,59745,59746,59747,59748,59749,59750,59751,59752,59753,59754,59755,59756,59757,59758,59759,59760,59761,59762,59763,59764,59765,59766,59767,59768,59769,59770,59771,59772,59773,59774,59776,59777,59778,59779,59780,59781,59782,59783,59784,59785,59786,59787,59788,59789,59790,59791,59792,59793,59794,59795,59796,59797,59798,59799,59800,59801,59802,59803,59804,59805,59806,59807,59808,59968,59969,59970,59971,59972,59973,59974,59975,59976,59977,59978,59979,59980,59981,59982,59983,59984,59985,59986,59987,59988,59989,59990,59991,59992,59993,59994,59995,50117,58309,51649,58310,59996,45525,52938,46259,51442,58311,53200,58312,48356,58313,58314,50118,54690,50390,47595,52933,58315,50166,58316,59997,47015,47347,47826,58317,58318,54468,58319,59998,58320,53707,58321,58322,58323,58324,53718,58325,45819,49339,58326,59999,49323,58327,58328,58329,60000,58330,58331,60001,47287,56034,60002,46803,60003,56036,56035,60004,60005,60006,60007,60008,60009,60010,56038,60011,60012,60013,51438,60014,60015,56037,47040,53748,54005,54771,48599,60016,60017,60018,60019,55272,56040,56039,60020,45218,52691,60021,56041,60022,47293,48330,49853,49828,46018,56042,60023,49834,50352,48565,60024,60025,53214,60026,60027,60028,56043,51650,60029,60030,60032,60033,60034,45533,60035,60036,60037,56044,60038,46776,54458,60039,46077,60040,60041,56045,54473,53205,50659,60042,56046,60043,60044,60045,60046,60047,56047,60048,56048,49642,52437,53213,60049,60050,60051,60052,60053,60054,60055,60056,60057,60058,60059,60060,60061,54247,49825,60062,56049,60063,60064,52197,60224,56050,60225,52198,54014,60226,60227,60228,47348,60229,60230,56051,45231,53174,60231,60232,54735,60233,60234,60235,60236,60237,60238,60239,60240,60241,60242,52205,60243,60244,60245,60246,60247,60248,60249,60250,56052,60251,60252,58308,60253,60254,49573,60255,60256,63167,60257,60258,63168,63169,50385,60259,51384,53731,60260,60261,53467,53701,48303,47565,60262,61428,60263,60264,46278,54202,63170,46075,60265,60266,63171,60267,60268,46577,60269,60270,60271,60272,60273,60274,60275,60276,60277,60278,63173,60279,60280,60281,60282,60283,60284,60285,54250,63143,53673,60286,60288,60289,60290,63145,60291,60292,60293,63144,60294,60295,49635,49367,60296,45474,60297,60298,60299,60300,52973,60301,53480,63147,60302,60303,53238,60304,63146,54768,63148,50105,60305,60306,60307,48116,63150,63149,60308,60309,60310,50398,60311,60312,49624,60313,60314,60315,60316,60317,52138,60318,53180,60319,60320,60480,60481,60482,60483,60484,60485,60486,60487,60488,63151,60489,60490,63152,60491,60492,63153,60493,49846,60494,60495,60496,60497,60498,45268,50681,60499,60500,60501,60502,63154,60503,60504,60505,60506,60507,60508,60509,60510,60511,60512,60513,60514,60515,60516,60517,60518,60519,60520,60521,51168,63142,60522,60523,48824,60524,60525,48818,60526,46565,60527,60528,47047,60529,49087,50130,50150,60530,60531,55500,60532,60533,60534,47343,60535,60536,60537,60538,60539,60540,60541,60542,60544,48633,53669,60545,45264,60546,60547,60548,60549,60550,63408,60551,60552,60553,60554,60555,60556,60557,60558,63409,60559,60560,60561,60562,60563,53420,60564,45232,60565,60566,60567,63410,63411,60568,63412,60569,60570,60571,51146,60572,60573,60574,60575,60576,60736,60737,48847,60738,60739,63415,60740,60741,60742,60743,60744,60745,60746,63414,60747,45534,60748,63413,60749,60750,63416,60751,63417,60752,60753,60754,60755,60756,60757,60758,60759,60760,60761,60762,60763,60764,60765,60766,60767,60768,60769,60770,60771,60772,60773,60774,60775,60776,60777,60778,60779,60780,60781,60782,60783,60784,60785,60786,60787,60788,60789,60790,60791,60792,60793,60794,60795,60796,60797,60798,60800,60801,52900,51405,60802,47787,59576,59577,59578,48834,60803,60804,60805,60806,60807,54004,60808,54479,51672,60809,60810,60811,60812,60813,60814,60815,60816,60817,60818,60819,60820,60821,60822,60823,60824,60825,60826,60827,60828,60829,60830,60831,60832,60992,60993,60994,60995,60996,60997,60998,60999,61000,61001,61002,61003,61004,61005,61006,61007,61008,61009,61010,61011,61012,61013,61014,61015,61016,61017,61018,61019,61020,61021,61022,61023,61024,61025,61026,61027,61028,61029,61030,61031,61032,61033,61034,61035,61036,61037,61038,61039,61040,61041,61042,61043,61044,61045,61046,61047,61048,61049,61050,61051,61052,61053,61054,61056,61057,61058,61059,61060,61061,61062,61063,61064,61065,61066,61067,61068,61069,61070,61071,61072,61073,61074,61075,61076,61077,61078,61079,61080,61081,61082,61083,61084,61085,61086,61087,61088,61248,61249,61250,61251,61252,61253,53939,46757,51178,61948,53230,52147,53483,59375,52711,47563,46809,61949,45284,52172,61950,54436,49837,49644,50884,48817,62113,48341,61254,62114,62115,61255,62116,53955,50869,61256,52679,62117,61257,54193,49093,52450,61258,62118,62119,53717,46830,62120,62121,46559,62122,62123,61259,45820,62124,62125,51367,61260,61261,61262,61263,61264,61265,61266,61267,61268,61269,61270,61271,61272,61273,61274,61275,61276,61277,61278,61279,61280,61281,61282,61283,61284,61285,61286,61287,61288,61289,61290,61291,61292,61293,61294,61295,61296,61297,47079,61298,61299,60585,60586,60587,61300,60588,61301,61302,50862,60589,60590,61303,61304,61305,47049,51891,61306,61307,61308,61309,61310,61312,61313,58040,63439,61314,61315,61316,61317,61318,61319,61320,61321,61322,61323,61324,61325,61326,61327,61328,61329,61330,61331,61332,61333,61334,61335,61336,61337,61338,61339,61340,61341,61342,61343,61344,61504,61505,61506,61507,61508,63440,61509,61510,45773,61511,61512,61513,61514,61515,61516,61517,61518,61519,61520,61521,61522,61523,61524,61525,61526,61527,61528,61529,61530,61531,61532,61533,61534,61535,61536,61537,61538,61539,63441,61540,61541,61542,61543,61544,61545,61546,61547,61548,61549,61550,61551,61552,61553,61554,61555,61556,61557,61558,61559,61560,61561,61562,61563,61564,61565,61566,61568,61569,61570,61571,61572,61573,61574,61575,61576,61577,63443,63442,61578,61579,61580,61581,61582,61583,61584,61585,61586,61587,61588,61589,61590,58043,61591,48290,61592,58044,58045,58046,58047,58048,58049,47033,54011,48548,51918,45477,52167,61593,58050,46844,51396,58051,61594,61595,48584,61596,45565,58052,61597,46838,58053,50393,61598,61599,58054,53210,47581,58055,49313,61600,58056,45814,61760,58057,61761,49651,58058,58059,49912,58060,58061,58062,51927,55480,55781,53219,61762,61763,61764,61765,61766,61767,61768,61769,61770,61771,61772,61605,61773,61774,56496,61775,61776,61777,61778,61779,61780,61781,61782,61783,61784,61785,61786,61787,61788,61789,61790,61791,61792,61793,61794,61795,61796,61797,61798,61799,61800,61801,61802,61803,61804,61805,61806,61807,61808,61809,61810,61811,61812,61813,61814,61815,61816,61817,61818,61819,61820,61821,61822,61824,61825,61826,61827,61828,61829,61830,61831,61832,61833,61834,61835,61836,61837,61838,61839,61840,61841,61842,61843,61844,61845,61846,61847,61848,61849,61850,61851,61852,61853,61854,61855,61856,62016,62017,62018,62019,62020,62021,62022,62023,62024,62025,62026,62027,62028,62029,62030,62031,62032,62033,62034,62035,62036,62037,62038,62039,62040,62041,62042,62043,62044,62045,62046,62047,62048,62049,62050,62051,62052,62053,62054,62055,62056,62057,62058,62059,62060,62061,62062,62063,62064,62065,62066,62067,62068,62069,62070,62071,62072,62073,62074,62075,62076,62077,62078,62080,62081,62082,62083,62084,62085,62086,62087,62088,62089,62090,62091,62092,62093,62094,62095,62096,62097,62098,62099,62100,62101,62102,62103,62104,62105,62106,62107,62108,62109,62110,62111,62112,62272,62273,62274,62275,62276,62277,62278,62279,62280,62281,62282,62283,62284,62285,62286,62287,62288,62289,49901,54438,52692,53681,46043,51197,62290,45749,49855,59104,51899,59105,59106,48852,59107,55204,52693,59109,48349,59108,59110,59111,49902,62291,48574,59112,49894,47783,59113,62292,59114,46034,53737,62293,62294,49061,59115,50927,59116,59117,62295,62296,59118,50861,59119,62297,51623,59120,59121,59122,58809,59123,59124,49890,59125,59126,55016,59127,62298,59128,47559,62299,62300,62301,62302,62303,62304,62305,63419,63418,62306,62307,62308,62309,63422,63420,47777,62310,63423,62311,63424,62312,62313,62314,63426,63425,63428,62315,62316,63427,62317,62318,62319,62320,62321,63429,63430,62322,62323,62324,62325,63431,62326,52200,62327,62328,62329,62330,47327,62331,62332,62333,62334,62336,62337,63444,62338,63445,62339,62340,62341,62342,63446,62343,62344,62345,62346,63448,62347,63450,62348,63447,62349,62350,62351,62352,62353,62354,62355,62356,62357,63451,62358,63449,62359,62360,62361,62362,62363,62364,62365,55255,62366,62367,62368,62528,63452,62529,62530,62531,62532,62533,62534,63453,62535,62536,62537,63454,62538,62539,62540,62541,62542,62543,62544,62545,62546,62547,62548,63455,62549,62550,62551,63456,62552,62553,62554,62555,62556,62557,62558,62559,62560,62561,62562,56267,62563,62564,55466,62565,62566,62567,62568,62569,62570,62571,62572,58871,47597,62573,62574,62575,62576,49149,48106,63433,50887,63432,62577,63434,63436,63435,62578,62579,62580,63437,62581,52922,62582,63438,62583,62584,50343,62585,62586,62587,62588,62589,62590,62592,62593,62594,62595,62596,62597,62598,62599,62600,62601,62602,62603,62604,62605,62606,62607,62608,62609,62610,62611,62612,62613,62614,62615,62616,62617,62618,62619,62620,62621,62622,62623,62624,62784,62785,62786,62787,62788,62789,62790,62791,62792,62793,62794,62795,62796,62797,62798,62799,62800,62801,62802,62803,62804,62805,62806,62807,62808,62809,62810,62811,62812,62813,62814,62815,62816,62817,62818,62819,62820,62821,62822,62823,62824,62825,62826,62827,62828,62829,62830,62831,62832,62833,62834,62835,62836,62837,62838,62839,62840,62841,62842,62843,62844,62845,62846,62848,62849,62850,62851,62852,62853,62854,62855,62856,62857,62858,62859,62860,62861,62862,62863,62864,62865,62866,62867,62868,62869,62870,62871,62872,62873,62874,62875,62876,62877,62878,62879,62880,63040,63041,63042,63043,63044,63045,63046,63047,63048,63049,63050,63051,63052,63053,63054,63055,63056,63057,63058,63059,63060,63061,63062,63063,63064,63065,63066,63067,63068,63069,63070,63071,63072,63073,63074,63075,63076,63077,63078,63079,63080,63081,63082,63083,63084,63085,63086,63087,63088,63089,63090,63091,63092,63093,63094,63095,63096,63097,63098,63099,63100,63101,63102,63104,63105,63106,63107,63108,63109,63110,63111,63112,63113,63114,63115,63116,63117,63118,63119,63120,63121,63122,63123,63124,63125,63126,63127,63128,63129,63130,63131,63132,63133,63134,63135,63136,63296,63297,63298,63299,63300,63301,63302,63303,63304,63305,63306,63307,63308,63309,63310,63311,63312,63313,63314,63315,63316,63317,63318,63319,63320,63321,63322,63323,63324,63325,63326,63327,63328,63329,63330,63331,63332,63333,63334,63335,63336,63337,63338,63339,63340,63341,63342,63343,63344,63345,63346,63347,63348,63349,63350,63351,63352,63353,63354,63355,63356,63357,63358,63360,54243,63361,63362,63183,63363,49843,63184,63364,63365,63185,63186,63187,63188,63366,63367,63190,63368,45483,63191,63369,63192,63193,63194,63370,63195,63196,63371,63372,63373,63374,63197,63198,53194,63375,63199,63200,63201,63202,63203,63204,49392,63205,63206,63207,63208,63209,63376,63210,63377,63211,63212,63378,63213,63214,63215,63216,63217,63218,63219,63220,48808,63379,63221,63222,63223,63224,63380,63381,63382,63383,63384,51450,63225,63226,63227,63228,63385,63386,63229,63230,63393,63394,63395,63396,63397,63387,63388,63398,63399,63400,45550,63401,63402,63403,63389,63390,63404,63405,49627,63406,63391,63392,63407,63552,63553,63554,63555,63556,63557,63558,63559,63560,63561,63562,63563,63564,63565,63566,63567,63568,63569,63570,63571,63572,63573,63574,63575,63576,63577,63578,63579,63580,63581,63582,63583,63584,63585,63586,63587,63588,63589,63590,63591,63592,63593,63594,63595,63596,63597,63598,63599,63600,63601,63602,63603,63604,63605,63606,63607,63608,63609,63610,63611,63612,63613,63614,63616,63617,63618,63619,63620,63621,63622,63623,63624,63625,63626,63627,63628,63629,63630,63631,63632,63633,63634,63635,63636,63637,63638,63639,63640,63641,63642,63643,63644,63645,63646,63647,63648,63808,63809,63810,63811,63812,63813,63814,63815,63816,63817,63818,63819,63820,63821,63822,63823,63824,63825,63826,63827,63828,63829,63830,63831,63832,63833,63834,63835,63836,63837,63838,63839,63840,63841,63842,63843,63844,63845,63846,63847,63848,63849,63850,63851,63852,63853,63854,63855,63856,63857,63858,63859,63860,63861,63862,63863,63864,63865,63866,63867,63868,63869,63870,63872,63873,63874,63875,63876,63877,63878,63879,63880,63881,63882,63883,63884,63885,63886,63887,63888,63889,63890,63891,63892,63893,63894,63895,63896,63897,63898,63899,63900,63901,63902,63903,63904,64064,64065,64066,64067,64068,64069,64070,64071,64072,64073,64074,64075,64076,64077,64078,64079,64080,64081,64082,64083,64084,64085,64086,64087,64088,64089,64090,64091,64092,64093,64094,64095,64096,64097,64098,64099,64100,64101,64102,64103,64104,64105,64106,64107,64108,64109,64110,64111,64112,64113,64114,64115,64116,64117,64118,64119,64120,64121,64122,64123,64124,64125,64126,64128,64129,64130,64131,64132,64133,64134,64135,64136,64137,64138,64139,64140,64141,64142,64143,64144,64145,64146,64147,64148,64149,64150,64151,64152,64153,64154,64155,64156,64157,64158,64159,64160,64320,64321,64322,64323,64324,64325,64326,64327,64328,64329,64330,64331,64332,64333,64334,64335,64336,64337,64338,64339,64340,64341,64342,64343,64344,64345,64346,64347,50417,61615,48294,61616,50169,64348,50616,53691,64349,61617,61618,61619,61620,61621,53692,64350,53740,64351,61623,61622,54439,64352,52690,61624,61626,61625,61627,61628,64353,64354,47339,61629,47848,64355,61630,61631,48873,61632,46828,61633,61634,61635,61636,51381,61637,61638,64356,61639,50676,64357,61640,64358,64359,64360,61641,64361,61642,63421,64362,61643,61644,61645,64363,61646,64364,64365,64366,64367,61647,47831,64368,61648,61649,61650,61651,61652,61653,61654,61656,64369,64370,54181,61655,64371,61657,64372,64373,64374,64375,64376,64377,64378,64379,64380,64381,62906,49849,64382,64384,63460,64385,64386,64387,64388,63461,63462,64389,64390,63463,64391,64392,64393,64394,64395,64396,63464,49844,64397,64398,64399,64400,64401,64402,64403,64404,64405,63466,64406,63467,64407,64408,64409,64410,64411,64412,49907,64413,64414,64415,64416,64576,64577,64578,64579,64580,64581,64582,64583,64584,62704,64585,64586,64587,62703,64588,64589,49897,64590,63457,63458,64591,64592,64593,64594,64595,48070,64596,64597,64598,64599,55780,64600,64601,64602,51954,49384,61604,64603,47834,64604,64605,51117,64606,64607,64608,50348,64609,64610,63468,63469,63470,64611,63472,63471,64612,63473,64613,64614,63476,64615,63475,64616,63474,63477,64617,64618,64619,64620,63478,64621,64622,64623,64624,64625,64626,64627,64628,64629,60905,64630,60906,60907,64631,63164,64632,64633,64634,64635,64636,64637,64638,64640,64641,64642,64643,64644,63165,64645,63166,46758,64646,55486,64647,64648,47556,64649,64650,64651,55483,64652,56497,64653,64654,64655,64656,64657,64658,51955,64659,63479,64660,64661,64662,64663,64664,64665,64666,64667,64668,63480,64669,64670,63481,64671,64672,64832,64833,64834,64835,64836,63483,64837,63482,64838,45511,64839,63484,63485,64840,64841,64842,64843,64844,63486,64845,64846,64847,64848,64849,64850,64851,64852,64853,64854,64855,50923,60596,64856,64857,64858,64859,64860,64861,64862,64863,64864,64865,64866,64867,64868,64869,64870,64871,64872,64873,64874,64875,64876,64877,64878,64879,64880,64881,64882,64883,64884,64885,64886,64887,64888,64889,64890,64891,64892,64893,64894,64896,64897,64898,64899,64900,64901,46045,63155,64902,64903,63156,49636,63157,63158,63159,63160,63161,63162,51363,63163,64904,64905,64906,64907,64908,64909,64910,64911,64912,64913,64914,64915,49658,47528,60904,64916,64917,64918,47594,55775,64919,64920,64921,64922,64923,43681,43682,43683,43684,43685,43686,43687,43688,43689,43690,43691,43692,43693,43694,43695,43696,43697,43698,43699,43700,43701,43702,43703,43704,43705,43706,43707,43708,43709,43710,43711,43712,43713,43714,43715,43716,43717,43718,43719,43720,43721,43722,43723,43724,43725,43726,43727,43728,43729,43730,43731,43732,43733,43734,43735,43736,43737,43738,43739,43740,43741,43742,43743,43744,43745,43746,43747,43748,43749,43750,43751,43752,43753,43754,43755,43756,43757,43758,43759,43760,43761,43762,43763,43764,43765,43766,43767,43768,43769,43770,43771,43772,43773,43774,43937,43938,43939,43940,43941,43942,43943,43944,43945,43946,43947,43948,43949,43950,43951,43952,43953,43954,43955,43956,43957,43958,43959,43960,43961,43962,43963,43964,43965,43966,43967,43968,43969,43970,43971,43972,43973,43974,43975,43976,43977,43978,43979,43980,43981,43982,43983,43984,43985,43986,43987,43988,43989,43990,43991,43992,43993,43994,43995,43996,43997,43998,43999,44000,44001,44002,44003,44004,44005,44006,44007,44008,44009,44010,44011,44012,44013,44014,44015,44016,44017,44018,44019,44020,44021,44022,44023,44024,44025,44026,44027,44028,44029,44030,44193,44194,44195,44196,44197,44198,44199,44200,44201,44202,44203,44204,44205,44206,44207,44208,44209,44210,44211,44212,44213,44214,44215,44216,44217,44218,44219,44220,44221,44222,44223,44224,44225,44226,44227,44228,44229,44230,44231,44232,44233,44234,44235,44236,44237,44238,44239,44240,44241,44242,44243,44244,44245,44246,44247,44248,44249,44250,44251,44252,44253,44254,44255,44256,44257,44258,44259,44260,44261,44262,44263,44264,44265,44266,44267,44268,44269,44270,44271,44272,44273,44274,44275,44276,44277,44278,44279,44280,44281,44282,44283,44284,44285,44286,44449,44450,44451,44452,44453,44454,44455,44456,44457,44458,44459,44460,44461,44462,44463,44464,44465,44466,44467,44468,44469,44470,44471,44472,44473,44474,44475,44476,44477,44478,44479,44480,44481,44482,44483,44484,44485,44486,44487,44488,44489,44490,44491,44492,44493,44494,44495,44496,44497,44498,44499,44500,44501,44502,44503,44504,44505,44506,44507,44508,44509,44510,44511,44512,44513,44514,44515,44516,44517,44518,44519,44520,44521,44522,44523,44524,44525,44526,44527,44528,44529,44530,44531,44532,44533,44534,44535,44536,44537,44538,44539,44540,44541,44542,44705,44706,44707,44708,44709,44710,44711,44712,44713,44714,44715,44716,44717,44718,44719,44720,44721,44722,44723,44724,44725,44726,44727,44728,44729,44730,44731,44732,44733,44734,44735,44736,44737,44738,44739,44740,44741,44742,44743,44744,44745,44746,44747,44748,44749,44750,44751,44752,44753,44754,44755,44756,44757,44758,44759,44760,44761,44762,44763,44764,44765,44766,44767,44768,44769,44770,44771,44772,44773,44774,44775,44776,44777,44778,44779,44780,44781,44782,44783,44784,44785,44786,44787,44788,44789,44790,44791,44792,44793,44794,44795,44796,44797,44798,44961,44962,44963,44964,44965,44966,44967,44968,44969,44970,44971,44972,44973,44974,44975,44976,44977,44978,44979,44980,44981,44982,44983,44984,44985,44986,44987,44988,44989,44990,44991,44992,44993,44994,44995,44996,44997,44998,44999,45000,45001,45002,45003,45004,45005,45006,45007,45008,45009,45010,45011,45012,45013,45014,45015,45016,45017,45018,45019,45020,45021,45022,45023,45024,45025,45026,45027,45028,45029,45030,45031,45032,45033,45034,45035,45036,45037,45038,45039,45040,45041,45042,45043,45044,45045,45046,45047,45048,45049,45050,45051,45052,45053,45054,63649,63650,63651,63652,63653,63654,63655,63656,63657,63658,63659,63660,63661,63662,63663,63664,63665,63666,63667,63668,63669,63670,63671,63672,63673,63674,63675,63676,63677,63678,63679,63680,63681,63682,63683,63684,63685,63686,63687,63688,63689,63690,63691,63692,63693,63694,63695,63696,63697,63698,63699,63700,63701,63702,63703,63704,63705,63706,63707,63708,63709,63710,63711,63712,63713,63714,63715,63716,63717,63718,63719,63720,63721,63722,63723,63724,63725,63726,63727,63728,63729,63730,63731,63732,63733,63734,63735,63736,63737,63738,63739,63740,63741,63742,63905,63906,63907,63908,63909,63910,63911,63912,63913,63914,63915,63916,63917,63918,63919,63920,63921,63922,63923,63924,63925,63926,63927,63928,63929,63930,63931,63932,63933,63934,63935,63936,63937,63938,63939,63940,63941,63942,63943,63944,63945,63946,63947,63948,63949,63950,63951,63952,63953,63954,63955,63956,63957,63958,63959,63960,63961,63962,63963,63964,63965,63966,63967,63968,63969,63970,63971,63972,63973,63974,63975,63976,63977,63978,63979,63980,63981,63982,63983,63984,63985,63986,63987,63988,63989,63990,63991,63992,63993,63994,63995,63996,63997,63998,64161,64162,64163,64164,64165,64166,64167,64168,64169,64170,64171,64172,64173,64174,64175,64176,64177,64178,64179,64180,64181,64182,64183,64184,64185,64186,64187,64188,64189,64190,64191,64192,64193,64194,64195,64196,64197,64198,64199,64200,64201,64202,64203,64204,64205,64206,64207,64208,64209,64210,64211,64212,64213,64214,64215,64216,64217,64218,64219,64220,64221,64222,64223,64224,64225,64226,64227,64228,64229,64230,64231,64232,64233,64234,64235,64236,64237,64238,64239,64240,64241,64242,64243,64244,64245,64246,64247,64248,64249,64250,64251,64252,64253,64254,64417,64418,64419,64420,64421,64422,64423,64424,64425,64426,64427,64428,64429,64430,64431,64432,64433,64434,64435,64436,64437,64438,64439,64440,64441,64442,64443,64444,64445,64446,64447,64448,64449,64450,64451,64452,64453,64454,64455,64456,64457,64458,64459,64460,64461,64462,64463,64464,64465,64466,64467,64468,64469,64470,64471,64472,64473,64474,64475,64476,64477,64478,64479,64480,64481,64482,64483,64484,64485,64486,64487,64488,64489,64490,64491,64492,64493,64494,64495,64496,64497,64498,64499,64500,64501,64502,64503,64504,64505,64506,64507,64508,64509,64510,64673,64674,64675,64676,64677,64678,64679,64680,64681,64682,64683,64684,64685,64686,64687,64688,64689,64690,64691,64692,64693,64694,64695,64696,64697,64698,64699,64700,64701,64702,64703,64704,64705,64706,64707,64708,64709,64710,64711,64712,64713,64714,64715,64716,64717,64718,64719,64720,64721,64722,64723,64724,64725,64726,64727,64728,64729,64730,64731,64732,64733,64734,64735,64736,64737,64738,64739,64740,64741,64742,64743,64744,64745,64746,64747,64748,64749,64750,64751,64752,64753,64754,64755,64756,64757,64758,64759,64760,64761,64762,64763,64764,64765,64766,64929,64930,64931,64932,64933,64934,64935,64936,64937,64938,64939,64940,64941,64942,64943,64944,64945,64946,64947,64948,64949,64950,64951,64952,64953,64954,64955,64956,64957,64958,64959,64960,64961,64962,64963,64964,64965,64966,64967,64968,64969,64970,64971,64972,64973,64974,64975,64976,64977,64978,64979,64980,64981,64982,64983,64984,64985,64986,64987,64988,64989,64990,64991,64992,64993,64994,64995,64996,64997,64998,64999,65000,65001,65002,65003,65004,65005,65006,65007,65008,65009,65010,65011,65012,65013,65014,65015,65016,65017,65018,65019,65020,65021,65022,65185,65186,65187,65188,65189,65190,65191,65192,65193,65194,65195,65196,65197,65198,65199,65200,65201,65202,65203,65204,65205,65206,65207,65208,65209,65210,65211,65212,65213,65214,65215,65216,65217,65218,65219,65220,65221,65222,65223,65224,65225,65226,65227,65228,65229,65230,65231,65232,65233,65234,65235,65236,65237,65238,65239,65240,65241,65242,65243,65244,65245,65246,65247,65248,65249,65250,65251,65252,65253,65254,65255,65256,65257,65258,65259,65260,65261,65262,65263,65264,65265,65266,65267,65268,65269,65270,65271,65272,65273,65274,65275,65276,65277,65278,41280,41281,41282,41283,41284,41285,41286,41287,41288,41289,41290,41291,41292,41293,41294,41295,41296,41297,41298,41299,41300,41301,41302,41303,41304,41305,41306,41307,41308,41309,41310,41311,41312,41313,41314,41315,41316,41317,41318,41319,41320,41321,41322,41323,41324,41325,41326,41327,41328,41329,41330,41331,41332,41333,41334,41335,41336,41337,41338,41339,41340,41341,41342,41344,41345,41346,41347,41348,41349,41350,41351,41352,41353,41354,41355,41356,41357,41358,41359,41360,41361,41362,41363,41364,41365,41366,41367,41368,41369,41370,41371,41372,41373,41374,41375,41376,41536,41537,41538,41539,41540,41541,41542,41543,41544,41545,41546,41547,41548,41549,41550,41551,41552,41553,41554,41555,41556,41557,41558,41559,41560,41561,41562,41563,41564,41565,41566,41567,41568,41569,41570,41571,41572,41573,41574,41575,41576,41577,41578,41579,41580,41581,41582,41583,41584,41585,41586,41587,41588,41589,41590,41591,41592,41593,41594,41595,41596,41597,41598,41600,41601,41602,41603,41604,41605,41606,41607,41608,41609,41610,41611,41612,41613,41614,41615,41616,41617,41618,41619,41620,41621,41622,41623,41624,41625,41626,41627,41628,41629,41630,41631,41632,41792,41793,41794,41795,41796,41797,41798,41799,41800,41801,41802,41803,41804,41805,41806,41807,41808,41809,41810,41811,41812,41813,41814,41815,41816,41817,41818,41819,41820,41821,41822,41823,41824,41825,41826,41827,41828,41829,41830,41831,41832,41833,41834,41835,41836,41837,41838,41839,41840,41841,41842,41843,41844,41845,41846,41847,41848,41849,41850,41851,41852,41853,41854,41856,41857,41858,41859,41860,41861,41862,41863,41864,41865,41866,41867,41868,41869,41870,41871,41872,41873,41874,41875,41876,41877,41878,41879,41880,41881,41882,41883,41884,41885,41886,41887,41888,42048,42049,42050,42051,42052,42053,42054,42055,42056,42057,42058,42059,42060,42061,42062,42063,42064,42065,42066,42067,42068,42069,42070,42071,42072,42073,42074,42075,42076,42077,42078,42079,42080,42081,42082,42083,42084,42085,42086,42087,42088,42089,42090,42091,42092,42093,42094,42095,42096,42097,42098,42099,42100,42101,42102,42103,42104,42105,42106,42107,42108,42109,42110,42112,42113,42114,42115,42116,42117,42118,42119,42120,42121,42122,42123,42124,42125,42126,42127,42128,42129,42130,42131,42132,42133,42134,42135,42136,42137,42138,42139,42140,42141,42142,42143,42144,42304,42305,42306,42307,42308,42309,42310,42311,42312,42313,42314,42315,42316,42317,42318,42319,42320,42321,42322,42323,42324,42325,42326,42327,42328,42329,42330,42331,42332,42333,42334,42335,42336,42337,42338,42339,42340,42341,42342,42343,42344,42345,42346,42347,42348,42349,42350,42351,42352,42353,42354,42355,42356,42357,42358,42359,42360,42361,42362,42363,42364,42365,42366,42368,42369,42370,42371,42372,42373,42374,42375,42376,42377,42378,42379,42380,42381,42382,42383,42384,42385,42386,42387,42388,42389,42390,42391,42392,42393,42394,42395,42396,42397,42398,42399,42400,42560,42561,42562,42563,42564,42565,42566,42567,42568,42569,42570,42571,42572,42573,42574,42575,42576,42577,42578,42579,42580,42581,42582,42583,42584,42585,42586,42587,42588,42589,42590,42591,42592,42593,42594,42595,42596,42597,42598,42599,42600,42601,42602,42603,42604,42605,42606,42607,42608,42609,42610,42611,42612,42613,42614,42615,42616,42617,42618,42619,42620,42621,42622,42624,42625,42626,42627,42628,42629,42630,42631,42632,42633,42634,42635,42636,42637,42638,42639,42640,42641,42642,42643,42644,42645,42646,42647,42648,42649,42650,42651,42652,42653,42654,42655,42656,42816,42817,42818,42819,42820,42821,42822,42823,42824,42825,42826,42827,42828,42829,42830,42831,42832,42833,42834,42835,42836,42837,42838,42839,42840,42841,42842,42843,42844,42845,42846,42847,42848,42849,42850,42851,42852,42853,42854,42855,42856,42857,42858,42859,42860,42861,42862,42863,42864,42865,42866,42867,42868,42869,42870,42871,42872,42873,42874,42875,42876,42877,42878,42880,42881,42882,42883,42884,42885,42886,42887,42888,42889,42890,42891,42892,42893,42894,42895,42896,42897,42898,42899,42900,42901,42902,42903,42904,42905,42906,42907,42908,42909,42910,42911,42912,41643,41644,41645,41646,41647,41648,41699,41700,41711,41712,41725,41726,42228,42229,42230,42231,42232,42233,42234,42235,42236,42237,42238,42487,42488,42489,42490,42491,42492,42493,42494,42681,42682,42683,42684,42685,42686,42687,42688,42713,42714,42715,42716,42717,42718,42719,42732,42733,42739,42742,42743,42744,42745,42746,42747,42748,42749,42750,42946,42947,42948,42949,42950,42951,42952,42953,42954,42955,42956,42957,42958,42959,42960,42994,42995,42996,42997,42998,42999,43000,43001,43002,43003,43004,43005,43006,43158,43159,43160,43161,43162,43163,43164,43165,43166,43167,43168,43196,43199,43201,43202,43203,43204,43242,43243,43244,43245,43246,43247,43248,43249,43250,43251,43252,43253,43254,43255,43256,43257,43258,43259,43260,43261,43262,43352,43355,43357,43358,43359,43401,43402,43403,43404,43405,43406,43407,43408,43409,43410,43411,43412,43413,43415,43416,43417,43418,43419,43420,43421,43422,43423,43424,43425,43426,43427,43504,43505,43506,43507,43508,43509,43510,43511,43512,43513,43514,43515,43516,43517,43518,55290,55291,55292,55293,55294,65104,65105,65106,65107,65108,65109,65110,65111,65112,65113,65114,65115,65116,65117,65118,65119,65120,65121,65122,65123,65124,65125,65126,65127,65128,65129,65130,65131,65132,65133,65134,65135,65136,65137,65138,65139,65140,65141,65142,65143,65144,65145,65146,65147,65148,65149,65150,65152,65153,65154,65155,65156,65157,65158,65159,65160,65161,65162,65163,65164,65165,65166,65167,65168,65169,65170,65171,65172,65173,65174,65175,65176,65177,65178,65179,65180,65181,65182,65183,65184,64924,64925,64926,64927,64928,65088,65089,65090,65091,65092,65093,65094,65095,65096,65097,65098,65099,65100,65101,65102,65103,43349,42738,42740,42741,42720,42721,42736,42737,42722,42723,42734,42735,42726,42727,42724,42725,42728,42729,42730,42731,43368,43369,43370,43371,43372,43373,43374,43375,43376,43377,43378,43379,43380,43381,43382,43383,43384,43385,43386,43387,43388,43389,43390,43392,43393,43394,43395,43396,43397,43398,43399,43400,41889,41890,41891,41447,41893,41894,41895,41896,41897,41898,41899,41900,41901,41902,41903,41904,41905,41906,41907,41908,41909,41910,41911,41912,41913,41914,41915,41916,41917,41918,41919,41920,41921,41922,41923,41924,41925,41926,41927,41928,41929,41930,41931,41932,41933,41934,41935,41936,41937,41938,41939,41940,41941,41942,41943,41944,41945,41946,41947,41948,41949,41950,41951,41952,41953,41954,41955,41956,41957,41958,41959,41960,41961,41962,41963,41964,41965,41966,41967,41968,41969,41970,41971,41972,41973,41974,41975,41976,41977,41978,41979,41980,41981,41387,41449,41450,43350,41982,43351,41892 };

unsigned short gbk1_to_unicode_keys[] = { 33088,33089,33090,33091,33092,33093,33094,33095,33096,33097,33098,33099,33100,33101,33102,33103,33104,33105,33106,33107,33108,33109,33110,33111,33112,33113,33114,33115,33116,33117,33118,33119,33120,33121,33122,33123,33124,33125,33126,33127,33128,33129,33130,33131,33132,33133,33134,33135,33136,33137,33138,33139,33140,33141,33142,33143,33144,33145,33146,33147,33148,33149,33150,33152,33153,33154,33155,33156,33157,33158,33159,33160,33161,33162,33163,33164,33165,33166,33167,33168,33169,33170,33171,33172,33173,33174,33175,33176,33177,33178,33179,33180,33181,33182,33183,33184,33185,33186,33187,33188,33189,33190,33191,33192,33193,33194,33195,33196,33197,33198,33199,33200,33201,33202,33203,33204,33205,33206,33207,33208,33209,33210,33211,33212,33213,33214,33215,33216,33217,33218,33219,33220,33221,33222,33223,33224,33225,33226,33227,33228,33229,33230,33231,33232,33233,33234,33235,33236,33237,33238,33239,33240,33241,33242,33243,33244,33245,33246,33247,33248,33249,33250,33251,33252,33253,33254,33255,33256,33257,33258,33259,33260,33261,33262,33263,33264,33265,33266,33267,33268,33269,33270,33271,33272,33273,33274,33275,33276,33277,33278,33344,33345,33346,33347,33348,33349,33350,33351,33352,33353,33354,33355,33356,33357,33358,33359,33360,33361,33362,33363,33364,33365,33366,33367,33368,33369,33370,33371,33372,33373,33374,33375,33376,33377,33378,33379,33380,33381,33382,33383,33384,33385,33386,33387,33388,33389,33390,33391,33392,33393,33394,33395,33396,33397,33398,33399,33400,33401,33402,33403,33404,33405,33406,33408,33409,33410,33411,33412,33413,33414,33415,33416,33417,33418,33419,33420,33421,33422,33423,33424,33425,33426,33427,33428,33429,33430,33431,33432,33433,33434,33435,33436,33437,33438,33439,33440,33441,33442,33443,33444,33445,33446,33447,33448,33449,33450,33451,33452,33453,33454,33455,33456,33457,33458,33459,33460,33461,33462,33463,33464,33465,33466,33467,33468,33469,33470,33471,33472,33473,33474,33475,33476,33477,33478,33479,33480,33481,33482,33483,33484,33485,33486,33487,33488,33489,33490,33491,33492,33493,33494,33495,33496,33497,33498,33499,33500,33501,33502,33503,33504,33505,33506,33507,33508,33509,33510,33511,33512,33513,33514,33515,33516,33517,33518,33519,33520,33521,33522,33523,33524,33525,33526,33527,33528,33529,33530,33531,33532,33533,33534,33600,33601,33602,33603,33604,33605,33606,33607,33608,33609,33610,33611,33612,33613,33614,33615,33616,33617,33618,33619,33620,33621,33622,33623,33624,33625,33626,33627,33628,33629,33630,33631,33632,33633,33634,33635,33636,33637,33638,33639,33640,33641,33642,33643,33644,33645,33646,33647,33648,33649,33650,33651,33652,33653,33654,33655,33656,33657,33658,33659,33660,33661,33662,33664,33665,33666,33667,33668,33669,33670,33671,33672,33673,33674,33675,33676,33677,33678,33679,33680,33681,33682,33683,33684,33685,33686,33687,33688,33689,33690,33691,33692,33693,33694,33695,33696,33697,33698,33699,33700,33701,33702,33703,33704,33705,33706,33707,33708,33709,33710,33711,33712,33713,33714,33715,33716,33717,33718,33719,33720,33721,33722,33723,33724,33725,33726,33727,33728,33729,33730,33731,33732,33733,33734,33735,33736,33737,33738,33739,33740,33741,33742,33743,33744,33745,33746,33747,33748,33749,33750,33751,33752,33753,33754,33755,33756,33757,33758,33759,33760,33761,33762,33763,33764,33765,33766,33767,33768,33769,33770,33771,33772,33773,33774,33775,33776,33777,33778,33779,33780,33781,33782,33783,33784,33785,33786,33787,33788,33789,33790,33856,33857,33858,33859,33860,33861,33862,33863,33864,33865,33866,33867,33868,33869,33870,33871,33872,33873,33874,33875,33876,33877,33878,33879,33880,33881,33882,33883,33884,33885,33886,33887,33888,33889,33890,33891,33892,33893,33894,33895,33896,33897,33898,33899,33900,33901,33902,33903,33904,33905,33906,33907,33908,33909,33910,33911,33912,33913,33914,33915,33916,33917,33918,33920,33921,33922,33923,33924,33925,33926,33927,33928,33929,33930,33931,33932,33933,33934,33935,33936,33937,33938,33939,33940,33941,33942,33943,33944,33945,33946,33947,33948,33949,33950,33951,33952,33953,33954,33955,33956,33957,33958,33959,33960,33961,33962,33963,33964,33965,33966,33967,33968,33969,33970,33971,33972,33973,33974,33975,33976,33977,33978,33979,33980,33981,33982,33983,33984,33985,33986,33987,33988,33989,33990,33991,33992,33993,33994,33995,33996,33997,33998,33999,34000,34001,34002,34003,34004,34005,34006,34007,34008,34009,34010,34011,34012,34013,34014,34015,34016,34017,34018,34019,34020,34021,34022,34023,34024,34025,34026,34027,34028,34029,34030,34031,34032,34033,34034,34035,34036,34037,34038,34039,34040,34041,34042,34043,34044,34045,34046,34112,34113,34114,34115,34116,34117,34118,34119,34120,34121,34122,34123,34124,34125,34126,34127,34128,34129,34130,34131,34132,34133,34134,34135,34136,34137,34138,34139,34140,34141,34142,34143,34144,34145,34146,34147,34148,34149,34150,34151,34152,34153,34154,34155,34156,34157,34158,34159,34160,34161,34162,34163,34164,34165,34166,34167,34168,34169,34170,34171,34172,34173,34174,34176,34177,34178,34179,34180,34181,34182,34183,34184,34185,34186,34187,34188,34189,34190,34191,34192,34193,34194,34195,34196,34197,34198,34199,34200,34201,34202,34203,34204,34205,34206,34207,34208,34209,34210,34211,34212,34213,34214,34215,34216,34217,34218,34219,34220,34221,34222,34223,34224,34225,34226,34227,34228,34229,34230,34231,34232,34233,34234,34235,34236,34237,34238,34239,34240,34241,34242,34243,34244,34245,34246,34247,34248,34249,34250,34251,34252,34253,34254,34255,34256,34257,34258,34259,34260,34261,34262,34263,34264,34265,34266,34267,34268,34269,34270,34271,34272,34273,34274,34275,34276,34277,34278,34279,34280,34281,34282,34283,34284,34285,34286,34287,34288,34289,34290,34291,34292,34293,34294,34295,34296,34297,34298,34299,34300,34301,34302,34368,34369,34370,34371,34372,34373,34374,34375,34376,34377,34378,34379,34380,34381,34382,34383,34384,34385,34386,34387,34388,34389,34390,34391,34392,34393,34394,34395,34396,34397,34398,34399,34400,34401,34402,34403,34404,34405,34406,34407,34408,34409,34410,34411,34412,34413,34414,34415,34416,34417,34418,34419,34420,34421,34422,34423,34424,34425,34426,34427,34428,34429,34430,34432,34433,34434,34435,34436,34437,34438,34439,34440,34441,34442,34443,34444,34445,34446,34447,34448,34449,34450,34451,34452,34453,34454,34455,34456,34457,34458,34459,34460,34461,34462,34463,34464,34465,34466,34467,34468,34469,34470,34471,34472,34473,34474,34475,34476,34477,34478,34479,34480,34481,34482,34483,34484,34485,34486,34487,34488,34489,34490,34491,34492,34493,34494,34495,34496,34497,34498,34499,34500,34501,34502,34503,34504,34505,34506,34507,34508,34509,34510,34511,34512,34513,34514,34515,34516,34517,34518,34519,34520,34521,34522,34523,34524,34525,34526,34527,34528,34529,34530,34531,34532,34533,34534,34535,34536,34537,34538,34539,34540,34541,34542,34543,34544,34545,34546,34547,34548,34549,34550,34551,34552,34553,34554,34555,34556,34557,34558,34624,34625,34626,34627,34628,34629,34630,34631,34632,34633,34634,34635,34636,34637,34638,34639,34640,34641,34642,34643,34644,34645,34646,34647,34648,34649,34650,34651,34652,34653,34654,34655,34656,34657,34658,34659,34660,34661,34662,34663,34664,34665,34666,34667,34668,34669,34670,34671,34672,34673,34674,34675,34676,34677,34678,34679,34680,34681,34682,34683,34684,34685,34686,34688,34689,34690,34691,34692,34693,34694,34695,34696,34697,34698,34699,34700,34701,34702,34703,34704,34705,34706,34707,34708,34709,34710,34711,34712,34713,34714,34715,34716,34717,34718,34719,34720,34721,34722,34723,34724,34725,34726,34727,34728,34729,34730,34731,34732,34733,34734,34735,34736,34737,34738,34739,34740,34741,34742,34743,34744,34745,34746,34747,34748,34749,34750,34751,34752,34753,34754,34755,34756,34757,34758,34759,34760,34761,34762,34763,34764,34765,34766,34767,34768,34769,34770,34771,34772,34773,34774,34775,34776,34777,34778,34779,34780,34781,34782,34783,34784,34785,34786,34787,34788,34789,34790,34791,34792,34793,34794,34795,34796,34797,34798,34799,34800,34801,34802,34803,34804,34805,34806,34807,34808,34809,34810,34811,34812,34813,34814,34880,34881,34882,34883,34884,34885,34886,34887,34888,34889,34890,34891,34892,34893,34894,34895,34896,34897,34898,34899,34900,34901,34902,34903,34904,34905,34906,34907,34908,34909,34910,34911,34912,34913,34914,34915,34916,34917,34918,34919,34920,34921,34922,34923,34924,34925,34926,34927,34928,34929,34930,34931,34932,34933,34934,34935,34936,34937,34938,34939,34940,34941,34942,34944,34945,34946,34947,34948,34949,34950,34951,34952,34953,34954,34955,34956,34957,34958,34959,34960,34961,34962,34963,34964,34965,34966,34967,34968,34969,34970,34971,34972,34973,34974,34975,34976,34977,34978,34979,34980,34981,34982,34983,34984,34985,34986,34987,34988,34989,34990,34991,34992,34993,34994,34995,34996,34997,34998,34999,35000,35001,35002,35003,35004,35005,35006,35007,35008,35009,35010,35011,35012,35013,35014,35015,35016,35017,35018,35019,35020,35021,35022,35023,35024,35025,35026,35027,35028,35029,35030,35031,35032,35033,35034,35035,35036,35037,35038,35039,35040,35041,35042,35043,35044,35045,35046,35047,35048,35049,35050,35051,35052,35053,35054,35055,35056,35057,35058,35059,35060,35061,35062,35063,35064,35065,35066,35067,35068,35069,35070,35136,35137,35138,35139,35140,35141,35142,35143,35144,35145,35146,35147,35148,35149,35150,35151,35152,35153,35154,35155,35156,35157,35158,35159,35160,35161,35162,35163,35164,35165,35166,35167,35168,35169,35170,35171,35172,35173,35174,35175,35176,35177,35178,35179,35180,35181,35182,35183,35184,35185,35186,35187,35188,35189,35190,35191,35192,35193,35194,35195,35196,35197,35198,35200,35201,35202,35203,35204,35205,35206,35207,35208,35209,35210,35211,35212,35213,35214,35215,35216,35217,35218,35219,35220,35221,35222,35223,35224,35225,35226,35227,35228,35229,35230,35231,35232,35233,35234,35235,35236,35237,35238,35239,35240,35241,35242,35243,35244,35245,35246,35247,35248,35249,35250,35251,35252,35253,35254,35255,35256,35257,35258,35259,35260,35261,35262,35263,35264,35265,35266,35267,35268,35269,35270,35271,35272,35273,35274,35275,35276,35277,35278,35279,35280,35281,35282,35283,35284,35285,35286,35287,35288,35289,35290,35291,35292,35293,35294,35295,35296,35297,35298,35299,35300,35301,35302,35303,35304,35305,35306,35307,35308,35309,35310,35311,35312,35313,35314,35315,35316,35317,35318,35319,35320,35321,35322,35323,35324,35325,35326,35392,35393,35394,35395,35396,35397,35398,35399,35400,35401,35402,35403,35404,35405,35406,35407,35408,35409,35410,35411,35412,35413,35414,35415,35416,35417,35418,35419,35420,35421,35422,35423,35424,35425,35426,35427,35428,35429,35430,35431,35432,35433,35434,35435,35436,35437,35438,35439,35440,35441,35442,35443,35444,35445,35446,35447,35448,35449,35450,35451,35452,35453,35454,35456,35457,35458,35459,35460,35461,35462,35463,35464,35465,35466,35467,35468,35469,35470,35471,35472,35473,35474,35475,35476,35477,35478,35479,35480,35481,35482,35483,35484,35485,35486,35487,35488,35489,35490,35491,35492,35493,35494,35495,35496,35497,35498,35499,35500,35501,35502,35503,35504,35505,35506,35507,35508,35509,35510,35511,35512,35513,35514,35515,35516,35517,35518,35519,35520,35521,35522,35523,35524,35525,35526,35527,35528,35529,35530,35531,35532,35533,35534,35535,35536,35537,35538,35539,35540,35541,35542,35543,35544,35545,35546,35547,35548,35549,35550,35551,35552,35553,35554,35555,35556,35557,35558,35559,35560,35561,35562,35563,35564,35565,35566,35567,35568,35569,35570,35571,35572,35573,35574,35575,35576,35577,35578,35579,35580,35581,35582,35648,35649,35650,35651,35652,35653,35654,35655,35656,35657,35658,35659,35660,35661,35662,35663,35664,35665,35666,35667,35668,35669,35670,35671,35672,35673,35674,35675,35676,35677,35678,35679,35680,35681,35682,35683,35684,35685,35686,35687,35688,35689,35690,35691,35692,35693,35694,35695,35696,35697,35698,35699,35700,35701,35702,35703,35704,35705,35706,35707,35708,35709,35710,35712,35713,35714,35715,35716,35717,35718,35719,35720,35721,35722,35723,35724,35725,35726,35727,35728,35729,35730,35731,35732,35733,35734,35735,35736,35737,35738,35739,35740,35741,35742,35743,35744,35745,35746,35747,35748,35749,35750,35751,35752,35753,35754,35755,35756,35757,35758,35759,35760,35761,35762,35763,35764,35765,35766,35767,35768,35769,35770,35771,35772,35773,35774,35775,35776,35777,35778,35779,35780,35781,35782,35783,35784,35785,35786,35787,35788,35789,35790,35791,35792,35793,35794,35795,35796,35797,35798,35799,35800,35801,35802,35803,35804,35805,35806,35807,35808,35809,35810,35811,35812,35813,35814,35815,35816,35817,35818,35819,35820,35821,35822,35823,35824,35825,35826,35827,35828,35829,35830,35831,35832,35833,35834,35835,35836,35837,35838,35904,35905,35906,35907,35908,35909,35910,35911,35912,35913,35914,35915,35916,35917,35918,35919,35920,35921,35922,35923,35924,35925,35926,35927,35928,35929,35930,35931,35932,35933,35934,35935,35936,35937,35938,35939,35940,35941,35942,35943,35944,35945,35946,35947,35948,35949,35950,35951,35952,35953,35954,35955,35956,35957,35958,35959,35960,35961,35962,35963,35964,35965,35966,35968,35969,35970,35971,35972,35973,35974,35975,35976,35977,35978,35979,35980,35981,35982,35983,35984,35985,35986,35987,35988,35989,35990,35991,35992,35993,35994,35995,35996,35997,35998,35999,36000,36001,36002,36003,36004,36005,36006,36007,36008,36009,36010,36011,36012,36013,36014,36015,36016,36017,36018,36019,36020,36021,36022,36023,36024,36025,36026,36027,36028,36029,36030,36031,36032,36033,36034,36035,36036,36037,36038,36039,36040,36041,36042,36043,36044,36045,36046,36047,36048,36049,36050,36051,36052,36053,36054,36055,36056,36057,36058,36059,36060,36061,36062,36063,36064,36065,36066,36067,36068,36069,36070,36071,36072,36073,36074,36075,36076,36077,36078,36079,36080,36081,36082,36083,36084,36085,36086,36087,36088,36089,36090,36091,36092,36093,36094,36160,36161,36162,36163,36164,36165,36166,36167,36168,36169,36170,36171,36172,36173,36174,36175,36176,36177,36178,36179,36180,36181,36182,36183,36184,36185,36186,36187,36188,36189,36190,36191,36192,36193,36194,36195,36196,36197,36198,36199,36200,36201,36202,36203,36204,36205,36206,36207,36208,36209,36210,36211,36212,36213,36214,36215,36216,36217,36218,36219,36220,36221,36222,36224,36225,36226,36227,36228,36229,36230,36231,36232,36233,36234,36235,36236,36237,36238,36239,36240,36241,36242,36243,36244,36245,36246,36247,36248,36249,36250,36251,36252,36253,36254,36255,36256,36257,36258,36259,36260,36261,36262,36263,36264,36265,36266,36267,36268,36269,36270,36271,36272,36273,36274,36275,36276,36277,36278,36279,36280,36281,36282,36283,36284,36285,36286,36287,36288,36289,36290,36291,36292,36293,36294,36295,36296,36297,36298,36299,36300,36301,36302,36303,36304,36305,36306,36307,36308,36309,36310,36311,36312,36313,36314,36315,36316,36317,36318,36319,36320,36321,36322,36323,36324,36325,36326,36327,36328,36329,36330,36331,36332,36333,36334,36335,36336,36337,36338,36339,36340,36341,36342,36343,36344,36345,36346,36347,36348,36349,36350,36416,36417,36418,36419,36420,36421,36422,36423,36424,36425,36426,36427,36428,36429,36430,36431,36432,36433,36434,36435,36436,36437,36438,36439,36440,36441,36442,36443,36444,36445,36446,36447,36448,36449,36450,36451,36452,36453,36454,36455,36456,36457,36458,36459,36460,36461,36462,36463,36464,36465,36466,36467,36468,36469,36470,36471,36472,36473,36474,36475,36476,36477,36478,36480,36481,36482,36483,36484,36485,36486,36487,36488,36489,36490,36491,36492,36493,36494,36495,36496,36497,36498,36499,36500,36501,36502,36503,36504,36505,36506,36507,36508,36509,36510,36511,36512,36513,36514,36515,36516,36517,36518,36519,36520,36521,36522,36523,36524,36525,36526,36527,36528,36529,36530,36531,36532,36533,36534,36535,36536,36537,36538,36539,36540,36541,36542,36543,36544,36545,36546,36547,36548,36549,36550,36551,36552,36553,36554,36555,36556,36557,36558,36559,36560,36561,36562,36563,36564,36565,36566,36567,36568,36569,36570,36571,36572,36573,36574,36575,36576,36577,36578,36579,36580,36581,36582,36583,36584,36585,36586,36587,36588,36589,36590,36591,36592,36593,36594,36595,36596,36597,36598,36599,36600,36601,36602,36603,36604,36605,36606,36672,36673,36674,36675,36676,36677,36678,36679,36680,36681,36682,36683,36684,36685,36686,36687,36688,36689,36690,36691,36692,36693,36694,36695,36696,36697,36698,36699,36700,36701,36702,36703,36704,36705,36706,36707,36708,36709,36710,36711,36712,36713,36714,36715,36716,36717,36718,36719,36720,36721,36722,36723,36724,36725,36726,36727,36728,36729,36730,36731,36732,36733,36734,36736,36737,36738,36739,36740,36741,36742,36743,36744,36745,36746,36747,36748,36749,36750,36751,36752,36753,36754,36755,36756,36757,36758,36759,36760,36761,36762,36763,36764,36765,36766,36767,36768,36769,36770,36771,36772,36773,36774,36775,36776,36777,36778,36779,36780,36781,36782,36783,36784,36785,36786,36787,36788,36789,36790,36791,36792,36793,36794,36795,36796,36797,36798,36799,36800,36801,36802,36803,36804,36805,36806,36807,36808,36809,36810,36811,36812,36813,36814,36815,36816,36817,36818,36819,36820,36821,36822,36823,36824,36825,36826,36827,36828,36829,36830,36831,36832,36833,36834,36835,36836,36837,36838,36839,36840,36841,36842,36843,36844,36845,36846,36847,36848,36849,36850,36851,36852,36853,36854,36855,36856,36857,36858,36859,36860,36861,36862,36928,36929,36930,36931,36932,36933,36934,36935,36936,36937,36938,36939,36940,36941,36942,36943,36944,36945,36946,36947,36948,36949,36950,36951,36952,36953,36954,36955,36956,36957,36958,36959,36960,36961,36962,36963,36964,36965,36966,36967,36968,36969,36970,36971,36972,36973,36974,36975,36976,36977,36978,36979,36980,36981,36982,36983,36984,36985,36986,36987,36988,36989,36990,36992,36993,36994,36995,36996,36997,36998,36999,37000,37001,37002,37003,37004,37005,37006,37007,37008,37009,37010,37011,37012,37013,37014,37015,37016,37017,37018,37019,37020,37021,37022,37023,37024,37025,37026,37027,37028,37029,37030,37031,37032,37033,37034,37035,37036,37037,37038,37039,37040,37041,37042,37043,37044,37045,37046,37047,37048,37049,37050,37051,37052,37053,37054,37055,37056,37057,37058,37059,37060,37061,37062,37063,37064,37065,37066,37067,37068,37069,37070,37071,37072,37073,37074,37075,37076,37077,37078,37079,37080,37081,37082,37083,37084,37085,37086,37087,37088,37089,37090,37091,37092,37093,37094,37095,37096,37097,37098,37099,37100,37101,37102,37103,37104,37105,37106,37107,37108,37109,37110,37111,37112,37113,37114,37115,37116,37117,37118,37184,37185,37186,37187,37188,37189,37190,37191,37192,37193,37194,37195,37196,37197,37198,37199,37200,37201,37202,37203,37204,37205,37206,37207,37208,37209,37210,37211,37212,37213,37214,37215,37216,37217,37218,37219,37220,37221,37222,37223,37224,37225,37226,37227,37228,37229,37230,37231,37232,37233,37234,37235,37236,37237,37238,37239,37240,37241,37242,37243,37244,37245,37246,37248,37249,37250,37251,37252,37253,37254,37255,37256,37257,37258,37259,37260,37261,37262,37263,37264,37265,37266,37267,37268,37269,37270,37271,37272,37273,37274,37275,37276,37277,37278,37279,37280,37281,37282,37283,37284,37285,37286,37287,37288,37289,37290,37291,37292,37293,37294,37295,37296,37297,37298,37299,37300,37301,37302,37303,37304,37305,37306,37307,37308,37309,37310,37311,37312,37313,37314,37315,37316,37317,37318,37319,37320,37321,37322,37323,37324,37325,37326,37327,37328,37329,37330,37331,37332,37333,37334,37335,37336,37337,37338,37339,37340,37341,37342,37343,37344,37345,37346,37347,37348,37349,37350,37351,37352,37353,37354,37355,37356,37357,37358,37359,37360,37361,37362,37363,37364,37365,37366,37367,37368,37369,37370,37371,37372,37373,37374,37440,37441,37442,37443,37444,37445,37446,37447,37448,37449,37450,37451,37452,37453,37454,37455,37456,37457,37458,37459,37460,37461,37462,37463,37464,37465,37466,37467,37468,37469,37470,37471,37472,37473,37474,37475,37476,37477,37478,37479,37480,37481,37482,37483,37484,37485,37486,37487,37488,37489,37490,37491,37492,37493,37494,37495,37496,37497,37498,37499,37500,37501,37502,37504,37505,37506,37507,37508,37509,37510,37511,37512,37513,37514,37515,37516,37517,37518,37519,37520,37521,37522,37523,37524,37525,37526,37527,37528,37529,37530,37531,37532,37533,37534,37535,37536,37537,37538,37539,37540,37541,37542,37543,37544,37545,37546,37547,37548,37549,37550,37551,37552,37553,37554,37555,37556,37557,37558,37559,37560,37561,37562,37563,37564,37565,37566,37567,37568,37569,37570,37571,37572,37573,37574,37575,37576,37577,37578,37579,37580,37581,37582,37583,37584,37585,37586,37587,37588,37589,37590,37591,37592,37593,37594,37595,37596,37597,37598,37599,37600,37601,37602,37603,37604,37605,37606,37607,37608,37609,37610,37611,37612,37613,37614,37615,37616,37617,37618,37619,37620,37621,37622,37623,37624,37625,37626,37627,37628,37629,37630,37696,37697,37698,37699,37700,37701,37702,37703,37704,37705,37706,37707,37708,37709,37710,37711,37712,37713,37714,37715,37716,37717,37718,37719,37720,37721,37722,37723,37724,37725,37726,37727,37728,37729,37730,37731,37732,37733,37734,37735,37736,37737,37738,37739,37740,37741,37742,37743,37744,37745,37746,37747,37748,37749,37750,37751,37752,37753,37754,37755,37756,37757,37758,37760,37761,37762,37763,37764,37765,37766,37767,37768,37769,37770,37771,37772,37773,37774,37775,37776,37777,37778,37779,37780,37781,37782,37783,37784,37785,37786,37787,37788,37789,37790,37791,37792,37793,37794,37795,37796,37797,37798,37799,37800,37801,37802,37803,37804,37805,37806,37807,37808,37809,37810,37811,37812,37813,37814,37815,37816,37817,37818,37819,37820,37821,37822,37823,37824,37825,37826,37827,37828,37829,37830,37831,37832,37833,37834,37835,37836,37837,37838,37839,37840,37841,37842,37843,37844,37845,37846,37847,37848,37849,37850,37851,37852,37853,37854,37855,37856,37857,37858,37859,37860,37861,37862,37863,37864,37865,37866,37867,37868,37869,37870,37871,37872,37873,37874,37875,37876,37877,37878,37879,37880,37881,37882,37883,37884,37885,37886,37952,37953,37954,37955,37956,37957,37958,37959,37960,37961,37962,37963,37964,37965,37966,37967,37968,37969,37970,37971,37972,37973,37974,37975,37976,37977,37978,37979,37980,37981,37982,37983,37984,37985,37986,37987,37988,37989,37990,37991,37992,37993,37994,37995,37996,37997,37998,37999,38000,38001,38002,38003,38004,38005,38006,38007,38008,38009,38010,38011,38012,38013,38014,38016,38017,38018,38019,38020,38021,38022,38023,38024,38025,38026,38027,38028,38029,38030,38031,38032,38033,38034,38035,38036,38037,38038,38039,38040,38041,38042,38043,38044,38045,38046,38047,38048,38049,38050,38051,38052,38053,38054,38055,38056,38057,38058,38059,38060,38061,38062,38063,38064,38065,38066,38067,38068,38069,38070,38071,38072,38073,38074,38075,38076,38077,38078,38079,38080,38081,38082,38083,38084,38085,38086,38087,38088,38089,38090,38091,38092,38093,38094,38095,38096,38097,38098,38099,38100,38101,38102,38103,38104,38105,38106,38107,38108,38109,38110,38111,38112,38113,38114,38115,38116,38117,38118,38119,38120,38121,38122,38123,38124,38125,38126,38127,38128,38129,38130,38131,38132,38133,38134,38135,38136,38137,38138,38139,38140,38141,38142,38208,38209,38210,38211,38212,38213,38214,38215,38216,38217,38218,38219,38220,38221,38222,38223,38224,38225,38226,38227,38228,38229,38230,38231,38232,38233,38234,38235,38236,38237,38238,38239,38240,38241,38242,38243,38244,38245,38246,38247,38248,38249,38250,38251,38252,38253,38254,38255,38256,38257,38258,38259,38260,38261,38262,38263,38264,38265,38266,38267,38268,38269,38270,38272,38273,38274,38275,38276,38277,38278,38279,38280,38281,38282,38283,38284,38285,38286,38287,38288,38289,38290,38291,38292,38293,38294,38295,38296,38297,38298,38299,38300,38301,38302,38303,38304,38305,38306,38307,38308,38309,38310,38311,38312,38313,38314,38315,38316,38317,38318,38319,38320,38321,38322,38323,38324,38325,38326,38327,38328,38329,38330,38331,38332,38333,38334,38335,38336,38337,38338,38339,38340,38341,38342,38343,38344,38345,38346,38347,38348,38349,38350,38351,38352,38353,38354,38355,38356,38357,38358,38359,38360,38361,38362,38363,38364,38365,38366,38367,38368,38369,38370,38371,38372,38373,38374,38375,38376,38377,38378,38379,38380,38381,38382,38383,38384,38385,38386,38387,38388,38389,38390,38391,38392,38393,38394,38395,38396,38397,38398,38464,38465,38466,38467,38468,38469,38470,38471,38472,38473,38474,38475,38476,38477,38478,38479,38480,38481,38482,38483,38484,38485,38486,38487,38488,38489,38490,38491,38492,38493,38494,38495,38496,38497,38498,38499,38500,38501,38502,38503,38504,38505,38506,38507,38508,38509,38510,38511,38512,38513,38514,38515,38516,38517,38518,38519,38520,38521,38522,38523,38524,38525,38526,38528,38529,38530,38531,38532,38533,38534,38535,38536,38537,38538,38539,38540,38541,38542,38543,38544,38545,38546,38547,38548,38549,38550,38551,38552,38553,38554,38555,38556,38557,38558,38559,38560,38561,38562,38563,38564,38565,38566,38567,38568,38569,38570,38571,38572,38573,38574,38575,38576,38577,38578,38579,38580,38581,38582,38583,38584,38585,38586,38587,38588,38589,38590,38591,38592,38593,38594,38595,38596,38597,38598,38599,38600,38601,38602,38603,38604,38605,38606,38607,38608,38609,38610,38611,38612,38613,38614,38615,38616,38617,38618,38619,38620,38621,38622,38623,38624,38625,38626,38627,38628,38629,38630,38631,38632,38633,38634,38635,38636,38637,38638,38639,38640,38641,38642,38643,38644,38645,38646,38647,38648,38649,38650,38651,38652,38653,38654,38720,38721,38722,38723,38724,38725,38726,38727,38728,38729,38730,38731,38732,38733,38734,38735,38736,38737,38738,38739,38740,38741,38742,38743,38744,38745,38746,38747,38748,38749,38750,38751,38752,38753,38754,38755,38756,38757,38758,38759,38760,38761,38762,38763,38764,38765,38766,38767,38768,38769,38770,38771,38772,38773,38774,38775,38776,38777,38778,38779,38780,38781,38782,38784,38785,38786,38787,38788,38789,38790,38791,38792,38793,38794,38795,38796,38797,38798,38799,38800,38801,38802,38803,38804,38805,38806,38807,38808,38809,38810,38811,38812,38813,38814,38815,38816,38817,38818,38819,38820,38821,38822,38823,38824,38825,38826,38827,38828,38829,38830,38831,38832,38833,38834,38835,38836,38837,38838,38839,38840,38841,38842,38843,38844,38845,38846,38847,38848,38849,38850,38851,38852,38853,38854,38855,38856,38857,38858,38859,38860,38861,38862,38863,38864,38865,38866,38867,38868,38869,38870,38871,38872,38873,38874,38875,38876,38877,38878,38879,38880,38881,38882,38883,38884,38885,38886,38887,38888,38889,38890,38891,38892,38893,38894,38895,38896,38897,38898,38899,38900,38901,38902,38903,38904,38905,38906,38907,38908,38909,38910,38976,38977,38978,38979,38980,38981,38982,38983,38984,38985,38986,38987,38988,38989,38990,38991,38992,38993,38994,38995,38996,38997,38998,38999,39000,39001,39002,39003,39004,39005,39006,39007,39008,39009,39010,39011,39012,39013,39014,39015,39016,39017,39018,39019,39020,39021,39022,39023,39024,39025,39026,39027,39028,39029,39030,39031,39032,39033,39034,39035,39036,39037,39038,39040,39041,39042,39043,39044,39045,39046,39047,39048,39049,39050,39051,39052,39053,39054,39055,39056,39057,39058,39059,39060,39061,39062,39063,39064,39065,39066,39067,39068,39069,39070,39071,39072,39073,39074,39075,39076,39077,39078,39079,39080,39081,39082,39083,39084,39085,39086,39087,39088,39089,39090,39091,39092,39093,39094,39095,39096,39097,39098,39099,39100,39101,39102,39103,39104,39105,39106,39107,39108,39109,39110,39111,39112,39113,39114,39115,39116,39117,39118,39119,39120,39121,39122,39123,39124,39125,39126,39127,39128,39129,39130,39131,39132,39133,39134,39135,39136,39137,39138,39139,39140,39141,39142,39143,39144,39145,39146,39147,39148,39149,39150,39151,39152,39153,39154,39155,39156,39157,39158,39159,39160,39161,39162,39163,39164,39165,39166,39232,39233,39234,39235,39236,39237,39238,39239,39240,39241,39242,39243,39244,39245,39246,39247,39248,39249,39250,39251,39252,39253,39254,39255,39256,39257,39258,39259,39260,39261,39262,39263,39264,39265,39266,39267,39268,39269,39270,39271,39272,39273,39274,39275,39276,39277,39278,39279,39280,39281,39282,39283,39284,39285,39286,39287,39288,39289,39290,39291,39292,39293,39294,39296,39297,39298,39299,39300,39301,39302,39303,39304,39305,39306,39307,39308,39309,39310,39311,39312,39313,39314,39315,39316,39317,39318,39319,39320,39321,39322,39323,39324,39325,39326,39327,39328,39329,39330,39331,39332,39333,39334,39335,39336,39337,39338,39339,39340,39341,39342,39343,39344,39345,39346,39347,39348,39349,39350,39351,39352,39353,39354,39355,39356,39357,39358,39359,39360,39361,39362,39363,39364,39365,39366,39367,39368,39369,39370,39371,39372,39373,39374,39375,39376,39377,39378,39379,39380,39381,39382,39383,39384,39385,39386,39387,39388,39389,39390,39391,39392,39393,39394,39395,39396,39397,39398,39399,39400,39401,39402,39403,39404,39405,39406,39407,39408,39409,39410,39411,39412,39413,39414,39415,39416,39417,39418,39419,39420,39421,39422,39488,39489,39490,39491,39492,39493,39494,39495,39496,39497,39498,39499,39500,39501,39502,39503,39504,39505,39506,39507,39508,39509,39510,39511,39512,39513,39514,39515,39516,39517,39518,39519,39520,39521,39522,39523,39524,39525,39526,39527,39528,39529,39530,39531,39532,39533,39534,39535,39536,39537,39538,39539,39540,39541,39542,39543,39544,39545,39546,39547,39548,39549,39550,39552,39553,39554,39555,39556,39557,39558,39559,39560,39561,39562,39563,39564,39565,39566,39567,39568,39569,39570,39571,39572,39573,39574,39575,39576,39577,39578,39579,39580,39581,39582,39583,39584,39585,39586,39587,39588,39589,39590,39591,39592,39593,39594,39595,39596,39597,39598,39599,39600,39601,39602,39603,39604,39605,39606,39607,39608,39609,39610,39611,39612,39613,39614,39615,39616,39617,39618,39619,39620,39621,39622,39623,39624,39625,39626,39627,39628,39629,39630,39631,39632,39633,39634,39635,39636,39637,39638,39639,39640,39641,39642,39643,39644,39645,39646,39647,39648,39649,39650,39651,39652,39653,39654,39655,39656,39657,39658,39659,39660,39661,39662,39663,39664,39665,39666,39667,39668,39669,39670,39671,39672,39673,39674,39675,39676,39677,39678,39744,39745,39746,39747,39748,39749,39750,39751,39752,39753,39754,39755,39756,39757,39758,39759,39760,39761,39762,39763,39764,39765,39766,39767,39768,39769,39770,39771,39772,39773,39774,39775,39776,39777,39778,39779,39780,39781,39782,39783,39784,39785,39786,39787,39788,39789,39790,39791,39792,39793,39794,39795,39796,39797,39798,39799,39800,39801,39802,39803,39804,39805,39806,39808,39809,39810,39811,39812,39813,39814,39815,39816,39817,39818,39819,39820,39821,39822,39823,39824,39825,39826,39827,39828,39829,39830,39831,39832,39833,39834,39835,39836,39837,39838,39839,39840,39841,39842,39843,39844,39845,39846,39847,39848,39849,39850,39851,39852,39853,39854,39855,39856,39857,39858,39859,39860,39861,39862,39863,39864,39865,39866,39867,39868,39869,39870,39871,39872,39873,39874,39875,39876,39877,39878,39879,39880,39881,39882,39883,39884,39885,39886,39887,39888,39889,39890,39891,39892,39893,39894,39895,39896,39897,39898,39899,39900,39901,39902,39903,39904,39905,39906,39907,39908,39909,39910,39911,39912,39913,39914,39915,39916,39917,39918,39919,39920,39921,39922,39923,39924,39925,39926,39927,39928,39929,39930,39931,39932,39933,39934,40000,40001,40002,40003,40004,40005,40006,40007,40008,40009,40010,40011,40012,40013,40014,40015,40016,40017,40018,40019,40020,40021,40022,40023,40024,40025,40026,40027,40028,40029,40030,40031,40032,40033,40034,40035,40036,40037,40038,40039,40040,40041,40042,40043,40044,40045,40046,40047,40048,40049,40050,40051,40052,40053,40054,40055,40056,40057,40058,40059,40060,40061,40062,40064,40065,40066,40067,40068,40069,40070,40071,40072,40073,40074,40075,40076,40077,40078,40079,40080,40081,40082,40083,40084,40085,40086,40087,40088,40089,40090,40091,40092,40093,40094,40095,40096,40097,40098,40099,40100,40101,40102,40103,40104,40105,40106,40107,40108,40109,40110,40111,40112,40113,40114,40115,40116,40117,40118,40119,40120,40121,40122,40123,40124,40125,40126,40127,40128,40129,40130,40131,40132,40133,40134,40135,40136,40137,40138,40139,40140,40141,40142,40143,40144,40145,40146,40147,40148,40149,40150,40151,40152,40153,40154,40155,40156,40157,40158,40159,40160,40161,40162,40163,40164,40165,40166,40167,40168,40169,40170,40171,40172,40173,40174,40175,40176,40177,40178,40179,40180,40181,40182,40183,40184,40185,40186,40187,40188,40189,40190,40256,40257,40258,40259,40260,40261,40262,40263,40264,40265,40266,40267,40268,40269,40270,40271,40272,40273,40274,40275,40276,40277,40278,40279,40280,40281,40282,40283,40284,40285,40286,40287,40288,40289,40290,40291,40292,40293,40294,40295,40296,40297,40298,40299,40300,40301,40302,40303,40304,40305,40306,40307,40308,40309,40310,40311,40312,40313,40314,40315,40316,40317,40318,40320,40321,40322,40323,40324,40325,40326,40327,40328,40329,40330,40331,40332,40333,40334,40335,40336,40337,40338,40339,40340,40341,40342,40343,40344,40345,40346,40347,40348,40349,40350,40351,40352,40353,40354,40355,40356,40357,40358,40359,40360,40361,40362,40363,40364,40365,40366,40367,40368,40369,40370,40371,40372,40373,40374,40375,40376,40377,40378,40379,40380,40381,40382,40383,40384,40385,40386,40387,40388,40389,40390,40391,40392,40393,40394,40395,40396,40397,40398,40399,40400,40401,40402,40403,40404,40405,40406,40407,40408,40409,40410,40411,40412,40413,40414,40415,40416,40417,40418,40419,40420,40421,40422,40423,40424,40425,40426,40427,40428,40429,40430,40431,40432,40433,40434,40435,40436,40437,40438,40439,40440,40441,40442,40443,40444,40445,40446,40512,40513,40514,40515,40516,40517,40518,40519,40520,40521,40522,40523,40524,40525,40526,40527,40528,40529,40530,40531,40532,40533,40534,40535,40536,40537,40538,40539,40540,40541,40542,40543,40544,40545,40546,40547,40548,40549,40550,40551,40552,40553,40554,40555,40556,40557,40558,40559,40560,40561,40562,40563,40564,40565,40566,40567,40568,40569,40570,40571,40572,40573,40574,40576,40577,40578,40579,40580,40581,40582,40583,40584,40585,40586,40587,40588,40589,40590,40591,40592,40593,40594,40595,40596,40597,40598,40599,40600,40601,40602,40603,40604,40605,40606,40607,40608,40609,40610,40611,40612,40613,40614,40615,40616,40617,40618,40619,40620,40621,40622,40623,40624,40625,40626,40627,40628,40629,40630,40631,40632,40633,40634,40635,40636,40637,40638,40639,40640,40641,40642,40643,40644,40645,40646,40647,40648,40649,40650,40651,40652,40653,40654,40655,40656,40657,40658,40659,40660,40661,40662,40663,40664,40665,40666,40667,40668,40669,40670,40671,40672,40673,40674,40675,40676,40677,40678,40679,40680,40681,40682,40683,40684,40685,40686,40687,40688,40689,40690,40691,40692,40693,40694,40695,40696,40697,40698,40699,40700,40701,40702,40768,40769,40770,40771,40772,40773,40774,40775,40776,40777,40778,40779,40780,40781,40782,40783,40784,40785,40786,40787,40788,40789,40790,40791,40792,40793,40794,40795,40796,40797,40798,40799,40800,40801,40802,40803,40804,40805,40806,40807,40808,40809,40810,40811,40812,40813,40814,40815,40816,40817,40818,40819,40820,40821,40822,40823,40824,40825,40826,40827,40828,40829,40830,40832,40833,40834,40835,40836,40837,40838,40839,40840,40841,40842,40843,40844,40845,40846,40847,40848,40849,40850,40851,40852,40853,40854,40855,40856,40857,40858,40859,40860,40861,40862,40863,40864,40865,40866,40867,40868,40869,40870,40871,40872,40873,40874,40875,40876,40877,40878,40879,40880,40881,40882,40883,40884,40885,40886,40887,40888,40889,40890,40891,40892,40893,40894,40895,40896,40897,40898,40899,40900,40901,40902,40903,40904,40905,40906,40907,40908,40909,40910,40911,40912,40913,40914,40915,40916,40917,40918,40919,40920,40921,40922,40923,40924,40925,40926,40927,40928,40929,40930,40931,40932,40933,40934,40935,40936,40937,40938,40939,40940,40941,40942,40943,40944,40945,40946,40947,40948,40949,40950,40951,40952,40953,40954,40955,40956,40957,40958,41024,41025,41026,41027,41028,41029,41030,41031,41032,41033,41034,41035,41036,41037,41038,41039,41040,41041,41042,41043,41044,41045,41046,41047,41048,41049,41050,41051,41052,41053,41054,41055,41056,41057,41058,41059,41060,41061,41062,41063,41064,41065,41066,41067,41068,41069,41070,41071,41072,41073,41074,41075,41076,41077,41078,41079,41080,41081,41082,41083,41084,41085,41086,41088,41089,41090,41091,41092,41093,41094,41095,41096,41097,41098,41099,41100,41101,41102,41103,41104,41105,41106,41107,41108,41109,41110,41111,41112,41113,41114,41115,41116,41117,41118,41119,41120,41121,41122,41123,41124,41125,41126,41127,41128,41129,41130,41131,41132,41133,41134,41135,41136,41137,41138,41139,41140,41141,41142,41143,41144,41145,41146,41147,41148,41149,41150,41151,41152,41153,41154,41155,41156,41157,41158,41159,41160,41161,41162,41163,41164,41165,41166,41167,41168,41169,41170,41171,41172,41173,41174,41175,41176,41177,41178,41179,41180,41181,41182,41183,41184,41185,41186,41187,41188,41189,41190,41191,41192,41193,41194,41195,41196,41197,41198,41199,41200,41201,41202,41203,41204,41205,41206,41207,41208,41209,41210,41211,41212,41213,41214,41280,41281,41282,41283,41284,41285,41286,41287,41288,41289,41290,41291,41292,41293,41294,41295,41296,41297,41298,41299,41300,41301,41302,41303,41304,41305,41306,41307,41308,41309,41310,41311,41312,41313,41314,41315,41316,41317,41318,41319,41320,41321,41322,41323,41324,41325,41326,41327,41328,41329,41330,41331,41332,41333,41334,41335,41336,41337,41338,41339,41340,41341,41342,41344,41345,41346,41347,41348,41349,41350,41351,41352,41353,41354,41355,41356,41357,41358,41359,41360,41361,41362,41363,41364,41365,41366,41367,41368,41369,41370,41371,41372,41373,41374,41375,41376,41377,41378,41379,41380,41381,41382,41383,41384,41385,41386,41387,41388,41389,41390,41391,41392,41393,41394,41395,41396,41397,41398,41399,41400,41401,41402,41403,41404,41405,41406,41407,41408,41409,41410,41411,41412,41413,41414,41415,41416,41417,41418,41419,41420,41421,41422,41423,41424,41425,41426,41427,41428,41429,41430,41431,41432,41433,41434,41435,41436,41437,41438,41439,41440,41441,41442,41443,41444,41445,41446,41447,41448,41449,41450,41451,41452,41453,41454,41455,41456,41457,41458,41459,41460,41461,41462,41463,41464,41465,41466,41467,41468,41469,41470,41536,41537,41538,41539,41540,41541,41542,41543,41544,41545,41546,41547,41548,41549,41550,41551,41552,41553,41554,41555,41556,41557,41558,41559,41560,41561,41562,41563,41564,41565,41566,41567,41568,41569,41570,41571,41572,41573,41574,41575,41576,41577,41578,41579,41580,41581,41582,41583,41584,41585,41586,41587,41588,41589,41590,41591,41592,41593,41594,41595,41596,41597,41598,41600,41601,41602,41603,41604,41605,41606,41607,41608,41609,41610,41611,41612,41613,41614,41615,41616,41617,41618,41619,41620,41621,41622,41623,41624,41625,41626,41627,41628,41629,41630,41631,41632,41633,41634,41635,41636,41637,41638,41639,41640,41641,41642,41643,41644,41645,41646,41647,41648,41649,41650,41651,41652,41653,41654,41655,41656,41657,41658,41659,41660,41661,41662,41663,41664,41665,41666,41667,41668,41669,41670,41671,41672,41673,41674,41675,41676,41677,41678,41679,41680,41681,41682,41683,41684,41685,41686,41687,41688,41689,41690,41691,41692,41693,41694,41695,41696,41697,41698,41699,41700,41701,41702,41703,41704,41705,41706,41707,41708,41709,41710,41711,41712,41713,41714,41715,41716,41717,41718,41719,41720,41721,41722,41723,41724,41725,41726,41792,41793,41794,41795,41796,41797,41798,41799,41800,41801,41802,41803,41804,41805,41806,41807,41808,41809,41810,41811,41812,41813,41814,41815,41816,41817,41818,41819,41820,41821,41822,41823,41824,41825,41826,41827,41828,41829,41830,41831,41832,41833,41834,41835,41836,41837,41838,41839,41840,41841,41842,41843,41844,41845,41846,41847,41848,41849,41850,41851,41852,41853,41854,41856,41857,41858,41859,41860,41861,41862,41863,41864,41865,41866,41867,41868,41869,41870,41871,41872,41873,41874,41875,41876,41877,41878,41879,41880,41881,41882,41883,41884,41885,41886,41887,41888,41889,41890,41891,41892,41893,41894,41895,41896,41897,41898,41899,41900,41901,41902,41903,41904,41905,41906,41907,41908,41909,41910,41911,41912,41913,41914,41915,41916,41917,41918,41919,41920,41921,41922,41923,41924,41925,41926,41927,41928,41929,41930,41931,41932,41933,41934,41935,41936,41937,41938,41939,41940,41941,41942,41943,41944,41945,41946,41947,41948,41949,41950,41951,41952,41953,41954,41955,41956,41957,41958,41959,41960,41961,41962,41963,41964,41965,41966,41967,41968,41969,41970,41971,41972,41973,41974,41975,41976,41977,41978,41979,41980,41981,41982,42048,42049,42050,42051,42052,42053,42054,42055,42056,42057,42058,42059,42060,42061,42062,42063,42064,42065,42066,42067,42068,42069,42070,42071,42072,42073,42074,42075,42076,42077,42078,42079,42080,42081,42082,42083,42084,42085,42086,42087,42088,42089,42090,42091,42092,42093,42094,42095,42096,42097,42098,42099,42100,42101,42102,42103,42104,42105,42106,42107,42108,42109,42110,42112,42113,42114,42115,42116,42117,42118,42119,42120,42121,42122,42123,42124,42125,42126,42127,42128,42129,42130,42131,42132,42133,42134,42135,42136,42137,42138,42139,42140,42141,42142,42143,42144,42145,42146,42147,42148,42149,42150,42151,42152,42153,42154,42155,42156,42157,42158,42159,42160,42161,42162,42163,42164,42165,42166,42167,42168,42169,42170,42171,42172,42173,42174,42175,42176,42177,42178,42179,42180,42181,42182,42183,42184,42185,42186,42187,42188,42189,42190,42191,42192,42193,42194,42195,42196,42197,42198,42199,42200,42201,42202,42203,42204,42205,42206,42207,42208,42209,42210,42211,42212,42213,42214,42215,42216,42217,42218,42219,42220,42221,42222,42223,42224,42225,42226,42227,42228,42229,42230,42231,42232,42233,42234,42235,42236,42237,42238,42304,42305,42306,42307,42308,42309,42310,42311,42312,42313,42314,42315,42316,42317,42318,42319,42320,42321,42322,42323,42324,42325,42326,42327,42328,42329,42330,42331,42332,42333,42334,42335,42336,42337,42338,42339,42340,42341,42342,42343,42344,42345,42346,42347,42348,42349,42350,42351,42352,42353,42354,42355,42356,42357,42358,42359,42360,42361,42362,42363,42364,42365,42366,42368,42369,42370,42371,42372,42373,42374,42375,42376,42377,42378,42379,42380,42381,42382,42383,42384,42385,42386,42387,42388,42389,42390,42391,42392,42393,42394,42395,42396,42397,42398,42399,42400,42401,42402,42403,42404,42405,42406,42407,42408,42409,42410,42411,42412,42413,42414,42415,42416,42417,42418,42419,42420,42421,42422,42423,42424,42425,42426,42427,42428,42429,42430,42431,42432,42433,42434,42435,42436,42437,42438,42439,42440,42441,42442,42443,42444,42445,42446,42447,42448,42449,42450,42451,42452,42453,42454,42455,42456,42457,42458,42459,42460,42461,42462,42463,42464,42465,42466,42467,42468,42469,42470,42471,42472,42473,42474,42475,42476,42477,42478,42479,42480,42481,42482,42483,42484,42485,42486,42487,42488,42489,42490,42491,42492,42493,42494,42560,42561,42562,42563,42564,42565,42566,42567,42568,42569,42570,42571,42572,42573,42574,42575,42576,42577,42578,42579,42580,42581,42582,42583,42584,42585,42586,42587,42588,42589,42590,42591,42592,42593,42594,42595,42596,42597,42598,42599,42600,42601,42602,42603,42604,42605,42606,42607,42608,42609,42610,42611,42612,42613,42614,42615,42616,42617,42618,42619,42620,42621,42622,42624,42625,42626,42627,42628,42629,42630,42631,42632,42633,42634,42635,42636,42637,42638,42639,42640,42641,42642,42643,42644,42645,42646,42647,42648,42649,42650,42651,42652,42653,42654,42655,42656,42657,42658,42659,42660,42661,42662,42663,42664,42665,42666,42667,42668,42669,42670,42671,42672,42673,42674,42675,42676,42677,42678,42679,42680,42681,42682,42683,42684,42685,42686,42687,42688,42689,42690,42691,42692,42693,42694,42695,42696,42697,42698,42699,42700,42701,42702,42703,42704,42705,42706,42707,42708,42709,42710,42711,42712,42713,42714,42715,42716,42717,42718,42719,42720,42721,42722,42723,42724,42725,42726,42727,42728,42729,42730,42731,42732,42733,42734,42735,42736,42737,42738,42739,42740,42741,42742,42743,42744,42745,42746,42747,42748,42749,42750,42816,42817,42818,42819,42820,42821,42822,42823,42824,42825,42826,42827,42828,42829,42830,42831,42832,42833,42834,42835,42836,42837,42838,42839,42840,42841,42842,42843,42844,42845,42846,42847,42848,42849,42850,42851,42852,42853,42854,42855,42856,42857,42858,42859,42860,42861,42862,42863,42864,42865,42866,42867,42868,42869,42870,42871,42872,42873,42874,42875,42876,42877,42878,42880,42881,42882,42883,42884,42885,42886,42887,42888,42889,42890,42891,42892,42893,42894,42895,42896,42897,42898,42899,42900,42901,42902,42903,42904,42905,42906,42907,42908,42909,42910,42911,42912,42913,42914,42915,42916,42917,42918,42919,42920,42921,42922,42923,42924,42925,42926,42927,42928,42929,42930,42931,42932,42933,42934,42935,42936,42937,42938,42939,42940,42941,42942,42943,42944,42945,42946,42947,42948,42949,42950,42951,42952,42953,42954,42955,42956,42957,42958,42959,42960,42961,42962,42963,42964,42965,42966,42967,42968,42969,42970,42971,42972,42973,42974,42975,42976,42977,42978,42979,42980,42981,42982,42983,42984,42985,42986,42987,42988,42989,42990,42991,42992,42993,42994,42995,42996,42997,42998,42999,43000,43001,43002,43003,43004,43005,43006,43072,43073,43074,43075,43076,43077,43078,43079,43080,43081,43082,43083,43084,43085,43086,43087,43088,43089,43090,43091,43092,43093,43094,43095,43096,43097,43098,43099,43100,43101,43102,43103,43104,43105,43106,43107,43108,43109,43110,43111,43112,43113,43114,43115,43116,43117,43118,43119,43120,43121,43122,43123,43124,43125,43126,43127,43128,43129,43130,43131,43132,43133,43134,43136,43137,43138,43139,43140,43141,43142,43143,43144,43145,43146,43147,43148,43149,43150,43151,43152,43153,43154,43155,43156,43157,43158,43159,43160,43161,43162,43163,43164,43165,43166,43167,43168,43169,43170,43171,43172,43173,43174,43175,43176,43177,43178,43179,43180,43181,43182,43183,43184,43185,43186,43187,43188,43189,43190,43191,43192,43193,43194,43195,43196,43197,43198,43199,43200,43201,43202,43203,43204,43205,43206,43207,43208,43209,43210,43211,43212,43213,43214,43215,43216,43217,43218,43219,43220,43221,43222,43223,43224,43225,43226,43227,43228,43229,43230,43231,43232,43233,43234,43235,43236,43237,43238,43239,43240,43241,43242,43243,43244,43245,43246,43247,43248,43249,43250,43251,43252,43253,43254,43255,43256,43257,43258,43259,43260,43261,43262,43328,43329,43330,43331,43332,43333,43334,43335,43336,43337,43338,43339,43340,43341,43342,43343,43344,43345,43346,43347,43348,43349,43350,43351,43352,43353,43354,43355,43356,43357,43358,43359,43360,43361,43362,43363,43364,43365,43366,43367,43368,43369,43370,43371,43372,43373,43374,43375,43376,43377,43378,43379,43380,43381,43382,43383,43384,43385,43386,43387,43388,43389,43390,43392,43393,43394,43395,43396,43397,43398,43399,43400,43401,43402,43403,43404,43405,43406,43407,43408,43409,43410,43411,43412,43413,43414,43415,43416,43417,43418,43419,43420,43421,43422,43423,43424,43425,43426,43427,43428,43429,43430,43431,43432,43433,43434,43435,43436,43437,43438,43439,43440,43441,43442,43443,43444,43445,43446,43447,43448,43449,43450,43451,43452,43453,43454,43455,43456,43457,43458,43459,43460,43461,43462,43463,43464,43465,43466,43467,43468,43469,43470,43471,43472,43473,43474,43475,43476,43477,43478,43479,43480,43481,43482,43483,43484,43485,43486,43487,43488,43489,43490,43491,43492,43493,43494,43495,43496,43497,43498,43499,43500,43501,43502,43503,43504,43505,43506,43507,43508,43509,43510,43511,43512,43513,43514,43515,43516,43517,43518,43584,43585,43586,43587,43588,43589,43590,43591,43592,43593,43594,43595,43596,43597,43598,43599,43600,43601,43602,43603,43604,43605,43606,43607,43608,43609,43610,43611,43612,43613,43614,43615,43616,43617,43618,43619,43620,43621,43622,43623,43624,43625,43626,43627,43628,43629,43630,43631,43632,43633,43634,43635,43636,43637,43638,43639,43640,43641,43642,43643,43644,43645,43646,43648,43649,43650,43651,43652,43653,43654,43655,43656,43657,43658,43659,43660,43661,43662,43663,43664,43665,43666,43667,43668,43669,43670,43671,43672,43673,43674,43675,43676,43677,43678,43679,43680,43681,43682,43683,43684,43685,43686,43687,43688,43689,43690,43691,43692,43693,43694,43695,43696,43697,43698,43699,43700,43701,43702,43703,43704,43705,43706,43707,43708,43709,43710,43711,43712,43713,43714,43715,43716,43717,43718,43719,43720,43721,43722,43723,43724,43725,43726,43727,43728,43729,43730,43731,43732,43733,43734,43735,43736,43737,43738,43739,43740,43741,43742,43743,43744,43745,43746,43747,43748,43749,43750,43751,43752,43753,43754,43755,43756,43757,43758,43759,43760,43761,43762,43763,43764,43765,43766,43767,43768,43769,43770,43771,43772,43773,43774,43840,43841,43842,43843,43844,43845,43846,43847,43848,43849,43850,43851,43852,43853,43854,43855,43856,43857,43858,43859,43860,43861,43862,43863,43864,43865,43866,43867,43868,43869,43870,43871,43872,43873,43874,43875,43876,43877,43878,43879,43880,43881,43882,43883,43884,43885,43886,43887,43888,43889,43890,43891,43892,43893,43894,43895,43896,43897,43898,43899,43900,43901,43902,43904,43905,43906,43907,43908,43909,43910,43911,43912,43913,43914,43915,43916,43917,43918,43919,43920,43921,43922,43923,43924,43925,43926,43927,43928,43929,43930,43931,43932,43933,43934,43935,43936,43937,43938,43939,43940,43941,43942,43943,43944,43945,43946,43947,43948,43949,43950,43951,43952,43953,43954,43955,43956,43957,43958,43959,43960,43961,43962,43963,43964,43965,43966,43967,43968,43969,43970,43971,43972,43973,43974,43975,43976,43977,43978,43979,43980,43981,43982,43983,43984,43985,43986,43987,43988,43989,43990,43991,43992,43993,43994,43995,43996,43997,43998,43999,44000,44001,44002,44003,44004,44005,44006,44007,44008,44009,44010,44011,44012,44013,44014,44015,44016,44017,44018,44019,44020,44021,44022,44023,44024,44025,44026,44027,44028,44029,44030,44096,44097,44098,44099,44100,44101,44102,44103,44104,44105,44106,44107,44108,44109,44110,44111,44112,44113,44114,44115,44116,44117,44118,44119,44120,44121,44122,44123,44124,44125,44126,44127,44128,44129,44130,44131,44132,44133,44134,44135,44136,44137,44138,44139,44140,44141,44142,44143,44144,44145,44146,44147,44148,44149,44150,44151,44152,44153,44154,44155,44156,44157,44158,44160,44161,44162,44163,44164,44165,44166,44167,44168,44169,44170,44171,44172,44173,44174,44175,44176,44177,44178,44179,44180,44181,44182,44183,44184,44185,44186,44187,44188,44189,44190,44191,44192,44193,44194,44195,44196,44197,44198,44199,44200,44201,44202,44203,44204,44205,44206,44207,44208,44209,44210,44211,44212,44213,44214,44215,44216,44217,44218,44219,44220,44221,44222,44223,44224,44225,44226,44227,44228,44229,44230,44231,44232,44233,44234,44235,44236,44237,44238,44239,44240,44241,44242,44243,44244,44245,44246,44247,44248,44249,44250,44251,44252,44253,44254,44255,44256,44257,44258,44259,44260,44261,44262,44263,44264,44265,44266,44267,44268,44269,44270,44271,44272,44273,44274,44275,44276,44277,44278,44279,44280,44281,44282,44283,44284,44285,44286,44352,44353,44354,44355,44356,44357,44358,44359,44360,44361,44362,44363,44364,44365,44366,44367,44368,44369,44370,44371,44372,44373,44374,44375,44376,44377,44378,44379,44380,44381,44382,44383,44384,44385,44386,44387,44388,44389,44390,44391,44392,44393,44394,44395,44396,44397,44398,44399,44400,44401,44402,44403,44404,44405,44406,44407,44408,44409,44410,44411,44412,44413,44414,44416,44417,44418,44419,44420,44421,44422,44423,44424,44425,44426,44427,44428,44429,44430,44431,44432,44433,44434,44435,44436,44437,44438,44439,44440,44441,44442,44443,44444,44445,44446,44447,44448,44449,44450,44451,44452,44453,44454,44455,44456,44457,44458,44459,44460,44461,44462,44463,44464,44465,44466,44467,44468,44469,44470,44471,44472,44473,44474,44475,44476,44477,44478,44479,44480,44481,44482,44483,44484,44485,44486,44487,44488,44489,44490,44491,44492,44493,44494,44495,44496,44497,44498,44499,44500,44501,44502,44503,44504,44505,44506,44507,44508,44509,44510,44511,44512,44513,44514,44515,44516,44517,44518,44519,44520,44521,44522,44523,44524,44525,44526,44527,44528,44529,44530,44531,44532,44533,44534,44535,44536,44537,44538,44539,44540,44541,44542,44608,44609,44610,44611,44612,44613,44614,44615,44616,44617,44618,44619,44620,44621,44622,44623,44624,44625,44626,44627,44628,44629,44630,44631,44632,44633,44634,44635,44636,44637,44638,44639,44640,44641,44642,44643,44644,44645,44646,44647,44648,44649,44650,44651,44652,44653,44654,44655,44656,44657,44658,44659,44660,44661,44662,44663,44664,44665,44666,44667,44668,44669,44670,44672,44673,44674,44675,44676,44677,44678,44679,44680,44681,44682,44683,44684,44685,44686,44687,44688,44689,44690,44691,44692,44693,44694,44695,44696,44697,44698,44699,44700,44701,44702,44703,44704,44705,44706,44707,44708,44709,44710,44711,44712,44713,44714,44715,44716,44717,44718,44719,44720,44721,44722,44723,44724,44725,44726,44727,44728,44729,44730,44731,44732,44733,44734,44735,44736,44737,44738,44739,44740,44741,44742,44743,44744,44745,44746,44747,44748,44749,44750,44751,44752,44753,44754,44755,44756,44757,44758,44759,44760,44761,44762,44763,44764,44765,44766,44767,44768,44769,44770,44771,44772,44773,44774,44775,44776,44777,44778,44779,44780,44781,44782,44783,44784,44785,44786,44787,44788,44789,44790,44791,44792,44793,44794,44795,44796,44797,44798,44864,44865,44866,44867,44868,44869,44870,44871,44872,44873,44874,44875,44876,44877,44878,44879,44880,44881,44882,44883,44884,44885,44886,44887,44888,44889,44890,44891,44892,44893,44894,44895,44896,44897,44898,44899,44900,44901,44902,44903,44904,44905,44906,44907,44908,44909,44910,44911,44912,44913,44914,44915,44916,44917,44918,44919,44920,44921,44922,44923,44924,44925,44926,44928,44929,44930,44931,44932,44933,44934,44935,44936,44937,44938,44939,44940,44941,44942,44943,44944,44945,44946,44947,44948,44949,44950,44951,44952,44953,44954,44955,44956,44957,44958,44959,44960,44961,44962,44963,44964,44965,44966,44967,44968,44969,44970,44971,44972,44973,44974,44975,44976,44977,44978,44979,44980,44981,44982,44983,44984,44985,44986,44987,44988,44989,44990,44991,44992,44993,44994,44995,44996,44997,44998,44999,45000,45001,45002,45003,45004,45005,45006,45007,45008,45009,45010,45011,45012,45013,45014,45015,45016,45017,45018,45019,45020,45021,45022,45023,45024,45025,45026,45027,45028,45029,45030,45031,45032,45033,45034,45035,45036,45037,45038,45039,45040,45041,45042,45043,45044,45045,45046,45047,45048,45049,45050,45051,45052,45053,45054,45120,45121,45122,45123,45124,45125,45126,45127,45128,45129,45130,45131,45132,45133,45134,45135,45136,45137,45138,45139,45140,45141,45142,45143,45144,45145,45146,45147,45148,45149,45150,45151,45152,45153,45154,45155,45156,45157,45158,45159,45160,45161,45162,45163,45164,45165,45166,45167,45168,45169,45170,45171,45172,45173,45174,45175,45176,45177,45178,45179,45180,45181,45182,45184,45185,45186,45187,45188,45189,45190,45191,45192,45193,45194,45195,45196,45197,45198,45199,45200,45201,45202,45203,45204,45205,45206,45207,45208,45209,45210,45211,45212,45213,45214,45215,45216,45217,45218,45219,45220,45221,45222,45223,45224,45225,45226,45227,45228,45229,45230,45231,45232,45233,45234,45235,45236,45237,45238,45239,45240,45241,45242,45243,45244,45245,45246,45247,45248,45249,45250,45251,45252,45253,45254,45255,45256,45257,45258,45259,45260,45261,45262,45263,45264,45265,45266,45267,45268,45269,45270,45271,45272,45273,45274,45275,45276,45277,45278,45279,45280,45281,45282,45283,45284,45285,45286,45287,45288,45289,45290,45291,45292,45293,45294,45295,45296,45297,45298,45299,45300,45301,45302,45303,45304,45305,45306,45307,45308,45309,45310,45376,45377,45378,45379,45380,45381,45382,45383,45384,45385,45386,45387,45388,45389,45390,45391,45392,45393,45394,45395,45396,45397,45398,45399,45400,45401,45402,45403,45404,45405,45406,45407,45408,45409,45410,45411,45412,45413,45414,45415,45416,45417,45418,45419,45420,45421,45422,45423,45424,45425,45426,45427,45428,45429,45430,45431,45432,45433,45434,45435,45436,45437,45438,45440,45441,45442,45443,45444,45445,45446,45447,45448,45449,45450,45451,45452,45453,45454,45455,45456,45457,45458,45459,45460,45461,45462,45463,45464,45465,45466,45467,45468,45469,45470,45471,45472,45473,45474,45475,45476,45477,45478,45479,45480,45481,45482,45483,45484,45485,45486,45487,45488,45489,45490,45491,45492,45493,45494,45495,45496,45497,45498,45499,45500,45501,45502,45503,45504,45505,45506,45507,45508,45509,45510,45511,45512,45513,45514,45515,45516,45517,45518,45519,45520,45521,45522,45523,45524,45525,45526,45527,45528,45529,45530,45531,45532,45533,45534,45535,45536,45537,45538,45539,45540,45541,45542,45543,45544,45545,45546,45547,45548,45549,45550,45551,45552,45553,45554,45555,45556,45557,45558,45559,45560,45561,45562,45563,45564,45565,45566,45632,45633,45634,45635,45636,45637,45638,45639,45640,45641,45642,45643,45644,45645,45646,45647,45648,45649,45650,45651,45652,45653,45654,45655,45656,45657,45658,45659,45660,45661,45662,45663,45664,45665,45666,45667,45668,45669,45670,45671,45672,45673,45674,45675,45676,45677,45678,45679,45680,45681,45682,45683,45684,45685,45686,45687,45688,45689,45690,45691,45692,45693,45694,45696,45697,45698,45699,45700,45701,45702,45703,45704,45705,45706,45707,45708,45709,45710,45711,45712,45713,45714,45715,45716,45717,45718,45719,45720,45721,45722,45723,45724,45725,45726,45727,45728,45729,45730,45731,45732,45733,45734,45735,45736,45737,45738,45739,45740,45741,45742,45743,45744,45745,45746,45747,45748,45749,45750,45751,45752,45753,45754,45755,45756,45757,45758,45759,45760,45761,45762,45763,45764,45765,45766,45767,45768,45769,45770,45771,45772,45773,45774,45775,45776,45777,45778,45779,45780,45781,45782,45783,45784,45785,45786,45787,45788,45789,45790,45791,45792,45793,45794,45795,45796,45797,45798,45799,45800,45801,45802,45803,45804,45805,45806,45807,45808,45809,45810,45811,45812,45813,45814,45815,45816,45817,45818,45819,45820,45821,45822,45888,45889,45890,45891,45892,45893,45894,45895,45896,45897,45898,45899,45900,45901,45902,45903,45904,45905,45906,45907,45908,45909,45910,45911,45912,45913,45914,45915,45916,45917,45918,45919,45920,45921,45922,45923,45924,45925,45926,45927,45928,45929,45930,45931,45932,45933,45934,45935,45936,45937,45938,45939,45940,45941,45942,45943,45944,45945,45946,45947,45948,45949,45950,45952,45953,45954,45955,45956,45957,45958,45959,45960,45961,45962,45963,45964,45965,45966,45967,45968,45969,45970,45971,45972,45973,45974,45975,45976,45977,45978,45979,45980,45981,45982,45983,45984,45985,45986,45987,45988,45989,45990,45991,45992,45993,45994,45995,45996,45997,45998,45999,46000,46001,46002,46003,46004,46005,46006,46007,46008,46009,46010,46011,46012,46013,46014,46015,46016,46017,46018,46019,46020,46021,46022,46023,46024,46025,46026,46027,46028,46029,46030,46031,46032,46033,46034,46035,46036,46037,46038,46039,46040,46041,46042,46043,46044,46045,46046,46047,46048,46049,46050,46051,46052,46053,46054,46055,46056,46057,46058,46059,46060,46061,46062,46063,46064,46065,46066,46067,46068,46069,46070,46071,46072,46073,46074,46075,46076,46077,46078,46144,46145,46146,46147,46148,46149,46150,46151,46152,46153,46154,46155,46156,46157,46158,46159,46160,46161,46162,46163,46164,46165,46166,46167,46168,46169,46170,46171,46172,46173,46174,46175,46176,46177,46178,46179,46180,46181,46182,46183,46184,46185,46186,46187,46188,46189,46190,46191,46192,46193,46194,46195,46196,46197,46198,46199,46200,46201,46202,46203,46204,46205,46206,46208,46209,46210,46211,46212,46213,46214,46215,46216,46217,46218,46219,46220,46221,46222,46223,46224,46225,46226,46227,46228,46229,46230,46231,46232,46233,46234,46235,46236,46237,46238,46239,46240,46241,46242,46243,46244,46245,46246,46247,46248,46249,46250,46251,46252,46253,46254,46255,46256,46257,46258,46259,46260,46261,46262,46263,46264,46265,46266,46267,46268,46269,46270,46271,46272,46273,46274,46275,46276,46277,46278,46279,46280,46281,46282,46283,46284,46285,46286,46287,46288,46289,46290,46291,46292,46293,46294,46295,46296,46297,46298,46299,46300,46301,46302,46303,46304,46305,46306,46307,46308,46309,46310,46311,46312,46313,46314,46315,46316,46317,46318,46319,46320,46321,46322,46323,46324,46325,46326,46327,46328,46329,46330,46331,46332,46333,46334,46400,46401,46402,46403,46404,46405,46406,46407,46408,46409,46410,46411,46412,46413,46414,46415,46416,46417,46418,46419,46420,46421,46422,46423,46424,46425,46426,46427,46428,46429,46430,46431,46432,46433,46434,46435,46436,46437,46438,46439,46440,46441,46442,46443,46444,46445,46446,46447,46448,46449,46450,46451,46452,46453,46454,46455,46456,46457,46458,46459,46460,46461,46462,46464,46465,46466,46467,46468,46469,46470,46471,46472,46473,46474,46475,46476,46477,46478,46479,46480,46481,46482,46483,46484,46485,46486,46487,46488,46489,46490,46491,46492,46493,46494,46495,46496,46497,46498,46499,46500,46501,46502,46503,46504,46505,46506,46507,46508,46509,46510,46511,46512,46513,46514,46515,46516,46517,46518,46519,46520,46521,46522,46523,46524,46525,46526,46527,46528,46529,46530,46531,46532,46533,46534,46535,46536,46537,46538,46539,46540,46541,46542,46543,46544,46545,46546,46547,46548,46549,46550,46551,46552,46553,46554,46555,46556,46557,46558,46559,46560,46561,46562,46563,46564,46565,46566,46567,46568,46569,46570,46571,46572,46573,46574,46575,46576,46577,46578,46579,46580,46581,46582,46583,46584,46585,46586,46587,46588,46589,46590,46656,46657,46658,46659,46660,46661,46662,46663,46664,46665,46666,46667,46668,46669,46670,46671,46672,46673,46674,46675,46676,46677,46678,46679,46680,46681,46682,46683,46684,46685,46686,46687,46688,46689,46690,46691,46692,46693,46694,46695,46696,46697,46698,46699,46700,46701,46702,46703,46704,46705,46706,46707,46708,46709,46710,46711,46712,46713,46714,46715,46716,46717,46718,46720,46721,46722,46723,46724,46725,46726,46727,46728,46729,46730,46731,46732,46733,46734,46735,46736,46737,46738,46739,46740,46741,46742,46743,46744,46745,46746,46747,46748,46749,46750,46751,46752,46753,46754,46755,46756,46757,46758,46759,46760,46761,46762,46763,46764,46765,46766,46767,46768,46769,46770,46771,46772,46773,46774,46775,46776,46777,46778,46779,46780,46781,46782,46783,46784,46785,46786,46787,46788,46789,46790,46791,46792,46793,46794,46795,46796,46797,46798,46799,46800,46801,46802,46803,46804,46805,46806,46807,46808,46809,46810,46811,46812,46813,46814,46815,46816,46817,46818,46819,46820,46821,46822,46823,46824,46825,46826,46827,46828,46829,46830,46831,46832,46833,46834,46835,46836,46837,46838,46839,46840,46841,46842,46843,46844,46845,46846,46912,46913,46914,46915,46916,46917,46918,46919,46920,46921,46922,46923,46924,46925,46926,46927,46928,46929,46930,46931,46932,46933,46934,46935,46936,46937,46938,46939,46940,46941,46942,46943,46944,46945,46946,46947,46948,46949,46950,46951,46952,46953,46954,46955,46956,46957,46958,46959,46960,46961,46962,46963,46964,46965,46966,46967,46968,46969,46970,46971,46972,46973,46974,46976,46977,46978,46979,46980,46981,46982,46983,46984,46985,46986,46987,46988,46989,46990,46991,46992,46993,46994,46995,46996,46997,46998,46999,47000,47001,47002,47003,47004,47005,47006,47007,47008,47009,47010,47011,47012,47013,47014,47015,47016,47017,47018,47019,47020,47021,47022,47023,47024,47025,47026,47027,47028,47029,47030,47031,47032,47033,47034,47035,47036,47037,47038,47039,47040,47041,47042,47043,47044,47045,47046,47047,47048,47049,47050,47051,47052,47053,47054,47055,47056,47057,47058,47059,47060,47061,47062,47063,47064,47065,47066,47067,47068,47069,47070,47071,47072,47073,47074,47075,47076,47077,47078,47079,47080,47081,47082,47083,47084,47085,47086,47087,47088,47089,47090,47091,47092,47093,47094,47095,47096,47097,47098,47099,47100,47101,47102,47168,47169,47170,47171,47172,47173,47174,47175,47176,47177,47178,47179,47180,47181,47182,47183,47184,47185,47186,47187,47188,47189,47190,47191,47192,47193,47194,47195,47196,47197,47198,47199,47200,47201,47202,47203,47204,47205,47206,47207,47208,47209,47210,47211,47212,47213,47214,47215,47216,47217,47218,47219,47220,47221,47222,47223,47224,47225,47226,47227,47228,47229,47230,47232,47233,47234,47235,47236,47237,47238,47239,47240,47241,47242,47243,47244,47245,47246,47247,47248,47249,47250,47251,47252,47253,47254,47255,47256,47257,47258,47259,47260,47261,47262,47263,47264,47265,47266,47267,47268,47269,47270,47271,47272,47273,47274,47275,47276,47277,47278,47279,47280,47281,47282,47283,47284,47285,47286,47287,47288,47289,47290,47291,47292,47293,47294,47295,47296,47297,47298,47299,47300,47301,47302,47303,47304,47305,47306,47307,47308,47309,47310,47311,47312,47313,47314,47315,47316,47317,47318,47319,47320,47321,47322,47323,47324,47325,47326,47327,47328,47329,47330,47331,47332,47333,47334,47335,47336,47337,47338,47339,47340,47341,47342,47343,47344,47345,47346,47347,47348,47349,47350,47351,47352,47353,47354,47355,47356,47357,47358,47424,47425,47426,47427,47428,47429,47430,47431,47432,47433,47434,47435,47436,47437,47438,47439,47440,47441,47442,47443,47444,47445,47446,47447,47448,47449,47450,47451,47452,47453,47454,47455,47456,47457,47458,47459,47460,47461,47462,47463,47464,47465,47466,47467,47468,47469,47470,47471,47472,47473,47474,47475,47476,47477,47478,47479,47480,47481,47482,47483,47484,47485,47486,47488,47489,47490,47491,47492,47493,47494,47495,47496,47497,47498,47499,47500,47501,47502,47503,47504,47505,47506,47507,47508,47509,47510,47511,47512,47513,47514,47515,47516,47517,47518,47519,47520,47521,47522,47523,47524,47525,47526,47527,47528,47529,47530,47531,47532,47533,47534,47535,47536,47537,47538,47539,47540,47541,47542,47543,47544,47545,47546,47547,47548,47549,47550,47551,47552,47553,47554,47555,47556,47557,47558,47559,47560,47561,47562,47563,47564,47565,47566,47567,47568,47569,47570,47571,47572,47573,47574,47575,47576,47577,47578,47579,47580,47581,47582,47583,47584,47585,47586,47587,47588,47589,47590,47591,47592,47593,47594,47595,47596,47597,47598,47599,47600,47601,47602,47603,47604,47605,47606,47607,47608,47609,47610,47611,47612,47613,47614,47680,47681,47682,47683,47684,47685,47686,47687,47688,47689,47690,47691,47692,47693,47694,47695,47696,47697,47698,47699,47700,47701,47702,47703,47704,47705,47706,47707,47708,47709,47710,47711,47712,47713,47714,47715,47716,47717,47718,47719,47720,47721,47722,47723,47724,47725,47726,47727,47728,47729,47730,47731,47732,47733,47734,47735,47736,47737,47738,47739,47740,47741,47742,47744,47745,47746,47747,47748,47749,47750,47751,47752,47753,47754,47755,47756,47757,47758,47759,47760,47761,47762,47763,47764,47765,47766,47767,47768,47769,47770,47771,47772,47773,47774,47775,47776,47777,47778,47779,47780,47781,47782,47783,47784,47785,47786,47787,47788,47789,47790,47791,47792,47793,47794,47795,47796,47797,47798,47799,47800,47801,47802,47803,47804,47805,47806,47807,47808,47809,47810,47811,47812,47813,47814,47815,47816,47817,47818,47819,47820,47821,47822,47823,47824,47825,47826,47827,47828,47829,47830,47831,47832,47833,47834,47835,47836,47837,47838,47839,47840,47841,47842,47843,47844,47845,47846,47847,47848,47849,47850,47851,47852,47853,47854,47855,47856,47857,47858,47859,47860,47861,47862,47863,47864,47865,47866,47867,47868,47869,47870,47936,47937,47938,47939,47940,47941,47942,47943,47944,47945,47946,47947,47948,47949,47950,47951,47952,47953,47954,47955,47956,47957,47958,47959,47960,47961,47962,47963,47964,47965,47966,47967,47968,47969,47970,47971,47972,47973,47974,47975,47976,47977,47978,47979,47980,47981,47982,47983,47984,47985,47986,47987,47988,47989,47990,47991,47992,47993,47994,47995,47996,47997,47998,48000,48001,48002,48003,48004,48005,48006,48007,48008,48009,48010,48011,48012,48013,48014,48015,48016,48017,48018,48019,48020,48021,48022,48023,48024,48025,48026,48027,48028,48029,48030,48031,48032,48033,48034,48035,48036,48037,48038,48039,48040,48041,48042,48043,48044,48045,48046,48047,48048,48049,48050,48051,48052,48053,48054,48055,48056,48057,48058,48059,48060,48061,48062,48063,48064,48065,48066,48067,48068,48069,48070,48071,48072,48073,48074,48075,48076,48077,48078,48079,48080,48081,48082,48083,48084,48085,48086,48087,48088,48089,48090,48091,48092,48093,48094,48095,48096,48097,48098,48099,48100,48101,48102,48103,48104,48105,48106,48107,48108,48109,48110,48111,48112,48113,48114,48115,48116,48117,48118,48119,48120,48121,48122,48123,48124,48125,48126,48192,48193,48194,48195,48196,48197,48198,48199,48200,48201,48202,48203,48204,48205,48206,48207,48208,48209,48210,48211,48212,48213,48214,48215,48216,48217,48218,48219,48220,48221,48222,48223,48224,48225,48226,48227,48228,48229,48230,48231,48232,48233,48234,48235,48236,48237,48238,48239,48240,48241,48242,48243,48244,48245,48246,48247,48248,48249,48250,48251,48252,48253,48254,48256,48257,48258,48259,48260,48261,48262,48263,48264,48265,48266,48267,48268,48269,48270,48271,48272,48273,48274,48275,48276,48277,48278,48279,48280,48281,48282,48283,48284,48285,48286,48287,48288,48289,48290,48291,48292,48293,48294,48295,48296,48297,48298,48299,48300,48301,48302,48303,48304,48305,48306,48307,48308,48309,48310,48311,48312,48313,48314,48315,48316,48317,48318,48319,48320,48321,48322,48323,48324,48325,48326,48327,48328,48329,48330,48331,48332,48333,48334,48335,48336,48337,48338,48339,48340,48341,48342,48343,48344,48345,48346,48347,48348,48349,48350,48351,48352,48353,48354,48355,48356,48357,48358,48359,48360,48361,48362,48363,48364,48365,48366,48367,48368,48369,48370,48371,48372,48373,48374,48375,48376,48377,48378,48379,48380,48381,48382,48448,48449,48450,48451,48452,48453,48454,48455,48456,48457,48458,48459,48460,48461,48462,48463,48464,48465,48466,48467,48468,48469,48470,48471,48472,48473,48474,48475,48476,48477,48478,48479,48480,48481,48482,48483,48484,48485,48486,48487,48488,48489,48490,48491,48492,48493,48494,48495,48496,48497,48498,48499,48500,48501,48502,48503,48504,48505,48506,48507,48508,48509,48510,48512,48513,48514,48515,48516,48517,48518,48519,48520,48521,48522,48523,48524,48525,48526,48527,48528,48529,48530,48531,48532,48533,48534,48535,48536,48537,48538,48539,48540,48541,48542,48543,48544,48545,48546,48547,48548,48549,48550,48551,48552,48553,48554,48555,48556,48557,48558,48559,48560,48561,48562,48563,48564,48565,48566,48567,48568,48569,48570,48571,48572,48573,48574,48575,48576,48577,48578,48579,48580,48581,48582,48583,48584,48585,48586,48587,48588,48589,48590,48591,48592,48593,48594,48595,48596,48597,48598,48599,48600,48601,48602,48603,48604,48605,48606,48607,48608,48609,48610,48611,48612,48613,48614,48615,48616,48617,48618,48619,48620,48621,48622,48623,48624,48625,48626,48627,48628,48629,48630,48631,48632,48633,48634,48635,48636,48637,48638,48704,48705,48706,48707,48708,48709,48710,48711,48712,48713,48714,48715,48716,48717,48718,48719,48720,48721,48722,48723,48724,48725,48726,48727,48728,48729,48730,48731,48732,48733,48734,48735,48736,48737,48738,48739,48740,48741,48742,48743,48744,48745,48746,48747,48748,48749,48750,48751,48752,48753,48754,48755,48756,48757,48758,48759,48760,48761,48762,48763,48764,48765,48766,48768,48769,48770,48771,48772,48773,48774,48775,48776,48777,48778,48779,48780,48781,48782,48783,48784,48785,48786,48787,48788,48789,48790,48791,48792,48793,48794,48795,48796,48797,48798,48799,48800,48801,48802,48803,48804,48805,48806,48807,48808,48809,48810,48811,48812,48813,48814,48815,48816,48817,48818,48819,48820,48821,48822,48823,48824,48825,48826,48827,48828,48829,48830,48831,48832,48833,48834,48835,48836,48837,48838,48839,48840,48841,48842,48843,48844,48845,48846,48847,48848,48849,48850,48851,48852,48853,48854,48855,48856,48857,48858,48859,48860,48861,48862,48863,48864,48865,48866,48867,48868,48869,48870,48871,48872,48873,48874,48875,48876,48877,48878,48879,48880,48881,48882,48883,48884,48885,48886,48887,48888,48889,48890,48891,48892,48893,48894,48960,48961,48962,48963,48964,48965,48966,48967,48968,48969,48970,48971,48972,48973,48974,48975,48976,48977,48978,48979,48980,48981,48982,48983,48984,48985,48986,48987,48988,48989,48990,48991,48992,48993,48994,48995,48996,48997,48998,48999,49000,49001,49002,49003,49004,49005,49006,49007,49008,49009,49010,49011,49012,49013,49014,49015,49016,49017,49018,49019,49020,49021,49022,49024,49025,49026,49027,49028,49029,49030,49031,49032,49033,49034,49035,49036,49037,49038,49039,49040,49041,49042,49043,49044,49045,49046,49047,49048,49049,49050,49051,49052,49053,49054,49055,49056,49057,49058,49059,49060,49061,49062,49063,49064,49065,49066,49067,49068,49069,49070,49071,49072,49073,49074,49075,49076,49077,49078,49079,49080,49081,49082,49083,49084,49085,49086,49087,49088,49089,49090,49091,49092,49093,49094,49095,49096,49097,49098,49099,49100,49101,49102,49103,49104,49105,49106,49107,49108,49109,49110,49111,49112,49113,49114,49115,49116,49117,49118,49119,49120,49121,49122,49123,49124,49125,49126,49127,49128,49129,49130,49131,49132,49133,49134,49135,49136,49137,49138,49139,49140,49141,49142,49143,49144,49145,49146,49147,49148,49149,49150,49216,49217,49218,49219,49220,49221,49222,49223,49224,49225,49226,49227,49228,49229,49230,49231,49232,49233,49234,49235,49236,49237,49238,49239,49240,49241,49242,49243,49244,49245,49246,49247,49248,49249,49250,49251,49252,49253,49254,49255,49256,49257,49258,49259,49260,49261,49262,49263,49264,49265,49266,49267,49268,49269,49270,49271,49272,49273,49274,49275,49276,49277,49278,49280,49281,49282,49283,49284,49285,49286,49287,49288,49289,49290,49291,49292,49293,49294,49295,49296,49297,49298,49299,49300,49301,49302,49303,49304,49305,49306,49307,49308,49309,49310,49311,49312,49313,49314,49315,49316,49317,49318,49319,49320,49321,49322,49323,49324,49325,49326,49327,49328,49329,49330,49331,49332,49333,49334,49335,49336,49337,49338,49339,49340,49341,49342,49343,49344,49345,49346,49347,49348,49349,49350,49351,49352,49353,49354,49355,49356,49357,49358,49359,49360,49361,49362,49363,49364,49365,49366,49367,49368,49369,49370,49371,49372,49373,49374,49375,49376,49377,49378,49379,49380,49381,49382,49383,49384,49385,49386,49387,49388,49389,49390,49391,49392,49393,49394,49395,49396,49397,49398,49399,49400,49401,49402,49403,49404,49405,49406,49472,49473,49474,49475,49476,49477,49478,49479,49480,49481,49482,49483,49484,49485,49486,49487,49488,49489,49490,49491,49492,49493,49494,49495,49496,49497,49498,49499,49500,49501,49502,49503,49504,49505,49506,49507,49508,49509,49510,49511,49512,49513,49514,49515,49516,49517,49518,49519,49520,49521,49522,49523,49524,49525,49526,49527,49528,49529,49530,49531,49532,49533,49534,49536,49537,49538,49539,49540,49541,49542,49543,49544,49545,49546,49547,49548,49549,49550,49551,49552,49553,49554,49555,49556,49557,49558,49559,49560,49561,49562,49563,49564,49565,49566,49567,49568,49569,49570,49571,49572,49573,49574,49575,49576,49577,49578,49579,49580,49581,49582,49583,49584,49585,49586,49587,49588,49589,49590,49591,49592,49593,49594,49595,49596,49597,49598,49599,49600,49601,49602,49603,49604,49605,49606,49607,49608,49609,49610,49611,49612,49613,49614,49615,49616,49617,49618,49619,49620,49621,49622,49623,49624,49625,49626,49627,49628,49629,49630,49631,49632,49633,49634,49635,49636,49637,49638,49639,49640,49641,49642,49643,49644,49645,49646,49647,49648,49649,49650,49651,49652,49653,49654,49655,49656,49657,49658,49659,49660,49661,49662,49728,49729,49730,49731,49732,49733,49734,49735,49736,49737,49738,49739,49740,49741,49742,49743,49744,49745,49746,49747,49748,49749,49750,49751,49752,49753,49754,49755,49756,49757,49758,49759,49760,49761,49762,49763,49764,49765,49766,49767,49768,49769,49770,49771,49772,49773,49774,49775,49776,49777,49778,49779,49780,49781,49782,49783,49784,49785,49786,49787,49788,49789,49790,49792,49793,49794,49795,49796,49797,49798,49799,49800,49801,49802,49803,49804,49805,49806,49807,49808,49809,49810,49811,49812,49813,49814,49815,49816,49817,49818,49819,49820,49821,49822,49823,49824,49825,49826,49827,49828,49829,49830,49831,49832,49833,49834,49835,49836,49837,49838,49839,49840,49841,49842,49843,49844,49845,49846,49847,49848,49849,49850,49851,49852,49853,49854,49855,49856,49857,49858,49859,49860,49861,49862,49863,49864,49865,49866,49867,49868,49869,49870,49871,49872,49873,49874,49875,49876,49877,49878,49879,49880,49881,49882,49883,49884,49885,49886,49887,49888,49889,49890,49891,49892,49893,49894,49895,49896,49897,49898,49899,49900,49901,49902,49903,49904,49905,49906,49907,49908,49909,49910,49911,49912,49913,49914,49915,49916,49917,49918,49984,49985,49986,49987,49988,49989,49990,49991,49992,49993,49994,49995,49996,49997,49998,49999,50000,50001,50002,50003,50004,50005,50006,50007,50008,50009,50010,50011,50012,50013,50014,50015,50016,50017,50018,50019,50020,50021,50022,50023,50024,50025,50026,50027,50028,50029,50030,50031,50032,50033,50034,50035,50036,50037,50038,50039,50040,50041,50042,50043,50044,50045,50046,50048,50049,50050,50051,50052,50053,50054,50055,50056,50057,50058,50059,50060,50061,50062,50063,50064,50065,50066,50067,50068,50069,50070,50071,50072,50073,50074,50075,50076,50077,50078,50079,50080,50081,50082,50083,50084,50085,50086,50087,50088,50089,50090,50091,50092,50093,50094,50095,50096,50097,50098,50099,50100,50101,50102,50103,50104,50105,50106,50107,50108,50109,50110,50111,50112,50113,50114,50115,50116,50117,50118,50119,50120,50121,50122,50123,50124,50125,50126,50127,50128,50129,50130,50131,50132,50133,50134,50135,50136,50137,50138,50139,50140,50141,50142,50143,50144,50145,50146,50147,50148,50149,50150,50151,50152,50153,50154,50155,50156,50157,50158,50159,50160,50161,50162,50163,50164,50165,50166,50167,50168,50169,50170,50171,50172,50173,50174,50240,50241,50242,50243,50244,50245,50246,50247,50248,50249,50250,50251,50252,50253,50254,50255,50256,50257,50258,50259,50260,50261,50262,50263,50264,50265,50266,50267,50268,50269,50270,50271,50272,50273,50274,50275,50276,50277,50278,50279,50280,50281,50282,50283,50284,50285,50286,50287,50288,50289,50290,50291,50292,50293,50294,50295,50296,50297,50298,50299,50300,50301,50302,50304,50305,50306,50307,50308,50309,50310,50311,50312,50313,50314,50315,50316,50317,50318,50319,50320,50321,50322,50323,50324,50325,50326,50327,50328,50329,50330,50331,50332,50333,50334,50335,50336,50337,50338,50339,50340,50341,50342,50343,50344,50345,50346,50347,50348,50349,50350,50351,50352,50353,50354,50355,50356,50357,50358,50359,50360,50361,50362,50363,50364,50365,50366,50367,50368,50369,50370,50371,50372,50373,50374,50375,50376,50377,50378,50379,50380,50381,50382,50383,50384,50385,50386,50387,50388,50389,50390,50391,50392,50393,50394,50395,50396,50397,50398,50399,50400,50401,50402,50403,50404,50405,50406,50407,50408,50409,50410,50411,50412,50413,50414,50415,50416,50417,50418,50419,50420,50421,50422,50423,50424,50425,50426,50427,50428,50429,50430,50496,50497,50498,50499,50500,50501,50502,50503,50504,50505,50506,50507,50508,50509,50510,50511,50512,50513,50514,50515,50516,50517,50518,50519,50520,50521,50522,50523,50524,50525,50526,50527,50528,50529,50530,50531,50532,50533,50534,50535,50536,50537,50538,50539,50540,50541,50542,50543,50544,50545,50546,50547,50548,50549,50550,50551,50552,50553,50554,50555,50556,50557,50558,50560,50561,50562,50563,50564,50565,50566,50567,50568,50569,50570,50571,50572,50573,50574,50575,50576,50577,50578,50579,50580,50581,50582,50583,50584,50585,50586,50587,50588,50589,50590,50591,50592,50593,50594,50595,50596,50597,50598,50599,50600,50601,50602,50603,50604,50605,50606,50607,50608,50609,50610,50611,50612,50613,50614,50615,50616,50617,50618,50619,50620,50621,50622,50623,50624,50625,50626,50627,50628,50629,50630,50631,50632,50633,50634,50635,50636,50637,50638,50639,50640,50641,50642,50643,50644,50645,50646,50647,50648,50649,50650,50651,50652,50653,50654,50655,50656,50657,50658,50659,50660,50661,50662,50663,50664,50665,50666,50667,50668,50669,50670,50671,50672,50673,50674,50675,50676,50677,50678,50679,50680,50681,50682,50683,50684,50685,50686,50752,50753,50754,50755,50756,50757,50758,50759,50760,50761,50762,50763,50764,50765,50766,50767,50768,50769,50770,50771,50772,50773,50774,50775,50776,50777,50778,50779,50780,50781,50782,50783,50784,50785,50786,50787,50788,50789,50790,50791,50792,50793,50794,50795,50796,50797,50798,50799,50800,50801,50802,50803,50804,50805,50806,50807,50808,50809,50810,50811,50812,50813,50814,50816,50817,50818,50819,50820,50821,50822,50823,50824,50825,50826,50827,50828,50829,50830,50831,50832,50833,50834,50835,50836,50837,50838,50839,50840,50841,50842,50843,50844,50845,50846,50847,50848,50849,50850,50851,50852,50853,50854,50855,50856,50857,50858,50859,50860,50861,50862,50863,50864,50865,50866,50867,50868,50869,50870,50871,50872,50873,50874,50875,50876,50877,50878,50879,50880,50881,50882,50883,50884,50885,50886,50887,50888,50889,50890,50891,50892,50893,50894,50895,50896,50897,50898,50899,50900,50901,50902,50903,50904,50905,50906,50907,50908,50909,50910,50911,50912,50913,50914,50915,50916,50917,50918,50919,50920,50921,50922,50923,50924,50925,50926,50927,50928,50929,50930,50931,50932,50933,50934,50935,50936,50937,50938,50939,50940,50941,50942,51008,51009,51010,51011,51012,51013,51014,51015,51016,51017,51018,51019,51020,51021,51022,51023,51024,51025,51026,51027,51028,51029,51030,51031,51032,51033,51034,51035,51036,51037,51038,51039,51040,51041,51042,51043,51044,51045,51046,51047,51048,51049,51050,51051,51052,51053,51054,51055,51056,51057,51058,51059,51060,51061,51062,51063,51064,51065,51066,51067,51068,51069,51070,51072,51073,51074,51075,51076,51077,51078,51079,51080,51081,51082,51083,51084,51085,51086,51087,51088,51089,51090,51091,51092,51093,51094,51095,51096,51097,51098,51099,51100,51101,51102,51103,51104,51105,51106,51107,51108,51109,51110,51111,51112,51113,51114,51115,51116,51117,51118,51119,51120,51121,51122,51123,51124,51125,51126,51127,51128,51129,51130,51131,51132,51133,51134,51135,51136,51137,51138,51139,51140,51141,51142,51143,51144,51145,51146,51147,51148,51149,51150,51151,51152,51153,51154,51155,51156,51157,51158,51159,51160,51161,51162,51163,51164,51165,51166,51167,51168,51169,51170,51171,51172,51173,51174,51175,51176,51177,51178,51179,51180,51181,51182,51183,51184,51185,51186,51187,51188,51189,51190,51191,51192,51193,51194,51195,51196,51197,51198,51264,51265,51266,51267,51268,51269,51270,51271,51272,51273,51274,51275,51276,51277,51278,51279,51280,51281,51282,51283,51284,51285,51286,51287,51288,51289,51290,51291,51292,51293,51294,51295,51296,51297,51298,51299,51300,51301,51302,51303,51304,51305,51306,51307,51308,51309,51310,51311,51312,51313,51314,51315,51316,51317,51318,51319,51320,51321,51322,51323,51324,51325,51326,51328,51329,51330,51331,51332,51333,51334,51335,51336,51337,51338,51339,51340,51341,51342,51343,51344,51345,51346,51347,51348,51349,51350,51351,51352,51353,51354,51355,51356,51357,51358,51359,51360,51361,51362,51363,51364,51365,51366,51367,51368,51369,51370,51371,51372,51373,51374,51375,51376,51377,51378,51379,51380,51381,51382,51383,51384,51385,51386,51387,51388,51389,51390,51391,51392,51393,51394,51395,51396,51397,51398,51399,51400,51401,51402,51403,51404,51405,51406,51407,51408,51409,51410,51411,51412,51413,51414,51415,51416,51417,51418,51419,51420,51421,51422,51423,51424,51425,51426,51427,51428,51429,51430,51431,51432,51433,51434,51435,51436,51437,51438,51439,51440,51441,51442,51443,51444,51445,51446,51447,51448,51449,51450,51451,51452,51453,51454,51520,51521,51522,51523,51524,51525,51526,51527,51528,51529,51530,51531,51532,51533,51534,51535,51536,51537,51538,51539,51540,51541,51542,51543,51544,51545,51546,51547,51548,51549,51550,51551,51552,51553,51554,51555,51556,51557,51558,51559,51560,51561,51562,51563,51564,51565,51566,51567,51568,51569,51570,51571,51572,51573,51574,51575,51576,51577,51578,51579,51580,51581,51582,51584,51585,51586,51587,51588,51589,51590,51591,51592,51593,51594,51595,51596,51597,51598,51599,51600,51601,51602,51603,51604,51605,51606,51607,51608,51609,51610,51611,51612,51613,51614,51615,51616,51617,51618,51619,51620,51621,51622,51623,51624,51625,51626,51627,51628,51629,51630,51631,51632,51633,51634,51635,51636,51637,51638,51639,51640,51641,51642,51643,51644,51645,51646,51647,51648,51649,51650,51651,51652,51653,51654,51655,51656,51657,51658,51659,51660,51661,51662,51663,51664,51665,51666,51667,51668,51669,51670,51671,51672,51673,51674,51675,51676,51677,51678,51679,51680,51681,51682,51683,51684,51685,51686,51687,51688,51689,51690,51691,51692,51693,51694,51695,51696,51697,51698,51699,51700,51701,51702,51703,51704,51705,51706,51707,51708,51709,51710,51776,51777,51778,51779,51780,51781,51782,51783,51784,51785,51786,51787,51788,51789,51790,51791,51792,51793,51794,51795,51796,51797,51798,51799,51800,51801,51802,51803,51804,51805,51806,51807,51808,51809,51810,51811,51812,51813,51814,51815,51816,51817,51818,51819,51820,51821,51822,51823,51824,51825,51826,51827,51828,51829,51830,51831,51832,51833,51834,51835,51836,51837,51838,51840,51841,51842,51843,51844,51845,51846,51847,51848,51849,51850,51851,51852,51853,51854,51855,51856,51857,51858,51859,51860,51861,51862,51863,51864,51865,51866,51867,51868,51869,51870,51871,51872,51873,51874,51875,51876,51877,51878,51879,51880,51881,51882,51883,51884,51885,51886,51887,51888,51889,51890,51891,51892,51893,51894,51895,51896,51897,51898,51899,51900,51901,51902,51903,51904,51905,51906,51907,51908,51909,51910,51911,51912,51913,51914,51915,51916,51917,51918,51919,51920,51921,51922,51923,51924,51925,51926,51927,51928,51929,51930,51931,51932,51933,51934,51935,51936,51937,51938,51939,51940,51941,51942,51943,51944,51945,51946,51947,51948,51949,51950,51951,51952,51953,51954,51955,51956,51957,51958,51959,51960,51961,51962,51963,51964,51965,51966,52032,52033,52034,52035,52036,52037,52038,52039,52040,52041,52042,52043,52044,52045,52046,52047,52048,52049,52050,52051,52052,52053,52054,52055,52056,52057,52058,52059,52060,52061,52062,52063,52064,52065,52066,52067,52068,52069,52070,52071,52072,52073,52074,52075,52076,52077,52078,52079,52080,52081,52082,52083,52084,52085,52086,52087,52088,52089,52090,52091,52092,52093,52094,52096,52097,52098,52099,52100,52101,52102,52103,52104,52105,52106,52107,52108,52109,52110,52111,52112,52113,52114,52115,52116,52117,52118,52119,52120,52121,52122,52123,52124,52125,52126,52127,52128,52129,52130,52131,52132,52133,52134,52135,52136,52137,52138,52139,52140,52141,52142,52143,52144,52145,52146,52147,52148,52149,52150,52151,52152,52153,52154,52155,52156,52157,52158,52159,52160,52161,52162,52163,52164,52165,52166,52167,52168,52169,52170,52171,52172,52173,52174,52175,52176,52177,52178,52179,52180,52181,52182,52183,52184,52185,52186,52187,52188,52189,52190,52191,52192,52193,52194,52195,52196,52197,52198,52199,52200,52201,52202,52203,52204,52205,52206,52207,52208,52209,52210,52211,52212,52213,52214,52215,52216,52217,52218,52219,52220,52221,52222,52288,52289,52290,52291,52292,52293,52294,52295,52296,52297,52298,52299,52300,52301,52302,52303,52304,52305,52306,52307,52308,52309,52310,52311,52312,52313,52314,52315,52316,52317,52318,52319,52320,52321,52322,52323,52324,52325,52326,52327,52328,52329,52330,52331,52332,52333,52334,52335,52336,52337,52338,52339,52340,52341,52342,52343,52344,52345,52346,52347,52348,52349,52350,52352,52353,52354,52355,52356,52357,52358,52359,52360,52361,52362,52363,52364,52365,52366,52367,52368,52369,52370,52371,52372,52373,52374,52375,52376,52377,52378,52379,52380,52381,52382,52383,52384,52385,52386,52387,52388,52389,52390,52391,52392,52393,52394,52395,52396,52397,52398,52399,52400,52401,52402,52403,52404,52405,52406,52407,52408,52409,52410,52411,52412,52413,52414,52415,52416,52417,52418,52419,52420,52421,52422,52423,52424,52425,52426,52427,52428,52429,52430,52431,52432,52433,52434,52435,52436,52437,52438,52439,52440,52441,52442,52443,52444,52445,52446,52447,52448,52449,52450,52451,52452,52453,52454,52455,52456,52457,52458,52459,52460,52461,52462,52463,52464,52465,52466,52467,52468,52469,52470,52471,52472,52473,52474,52475,52476,52477,52478,52544,52545,52546,52547,52548,52549,52550,52551,52552,52553,52554,52555,52556,52557,52558,52559,52560,52561,52562,52563,52564,52565,52566,52567,52568,52569,52570,52571,52572,52573,52574,52575,52576,52577,52578,52579,52580,52581,52582,52583,52584,52585,52586,52587,52588,52589,52590,52591,52592,52593,52594,52595,52596,52597,52598,52599,52600,52601,52602,52603,52604,52605,52606,52608,52609,52610,52611,52612,52613,52614,52615,52616,52617,52618,52619,52620,52621,52622,52623,52624,52625,52626,52627,52628,52629,52630,52631,52632,52633,52634,52635,52636,52637,52638,52639,52640,52641,52642,52643,52644,52645,52646,52647,52648,52649,52650,52651,52652,52653,52654,52655,52656,52657,52658,52659,52660,52661,52662,52663,52664,52665,52666,52667,52668,52669,52670,52671,52672,52673,52674,52675,52676,52677,52678,52679,52680,52681,52682,52683,52684,52685,52686,52687,52688,52689,52690,52691,52692,52693,52694,52695,52696,52697,52698,52699,52700,52701,52702,52703,52704,52705,52706,52707,52708,52709,52710,52711,52712,52713,52714,52715,52716,52717,52718,52719,52720,52721,52722,52723,52724,52725,52726,52727,52728,52729,52730,52731,52732,52733,52734,52800,52801,52802,52803,52804,52805,52806,52807,52808,52809,52810,52811,52812,52813,52814,52815,52816,52817,52818,52819,52820,52821,52822,52823,52824,52825,52826,52827,52828,52829,52830,52831,52832,52833,52834,52835,52836,52837,52838,52839,52840,52841,52842,52843,52844,52845,52846,52847,52848,52849,52850,52851,52852,52853,52854,52855,52856,52857,52858,52859,52860,52861,52862,52864,52865,52866,52867,52868,52869,52870,52871,52872,52873,52874,52875,52876,52877,52878,52879,52880,52881,52882,52883,52884,52885,52886,52887,52888,52889,52890,52891,52892,52893,52894,52895,52896,52897,52898,52899,52900,52901,52902,52903,52904,52905,52906,52907,52908,52909,52910,52911,52912,52913,52914,52915,52916,52917,52918,52919,52920,52921,52922,52923,52924,52925,52926,52927,52928,52929,52930,52931,52932,52933,52934,52935,52936,52937,52938,52939,52940,52941,52942,52943,52944,52945,52946,52947,52948,52949,52950,52951,52952,52953,52954,52955,52956,52957,52958,52959,52960,52961,52962,52963,52964,52965,52966,52967,52968,52969,52970,52971,52972,52973,52974,52975,52976,52977,52978,52979,52980,52981,52982,52983,52984,52985,52986,52987,52988,52989,52990,53056,53057,53058,53059,53060,53061,53062,53063,53064,53065,53066,53067,53068,53069,53070,53071,53072,53073,53074,53075,53076,53077,53078,53079,53080,53081,53082,53083,53084,53085,53086,53087,53088,53089,53090,53091,53092,53093,53094,53095,53096,53097,53098,53099,53100,53101,53102,53103,53104,53105,53106,53107,53108,53109,53110,53111,53112,53113,53114,53115,53116,53117,53118,53120,53121,53122,53123,53124,53125,53126,53127,53128,53129,53130,53131,53132,53133,53134,53135,53136,53137,53138,53139,53140,53141,53142,53143,53144,53145,53146,53147,53148,53149,53150,53151,53152,53153,53154,53155,53156,53157,53158,53159,53160,53161,53162,53163,53164,53165,53166,53167,53168,53169,53170,53171,53172,53173,53174,53175,53176,53177,53178,53179,53180,53181,53182,53183,53184,53185,53186,53187,53188,53189,53190,53191,53192,53193,53194,53195,53196,53197,53198,53199,53200,53201,53202,53203,53204,53205,53206,53207,53208,53209,53210,53211,53212,53213,53214,53215,53216,53217,53218,53219,53220,53221,53222,53223,53224,53225,53226,53227,53228,53229,53230,53231,53232,53233,53234,53235,53236,53237,53238,53239,53240,53241,53242,53243,53244,53245,53246,53312,53313,53314,53315,53316,53317,53318,53319,53320,53321,53322,53323,53324,53325,53326,53327,53328,53329,53330,53331,53332,53333,53334,53335,53336,53337,53338,53339,53340,53341,53342,53343,53344,53345,53346,53347,53348,53349,53350,53351,53352,53353,53354,53355,53356,53357,53358,53359,53360,53361,53362,53363,53364,53365,53366,53367,53368,53369,53370,53371,53372,53373,53374,53376,53377,53378,53379,53380,53381,53382,53383,53384,53385,53386,53387,53388,53389,53390,53391,53392,53393,53394,53395,53396,53397,53398,53399,53400,53401,53402,53403,53404,53405,53406,53407,53408,53409,53410,53411,53412,53413,53414,53415,53416,53417,53418,53419,53420,53421,53422,53423,53424,53425,53426,53427,53428,53429,53430,53431,53432,53433,53434,53435,53436,53437,53438,53439,53440,53441,53442,53443,53444,53445,53446,53447,53448,53449,53450,53451,53452,53453,53454,53455,53456,53457,53458,53459,53460,53461,53462,53463,53464,53465,53466,53467,53468,53469,53470,53471,53472,53473,53474,53475,53476,53477,53478,53479,53480,53481,53482,53483,53484,53485,53486,53487,53488,53489,53490,53491,53492,53493,53494,53495,53496,53497,53498,53499,53500,53501,53502,53568,53569,53570,53571,53572,53573,53574,53575,53576,53577,53578,53579,53580,53581,53582,53583,53584,53585,53586,53587,53588,53589,53590,53591,53592,53593,53594,53595,53596,53597,53598,53599,53600,53601,53602,53603,53604,53605,53606,53607,53608,53609,53610,53611,53612,53613,53614,53615,53616,53617,53618,53619,53620,53621,53622,53623,53624,53625,53626,53627,53628,53629,53630,53632,53633,53634,53635,53636,53637,53638,53639,53640,53641,53642,53643,53644,53645,53646,53647,53648,53649,53650,53651,53652,53653,53654,53655,53656,53657,53658,53659,53660,53661,53662,53663,53664,53665,53666,53667,53668,53669,53670,53671,53672,53673,53674,53675,53676,53677,53678,53679,53680,53681,53682,53683,53684,53685,53686,53687,53688,53689,53690,53691,53692,53693,53694,53695,53696,53697,53698,53699,53700,53701,53702,53703,53704,53705,53706,53707,53708,53709,53710,53711,53712,53713,53714,53715,53716,53717,53718,53719,53720,53721,53722,53723,53724,53725,53726,53727,53728,53729,53730,53731,53732,53733,53734,53735,53736,53737,53738,53739,53740,53741,53742,53743,53744,53745,53746,53747,53748,53749,53750,53751,53752,53753,53754,53755,53756,53757,53758,53824,53825,53826,53827,53828,53829,53830,53831,53832,53833,53834,53835,53836,53837,53838,53839,53840,53841,53842,53843,53844,53845,53846,53847,53848,53849,53850,53851,53852,53853,53854,53855,53856,53857,53858,53859,53860,53861,53862,53863,53864,53865,53866,53867,53868,53869,53870,53871,53872,53873,53874,53875,53876,53877,53878,53879,53880,53881,53882,53883,53884,53885,53886,53888,53889,53890,53891,53892,53893,53894,53895,53896,53897,53898,53899,53900,53901,53902,53903,53904,53905,53906,53907,53908,53909,53910,53911,53912,53913,53914,53915,53916,53917,53918,53919,53920,53921,53922,53923,53924,53925,53926,53927,53928,53929,53930,53931,53932,53933,53934,53935,53936,53937,53938,53939,53940,53941,53942,53943,53944,53945,53946,53947,53948,53949,53950,53951,53952,53953,53954,53955,53956,53957,53958,53959,53960,53961,53962,53963,53964,53965,53966,53967,53968,53969,53970,53971,53972,53973,53974,53975,53976,53977,53978,53979,53980,53981,53982,53983,53984,53985,53986,53987,53988,53989,53990,53991,53992,53993,53994,53995,53996,53997,53998,53999,54000,54001,54002,54003,54004,54005,54006,54007,54008,54009,54010,54011,54012,54013,54014,54080,54081,54082,54083,54084,54085,54086,54087,54088,54089,54090,54091,54092,54093,54094,54095,54096,54097,54098,54099,54100,54101,54102,54103,54104,54105,54106,54107,54108,54109,54110,54111,54112,54113,54114,54115,54116,54117,54118,54119,54120,54121,54122,54123,54124,54125,54126,54127,54128,54129,54130,54131,54132,54133,54134,54135,54136,54137,54138,54139,54140,54141,54142,54144,54145,54146,54147,54148,54149,54150,54151,54152,54153,54154,54155,54156,54157,54158,54159,54160,54161,54162,54163,54164,54165,54166,54167,54168,54169,54170,54171,54172,54173,54174,54175,54176,54177,54178,54179,54180,54181,54182,54183,54184,54185,54186,54187,54188,54189,54190,54191,54192,54193,54194,54195,54196,54197,54198,54199,54200,54201,54202,54203,54204,54205,54206,54207,54208,54209,54210,54211,54212,54213,54214,54215,54216,54217,54218,54219,54220,54221,54222,54223,54224,54225,54226,54227,54228,54229,54230,54231,54232,54233,54234,54235,54236,54237,54238,54239,54240,54241,54242,54243,54244,54245,54246,54247,54248,54249,54250,54251,54252,54253,54254,54255,54256,54257,54258,54259,54260,54261,54262,54263,54264,54265,54266,54267,54268,54269,54270,54336,54337,54338,54339,54340,54341,54342,54343,54344,54345,54346,54347,54348,54349,54350,54351,54352,54353,54354,54355,54356,54357,54358,54359,54360,54361,54362,54363,54364,54365,54366,54367,54368,54369,54370,54371,54372,54373,54374,54375,54376,54377,54378,54379,54380,54381,54382,54383,54384,54385,54386,54387,54388,54389,54390,54391,54392,54393,54394,54395,54396,54397,54398,54400,54401,54402,54403,54404,54405,54406,54407,54408,54409,54410,54411,54412,54413,54414,54415,54416,54417,54418,54419,54420,54421,54422,54423,54424,54425,54426,54427,54428,54429,54430,54431,54432,54433,54434,54435,54436,54437,54438,54439,54440,54441,54442,54443,54444,54445,54446,54447,54448,54449,54450,54451,54452,54453,54454,54455,54456,54457,54458,54459,54460,54461,54462,54463,54464,54465,54466,54467,54468,54469,54470,54471,54472,54473,54474,54475,54476,54477,54478,54479,54480,54481,54482,54483,54484,54485,54486,54487,54488,54489,54490,54491,54492,54493,54494,54495,54496,54497,54498,54499,54500,54501,54502,54503,54504,54505,54506,54507,54508,54509,54510,54511,54512,54513,54514,54515,54516,54517,54518,54519,54520,54521,54522,54523,54524,54525,54526,54592,54593,54594,54595,54596,54597,54598,54599,54600,54601,54602,54603,54604,54605,54606,54607,54608,54609,54610,54611,54612,54613,54614,54615,54616,54617,54618,54619,54620,54621,54622,54623,54624,54625,54626,54627,54628,54629,54630,54631,54632,54633,54634,54635,54636,54637,54638,54639,54640,54641,54642,54643,54644,54645,54646,54647,54648,54649,54650,54651,54652,54653,54654,54656,54657,54658,54659,54660,54661,54662,54663,54664,54665,54666,54667,54668,54669,54670,54671,54672,54673,54674,54675,54676,54677,54678,54679,54680,54681,54682,54683,54684,54685,54686,54687,54688,54689,54690,54691,54692,54693,54694,54695,54696,54697,54698,54699,54700,54701,54702,54703,54704,54705,54706,54707,54708,54709,54710,54711,54712,54713,54714,54715,54716,54717,54718,54719,54720,54721,54722,54723,54724,54725,54726,54727,54728,54729,54730,54731,54732,54733,54734,54735,54736,54737,54738,54739,54740,54741,54742,54743,54744,54745,54746,54747,54748,54749,54750,54751,54752,54753,54754,54755,54756,54757,54758,54759,54760,54761,54762,54763,54764,54765,54766,54767,54768,54769,54770,54771,54772,54773,54774,54775,54776,54777,54778,54779,54780,54781,54782,54848,54849,54850,54851,54852,54853,54854,54855,54856,54857,54858,54859,54860,54861,54862,54863,54864,54865,54866,54867,54868,54869,54870,54871,54872,54873,54874,54875,54876,54877,54878,54879,54880,54881,54882,54883,54884,54885,54886,54887,54888,54889,54890,54891,54892,54893,54894,54895,54896,54897,54898,54899,54900,54901,54902,54903,54904,54905,54906,54907,54908,54909,54910,54912,54913,54914,54915,54916,54917,54918,54919,54920,54921,54922,54923,54924,54925,54926,54927,54928,54929,54930,54931,54932,54933,54934,54935,54936,54937,54938,54939,54940,54941,54942,54943,54944,54945,54946,54947,54948,54949,54950,54951,54952,54953,54954,54955,54956,54957,54958,54959,54960,54961,54962,54963,54964,54965,54966,54967,54968,54969,54970,54971,54972,54973,54974,54975,54976,54977,54978,54979,54980,54981,54982,54983,54984,54985,54986,54987,54988,54989,54990,54991,54992,54993,54994,54995,54996,54997,54998,54999,55000,55001,55002,55003,55004,55005,55006,55007,55008,55009,55010,55011,55012,55013,55014,55015,55016,55017,55018,55019,55020,55021,55022,55023,55024,55025,55026,55027,55028,55029,55030,55031,55032,55033,55034,55035,55036,55037,55038,55104,55105,55106,55107,55108,55109,55110,55111,55112,55113,55114,55115,55116,55117,55118,55119,55120,55121,55122,55123,55124,55125,55126,55127,55128,55129,55130,55131,55132,55133,55134,55135,55136,55137,55138,55139,55140,55141,55142,55143,55144,55145,55146,55147,55148,55149,55150,55151,55152,55153,55154,55155,55156,55157,55158,55159,55160,55161,55162,55163,55164,55165,55166,55168,55169,55170,55171,55172,55173,55174,55175,55176,55177,55178,55179,55180,55181,55182,55183,55184,55185,55186,55187,55188,55189,55190,55191,55192,55193,55194,55195,55196,55197,55198,55199,55200,55201,55202,55203,55204,55205,55206,55207,55208,55209,55210,55211,55212,55213,55214,55215,55216,55217,55218,55219,55220,55221,55222,55223,55224,55225,55226,55227,55228,55229,55230,55231,55232,55233,55234,55235,55236,55237,55238,55239,55240,55241,55242,55243,55244,55245,55246,55247,55248,55249,55250,55251,55252,55253,55254,55255,55256,55257,55258,55259,55260,55261,55262,55263,55264,55265,55266,55267,55268,55269,55270,55271,55272,55273,55274,55275,55276,55277,55278,55279,55280,55281,55282,55283,55284,55285,55286,55287,55288,55289,55290,55291,55292,55293,55294,55360,55361,55362,55363,55364,55365,55366,55367,55368,55369,55370,55371,55372,55373,55374,55375,55376,55377,55378,55379,55380,55381,55382,55383,55384,55385,55386,55387,55388,55389,55390,55391,55392,55393,55394,55395,55396,55397,55398,55399,55400,55401,55402,55403,55404,55405,55406,55407,55408,55409,55410,55411,55412,55413,55414,55415,55416,55417,55418,55419,55420,55421,55422,55424,55425,55426,55427,55428,55429,55430,55431,55432,55433,55434,55435,55436,55437,55438,55439,55440,55441,55442,55443,55444,55445,55446,55447,55448,55449,55450,55451,55452,55453,55454,55455,55456,55457,55458,55459,55460,55461,55462,55463,55464,55465,55466,55467,55468,55469,55470,55471,55472,55473,55474,55475,55476,55477,55478,55479,55480,55481,55482,55483,55484,55485,55486,55487,55488,55489,55490,55491,55492,55493,55494,55495,55496,55497,55498,55499,55500,55501,55502,55503,55504,55505,55506,55507,55508,55509,55510,55511,55512,55513,55514,55515,55516,55517,55518,55519,55520,55521,55522,55523,55524,55525,55526,55527,55528,55529,55530,55531,55532,55533,55534,55535,55536,55537,55538,55539,55540,55541,55542,55543,55544,55545,55546,55547,55548,55549,55550,55616,55617,55618,55619,55620,55621,55622,55623,55624,55625,55626,55627,55628,55629,55630,55631,55632,55633,55634,55635,55636,55637,55638,55639,55640,55641,55642,55643,55644,55645,55646,55647,55648,55649,55650,55651,55652,55653,55654,55655,55656,55657,55658,55659,55660,55661,55662,55663,55664,55665,55666,55667,55668,55669,55670,55671,55672,55673,55674,55675,55676,55677,55678,55680,55681,55682,55683,55684,55685,55686,55687,55688,55689,55690,55691,55692,55693,55694,55695,55696,55697,55698,55699,55700,55701,55702,55703,55704,55705,55706,55707,55708,55709,55710,55711,55712,55713,55714,55715,55716,55717,55718,55719,55720,55721,55722,55723,55724,55725,55726,55727,55728,55729,55730,55731,55732,55733,55734,55735,55736,55737,55738,55739,55740,55741,55742,55743,55744,55745,55746,55747,55748,55749,55750,55751,55752,55753,55754,55755,55756,55757,55758,55759,55760,55761,55762,55763,55764,55765,55766,55767,55768,55769,55770,55771,55772,55773,55774,55775,55776,55777,55778,55779,55780,55781,55782,55783,55784,55785,55786,55787,55788,55789,55790,55791,55792,55793,55794,55795,55796,55797,55798,55799,55800,55801,55802,55803,55804,55805,55806,55872,55873,55874,55875,55876,55877,55878,55879,55880,55881,55882,55883,55884,55885,55886,55887,55888,55889,55890,55891,55892,55893,55894,55895,55896,55897,55898,55899,55900,55901,55902,55903,55904,55905,55906,55907,55908,55909,55910,55911,55912,55913,55914,55915,55916,55917,55918,55919,55920,55921,55922,55923,55924,55925,55926,55927,55928,55929,55930,55931,55932,55933,55934,55936,55937,55938,55939,55940,55941,55942,55943,55944,55945,55946,55947,55948,55949,55950,55951,55952,55953,55954,55955,55956,55957,55958,55959,55960,55961,55962,55963,55964,55965,55966,55967,55968,55969,55970,55971,55972,55973,55974,55975,55976,55977,55978,55979,55980,55981,55982,55983,55984,55985,55986,55987,55988,55989,55990,55991,55992,55993,55994,55995,55996,55997,55998,55999,56000,56001,56002,56003,56004,56005,56006,56007,56008,56009,56010,56011,56012,56013,56014,56015,56016,56017,56018,56019,56020,56021,56022,56023,56024,56025,56026,56027,56028,56029,56030,56031,56032,56033,56034,56035,56036,56037,56038,56039,56040,56041,56042,56043,56044,56045,56046,56047,56048,56049,56050,56051,56052,56053,56054,56055,56056,56057,56058,56059,56060,56061,56062,56128,56129,56130,56131,56132,56133,56134,56135,56136,56137,56138,56139,56140,56141,56142,56143,56144,56145,56146,56147,56148,56149,56150,56151,56152,56153,56154,56155,56156,56157,56158,56159,56160,56161,56162,56163,56164,56165,56166,56167,56168,56169,56170,56171,56172,56173,56174,56175,56176,56177,56178,56179,56180,56181,56182,56183,56184,56185,56186,56187,56188,56189,56190,56192,56193,56194,56195,56196,56197,56198,56199,56200,56201,56202,56203,56204,56205,56206,56207,56208,56209,56210,56211,56212,56213,56214,56215,56216,56217,56218,56219,56220,56221,56222,56223,56224,56225,56226,56227,56228,56229,56230,56231,56232,56233,56234,56235,56236,56237,56238,56239,56240,56241,56242,56243,56244,56245,56246,56247,56248,56249,56250,56251,56252,56253,56254,56255,56256,56257,56258,56259,56260,56261,56262,56263,56264,56265,56266,56267,56268,56269,56270,56271,56272,56273,56274,56275,56276,56277,56278,56279,56280,56281,56282,56283,56284,56285,56286,56287,56288,56289,56290,56291,56292,56293,56294,56295,56296,56297,56298,56299,56300,56301,56302,56303,56304,56305,56306,56307,56308,56309,56310,56311,56312,56313,56314,56315,56316,56317,56318,56384,56385,56386,56387,56388,56389,56390,56391,56392,56393,56394,56395,56396,56397,56398,56399,56400,56401,56402,56403,56404,56405,56406,56407,56408,56409,56410,56411,56412,56413,56414,56415,56416,56417,56418,56419,56420,56421,56422,56423,56424,56425,56426,56427,56428,56429,56430,56431,56432,56433,56434,56435,56436,56437,56438,56439,56440,56441,56442,56443,56444,56445,56446,56448,56449,56450,56451,56452,56453,56454,56455,56456,56457,56458,56459,56460,56461,56462,56463,56464,56465,56466,56467,56468,56469,56470,56471,56472,56473,56474,56475,56476,56477,56478,56479,56480,56481,56482,56483,56484,56485,56486,56487,56488,56489,56490,56491,56492,56493,56494,56495,56496,56497,56498,56499,56500,56501,56502,56503,56504,56505,56506,56507,56508,56509,56510,56511,56512,56513,56514,56515,56516,56517,56518,56519,56520,56521,56522,56523,56524,56525,56526,56527,56528,56529,56530,56531,56532,56533,56534,56535,56536,56537,56538,56539,56540,56541,56542,56543,56544,56545,56546,56547,56548,56549,56550,56551,56552,56553,56554,56555,56556,56557,56558,56559,56560,56561,56562,56563,56564,56565,56566,56567,56568,56569,56570,56571,56572,56573,56574,56640,56641,56642,56643,56644,56645,56646,56647,56648,56649,56650,56651,56652,56653,56654,56655,56656,56657,56658,56659,56660,56661,56662,56663,56664,56665,56666,56667,56668,56669,56670,56671,56672,56673,56674,56675,56676,56677,56678,56679,56680,56681,56682,56683,56684,56685,56686,56687,56688,56689,56690,56691,56692,56693,56694,56695,56696,56697,56698,56699,56700,56701,56702,56704,56705,56706,56707,56708,56709,56710,56711,56712,56713,56714,56715,56716,56717,56718,56719,56720,56721,56722,56723,56724,56725,56726,56727,56728,56729,56730,56731,56732,56733,56734,56735,56736,56737,56738,56739,56740,56741,56742,56743,56744,56745,56746,56747,56748,56749,56750,56751,56752,56753,56754,56755,56756,56757,56758,56759,56760,56761,56762,56763,56764,56765,56766,56767,56768,56769,56770,56771,56772,56773,56774,56775,56776,56777,56778,56779,56780,56781,56782,56783,56784,56785,56786,56787,56788,56789,56790,56791,56792,56793,56794,56795,56796,56797,56798,56799,56800,56801,56802,56803,56804,56805,56806,56807,56808,56809,56810,56811,56812,56813,56814,56815,56816,56817,56818,56819,56820,56821,56822,56823,56824,56825,56826,56827,56828,56829,56830,56896,56897,56898,56899,56900,56901,56902,56903,56904,56905,56906,56907,56908,56909,56910,56911,56912,56913,56914,56915,56916,56917,56918,56919,56920,56921,56922,56923,56924,56925,56926,56927,56928,56929,56930,56931,56932,56933,56934,56935,56936,56937,56938,56939,56940,56941,56942,56943,56944,56945,56946,56947,56948,56949,56950,56951,56952,56953,56954,56955,56956,56957,56958,56960,56961,56962,56963,56964,56965,56966,56967,56968,56969,56970,56971,56972,56973,56974,56975,56976,56977,56978,56979,56980,56981,56982,56983,56984,56985,56986,56987,56988,56989,56990,56991,56992,56993,56994,56995,56996,56997,56998,56999,57000,57001,57002,57003,57004,57005,57006,57007,57008,57009,57010,57011,57012,57013,57014,57015,57016,57017,57018,57019,57020,57021,57022,57023,57024,57025,57026,57027,57028,57029,57030,57031,57032,57033,57034,57035,57036,57037,57038,57039,57040,57041,57042,57043,57044,57045,57046,57047,57048,57049,57050,57051,57052,57053,57054,57055,57056,57057,57058,57059,57060,57061,57062,57063,57064,57065,57066,57067,57068,57069,57070,57071,57072,57073,57074,57075,57076,57077,57078,57079,57080,57081,57082,57083,57084,57085,57086,57152,57153,57154,57155,57156,57157,57158,57159,57160,57161,57162,57163,57164,57165,57166,57167,57168,57169,57170,57171,57172,57173,57174,57175,57176,57177,57178,57179,57180,57181,57182,57183,57184,57185,57186,57187,57188,57189,57190,57191,57192,57193,57194,57195,57196,57197,57198,57199,57200,57201,57202,57203,57204,57205,57206,57207,57208,57209,57210,57211,57212,57213,57214,57216,57217,57218,57219,57220,57221,57222,57223,57224,57225,57226,57227,57228,57229,57230,57231,57232,57233,57234,57235,57236,57237,57238,57239,57240,57241,57242,57243,57244,57245,57246,57247,57248,57249,57250,57251,57252,57253,57254,57255,57256,57257,57258,57259,57260,57261,57262,57263,57264,57265,57266,57267,57268,57269,57270,57271,57272,57273,57274,57275,57276,57277,57278,57279,57280,57281,57282,57283,57284,57285,57286,57287,57288,57289,57290,57291,57292,57293,57294,57295,57296,57297,57298,57299,57300,57301,57302,57303,57304,57305,57306,57307,57308,57309,57310,57311,57312,57313,57314,57315,57316,57317,57318,57319,57320,57321,57322,57323,57324,57325,57326,57327,57328,57329,57330,57331,57332,57333,57334,57335,57336,57337,57338,57339,57340,57341,57342,57408,57409,57410,57411,57412,57413,57414,57415,57416,57417,57418,57419,57420,57421,57422,57423,57424,57425,57426,57427,57428,57429,57430,57431,57432,57433,57434,57435,57436,57437,57438,57439,57440,57441,57442,57443,57444,57445,57446,57447,57448,57449,57450,57451,57452,57453,57454,57455,57456,57457,57458,57459,57460,57461,57462,57463,57464,57465,57466,57467,57468,57469,57470,57472,57473,57474,57475,57476,57477,57478,57479,57480,57481,57482,57483,57484,57485,57486,57487,57488,57489,57490,57491,57492,57493,57494,57495,57496,57497,57498,57499,57500,57501,57502,57503,57504,57505,57506,57507,57508,57509,57510,57511,57512,57513,57514,57515,57516,57517,57518,57519,57520,57521,57522,57523,57524,57525,57526,57527,57528,57529,57530,57531,57532,57533,57534,57535,57536,57537,57538,57539,57540,57541,57542,57543,57544,57545,57546,57547,57548,57549,57550,57551,57552,57553,57554,57555,57556,57557,57558,57559,57560,57561,57562,57563,57564,57565,57566,57567,57568,57569,57570,57571,57572,57573,57574,57575,57576,57577,57578,57579,57580,57581,57582,57583,57584,57585,57586,57587,57588,57589,57590,57591,57592,57593,57594,57595,57596,57597,57598,57664,57665,57666,57667,57668,57669,57670,57671,57672,57673,57674,57675,57676,57677,57678,57679,57680,57681,57682,57683,57684,57685,57686,57687,57688,57689,57690,57691,57692,57693,57694,57695,57696,57697,57698,57699,57700,57701,57702,57703,57704,57705,57706,57707,57708,57709,57710,57711,57712,57713,57714,57715,57716,57717,57718,57719,57720,57721,57722,57723,57724,57725,57726,57728,57729,57730,57731,57732,57733,57734,57735,57736,57737,57738,57739,57740,57741,57742,57743,57744,57745,57746,57747,57748,57749,57750,57751,57752,57753,57754,57755,57756,57757,57758,57759,57760,57761,57762,57763,57764,57765,57766,57767,57768,57769,57770,57771,57772,57773,57774,57775,57776,57777,57778,57779,57780,57781,57782,57783,57784,57785,57786,57787,57788,57789,57790,57791,57792,57793,57794,57795,57796,57797,57798,57799,57800,57801,57802,57803,57804,57805,57806,57807,57808,57809,57810,57811,57812,57813,57814,57815,57816,57817,57818,57819,57820,57821,57822,57823,57824,57825,57826,57827,57828,57829,57830,57831,57832,57833,57834,57835,57836,57837,57838,57839,57840,57841,57842,57843,57844,57845,57846,57847,57848,57849,57850,57851,57852,57853,57854,57920,57921,57922,57923,57924,57925,57926,57927,57928,57929,57930,57931,57932,57933,57934,57935,57936,57937,57938,57939,57940,57941,57942,57943,57944,57945,57946,57947,57948,57949,57950,57951,57952,57953,57954,57955,57956,57957,57958,57959,57960,57961,57962,57963,57964,57965,57966,57967,57968,57969,57970,57971,57972,57973,57974,57975,57976,57977,57978,57979,57980,57981,57982,57984,57985,57986,57987,57988,57989,57990,57991,57992,57993,57994,57995,57996,57997,57998,57999,58000,58001,58002,58003,58004,58005,58006,58007,58008,58009,58010,58011,58012,58013,58014,58015,58016,58017,58018,58019,58020,58021,58022,58023,58024,58025,58026,58027,58028,58029,58030,58031,58032,58033,58034,58035,58036,58037,58038,58039,58040,58041,58042,58043,58044,58045,58046,58047,58048,58049,58050,58051,58052,58053,58054,58055,58056,58057,58058,58059,58060,58061,58062,58063,58064,58065,58066,58067,58068,58069,58070,58071,58072,58073,58074,58075,58076,58077,58078,58079,58080,58081,58082,58083,58084,58085,58086,58087,58088,58089,58090,58091,58092,58093,58094,58095,58096,58097,58098,58099,58100,58101,58102,58103,58104,58105,58106,58107,58108,58109,58110,58176,58177,58178,58179,58180,58181,58182,58183,58184,58185,58186,58187,58188,58189,58190,58191,58192,58193,58194,58195,58196,58197,58198,58199,58200,58201,58202,58203,58204,58205,58206,58207,58208,58209,58210,58211,58212,58213,58214,58215,58216,58217,58218,58219,58220,58221,58222,58223,58224,58225,58226,58227,58228,58229,58230,58231,58232,58233,58234,58235,58236,58237,58238,58240,58241,58242,58243,58244,58245,58246,58247,58248,58249,58250,58251,58252,58253,58254,58255,58256,58257,58258,58259,58260,58261,58262,58263,58264,58265,58266,58267,58268,58269,58270,58271,58272,58273,58274,58275,58276,58277,58278,58279,58280,58281,58282,58283,58284,58285,58286,58287,58288,58289,58290,58291,58292,58293,58294,58295,58296,58297,58298,58299,58300,58301,58302,58303,58304,58305,58306,58307,58308,58309,58310,58311,58312,58313,58314,58315,58316,58317,58318,58319,58320,58321,58322,58323,58324,58325,58326,58327,58328,58329,58330,58331,58332,58333,58334,58335,58336,58337,58338,58339,58340,58341,58342,58343,58344,58345,58346,58347,58348,58349,58350,58351,58352,58353,58354,58355,58356,58357,58358,58359,58360,58361,58362,58363,58364,58365,58366,58432,58433,58434,58435,58436,58437,58438,58439,58440,58441,58442,58443,58444,58445,58446,58447,58448,58449,58450,58451,58452,58453,58454,58455,58456,58457,58458,58459,58460,58461,58462,58463,58464,58465,58466,58467,58468,58469,58470,58471,58472,58473,58474,58475,58476,58477,58478,58479,58480,58481,58482,58483,58484,58485,58486,58487,58488,58489,58490,58491,58492,58493,58494,58496,58497,58498,58499,58500,58501,58502,58503,58504,58505,58506,58507,58508,58509,58510,58511,58512,58513,58514,58515,58516,58517,58518,58519,58520,58521,58522,58523,58524,58525,58526,58527,58528,58529,58530,58531,58532,58533,58534,58535,58536,58537,58538,58539,58540,58541,58542,58543,58544,58545,58546,58547,58548,58549,58550,58551,58552,58553,58554,58555,58556,58557,58558,58559,58560,58561,58562,58563,58564,58565,58566,58567,58568,58569,58570,58571,58572,58573,58574,58575,58576,58577,58578,58579,58580,58581,58582,58583,58584,58585,58586,58587,58588,58589,58590,58591,58592,58593,58594,58595,58596,58597,58598,58599,58600,58601,58602,58603,58604,58605,58606,58607,58608,58609,58610,58611,58612,58613,58614,58615,58616,58617,58618,58619,58620,58621,58622,58688,58689,58690,58691,58692,58693,58694,58695,58696,58697,58698,58699,58700,58701,58702,58703,58704,58705,58706,58707,58708,58709,58710,58711,58712,58713,58714,58715,58716,58717,58718,58719,58720,58721,58722,58723,58724,58725,58726,58727,58728,58729,58730,58731,58732,58733,58734,58735,58736,58737,58738,58739,58740,58741,58742,58743,58744,58745,58746,58747,58748,58749,58750,58752,58753,58754,58755,58756,58757,58758,58759,58760,58761,58762,58763,58764,58765,58766,58767,58768,58769,58770,58771,58772,58773,58774,58775,58776,58777,58778,58779,58780,58781,58782,58783,58784,58785,58786,58787,58788,58789,58790,58791,58792,58793,58794,58795,58796,58797,58798,58799,58800,58801,58802,58803,58804,58805,58806,58807,58808,58809,58810,58811,58812,58813,58814,58815,58816,58817,58818,58819,58820,58821,58822,58823,58824,58825,58826,58827,58828,58829,58830,58831,58832,58833,58834,58835,58836,58837,58838,58839,58840,58841,58842,58843,58844,58845,58846,58847,58848,58849,58850,58851,58852,58853,58854,58855,58856,58857,58858,58859,58860,58861,58862,58863,58864,58865,58866,58867,58868,58869,58870,58871,58872,58873,58874,58875,58876,58877,58878,58944,58945,58946,58947,58948,58949,58950,58951,58952,58953,58954,58955,58956,58957,58958,58959,58960,58961,58962,58963,58964,58965,58966,58967,58968,58969,58970,58971,58972,58973,58974,58975,58976,58977,58978,58979,58980,58981,58982,58983,58984,58985,58986,58987,58988,58989,58990,58991,58992,58993,58994,58995,58996,58997,58998,58999,59000,59001,59002,59003,59004,59005,59006,59008,59009,59010,59011,59012,59013,59014,59015,59016,59017,59018,59019,59020,59021,59022,59023,59024,59025,59026,59027,59028,59029,59030,59031,59032,59033,59034,59035,59036,59037,59038,59039,59040,59041,59042,59043,59044,59045,59046,59047,59048,59049,59050,59051,59052,59053,59054,59055,59056,59057,59058,59059,59060,59061,59062,59063,59064,59065,59066,59067,59068,59069,59070,59071,59072,59073,59074,59075,59076,59077,59078,59079,59080,59081,59082,59083,59084,59085,59086,59087,59088,59089,59090,59091,59092,59093,59094,59095,59096,59097,59098,59099,59100,59101,59102,59103,59104,59105,59106,59107,59108,59109,59110,59111,59112,59113,59114,59115,59116,59117,59118,59119,59120,59121,59122,59123,59124,59125,59126,59127,59128,59129,59130,59131,59132,59133,59134,59200,59201,59202,59203,59204,59205,59206,59207,59208,59209,59210,59211,59212,59213,59214,59215,59216,59217,59218,59219,59220,59221,59222,59223,59224,59225,59226,59227,59228,59229,59230,59231,59232,59233,59234,59235,59236,59237,59238,59239,59240,59241,59242,59243,59244,59245,59246,59247,59248,59249,59250,59251,59252,59253,59254,59255,59256,59257,59258,59259,59260,59261,59262,59264,59265,59266,59267,59268,59269,59270,59271,59272,59273,59274,59275,59276,59277,59278,59279,59280,59281,59282,59283,59284,59285,59286,59287,59288,59289,59290,59291,59292,59293,59294,59295,59296,59297,59298,59299,59300,59301,59302,59303,59304,59305,59306,59307,59308,59309,59310,59311,59312,59313,59314,59315,59316,59317,59318,59319,59320,59321,59322,59323,59324,59325,59326,59327,59328,59329,59330,59331,59332,59333,59334,59335,59336,59337,59338,59339,59340,59341,59342,59343,59344,59345,59346,59347,59348,59349,59350,59351,59352,59353,59354,59355,59356,59357,59358,59359,59360,59361,59362,59363,59364,59365,59366,59367,59368,59369,59370,59371,59372,59373,59374,59375,59376,59377,59378,59379,59380,59381,59382,59383,59384,59385,59386,59387,59388,59389,59390,59456,59457,59458,59459,59460,59461,59462,59463,59464,59465,59466,59467,59468,59469,59470,59471,59472,59473,59474,59475,59476,59477,59478,59479,59480,59481,59482,59483,59484,59485,59486,59487,59488,59489,59490,59491,59492,59493,59494,59495,59496,59497,59498,59499,59500,59501,59502,59503,59504,59505,59506,59507,59508,59509,59510,59511,59512,59513,59514,59515,59516,59517,59518,59520,59521,59522,59523,59524,59525,59526,59527,59528,59529,59530,59531,59532,59533,59534,59535,59536,59537,59538,59539,59540,59541,59542,59543,59544,59545,59546,59547,59548,59549,59550,59551,59552,59553,59554,59555,59556,59557,59558,59559,59560,59561,59562,59563,59564,59565,59566,59567,59568,59569,59570,59571,59572,59573,59574,59575,59576,59577,59578,59579,59580,59581,59582,59583,59584,59585,59586,59587,59588,59589,59590,59591,59592,59593,59594,59595,59596,59597,59598,59599,59600,59601,59602,59603,59604,59605,59606,59607,59608,59609,59610,59611,59612,59613,59614,59615,59616,59617,59618,59619,59620,59621,59622,59623,59624,59625,59626,59627,59628,59629,59630,59631,59632,59633,59634,59635,59636,59637,59638,59639,59640,59641,59642,59643,59644,59645,59646,59712,59713,59714,59715,59716,59717,59718,59719,59720,59721,59722,59723,59724,59725,59726,59727,59728,59729,59730,59731,59732,59733,59734,59735,59736,59737,59738,59739,59740,59741,59742,59743,59744,59745,59746,59747,59748,59749,59750,59751,59752,59753,59754,59755,59756,59757,59758,59759,59760,59761,59762,59763,59764,59765,59766,59767,59768,59769,59770,59771,59772,59773,59774,59776,59777,59778,59779,59780,59781,59782,59783,59784,59785,59786,59787,59788,59789,59790,59791,59792,59793,59794,59795,59796,59797,59798,59799,59800,59801,59802,59803,59804,59805,59806,59807,59808,59809,59810,59811,59812,59813,59814,59815,59816,59817,59818,59819,59820,59821,59822,59823,59824,59825,59826,59827,59828,59829,59830,59831,59832,59833,59834,59835,59836,59837,59838,59839,59840,59841,59842,59843,59844,59845,59846,59847,59848,59849,59850,59851,59852,59853,59854,59855,59856,59857,59858,59859,59860,59861,59862,59863,59864,59865,59866,59867,59868,59869,59870,59871,59872,59873,59874,59875,59876,59877,59878,59879,59880,59881,59882,59883,59884,59885,59886,59887,59888,59889,59890,59891,59892,59893,59894,59895,59896,59897,59898,59899,59900,59901,59902,59968,59969,59970,59971,59972,59973,59974,59975,59976,59977,59978,59979,59980,59981,59982,59983,59984,59985,59986,59987,59988,59989,59990,59991,59992,59993,59994,59995,59996,59997,59998,59999,60000,60001,60002,60003,60004,60005,60006,60007,60008,60009,60010,60011,60012,60013,60014,60015,60016,60017,60018,60019,60020,60021,60022,60023,60024,60025,60026,60027,60028,60029,60030,60032,60033,60034,60035,60036,60037,60038,60039,60040,60041,60042,60043,60044,60045,60046,60047,60048,60049,60050,60051,60052,60053,60054,60055,60056,60057,60058,60059,60060,60061,60062,60063,60064,60065,60066,60067,60068,60069,60070,60071,60072,60073,60074,60075,60076,60