/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  FmAwActions.c
  
  (c) Simon Marlow 1990-92
  (c) Albert Graef 1994

  modified 2004,2005,2006,2007 by Bernhard R. Link (see Changelog)

  Action procedures for widgets in the application window
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <xfmconfig.h>

#include <assert.h>
#include <string.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw3d/Toggle.h>

#include "global.h"

#include "Am.h"
#include "execute.h"

/*---------------------------------------------------------------------------
  PRIVATE FUNCTIONS
---------------------------------------------------------------------------*/

static void awDragGrab(Boolean dropable) {
	Cursor c;

	if (dropable)
		c = curs[EXEC_CUR];
	else
		c = curs[NOENTRY_CUR];

	XGrabPointer(XtDisplay(aw.shell), XtWindow(aw.shell), True,
			EnterWindowMask | LeaveWindowMask | ButtonReleaseMask, 
			GrabModeAsync, GrabModeAsync, None, c, CurrentTime);
}

/* perform a click-drag, returning the widget that the user released the 
   button on */

static Boolean dragging_app = False;

static Widget drag(Widget w, unsigned int button,
		   Boolean *on_root_return)
{
  XEvent e;
  int x, y;
  Window child;

  awDragGrab(True);

  freeze = dragging_app = True;
  
  for (;;) {
    XtAppNextEvent(app_context, &e);
    switch (e.type) {
    case ButtonPress:        /* Ignore button presses */
      continue;
    case ButtonRelease:      /* Ignore button releases except 2 */
      if (e.xbutton.button != button)
	continue;
      break;
    default:
      XtDispatchEvent(&e);
      continue;
    }
    break;
  }

  XUngrabPointer(XtDisplay(aw.shell),CurrentTime);
  XtDispatchEvent(&e);
  freeze = dragging_app = False;
  
  /* Find out if the drag was released on the root window (child = None) */
  XTranslateCoordinates(XtDisplay(w),e.xbutton.window,e.xbutton.root,
			e.xbutton.x,e.xbutton.y,&x,&y,&child);
  if (child == None)
    *on_root_return = True;
  else
    *on_root_return = False;

  return XtWindowToWidget(XtDisplay(w),e.xbutton.window);
}

/*---------------------------------------------------------------------------
  PUBLIC FUNCTIONS
---------------------------------------------------------------------------*/

int findAppWidget(Widget w)
{
  Cardinal i;

  for (i=0; i<aw.n_apps; i++)
    if (aw.apps[i].toggle == w)
      return i;
  return -1;
}

/*---------------------------------------------------------------------------*/

void appPopup(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
  Display *dpy;
  Window root, child;
  int x, y, x_win, y_win;
  unsigned int mask;
  int i = findAppWidget(w);

  dpy = XtDisplay(aw.shell);

  XQueryPointer(dpy, XtWindow(w), &root, &child, &x, &y, 
		&x_win, &y_win, &mask);

  /* check whether icon was selected */
  if (child != None) {
    XTranslateCoordinates(dpy, XtWindow(w), child, x_win, y_win,
			  &x_win, &y_win, &child);
    if (child != None) w = XtWindowToWidget(dpy, child);
  }

  i = findAppWidget(w);
  if (i != -1) appSelect(w, event, params, num_params);

  if (i == -1) {
    struct stat stats;
    if (aw.n_selections == 0) {
      grayOut(app_popup_items[3]);
      grayOut(app_popup_items[4]);
      grayOut(app_popup_items[7]);
      grayOut(app_popup_items[10]);
    } else {
      if (aw.readonly) {
        grayOut(app_popup_items[3]);
        grayOut(app_popup_items[7]);
      } else {
        fillIn(app_popup_items[3]);
	fillIn(app_popup_items[7]);
      }
      fillIn(app_popup_items[4]);
      fillIn(app_popup_items[10]);
    }
    if (stat(resources.app_clip, &stats) || aw.readonly)
      grayOut(app_popup_items[5]);
    else
      fillIn(app_popup_items[5]);
    if (aw.readonly) {
      grayOut(app_popup_items[0]);
      grayOut(app_popup_items[1]);
    } else {
      fillIn(app_popup_items[0]);
      fillIn(app_popup_items[1]);
    }

    XQueryPointer(dpy, DefaultRootWindow(dpy), &root, &child, &x, &y, 
		  &x_win, &y_win, &mask);
  
    XtVaSetValues(app_popup_widget, XtNx, (XtArgVal) x, XtNy, (XtArgVal) y,
		  NULL);
  
    XtPopupSpringLoaded(app_popup_widget);
  } else {
    if (aw.readonly) {
	    grayOut(app_popup_items1[0]);
	    grayOut(app_popup_items1[2]);
	    grayOut(app_popup_items1[5]);
    } else {
	    fillIn(app_popup_items1[0]);
	    fillIn(app_popup_items1[2]);
	    fillIn(app_popup_items1[5]);
    }
    XQueryPointer(dpy, DefaultRootWindow(dpy), &root, &child, &x, &y, 
		  &x_win, &y_win, &mask);
  
    XtVaSetValues(app_popup_widget1, XtNx, (XtArgVal) x, XtNy, (XtArgVal) y,
		  NULL);
  
    XtPopupSpringLoaded(app_popup_widget1);
  }
}  

/*---------------------------------------------------------------------------*/

void appMaybeHighlight(Widget w, XEvent *event,
		       UNUSED(String *params), UNUSED(Cardinal *num_params))
{
  int i;

  if (dragging) {
    i = findAppWidget(w);
    if (*aw.apps[i].drop_action) {
      XtCallActionProc(w, "highlight", event, NULL, 0);
      drag_set_dropable(XtDisplay(w), True);
    } else {
      drag_set_dropable(XtDisplay(w), False);
    }
  } else if (dragging_app) {
    XtCallActionProc(w, "highlight", event, NULL, 0);
    awDragGrab(True);
  }
}

/*---------------------------------------------------------------------------*/

void runApp(Widget w, UNUSED(XEvent *event),
	    UNUSED(String *params), UNUSED(Cardinal *num_params))
{
  int i;
  char *directory;

  i = findAppWidget(w);
  directory = expanddirectory(aw.apps[i].directory);

  if (*aw.apps[i].push_action) {
    if (!strncmp(aw.apps[i].push_action, "EDIT ", 5))
      doEdit(directory, aw.apps[i].push_action+5);
    else if (!strncmp(aw.apps[i].push_action, "VIEW ", 5))
      doView(directory, aw.apps[i].push_action+5);
    else if (!strcmp(aw.apps[i].push_action, "OPEN")) {
      newFileWindow(directory, resources.default_display_type,
		    False);
    } else if (!strcmp(aw.apps[i].push_action, "LOAD")) {
      zzz();
      AmPushLoadFile(directory,False);
      wakeUp();
    } else {
      char *action = varPopup(aw.shell, aw.apps[i].icon_bm, aw.apps[i].push_action);
      const char **args = makeShellArgs(action,NULL);
      execute_external_program(args[0],directory,args,NULL,NULL);
      XTFREE(args);
      XtFree(action);
    }
  }
}

/*---------------------------------------------------------------------------*/
static void appEndMoveInBox(void);
static void appEndCopyInBox(void);
static void appEndCopy(int i);
static void appEndMove(Cardinal i);

void appBeginDrag(Widget w, XEvent *event, String *params, 
		  Cardinal *num_params)
{
  int i, button;
  Boolean on_root_return, move = True;

  if (*num_params != 2) {
    error("Internal error:","wrong number of parameters to appBeginDrag");
    return;
  }

  button = *params[0] - '0';
  if (!strcmp(params[1],"copy"))
    move = False;

  i =  findAppWidget(w);
  
  if (i != -1) {
    if (!aw.apps[i].selected)
      appSelect(w, event, params, num_params);
  } else
    return;
  
  if (!(w = drag(w,button,&on_root_return)))
    return;
  else if (on_root_return)
    return;

  if (w == aw.icon_box)
    if (move)
      appEndMoveInBox();
    else
      appEndCopyInBox();
  else if ((i = findAppWidget(w)) >= 0) {
    if (move)
      appEndMove(i);
    else
      appEndCopy(i);
  }
}
  
/*---------------------------------------------------------------------------*/

static void appEndMoveInBox(void)
{
  char s[0xff];
  Cardinal j, k, l;
  AppRec *apps;

  if (aw.n_selections == 0) return;

  if (resources.confirm_moves) {
    sprintf(s, "Moving %d item%s in", aw.n_selections,
	    aw.n_selections > 1 ? "s" : "" );
    if (!amConfirm(s, "the application window", "", NULL))
      return;
  }

  apps = (AppRec*) XtMalloc(aw.n_apps*sizeof(AppRec));
  assert(aw.n_apps >= aw.n_selections);
  for (j = k = 0, l = aw.n_apps - aw.n_selections; j < aw.n_apps; j++)
    if (aw.apps[j].selected)
      memcpy(&apps[l++], &aw.apps[j], sizeof(AppRec));
    else
      memcpy(&apps[k++], &aw.apps[j], sizeof(AppRec));
  XTFREE(aw.apps);
  aw.apps = apps;
  updateApplicationDisplay();
  writeApplicationData();
}

/*---------------------------------------------------------------------------*/

static void appEndMove(Cardinal i)
{
  char s[0xff];
  Cardinal j, k, l, m;
  AppRec *apps;

  if (aw.n_selections == 0 || aw.apps[i].selected) return;

  if (resources.confirm_moves) {
    sprintf(s, "Moving %d item%s in", aw.n_selections,
	    aw.n_selections > 1 ? "s" : "" );
    if (!amConfirm(s, "the application window", "", NULL))
      return;
  }

  apps = (AppRec*) XtMalloc(aw.n_apps*sizeof(AppRec));
  for (j = l = 0; j < i; j++)
    if (!aw.apps[j].selected) l++;
  for (j = k = 0, m = l + aw.n_selections; j < aw.n_apps; j++)
    if (aw.apps[j].selected)
      memcpy(&apps[l++], &aw.apps[j], sizeof(AppRec));
    else if (j < i)
      memcpy(&apps[k++], &aw.apps[j], sizeof(AppRec));
    else
      memcpy(&apps[m++], &aw.apps[j], sizeof(AppRec));
  XTFREE(aw.apps);
  aw.apps = apps;
  updateApplicationDisplay();
  writeApplicationData();
}

/*---------------------------------------------------------------------------*/

static void copyApps(void)
{
  int j, n_apps = aw.n_apps;

  aw.apps = (AppList) XTREALLOC(aw.apps, (aw.n_apps+aw.n_selections) *
					  sizeof(AppRec));
  for (j=0; j<n_apps; j++)
    if (aw.apps[j].selected) {
      aw.apps[aw.n_apps].name = XtNewString(aw.apps[j].name);
      aw.apps[aw.n_apps].directory = XtNewString(aw.apps[j].directory);
      aw.apps[aw.n_apps].icon = XtNewString(aw.apps[j].icon);
      aw.apps[aw.n_apps].push_action = XtNewString(aw.apps[j].push_action);
      aw.apps[aw.n_apps].drop_action = XtNewString(aw.apps[j].drop_action);
      aw.apps[aw.n_apps].icon_bm = aw.apps[j].icon_bm;
      aw.apps[aw.n_apps].form = aw.apps[aw.n_apps].toggle =
	aw.apps[aw.n_apps].label = NULL;
      aw.n_apps++;
    }
}

/*---------------------------------------------------------------------------*/

void appEndCopyInBox(void)
{
  char s[0xff];
  if (aw.n_selections == 0) return;

  if (resources.confirm_copies) {
    sprintf(s, "Copying %d item%s in", aw.n_selections,
	    aw.n_selections > 1 ? "s" : "" );
    if (!amConfirm(s, "the application window", "", NULL))
      return;
  }

  copyApps();
  updateApplicationDisplay();
  writeApplicationData();
}

/*---------------------------------------------------------------------------*/

void appEndCopy(int i)
{
  char s[0xff];
  int n_apps = aw.n_apps;
  AppRec *apps;

  if (aw.n_selections == 0) return;

  if (resources.confirm_copies) {
    sprintf(s, "Copying %d item%s in", aw.n_selections,
	    aw.n_selections > 1 ? "s" : "" );
    if (!amConfirm(s, "the application window", "", NULL))
      return;
  }

  copyApps();
  apps = (AppRec*) XtMalloc(aw.n_apps * sizeof(AppRec));
  memcpy(apps, aw.apps, i*sizeof(AppRec));
  memcpy(&apps[i], &aw.apps[n_apps], (aw.n_apps-n_apps)*sizeof(AppRec));
  memcpy(&apps[i+aw.n_apps-n_apps], &aw.apps[i], (n_apps-i)*sizeof(AppRec));
  XTFREE(aw.apps);
  aw.apps = apps;
  updateApplicationDisplay();
  writeApplicationData();
}

/*---------------------------------------------------------------------------*/

void appEndDrag(int i)
{
  if (*aw.apps[i].drop_action) {
    char *action = varPopup(aw.shell, aw.apps[i].icon_bm, aw.apps[i].drop_action);
    if (!action) return;
    fileCallWithSelected(action);
    XTFREE(action);
  }
}

/*---------------------------------------------------------------------------*/

void appEndDragInBox(void)
{
  int i;

  if (aw.readonly)
	  return;

  for (i=0; i<move_info.fw->n_files; i++)
    if (move_info.fw->files[i]->selected) {
      if (S_ISDIR(move_info.fw->files[i]->stats.st_mode)) {
	char *fullname = dirConcat(move_info.fw->directory,
			move_info.fw->files[i]->name);
	installApplication(move_info.fw->files[i]->name,
                           fullname,
			   "",
			   "OPEN",
			   "");
	XtFree(fullname);
      } else if (move_info.fw->files[i]->stats.st_mode &
		 (S_IXUSR | S_IXGRP | S_IXOTH)) {
	char *push_action, *drop_action;
	push_action = move_info.fw->files[i]->name;
	drop_action = (char *)alloca(strlen(push_action)+6);
	strcpy(drop_action, push_action);
	strcat(drop_action, " \"$*\"");
	installApplication(move_info.fw->files[i]->name,
			   move_info.fw->directory,
			   "",
			   push_action,
			   drop_action);
      } else {
// TODO: re-add support for storing how to start that file here
	char *editcmd = spaceConcat("EDIT",
			move_info.fw->files[i]->name);
	installApplication(move_info.fw->files[i]->name,
			   move_info.fw->directory,
			   "",
			   editcmd,
			   "");
	XtFree(editcmd);
      }
    }
  updateApplicationDisplay();
  writeApplicationData();
}

/*---------------------------------------------------------------------------*/

void appSelect(Widget w, UNUSED(XEvent *event),
	       UNUSED(String *params), UNUSED(Cardinal *num_params))
{
  Cardinal i;
  int j;
  Pixel back, fore;
  
  j = findAppWidget(w);
  if (j == -1) {
    error("Internal error:", "widget not found in appSelect");
    return;
  }
  
  for (i=0; i<aw.n_apps; i++)
    if (aw.apps[i].selected) {
      XtVaGetValues(aw.apps[i].toggle, XtNbackground, &back, NULL);
      XtVaSetValues(aw.apps[i].toggle, XtNborder, (XtArgVal) back, NULL);
      aw.apps[i].selected = False;
    }

  XtVaGetValues(w, XtNforeground, &fore, NULL);
  XtVaSetValues(w, XtNborder, (XtArgVal) fore, NULL);
  
  aw.apps[j].selected = True;
  aw.n_selections = 1;
}

/*---------------------------------------------------------------------------*/

void appToggle(Widget w, UNUSED(XEvent *event),
               UNUSED(String *params), UNUSED(Cardinal *num_params))
{
  int i;
  Pixel pix;
  
  i = findAppWidget(w);
  if (i == -1) {
    error("Internal error:", "widget not found in appToggle");
    return;
  }
  
  XtVaGetValues(w, aw.apps[i].selected?XtNbackground:XtNforeground, &pix,
		NULL);
  XtVaSetValues(w, XtNborder, (XtArgVal) pix, NULL);
  
  aw.apps[i].selected = !aw.apps[i].selected;
  if (aw.apps[i].selected)
    aw.n_selections++;
  else
    aw.n_selections--;
}

/*---------------------------------------------------------------------------*/
void appTrackCursor(Widget w, XEvent *event, UNUSED(String *params),
		 UNUSED(Cardinal *num_params))
{
	if (dragging)
		drag_set_dropable(XtDisplay(w), 
				event->type != LeaveNotify && !aw.readonly);
	else if (dragging_app)
		awDragGrab(event->type != LeaveNotify && !aw.readonly);
}

