/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2008-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

/**
 * @defgroup PublicHighLevelAPI High Level API
 * @ingroup PublicAPI
 * @brief The public part
 */
/**
 * @defgroup PublicHighLevelStandard High Level Standard Functions
 * @ingroup PublicHighLevelAPI
 * @brief The public part
 */
/*@{*/

#ifndef _SML_DATA_SYNC_API_STANDARD_H_
#define _SML_DATA_SYNC_API_STANDARD_H_

#include <libsyncml/syncml.h>

/*! @brief These are all possible events which the high level API sends.
 */
typedef enum SmlDataSyncEventType {
	SML_DATA_SYNC_EVENT_ERROR = 0,
	SML_DATA_SYNC_EVENT_CONNECT = 1,
	SML_DATA_SYNC_EVENT_GOT_ALL_ALERTS = 2,
	SML_DATA_SYNC_EVENT_GOT_ALL_CHANGES = 3,
	SML_DATA_SYNC_EVENT_GOT_ALL_MAPPINGS = 4,
	SML_DATA_SYNC_EVENT_DISCONNECT = 5,
	SML_DATA_SYNC_EVENT_FINISHED = 6,
} SmlDataSyncEventType;

typedef struct SmlDataSyncObject SmlDataSyncObject;

/* object creation and configuration */

SmlDataSyncObject *smlDataSyncNew(
			SmlSessionType dsType,
			SmlTransportType tspType,
			SmlError **error);
void smlDataSyncObjectRef(SmlDataSyncObject *dsObject);
SmlBool smlDataSyncSetOption(
			SmlDataSyncObject *dsObject,
			const char *name,
			const char *value,
			SmlError **error);
SmlBool smlDataSyncAddDatastore(SmlDataSyncObject *dsObject,
			const char *contentType,
			const char *target,
			const char *source,
			SmlError **error);

/* init session */

SmlBool smlDataSyncInit(SmlDataSyncObject *dsObject, SmlError **error);
SmlBool smlDataSyncRun(SmlDataSyncObject *dsObject, SmlError **error);

/* synchronize */

SmlBool smlDataSyncAddChange(
			SmlDataSyncObject *dsObject,
			const char *source,
			SmlChangeType type,
			const char *name,
			const char *data,
			unsigned int size,
			void *userdata, /* for SmlDataSyncChangeStatusCallback */
			SmlError **error);
SmlBool smlDataSyncSendChanges(SmlDataSyncObject *dsObject, SmlError **error);
SmlBool smlDataSyncAddMapping(
			SmlDataSyncObject *dsObject,
			const char *source,
			const char *remoteID,
			const char *localID,
			SmlError **error);


/**
 * Returns the identity of the remote sync peer. This is the
 * <Source></Source> tag. The returned reference is not owned
 * by the caller and must NOT be freed or unrefed by the caller.
 *
 * @param dsObject The data sync object
 * @param error The error condition
 * @return The identity of the remote peer
 */
const SmlLocation *smlDataSyncGetTarget(
			SmlDataSyncObject *dsObject, 
			SmlError **error);

/* close session */

void smlDataSyncObjectUnref(SmlDataSyncObject **dsObject);

#endif /* _SML_DATA_SYNC_API_STANDARD_H_ */

/*@}*/

