/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2008-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "transport.h"
#include "libsyncml/sml_error_internals.h"
#include "libsyncml/sml_support.h"

#include "data_sync_client.h"
#include "data_sync_devinf.h"


SmlBool smlDataSyncTransportHttpClientInit(
			SmlDataSyncObject *dsObject,
			SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, dsObject, error);
	CHECK_ERROR_REF

	if (dsObject->url != NULL &&
	    !smlTransportSetConfigOption(
			dsObject->tsp,
			SML_TRANSPORT_CONFIG_URL, dsObject->url,
			error))
		goto error;
	if (dsObject->username != NULL &&
	    !smlTransportSetConfigOption(
			dsObject->tsp,
			SML_TRANSPORT_CONFIG_USERNAME, dsObject->username,
			error))
		goto error;
	if (dsObject->password != NULL &&
	    !smlTransportSetConfigOption(
			dsObject->tsp,
			SML_TRANSPORT_CONFIG_PASSWORD, dsObject->password,
			error))
		goto error;

	smlTrace(TRACE_EXIT, "%s - TRUE", __func__);
	return TRUE;
error:
	smlTrace(TRACE_EXIT_ERROR, "%s - %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool smlDataSyncTransportHttpClientInitNewSession (SmlDataSyncObject *dsObject, SmlError **error)
{
	smlAssert(dsObject->url);
	smlAssert(dsObject->identifier);
	smlAssert(dsObject->tsp);
	smlAssert(dsObject->manager);
	smlAssert(dsObject->agent);

	SmlLocation *target = NULL;
	SmlLocation *source = NULL;

	/* If there is an old session then this is a bug.
	 * the high level API only support one session per
	 * OMA DS object.
	 */
	if (dsObject->session) {
		smlErrorSet(error, SML_ERROR_GENERIC,
			"Only one SyncML session is allowed per SmlDataSyncObject instance.");
		goto error;
	}

	/* create session */

	target = smlLocationNew(dsObject->target, NULL, error);
	if (!target)
		goto error;
	source = smlLocationNew(dsObject->identifier, NULL, error);
	if (!source)
		goto error;

	char *sessionString = smlManagerGetNewSessionID(dsObject->manager);
	dsObject->session = smlSessionNew(SML_SESSION_TYPE_CLIENT,
					SML_MIMETYPE_XML,
					dsObject->version,
					SML_PROTOCOL_SYNCML,
					target, source,
					sessionString, 0, error);
	smlSafeCFree(&sessionString);
	if (!dsObject->session)
		goto error;

	smlLocationUnref(target);
	target = NULL;
	smlLocationUnref(source);
	source = NULL;

	/* register all the add-ons */
	if (!smlManagerSessionAdd(dsObject->manager, dsObject->session, NULL, error))
		goto error;
	if (!smlDevInfAgentRegisterSession(dsObject->agent, dsObject->manager, dsObject->session, error))
        	goto error;

	return TRUE;
error:
	if (target) {
		smlLocationUnref(target);
		target = NULL;
	}
	if (source) {
		smlLocationUnref(source);
		source = NULL;
	}
	return FALSE;
}

SmlBool smlDataSyncTransportHttpClientConnect(SmlDataSyncObject *dsObject, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, dsObject, error);
	CHECK_ERROR_REF

	dsObject->tryDisconnect = FALSE;

	/* init new session */
	if (!smlDataSyncTransportHttpClientInitNewSession(dsObject, error))
		goto error;

	/* send the device information */
	if (!smlDataSyncManageDevInf(dsObject, error))
		goto error;
	smlTrace(TRACE_INTERNAL, "%s: sent devinf", __func__);

	/* prepare correct alert type */
	GList *o = dsObject->datastores;
	for (; o; o = o->next) { 
		SmlDataSyncDatastore *datastore = o->data;
		if (!smlDataSyncClientSendAlert(datastore, SML_ALERT_SLOW_SYNC, error))
			goto error;
	}
	smlTrace(TRACE_INTERNAL, "%s: all datastores created their alerts", __func__);

	/* If no SAN and no server alerted sync is used
	 * then we must set the actual package to 1.
	 * Otherwise the state management thinks the first
	 * received final is part of a SAN package.
	 */
	dsObject->actualPackage = SML_PACKAGE_1;

	/* flush package 1 */
	if (!smlSessionFlush(dsObject->session, TRUE, error))
		goto error;

	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
error:
	smlTrace(TRACE_EXIT_ERROR, "%s - %s", __func__, smlErrorPrint(error));
	return FALSE;
}

