/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "support.h"
#include <fnmatch.h>

START_TEST (check_location_new)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *location = smlLocationNew("uri", "name", &error);
	sml_fail_unless(location != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(!strcmp(smlLocationGetURI(location), "uri"), NULL);
	sml_fail_unless(!strcmp(smlLocationGetName(location), "name"), NULL);

	smlLocationUnref(location);
}
END_TEST

START_TEST (check_location_new2)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *location = smlLocationNew("uri", NULL, &error);
	sml_fail_unless(location != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(!strcmp(smlLocationGetURI(location), "uri"), NULL);
	sml_fail_unless(smlLocationGetName(location) == NULL, NULL);

	smlLocationUnref(location);
}
END_TEST

START_TEST (check_location_new3)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *location = smlLocationNew(NULL, NULL, &error);
	sml_fail_unless(location == NULL, NULL);
	sml_fail_unless(error != NULL, NULL);
	
	smlErrorDeref(&error);
}
END_TEST

START_TEST (check_location_compare_rel)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *objloc = smlLocationNew("./test", NULL, &error);
	sml_fail_unless(objloc != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);

	sml_fail_unless(smlLocationCompare(NULL, objloc, NULL, objloc), NULL);

	smlLocationUnref(objloc);
}
END_TEST

START_TEST (check_location_compare_rel1)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *objloc = smlLocationNew("/test", NULL, &error);
	sml_fail_unless(objloc != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);

	sml_fail_unless(smlLocationCompare(NULL, objloc, NULL, objloc), NULL);
	
	smlLocationUnref(objloc);
}
END_TEST

START_TEST (check_location_compare_rel2)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *objloc = smlLocationNew("./test", NULL, &error);
	sml_fail_unless(objloc != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc2 = smlLocationNew("./test2", NULL, &error);
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(!smlLocationCompare(NULL, objloc, NULL, objloc2), NULL);

	smlLocationUnref(objloc);
	smlLocationUnref(objloc2);
}
END_TEST

START_TEST (check_location_compare_abs)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *objbase = smlLocationNew("/test", NULL, &error);
	sml_fail_unless(objbase != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);

	sml_fail_unless(smlLocationCompare(objbase, NULL, objbase, NULL), NULL);

	smlLocationUnref(objbase);
}
END_TEST

START_TEST (check_location_compare_abs2)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *objbase = smlLocationNew("/test", NULL, &error);
	sml_fail_unless(objbase != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objbase2 = smlLocationNew("/test2", NULL, &error);
	sml_fail_unless(objbase2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);

	sml_fail_unless(smlLocationCompare(objbase, NULL, objbase2, NULL), NULL);

	smlLocationUnref(objbase);
	smlLocationUnref(objbase2);
}
END_TEST

START_TEST (check_location_compare_combined)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *objbase = smlLocationNew("/test", NULL, &error);
	sml_fail_unless(objbase != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc = smlLocationNew("./test", NULL, &error);
	sml_fail_unless(objloc != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	SmlLocation *objbase2 = smlLocationNew("/test", NULL, &error);
	sml_fail_unless(objbase2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc2 = smlLocationNew("./test", NULL, &error);
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(smlLocationCompare(objbase, objloc, objbase2, objloc2), NULL);
	
	smlLocationUnref(objbase);
	smlLocationUnref(objloc);
	smlLocationUnref(objbase2);
	smlLocationUnref(objloc2);
}
END_TEST

START_TEST (check_location_compare_combined2)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *objbase = smlLocationNew("/test", NULL, &error);
	sml_fail_unless(objbase != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc = smlLocationNew("test", NULL, &error);
	sml_fail_unless(objloc != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	SmlLocation *objbase2 = smlLocationNew("/test", NULL, &error);
	sml_fail_unless(objbase2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc2 = smlLocationNew("./test", NULL, &error);
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(smlLocationCompare(objbase, objloc, objbase2, objloc2), NULL);
	
	smlLocationUnref(objbase);
	smlLocationUnref(objloc);
	smlLocationUnref(objbase2);
	smlLocationUnref(objloc2);
}
END_TEST

START_TEST (check_location_compare_combined3)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *objbase = smlLocationNew("/test/asd", NULL, &error);
	sml_fail_unless(objbase != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc = smlLocationNew("test/", NULL, &error);
	sml_fail_unless(objloc != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	SmlLocation *objbase2 = smlLocationNew("/test/.//asd//", NULL, &error);
	sml_fail_unless(objbase2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc2 = smlLocationNew("././test", NULL, &error);
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(smlLocationCompare(objbase, objloc, objbase2, objloc2), NULL);
	
	smlLocationUnref(objbase);
	smlLocationUnref(objloc);
	smlLocationUnref(objbase2);
	smlLocationUnref(objloc2);
}
END_TEST

START_TEST (check_location_compare_combined4)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *objbase = smlLocationNew("/test/asd2", NULL, &error);
	sml_fail_unless(objbase != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc = smlLocationNew("test", NULL, &error);
	sml_fail_unless(objloc != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	SmlLocation *objbase2 = smlLocationNew("/test/.//asd//", NULL, &error);
	sml_fail_unless(objbase2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc2 = smlLocationNew("././test", NULL, &error);
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(!smlLocationCompare(objbase, objloc, objbase2, objloc2), NULL);
	
	smlLocationUnref(objbase);
	smlLocationUnref(objloc);
	smlLocationUnref(objbase2);
	smlLocationUnref(objloc2);
}
END_TEST

START_TEST (check_location_compare_combined5)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *objbase = smlLocationNew("/test/asd", NULL, &error);
	sml_fail_unless(objbase != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc = smlLocationNew("tes3t", NULL, &error);
	sml_fail_unless(objloc != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	SmlLocation *objbase2 = smlLocationNew("/test/.//asd//", NULL, &error);
	sml_fail_unless(objbase2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc2 = smlLocationNew("././test", NULL, &error);
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(!smlLocationCompare(objbase, objloc, objbase2, objloc2), NULL);
	
	smlLocationUnref(objbase);
	smlLocationUnref(objloc);
	smlLocationUnref(objbase2);
	smlLocationUnref(objloc2);
}
END_TEST

START_TEST (check_location_compare_combined6)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlLocation *objbase = smlLocationNew("/", NULL, &error);
	sml_fail_unless(objbase != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc = smlLocationNew("", NULL, &error);
	sml_fail_unless(objloc != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	SmlLocation *objbase2 = smlLocationNew(".//", NULL, &error);
	sml_fail_unless(objbase2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	SmlLocation *objloc2 = smlLocationNew("///", NULL, &error);
	sml_fail_unless(objloc2 != NULL, NULL);
	sml_fail_unless(error == NULL, NULL);
	
	sml_fail_unless(smlLocationCompare(objbase, objloc, objbase2, objloc2), NULL);
	
	smlLocationUnref(objbase);
	smlLocationUnref(objloc);
	smlLocationUnref(objbase2);
	smlLocationUnref(objloc2);
}
END_TEST

Suite *elements_suite(void)
{
	Suite *s = suite_create("Elements");
	//Suite *s2 = suite_create("Elements");
	
	create_case(s, "check_location_new", check_location_new);
	create_case(s, "check_location_new2", check_location_new2);
	create_case(s, "check_location_new3", check_location_new3);
	create_case(s, "check_location_compare_rel", check_location_compare_rel);
	create_case(s, "check_location_compare_rel1", check_location_compare_rel1);
	create_case(s, "check_location_compare_rel2", check_location_compare_rel2);
	create_case(s, "check_location_compare_abs", check_location_compare_abs);
	create_case(s, "check_location_compare_abs2", check_location_compare_abs2);
	create_case(s, "check_location_compare_combined", check_location_compare_combined);
	create_case(s, "check_location_compare_combined2", check_location_compare_combined2);
	create_case(s, "check_location_compare_combined3", check_location_compare_combined3);
	create_case(s, "check_location_compare_combined4", check_location_compare_combined4);
	create_case(s, "check_location_compare_combined5", check_location_compare_combined5);
	create_case(s, "check_location_compare_combined6", check_location_compare_combined6);
	
	return s;
}

int main(void)
{
	configure_environment();

	int nf;

	Suite *s = elements_suite();
	
	SRunner *sr;
	sr = srunner_create(s);
	srunner_run_all(sr, CK_VERBOSE);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);

	cleanup_environment();

	return (nf == 0) ? 0 : 1;
}
