/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2007-2008  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "support.h"

#include "libsyncml/parser/sml_xml_parse.c"

START_TEST (check_no_item)
{
	const char *input_data = "askfdjasdkfjasd masldfjasldk<Data>sdafs</Data>adada";

	char *fixed_data = NULL;
	unsigned int fixed_size = 0;
	SmlError *error = NULL;

	sml_fail_unless(
		_smlXmlParserFixBrokenItemData(
			input_data, strlen(input_data),
			&fixed_data, &fixed_size,
			&error),
		"%s", smlErrorPrint(&error));
	sml_fail_unless(error == NULL, "%s", smlErrorPrint(&error));
	sml_fail_unless(fixed_size == strlen(input_data), "The data must not be fixed (size has changed).");
	sml_fail_unless(strcmp(input_data, fixed_data) == 0, "The data must not be fixed.");
	g_free(fixed_data);
}
END_TEST

START_TEST (check_no_item_data)
{
	const char *input_data = "askfdjasdkfj<Item>sdafs</Item>adada";

	char *fixed_data = NULL;
	unsigned int fixed_size = 0;
	SmlError *error = NULL;

	sml_fail_unless(
		_smlXmlParserFixBrokenItemData(
			input_data, strlen(input_data),
			&fixed_data, &fixed_size,
			&error),
		"%s", smlErrorPrint(&error));
	sml_fail_unless(error == NULL, "%s", smlErrorPrint(&error));
	sml_fail_unless(fixed_size == strlen(input_data), "The data must not be fixed (size has changed).");
	sml_fail_unless(strcmp(input_data, fixed_data) == 0, "The data must not be fixed.");
	g_free(fixed_data);
}
END_TEST

START_TEST (check_no_broken_item_data)
{
	const char *input_data = "askfdjasdkfj<Item><Data><![CDATA[This is binary data.]]></Data></Item>adada";

	char *fixed_data = NULL;
	unsigned int fixed_size = 0;
	SmlError *error = NULL;

	sml_fail_unless(
		_smlXmlParserFixBrokenItemData(
			input_data, strlen(input_data),
			&fixed_data, &fixed_size,
			&error),
		"%s", smlErrorPrint(&error));
	sml_fail_unless(error == NULL, "%s", smlErrorPrint(&error));
	sml_fail_unless(fixed_size == strlen(input_data), "The data must not be fixed (size has changed).");
	sml_fail_unless(strcmp(input_data, fixed_data) == 0, "The data must not be fixed.");
	g_free(fixed_data);
}
END_TEST

START_TEST (check_one_broken_item_data)
{
	const char *input_data = "askfdjasdkfj<Item><Data>This is binary data.</Data></Item>adada";
	const char *output_data = "askfdjasdkfj<Item><Data><![CDATA[This is binary data.]]></Data></Item>adada";

	char *fixed_data = NULL;
	unsigned int fixed_size = 0;
	SmlError *error = NULL;

	sml_fail_unless(
		_smlXmlParserFixBrokenItemData(
			input_data, strlen(input_data),
			&fixed_data, &fixed_size,
			&error),
		"%s", smlErrorPrint(&error));
	sml_fail_unless(error == NULL, "%s", smlErrorPrint(&error));

	/* compare it and check the expected length */
	sml_fail_unless(fixed_size != strlen(input_data), "The data must be fixed (size has not changed).");
	sml_fail_unless(strcmp(input_data, fixed_data) != 0, "The data must be fixed.");
	sml_fail_unless(fixed_size == strlen(output_data), "The length of the fixed data is wrong.");
	sml_fail_unless(strcmp(output_data, fixed_data) == 0, "The fixed data is wrong.\nExpected: %s\nOutput: %s", output_data, fixed_data);

	g_free(fixed_data);
}
END_TEST

START_TEST (check_multi_broken_item_data)
{
	const char *input_data = "askf<Item><Data>test1</Data></Item>" \
				"adad<Item><Data>test2</Data></Item>adada";
	const char *output_data = "askf<Item><Data><![CDATA[test1]]></Data></Item>" \
				"adad<Item><Data><![CDATA[test2]]></Data></Item>adada";

	char *fixed_data = NULL;
	unsigned int fixed_size = 0;
	SmlError *error = NULL;

	sml_fail_unless(
		_smlXmlParserFixBrokenItemData(
			input_data, strlen(input_data),
			&fixed_data, &fixed_size,
			&error),
		"%s", smlErrorPrint(&error));
	sml_fail_unless(error == NULL, "%s", smlErrorPrint(&error));

	/* compare it and check the expected length */
	sml_fail_unless(fixed_size != strlen(input_data), "The data must be fixed (size has not changed).");
	sml_fail_unless(strcmp(input_data, fixed_data) != 0, "The data must be fixed.");
	sml_fail_unless(fixed_size == strlen(output_data), "The length of the fixed data is wrong.");
	sml_fail_unless(strcmp(output_data, fixed_data) == 0, "The fixed data is wrong.\nExpected: %s\nOutput: %s", output_data, fixed_data);

	g_free(fixed_data);
}
END_TEST

START_TEST (check_mixed_broken_item_data)
{
	const char *input_data = "askf<Item><Data>test1</Data></Item>" \
				"adad<Item><Data><![CDATA[test2]]></Data></Item>adada" \
				"adad<Item><Data>test3</Data></Item>adada" \
				"adad<Item><Data><![CDATA[test4]]></Data></Item>adada" \
				"adad<Item><Data>test5</Data></Item>adada";
	const char *output_data = "askf<Item><Data><![CDATA[test1]]></Data></Item>" \
				"adad<Item><Data><![CDATA[test2]]></Data></Item>adada" \
				"adad<Item><Data><![CDATA[test3]]></Data></Item>adada" \
				"adad<Item><Data><![CDATA[test4]]></Data></Item>adada" \
				"adad<Item><Data><![CDATA[test5]]></Data></Item>adada";

	char *fixed_data = NULL;
	unsigned int fixed_size = 0;
	SmlError *error = NULL;

	sml_fail_unless(
		_smlXmlParserFixBrokenItemData(
			input_data, strlen(input_data),
			&fixed_data, &fixed_size,
			&error),
		"%s", smlErrorPrint(&error));
	sml_fail_unless(error == NULL, "%s", smlErrorPrint(&error));

	/* compare it and check the expected length */
	sml_fail_unless(fixed_size != strlen(input_data), "The data must be fixed (size has not changed).");
	sml_fail_unless(strcmp(input_data, fixed_data) != 0, "The data must be fixed.");
	sml_fail_unless(fixed_size == strlen(output_data), "The length of the fixed data is wrong.");
	sml_fail_unless(strcmp(output_data, fixed_data) == 0, "The fixed data is wrong.\nExpected: %s\nOutput: %s", output_data, fixed_data);

	g_free(fixed_data);
}
END_TEST

START_TEST (check_utf16_conversion)
{
	SmlError *error = NULL;
	char *input_data = smlTryMalloc0(20, &error);
	sml_fail_unless(error == NULL, "%s", smlErrorPrint(&error));
	input_data[0] = 65; /* A */
	input_data[1] = 66; /* B */
	input_data[3] = 67; /* C */
	input_data[5] = 68; /* D */
	input_data[6] = 69; /* E */
	input_data[7] = 70; /* F*/
	input_data[8] = 71; /* G */
	input_data[10] = 72; /* H */
	input_data[12] = 73; /* I */
	input_data[13] = 74; /* J */
	size_t length = 14;

	char *fixed_data = NULL;
	unsigned int fixed_size = 0;

	/* check UTF-16 conversion of_smlXmlParserFixBrokenItemData */

	sml_fail_unless(
		_smlXmlParserFixBrokenItemData(
			input_data, length,
			&fixed_data, &fixed_size,
			&error),
		"%s", smlErrorPrint(&error));
	sml_fail_unless(error == NULL, "%s", smlErrorPrint(&error));
	sml_fail_unless(fixed_size == strlen("ABCDEFGHIJ"), "%d != %d", fixed_size, strlen("ABCDEFGHIJ"));
	sml_fail_unless(strcmp(fixed_data, "ABCDEFGHIJ") == 0, NULL);
	g_free(input_data);
	g_free(fixed_data);
}
END_TEST

Suite *parser_suite(void)
{
	Suite *s = suite_create("Parser");
	//Suite *s2 = suite_create("Parser");
	
	create_case(s, "check_no_item", check_no_item);
	create_case(s, "check_no_item_data", check_no_item_data);
	create_case(s, "check_no_broken_item_data", check_no_broken_item_data);
	create_case(s, "check_one_broken_item_data", check_one_broken_item_data);
	create_case(s, "check_multi_broken_item_data", check_multi_broken_item_data);
	create_case(s, "check_mixed_broken_item_data", check_mixed_broken_item_data);
	create_case(s, "check_utf16_conversion", check_utf16_conversion);
	
	return s;
}

int main(void)
{
	configure_environment();

	int nf;

	Suite *s = parser_suite();
	
	SRunner *sr;
	sr = srunner_create(s);
	srunner_run_all(sr, CK_VERBOSE);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);

	cleanup_environment();

	return (nf == 0) ? 0 : 1;
}
