/*
 * diskutil.c - simple bindings for various disk related functions.
 * blatently stolen from anaconda/isys/isys.c
 *
 * Copyright 1999-2002 Red Hat, Inc.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <sys/mount.h>
#include <linux/cdrom.h>

/* fixup dev_t type declaration (needed for loop.h) */
#include <linux/posix_types.h>
#undef dev_t
#define dev_t __kernel_dev_t
#include <linux/loop.h>

#include "Python.h"


#ifndef CDROMEJECT
#define CDROMEJECT 0x5309
#endif

static PyObject * doMount(PyObject * s, PyObject * args);
static PyObject * doUMount(PyObject * s, PyObject * args);
static PyObject * doEjectCdrom(PyObject * s, PyObject * args);
static PyObject * doSync(PyObject * s, PyObject * args);
static PyObject * doLoSetup(PyObject * s, PyObject * args);
static PyObject * doUnLoSetup(PyObject * s, PyObject * args);
static PyObject * doFindUnusedLoopDev(PyObject * s, PyObject * args);

static PyMethodDef diskutilModuleMethods[] = {
    { "ejectcdrom", (PyCFunction) doEjectCdrom, METH_VARARGS, NULL },
    { "mount", (PyCFunction) doMount, METH_VARARGS, NULL },
    { "umount", (PyCFunction) doUMount, METH_VARARGS, NULL },
    { "sync", (PyCFunction) doSync, METH_VARARGS, NULL},
    { "losetup", (PyCFunction) doLoSetup, METH_VARARGS, NULL},
    { "unlosetup", (PyCFunction) doUnLoSetup, METH_VARARGS, NULL},
    { "findunusedloopdev", (PyCFunction) doFindUnusedLoopDev, METH_VARARGS, 
      NULL},
    { NULL }
};


static PyObject * doMount(PyObject * s, PyObject * args) {
    char * fs, * device, * mntpoint;
    int rc;
    int readOnly;

    if (!PyArg_ParseTuple(args, "sssi", &fs, &device, &mntpoint,
			  &readOnly)) return NULL;

    rc = mount(device, mntpoint, fs, readOnly, NULL);
    if (rc == -1)
	PyErr_SetFromErrno(PyExc_SystemError);
    else if (rc)
	PyErr_SetString(PyExc_SystemError, "mount failed");

    if (rc) return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject * doUMount(PyObject * s, PyObject * args) {
    char * fs;

    if (!PyArg_ParseTuple(args, "s", &fs)) return NULL;

    if (umount(fs)) {
	PyErr_SetFromErrno(PyExc_SystemError);
	return NULL;
    }

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject * doEjectCdrom(PyObject * s, PyObject * args) {
    int fd;

    if (!PyArg_ParseTuple(args, "i", &fd)) return NULL;

    if (ioctl(fd, CDROMEJECT, 1)) {
	PyErr_SetFromErrno(PyExc_SystemError);
	return NULL;
    }

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject * doSync(PyObject * s, PyObject * args) {
    int fd;

    if (!PyArg_ParseTuple(args, "", &fd)) return NULL;
    sync();

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject * doLoSetup(PyObject * s, PyObject * args) {
    int loopfd;
    int targfd;
    struct loop_info loopInfo;
    char * loopName;

    if (!PyArg_ParseTuple(args, "iis", &loopfd, &targfd, &loopName)) 
	return NULL;
    if (ioctl(loopfd, LOOP_SET_FD, targfd)) {
	PyErr_SetFromErrno(PyExc_SystemError);
	return NULL;
    }

    memset(&loopInfo, 0, sizeof(loopInfo));
    strcpy(loopInfo.lo_name, loopName);

    if (ioctl(loopfd, LOOP_SET_STATUS, &loopInfo)) {
	PyErr_SetFromErrno(PyExc_SystemError);
	return NULL;
    }

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject * doUnLoSetup(PyObject * s, PyObject * args) {
    int loopfd;

    if (!PyArg_ParseTuple(args, "i", &loopfd)) return NULL;
    if (ioctl(loopfd, LOOP_CLR_FD, 0)) {
	PyErr_SetFromErrno(PyExc_SystemError);
	return NULL;
    }

    Py_INCREF(Py_None);
    return Py_None;
}

void init_diskutil(void) {
    PyObject * m, * d;

    m = Py_InitModule("_diskutil", diskutilModuleMethods);
    d = PyModule_GetDict(m);
}

/* taken and tweaked from util-linux/mount/lomount.c:find_unused_loop_device */
static PyObject * doFindUnusedLoopDev(PyObject * s, PyObject * args) {
    char dev[20];
    int i, fd, somedev = 0, someloop = 0;
    struct stat statbuf;
    struct loop_info loopinfo;

    if (!PyArg_ParseTuple(args, "")) return NULL;

    for(i = 0; i < 256; i++) {
        sprintf(dev, "/dev/loop%d", i);
        if (stat (dev, &statbuf) == 0 && S_ISBLK(statbuf.st_mode)) {
            somedev++;
            fd = open (dev, O_RDONLY);
            if (fd >= 0) {
                if(ioctl (fd, LOOP_GET_STATUS, &loopinfo) == 0)
                    someloop++;             /* in use */
                else if (errno == ENXIO) {
                    close (fd);
                    return Py_BuildValue("s", dev);/* probably free */
                }
                close (fd);
            }
            continue;/* continue trying as long as devices exist */
        }
        break;
    }
    
    Py_INCREF(Py_None);
    return Py_None;
}
